////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   IConsoleInterface.cpp
//  Version:     v1.00
//  Created:     28/09/2009 by Steve Barnett.
//  Description: Interface for communication with console platforms
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef __ICONSOLEINTERFACE_H__
#define __ICONSOLEINTERFACE_H__

#define BUFFERSIZE 1024

#include "Common/Logger.h"

class IConsoleInterface
{
public:
	enum EResponseType
	{
		eRT_DumpFilename = 0,
		eRT_CVARValue,
		eRT_Max,
	};
	struct SResponseTarget
	{
	const char* const m_pHeader;
	char* const m_pTarget;
	};

	IConsoleInterface( const char* const pExePath, const char* const pArguments, const char* const pTarget, const char* const pPipeName, SLogger* const pLogger, bool verbose = false, bool outputTTY = true ) : m_pExePath( pExePath ), m_pArguments(pArguments), m_pTarget( pTarget ), m_pPipeName( pPipeName ), m_pLogger( pLogger ), m_verbose( verbose), m_outputTTY( outputTTY ) , m_quit( false ) { };
	virtual ~IConsoleInterface( void ) {};

	virtual bool Connect( void ) = 0;
	bool Reset( const std::vector<std::string> &cmdLineParams, bool hardReset = true ) { return ResetImpl( cmdLineParams, hardReset ); }
	virtual void Shutdown( void ) = 0;

	virtual void UpdateComms( void ) = 0;
	virtual bool CreatePipe( void ) = 0;
	virtual void SendToPipe( const char* const ) = 0;
	// Read from pipe takes a pointer to a buffer and the length of the buffer. A pointer to the buffer is returned on success otherwise NULL.
	virtual const char* ReadFromPipe( char* const, const int) = 0;
	virtual void ClosePipe( void ) = 0;
	virtual bool WaitForConnected( void ) = 0;
	virtual bool WaitForFinished( void ) = 0;
	// Wait for a response marked with <pIdentifier> (response is stored in the appropriate static buffer in this class and a pointer to it returned)
	virtual const char* const GetMatchingResponse( const EResponseType eExpectedType ) = 0;

	virtual bool GatherDumpFile( const char* const pMapName, const char* const pDumpFile, const char* const pOutputDir ) = 0;

	bool TargetRequestedQuit( void ) { return m_quit; };

	bool IsVerbose( void ) const { return m_verbose; }
	void SetVerbose( const bool verbose ) { m_verbose = verbose; }

	bool ShouldOutputTTY( void ) const { return m_outputTTY; }
	void SetOutputTTY( const bool outputTTY ) { m_outputTTY = outputTTY; }

protected:

	// Allow for platform implementation of Reset without using default parameters on a virtual method
	virtual bool ResetImpl( const std::vector<std::string> &cmdLineParams, bool hardReset ) = 0;

	virtual void PrintError( const char* const pError, const char* const pErrorCode = NULL )
	{
		if ( m_pLogger )
		{
			if ( pErrorCode )
			{
				m_pLogger->LogError( "%s (%s)", pError, pErrorCode );
			}
			else
			{
				m_pLogger->LogError( "%s (%s)", pError, pErrorCode );
			}
		}
	}

	SResponseTarget GetResponseData( const EResponseType eExpectedType )
	{
		// Use the expected response type to select an appropriate header and target to match
		static SResponseTarget responseInfo[eRT_Max] =
		{
			{ "dumpfile ", m_dumpFile },	// eRT_DumpFilename
			{ "cvarval ",	m_CVARValue },	// eRT_CVARValue
		};
		return responseInfo[eExpectedType];
	}

	SLogger* m_pLogger;

	const char* m_pExePath;
	const char* m_pArguments;
	const char* m_pTarget;
	const char* m_pPipeName;

	char m_dumpFile[BUFFERSIZE];
	char m_CVARValue[BUFFERSIZE];

	bool m_verbose;
	bool m_outputTTY;
	bool m_quit;

};

#endif
