////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   LoadTest.cpp
//  Version:     v1.00
//  Created:     01/10/2009 by Steve Barnett.
//  Description: Gather statistics by loading and running the Crysis exe and
//               gathering memory or performance logs per level
// -------------------------------------------------------------------------
//  History:
//
//	01/10/2009: Initial implementation
//
////////////////////////////////////////////////////////////////////////////

// Prevent windows.h (pulled in from Mailer.h via winsock.h) from #defining min and max as macros
#define NOMINMAX

// Headers
#include <string.h>
#include <conio.h>
#include <time.h>

#include "LoadTestOptions.h"
#include "LoadTestJob.h"

#include "Mailer.h"
#include "TelnetServer.h"

#include "Common/Config.h"
#include "Common/Logger.h"

#include <iostream>
#include <string>
#include <limits>

using namespace std;

// Static data
static JobList s_jobList;
static SLogger s_globalLog;

// Implementation
static void Usage( void )
{
	printf( "LoadTest: Run Crysis on a target console and gather memory stats for each level\n" );
	printf( "Usage:\n");
	printf( "\tLoadTest.exe <config>\n");
}

static bool Finished( void )
{
	// Check if all jobs are either complete or failed
	for ( JobList::iterator it = s_jobList.begin(), end = s_jobList.end(); it != end; it++ )
	{
		if ( !it->m_complete && !it->m_failed ) return false;
	}
	return true;
}

static void RunNextJob( void )
{
	SJob* pBestJob = NULL;
	int leastErrors = numeric_limits<int>::max();

	// Find the job with the least errors (failed runs) that's not already running, finished or failed
	for ( JobList::iterator it = s_jobList.begin(), end = s_jobList.end(); it != end; it++ )
	{
		SJob& jobDesc = *it;
		if ( !it->m_running && !it->m_complete && !it->m_failed )
		{
			if ( jobDesc.m_errors < leastErrors )
			{
				pBestJob = &jobDesc;
				leastErrors = jobDesc.m_errors;
			}
		}
	}

	if ( pBestJob )
	{
		CLoadTestJob job( pBestJob );
		// Job will mark itself complete or failed and track errors but we need to mark it as running or not
		// ToDo: When multithreading lock the job list for writing (running threads can are safe as we only need to lock the m_running flags and they leave this alone)
		pBestJob->m_running = true;
		CLoadTestJob::RunJob( &job );
		pBestJob->m_running = false;
	}
}

static void stdsprintf( std::string& sBuffer, const char* const pFormat, ... )
{
	char pBuffer[BUFFER_SIZE];
	va_list ap;
	va_start( ap, pFormat );
	vsnprintf_s( pBuffer, 512, 512, pFormat, ap );
	sBuffer += pBuffer;
	va_end( ap );
}

void OutputJobStatusReport( std::string& sStr, const char* const pTitle = NULL )
{
	const char* const pPending = "Pending";
	const char* const pComplete = "Complete";
	const char* const pFailed = "Failed";
	const char* const pRunning = "InProgress";
	const char* const pError = "Error";
	const char* const pStatusMap[8] =
	{						//	C	F	P		- (complete & 4) | (failed & 2) | (in progress & 1)
		pPending,	//	0	0	0
		pRunning,	//	0	0	1
		pFailed,	//	0	1	0
		pError,		//	0	1	1
		pComplete,//	1	0	0
		pError,		//	1	0	1
		pError,		//	1	1	0
		pError,		//	1	1	1
	};

	// Print out the status of all running jobs
	if ( pTitle ) { stdsprintf( sStr, "%s\n", pTitle ); }

	stdsprintf( sStr, "----------------------------------------------------------------------\n" );
	stdsprintf( sStr, "Job status:\n" );
	stdsprintf( sStr, "Map         Tries   Status          ...LogFiles, Errors and Warnings\n" );
	stdsprintf( sStr, "----------------------------------------------------------------------\n" );
	for ( JobList::iterator jobIt = s_jobList.begin(), jobEnd = s_jobList.end(); jobIt != jobEnd; ++jobIt )
	{
		SJob& jobDesc = *jobIt;
		// Job status line
		int iStatus = (jobDesc.m_complete ? 4 : 0) | (jobDesc.m_failed ? 2 : 0) | (jobDesc.m_running ? 1 : 0);
		stdsprintf( sStr, "%-12s%2d/%-2d   ", jobDesc.m_pConfig->mapList.GetMapNames()[0].c_str(), jobDesc.m_errors, jobDesc.m_pConfig->iMaxErrors );
		stdsprintf( sStr, "%-10s(%c%c%c)\n", pStatusMap[iStatus], jobDesc.m_complete ? 'c' : '-', jobDesc.m_failed ? 'f' : '-', jobDesc.m_running ? 'r' : '-' );
		for ( SJob::LogList::iterator logIt = jobDesc.m_logs.begin(), logEnd = jobDesc.m_logs.end(); logIt != logEnd; ++logIt )
		{
			stdsprintf( sStr, "  %-40s(%d error%s, %d warning%s):\n", (*logIt).m_logfile.c_str(), (*logIt).m_logger.m_errors, ((*logIt).m_logger.m_errors != 1) ? "s" : "", (*logIt).m_logger.m_warnings, ((*logIt).m_logger.m_warnings != 1) ? "s" : "" );
			// Error report for this logfile
			for ( SLogger::ErrorList::iterator errIt = (*logIt).m_logger.m_output.begin(), errEnd = (*logIt).m_logger.m_output.end(); errIt != errEnd; ++errIt )
			{
				stdsprintf( sStr, "    %s\n", (*errIt).c_str() );
			}
		}
		stdsprintf( sStr, "----------------------------------------------------------------------\n" );
	}
	// Any global errors (not from any specific job)
	if ( s_globalLog.m_errors > 0 || s_globalLog.m_warnings > 0 )
	{
		stdsprintf( sStr, "Other warnings and errors:\n" );
			for ( SLogger::ErrorList::iterator errIt = s_globalLog.m_output.begin(), errEnd = s_globalLog.m_output.end(); errIt != errEnd; ++errIt )
		{
			stdsprintf( sStr, "  %s\n", (*errIt).c_str() );
		}
		stdsprintf( sStr, "----------------------------------------------------------------------\n" );
	}
}

void OutputJobStatusReport( FILE* f, const char* const pTitle = NULL )
{
	std::string str;
	OutputJobStatusReport( str, pTitle );
	fprintf( f, str.c_str() );
}

static void MailStatusReport( const char* const pFilename, const char* const pTitle, const SConfig& config )
{
	// If a report was created mail it to whoever is interested
	if ( !config.sSMTPServer.empty() && !config.sSMTPSender.empty() && !config.recipients.empty() )
	{
		CSMTPMailer::tstrcol rcpt;
		CSMTPMailer::tstrcol cc;
		CSMTPMailer::tstrcol bcc;
		CSMTPMailer::tstrcol attachments;
		CSMTPMailer::tstr mailSubject( pTitle );

		for ( SConfig::MailingList::const_iterator it = config.recipients.begin(), end = config.recipients.end(); it != end; ++it )
		{
			rcpt.push_back( *it );
		}

		CSMTPMailer::tstr mailBody( "Message generated from'" );
		mailBody += pFilename;
		mailBody += "':\n\n";

		FILE* pFile;
		fopen_s( &pFile, pFilename, "r" );
		if ( pFile )
		{
			while ( !feof( pFile ) )
			{
				char c = fgetc( pFile );
				if ( !feof( pFile ) )
				{
					mailBody += c;
				}
			}
			fclose( pFile );

			CSMTPMailer mail("", "", config.sSMTPServer, config.uSMTPPort );
			bool res = mail.Send( config.sSMTPSender, rcpt, cc, bcc, mailSubject, mailBody, attachments );
			if ( !res )
			{
				s_globalLog.LogError( "[Error] Failed to send status report using server '%s'.\n", config.sSMTPServer.c_str() );
			}
		}
	}
}

int main(int argc, char* argv[])
{
	const char* pConfig = NULL;
	bool error = false;

	if ( argc == 2 )
	{
		pConfig = argv[1];
	}
	else
	{
		Usage();
		return 0;
	}

	// Create a telnet server to allow remote control of this LoadTest instance
	// Listener not started yet...
	CTelnetServer server( &s_jobList );

	// This config instance stores the global settings and takes care of the config parsing
	SConfig config;
	// Process the config file and generate a job list
	if ( !config.ParseConfigFile( pConfig, &s_globalLog, &s_jobList ) )
	{
		s_globalLog.LogError( "Failed to load config file '%s'.\n", pConfig );
		error = true;
	}
	else
	{
		if ( config.bTelnet )
		{
			server.StartListenerThread( config.uTelnetPort );
		}

		OutputJobStatusReport( stdout );
		// Run jobs until all are finished or have failed
		while ( !Finished() )
		{
			RunNextJob();
			OutputJobStatusReport( stdout );
		}
	}

	// Write a final status report to the output directory and mail it if necessary
	string sReportTitle = "LoadTest status report";
	if ( !config.sSubjectSuffixFile.empty() )
	{
		sReportTitle += " ";
		
		FILE* pFile;
		fopen_s( &pFile, config.sSubjectSuffixFile.c_str(), "r" );
		if ( pFile )
		{
			while ( !feof( pFile ) )
			{
				char c = fgetc( pFile );
				if ( !feof( pFile ) && c != '\n' && c != '\r' ) sReportTitle += c;
			}
			fclose( pFile );
		}
	}

	time_t rawtime = time( NULL );
	tm ts;
	localtime_s( &ts, &rawtime );
	char filename[512];
	bool bReportWritten = false;
	for ( int i = 0; i < 2; i++ )
	{
		// Two passes to output either to the memory log directory, perf stats directory or both
		const char* dir = NULL;
		switch ( i )
		{
			case 0:
			{
				if ( config.sMemoryStatsDir.empty() ) continue;
				dir = config.sMemoryStatsDir.c_str();
			} break;
			case 1:
			{
				if ( config.sPerfStatsDir.empty() ) continue;
				dir = config.sPerfStatsDir.c_str();
			} break;
			default: continue;
		}
		sprintf_s( filename, sizeof(filename), "%s/loadtest-status-%04i%02i%02i-%02i%02i%02i.log", dir, 1900 + ts.tm_year, 1 + ts.tm_mon, ts.tm_mday, ts.tm_hour, ts.tm_min, ts.tm_sec );
		FILE* pFile;
		fopen_s( &pFile, filename, "w+" );
		if ( pFile )
		{
			OutputJobStatusReport( pFile, sReportTitle.c_str() );
			fclose( pFile );
			bReportWritten = true;
		}
		else
		{
			s_globalLog.LogError( "Could not open '%s' to write status report.\n", filename );
		}
	}

	if ( bReportWritten )
	{
		MailStatusReport( filename, sReportTitle.c_str(), config );
	}

	// Close the telnet server if it is running
	if ( config.bTelnet )
	{
		server.StopListenerThread();
	}

	// Finish up
	if ( error )
	{
		if ( config.bPauseOnError )
		{
			fprintf( stderr, "Press any key");
			_getch();
		}
		return 1;
	}
	else
	{
		printf( "Finished." );
		return 0;
	}
}
