////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2010.
// -------------------------------------------------------------------------
//  File name:   LoadTestJob.cpp
//  Version:     v1.00
//  Created:     26/01/2010 by Steve Barnett.
//  Description: LoadTest a single level from an SJob
////////////////////////////////////////////////////////////////////////////

// Headers
#include "LoadTestOptions.h"

#include <assert.h>
#include <string.h>
#include <conio.h>
#include <time.h>
#include <windows.h>
#include <iostream>
#include <string>

#include "LoadTestJob.h"

#include "LoadTestOptions.h"

#include "IConsoleInterface.h"
#include "ConsolePS3.h"
#include "ConsoleXBox.h"

#include "Common/Config.h"

using namespace std;

// Utility functions
void GeneratePipeName( SConfig& config )
{
	// Get time in seconds since epoch
	time_t seconds;
	seconds = time( NULL ) & 0xffffffff;

	// Encode as hex into a string - backwards but all that matters is it is roughly unique
	char* enc = "0123456789abcdef";
	char buf[2] = { ' ', '\0' };
	config.sPipeName = "";
	while ( seconds != 0 )
	{
		buf[0] = enc[seconds & 0x0f];
		config.sPipeName.append( buf );
		seconds /= 16;
	}
}

void sleep(unsigned int mseconds)
{
  clock_t goal = mseconds + clock();
	while ( goal > clock() );
}

static bool WaitForKeypressIfNecessary( bool bWait)
{
	if ( bWait )
	{
		printf( "Press a key to continue...\n" );
		while ( !_kbhit() );
	}
	return true;
}


// LoadTestJob implementation
CLoadTestJob::CLoadTestJob( SJob* pJob )
: m_pTarget( NULL )
, m_pJob( pJob )
{
	assert( m_pJob );
}

CLoadTestJob::~CLoadTestJob( void )
{

}

bool CLoadTestJob::RunJob( CLoadTestJob* pJob )
{
	if ( pJob )
	{
		return pJob->Run();
	}
	return false;
}

SConfig& CLoadTestJob::GetConfig( void )
{
	return *(m_pJob->m_pConfig);
}

bool CLoadTestJob::RunPostprocessTask( const char* const pOutFile, const char* const pExecutable )
{
	SConfig& config = GetConfig();
	if ( config.sPostprocessTool.empty() ) return false;

	string sCmd = "\"" + config.sPostprocessTool + "\" \"" + pOutFile + "\" \"" + pExecutable + "\"";
	cout << "[Auto] Running post process task '" << sCmd << "'" << endl;

	STARTUPINFOA startupInfo;
	PROCESS_INFORMATION processInformation;

	ZeroMemory( &startupInfo, sizeof(startupInfo) );
	startupInfo.cb = sizeof(startupInfo);
	ZeroMemory( &processInformation, sizeof(processInformation) );

	//+2 makes the buffer slightly larger than the command line, leaving room for a trailing null
	const int iBufferSize = sCmd.length() + 2;
	char* args = new char[iBufferSize];
	strncpy_s(args, iBufferSize, sCmd.c_str(), sCmd.length());
	CreateProcessA(config.sPostprocessTool.c_str(),
		args, NULL, NULL, FALSE, 0, NULL, NULL, &startupInfo, &processInformation);

	// Wait until child process exits.
	WaitForSingleObject( processInformation.hProcess, INFINITE );
	DWORD exitCode = -1;
	GetExitCodeProcess(processInformation.hProcess, &exitCode);

	// Close process and thread handles. 
	CloseHandle( processInformation.hProcess );
	CloseHandle( processInformation.hThread );

	delete[] args;

	return exitCode == 0;
}

// Return true if an error occured or the kill flag is set
#define IsOkay() (!m_pJob->m_kill && (status.m_logger.m_errors == 0) )

bool CLoadTestJob::Run( void )
{
	char* pCur = m_pManualBuffer;
	char* pDumpFilename = NULL;
	SConfig& config = GetConfig();
	const char* const pMapName = config.mapList.GetCurrentMap();
	bool bGameLoaded = false;
	SLog status;

	status.m_logfile = "<unable to gather>";

	// Has target name been set?
	const char* pTargetName = NULL;
	if ( !config.sTargetName.empty() )
	{
		pTargetName = config.sTargetName.c_str();
	}

	GeneratePipeName( config );

	if ( config.ePlatform == SConfig::ePL_PS3 )
	{
		m_pTarget = new CConsolePS3( config.sExecutable.c_str(), config.sArguments.c_str(), pTargetName, config.sPipeName.c_str(), &status.m_logger, config.bVerbose, config.bOutputTTY );
	}
	else if ( config.ePlatform == SConfig::ePL_XBOX360 )
	{
		m_pTarget = new CConsoleXBox( config.sExecutable.c_str(), config.sArguments.c_str(), pTargetName, config.sPipeName.c_str(), &status.m_logger, config.bVerbose, config.bOutputTTY );
	}

	if ( !m_pTarget )
	{
		status.m_logger.LogError( "Unable to create console interface." );
	}

	if ( IsOkay() && !m_pTarget->Connect() )
	{
		status.m_logger.LogError( "Unable to connect to console." );
	}

	// Automated map loading
	if ( IsOkay() && config.mapList.GetNumMaps() == 0) 
	{
		status.m_logger.LogError( "No maps to process." );
	}

	if ( IsOkay() )
	{
		printf( "[Auto] Resetting target\n" );
		std::vector<std::string> cmdLineParams;
		printf( "[Auto] Loading map '%s'\n", pMapName );
		cmdLineParams.push_back("+map");
		cmdLineParams.push_back(pMapName);

		if (config.eLoggingType == config.eLT_PERF_STATS)
		{
			char perfStatsPeriodStr[8];
			sprintf_s(perfStatsPeriodStr, sizeof(perfStatsPeriodStr), "%d", config.iPerfStatsPeriod);
			cmdLineParams.push_back("+e_PerfStatsDumpPeriod");
			cmdLineParams.push_back(perfStatsPeriodStr);
		}
		else if ( config.eLoggingType == config.eLT_MEM_REPLAY )
		{
			cmdLineParams.push_back( "-memReplay" );
		}
	
		// For performance stats we should use a soft reset to preserve the shader cache
		if ( IsOkay() && !m_pTarget->Reset( cmdLineParams, config.eLoggingType == config.eLT_MEM_REPLAY ) )
		{
			status.m_logger.LogError( "Unable to reset console." );
		}
	
		if ( IsOkay() && WaitForKeypressIfNecessary( config.bPauseOnConnect ) && !m_pTarget->CreatePipe() )
		{
			status.m_logger.LogError( "Unable to create pipe." );
		}

	// Perform any work required to maintain connection to target
		m_pTarget->UpdateComms();
	}

	// Wait for "Connected" message
	if ( IsOkay() && !m_pTarget->WaitForConnected() )
	{
		status.m_logger.LogError( "Target not responding." );
		m_pTarget->ClosePipe();
	}

	if ( config.eLoggingType == SConfig::eLT_MEM_REPLAY )
	{
		if ( IsOkay() )
		{
			// Enable memory logging
			printf( "[Auto] Enabling memory logging\n" );
			m_pTarget->SendToPipe( "exec memDumpAllocs" );
			if ( !m_pTarget->WaitForFinished() )
			{
				status.m_logger.LogError( "Sending command 'exec memDumpAllocs' timed out." );
			}
		}

		if ( IsOkay() )
		{
			// Get the dump filename
			printf( "[Auto] Requesting dump filename\n" );
			m_pTarget->SendToPipe( "getdumpfilename" );
			const char* pTemp = m_pTarget->GetMatchingResponse( IConsoleInterface::eRT_DumpFilename );
			if ( pTemp && !strncmp( pTemp, "dumpfile ", 9 ) )
			{
				pTemp += 9;
			}
			else
			{
				status.m_logger.LogError( "Getting dump file name (Response: '%s').", pTemp );
				m_pTarget->ClosePipe();
			}
	
			if ( IsOkay() )
			{
				// Copy it into a new buffer as what we currently have is a pointer to the pipe read buffer
				const int len = strlen( pTemp );
				pDumpFilename = new char[len + 1];
				strncpy_s( pDumpFilename, len + 1, pTemp, len );
			}
		}
	}

	// If we got this far without errors the game has loaded and there should be something worth gathering
	if ( IsOkay() )
	{
		bGameLoaded = true;
	}

	// Null cmd to ensure the map has loaded
	if ( IsOkay() )
	{
		m_pTarget->SendToPipe( "exec  " );
		if ( !m_pTarget->WaitForFinished() )
		{
			status.m_logger.LogError( "Console stopped resonding during load." );
		}
	}

	// Map is loaded - now we should wait for a short while
	// Based on the PollCVARType we either wait for a set time or until a CVAR is set
	if ( config.ePollCVARAction == SConfig::ePCA_FINISH )
	{
		printf( "[Auto] Waiting for condition CVAR(\"%s\") == %i\n", config.sPollCVARName.c_str(), config.iPollCVARValue );
		printf( "[Auto] Received values...");
		while ( IsOkay() )
		{
			// Get the CVAR value
			std::string sCommand = "getcvarvalue " + config.sPollCVARName;
			m_pTarget->SendToPipe( sCommand.c_str() );
			const char* pTemp = m_pTarget->GetMatchingResponse( IConsoleInterface::eRT_CVARValue );
			if ( pTemp && !strncmp( pTemp, "cvarval ", 8 ) )
			{
				pTemp += 8;
			}
			else
			{
				status.m_logger.LogError( "Getting CVAR value for %s (Response: '%s').", config.sPollCVARName.c_str(), pTemp );
			}

			if ( IsOkay() )
			{
				// Test the returned value
				const int iValue = atoi( pTemp );
				printf( "%i ", iValue );
				if ( iValue == config.iPollCVARValue )
				{
					printf( "\n[Auto] Matched.\n", iValue );
					break;
				}
			}

			// Pause to avoid spamming the console - more of an issue on the 360 where responses are very fast
			sleep( 1000 );
		}
	}
	else
	{
		printf( "[Auto] Waiting..." );
		for ( int i = 0; i < config.iDelay; i++ )
		{
			if ( !IsOkay() ) break;
			sleep( 1000 );
			printf( "%i ", i + 1 );
		}
		printf( "\n" );
	}

	// Gather the log file as long as the game loaded - try it regardless of error any logfile we can salvage is worthwhile
	if ( bGameLoaded )
	{
		if ( config.eLoggingType == SConfig::eLT_MEM_REPLAY && pDumpFilename )
		{
			// Close and gather dump file
			printf( "[Auto] Requesting memory dump\n" );
			m_pTarget->SendToPipe( "exec memReplayStop" );
	
			// Wait for notification that the command has finished
			m_pTarget->WaitForFinished();
	
			// Gather dump files and run post processing if necessary
			printf( "[Auto] Gathering output file\n" );
			if ( !m_pTarget->GatherDumpFile( config.mapList.GetCurrentMap(), pDumpFilename, config.sMemoryStatsDir.c_str() ) )
			{
				status.m_logger.LogError( "Failed to gather memReplay log." );
				m_pTarget->ClosePipe();
			}
			else
			{
				// Store the log filename for the status report
				// Do not store filename if the gather failed
				status.m_logfile = pDumpFilename;
			}
	
			// Gather the game log
			printf( "[Auto] Gathering game.log\n" );
			if ( !m_pTarget->GatherDumpFile( config.mapList.GetCurrentMap(), "game.log", config.sMemoryStatsDir.c_str() ) )
			{
				status.m_logger.LogError( "Failed to gather game.log." );
				m_pTarget->ClosePipe();
			}
	
			// Give game.log a time stamp to match the logfile
	 		string sGameLogName( pDumpFilename );
			string::size_type truncLen = sGameLogName.find_first_of( '.' );
			if ( truncLen != string::npos )
			{
				sGameLogName.resize( truncLen );
				sGameLogName.append( "_game.log" );
				string cmdStr = "copy \"";
				cmdStr += config.sMemoryStatsDir;
				cmdStr += "\\";
				cmdStr += config.mapList.GetCurrentMap();
				cmdStr += "\\game.log";
				cmdStr += "\" \"";
				cmdStr += config.sMemoryStatsDir;
				cmdStr += "\\";
				cmdStr += config.mapList.GetCurrentMap();
				cmdStr += "\\";
				cmdStr += sGameLogName.c_str();
				cmdStr += "\"";
				printf( "[Auto] running command: %s\n", cmdStr.c_str() );
				system(cmdStr.c_str());	// need to replace this with a platform version
			}
	
			if ( !config.sPostprocessTool.empty() )
			{
				const char* const pMapName = config.mapList.GetCurrentMap();
				string sOutFile = config.sMemoryStatsDir + pMapName + "/" + string(pDumpFilename);
				printf( "[Auto] Executing postprocess task\n" );
				if ( !config.sPostprocessTool.empty() && !RunPostprocessTask( sOutFile.c_str(), config.sExecutable.c_str() ) )
				{
					status.m_logger.LogWarning( "Failed to spawn postprocess task '%s %s %s'.", config.sPostprocessTool.c_str(), sOutFile.c_str(), config.sExecutable.c_str() );
				}
			}
			else
			{
				status.m_logger.LogWarning( "No postprocess task." );
			}
		}
	
		if ( config.eLoggingType == SConfig::eLT_PERF_STATS )
		{
			// First move the logfile to the output directory in a target independent way
			printf( "[Auto] Gathering output file\n" );
			string sourceFile( "perflog_" );
			sourceFile += pMapName;
			sourceFile += ".log";
			if ( !m_pTarget->GatherDumpFile( "", sourceFile.c_str(), config.sPerfStatsDir.c_str() ) )
			{
				status.m_logger.LogError( "Failed to copy logfile to output directory." );
			}
	
			// Generate a timestamp
			time_t rawtime = time( NULL );
			tm ts;
			localtime_s( &ts, &rawtime );
			char timestamp[32];
			sprintf_s(timestamp, sizeof(timestamp), "%04i%02i%02i-%02i%02i%02i", 1900 + ts.tm_year, 1 + ts.tm_mon, ts.tm_mday, ts.tm_hour, ts.tm_min, ts.tm_sec );
	
			// Generate the log filename
			string sLogFilename = "perflog_";
			sLogFilename += pMapName;
			sLogFilename += "_";
			sLogFilename += timestamp;
			sLogFilename += ".log";
	
			// Now rename the logfile 
			string cmdStr = "copy \"";
			cmdStr += config.sPerfStatsDir;
			cmdStr += "\\";
			cmdStr += sourceFile;
			cmdStr += "\" \"";
			cmdStr += config.sPerfStatsDir;
			cmdStr += "\\";
			cmdStr += sLogFilename;
			cmdStr += "\"";
			printf( "[Auto] running command: %s\n", cmdStr.c_str() );
			system(cmdStr.c_str());
	
			// Store the log filename for the status report
			status.m_logfile = sLogFilename;
	
			// Gather the game log
			printf( "[Auto] Gathering game.log\n" );
			if ( !m_pTarget->GatherDumpFile( "", "game.log", config.sPerfStatsDir.c_str() ) )
			{
				status.m_logger.LogError( "Failed to gather game log." );
				m_pTarget->ClosePipe();
			}
			// Give game.log a new name to match the logfile
			cmdStr = "copy \"";
			cmdStr += config.sPerfStatsDir;
			cmdStr += "\\game.log";
			cmdStr += "\" \"";
			cmdStr += config.sPerfStatsDir;
			cmdStr += "\\gamelog_";
			cmdStr += pMapName;
			cmdStr += "_";
			cmdStr += timestamp;
			cmdStr += ".log\"";
			printf( "[Auto] running command: %s\n", cmdStr.c_str() );
			system(cmdStr.c_str());
		}
	}
	else
	{
		status.m_logger.LogError( "No log could be salvaged as the game failed to load." );
	}

	// Close pipe
	if ( IsOkay() )
	{
		m_pTarget->ClosePipe();
	}

	delete[] pDumpFilename;
	pDumpFilename = NULL;

	if ( m_pTarget ) m_pTarget->Shutdown();
	delete m_pTarget;
	m_pTarget = NULL;

	// Output the job status report
	m_pJob->m_logs.push_back( status );

	// Mark task as complete, failed or neither (in which case we bump the number of errors)
	const int error = status.m_logger.m_errors;
	if ( !m_pJob->m_kill )
	{
		if ( error == 0 )
		{
			m_pJob->m_complete = true;
		}
		else
		{
			++m_pJob->m_errors;
			if ( config.iMaxErrors > 0 && m_pJob->m_errors >= config.iMaxErrors )
			{
				m_pJob->m_failed = true;
			}
		}
	}

	// Return true if no errors and not killed
	return (error == 0) && !m_pJob->m_kill;
}
