////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2010.
// -------------------------------------------------------------------------
//  File name:   TelnetServer.cpp
//  Version:     v1.00
//  Created:     12/03/2010 by Steve Barnett.
//  Description: Telnet server for remotely managing a LoadTest instance
////////////////////////////////////////////////////////////////////////////

#include "TelnetServer.h"

#include "LoadTestOptions.h"
#include "Common/Config.h"

#include <process.h>

#pragma comment(lib, "wininet.lib")

using namespace std;

static void FixLineBreaks( string& sStr )
{
	const int length = sStr.length();
	string buffer = "";
	char buf[2] = " ";
	for ( unsigned int i = 0; i < sStr.length(); i++ )
	{
		if ( sStr[i] == '\n' )
		{
			buffer += "\r\n";
		}
		else
		{
			buf[0] = sStr[i];
			buffer += buf;
		}
	}
	sStr = buffer;
}

void OutputJobStatusReport( string& sStr, const char* const pTitle = NULL );

CTelnetServer::~CTelnetServer( void )
{
	if ( m_isListener )
	{
		if ( m_hExit )
		{
			CloseHandle( m_hExit );
			m_hExit = NULL;
		}

		// Listener is the last one to finish so cleanup after WinSock
		if ( m_pSock )
		{
			closesocket( *m_pSock );
		}
		WSACleanup();
	}

	m_pJobs = NULL;
	delete m_pSock;	// do we need to close it?
	m_pSock = NULL;
}

void CTelnetServer::StartListenerThread( unsigned int port )
{
	m_port = port;
	m_hExit = CreateEvent( NULL, 0, 0, NULL );

	_beginthread( ListenerThread, 0, this );
}

void CTelnetServer::StopListenerThread( void )
{
	if ( m_isListener )
	{
		// Signal the listener thread to exit
		SetEvent( m_hExit );
	}
}

int CTelnetServer::ListenerThread_Impl( void )
{
	WORD wVersionRequested = MAKEWORD( 1, 1 );
	WSADATA wsaData;
	int err = WSAStartup( wVersionRequested, &wsaData );
	if ( err != 0 )
	{
		fprintf( stderr, "Failed to initialise winsock.\n" );
		return 0;
	}

	if ( m_isListener )
	{
		// Open the socket
		m_pSock = NULL;
		u_long addr = inet_addr( "127.0.0.1" );
		if ( addr != INADDR_NONE )
		{
			SOCKET sock = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
			if ( sock != INVALID_SOCKET )
			{
				sockaddr_in sinIf;
				sinIf.sin_family = AF_INET;
				sinIf.sin_addr.s_addr = addr;
				sinIf.sin_port = htons( m_port );
				if ( bind( sock, (sockaddr*)&sinIf, sizeof( sockaddr_in ) ) != SOCKET_ERROR )
				{
					listen( sock, SOMAXCONN );
					m_pSock = new SOCKET;
					*m_pSock = sock;
				}
			}
		}

		if ( !m_pSock )
		{
			// Print a proper error
			fprintf( stderr, "Could not create listener socket.\n" );
		}
		else
		{
			bool exitSignalled = false;
			while ( !exitSignalled )
			{
				// Print a proper log message
				printf( "Listener restarting\n" );

				sockaddr_in sinRem;
				int addrsize = sizeof( sinRem );

				while ( !exitSignalled )
				{
					// Check if this thread has been signalled to exit
					if ( WaitForSingleObject( m_hExit, 0 ) == WAIT_OBJECT_0 )
					{
						exitSignalled = true;
					}

					SOCKET sd = accept( *m_pSock, (sockaddr*)&sinRem, &addrsize );
					if ( sd != INVALID_SOCKET )
					{
						printf( "Accepted connection from %s:%hd\n", inet_ntoa( sinRem.sin_addr ), ntohs( sinRem.sin_port ) );

						// Create a new instance of this class and run it as a server on a new thread
						CTelnetServer* pSvr = new CTelnetServer( m_pJobs, new SOCKET( sd ) );
						pSvr->StartServerThread();
					}
					else
					{
						// Print a proper error
						fprintf( stderr, "Failed to accept connection\n" );
					}
				}
			}
		}
	}

	printf( "Listener thread exiting.\n" );

	return 0;
}

void CTelnetServer::StartServerThread( void )
{
	_beginthread( ServerThread, 0, this );
}

bool CTelnetServer::ProcessCommands( void )
{
	if ( m_pSock )
	{
		char buf[BUFFER_SIZE];
		int read;
		int unconsumed = 0;

		// Send an initial status report
		SendStatusReport();

		SendData( "Type 'help' for usage information...\r\n" );
	
		// Then begin processing commands
		do
		{
			read = recv( *m_pSock, buf + unconsumed, BUFFER_SIZE - unconsumed, 0 );
			if ( read > 0 )
			{
				int consumed;
				//printf( "Received %d bytes from client.\n", read );
				buf[read + unconsumed] = '\0';
				switch ( ProcessData( buf, consumed ) )
				{
					case ePRValid: break;
					case ePRError: return false;
					case ePRDisconnect: return true;
					default: break;
				}
				unconsumed = ( read + unconsumed ) - consumed;
				for ( int i = 0; i < unconsumed; i++ )
				{
					// Always copying backwards in the buffer
					buf[i] = buf[i + consumed];
				}
			}
			else if ( read == SOCKET_ERROR )
			{
				return false;
			}
		} while ( read != 0 );

		printf( "Connection closed by client.\n" );
		return true;
	}
	fprintf( stderr, "Could not create socket for server thread.\n" );
	return true;
}

CTelnetServer::EProcessResponse CTelnetServer::ProcessData( const char* const pData, int& consumed )
{
	// Look for control characters
	const int len = strlen( pData );
	const char* pCommandStart = pData;
	for ( int i = 0; i < len; ++i )
	{
		if ( pData[i] == '\n' )	// End of a command
		{
			switch ( ProcessCommand( pCommandStart ) )
			{
				case ePRValid: break;
				case ePRError: return ePRError;
				case ePRDisconnect: return ePRDisconnect;
				default: break;
			}
			pCommandStart = &pData[i + 1];
		}
		else if ( pData[i] == 0x04 )	// End of transmission
		{
			return ePRDisconnect;
		}
	}

	// Allow the caller to track any unconsumed data at the end of the buffer
	consumed = pCommandStart - pData;

	return ePRValid;
}

bool CTelnetServer::SendData( const char* pData )
{
	int len = strlen( pData );
	int sent = 0;
	while (sent < len )
	{
		int temp = send( *m_pSock, pData + sent, len - sent, 0 );
		if ( temp > 0 )
		{
			//printf( "Send %d bytes to client.\n", temp );
			sent += temp;
		}
		else if ( temp == SOCKET_ERROR )
		{
			return false;
		}
		else
		{
			fprintf( stderr, "Client unexpectedly closed connection.\n" );
			return true;
		}
	}
	return true;
}

int CTelnetServer::ServerThread_Impl( void )
{
	if ( m_pSock )
	{
		int ret = 0;
		if ( !ProcessCommands() )
		{
			fprintf( stderr, "Failed to create server thread.\n" );
			ret = 3;
		}

		printf( "Closing connection.\n" );
		if ( shutdown( *m_pSock, 0/*SD_SEND*/ ) == SOCKET_ERROR )
		{
			ret = 4;
		}
		else
		{
			// Read the remaining data from the socket
			char buf[BUFFER_SIZE];
			while ( 1)
			{
				int bytes = recv( *m_pSock, buf, BUFFER_SIZE, 0 );
				if ( bytes == SOCKET_ERROR )
				{
					ret = 4;
					break;
				}
				else if ( bytes != 0 )
				{
					fprintf( stderr, "Unexpected bytes at shutdown.\n" );
				}
				else
				{
					break;
				}
		
				if ( closesocket( *m_pSock ) == SOCKET_ERROR )
				{
					delete[] m_pSock;
					m_pSock = NULL;
					return ret;
				}
			}
		}

		delete[] m_pSock;
		m_pSock = NULL;

		return ret;
	}

	return 0;
}

CTelnetServer::EProcessResponse CTelnetServer::ProcessCommand( const char* const pCommand )
{
	if ( !strncmp( pCommand, "disconnect", 10 ) )
	{
		return ePRDisconnect;
	}
	else if ( !strncmp( pCommand, "cancelall", 9 ) )
	{
		CancelAllJobs();
		return ePRValid;
	}
	else if ( !strncmp( pCommand, "cancelrunning", 9 ) )
	{
		CancelCurrentJobs();
		return ePRValid;
	}
	else if ( !strncmp( pCommand, "cancel ", 7 ) )
	{
		// Extract the map name
		char pBuf[BUFFER_SIZE];
		strncpy_s( pBuf, BUFFER_SIZE, pCommand + 7, BUFFER_SIZE - 1 );
		char* pEnd = min( strchr( pBuf, '\n' ), strchr( pBuf, '\r' ) );
		if ( pEnd )
		{
			*pEnd = '\0';
		}
		CancelJobByMapName( pBuf );
		return ePRValid;
	}
	else if ( !strncmp( pCommand, "status", 6 ) )
	{
		SendStatusReport();
		return ePRValid;
	}
	else if ( !strncmp( pCommand, "help", 4 ) )
	{
		SendHelpMessage();
		return ePRValid;
	}

	char pBuffer[BUFFER_SIZE];
	sprintf_s( pBuffer, BUFFER_SIZE, "Unexpected command '%s'\r\n", pCommand );
	SendData( pBuffer );
	return ePRValid;
}

void CTelnetServer::CancelAllJobs( void )
{
	for ( JobList::iterator jobIt = m_pJobs->begin(), jobEnd = m_pJobs->end(); jobIt != jobEnd; ++jobIt )
	{
		SJob& jobDesc = *jobIt;
		SLog log;
		log.m_logfile = "<job cancelled by remote user>";
		if ( jobDesc.m_running )
		{
			jobDesc.m_kill = true;
			jobDesc.m_failed = true;
			// Can't write to this job's log for thread safety but it will log it's own message
		}
		else
		{
			jobDesc.m_failed = true;
			jobDesc.m_logs.push_back( log );
		}
	}
	SendStatusReport();
}

void CTelnetServer::CancelCurrentJobs( void )
{
	for ( JobList::iterator jobIt = m_pJobs->begin(), jobEnd = m_pJobs->end(); jobIt != jobEnd; ++jobIt )
	{
		SJob& jobDesc = *jobIt;
		SLog log;
		log.m_logfile = "<job cancelled by remote user>";
		if ( jobDesc.m_running )
		{
			jobDesc.m_kill = true;
			jobDesc.m_failed = true;
			// Can't write to this job's log for thread safety but it will log it's own message
		}
	}
	SendStatusReport();
}

void CTelnetServer::CancelJobByMapName( const char* const pMap )
{
	for ( JobList::iterator jobIt = m_pJobs->begin(), jobEnd = m_pJobs->end(); jobIt != jobEnd; ++jobIt )
	{
		SJob& jobDesc = *jobIt;
		SLog log;
		log.m_logfile = "<job cancelled by remote user>";
		if ( jobDesc.m_pConfig )
		{
			const char* const pMapName = jobDesc.m_pConfig->mapList.GetMapNames()[0].c_str();
			if ( !strcmp( pMap, pMapName ) )
			{
				if ( jobDesc.m_running )
				{
					jobDesc.m_kill = true;
					jobDesc.m_failed = true;
					// Can't write to this job's log for thread safety but it will log it's own message
				}
				else
				{
					jobDesc.m_failed = true;
					jobDesc.m_logs.push_back( log );
				}
			}
		}
	}
	SendStatusReport();
}

void CTelnetServer::SendStatusReport( void )
{
	std::string sReport;
	OutputJobStatusReport( sReport );
	FixLineBreaks( sReport );
	SendData( sReport.c_str() );
}

void CTelnetServer::SendHelpMessage( void )
{
	SendData( "LoadTest Telnet Commands: \r\n" );
	SendData( "\thelp - Display this message.\r\n" );
	SendData( "\tstatus - Display a job status report.\r\n" );
	SendData( "\tcancelall - Cancel all jobs.\r\n" );
	SendData( "\tcancelrunning - Cancel all running jobs.\r\n" );
	SendData( "\tcancel <map> - Cancel all jobs for a specific map.\r\n" );
	SendData( "\tdisconnect - Close the telnet session.\r\n" );
	SendData( "\r\n" );
}

void __cdecl CTelnetServer::ListenerThread( void* pInstVoid )
{
	if ( pInstVoid )
	{
		CTelnetServer* const pInstance = reinterpret_cast<CTelnetServer*>( pInstVoid );

		// Run the listener thread instance, delete it when it's done
		int ret = pInstance->ListenerThread_Impl();
		delete pInstance;
	}
	return;
}

void __cdecl CTelnetServer::ServerThread( void* pInstVoid )
{
	if ( pInstVoid )
	{
		CTelnetServer* const pInstance = reinterpret_cast<CTelnetServer*>( pInstVoid );

		// Run the worker thread instance, delete it when it's done
		int ret = pInstance->ServerThread_Impl();
		delete pInstance;
	}
	return;
}
