////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   ReportGen.cpp
//  Version:     v1.00
//  Created:     28/09/2009 by Steve Barnett.
//  Description: Merge StatsTool output into a useful report
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include <windows.h>
#include <vector>
#include <utility>
#include <algorithm>
#include <stdlib.h>
#include "tinyxml.h"

#include <fstream>
#include <iomanip>
#include <iostream>
#include <map>

#include "Common/Config.h"

using namespace std;

#define SUPPORT_CONFLUENCE																							(1)

#if defined( SUPPORT_CONFLUENCE )
#include "ConfluenceWrapper/ConfluenceStubs.h"
#endif

#define EXCEL_WILDCARD	"*.zmrl.xml"
#define MAINMEM_EXTENSION1	"_out.png"
#define MAINMEM_EXTENSION2	"_out.svg"
#define DIRECTORY_WILDCARD	"*"
#define ASSET_OUTFILE		"assetview.xml"
#define PROGRESS_OUTFILE		"ProgressView.csv"
#define PROGRESS_GRAPH_TEMPFILE		"temp.gnuplot"
#define PROGRESS_GRAPH_OUTFILE		"out.svg"
#define RESOURCE_OUTFILE	"ResourceView"
#define RESOURCE_EXTENSION ".csv"
#define RESOURCE_GRAPH_OUTFILE	"resource.svg"
#define ONE_MEG		( 1024.0f * 1024.0f )

enum EReportMode
{
	eRM_None,
	eRM_AssetView,
	eRM_ProgressView,
	eRM_ResourceView,
	eRM_Submit,
	eRM_All,
};

enum EGraphType
{
	eGT_MemoryTypes,
	eGT_LevelTotals,
	eGT_ResourceByLevel,
	eGT_ResourceByType,
};

enum EGraphFormat
{
	eGF_SVG,
	eGF_PNG,

	eGF_NUM_FORMATS,
};

const char* szGraphOutputFiles[eGF_NUM_FORMATS] = { "out.svg", "out.png" };

struct SSummary
{
	SYSTEMTIME m_date;

	typedef map<string, int> TMemorySummary;

	TMemorySummary m_memory;
};

struct SAsset
{
	char* m_pName;
	u_int m_memory;
	u_int m_memoryRSX;
	vector<const char*> m_levels;
};

struct SAssetType
{
	SAssetType( const char* const pName, const u_int columns ) : m_pName( pName ), m_columns( columns ) {};
	const char* m_pName;
	u_int m_columns;
	vector<SAsset> m_data;
};

// Second param is number of columns up to the main memory column
SAssetType pAssetTypes[] =
{
	SAssetType( "CGF Physics", 3 ),
	SAssetType( "Textures", 2 ),
	SAssetType( "Render Mesh", 3 ),
	SAssetType( "Animation", 2 ),
};
const u_int numAssetTypes = ( sizeof( pAssetTypes ) / sizeof( SAssetType ) );

static SConfig config;

typedef vector<string> TStrings;
const TStrings& GetMemoryTypesWithoutRSX( bool bIgnoreTotal = false, bool bIgnoreFree = false, bool bIgnoreOther = false );
const TStrings& GetMemoryTypesWithRSX( bool bIgnoreTotal = false, bool bIgnoreFree = false, bool bIgnoreOther = false );
string GetRSXMemoryName(const string& memoryType);

bool CopyFileToReportDir( const string& sFilePath );

bool GenerateProgressView();
bool GenerateProgressViewForLevel(const string& sDirectory);
bool GenerateAssetView();
bool GenerateResourceView();
bool ProcessSubmit();

bool GenerateSummary( const char* const pFilename, SYSTEMTIME& date, SSummary& summary );
bool ProcessSummaryRow( TiXmlNode* row, SSummary& summary );
bool WriteProgressView( const string& sFilename, vector<SSummary>& list );
bool CreateGraph( const string& sDirectory, const char* const pTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat );
bool CreateBarCharts( const string& sDirectory, const char* const pTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat );
void PrintSummary( SSummary& summary );

bool FindNewestLogfile( const string& sDirectory, string& sOutput );
bool ProcessAssetSpreadsheet( const char* const pFilename, const char* const pMap );
bool ProcessAssetTypeWorksheet( SAssetType& assetType, TiXmlNode* sheet, const char* const pMap );
bool ProcessAssetTypeRow( SAssetType& assetType, TiXmlNode* row, const char* const pMap );
bool StoreAsset( SAssetType& assetType, const char* const pName, const unsigned int mem, const unsigned int rsx, const char* const pMap );
bool WriteAssetViewSpreadsheet( const string& sFilename );
bool WriteAssetViewWorksheet( TiXmlNode* root, unsigned int assetType );
bool WriteAssetViewRow( TiXmlNode* table, SAsset& row );

bool CreateGnuplotInputFile(const string& sDirectory, const string& sTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat );
bool CreateGnuplotInputFileForBarChart(const string& sDirectory, const string& sTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat, int iChartNum );
bool AddGraphsToSubmit( const char* const pExtension, TStrings& files );
bool WriteResourceView( const string sFilename, const string sFilenameExtension, vector< pair<string, SSummary> >& summaries );
bool WriteResourceViewSpreadsheet( const string sFilename, vector< pair<string, SSummary> >& summaries, bool bFlip );

#if defined( SUPPORT_CONFLUENCE )
static const int s_AUTHTOKEN_MAX_LEN = 255;
ConfluenceStubs g_Conf;
HMODULE g_hConfluence = 0;
char g_authToken[s_AUTHTOKEN_MAX_LEN];

bool InitialiseConfluence( void );
bool ShutdownConfluence( void );

bool ConfluenceLogin( string& username, string& password );
void ConfluenceLogout( void );
bool ConfluenceGetPage( string& space, string& page, ConfluencePage& remotePage );
void ConfluenceFreePage( ConfluencePage& remotePage );
bool ConfluenceAttachFile( string& filename, string& shortFilename, string& title, string& mimeType, ConfluencePage& page );
#endif

bool IsString( TiXmlElement* elem );
bool IsNumber( TiXmlElement* elem );
bool ProcessTimeStamp( const char* const pFilename, SYSTEMTIME& date );

void Usage( void )
{
	printf( "ReportGen.exe:\tGenerate useful reports from memReplay log spreadsheets\n" );
	printf( "Usage:\n" );
	printf( "\tReportGen <config file> assets\n" );
	printf( "Analyse latest logfiles for all levels and produce overall memory\n");
	printf( "and usage statistics for each asset across the game.\n" );
	printf( "\n" );
	printf( "\tReportGen <config file> progress\n" );
	printf( "Analyse logfiles for a single level and graph usage totals from\n" );
	printf( "overview sheet.\n" );
	printf( "\n" );
	printf( "\tReportGen <config file> resource\n" );
	printf( "Analyse logfiles for all levels and produce memory statistics by\n" );
	printf( "resource type.\n" );
	printf( "\n" );
	printf( "\tReportGen <config file> submit\n" );
	printf( "Upload graphs generated from progress view to confluence. Login\n" );
	printf( "details can be entered manually or piped in from a file.\n" );
	printf( "\n" );
	printf( "\tReportGen <config file> all\n" );
	printf( "Run assets, then progress, then resource, then submit\n" );
	printf( "\n" );

}

int main(int argc, char* argv[])
{
	EReportMode eMode = eRM_None;

	// This app will have several operating modes:
	//	Progress View:	Graph overview of progress over time
	//	Asset View:			Summarise asset size and usage over all maps
	//	Resource View:	Summaries memory usage by resource type.
	//	Submit:					Upload the generated graphs (and reports?) to confluence
	//  Full:						Run assets, then progress, then extra graph processing, then Sumbit
	if ( argc != 3 )
	{
		Usage();
		return 0;
	}
	else
	{
		// First arg is config
		if ( !config.ParseConfigFile( argv[1] ) )
		{
			Usage();
			return  0;
		}

		// Second arg is mode
		if ( !strcmp( argv[2], "assets" ) )
		{
			eMode = eRM_AssetView;
		}
		else if ( !strcmp( argv[2], "resource" ) )
		{
			eMode = eRM_ResourceView;
		}
		else if ( !strcmp( argv[2], "progress" ) )
		{
			eMode = eRM_ProgressView;
		}
		else if ( !strcmp( argv[2], "submit" ) )
		{
			eMode = eRM_Submit;
		}
		else if ( !strcmp( argv[2], "all" ) )
		{
			eMode = eRM_All;
		}
		else
		{
			Usage();
			return 0;
		}
	}

	switch ( eMode )
	{
		case eRM_None:
		{
			Usage();
		}
		case eRM_All: // fall through
		case eRM_AssetView:
		{
			if ( !GenerateAssetView() )
			{
				return 1;
			}
		} if(eMode != eRM_All) break; // fall through if we're running all
		case eRM_ResourceView:
		{
			if ( !GenerateResourceView() )
			{
				return 1;
			}
		} if(eMode != eRM_All) break; // fall through if we're running all
		case eRM_ProgressView:
		{
			if ( !GenerateProgressView() )
			{
				return 1;
			}
		} if(eMode != eRM_All) break; // fall through if we're running all
		case eRM_Submit:
		{
			if ( !ProcessSubmit() )
			{
				return 1;
			}
		} break;
	}

	return 0;
}

const TStrings& GetMemoryTypesWithoutRSX( bool bIgnoreTotal /*= false*/, bool bIgnoreFree /*= false*/, bool bIgnoreOther /*= false*/ )
{
	static TStrings memoryTypes;
	static bool bLastIgnoreTotal;
	static bool bLastIgnoreFree;
	static bool bLastIgnoreOther;
	if ( memoryTypes.empty() || bLastIgnoreTotal != bIgnoreTotal || bLastIgnoreFree != bIgnoreFree || bLastIgnoreOther != bIgnoreOther )
	{
		memoryTypes.clear();	// Cache the last version requested
		bLastIgnoreTotal = bIgnoreTotal;
		bLastIgnoreFree = bIgnoreFree;
		bLastIgnoreOther = bIgnoreOther;
		memoryTypes.push_back( "CGF Physics" );
		memoryTypes.push_back( "Textures" );
		memoryTypes.push_back( "Render Mesh" );
		memoryTypes.push_back( "Animation" );
		memoryTypes.push_back( "Navigation" );
		memoryTypes.push_back( "Entity" );
		if ( !bIgnoreOther )
		{
			memoryTypes.push_back( "Other" );
		}
		if ( !bIgnoreTotal )
		{
			memoryTypes.push_back( "Total" );
		}
		if ( !bIgnoreFree )
		{
			memoryTypes.push_back( "Free" );
		}
	}

	return memoryTypes;
}

const TStrings& GetMemoryTypesWithRSX( bool bIgnoreTotal /*= false*/, bool bIgnoreFree /*= false*/, bool bIgnoreOther /*= false*/ )
{
	static TStrings memoryTypesWithRSX;
	static bool bLastIgnoreTotal;
	static bool bLastIgnoreFree;
	static bool bLastIgnoreOther;
	if ( memoryTypesWithRSX.empty() || bLastIgnoreTotal != bIgnoreTotal || bLastIgnoreFree != bIgnoreFree || bLastIgnoreOther != bIgnoreOther )
	{
		bLastIgnoreTotal = bIgnoreTotal;
		bLastIgnoreFree = bIgnoreFree;
		bLastIgnoreOther = bIgnoreOther;

		const TStrings& memoryTypes = GetMemoryTypesWithoutRSX( bIgnoreTotal, bIgnoreFree, bLastIgnoreOther );
		memoryTypesWithRSX = memoryTypes;

		for(TStrings::const_iterator it = memoryTypes.begin(), itEnd = memoryTypes.end(); it != itEnd; ++it)
		{
			memoryTypesWithRSX.push_back( GetRSXMemoryName(*it) );
		}
	}
	
	return memoryTypesWithRSX;
}

string GetRSXMemoryName(const string& memoryType)
{
	return memoryType + " RSX";
}

bool CopyFileToReportDir( const string& sFilePath )
{
	const size_t rootDirPos = sFilePath.find(config.sMemoryStatsDir);
	if ( rootDirPos == string::npos )
	{
		return false;
	}

	string sFileName = sFilePath.substr( rootDirPos + config.sMemoryStatsDir.size() );

	// strip out the directories
	size_t dirPos;
	while ( ( dirPos = sFileName.find('/') ) != string:: npos )
	{
		sFileName.replace(dirPos, 1, "_");
	}

	string sOutputPath = config.sReportOutDir + sFileName;

	return ( CopyFileA( sFilePath.c_str(), sOutputPath.c_str(), false) != 0 );
}

// Operating modes

bool GenerateProgressView()
{
	for ( size_t iMap = 0; iMap < config.mapList.GetNumMaps(); ++iMap)
	{
		string sDirectory = config.sMemoryStatsDir + config.mapList.GetMap(iMap) + string("/");
		if(!GenerateProgressViewForLevel(sDirectory))
		{
			return false;
		}
	}

	CreateGraph(config.sMemoryStatsDir, PROGRESS_GRAPH_TEMPFILE, eGT_LevelTotals, eGF_SVG);
	CreateGraph(config.sMemoryStatsDir, PROGRESS_GRAPH_TEMPFILE, eGT_LevelTotals, eGF_PNG);

	return true;
}

bool GenerateProgressViewForLevel(const string& sDirectory)
{
	// Iterate over the mem replay Excel files in the input directory and gather summary information
	vector<SSummary> list;

	// Add a wildcard to the end of the directory

	string sSearchString = sDirectory + EXCEL_WILDCARD;

	WIN32_FIND_DATAA result;
	HANDLE hSearch = FindFirstFileA( sSearchString.c_str(), &result );
	if ( hSearch != INVALID_HANDLE_VALUE)
	{
		do
		{
			SSummary summary;

			// Get the filename
			string sFileName = sDirectory + result.cFileName;

			// Get the date from the filename
			SYSTEMTIME date;
			ProcessTimeStamp( result.cFileName, date );

			// Generate the summary
			GenerateSummary( sFileName.c_str(), date, summary );
			list.push_back( summary );
		} while ( FindNextFileA( hSearch, &result ) );
		 
	}
	FindClose( hSearch );

	// Sort the list by date
	sort( list.begin(), list.end() );

	// Now we output a CSV of the data for GNUPlot to use
	string sCSVFile = sDirectory + PROGRESS_OUTFILE;

	if ( !WriteProgressView( sCSVFile, list ) ) return false;

	if ( !CreateGraph( sDirectory, PROGRESS_GRAPH_TEMPFILE, eGT_MemoryTypes, eGF_SVG ) ) return false;
	if ( !CreateGraph( sDirectory, PROGRESS_GRAPH_TEMPFILE, eGT_MemoryTypes, eGF_PNG ) ) return false;

	return true;
}

bool GenerateResourceView()
{
	// Iterate over each directory within the working directory
	vector< pair<string, SSummary> > summaries;

	// Now we need to find the appropriate file within each directory
	const size_t numMaps = config.mapList.GetNumMaps();
	for ( size_t u = 0; u < numMaps; u++ )
	{
		const char* const szMap = config.mapList.GetMap(u);
		string sDirectory = config.sMemoryStatsDir + szMap + "/";

		string sLogFile;
		if ( FindNewestLogfile( sDirectory, sLogFile ) )
		{
			printf( "Processing '%s'\n", sLogFile.c_str() );
			SYSTEMTIME date;
			SSummary summary;
			ProcessTimeStamp( sLogFile.c_str(), date );	// Date is not used but useful for debugging
			if ( !GenerateSummary( sLogFile.c_str(), date, summary ) )
			{
				fprintf( stderr, "Could not process '%s'\n", sLogFile.c_str() );
			}
			else
			{
				summaries.push_back( make_pair( szMap, summary ) );
			}
		}
	}

	// Now we need to create the output spreadsheet
	{
		string sOutFile = config.sMemoryStatsDir + RESOURCE_OUTFILE;
		string sOutFileExtension( RESOURCE_EXTENSION );
		if ( !WriteResourceView( sOutFile, sOutFileExtension, summaries ) )
		{
			fprintf( stderr, "Failed to write resource view spreadsheet.\n" );
			return false;
		}
	}

	return true;
}

bool GenerateAssetView()
{
	// Iterate over each directory within the working directory

	// Now we need to find the appropriate file within each directory
	const size_t numMaps = config.mapList.GetNumMaps();
	for ( size_t u = 0; u < numMaps; u++ )
	{
		const char* const szMap = config.mapList.GetMap(u);
		string sDirectory = config.sMemoryStatsDir + szMap + "/";

		string sLogFile;
		if ( FindNewestLogfile( sDirectory, sLogFile ) )
		{
			printf( "Processing '%s'\n", sLogFile.c_str() );
			if ( !ProcessAssetSpreadsheet( sLogFile.c_str(), szMap ) )
			{
				cerr << "Could not process '" << sLogFile << "'" << endl;
				return false;
			}
		}
	}

	// Now we need to create the output spreadsheet
	{
		string sOutFile = config.sMemoryStatsDir + ASSET_OUTFILE;
		if ( !WriteAssetViewSpreadsheet( sOutFile ) )
		{
			cerr << "Failed to write asset view spreadsheet." << endl;
			return false;
		}
	}

	return true;
}

bool ProcessSubmit()
{
#if defined( SUPPORT_CONFLUENCE )
	if ( !InitialiseConfluence() )
	{
		return false;
	}
	TStrings files;

	if ( !AddGraphsToSubmit( MAINMEM_EXTENSION1, files ) ) return false;
	if ( !AddGraphsToSubmit( MAINMEM_EXTENSION2, files ) ) return false;

	files.push_back("out.png");
	files.push_back("out.svg");

	// Read in username and password - should be possible to pipe these in
	char ch;
	string user = config.sConfluenceUser;
	if ( user.empty() )
	{
		printf( "[Confluence] Username: " );
		while ( ch = getc( stdin ) )
		{
			if ( ch == 13 || ch == '\n' || ch == '\0' ) break;
			user.append( 1, ch );
		}
		printf( "\n" );
	}

	string pass = config.sConfluencePassword;
	if ( pass.empty() )
	{
		printf( "[Confluence] Password: " );
		while ( ch = getc( stdin ) )
		{
			if ( ch == 13 || ch == '\n' || ch == '\0' ) break;
			pass.append( 1, ch );
		}
		printf( "\n" );
	}

	if ( ConfluenceLogin( user, pass ) )
	{
		ConfluencePage page;
		if ( ConfluenceGetPage( config.sConfluenceSpace, config.sConfluencePage, page ) )
		{
			// Upload the graphs
			for ( TStrings::iterator it = files.begin(), end = files.end(); it != end; ++it )
			{
				string fullName = config.sReportOutDir + *it;
				// Get the MIME type
				string mimeType;
				if ( strstr( fullName.c_str(), ".png" ) )
				{
					mimeType = "image/png";
				}
				else
				{
					mimeType = "image/svg+xml";
				}
	
				// Upload
				cout << "[Confluence] Uploading graph '" << fullName << "' as '" << *it << "'" << endl;
				string title( "Main Memory Graph" );
				if ( !ConfluenceAttachFile( fullName, *it, title, mimeType, page ) )
				{
					cerr << "[Confluence] Couldn't upload graph '" << fullName << "' as '" << *it << "'" << endl;
				}
			}

			// Release the data allocated in the page structure
			ConfluenceFreePage( page );
		}

		ConfluenceLogout();
	}
#else
	fprintf( stderr, "Confluence support is not enabled in this build. Rebuild with SUPPORT_CONFLUENCE to enable support.\n" );
#endif

	return true;
}

// Progress view utility functions

bool GenerateSummary( const char* const pFilename, SYSTEMTIME& date, SSummary& summary )
{
	if ( !pFilename ) return false;

	// Store the time/date
	summary.m_date = date;

	// Load a file
	TiXmlDocument doc( pFilename );
	bool loaded = doc.LoadFile();

	if ( !loaded )
	{
		fprintf( stderr, "Could not load file '%s'. Error='%s'.\n", pFilename, doc.ErrorDesc() );
		return false;
	}

	// Start processing the doc
	TiXmlNode* node = doc.RootElement();
	TiXmlElement* element = NULL;

	// Strip off the Workbook and check for an ExcelWorkbook element
	node = doc.FirstChild( "Workbook" );

	if ( !node || !node->FirstChild( "ExcelWorkbook" ) )
	{
		fprintf( stderr, "File is not a valid Excel XML document.\n" );
		return false;
	}

	// We're looking for the "Worksheet" elements
	// Specifically one with name "Overview"
	bool foundOverview = false;
	node = node->FirstChild( "Worksheet" );
	while ( node )
	{
		if ( element = node->ToElement() )
		{
			const char* pName = element->Attribute( "ss:Name" );
			if ( pName && !strcmp( pName, "Overview" ) )
			{
				foundOverview = true;
				break;
			}
		}

		node = node->NextSibling( "Worksheet" );
	}
	// Get the table from the worksheet
	node = node ? node->FirstChild( "Table" ) : NULL;
	if ( !node || !foundOverview )
	{
		fprintf( stderr, "Could not Overview worksheet.\n" );
		return false;
	}

	// Now get the 'Row' elements of the overview worksheet
	TiXmlNode* row = node->FirstChild( "Row" );
	while ( row )
	{
		if ( !ProcessSummaryRow( row, summary ) )
		{
			fprintf( stderr, "Unexpected summary row.\n" );
		}
		row = row->NextSibling( "Row" );
	}
	// Print the summary
	PrintSummary( summary );

	return true;
}

bool operator < (const SSummary& left, const SSummary& right)
{
	// For sorting SSummary
	FILETIME leftTime;
	FILETIME rightTime;

	SystemTimeToFileTime( &left.m_date, &leftTime );
	SystemTimeToFileTime( &right.m_date, &rightTime );

	if ( CompareFileTime( &leftTime, &rightTime ) == - 1 ) return true;

	return false;
} 

bool ProcessSummaryRow( TiXmlNode* row, SSummary& summary )
{
	// Expecting four 'Cell' element - one heading, main mem, RSX mem and total mem
	TiXmlNode* titleCell = row ? row->FirstChild( "Cell" ) : NULL;
	TiXmlNode* dataCell = titleCell ? titleCell->NextSibling( "Cell" ) : NULL;
	TiXmlNode* dataCellRSX = dataCell ? dataCell->NextSibling( "Cell" ) : NULL;
	if ( !titleCell || !dataCell || !dataCellRSX ) return false;

	// Each cell should contain a 'Data' element
	TiXmlNode* titleData = titleCell->FirstChild( "Data" );
	TiXmlNode* dataData = dataCell->FirstChild( "Data" );
	TiXmlNode* dataDataRSX = dataCellRSX->FirstChild( "Data" );
	if ( !titleData || !dataCell ) return false;

	// Get the elements and check their types
	TiXmlElement* titleElement = titleData->ToElement();
	TiXmlElement* dataElement = dataData->ToElement();
	TiXmlElement* dataElementRSX = dataDataRSX->ToElement();
	if ( !titleElement || !dataElement || !dataElementRSX ) return false;
	if ( !IsString( titleElement ) ) return false;
	if ( !IsNumber( dataElement ) )	return true; // Looks like we hit the heading row
	if ( !IsNumber( dataElementRSX ) )	return true; // Looks like we hit the heading row
	
	// Grab the data
	const char* const pData = dataElement->GetText();
	if ( !pData ) return false;
	const int iData = atoi( pData );

	const char* const pDataRSX = dataElementRSX->GetText();
	if ( !pDataRSX ) return false;
	const int iDataRSX = atoi( pDataRSX );

	// Grab the title
	const char* const pTitle = titleElement->GetText();
	if ( !pTitle ) return false;

	// And put the data in the appropriate field
	const TStrings& memoryTypes = GetMemoryTypesWithoutRSX();
	if( find( memoryTypes.begin(), memoryTypes.end(), pTitle ) != memoryTypes.end() )
	{
		summary.m_memory[pTitle] = iData;
		summary.m_memory[ GetRSXMemoryName(pTitle) ] = iDataRSX;
	}
	else
	{
		// Not recognised but not necessarily a fatal error
		return false;
	}

	return true;
}

bool WriteProgressView( const string& sFilename, vector<SSummary>& list )
{
	if ( sFilename.empty() ) return false;

	ofstream file(sFilename.c_str());
	if ( !file.good() )
	{
		cerr << "Cannot open output file '" << sFilename << "'." << endl;
		return false;
	}

	file << "# Date";
	const TStrings& memoryTypes = GetMemoryTypesWithRSX();
	for ( TStrings::const_iterator itMemoryType = memoryTypes.begin(), itEnd = memoryTypes.end(); itMemoryType != itEnd; ++itMemoryType)
	{
		file << ", " << *itMemoryType;
	}
	file << endl;

	for ( vector<SSummary>::iterator it = list.begin(), end = list.end(); it != end; it++ )
	{
		SSummary& summary = *it;
		file << summary.m_date.wHour << ":" << summary.m_date.wMinute << ":" << summary.m_date.wSecond
			<< " " << summary.m_date.wDay << "/" << summary.m_date.wMonth << "/" << summary.m_date.wYear;

		for ( TStrings::const_iterator itMemoryType = memoryTypes.begin(), itEnd = memoryTypes.end(); itMemoryType != itEnd; ++itMemoryType)
		{
			float fMemoryUsage = static_cast<float>(summary.m_memory[*itMemoryType] / ONE_MEG);
			file << "," << fMemoryUsage;
		}
		file << endl;
	}
	file << endl;

	file.close();

	return CopyFileToReportDir( sFilename );
}

bool CreateGnuplotInputFile(const string& sDirectory, const string& sTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat )
{
	assert( (eGraphFormat == eGF_SVG) || (eGraphFormat == eGF_PNG) );
	assert( (eGraphType == eGT_MemoryTypes) || (eGraphType == eGT_LevelTotals) );

	const char* const szTempFileName = sTempFilename.c_str();

	ofstream tempFile(szTempFileName);
	if(!tempFile.good())
	{
		return false;
	}

	tempFile << "set datafile separator \",\"" << endl;

	if(eGraphFormat == eGF_SVG)
	{
		tempFile << "set terminal svg enhanced size 800 480";
	}
	else
	{
		assert( eGraphFormat == eGF_PNG );
		tempFile << "set terminal png";
	}
	tempFile << endl;

	tempFile << "set xdata time" << endl;
	tempFile << "set timefmt \"%H:%M:%S %d/%m/%Y\"" << endl;
	tempFile << "set format x \"%d/%m\"" << endl;
	tempFile << "set xtics rotate by -90" << endl;
	tempFile << "set key outside bottom" << endl;
	const bool bPS3LevelTotals = ( eGraphType == eGT_LevelTotals && config.ePlatform == config.ePL_PS3 );
	if ( bPS3LevelTotals )
	{
		// This defines a function defining a horizontal line at y = 212 for illustrating the available main memory on PS3
		tempFile << "am(x) = 212" << endl;
	}

	const TStrings& lineNames = (eGraphType == eGT_MemoryTypes) ? GetMemoryTypesWithoutRSX( false, true ) : config.mapList.GetMapNames();
	for ( size_t i = 0, numTypes = lineNames.size(); i < numTypes; ++i )
	{
		const char* szFirstToken = ( ( i == 0 ) ? "plot " : ", ");
		tempFile << szFirstToken;

		if ( bPS3LevelTotals && i == 0 )
		{
			tempFile << "am(x) title 'Total' with lines linewidth 2 linetype 1, '";
		}
		else
		{
			tempFile << "'";
		}

		tempFile << sDirectory;
		
		int graphNumber = 0;
		if ( eGraphType == eGT_LevelTotals )
		{
			graphNumber = 9;
			tempFile << lineNames[i] << "/";
		}
		else
		{
			assert( eGraphType == eGT_MemoryTypes );
			graphNumber = i + 2;
		}
		tempFile << "ProgressView.csv' using 1:" << graphNumber << " title '" << lineNames[i] <<"' with lines";
	}

	tempFile.close();

	return true;
}

bool CreateGnuplotInputFileForBarChart( const string& sDirectory, const string& sTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat, int iChartNum )
{
	assert( (eGraphFormat == eGF_SVG) || (eGraphFormat == eGF_PNG) );
	assert( (eGraphType == eGT_ResourceByLevel) || (eGraphType == eGT_ResourceByType) );

	const char* const szTempFileName = sTempFilename.c_str();

	ofstream tempFile(szTempFileName);
	if(!tempFile.good())
	{
		return false;
	}

	tempFile << "set datafile separator \",\"" << endl;

	if(eGraphFormat == eGF_SVG)
	{
		tempFile << "set terminal svg enhanced size 800 480";
	}
	else
	{
		assert( eGraphFormat == eGF_PNG );
		tempFile << "set terminal png";
	}
	tempFile << endl;

	const TStrings& lineNames = (eGraphType == eGT_ResourceByType) ? config.mapList.GetMapNames() : GetMemoryTypesWithoutRSX( true, true, true );
	const TStrings& chartNames = (eGraphType == eGT_ResourceByType) ? GetMemoryTypesWithoutRSX( true, true ) : config.mapList.GetMapNames();
	const char* const pSourceFile = ( eGraphType == eGT_ResourceByType ) ? ( RESOURCE_OUTFILE "_level" RESOURCE_EXTENSION ) : ( RESOURCE_OUTFILE RESOURCE_EXTENSION );	// Yes, this is the right way around
	const int graphNumber = iChartNum + 3;
	tempFile << "set nokey" << endl;
	tempFile << "set format x \"xtic %2.0f\"" << endl;
	tempFile << "set style line 1 lt 1 lw 50" << endl;
	tempFile << "set xrange [0:" << lineNames.size() + 1 << "]" << endl;
	tempFile << "set xtics ( \"\" 1 ";
	for ( int line = 0, last = lineNames.size(); line < last; ++line )
	{
		tempFile << ", \"" << lineNames[line] << "\" " << line + 1;
	}
	tempFile << ", \"\" " << lineNames.size() + 2 << ")" << endl;
	tempFile << "plot '" << sDirectory;
	tempFile << pSourceFile << "' using 2:" << graphNumber << " title '" << chartNames[iChartNum] <<"' with impulses ls 1";

	tempFile.close();

	return true;
}

string GetFileContents(const char* szFileName)
{
	ifstream file(szFileName);
	file.seekg(0, ifstream::end);
	size_t fileSize = file.tellg();
	file.seekg(0);

	char *szFileContents = new char[fileSize + 1];
	memset(szFileContents, 0, fileSize + 1);

	file.read(szFileContents, fileSize);
	file.close();
	string sFileContents(szFileContents);
	delete[] szFileContents;

	return sFileContents;
}

bool UpdateSvgLineNames(const string& sOutputFile, const TStrings& lineNames, bool bHasTotal /*= false*/)
{
	string sGraphContents = GetFileContents(sOutputFile.c_str());

	if ( sGraphContents.empty() )
	{
		return false;
	}

	for ( size_t i = 0, numTypes = lineNames.size(); i < numTypes; ++i )
	{
		string sPattern("Plot #");
		char szNum[16];
		sprintf_s( szNum, 16, "%i", i + ( bHasTotal ? 2 : 1 ) );
		sPattern += szNum;

		size_t patternPos = sGraphContents.find(sPattern);
		if(patternPos == string::npos)
		{
			return false;
		}

		sGraphContents.replace(patternPos, sPattern.size(), lineNames[i].c_str());
	}

	ofstream graphFile( sOutputFile.c_str() );
	graphFile << sGraphContents;
	graphFile.close();

	return true;
}

bool CreateGraph( const string& sDirectory, const char* const pTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat )
{
	string sTempFileName = sDirectory + pTempFilename;
	if ( !CreateGnuplotInputFile(sDirectory, sTempFileName, eGraphType, eGraphFormat) )
	{
		return false;
	}

	string sOutputFile = sDirectory + szGraphOutputFiles[eGraphFormat];

	string sCommandLine = config.sGnuplotPath + " \"" + sTempFileName + "\" > \"" + sOutputFile + "\"";
	const int exitCode = system(sCommandLine.c_str());
	if ( exitCode != 0 )
	{
		return false;
	}

	remove(sTempFileName.c_str());

	if ( eGraphFormat == eGF_SVG )
	{
		const TStrings& lineNames = (eGraphType == eGT_MemoryTypes) ? GetMemoryTypesWithoutRSX( false, true ) : config.mapList.GetMapNames();
		if ( !UpdateSvgLineNames(sOutputFile, lineNames, (eGraphType == eGT_LevelTotals)) )
		{
			return false;
		}
	}

	return CopyFileToReportDir( sOutputFile );
}

bool CreateBarCharts( const string& sDirectory, const char* const pTempFilename, EGraphType eGraphType, EGraphFormat eGraphFormat )
{
	bool bSuccess = true;
	const TStrings& chartNames = ( eGraphType == eGT_ResourceByType ) ? GetMemoryTypesWithoutRSX( true, true ) : config.mapList.GetMapNames();
	string sTempFileName = sDirectory + pTempFilename;
	for ( int i = 0, numCharts = chartNames.size(); i < numCharts; i++ )
	{
		if ( !CreateGnuplotInputFileForBarChart(sDirectory, sTempFileName, eGraphType, eGraphFormat, i ) )
		{
			return false;
		}
	
		string sOutputPrefix;
		if ( eGraphType == eGT_ResourceByType )
		{
			sOutputPrefix = chartNames[i] + "_";
		}
		else
		{
			assert( eGraphType == eGT_ResourceByLevel );
			sOutputPrefix = "resource_" + chartNames[i] + "_";
		}
	
		string sOutputFile = sDirectory + sOutputPrefix + szGraphOutputFiles[eGraphFormat];
	
		string sCommandLine = config.sGnuplotPath + " \"" + sTempFileName + "\" > \"" + sOutputFile + "\"";
		const int exitCode = system(sCommandLine.c_str());
		if ( exitCode != 0 )
		{
			return false;
		}
	
		remove(sTempFileName.c_str());
	
		bSuccess = bSuccess && CopyFileToReportDir( sOutputFile );
	}
	return bSuccess;
}

void PrintSummary( SSummary& summary )
{
	printf( "Summary: (%i/%i/%i %i:%i%:%i)\n", summary.m_date.wDay, summary.m_date.wMonth, summary.m_date.wYear, summary.m_date.wHour, summary.m_date.wMinute, summary.m_date.wSecond );
	
	const TStrings& memoryTypes = GetMemoryTypesWithRSX();
	for ( TStrings::const_iterator itMemoryType = memoryTypes.begin(), itEnd = memoryTypes.end(); itMemoryType != itEnd; ++itMemoryType )
	{
		const string& sMemoryType = *itMemoryType;
		cout << "\t" << sMemoryType << ": " << summary.m_memory[sMemoryType] << endl;
	}

	printf( "\n" );
}

// Asset view utility functions

bool FindNewestLogfile( const string& sDirectory, string& sOutput )
{
	bool res = true;

	// Store newest date
	FILETIME newest;
	bool bFirst = true;

	// Add a wildcard to the end of the directory
	string sSearchString = sDirectory + EXCEL_WILDCARD;

	WIN32_FIND_DATAA result;
	HANDLE hSearch = FindFirstFileA( sSearchString.c_str(), &result );
	if ( hSearch != INVALID_HANDLE_VALUE)
	{
		do
		{
			if ( bFirst || CompareFileTime( &newest, &result.ftCreationTime ) == -1 )
			{
				sOutput = sDirectory + result.cFileName;
				newest = result.ftCreationTime;
			}
		} while ( FindNextFileA( hSearch, &result ) );
		 
	}
	else
	{
		res = false;
	}
	FindClose( hSearch );

	return res;
}

bool ProcessAssetSpreadsheet( const char* const pFilename, const char* const pMap )
{
	if ( !pFilename ) return false;

	// Load a file
	TiXmlDocument doc( pFilename );
	bool loaded = doc.LoadFile();

	if ( !loaded )
	{
		fprintf( stderr, "Could not load file '%s'. Error='%s'.\n", pFilename, doc.ErrorDesc() );
		return false;
	}

	// Start processing the doc
	TiXmlNode* node = doc.RootElement();
	TiXmlElement* element = NULL;

	// Strip off the Workbook and check for an ExcelWorkbook element
	node = doc.FirstChild( "Workbook" );

	if ( !node || !node->FirstChild( "ExcelWorkbook" ) )
	{
		fprintf( stderr, "File is not a valid Excel XML document.\n" );
		return false;
	}

	// We're looking for the "Worksheet" elements matching the names of our various asset types
	bool foundOverview = false;
	node = node->FirstChild( "Worksheet" );
	while ( node )
	{
		if ( element = node->ToElement() )
		{
			const char* pName = element->Attribute( "ss:Name" );
			if ( pName )
			{
				// Is this a worksheet we are interested in?
				for ( u_int u = 0; u < numAssetTypes; u++ )
				{
					if ( !strcmp( pName, pAssetTypes[u].m_pName ) )
					{
						if ( !ProcessAssetTypeWorksheet( pAssetTypes[u], node, pMap ) )
						{
							fprintf( stderr, "Could not process worksheet %s\n", pAssetTypes[u].m_pName );
						}
					}
				}
			}
		}

		node = node->NextSibling( "Worksheet" );
	}

	return true;
}

bool ProcessAssetTypeWorksheet( SAssetType& assetType, TiXmlNode* sheet, const char* const pMap )
{
	// Get the table from the worksheet
	TiXmlNode* node = sheet ? sheet->FirstChild( "Table" ) : NULL;
	if ( !node )
	{
		fprintf( stderr, "Could not read worksheet %s.\n", assetType.m_pName );
		return false;
	}

	// Now process each row in turn
	TiXmlNode* row = node->FirstChild( "Row" );
	while ( row )
	{
		if ( !ProcessAssetTypeRow( assetType, row, pMap ) )
		{
			fprintf( stderr, "Unexpected asset type row.\n" );
		}
		row = row->NextSibling( "Row" );
	}

	return true;
}

bool ProcessAssetTypeRow( SAssetType& assetType, TiXmlNode* row, const char* const pMap )
{
	// Check how many cells we expect (though we only care for the first and last)
	const u_int cells = assetType.m_columns;

	// Buffer for the row's name
	string name = "";

	// Process the cells to get the name and memory usage
	// Name cell comes from concatenating all of the cells leading to the memory usage cell
	TiXmlNode* nameCell = row->FirstChild( "Cell" );
	for ( u_int u = 0; u < ( cells - 1 ); u++ )
	{
		// Get the data from the cell
		TiXmlNode* nameData = nameCell->FirstChild( "Data" );
		TiXmlElement* nameElement = nameData ? nameData->ToElement() : NULL;
		if ( nameElement )
		{
			const char* const pName = nameElement->GetText();
			if ( pName )
			{
				if ( u > 0 ) name.append( " (" );
				name.append( pName );
				if ( u > 0 ) name.append( ")" );
			}
		}
		else
		{
			return false;
		}
		nameCell = nameCell ? nameCell->NextSibling( "Cell" ) : NULL;
	}
	// Now get the memory cell - nameCell should now point to it
	TiXmlNode* memCell = nameCell;
	TiXmlNode* memCellRSX = nameCell->NextSibling( "Cell" );
	if ( !memCell || !memCellRSX ) return false;

	// Each cell should contain a 'Data' element
	TiXmlNode* memData = memCell->FirstChild( "Data" );
	TiXmlNode* memDataRSX = memCellRSX->FirstChild( "Data" );
	if ( !memData || !memData ) return false;

	// Get the elements and check their types
	TiXmlElement* memElement = memData->ToElement();
	TiXmlElement* memElementRSX = memDataRSX->ToElement();
	if ( !memElement || !memElementRSX ) return false;
	//if ( !IsString( nameElement ) ) return false;
	if ( !IsNumber( memElement ) )	return true; // Looks like we hit the heading row
	if ( !IsNumber( memElementRSX ) )	return true; // Looks like we hit the heading row
	
	// Grab the data
	const char* const pData = memElement->GetText();
	if ( !pData ) return false;
	const int iData = atoi( pData );

	const char* const pDataRSX = memElementRSX->GetText();
	if ( !pDataRSX ) return false;
	const int iDataRSX = atoi( pDataRSX );

	// Grab the title
	const int len = name.length();
	char* const pName = new char[len + 1];
	strncpy_s( pName, len + 1, name.c_str(), len + 1 );
	if ( !pName ) return false;

	// Store the data
	return StoreAsset( assetType, pName, iData, iDataRSX, pMap );
}

bool StoreAsset( SAssetType& assetType, const char* const pName, const unsigned int mem, const unsigned int rsx, const char* const pMap )
{
	if ( !pName ) { return false; }

	// Look for an existing asset with the same name
	SAsset* pAsset = NULL;
	for ( vector<SAsset>::iterator it = assetType.m_data.begin(), end = assetType.m_data.end(); it != end; ++it )
	{
		// Compare names
		if ( !strcmp( pName, it->m_pName ) )
		{
			pAsset = &(*it);
			break;
		}
	}

	// Add or update the asset
	if ( pAsset )
	{
		pAsset->m_levels.push_back( pMap );
		if ( !pAsset->m_memory == mem )
		{
			fprintf( stderr, "Asset size mismatch for '%s' was %i now %i on %s\n", pName, pAsset->m_memory, mem, pMap );
		}
		pAsset->m_memory = max( pAsset->m_memory, mem );
		if ( !pAsset->m_memoryRSX == rsx )
		{
			fprintf( stderr, "Asset RSX size mismatch for '%s' was %i now %i on %s\n", pName, pAsset->m_memoryRSX, rsx, pMap );
		}
		pAsset->m_memoryRSX = max( pAsset->m_memoryRSX, rsx );
	}
	else
	{
		SAsset asset;
		// Need to copy the name string
		const int len = strlen( pName );
		asset.m_pName = new char[len +1 ];
		strncpy_s( asset.m_pName, len + 1, pName, len );
		asset.m_memory = mem;
		asset.m_memoryRSX = rsx;
		asset.m_levels.push_back( pMap );
		assetType.m_data.push_back( asset );
	}

	return true;
}

bool WriteAssetViewSpreadsheet( const string& sFilename )
{
	// Create a new TinyXml document

	TiXmlDocument doc( sFilename.c_str() );

	// Add the XML version declaration
	TiXmlDeclaration* decl = new TiXmlDeclaration( "1.0", "", "" );
	TiXmlElement* root = new TiXmlElement( "Workbook" );
	TiXmlElement* excel = new TiXmlElement( "ExcelWorkbook" );
	if ( !decl || !root || !excel)
	{
		delete decl;
		delete root;
		delete excel;
		return false;
	}
	root->SetAttribute( "xmlns", "urn:schemas-microsoft-com:office:spreadsheet" );
	excel->SetAttribute( "xmlns", "urn:schemas-microsoft-com:office:excel" );
	root->LinkEndChild( excel );

	doc.LinkEndChild( decl );
	doc.LinkEndChild( root );
	// Do we need to write out styles?
	for ( u_int u = 0; u < numAssetTypes; u++ )
	{
		if ( !WriteAssetViewWorksheet( root, u ) )
		{
			fprintf( stderr, "Error writing worksheet '%s'\n", pAssetTypes[u].m_pName );
		}
	}

	// While useful for debugging, spewing the contents of the file to the console is very slow!
	// It's also not amazingly useful as if verbose == true output is polluted with other info
	if( 0 && config.bVerbose )
	{
		doc.Print( stdout );
	}
	doc.SaveFile();

	return CopyFileToReportDir( sFilename );
}

bool WriteAssetViewWorksheet( TiXmlNode* root, unsigned int assetType )
{
	if ( !root ) return false;

	// Get the asset type
	SAssetType& assetList = pAssetTypes[assetType];

	// Create a new Worksheet containing a table
	TiXmlElement* worksheet = new TiXmlElement( "Worksheet" );
	TiXmlElement* table = new TiXmlElement( "Table" );
	if ( !worksheet || !table )
	{
		delete [] worksheet;
		delete [] table;
		return false;
	}
	root->LinkEndChild( worksheet );

	worksheet->SetAttribute( "ss:Name", assetList.m_pName );
	worksheet->LinkEndChild( table );

	// Declare the columns
	for ( u_int u = 0; u < 3 + config.mapList.GetNumMaps(); u++ )
	{
		TiXmlElement* column = new TiXmlElement( "Column" );
		if ( !column ) return false;
		column->SetAttribute( "ss:Width", "100" );
		table->LinkEndChild( column );
	}

	// Now write out the data row by row (the columns above are just a declaration,
	// they define how many data elements are expected in the row and column widths

	// Write heading?
	for ( vector<SAsset>::iterator it = assetList.m_data.begin(), end = assetList.m_data.end(); it != end; ++it )
	{
		WriteAssetViewRow( table, *it );
	}

	return true;
}

bool WriteAssetViewRow( TiXmlNode* table, SAsset& rowData )
{
	if ( !table ) return false;

	// Create a row
	TiXmlElement* row = new TiXmlElement( "Row" );
	TiXmlElement* cell1 = new TiXmlElement( "Cell" );
	TiXmlElement* cell2 = new TiXmlElement( "Cell" );
	TiXmlElement* cell3 = new TiXmlElement( "Cell" );
	vector<TiXmlElement*> cell4;
	cell4.resize(config.mapList.GetNumMaps());
	if ( !row || !cell1 || !cell2 || !cell3 )
	{
		delete row;
		delete cell1;
		delete cell2;
		delete cell3;
		return false;
	}
	row->LinkEndChild( cell1 );
	row->LinkEndChild( cell2 );
	row->LinkEndChild( cell3 );
	for ( u_int u = 0; u < config.mapList.GetNumMaps(); u++ )
	{
		cell4[u] = new TiXmlElement( "Cell" );
		if ( !cell4[u] ) return false;
		row->LinkEndChild( cell4[u] );
	}
	table->LinkEndChild( row );

	// Create the data elements
	TiXmlElement* name = new TiXmlElement( "Data" );
	TiXmlElement* mem = new TiXmlElement( "Data" );
	TiXmlElement* rsx = new TiXmlElement( "Data" );
	vector<TiXmlElement*> maps;
	maps.resize(config.mapList.GetNumMaps());
	if ( !name || !mem || !rsx )
	{
		delete name;
		delete mem;
		delete rsx;
		return false;
	}
	for ( u_int u = 0; u < config.mapList.GetNumMaps(); u++ )
	{
		maps[u] = new TiXmlElement( "Data" );
		if ( !maps[u] ) return false;
		maps[u]->SetAttribute( "ss:Type", "String" );
		cell4[u]->LinkEndChild( maps[u] );
	}
	name->SetAttribute( "ss:Type", "String" );
	mem->SetAttribute( "ss:Type", "Number" );
	rsx->SetAttribute( "ss:Type", "Number" );
	cell1->LinkEndChild( name );
	cell2->LinkEndChild( mem );
	cell3->LinkEndChild( rsx );

	// Name string - should be allocated for a larger scope than the document exists for
	TiXmlText* nameText = new TiXmlText( rowData.m_pName );

	// Need to allocate a buffer for the memory figure
	char pExtMem[64];
	_itoa_s( rowData.m_memory, pExtMem, 64, 10 );
	const int len = strlen( pExtMem );
	char* pMem = new char[len + 1];
	strncpy_s( pMem, len + 1, pExtMem, len + 1 );
	TiXmlText* memText = new TiXmlText( pMem );
	_itoa_s( rowData.m_memoryRSX, pExtMem, 64, 10 );
	const int lenRSX = strlen( pExtMem );
	char* pMemRSX = new char[lenRSX + 1];
	strncpy_s( pMemRSX, lenRSX + 1, pExtMem, lenRSX + 1 );
	TiXmlText* memTextRSX = new TiXmlText( pMemRSX );

	// Need to allocate and construct a string of map names
	bool *mapUsage = new bool[config.mapList.GetNumMaps()];
	for ( u_int u = 0; u < config.mapList.GetNumMaps(); u++ ) mapUsage[u] = false;
	for ( vector<const char*>::iterator it = rowData.m_levels.begin(), end = rowData.m_levels.end(); it != end; ++it )
	{
		// Find the matching map name and set as present
		for ( u_int u = 0; u < config.mapList.GetNumMaps(); u++ )
		{
			if ( !strcmp( *it, config.mapList.GetMap(u) ) )
			{
				mapUsage[u] = true;
				break;
			}
		}
	}
	if ( !nameText || !memText || !memTextRSX )
	{
		delete nameText;
		delete memText;
		delete [] pMem;
		delete memTextRSX;
		delete [] pMemRSX;
		delete [] mapUsage;
		return false;
	}
	name->LinkEndChild( nameText );
	mem->LinkEndChild( memText );
	rsx->LinkEndChild( memTextRSX );

	// Now fill in the rows accordingly
	for ( u_int u = 0; u < config.mapList.GetNumMaps(); u ++ )
	{
		if ( mapUsage[u] )
		{
			const int mapsLen = strlen( config.mapList.GetMap(u) );
			char* pMaps = new char[mapsLen + 1];
			strncpy_s( pMaps, mapsLen + 1, config.mapList.GetMap(u), mapsLen + 1 );
			TiXmlText* mapsText = new TiXmlText( pMaps );
			if ( !pMaps || !mapsText ) return false;
			maps[u]->LinkEndChild( mapsText );
		}
	}

	delete [] mapUsage;

	return true;
}

// Resource view utility functions

bool WriteResourceView( const string sFilename, const string sFilenameExtension, vector< pair<string, SSummary> >& summaries )
{
	// Write out the spreadsheet
	if ( !WriteResourceViewSpreadsheet( sFilename + sFilenameExtension, summaries, false ) )
	{
		return false;
	}
	if ( !WriteResourceViewSpreadsheet( sFilename + "_level" + sFilenameExtension, summaries, true ) )
	{
		return false;
	}
	
	// Now build sets of bar charts from it
	if ( !CreateBarCharts( config.sMemoryStatsDir, PROGRESS_GRAPH_TEMPFILE, eGT_ResourceByType, eGF_SVG ) ) return false;
	if ( !CreateBarCharts( config.sMemoryStatsDir, PROGRESS_GRAPH_TEMPFILE, eGT_ResourceByLevel, eGF_SVG ) ) return false;

	return true;
}

bool WriteResourceViewSpreadsheet( const string sFilename, vector< pair<string, SSummary> >& summaries, bool bFlip )
{
	if ( sFilename.empty() ) return false;

	ofstream file(sFilename.c_str());
	if ( !file.good() )
	{
		cerr << "Cannot open output file '" << sFilename << "'." << endl;
		return false;
	}

	if ( !bFlip )
	{
		// Write out headings for human readability
		file << "# Resource Type, Index";
		for ( vector< pair<string, SSummary> >::iterator itMap = summaries.begin(), endMap = summaries.end(); itMap != endMap; itMap++ )
		{
			file << ", " << itMap->first;
		}
		file << endl;
	
		// Now write out a row for each resource type
		const TStrings& memoryTypes = GetMemoryTypesWithoutRSX( true, true, true );
		int idx = 0;
		for ( TStrings::const_iterator itMemoryType = memoryTypes.begin(), itMemoryTypeEnd = memoryTypes.end(); itMemoryType != itMemoryTypeEnd; ++itMemoryType)
		{
			// Index 
			file << *itMemoryType << ", " << ++idx;
	
			// Data for each map
			for ( vector< pair<string, SSummary> >::iterator itMap = summaries.begin(), endMap = summaries.end(); itMap != endMap; itMap++ )
			{
				SSummary& summary = itMap->second;
				file << ", " << ( summary.m_memory[*itMemoryType] / ONE_MEG );
			}
			file << endl;
		}
	}
	else
	{
		// Write out headings for human readability
		file << "# Map, Index";
		const TStrings& memoryTypes = GetMemoryTypesWithoutRSX();
		for ( TStrings::const_iterator itMemoryType = memoryTypes.begin(), itMemoryTypeEnd = memoryTypes.end(); itMemoryType != itMemoryTypeEnd; ++itMemoryType)
		{
			file << ", " << itMemoryType->c_str();
		}
		file << endl;
	
		// Now write out a row for each map
		int idx = 0;
		for ( vector< pair<string, SSummary> >::iterator itMap = summaries.begin(), endMap = summaries.end(); itMap != endMap; itMap++ )
		{
			SSummary& summary = itMap->second;

			// Index 
			file << itMap->first.c_str() << ", " << ++idx;
	
			// Data for each memory type
			for ( TStrings::const_iterator itMemoryType = memoryTypes.begin(), itMemoryTypeEnd = memoryTypes.end(); itMemoryType != itMemoryTypeEnd; ++itMemoryType)
			{
				file << ", " << ( summary.m_memory[*itMemoryType] / ONE_MEG );
			}
			file << endl;
		}
	}

	file.close();

	return CopyFileToReportDir( sFilename );
}

// Confluence submit utility functions

bool AddGraphsToSubmit( const char* const pExtension, TStrings& files )
{
	// Per map views
	const TStrings& maps = config.mapList.GetMapNames();
	for ( TStrings::const_iterator it = maps.begin(), itEnd = maps.end(); it != itEnd; ++it)
	{
		string sGraphFileName = *it + pExtension;
		files.push_back(sGraphFileName);

		sGraphFileName = "resource_" + *it + pExtension;
		files.push_back(sGraphFileName);
	}

	// Per resource views
	const TStrings& resources = GetMemoryTypesWithoutRSX( true, true );
	for ( TStrings::const_iterator it = resources.begin(), itEnd = resources.end(); it != itEnd; ++it)
	{
		string sGraphFileName = *it + pExtension;
		files.push_back(sGraphFileName);
	}

	return true;
}

// Shared utility functions
bool IsString( TiXmlElement* elem )
{
	if ( elem )
	{
		const char* const pType = elem->Attribute( "ss:Type" );
		if ( pType )
		{
			return !strcmp( pType, "String" );
		}
	}
	return false;
}

bool IsNumber( TiXmlElement* elem )
{
	if ( elem )
	{
		const char* const pType = elem->Attribute( "ss:Type" );
		if ( pType )
		{
			return !strcmp( pType, "Number" );
		}
	}
	return false;
}

bool ProcessTimeStamp( const char* const pFilename, SYSTEMTIME& date )
{
	char pScratch[5];

	// Filename should start with 'memlog-' and end with '.zmrl.xml' or '.mrl.xml'
	// In newer versions of memReplay the filename will start with 'memlog-<map>-'
	if ( strncmp( pFilename, "memlog-", 7 ) ) return false;

	// Count the dashes to work out the version
	// Store the position of the second dash for use on the new format
	const char* pCur = pFilename;
	int dashes = 0;
	int secondDash = 0;
	while ( *pCur != '\0' )
	{
		if ( *(pCur) == '-' )
		{
			++dashes;
			if ( dashes == 2 )
			{
				secondDash = pCur - pFilename;
			}
		}
		++pCur;
	}

	if ( dashes == 2 )	// Original format
	{
		pCur = pFilename + 7;
	}
	else if ( dashes == 3 )	// New format
	{
		pCur = pFilename + secondDash + 1;
	}
	else	// Unknown format
	{
		fprintf( stderr, "Unknown filename format '%s'.\n", pFilename );
		return false;
	}

	// Get to the start of the date
	//const char* pCur = pFilename + 7;

	// Four digits of year
	strncpy_s( pScratch, 5, pCur, 4 );
	date.wYear = atoi( pScratch );

	// Two digits of month
	pCur += 4;
	strncpy_s( pScratch, 5, pCur, 2 );
	date.wMonth = atoi( pScratch );

	// Two digits of day
	pCur += 2;
	strncpy_s( pScratch, 5, pCur, 2 );
	date.wDay = atoi( pScratch );

	// Now expect a dash
	pCur += 2;
	if ( *pCur != '-' ) return false;

	// Two digits of hour
	pCur += 1;
	strncpy_s( pScratch, 5, pCur, 2 );
	date.wHour = atoi( pScratch );

	// Two digits of minute
	pCur += 2;
	strncpy_s( pScratch, 5, pCur, 2 );
	date.wMinute = atoi( pScratch );

	// Two digits of second
	pCur += 2;
	strncpy_s( pScratch, 5, pCur, 2 );
	date.wSecond = atoi( pScratch );

	// '.zmrl.xml' or '.mrl.xml'
	pCur += 2;
	if ( strncmp( pCur, ".zmrl.xml", 9 ) && strncmp( pCur, ".mrl.xml", 8 ) ) return false;

	// Zero the rest
	date.wDayOfWeek = 0;
	date.wMilliseconds = 0;

	return true;
}

#if defined( SUPPORT_CONFLUENCE )

bool InitialiseConfluence( void )
{
	if ( g_hConfluence == NULL )
	{
		const HMODULE g_hConfluence = LoadLibraryA( "ConfluenceWrapper.dll" );
		if ( g_hConfluence == NULL )
		{
			const DWORD error = GetLastError();
			fprintf( stderr, "Failed to load ConfluenceWrapper.dll (%i)\n", error );
			return false;
		}
	
		g_Conf.pfnLogin = (ConfLogin*)GetProcAddress( g_hConfluence, "Login" );
		g_Conf.pfnLogout = (ConfLogout*)GetProcAddress( g_hConfluence, "Logout" );
		g_Conf.pfnGetPage = (ConfGetPage*)GetProcAddress( g_hConfluence, "GetPage" );
		g_Conf.pfnFreePage = (ConfFreePage*)GetProcAddress( g_hConfluence, "FreePage" );
		g_Conf.pfnAttachFile = (ConfAttachFile*)GetProcAddress( g_hConfluence, "AttachFile" );
	}
	return true;
}

bool ShutdownConfluence( void )
{
	if ( g_hConfluence != NULL )
	{
		FreeLibrary( g_hConfluence );
		g_hConfluence = NULL;
	
		g_Conf.pfnLogin = NULL;
		g_Conf.pfnLogout = NULL;
		g_Conf.pfnGetPage = NULL;
		g_Conf.pfnFreePage = NULL;
		g_Conf.pfnAttachFile = NULL;
	}
	return true;
}

bool ConfluenceLogin( string& username, string& password )
{
	assert( g_Conf.pfnLogin );
	return g_Conf.pfnLogin( username.c_str(), password.c_str(), g_authToken, s_AUTHTOKEN_MAX_LEN );
}

void ConfluenceLogout( void )
{
	assert( g_Conf.pfnLogout );
	g_Conf.pfnLogout( g_authToken );
}

bool ConfluenceGetPage( string& space, string& page, ConfluencePage& remotePage )
{
	assert( g_Conf.pfnGetPage );
	if ( g_Conf.pfnGetPage( g_authToken, space.c_str(), page.c_str(), remotePage ) )
	{
		printf( "[Confluence] Retrieved page.\n" );
		printf( "[Confluence] Title:		%s\n", remotePage.titleField );
		printf( "[Confluence] Creator:	%s\n", remotePage.creatorField );
		return true;
	}
	else
	{
		printf( "[Confluence] Failed to retrieve page %s in %s.\n", page.c_str(), space.c_str() );
		return false;
	}
}

void ConfluenceFreePage( ConfluencePage& remotePage )
{
	assert( g_Conf.pfnFreePage );
	g_Conf.pfnFreePage( &remotePage );
}

bool ConfluenceAttachFile( string& filename, string& shortFilename, string& title, string& mimeType, ConfluencePage& page )
{
	assert( g_Conf.pfnAttachFile );
	const LONG64 pageId = page.idField;

	// Load in the data
	FILE* pF;
	fopen_s( &pF, filename.c_str(), "rb" );
	if ( !pF ) return false;
	fseek( pF, 0, SEEK_END );
	const int size = ftell( pF );
	fseek( pF, 0, SEEK_SET );

	unsigned char* pBuffer = new unsigned char[size];
	if ( !pBuffer )
	{
		fclose( pF );
		return false;
	}
	fread_s( pBuffer, size, sizeof( unsigned char ), size, pF );
	fclose( pF );

	bool bSuccess = g_Conf.pfnAttachFile( g_authToken, filename.c_str(), shortFilename.c_str(), title.c_str(), mimeType.c_str(), pBuffer, size, pageId );
	delete[] pBuffer;

	if ( bSuccess )
	{
		printf( "[Confluence] Attached file %s.\n", shortFilename.c_str() );
	}
	else
	{
		fprintf( stderr, "Failed to attach file '%s'.\n", shortFilename.c_str() );
	}

	return bSuccess;
}

#endif
