// DemoView.cpp : implementation of the CMemReplayView class
//

#include "stdafx.h"
#include "MemReplay.h"

#include "MemReplayDoc.h"
#include "MemReplayView.h"

#include "CodeTreePage.h"
#include "UsagePage.h"
#include "MemoryMapPage.h"
#include "ValidationPage.h"
#include "TypeInfoPage.h"

#include "ReplayProgressDlg.h"

#include "ReplaySDK/memHier2.h"
#include "ReplaySDK/ContextTree.h"
#include "ReplaySDK/ContextTreeFunctors.h"
#include "ReplaySDK/ExcelExport.h"
#include "ReplaySDK/FragmentationAnalysis.h"

/*
#ifdef _DEBUG
#define new DEBUG_NEW
#endif
*/

namespace
{
	class CreateMemHierFromContextReplayTask : public IProgressTask
	{
	public:
		CreateMemHierFromContextReplayTask(
			const SharedPtr<ReplayLogReader>& log,
			const SharedPtr<SymbolHelper>& symbols,
			const std::vector<ContextStreamOffsetSpan>& offsets)
			: m_log(log)
			, m_symbols(symbols)
			, m_offsets(offsets)
		{}

	public:
		int GetProgress() const { return m_log->GetPosition() / m_log->GetChunkSize(); }
		int GetMaximumProgress() const { return m_log->GetStreamLength() / m_log->GetChunkSize(); }

		void Start() { m_result = MemHier::FromReplayStream(m_log, m_symbols, m_offsets); }

		SharedPtr<MemHier> GetResult() const { return m_result; }

	private:
		SharedPtr<ReplayLogReader> m_log;
		SharedPtr<SymbolHelper> m_symbols;
		std::vector<ContextStreamOffsetSpan> m_offsets;

		SharedPtr<MemHier> m_result;
	};

	class CreateMemHierFromSpansReplayTask : public IProgressTask
	{
	public:
		CreateMemHierFromSpansReplayTask(
			const SharedPtr<ReplayLogReader>& log,
			const SharedPtr<SymbolHelper>& symbols,
			const std::vector<std::pair<u64, u64> >& offsets)
			: m_log(log)
			, m_symbols(symbols)
			, m_offsets(offsets)
		{}

	public:
		int GetProgress() const { return m_log->GetPosition() / m_log->GetChunkSize(); }
		int GetMaximumProgress() const { return m_log->GetStreamLength() / m_log->GetChunkSize(); }

		void Start() { m_result = MemHier::FromReplayStream(m_log, m_symbols, m_offsets); }

		SharedPtr<MemHier> GetResult() const { return m_result; }

	private:
		SharedPtr<ReplayLogReader> m_log;
		SharedPtr<SymbolHelper> m_symbols;
		std::vector<std::pair<u64, u64> > m_offsets;

		SharedPtr<MemHier> m_result;
	};
}

static void PopulateOffsets(const ContextTreeNode* node, std::vector<ContextStreamOffsetSpan>& offsetsOut)
{
	offsetsOut.insert(offsetsOut.end(), node->GetAllocStreamOffsets().begin(), node->GetAllocStreamOffsets().end());

	for (const ContextTreeNode* child = node->GetChildren(); child; child = child->GetNextSibling())
	{
		PopulateOffsets(child, offsetsOut);
	}
}

// CMemReplayView

IMPLEMENT_DYNCREATE(CMemReplayView, CView)

BEGIN_MESSAGE_MAP(CMemReplayView, CView)
	ON_WM_CREATE()
	ON_WM_SIZE()
	ON_WM_ERASEBKGND()
END_MESSAGE_MAP()

// CMemReplayView construction/destruction

CMemReplayView::CMemReplayView()
{
}

CMemReplayView::~CMemReplayView()
{
}

void CMemReplayView::OpenCodeViewFor(const char* name, const SharedPtr<MemHier>& hier)
{
	CCodeTreePage* page = new CCodeTreePage(hier, name);

	AddTab(page);
	m_tabWnd.SetActivePage(page);
}

void CMemReplayView::OpenCodeViewFor(const ContextTreeNode& node)
{
	CMemReplayDoc* doc = GetDocument();

	if (doc)
	{
		std::vector<ContextStreamOffsetSpan> offsets;
		PopulateOffsets(&node, offsets);

		CProgressTaskDlg progressDlg;
		CreateMemHierFromContextReplayTask task(doc->GetReplayStream(), doc->GetSymbolTable(), offsets);
		progressDlg.Run(task);

		SharedPtr<MemHier> hier = task.GetResult();
		OpenCodeViewFor(node.GetName(), hier);
	}
}

void CMemReplayView::OpenCodeViewForFrames(int frameBeginIdx, int frameEndIdx)
{
	using std::swap;

	CMemReplayDoc* doc = GetDocument();

	if (!doc)
		return;

	if (frameBeginIdx > frameEndIdx)
		swap(frameBeginIdx, frameEndIdx);

	const FrameUsageTracker& tracker = *doc->GetFrameUsageTracker();
	std::vector<std::pair<u64, u64> > spans(tracker.FrameOffsetsBegin() + frameBeginIdx, tracker.FrameOffsetsBegin() + frameEndIdx);

	CProgressTaskDlg progressDlg;
	CreateMemHierFromSpansReplayTask task(doc->GetReplayStream(), doc->GetSymbolTable(), spans);
	progressDlg.Run(task);

	SharedPtr<MemHier> hier = task.GetResult();

	char name[256];
	sprintf_s(name, 256, "Frames %i-%i", frameBeginIdx, frameEndIdx);

	OpenCodeViewFor(name, hier);
}

void CMemReplayView::OpenCodeViewForAllocEvRange(u64 begin, u64 end)
{
#if 1
	using std::swap;

	CMemReplayDoc* doc = GetDocument();

	if (!doc)
		return;

	if (begin > end)
		swap(begin, end);

	const FrameUsageTracker& tracker = *doc->GetFrameUsageTracker();
	std::vector<std::pair<u64, u64> > spans(tracker.AllocEvOffsetsBegin() + begin, tracker.AllocEvOffsetsBegin() + end);

	CProgressTaskDlg progressDlg;
	CreateMemHierFromSpansReplayTask task(doc->GetReplayStream(), doc->GetSymbolTable(), spans);
	progressDlg.Run(task);

	char name[256];
	sprintf_s(name, 256, "Alloc events %I64u-%I64u", begin * tracker.GetAllocEventSplit(), end * tracker.GetAllocEventSplit());

	CCodeTreePage* page = new CCodeTreePage(task.GetResult(), name);

	AddTab(page);
	m_tabWnd.SetActivePage(page);
#else
	using std::swap;

	CMemReplayDoc* doc = GetDocument();

	if (!doc)
		return;

	if (begin > end)
		swap(begin, end);

	FrameUsageTracker& tracker = doc->GetFrameUsageTracker();
	u64 evs = tracker.GetAllocEventSplit();

	AllocSetLST& lst = *doc->GetAllocSetLST();
	
	AllocSet set;
	lst.Add(set, begin * evs, end * evs);

	SharedPtr<MemHier> result = MemHier::FromAllocSet(
		doc->GetSymbolTable(), lst.GetCallstackTable(), set.GetAllocs());

	char name[256];
	sprintf_s(name, 256, "Alloc events %I64u-%I64u", begin * evs, end * evs);

	CCodeTreePage* page = new CCodeTreePage(result, name);

	AddTab(page);
	m_tabWnd.SetActivePage(page);
#endif
}

void CMemReplayView::OpenReverseCodeViewFor(const MemHier& hier, TAddress from)
{
	SharedPtr<MemHier> reversed = MemHier::Reverse(hier, from);

	if (reversed.IsValid())
	{
		CCodeTreePage* page = new CCodeTreePage(reversed, "Reverse");

		AddTab(page);
		m_tabWnd.SetActivePage(page);
	}
}

void CMemReplayView::OpenGatherCodeViewFor(const MemHier& hier, const std::vector<TAddress>& from)
{
	SharedPtr<MemHier> gathered = MemHier::Gather(hier, from);

	if (gathered.IsValid())
	{
		CCodeTreePage* page = new CCodeTreePage(gathered, "Gather");

		AddTab(page);
		m_tabWnd.SetActivePage(page);
	}
}

void CMemReplayView::OpenExcludeCodeViewFor(const MemHier& hier, const std::vector<TAddress>& from)
{
	SharedPtr<MemHier> gathered = MemHier::Exclude(hier, from);

	if (gathered.IsValid())
	{
		CCodeTreePage* page = new CCodeTreePage(gathered, "Exclude");

		AddTab(page);
		m_tabWnd.SetActivePage(page);
	}
}
	
void CMemReplayView::OpenTypeInfoViewFor(const s8 *typeName)
{
	if (GetDocument())
	{
		SharedPtr<MemHier> typeTree = MemHier::FromTypeInfo(typeName, GetDocument()->GetSymbolTable());

		if (typeTree.IsValid())
		{
			CTypeInfoPage* page = new CTypeInfoPage(typeTree, typeName);
			AddTab(page);
			m_tabWnd.SetActivePage(page);
		}
	}
}

void CMemReplayView::ExportToXLS()
{
	CFileDialog dlg(FALSE, "*.xml", NULL, OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT, "XML Files|*.xml", this);

	if (dlg.DoModal() == IDOK)
	{
		CString filename = dlg.GetFileName();

		ExcelExport xls;
		if (xls.Open(filename.GetString()))
		{
			int style = xls.BeginStyle();
			xls.EndStyle();

			for (std::vector<ITabPage*>::iterator it = m_tabs.begin(); it != m_tabs.end(); ++ it)
			{
				(*it)->Export(xls);
			}

			xls.Close();
		}
	}
}

ITabPage* CMemReplayView::GetActiveTab()
{
	CPropertyPage* activePage = m_tabWnd.GetActivePage();
	if (activePage)
	{
		int index = m_tabWnd.GetPageIndex(activePage);
		return m_tabs[index];
	}

	return NULL;
}

void CMemReplayView::SelectTab(ITabPage* page)
{ 
	m_tabWnd.SetActivePage(std::find(m_tabs.begin(), m_tabs.end(), page) - m_tabs.begin());
}

BOOL CMemReplayView::PreCreateWindow(CREATESTRUCT& cs)
{
	if (!CView::PreCreateWindow(cs))
		return FALSE;

	cs.style |= WS_CLIPCHILDREN;
	return TRUE;
}

int CMemReplayView::OnCreate(LPCREATESTRUCT lpcs)
{
	int result = CView::OnCreate(lpcs);

	m_usagePage = new CUsagePage(NULL);
	AddTab(m_usagePage);

	m_tabWnd.EnableStackedTabs(FALSE);
	m_tabWnd.Create(this, WS_CHILD | WS_VISIBLE | WS_TABSTOP, 0);

	// Fix for "Child CPropertySheet Hangs If Focus Is Switched" (Q149501)
	m_tabWnd.ModifyStyleEx(0, WS_EX_CONTROLPARENT);

	return result;
}

void CMemReplayView::OnInitialUpdate()
{
	using namespace ContextTreeFunctor;

	CView::OnInitialUpdate();

	for (int i = m_tabWnd.GetPageCount() - 1; i > 0; -- i)
	{
		CPropertyPage* page = m_tabWnd.GetPage(i);
		m_tabWnd.RemovePage(i);
		delete page;
	}

	m_tabs.erase(m_tabs.begin() + 1, m_tabs.end());

	CMemReplayDoc* doc = GetDocument();

	if (doc)
	{
		if (doc->GetFrameUsageTracker().IsValid())
		{
			m_usagePage = new CUsagePage(doc->GetFrameUsageTracker());
			AddTab(m_usagePage);
		}

		if (doc->GetValidation().IsValid() && doc->GetSymbolTable().IsValid())
			AddTab(new CValidationPage(doc->GetValidation(), doc->GetSymbolTable()));

		if (doc->GetMemMap().IsValid())
			AddTab(new CMemoryMapPage(*doc->GetFrameUsageTracker(), *doc->GetMemMap()));

		SharedPtr<ContextTree> assetTree = doc->GetAssetTree();

		if (assetTree.IsValid())
		{
			SharedPtr<ContextTree> physicsTree = doc->GetCgfPhysicsTree();
			SharedPtr<ContextTree> textureTree = doc->GetTextureTree();
			SharedPtr<ContextTree> renderMeshTree = doc->GetRenderMeshTree();
			SharedPtr<ContextTree> terrainTree = doc->GetTerrainTree();
			SharedPtr<ContextTree> animationTree = doc->GetAnimationTree();
			SharedPtr<ContextTree> navTree = doc->GetNavTree();
			SharedPtr<ContextTree> entityTree = doc->GetEntityTree();
			SharedPtr<ContextTree> soundProjectTree = doc->GetSoundProjectTree();
			SharedPtr<ContextTree> overviewTree = doc->GetOverviewTree();

			AddTab(new CContextPage(overviewTree, doc->GetOverviewExporter(), false));

			AddTab(new CContextPage(physicsTree, doc->GetCgfPhysicsExporter(), false));
			AddTab(new CContextPage(textureTree, doc->GetTextureExporter(), false));
			AddTab(new CContextPage(renderMeshTree, doc->GetRenderMeshExporter(), false));
			AddTab(new CContextPage(terrainTree, doc->GetRenderMeshExporter(), false));
			AddTab(new CContextPage(animationTree, doc->GetAnimationExporter(), false));
			AddTab(new CContextPage(navTree, doc->GetNavExporter(), true));
			AddTab(new CContextPage(entityTree, doc->GetEntityExporter(), true));
			AddTab(new CContextPage(soundProjectTree, doc->GetSoundProjectExporter(), true));

			AddTab(new CContextPage(doc->GetContextTreeByType(MemStatContextTypes::MSC_D3D), doc->GetExporterByType(MemStatContextTypes::MSC_D3D), true));
			AddTab(new CContextPage(doc->GetContextTreeByType(MemStatContextTypes::MSC_SoundBuffer), doc->GetExporterByType(MemStatContextTypes::MSC_SoundBuffer), true));

			AddTab(new CContextPage(assetTree, NULL, true));
		}

		if (m_tabs.size() > 1)
		{
			// Delete the first page now that there's guaranteed to be at least one page left.
			DeleteTab(0);
		}
	}
}

void CMemReplayView::OnDraw(CDC* pDC)
{
}

void CMemReplayView::OnSize(UINT nType, int cx, int cy)
{
	CView::OnSize(nType, cx, cy);

	if (m_tabWnd.GetSafeHwnd())
	{
		CRect rcClient;
		GetClientRect(rcClient);

		if (rcClient.Width() >= 100 && rcClient.Height() >= 100)
		{
			m_tabWnd.MoveWindow(rcClient);
		}
	}
}

BOOL CMemReplayView::OnEraseBkgnd(CDC* pDC)
{
	return TRUE;
}
