////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek, 1999-2009.
// -------------------------------------------------------------------------
//  File name:   SummaryTool.cpp
//  Version:     v1.00
//  Created:     07/10/2009 by Steve Barnett.
//  Description: This the implementation of the memReplay summary tool
// -------------------------------------------------------------------------
//  History:
//
//	07/10/2009: Use ReplaySDK to generate summaries of memory logs in
//							spreadsheet form
//
////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include "ReplayHandler.h"

#include <string>

#include "ReplaySDK/CSVCollator.h"

// Statics

static const char* s_overviewExportColumns[] = {
	"Item"
};

static const char* s_textureExportColumns[] = {
	"Texture"
};

static const char* s_renderMeshExportColumns[] = {
	"Name",
	"Type"
};

static const char* s_animationExportColumns[] = {
	"Name",
};

static const char* s_navExportColumns[] = {
	"Name"
};

static const char* s_entityExportColumns[] = {
	"Name",
	"Archetype"
};

static std::string s_symbolFile;

CReplayHandler::CReplayHandler( bool verbose /*= true */ ) : m_verbose( verbose )
{
	SetHooks( this );
}

bool CReplayHandler::LoadFile( const char* pFilename, const char* const pSymbolFilename )
{
	// Store the symbol filename
	s_symbolFile = pSymbolFilename;

	// Load the file
	const char *ext=strrchr(pFilename, '.');
	if (ext && (strcmp(ext, ".mrl") == 0) || (strcmp(ext, ".zmrl") == 0))
	{
		SharedPtr<ReplayLogReader> replayStream = ReplayLogReader::FromFile(pFilename);
		if (replayStream.IsValid())
		{
			m_replayStream = replayStream;

			if ( IsVerbose() )
				printf( "[Log] Reading '%s'\n", pFilename );

			if ( IsVerbose() )
				printf( "[Log] Reading symbol table\n" );
			SharedPtr<SymbolTableBuilder> symTableBuilder = ReadSymbolTable(pFilename);

			if ( IsVerbose() )
				printf( "[Log] Reading context tree\n" );
			SharedPtr<ContextTree> conTree = m_assetTree = ReadContextTree(pFilename);

			if ( IsVerbose() )
				printf( "[Log] Reading validator table\n" );
			SharedPtr<ReplayValidator> validator = m_validation = ReadValidatorTable(pFilename);

			SharedPtr<ContextTreeReplayBuilder> builder;
			
			if (conTree.IsValid() == false)
				builder = ContextTree::CreateBuilder();

			CompositeReplayListener builders;
			if (builder.IsValid())
				builders.AddListener(*builder);

			if (symTableBuilder.IsValid())
				builders.AddListener(*symTableBuilder);

			if ( IsVerbose() )
				printf( "[Log] Reading frame usages\n" );
			bool needsFUBuilding = !ReadFrameUsages(pFilename);

			if (needsFUBuilding)
				builders.AddListener(m_frameUsage);

			if (validator.IsValid() == false)
			{
				m_validation = new ReplayValidator();
				builders.AddListener(*m_validation);
			}

			if ( IsVerbose() )
				printf( "[Log] Replaying logfile\n" );
			if (builders.ListenerCount() > 0)
			{
				m_replayStream->Replay( builders );
			}

			if (builder.IsValid())
			{
				m_assetTree = builder->GetBuiltTree();

				std::string fn = std::string(pFilename) + ".contree";
				FileSerialiser ser(fn.c_str());
				if (ser.IsOpen())
				{
					m_assetTree->Serialise(ser);
				}
			}

			if (validator.IsValid() == false)
			{
				std::string fn = std::string(pFilename) + ".val";
				FileSerialiser ser(fn.c_str());
				if (ser.IsOpen())
				{
					m_validation->Serialise(ser);
				}
			}

			if (symTableBuilder.IsValid())
			{
				m_symbols = symTableBuilder->GetSymbols();
			}

			if (needsFUBuilding)
			{
				std::string fn = std::string(pFilename) + ".fu";
				FileSerialiser ser(fn.c_str());
				if (ser.IsOpen())
					m_frameUsage.Serialise(ser);
			}
		}
		else
		{
			fprintf( stderr, "[Error] Invalid filetype '%s'\n", pFilename );
			return false;
		}

		return true;
	}
	else
	{
		fprintf( stderr, "[Error] Invalid filetype '%s'\n", pFilename );
		return false;
	}
}

bool CReplayHandler::ProcessFile( void )
{
	if ( IsVerbose() )
		printf( "[Log] Gathering physics nodes\n" );
	m_physicsTree = ContextTree::GatherSubTrees(*m_assetTree, "CGF Physics", ContextTreeFunctor::CGFFilter());
	m_physicsTree = ContextTree::GroupChildren(*m_physicsTree, ContextTreeFunctor::NameGrouper());
	m_physicsTree = ContextTree::FilterLeaves(*m_physicsTree, ContextTreeFunctor::PhysicsLeafFilter());
	m_physicsTree = ContextTree::FoldLeaves(*m_physicsTree, ContextTreeFunctor::PhysicsLeafFolder());
	m_physicsTree = ContextTree::GroupChildren(*m_physicsTree, ContextTreeFunctor::NameGrouper());

	if ( IsVerbose() )
		printf( "[Log] Gathering texture nodes\n" );
	m_textureTree = ContextTree::GatherSubTrees(*m_assetTree, "Textures", ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_Texture));
	m_textureTree = ContextTree::GroupChildren(*m_textureTree, ContextTreeFunctor::NameGrouper());

	if ( IsVerbose() )
		printf( "[Log] Gathering render mesh nodes\n" );
	m_renderMeshTree = ContextTree::GatherSubTreesBottomUp(*m_assetTree, "Render Mesh", ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_RenderMesh));
	m_renderMeshTree = ContextTree::GroupChildren(*m_renderMeshTree, ContextTreeFunctor::NameGrouper());
	m_renderMeshTree = ContextTree::FoldLeaves(*m_renderMeshTree, ContextTreeFunctor::MiscLeafFolder());
	m_renderMeshTree = ContextTree::FilterLeaves(*m_renderMeshTree, ContextTreeFunctor::EmptyLeafFilter());

	if ( IsVerbose() )
		printf( "[Log] Gathering terrain nodes\n" );
	m_terrainTree = ContextTree::GatherSubTreesBottomUp(*m_assetTree, "Terrain", 
			ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_Terrain));
	m_terrainTree = ContextTree::Filter(*m_terrainTree,
		MakeNotFilter(
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_MAX),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_CGF),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_MTL),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_DBA),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_CHR),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_LMG),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_AG),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_Texture),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_RenderMesh),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_RenderMeshType),
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_ParticleLibrary),
		             ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_CDF))))))))))))));
	m_terrainTree = ContextTree::GroupChildren(*m_terrainTree, ContextTreeFunctor::NameGrouper());

	if ( IsVerbose() )
		printf( "[Log] Gathering animation nodes\n" );
	m_animationTree = ContextTree::GatherSubTrees(
		*m_assetTree, "Animation", 
		MakeOrFilter(
			ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_DBA),
			MakeOrFilter(
				ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_LMG),
				ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_AG))));
	m_animationTree = ContextTree::GroupChildren(*m_animationTree, ContextTreeFunctor::NameGrouper());

	if ( IsVerbose() )
		printf( "[Log] Gathering navigation nodes\n" );
	m_navTree = ContextTree::GatherSubTrees(*m_assetTree, "Navigation", ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_Navigation));
	m_navTree = ContextTree::GroupChildren(*m_navTree, ContextTreeFunctor::NameGrouper());

	if ( IsVerbose() )
		printf( "[Log] Gathering entity nodes\n" );
	m_entTree = ContextTree::GatherSubTrees(*m_assetTree, "Entity", ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_Entity));
	m_entTree = ContextTree::Filter(*m_entTree,
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_Entity), 
		MakeOrFilter(ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_Other),
		ContextTreeFunctor::TypeFilter(MemStatContextTypes::MSC_ScriptCall))));
	m_entTree = ContextTree::FoldLeaves(*m_entTree, ContextTreeFunctor::NotEntityLeafFolder());

	m_overviewTree = new ContextTree();
	{
		ContextTreeNode* overRoot = m_overviewTree->BeginEdit();
		overRoot->Rename("Overview");

		ContextTreeNode* ovCgfPhys = new ContextTreeNode("CGF Physics", MemStatContextTypes::MSC_Physics);
		if ( m_physicsTree->GetRoot() )
		{
			ovCgfPhys->GetSize() += m_physicsTree->GetRoot()->GetSize();
		}
		overRoot->AddChild(ovCgfPhys);

		ContextTreeNode* ovTextures = new ContextTreeNode("Textures", MemStatContextTypes::MSC_Texture);
		if ( m_textureTree->GetRoot() )
		{
			ovTextures->GetSize() += m_textureTree->GetRoot()->GetSize();
		}
		overRoot->AddChild(ovTextures);

		ContextTreeNode* ovRenderMesh = new ContextTreeNode("Render Mesh", MemStatContextTypes::MSC_RenderMesh);
		if ( m_renderMeshTree->GetRoot() )
		{
			ovRenderMesh->GetSize() += m_renderMeshTree->GetRoot()->GetSize();
		}
		overRoot->AddChild(ovRenderMesh);

		ContextTreeNode* ovAnimation = new ContextTreeNode("Animation", MemStatContextTypes::MSC_AG);
		if ( m_animationTree->GetRoot() )
		{
			ovAnimation->GetSize() += m_animationTree->GetRoot()->GetSize();
		}
		overRoot->AddChild(ovAnimation);

		ContextTreeNode* ovNavigation = new ContextTreeNode("Navigation", MemStatContextTypes::MSC_Navigation);
		if ( m_navTree->GetRoot() )
		{
			ovNavigation->GetSize() += m_navTree->GetRoot()->GetSize();
		}
		overRoot->AddChild(ovNavigation);

		ContextTreeNode* ovEntity = new ContextTreeNode("Entity", MemStatContextTypes::MSC_Entity);
		if ( m_entTree->GetRoot() )
		{
			ovEntity->GetSize() += m_entTree->GetRoot()->GetSize();
		}
		overRoot->AddChild(ovEntity);

		{
			ContextTreeNode* ovOther = new ContextTreeNode("Other", MemStatContextTypes::MSC_Other);
			if ( m_assetTree->GetRoot() )
			{
				ovOther->GetSize() += m_assetTree->GetRoot()->GetSize();
			}

			for (ContextTreeNode* ovChild = overRoot->GetChildren(); ovChild; ovChild = ovChild->GetNextSibling())
				ovOther->GetSize() -= ovChild->GetSize();

			overRoot->AddChild(ovOther);
		}

		{
			ContextTreeNode* ovTotal = new ContextTreeNode("Total", MemStatContextTypes::MSC_Other);
			FrameUsageTracker::ConstIterator lastFrameMain = m_frameUsage.FrameInfoEnd( MemGroups::Main ) - 1;
			FrameUsageTracker::ConstIterator lastFrameRSX = m_frameUsage.FrameInfoEnd( MemGroups::RSX ) - 1;
			SizeInfoGroups usage;
			SizeInfo frameMain = *lastFrameMain;
			SizeInfo frameRSX = *lastFrameRSX;
			frameMain.consumed = frameMain.global;
			frameMain.requested = frameMain.global;
			frameRSX.consumed = frameRSX.global;
			frameRSX.requested = frameRSX.global;
			usage.AddToGroup( MemGroups::Main, frameMain );
			usage.AddToGroup( MemGroups::RSX, frameRSX );
			ovTotal->GetSize() += usage;
			overRoot->AddChild( ovTotal );
		}

		{
			SizeInfoGroups freeSz;

			bool hasRSX = false;

			if ( IsVerbose() )
				printf( "[Log] Gathering RSX nodes\n" );

			{
				if ( m_frameUsage.AllocEvInfoEnd( MemGroups::RSX ) != m_frameUsage.AllocEvInfoBegin( MemGroups::RSX ) )
				{
					FrameUsageTracker::ConstIterator it = m_frameUsage.AllocEvInfoEnd(MemGroups::RSX);
					std::advance(it, -1);
					if (it->requested)
					{
						hasRSX = true;
						freeSz.AddToGroup(MemGroups::RSX, SizeInfo(0, 256 * 1024 * 1024, 256 * 1024 * 1024, 256 * 1024 * 1024) - *it);
					}
				}
			}

			{
				if ( m_frameUsage.AllocEvInfoEnd( MemGroups::Main ) != m_frameUsage.AllocEvInfoBegin( MemGroups::Main ) )
				{
					FrameUsageTracker::ConstIterator it = m_frameUsage.AllocEvInfoEnd(MemGroups::Main);
					std::advance(it, -1);
	
					SizeInfo mainMax = hasRSX 
						? SizeInfo(0, 212 * 1024 * 1024, 212 * 1024 * 1024, 212 * 1024 * 1024)
						: SizeInfo(0, 512 * 1024 * 1024, 512 * 1024 * 1024, 512 * 1024 * 1024);
	
					freeSz.AddToGroup(MemGroups::Main, mainMax - *it);
				}
			}

			if ( IsVerbose() )
				printf( "[Log] Gathering free nodes\n" );
			ContextTreeNode* ovFree = new ContextTreeNode("Free", MemStatContextTypes::MSC_Other);
			ovFree->GetSize() += freeSz;
			overRoot->AddChild(ovFree);
		}

		m_overviewTree->EndEdit();

	}

	return true;
}

bool CReplayHandler::ExportSummary( const char* const pFilename )
{
	// Exporters
	CGFPhysicsCSVExporter cgfPhysExporter;
	RangedContextExporter textureExporter(1, 2, s_textureExportColumns, false);
	RangedContextExporter renderMeshExporter(1, 3, s_renderMeshExportColumns, false);
	RangedContextExporter animationExporter(1, 2, s_animationExportColumns, false);
	RangedContextExporter navExporter(2, 3, s_navExportColumns, false);
	RangedContextExporter entityExporter(1, 3, s_entityExportColumns, true);
	RangedContextExporter overviewExporter(1, 2, s_overviewExportColumns, false);

	if ( IsVerbose() )
		printf( "[Log] Opening file for export '%s'\n", pFilename );

	ExcelExport xls;
	if ( xls.Open( pFilename ) )
	{
		int style = xls.BeginStyle();
		xls.EndStyle();

		// Export data
		if ( IsVerbose() ) printf( "[Log] Physics export\n" );
		if ( m_physicsTree.IsValid() )
			ExportWorksheet( xls, m_physicsTree, &cgfPhysExporter );

		if ( IsVerbose() ) printf( "[Log] Texture export\n" );
		if ( m_textureTree.IsValid() )
			ExportWorksheet( xls, m_textureTree, &textureExporter );

		if ( IsVerbose() ) printf( "[Log] Render mesh export\n" );
		if ( m_renderMeshTree.IsValid() )
			ExportWorksheet( xls, m_renderMeshTree, &renderMeshExporter );

		if ( IsVerbose() ) printf( "[Log] Terrain export\n" );
		if ( m_terrainTree.IsValid() )
			ExportWorksheet( xls, m_terrainTree, &renderMeshExporter );

		if ( IsVerbose() ) printf( "[Log] Animation export\n" );
		if ( m_animationTree.IsValid() )
			ExportWorksheet( xls, m_animationTree, &animationExporter );

		if ( IsVerbose() ) printf( "[Log] Nav export\n" );
		if ( m_navTree.IsValid() )
			ExportWorksheet( xls, m_navTree, &navExporter );

		if ( IsVerbose() ) printf( "[Log] Entity export\n" );
		if ( m_entTree.IsValid() )
			ExportWorksheet( xls, m_entTree, &entityExporter );

		if ( IsVerbose() ) printf( "[Log] Overview export\n" );
		if ( m_overviewTree.IsValid() )
			ExportWorksheet( xls, m_overviewTree, &overviewExporter );

		xls.Close();

		if ( IsVerbose() )
			printf( "[Log] Finished exporting\n" );

		return true;
	}
	else
	{
		fprintf( stderr, "[Error] Failed to open output file '%s'\n", pFilename );
		return false;
	}
}

void CReplayHandler::ExportWorksheet(ExcelExport& xls, const SharedPtr<ContextTree>& contextTree, ICSVExporter* exporter )
{
	if ( contextTree->GetRoot() )
	{
		ExcelExportWorksheet ws(xls, contextTree->GetRoot()->GetName());
		if (exporter)
		{
			WorksheetCSVCollator collator(ws);
			exporter->Export(collator, contextTree->GetRoot());
		}
	}
}


SharedPtr<SymbolTableBuilder> CReplayHandler::ReadSymbolTable(const char* filename)
{
	std::string symFilename = std::string(filename) + ".sym";
	if (GetFileAttributesA(symFilename.c_str()) != INVALID_FILE_ATTRIBUTES)
	{
		m_symbols = SymbolHelper::FromFile(symFilename.c_str());
		if (m_symbols.IsValid())
			return SharedPtr<SymbolTableBuilder>();
	}

	return new SymbolTableBuilder(symFilename);
}

SharedPtr<ContextTree> CReplayHandler::ReadContextTree(const char* filename)
{
	std::string fn = std::string(filename) + ".contree";

	FileDeserialiser ser(fn.c_str());
	if (ser.IsOpen())
		return ContextTree::Deserialise(ser);
	
	return SharedPtr<ContextTree>();
}

SharedPtr<ReplayValidator> CReplayHandler::ReadValidatorTable(const char* filename)
{
	std::string fn = std::string(filename) + ".val";

	FileDeserialiser ser(fn.c_str());
	if (ser.IsOpen())
	{
		SharedPtr<ReplayValidator> val = new ReplayValidator();

		if (val->Deserialise(ser))
			return val;
	}

	return SharedPtr<ReplayValidator>();
}

bool CReplayHandler::ReadFrameUsages(const char* filename)
{
	std::string fn = std::string(filename) + ".fu";
	FileDeserialiser ser(fn.c_str());
	if (ser.IsOpen())
		return m_frameUsage.Deserialise(ser);

	return false;
}

void CReplayHandler::InvokeOnMainImpl( InvokeHandler handler, void* user )
{
	// Just invoke the function
	(handler)( user );
}

void CReplayHandler::ShowMessageImpl( const TCHAR* msg, const TCHAR* title )
{
	printf( "[Message:%s] %s\n", title, msg );
}

std::string CReplayHandler::OpenFileImpl( const char* filter )
{
	// Should only be called in the command line version of the tool to get the symbol file
	(void) filter;
	return s_symbolFile;
}

std::string CReplayHandler::OpenFolderImpl( const TCHAR* title )
{
	// Should only be called in the command line version of the tool to get the symbol file
	fprintf( stdout, __FUNCTION__ " %s\n", title );
	return std::string();
}

void CReplayHandler::RunProgressTaskImpl(IProgressTask& task)
{
	task.Start();
}
