#include "stdafx.h"
#include "BuildSegmentedCompressionSpace.h"

static const int DIM = 4;

struct Pt
{
	Pt()
	{
		for (int i=0; i<DIM; i++)
			val[i] = 0;
	}
	bool operator<( const Pt& p ) const
	{
		for (int i=0; i<DIM; i++)
		{
			if (val[i] < p.val[i])
				return true;
			else if (val[i] > p.val[i])
				return false;
		}
		return false;
	}
	int val[DIM];
};

typedef std::pair<Pt, int> PtC;

static void flt2int( const SVec3& v, Pt& p, int dim, float quanta, const int * center )
{
	for (int i=0; i<dim; i++)
		p.val[i] = int(v[i] / quanta) - center[i];
}

static int CountInBlock( const PtC* quantized, size_t nSamples, float quanta, int dim, const int * mn, const int * mx, const int * center )
{
	int countIn = 0;
	for (size_t s=0; s<nSamples; s++)
	{
		bool in = true;
		for (int i=0; i<dim; i++)
			in &= (quantized[s].first.val[i] >= mn[i] && quantized[s].first.val[i] < mx[i]);
		countIn += in * quantized[s].second;
	}
	return countIn;
}

struct BuildTodo
{
	int ofs[DIM];
	int linkRow, linkCol;
	int outerShellSize;
};

struct BuildCtx
{
	std::vector< std::vector<int> > rows;
	std::vector< std::vector<int> > links;

	std::queue<BuildTodo> todo;
};

int BuildBlock( BuildCtx * ctx, const PtC* quantized, size_t nSamples, float quanta, int dim, int steps, const int * center, const int * ofs, int outerShellSize )
{
	int mod[DIM];
	for (int i=0; i<dim; i++)
	{
		mod[i] = steps;
		for (int j=0; j<i; j++)
			mod[i] *= steps;
	}

	int totSteps = 1;
	for (int i=0; i<steps; i++)
		totSteps *= steps;

	std::vector<int> row;
	row.resize(totSteps);

	int maxCount = 1;
	int minCount = 0x7fffffff;
	for (int n=0; n<totSteps; n++)
	{
		int mn[DIM], mx[DIM];
		for (int i=0; i<dim; i++)
		{
			mn[i] = ofs[i] + (outerShellSize/steps) * ((n % mod[i]) / (mod[i]/steps));
			mx[i] = mn[i] + outerShellSize/steps;
		}
		int count = CountInBlock(quantized, nSamples, quanta, dim, mn, mx, center);
		count += (count == 0);
		row[n] = count;
		if (count > maxCount)
			maxCount = count;
		if (count < minCount)
			minCount = count;
	}

	int idx = -1;
	if (maxCount > 2*minCount)
	{
		idx = ctx->rows.size();
		ctx->rows.push_back(row);
		
		std::vector<int> links;
		links.resize(totSteps, -1);
		ctx->links.push_back(links);

		if (outerShellSize > 1)
		{
			for (int n=0; n<totSteps; n++)
			{
				if (row[n] < totSteps*totSteps/2)
					continue;
				BuildTodo todo;
				for (int i=0; i<dim; i++)
					todo.ofs[i] = ofs[i] + (outerShellSize/steps) * ((n % mod[i]) / (mod[i]/steps));
				todo.outerShellSize = outerShellSize / steps;
				todo.linkRow = idx;
				todo.linkCol = n;
				ctx->todo.push(todo);
			}
		}
	}
	return idx;
}

void BuildSegmentedCompressionSpace(FILE* f, const SVec3* samples, size_t nSamples, float quanta, int dim, int steps)
{
	if (NULL == samples || 0 == nSamples)
		return;
	SVec3 wrlCenter(0,0,0);
	for (size_t s=0; s<nSamples; s++)
		wrlCenter = wrlCenter + samples[s];
	wrlCenter = wrlCenter / nSamples;

	int center[DIM];
	for (int i=0; i<dim; i++)
		center[i] = wrlCenter[i] / quanta;

	std::vector<PtC> pts;
	{
		std::map<Pt, int> m;
		for (size_t s=0; s<nSamples; s++)
		{
			Pt p;
			flt2int( samples[s], p, dim, quanta, center );
			m[p]++;
		}
		std::copy(m.begin(), m.end(), back_inserter(pts));
	}

	int outerShellSize = steps;
	while (true) 
	{
		int mnp = -outerShellSize/2;
		int mxp = mnp + outerShellSize;
		int mn[DIM], mx[DIM];
		for (int i=0; i<dim; i++)
		{
			mn[i] = mnp;
			mx[i] = mxp;
		}
		int countIn = CountInBlock(&pts[0], pts.size(), quanta, dim, mn, mx, center);

		fprintf(stderr, "%f -> %d/%d\n", outerShellSize*quanta, countIn, nSamples);

		if (countIn >= nSamples*0.99f)
			break;

		outerShellSize *= steps;
	}

	BuildCtx ctx;
	int iniofs[DIM];
	for (int i=0; i<dim; i++)
		iniofs[i] = -outerShellSize/2;
	BuildBlock( &ctx, &pts[0], pts.size(), quanta, dim, steps, center, iniofs, outerShellSize );

	while (!ctx.todo.empty())
	{
		BuildTodo todo = ctx.todo.front();
		ctx.todo.pop();

		int r = BuildBlock( &ctx, &pts[0], pts.size(), quanta, dim, steps, center, todo.ofs, todo.outerShellSize );
		fprintf(stderr, "build %d/%d,%d -> %d\n", todo.linkRow, ctx.rows.size(), todo.linkCol, r);
		if (r > 0)
		{
			ctx.links[todo.linkRow][todo.linkCol] = r;
		}
	}

	int totSteps = 1;
	for (int i=0; i<steps; i++)
		totSteps *= steps;
	for (int i=0; i<ctx.rows.size(); i++)
	{
		if (i == 0)
			fprintf(f, "<r oss=\"%d\" c=\"%d,%d,%d\">\n", outerShellSize, center[0], center[1], center[2]);
		else
			fprintf(f, "<r>\n");
		fprintf(f, "  <c>");
		for (int j=0; j<totSteps; j++)
			fprintf(f, "%s%d", j?",":"", ctx.rows[i][j]);
		fprintf(f, "</c>\n");
		fprintf(f, "  <l>");
		for (int j=0; j<totSteps; j++)
			fprintf(f, "%s%d", j?",":"", ctx.links[i][j]);
		fprintf(f, "</l>\n");
		fprintf(f, "</r>\n");
	}
}
