//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// Simple html generator 
// ---------------------------------------------------------------------------
// History:
// - June 13 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////

#ifndef __HTML_GEN 
#define __HTML_GEN

#include <fstream>
#include <string> 
#include <vector>
#include <map>

#include <boost/format.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>

using namespace boost;

namespace html 
{
  class document;
  class section;
  class list;
  class list_item;
  class table;
  class row;
  class column;
  class link;
  class paragraph;

  // Base class for html elements. Follows similar concepts as almost every other
  // class hierachy representing documents built up in tree hierachy.
  class element
  {
  protected:
    // Child elements other elements are stored in vector
    // of shared pointers
    typedef std::vector< shared_ptr<element> > child_array_t;
  private:
    // Links to other elements are stored in a map of weak ptrs
    // indexed by their name 
    typedef std::map< std::string, weak_ptr<element> > link_map_t;

    // Attributes are key/value pairs represented by string
    typedef std::map< std::string, std::string > attr_map_t;

    // The vector of children 
    child_array_t m_children;

    // The links to other elements 
    link_map_t m_links;

    // The attributes of the element
    attr_map_t m_attributes;

    // The optional payload of the element
    std::string m_payload;

  protected:
    // serialize this element out into oblivion
    friend bool serialize(std::ostream&, shared_ptr<element>); 
    virtual bool serialize(std::ostream&) const; 

    // Constructs elements 
    element(); 

  public:

    // Set an attribute to a given value
    element& set_attribute(const std::string&, const std::string&);
    element& set_attribute(const char* attr, const boost::format& fmt)
    { return set_attribute(std::string(attr), fmt.str()); }
    element& set_attribute(const char* attr, const char* value)
    { return set_attribute(std::string(attr), std::string(value)); }

    // Retrieve the value of an attribute (returns zero if the
    // attribute hasn't been set"
    const char* attribute(const std::string&) const;
    const char* attribute(const char* attr) const 
    { return attribute(std::string(attr)); }

    // Establish a weak link to another element under the given name
    element& set_link(const std::string&, weak_ptr<element>);
    element& set_link(const char* attr, weak_ptr<element> elem)
    { return set_link(std::string(attr), elem); }

    // Retrieve a named link 
    weak_ptr<element> link(const std::string&) const;
    weak_ptr<element> link(const char* attr) const 
    { return link(std::string(attr)); }

    // Add a child to this element 
    void add_child(shared_ptr<element> child) 
    { m_children.push_back(child); }
    
    // Retrieve the child iterators
    child_array_t::const_iterator children_begin() const
    { m_children.begin(); }
    child_array_t::const_iterator children_end() const
    { m_children.end(); }

    // Retrieve the payload 
    std::string& payload() { return m_payload; }
    const std::string& payload() const { return m_payload; }

    // Assigns a string to the element. This may or maynot make sense
    // depending on the concrete element implementation. 
    virtual element& operator<< (const std::string&); 
    element& operator<< (const char*); 
    element& operator<< (const boost::format&); 

    // create a section within the current context.This may or maynot
    // make sense depending on the concrete element implementation.
    virtual section& create_section(const std::string&); 

    // Create a table within the current context. This may or maynot make sense
    // depending on the concrete element implementation. 
    virtual table& create_table(); 

    // Create a list with the current context. This may or maynot make
    // sense depending on the concrete element implementation.
    virtual list& create_list(); 

    // Create a list item with the current context. This may or maynot make
    // sense depending on the concrete element implementation.
    virtual list_item& create_list_item(); 

    // Create a link to another element within the current
    // context. This may or maynot make sense depending on the
    // concrete element implementation.
    virtual html::link& create_link(const std::string&); 
  }; 

  // The document class that represents an html document. 
  // The html document can be used to create child elements within the
  // document that get serialized out
  class document : public element
  {
    // The title of the document
    std::string m_title; 

    // The number of sections within the document 
    signed m_section_index;

    // The sections within the document 
    std::vector< weak_ptr<section> > m_sections;

  protected:
    // Serialization implementation of document
    bool serialize(std::ostream&) const; 

  public:
    using element::operator<<; 

    // Constructor (initializes the document with a title)
    document(const std::string&); 

    // create a section within the current context 
    section& create_section(const std::string&, bool toc=true); 

    // Assigns a string to the document. Creates a paragraph and
    // attaches it as a child to the document
    element& operator<< (const std::string&); 
  }; 

  // A element representing a section within an html document
  class section : public element
  {
  protected:
    // Serialization implementation of section
    bool serialize(std::ostream&) const; 

    // The depth within the document structure
    int m_depth;

  public:
    using element::operator<<; 

    // Constructor
    section(int depth = 0); 

    // Change the depth 
    int& depth() { return m_depth; } 

    // create a section within the current context.
    section& create_section(const std::string&); 

    // Create a table within the current context. 
    table& create_table(); 

    // Assigns a string to the document. Creates a paragraph and
    // attaches it as a child to the document
    element& operator<< (const std::string&); 
  };


  // A element representing a list within the html document 
  class list : public element
  {
  protected:
    // Serialization implementation of section
    bool serialize(std::ostream&) const; 

  public:
    // Constructor
    list(); 

    // Create a list item with the current context. 
    list_item& create_list_item(); 
  };

  // An element representing a list item within an list 
  class list_item : public element
  {
    // Serialization implementation of section
    bool serialize(std::ostream&) const; 

  public:
    // Constructor
    list_item(); 

    // Assigns a string to the document. Creates a paragraph and
    // attaches it as a child to the document
    element& operator<< (const std::string&); 
  }; 


  // A element representing an html table
  class table : public element
  {
  protected:
    // Serialization implementation of table
    bool serialize(std::ostream&) const; 
  public:
    table(); 
    // create a row  within the table
    row& create_row(); 
  };

  // An element representing a row in an html table
  class row : public element
  {
  protected:
    // Serialization implementation of row
    bool serialize(std::ostream&) const; 

  public:
    // Constructor
    row(); 

    // create a row  within the table
    column& create_column(bool header=false); 
    column& create_column(
      const std::string& content, bool header=false); 
    column& create_column(
      const char* content, bool header=false)
    { return create_column(std::string(content),header); }
    column& create_column(
      const boost::format& content, bool header=false)
    { return create_column(content.str(),header); }


  };

  // An element representing a column entry in an html table
  class column : public element
  {
    // Is the column a table header?
    bool m_header : 1; 

  protected:
    // Serialization implementation of column
    bool serialize(std::ostream&) const; 

  public:
    column(); 

    // Set or clear the header flag
    void set_header(bool header) { m_header = header; }
  };

  // generates a table class
  class link : public element
  {
  protected:
    // Serialization implementation of link
    bool serialize(std::ostream&) const; 

  public:
    // Constructur
    link(); 
  };

  // generates a table class
  class paragraph : public element
  {
  protected:
    // Serialization implementation of link
    bool serialize(std::ostream&) const; 

  public:
    // Constructur
    paragraph(); 
  };

  // serialization method
  bool serialize(std::ostream&, shared_ptr<element>); 
}; 

#endif 
