//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// mtrace database for storing allocation information from cryengine3
// ps3 applications
// ---------------------------------------------------------------------------
// History:
// - June 13 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////
#ifndef _MTRACEDB_H
#define _MTRACEDB_H

#include <boost/cstdint.hpp>

#include <vector>
#include <map>
#include <set>
#include <string>
#include <fstream>
#include <iostream>


#include <boost/crc.hpp>
#include <boost/program_options/cmdline.hpp>
#include <boost/program_options/environment_iterator.hpp>
#include <boost/program_options/eof_iterator.hpp>
#include <boost/program_options/errors.hpp>
#include <boost/program_options/option.hpp>
#include <boost/program_options/options_description.hpp>
#include <boost/program_options/parsers.hpp>
#include <boost/program_options/positional_options.hpp>
#include <boost/program_options/value_semantic.hpp>
#include <boost/program_options/variables_map.hpp>
#include <boost/program_options/version.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/iterator_adaptors.hpp>
#include <boost/thread.hpp>
#include <boost/range.hpp>

using namespace boost; 

#include <boost/foreach.hpp>
#define foreach BOOST_FOREACH

#if defined(NDEBUG)
# define EXIT() std::exit(EXIT_FAILURE)
#else
# define EXIT() std::abort()
#endif

namespace debug { 
  static inline void check(bool value, const char* test, 
    const char* file, unsigned line) 
  { 
    if (!value) 
    {
      std::cerr << file << ':' << line << " assertion: " << test 
                << std::endl; 
    }
  } 
} 

#define DEBUG_CHECK(x) debug::check((x), #x, __FILE__, __LINE__);

namespace mtracedb 
{
  class memory_state_t; 
  class memory_op_range;
  class memory_monitor_t;


  // Emptry stream utility class 
  class null_stream : public std::ostream
  {
    public:
    null_stream() : std::ios(NULL), std::ostream(NULL) { }
  };

  // Simple log level handling 
  enum ELogLevel 
  { 
    LOG_NOISE = 1,
    LOG_DEBUG = 2,
    LOG_INFO  = 3, 
    LOG_WARNING = 4,
    LOG_ERROR = 5, 
  }; 
  // Conversion methods to and from string 
  ELogLevel from_string(const char* string);
  inline ELogLevel from_string(const std::string& s) { return from_string(s.c_str()); };
  const char* to_string(ELogLevel level);

  // Simple logging method -> performs formatted output like printf 
  // Note: newline is appended automatically
  void log(ELogLevel,const char*, ...);
  std::ostream &out(ELogLevel);
  std::ostream &cerr();

  // Log Callback handling 
  // Log call backs registered will recieve log messages over the "on_log_msg" routine"
  struct log_callback 
  { 
    // The constructor. 
    // Note: Will automatically register itself with the logging system 
    log_callback();
    // The destructor 
    // Note: Will automatically remove itself with the logging system 
    virtual ~log_callback();
    // virtual m
    virtual void on_log_msg(const std::string&) = 0; 
  }; 

  // Global options for the database and general operation are grouped
  // here
  struct options
  { 
    // The path to the ppu-lv2-addr2line executable
    std::string addr2line_exe; 
    // The path to the ppu-lv2-gdb executable
    std::string gdb_exe; 
    // The path to the ppu-lv2-size executable
    std::string size_exe; 
    // The path to the ppu-lv2-size executable
    std::string dot_exe; 
    // The path to the ps3bin executable
    std::string ps3bin_exe; 
    // The path to the gnuplot executable 
    std::string gnuplot_exe; 
    // The path the ps3 executable
    std::string elf_file; 
    // The path the database directory
    std::string data_directory; 
    // The path to the configuration file 
    std::string config_file; 
    // The path to the log file (if empty, no logging will be performed)
    std::string log_file; 
    // The path to ipconfig file. If set, the ip address of the host will be
    // written into this file
    std::string ipconfig_file; 
    // The current log level 
    ELogLevel log_level; 
    // Flag describing if the ps3bin executable should be used instead of
    // ppu-lv2 toolchain. 
    bool use_ps3bin; 
  }; 
  options& get_options(); 

  // The global configuration object. Contains a mapping of key value
  // pairs depending on type 
  typedef std::map<std::string, std::string> config_map_t;

  // Retrieve the config map instance
  config_map_t& config();   

  // Parse the command line 
  void parse_command_line(int argc, char* argv[],
    program_options::options_description&); 

  // The definitions for crymodule
  enum ECryModule
  {
    eCryM_3DEngine = 0,
    eCryM_Action = 1,
    eCryM_AISystem = 2,
    eCryM_Animation = 3,
    eCryM_EntitySystem = 4,
    eCryM_Font = 5,
    eCryM_Input = 6,
    eCryM_Movie = 7,
    eCryM_Network = 8,
    eCryM_Physics = 9,
    eCryM_ScriptSystem = 10,
    eCryM_SoundSystem = 11,
    eCryM_System = 12,
    eCryM_Game = 13,
    eCryM_Render = 14,
    eCryM_Launcher = 15,
    eCryM_Num = 16,
  };
  const char* to_string(ECryModule);
  

  // Statistics gathered for each module seperately 
  struct ModuleStats
  {
    // The current memory footprint in bytes
    size_t current_memory; 
    // The peak memory footprint 
    size_t peak_memory; 
    // The system memory footprint for the module (text+bss+data)
    size_t system_memory; 
    // Total Ammount of memory allocated during trace
    uint64_t allocated;
    // Total Ammount of memory freed during trace 
    uint64_t freed;
    // Total number of memory allocations
    uint64_t num_allocations;
  }; 

  // allocations statistics gathered by mtrace  
  struct AllocationStats
  {
    // The number of bytes allocated 
    uint64_t allocated_memory; 
    // The number of bytes freed 
    uint64_t freed_memory; 
    // The number of calls to malloc()
    size_t malloc_calls;
    // The number of calls to calloc()
    size_t calloc_calls;
    // The number of calls to realloc()
    size_t realloc_calls;
    // The number of calls to memalign()
    size_t memalign_calls;
    // The number of calls to free()
    size_t free_calls;
    // The system local thread id 
    uint16_t thread_local_id;
    // The thread id (~0U for accumulated system wide allocation statistics)
    uint32_t thread_id;
    // The thread id (~0U for accumulated system wide allocation statistics)
    uint32_t thread_sys_id;
    // Flag describing if tracing is enabled for this thread 
    bool   thread_recording;
  }; 

  // system wide statistics gathered by mtrace 
  struct SystemStats
  {
    // The overall allocations statistics for allocations 
    AllocationStats allocations; 
    // The statistics for each single module
    ModuleStats modules[eCryM_Num];
    // Statistics for code that do not belong to a module
    ModuleStats unknown;
  };  

  // system wide statistics - brief 
  struct BriefSystemStats
  {
    // The number of allocated bytes for the brief memstats 
    uint32_t total; 
    // The number of currently allocated bytes for each module
    uint32_t modules[eCryM_Num];
    // The total number of allocated bytes that are not accountable for any module
    uint32_t unknown; 
  };  

  // The possible types of memory operation 
  enum MEMORY_OPERATION 
  {
    MO_CALLOC=0,       // call to calloc
    MO_MALLOC=1,       // call to malloc
    MO_FREE=2,         // call to free
    MO_REALLOC=3,      // call to realloc
    MO_MEMALIGN=4,     // call to memalign 
    MO_SNAPSHOT=5,     // memory snapshot has been made
    MO_STAT_REQUEST=6, // a memory update request has been made
    MO_UNKNOWN=-1,     // unknown == error!
  }; 

  // tags sent by the ps3 application, keep in sync with
  // ps3launcher/mtrace.cpp
  enum MEMORY_FUNCTION
  {
    FN_NONE     = 0,
    FN_calloc   = 1,
    FN_malloc   = 2,
    FN_free     = 4,
    FN_realloc  = 8,
    FN_memalign = 16,
    FN_END      = 0x7fffffff
  };

  // The shorts character keys that are used to identify 
  // an memory operation type 
  static const char MEMORY_KEYS[] = 
  {
    'C', 	// call to calloc
    'M',  // call to malloc
    'F',  // call to free
    'R', 	// call to realloc
    'A',	// call to memalign 
  }; 

  // Various utility functions
  namespace util
  {
    // This function will Swap the endianess of any given input as
    // long as it is pod. (Warning, on visual studio it is actually
    // possible to call this 
    template<typename T> 
    inline T endian_swap(T input)
    {
      union { T src; char dst[sizeof T]; } a,b; 
      a.src = input; 
      for (unsigned i=0; i<sizeof T; ++i)
        b.dst[i] = a.dst[sizeof T - 1 - i]; 
      return b.src; 
    }

    // Specialization for byte types which require no swapping 
    template<>
    inline uint8_t endian_swap(uint8_t input)	{ return input;	}
    template<>
    inline int8_t endian_swap(int8_t input)	{ return input;	}

    // Char->Memory_Op type conversion helper
    inline MEMORY_OPERATION from_char(char type)
    {
      switch(type)
      {
        case 'C': 	// call to calloc
          return MO_CALLOC;
        case 'M':   // call to malloc
          return MO_MALLOC;
        case 'F':   // call to free
          return MO_FREE;
        case 'R': 	// call to realloc
          return MO_REALLOC;
        case 'A': 	// call to memalign 
          return MO_MEMALIGN;
      }
      return MO_UNKNOWN; 
    };

    // Char->Memory_Op type conversion helper
    inline MEMORY_OPERATION from_func(uint32_t type)
    {
      switch(type)
      {
        case FN_calloc: 	// call to calloc
          return MO_CALLOC;
        case FN_malloc:   // call to malloc
          return MO_MALLOC;
        case FN_free:   // call to free
          return MO_FREE;
        case FN_realloc: 	// call to realloc
          return MO_REALLOC;
        case FN_memalign: 	// call to memalign 
          return MO_MEMALIGN;
      }
      return MO_UNKNOWN; 
    };

    // Char->Memory_Op type conversion helper
    inline const char* to_string(uint32_t type)
    {
      switch(type)
      {
        case FN_calloc: 	// call to calloc
          return "MO_CALLOC";
        case FN_malloc:   // call to malloc
          return "MO_MALLOC";
        case FN_free:   // call to free
          return "MO_FREE";
        case FN_realloc: 	// call to realloc
          return "MO_REALLOC";
        case FN_memalign: 	// call to memalign 
          return "MO_MEMALIGN";
      }
      return "MO_UNKNOWN"; 
    };

    // Char->Memory_Op type conversion helper
    inline const char* to_string(MEMORY_OPERATION type)
    {
      switch(type)
      {
        case MO_CALLOC: 	// call to calloc
          return "MO_CALLOC";
        case MO_MALLOC:   // call to malloc
          return "MO_MALLOC";
        case MO_FREE:   // call to free
          return "MO_FREE";
        case MO_REALLOC: 	// call to realloc
          return "MO_REALLOC";
        case MO_MEMALIGN: 	// call to memalign 
          return "MO_MEMALIGN";
      }
      return "MO_UNKNOWN"; 
    };

    // Convert a ecrymodule enum to string
    inline const char* to_string(ECryModule type)
    {
      switch(type)
      {
        case eCryM_3DEngine:
          return "eCryM_3DEngine";
        case eCryM_Action:
          return "eCryM_Action";
        case eCryM_AISystem:
          return "eCryM_AISystem";
        case eCryM_Animation:
          return "eCryM_Animation";
        case eCryM_EntitySystem:
          return "eCryM_EntitySystem";
        case eCryM_Font:
          return "eCryM_Font";
        case eCryM_Input:
          return "eCryM_Input";
        case eCryM_Movie:
          return "eCryM_Movie";
        case eCryM_Network:
          return "eCryM_Network";
        case eCryM_Physics:
          return "eCryM_Physics";
        case eCryM_ScriptSystem:
          return "eCryM_ScriptSystem";
        case eCryM_SoundSystem:
          return "eCryM_SoundSystem";
        case eCryM_System:
          return "eCryM_System";
        case eCryM_Game:
          return "eCryM_Game";
        case eCryM_Render:
          return "eCryM_Render";
        case eCryM_Launcher:
          return "eCryM_Launcher";
        case eCryM_Num:
          return "unknown/os";
      }
      return "MO_UNKNOWN"; 
    };
  };

  // Use 32bit unsigned integers for any addresses maintaned by the
  // mtrace database 
  typedef uint32_t address_t;

  // Class that can resolve an address of the instruction pointer to a
  // source:line combination. Internally, the class executes
  // 'ppu-lvl2-addr2line.exe' to recieve the source line information.
  // To prevent unnessecary communication with the spawned process, it
  // caches previous results in a map.  Note: To retrieve an instance
  // of this class, call address_resolver::instance().
  class address_resolver
  {
    // The mapping of addresses to code lines 
    typedef std::map<mtracedb::address_t, std::string> addr_map_t;

    // The cache  of already resolved addresses
    addr_map_t m_resolved_cache; 

    // Constructor 
    address_resolver(); 

    // Initialize the resolver 
    bool init(); 

    // resolve the address with ps3bin 
    std::string& resolve_ps3bin(mtracedb::address_t addr); 

    // resolve the address with addr2line
    std::string resolve_addr2line(mtracedb::address_t addr); 

  public:
    // Resolves an address to the corresponding source line 
    std::string& resolve(mtracedb::address_t addr);

    // Static instance 
    static address_resolver& instance(); 
  }; 

  // Class that can resolve an address of the instruction pointer to a
  // function combination. Internally, the class executes
  // 'ppu-lvl2-gdb.exe' to recieve the function name per instruction address.
  // To prevent unnessecary communication with the spawned process, it
  // caches previous results in a map.  Note: To retrieve an instance
  // of this class, call function_resolver::instance().
  class function_resolver
  {
    // The mapping of addresses to code lines 
    typedef std::map<mtracedb::address_t, std::string> addr_map_t;

    // The cache  of already resolved addresses
    addr_map_t m_resolved_cache; 

    // Constructor 
    function_resolver(); 

    // Destructor
    ~function_resolver();

    // Initialize the resolver 
    bool init(); 

  public:
    // Resolves an address to the corresponding source line 
    std::string& resolve(mtracedb::address_t addr);

    // Static instance 
    static function_resolver& instance(); 
  }; 

  // that reads in the ps3 elf file and extracts the size of the 
  // text segment, bss segment and data segment 
  class segment_stats
  {
    // The size of the text segment in bytes 
    uint64_t m_text_size;

    // The size of the bss segment in bytes 
    uint64_t m_bss_size;

    // The size of the data segment in bytes 
    uint64_t m_data_size;

    // Initialize the segment statistics
    bool init(); 

    // Constructor for the segment stats
    segment_stats(); 

  public:

    // Retrieve the size of the text segment in bytes 
    uint64_t text_size() const { return m_text_size; }

    // The size of the bss segment in bytes 
    uint64_t bss_size() const { return m_bss_size; }

    // The size of the data segment in bytes 
    uint64_t data_size() const { return m_data_size; }
    
    // Static instance 
    static const segment_stats& instance(); 
  }; 

  // A structure representing a callstack. A callstack is a singly
  // linked list of stack addresses.
  struct callstack_info_t
  {
    // The return addresses of the callstack entry
    std::vector<address_t> addresses; 

    // Hash the callstack via boost crc32 (and hope we never have any
    // collisions)
    inline uint32_t hash() const 
    {
      const std::size_t num_addrs = addresses.size();
      if ( num_addrs ) 
      {
        boost::crc_optimal<32, 0x04C11DB7> crc;
        crc.process_bytes( 
          &(*addresses.begin()), num_addrs * sizeof(address_t) );
        return crc.checksum();
      }
      return 0; 
    }
  };

  // Structure that holds information about an memory operation
  // (allocation, deallocation, reallocation, snapshot, request)
  struct memory_stream_element_t
  {
    // The actual type of the memory operation 
    MEMORY_OPERATION type; 
    // The allocation id 
    uint64_t id; 
    // The id of the thread in which the memory operation was performed
    uint32_t thread_id; 
    // The address of the allocation (must not be null!)
    address_t primary; 
    // The address of the allocation (must not be null!)
    address_t secondary; 
    // The size of the allocation (might be null if unknown or unchanged?!)
    size_t size; 
    // The callstack at the point of allocation
    callstack_info_t callstack; 
    // If the operation is a snapshot, this is the name of the snapshot to be
    // made
    char snapshot[64]; 

    bool operator< (const memory_stream_element_t& b) const
    { return size > b.size; }
  };

  // Writable file storage backend for memory operations. 
  class file_writer_t
  {
  private:
    // The file containing all memory operations
    std::ostream& m_mem_op_file;

  public:
    // Constructor. Takes the path to the file to written as an argument
    file_writer_t(std::ostream& out); 

    // Append an memory operation to the file storage. 
    bool write(const memory_stream_element_t &);
  };

  // Readable storage backend for memory operations. 
  class file_reader_t
  {
  private:
    // The file containing all memory operations
    std::ifstream m_mem_op_file;

  public:
    // Constructor. Takes the path to the file to be read as an argument
    file_reader_t(const std::string&); 

    // Read the next memory operation from storage
    bool read(memory_stream_element_t &);

    // Retrieve the size of the file
    std::streampos file_size(); 

    // Retrieve the read position within the file 
    std::streampos current(); 
  };

  // The memory operation structure
  struct memory_op_t
  {
    // The actual type of the memory operation 
    MEMORY_OPERATION type; 
    // The allocation id 
    size_t id; 
    // The id of the thread in which the memory operation was performed
    uint32_t thread_id; 
    // The address of the allocation (must not be null!)
    address_t primary; 
    // The address of the allocation (must not be null!)
    address_t secondary; 
    // The size of the allocation (might be null if unknown or unchanged?!)
    size_t size; 
    // The crc of the callstack at the point of allocation.
    uint32_t callstack; 
    // The module this operation belongs to 
    ECryModule module; 
  };

  // A simple snapshot structure
  // name and operation id of the snapshot
  struct snapshot_t 
  { 
    // The name of the snapshot 
    std::string name; 

    // The memory id on which the snapshot was received
    size_t id; 
  }; 

  // Structure representing an active allocation 
  struct allocation_t
  {
    // The operation id of the allocation 
    uint64_t id; 

    // The address of the allocation 
    address_t address;
    
    // The size of the allocation 
    size_t size; 

    // The size of the allocation 
    ECryModule module; 

    
    // Default constructor
    allocation_t() 
      : id(), 
        address(), 
        size(), 
        module(eCryM_Num)
    {}
    
    // Initializing  constructor
    allocation_t(
      uint64_t _id, address_t addr, size_t _size, ECryModule _module) 
      : id(_id), 
        address(addr), 
        size(_size), 
        module(_module)
    {}
  }; 

  // Structure representing a free memory area within a memory state object
  struct freed_mem_t
  {
    // The address of the allocation 
    address_t address;
    
    // The size of the allocation 
    size_t size; 

    // The previous allocations occupying this memory area 
    std::vector<uint64_t> previous_alloc_id; 
  }; 

  // Class representing the active memory of a monitored process. Active
  // memory is the memory that is currently allocated. It is basically
  // an absolute address->(id, callstack, size) mapping so that at any
  // given time the active addresses can be queried to see who has
  // allocated them.
  class memory_state_t
  {
    typedef std::map<address_t, allocation_t> allocation_map_t;

    // Allocation count
    int64_t m_allocation_count;

    // Free count
    int64_t m_free_count;

    // The number of calls to malloc()
    uint64_t m_malloc_calls;

    // The number of calls to calloc()
    uint64_t m_calloc_calls;

    // The number of calls to realloc()
    uint64_t m_realloc_calls;

    // The number of calls to reallocalign()
    uint64_t m_reallocalign_calls;

    // The number of calls to memalign()
    uint64_t m_memalign_calls;

    // The number of calls to free()
    uint64_t m_free_calls;

    // The currently allocated memory 
    int64_t m_current; 

    // The currently allocated memory 
    int64_t m_allocated; 

    // The currently allocated memory 
    int64_t m_freed; 

    // The actual allocated memory 
    int64_t m_peak_memory; 

    // The statistics per module 
    ModuleStats m_module_stats[eCryM_Num]; 

    // The statistics for which we do not know to which module we
    // should attribute the memory allocation
    ModuleStats m_unknown;

    // The allocation map 
    allocation_map_t m_address_map;

    // Consume the memory operation given 
    signed consume(const memory_op_t& op);

    // Perform an allocation
    signed allocate(
      address_t addr, uint64_t id, size_t size, ECryModule module) 
    {
      allocation_map_t& map = m_address_map;
      if (!map.insert(
          std::make_pair(
            addr, 
            allocation_t(
              id,addr,size, module))).second)
        return -1;
      m_current += size;
      m_allocated += size;
      if (m_current > m_peak_memory)
        m_peak_memory = m_current; 
      ++m_allocation_count;
      return 0; 
    }

    // Perform a deallocation
    signed deallocate(address_t addr)
    {
      allocation_map_t& map = m_address_map;
      allocation_map_t::iterator it = map.find(addr); 
      if (it == map.end())
        return -1; 
      m_current -= it->second.size;
      m_freed += it->second.size;
      map.erase(it);
      return 0; 
    }

    // Update the size of a given allocation address 
    signed update(address_t addr, uint64_t id, size_t size)
    {
      allocation_map_t& map = m_address_map;
      allocation_map_t::iterator it =	map.find(addr); 
      if (it == map.end())
        return -1; 
      std::size_t old_size = 0;
      m_current -= it->second.size;
      m_freed += it->second.size;
      old_size = it->second.size;
      allocation_t alloc = it->second;
      alloc.size = size; 
      map.erase(it);
      if (map.insert(std::make_pair(addr,alloc)).second)
        return false; 
      m_current += size; 
      m_allocated += size; 
      if (size > old_size)
        ++m_allocation_count;
      return 0; 
    }

    // Determine if a given address was previously allocated 
    bool allocated(address_t addr, ECryModule& module) const
    {
      const allocation_map_t& map = m_address_map;
      allocation_map_t::const_iterator it =	map.find(addr); 
      if (it != map.end())
      {
        module = it->second.module;
        return true; 
      }
      return false;
    }

    // Determine if a given address was previously allocated 
    bool size(address_t addr, size_t& _size) const
    {
      const allocation_map_t& map = m_address_map;
      allocation_map_t::const_iterator it = map.find(addr); 
      if (it != map.end())
      {
        _size = it->second.size;
        return true;
      }
      return false; 
    }


    // Determine the module of a given address
    ECryModule module(address_t addr) const
    {
      const allocation_map_t& map = m_address_map;
      allocation_map_t::const_iterator it =	map.find(addr); 
      if (it != map.end())
      {
        return it->second.module; 
      }
      return eCryM_Num; 
    }

  public:

    // Determine if a given address was previously allocated 
    bool size(address_t addr, size_t& _size, ECryModule& module) const
    {
      const allocation_map_t& map = m_address_map;
      allocation_map_t::const_iterator it =	map.find(addr); 
      if (it != map.end())
      {
        _size = it->second.size;
        module = it->second.module; 
        return true;
      }
      return false; 
    }

    // Constructor 
    memory_state_t() 
      : m_allocation_count(),
        m_free_count(),
        m_malloc_calls(),
        m_calloc_calls(),
        m_realloc_calls(),
        m_reallocalign_calls(),
        m_memalign_calls(),
        m_free_calls(),
        m_current(),
        m_allocated(),
        m_freed(),
        m_peak_memory()
    {
      std::memset(m_module_stats, 0x0, sizeof (m_module_stats));
      std::memset(&m_unknown, 0x0, sizeof (m_unknown));
    }

    // Fill in the statistics for the memory fields
    void fill_stats(SystemStats& stats) const
    {
      stats.allocations.malloc_calls = (size_t)m_malloc_calls;
      stats.allocations.calloc_calls = (size_t)m_calloc_calls;
      stats.allocations.realloc_calls = (size_t)m_realloc_calls;
      stats.allocations.realloc_calls += (size_t)m_reallocalign_calls;
      stats.allocations.memalign_calls = (size_t)m_memalign_calls;
      stats.allocations.free_calls = (size_t)m_free_calls;
      stats.allocations.allocated_memory = (size_t)m_current; 
      stats.allocations.freed_memory = 0; 
      for (size_t i=0; i<eCryM_Num; ++i)
        stats.modules[i] = m_module_stats[i];
      stats.unknown = m_unknown;
    }

    // Fill in the statistics for the memory fields
    void fill_stats(BriefSystemStats& stats) const
    {
      stats.total = (uint32_t)(m_current);
      for (size_t i=0; i<eCryM_Num; ++i) 
      {
        const ModuleStats &module = m_module_stats[i];
        stats.modules[i] = (uint32_t)(module.allocated - module.freed);
      }
      stats.unknown = (uint32_t)(m_unknown.allocated - m_unknown.freed);
    }

    // Determine the module of a given module 
    const ModuleStats& module_stats(ECryModule module) const
    { return m_module_stats[module]; }

    // Apply the range of memory operations from begin to end to this memory
    // state object 
    signed apply(const memory_op_range& range);
    signed apply(const memory_op_t& op);

    // Remove the range of memory operations from this memory state object
    signed remove(const memory_op_range& range);

    
  }; 

  // Class representing the operations on memory over time.
  // Given a memory operation stream this class can recreate the global memory
  // state at any given point for the memory stream. 
  class memory_monitor_t 
  {
  public:
    // Unique callstacks 
    typedef std::pair<callstack_info_t, ECryModule> callstack_pair_t; 

    // Unique callstacks 
    typedef std::pair<uint32_t, callstack_pair_t> callstack_item_t; 

    // The type of snapshot
    typedef std::vector<snapshot_t> snapshot_vec_t; 

    // The mapping of callstacks crcs to actual callstacks
    typedef std::map<
        uint32_t, 
        callstack_pair_t
      > callstack_map_t; 

  private:

    friend class memory_op_range; 

    // The mapping of memory operation ids 
    typedef std::vector<memory_op_t*> memory_op_vec_t; 


    // The mutex lock for the monitor
    boost::mutex m_mutex;

    // The current number of memory operations 
    size_t m_num_memops; 

    // The current number of memory snapshots
    size_t m_num_snapshots; 

    // The mapping of crcs to callstacks and modules
    callstack_map_t m_callstacks; 

    // The vector of pointers to the memory operations 
    memory_op_vec_t m_memory_ops; 

    // The vector of pointers to the memory operations 
    snapshot_vec_t m_snapshots; 

    // Retrieve the module to which this callstack belongs
    ECryModule find_module(const callstack_info_t& callstack);
    
    // Extract the given range of memory operations from the monitor
    // Note: for performance reasons the caller must ensure that there is
    // enough space in the destination vector 
    void range(memory_op_vec_t& result, size_t from, size_t to) 
    {
      if (to - from > 0)
      {
        boost::unique_lock<boost::mutex> lock(m_mutex); 
        std::copy(
          m_memory_ops.begin() + from, 
          m_memory_ops.begin() + to, 
          result.begin());
      }
    }

  public: 
    // Constructor 
    memory_monitor_t(); 

    // Destructor 
    virtual ~memory_monitor_t(); 

    // Consume the memory operation stream. Note this method can be called from
    // a different thread. 
    bool process(const std::vector<memory_stream_element_t*>&); 
    bool process(const memory_stream_element_t&); 

    // retrieve the number of memory operations received 
    size_t num_ops() const { return m_num_memops; }

    // retrieve the number of snapshots recieved 
    size_t num_snapshots() const { return m_num_snapshots; }

    const memory_op_t* op(size_t id) 
    { 
      boost::unique_lock<boost::mutex> lock(m_mutex); 
      return m_memory_ops[id]; 
    };

    const callstack_pair_t* callstack(uint32_t hash) const
    { 
      callstack_map_t::const_iterator it = m_callstacks.find(hash);
      if (it != m_callstacks.end()) return &it->second; 
      return NULL;
    } 

    // Retrieve the callstacks iterators 
    std::pair<
      callstack_map_t::const_iterator, 
      callstack_map_t::const_iterator
    > callstacks() const 
    {
      return std::make_pair(m_callstacks.begin(), m_callstacks.end());
    }

    // Retrieve the snapshots iterator
    std::pair<
      snapshot_vec_t::const_iterator, 
      snapshot_vec_t::const_iterator
    > snapshots() const 
    {
      return std::make_pair(m_snapshots.begin(), m_snapshots.end());
    }
  }; 

  // The memory operation range iterator concept 
  class memory_op_range
  { 
  public:
   
    // The range_iterator 
    template<typename Value> 
    class range_iterator : 
      public boost::iterator_facade<
        range_iterator<Value>, 
        Value, 
        boost::random_access_traversal_tag
      >
    {
      friend class boost::iterator_core_access;
      template<typename Value> friend class range_iterator;

      // The hosting range 
      memory_op_range* m_range; 

      // The hosting index 
      size_t m_index; 
      
      // Increment the iterator 
      void increment()
      { ++m_index; }
      
      // Dereference the iterator 
      Value& dereference() const
      { return *m_range->m_ops[m_index]; }

      // Iterator equality check 
      template <class OtherValue>
      bool equal(const range_iterator<OtherValue>& other) const
      {
        return 
          this->m_range == other.m_range && 
          this->m_index == other.m_index;
      }

      // Calculate the distance between two iterators
      template <class OtherValue>
      size_t
      distance_to(const range_iterator<OtherValue>& other) const
      {
        return this->m_index - other.m_index; 
      }

    public:
      // Constructor
      explicit range_iterator(memory_op_range* p, size_t index = 0)
        : m_range(p), m_index(index)
      {}
    };

    typedef range_iterator<memory_op_t> iterator;
    typedef range_iterator<const memory_op_t> const_iterator; 

    // Constructor 
    memory_op_range(
      memory_monitor_t* monitor, 
      size_t begin = 0, 
      size_t end = ~0U)
    {
      const size_t real_end = end == ~0U ? monitor->num_ops() : end;
      m_ops.resize(real_end - begin, NULL);
      monitor->range(m_ops, begin, real_end); 
    }

    // Begin/End iterators 
    iterator begin() { return iterator(this); } 
    iterator end() { return iterator(this, m_ops.size()); }

    // Begin/End const_iterators 
    const_iterator begin() const 
    { 
      return const_iterator(const_cast<memory_op_range*>(this)); 
    } 
    const_iterator end() const 
    { 
      return const_iterator(const_cast<memory_op_range*>(this), m_ops.size()); 
    }

  private:
    template<typename Value> friend class range_iterator; 
    
    // The extracted range from the memory monitor
    memory_monitor_t::memory_op_vec_t m_ops; 
  };

  struct call_info
  {
    std::size_t target_id;
    std::string source_info; 
    std::size_t allocated;
    std::size_t freed;
    std::size_t fragmented;
    call_info() 
      : target_id(~0U), 
        source_info(),
        allocated(0), 
        freed(0), 
        fragmented(0) 
    {}
  }; 

  struct function_info
  {
    std::string name; 
    std::size_t id;
    std::map<std::string, call_info> calls;
    std::size_t total_allocated;
    std::size_t total_freed;
    std::size_t total_fragmented;
    std::size_t self_allocated;
    std::size_t self_freed;
    std::size_t self_fragmented;

    function_info() 
      : name(), id(~0U), calls(), 
        total_allocated(0), total_freed(0), total_fragmented(0),
        self_allocated(0), self_freed(0), self_fragmented(0)
    {
    }
  }; 

  template<typename T, T function_info::*member>
  struct function_info_cmp
  { 
    bool operator() (
      const shared_ptr<function_info> a, 
      const shared_ptr<function_info> b) const
    { return a.get()->*member > b.get()->*member; };
  }; 

  typedef function_info_cmp<
    std::size_t, &function_info::total_allocated> cmp_by_total_alloc;
  typedef function_info_cmp<
    std::size_t, &function_info::total_freed> cmp_by_total_live;
  typedef function_info_cmp<
    std::size_t, &function_info::total_fragmented> cmp_by_total_fragmented;
  typedef function_info_cmp<
    std::size_t, &function_info::self_allocated> cmp_by_self_alloc;
  typedef function_info_cmp<
    std::size_t, &function_info::self_freed> cmp_by_self_live;
  typedef function_info_cmp<
    std::size_t, &function_info::self_fragmented> cmp_by_self_fragmented;

  // Memory operations mapped to onto a call graph.
  // The graph provides efficient graph traversal functions to
  // accumulate and analysze the memory operations. 
  class call_graph
  {
  public:
    typedef std::vector< shared_ptr<function_info> > vertex_array_t;
  private:
    typedef std::map< std::string, std::size_t > function_name_map_t;
    typedef std::vector< memory_stream_element_t > operation_array_t;

    // The next function id 
    size_t m_next_function_id;

    // The vertices (found functions) of the graph 
    vertex_array_t m_vertices;

    // The vertices (found functions) of the graph 
    function_name_map_t m_function_names;

    // The vertices (found functions) of the graph 
    operation_array_t m_operations;

  public:

    // Constructor
    call_graph();

    // Analyze the memory operations that have happened
    signed analyze(memory_monitor_t* monitor);

    // The first vertex 
    vertex_array_t::const_iterator begin() const 
    { return m_vertices.begin(); }

    // The one-past the last vertex
    vertex_array_t::const_iterator end() const 
    { return m_vertices.end(); }

    // Retrieve the number of function infos vertices
    size_t size() const 
    { return m_vertices.size(); }

    // Index operator 
    shared_ptr<function_info> operator[] (size_t index) const 
    { return m_vertices[index]; }
  }; 

  // Has the application recieved a request to exit?
  extern volatile bool s_exit_request;
  // Has the application in the main loop?
  extern volatile bool s_main_loop;

  // netpump method 
  int netpump(  
    mtracedb::file_writer_t& file_writer, 
    mtracedb::memory_monitor_t& monitor);

  // filepump method 
  int filepump(  
    mtracedb::file_reader_t& reader, 
    mtracedb::memory_monitor_t& monitor);
}


#endif // _MTRACEDB_H
