//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// Simple html file generator 
// ---------------------------------------------------------------------------
// History:
// - June 13 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////

#include "html_gen.h"
#include <mtracedb/mtracedb.h>

using namespace html; 

////////////////////////////////////////////////////////////////////////
// element method defintions
////////////////////////////////////////////////////////////////////////

// Constructor 
element::element()
  : m_children(), 
    m_links(), 
    m_attributes(),
    m_payload()
{
}

bool element::serialize(std::ostream& out) const
{
  for (size_t i=0; i<m_children.size(); ++i)
  {
    shared_ptr<const element> child = m_children[i];
    if (!child->serialize(out))
      return false; 
  }
  return true;
}

// Set an attribute to a given value
element& element::set_attribute(
  const std::string& attr, const std::string& value)
{
  m_attributes[attr] = value;
  return *this;
}

// Set an attribute to a given value
const char* element::attribute(const std::string& attr) const 
{
  attr_map_t::const_iterator found = m_attributes.find(attr);
  if (found != m_attributes.end())
    return found->second.c_str();
  return NULL; 
}

// Establish a weak link to another element under the given name
element& element::set_link(
  const std::string& link_name, weak_ptr<element> elem)
{
  m_links[link_name] = elem;
  return *this;
}

// Retrieve a named link 
weak_ptr<element> element::link(const std::string& link_name) const
{
  link_map_t::const_iterator found = m_links.find(link_name);
  if (found != m_links.end())
    return found->second;
  return weak_ptr<element>();
}

// Assigns a string to the element. This may or maynot make sense
// depending on the concrete element implementation. 
element& element::operator<< (const std::string&)
{
  DEBUG_CHECK (false);
  throw std::exception(
    "operator<<() can never be called!"); 
  return *this; 
}
element& element::operator<< (const char* content)
{
  return (*this)<<std::string(content);
}
element& element::operator<< (const boost::format& fmt)
{
  return (*this)<<fmt.str();
}

// create a section within the current context.This may or maynot
// make sense depending on the concrete element implementation.
section& element::create_section(const std::string& section_name)
{
  throw std::exception(
    "create_section() not overriden!"); 
  return *(section*)(NULL); 
}

// Create a table within the current context. This may or maynot make sense
// depending on the concrete element implementation. 
table& element::create_table()
{
  shared_ptr<table> _table(new table());  
  add_child(_table);
  return *_table;
}

// Create a list with the current context. This may or maynot make
// sense depending on the concrete element implementation.
list& element::create_list()
{
  shared_ptr<list> _list(new list());  
  add_child(_list);
  return *_list;
}

// Create a list item with the current context. This may or maynot make
// sense depending on the concrete element implementation.
list_item& element::create_list_item()
{
  throw std::exception(
    "create_list_item() not overriden!"); 
  return *(list_item*)(NULL); 
}

// Create a link to another element within the current
// context. This may or maynot make sense depending on the
// concrete element implementation.
link& element::create_link(const std::string& name)
{
  shared_ptr<html::link> _link(new html::link); 
  _link->payload() = name;
  add_child(_link);
  return *_link;
}

////////////////////////////////////////////////////////////////////////
// document method defintions
////////////////////////////////////////////////////////////////////////

document::document(const std::string& title)
  : m_title(title),
    m_section_index()
{
}

// Serialization implementation of document
bool document::serialize(std::ostream& out) const
{
  out << "<html>"
      << "<head>"
      << "<title>"
      << m_title
      << "</title>"
      << "</head>"
      << "<body>" 
      << std::endl;
  out << "<h1><a name=\"_top\"><u>" << m_title << "</u></a></h1>" << std::endl;

  for (size_t i=0; i<m_sections.size(); ++i)
  {
    shared_ptr<const section> _section(m_sections[i]);
    const char* ref_name = _section->attribute("ref_name");
    DEBUG_CHECK(ref_name); 
    out << "<p><a href=\"#" << ref_name << "\">" <<
      _section->payload() << "</a></p>" << std::endl; 
  }

  if (!element::serialize(out))
    return false; 
  
  out << "</body>"
      << "</html>" << std::endl; 
  
  return true; 
}

// Create a section within the current context 
section& document::create_section(
  const std::string& section_name, bool toc)
{
  char section_index[128]; 
  sprintf(section_index, "section_%d", ++m_section_index);

  shared_ptr<section> _section(new section);  
  _section->payload() = section_name;
  _section->set_attribute("ref_name", section_index);
  if (toc)
    *_section << "<a href=\"#_top\"><b>Top</b></a></p>";

  add_child(_section);
  if (toc)
    m_sections.push_back(_section);

  return *_section;
}

// Assigns a string to the document. Creates a paragraph and
// attaches it as a child to the document
element& document::operator<< (const std::string& content)
{
  shared_ptr<paragraph> _p(new paragraph);  
  _p->payload() = content;
  add_child(_p);
  return *this; 
}

////////////////////////////////////////////////////////////////////////
// section method defintions
////////////////////////////////////////////////////////////////////////

// Constructor
section::section(int depth)
  : element(), 
    m_depth(depth)
{
}

// Serialization implementation of document
bool section::serialize(std::ostream& out) const
{
  switch (m_depth)
  {
  case 0:
    out << "<h1><u>"; 
    break;
  case 1:
    out << "<h2><u>"; 
    break;
  case 2:
    out << "<h3><u>"; 
    break;
  case 3:
    out << "<h4><u>"; 
    break;
  case 4:
    out << "<h5><u>"; 
    break;
  case 5:
    out << "<h6><u>"; 
    break;
  default:
    out << "<b><u>";
    break;
  }

  const char* ref_name = attribute("ref_name"); 
  if (ref_name) 
    out << "<a name=\"" << ref_name << "\">"; 

  out << payload(); 

  if (ref_name) 
    out << "</a>"; 

  switch (m_depth)
  {
  case 0:
    out << "</u></h1>"; 
    break;
  case 1:
    out << "</u></h2>"; 
    break;
  case 2:
    out << "</u></h3>"; 
    break;
  case 3:
    out << "</u></h4>"; 
    break;
  case 4:
    out << "</u></h5>"; 
    break;
  case 5:
    out << "</u></h6>"; 
    break;
  default:
    out << "</u></u>";
    break;
  }

  out << std::endl; 

  return element::serialize(out); 
}

// create a section within the current context.
section& section::create_section(const std::string& section_name)
{
  shared_ptr<section> _section(new section(m_depth+1));  
  _section->payload() = section_name;
  add_child(_section);
  *_section << "<a href=\"#_top\"><b>Top</b></a>"
            << "<a href=\"#" << attribute("ref_name") 
            <<  "\"><b>Up</b></a>";
  return *_section;
}

// Create a table within the current context. 
table& section::create_table()
{
  shared_ptr<table> _table(new table());  
  add_child(_table);
  return *_table;
} 

// Assigns a string to the document. Creates a paragraph and
// attaches it as a child to the document
element& section::operator<< (const std::string& content)
{
  shared_ptr<paragraph> _p(new paragraph);  
  _p->payload() = content;
  add_child(_p);
  return *this; 
}

////////////////////////////////////////////////////////////////////////
// list method defintions
////////////////////////////////////////////////////////////////////////

bool list::serialize(std::ostream& out) const
{
  out << "<ul>" << std::endl; 

  if (!element::serialize(out))
    return false; 
  
  out << "</ul>" << std::endl; 

  return true; 
}
  
list::list()
  : element()
{
}

// create a row  within the table
list_item& list::create_list_item()
{
  shared_ptr<list_item> _list_item(new list_item);
  add_child(_list_item);
  return *_list_item;
}


////////////////////////////////////////////////////////////////////////
// list_item method defintions
////////////////////////////////////////////////////////////////////////

bool list_item::serialize(std::ostream& out) const
{
  out << "<li>" << std::endl; 

  if (payload().length())
  {
    out << payload(); 
  }

  if (!element::serialize(out))
    return false; 
  
  out << "</li>" << std::endl; 

  return true; 
}
  
list_item::list_item()
  : element()
{
}

// Assigns a string to the document. Creates a paragraph and
// attaches it as a child to the document
element& list_item::operator<< (const std::string& content)
{
  payload() = content;
  return *this; 
}


////////////////////////////////////////////////////////////////////////
// table method defintions
////////////////////////////////////////////////////////////////////////

table::table() 
  : element()
{
}

// Serialization implementation of table
bool table::serialize(std::ostream& out) const
{
  const char* border = attribute("border"); 
  out << "<table "; 
  if (border)
    out << "border=\"" << border << "\""; 
  out << ">" << std::endl; 

  if (!element::serialize(out))
    return false; 

  out << "</table>"; 

  return true; 
}

// create a row  within the table
row& table::create_row()
{
  shared_ptr<row> _row(new row);
  add_child(_row);
  return *_row;
}

////////////////////////////////////////////////////////////////////////
// row method defintions
////////////////////////////////////////////////////////////////////////

row::row()
  : element()
{
}

// Serialization implementation of row
bool row::serialize(std::ostream& out) const
{
  
  out << "<tr>"; 

  if (!element::serialize(out))
    return false; 

  out << "</tr>" << std::endl; 

  return true; 
}

// create a row  within the table
column& row::create_column(bool header)
{
  shared_ptr<column> _c(new column);
  _c->set_header(header); 
  add_child(_c);
  return *_c;
}

// create a row  within the table
column& row::create_column(const std::string& content, bool header)
{
  shared_ptr<column> _c(new column);
  _c->set_header(header); 
  _c->payload() = content;
  add_child(_c);
  return *_c;
}


////////////////////////////////////////////////////////////////////////
// column method defintions
////////////////////////////////////////////////////////////////////////

column::column()
  : element(),
    m_header(false)
{
}

// Serialization implementation of column
bool column::serialize(std::ostream& out) const
{
  if (m_header)
    out << "<th>"; 
  else
    out << "<td>";

  if (payload().size())
    out << payload(); 

  if (!element::serialize(out))
    return false; 

  if (m_header)
    out << "</th>"; 
  else
    out << "</td>";

  return true; 
}

////////////////////////////////////////////////////////////////////////
// link method defintions
////////////////////////////////////////////////////////////////////////

link::link()
  : element()
{
}

// Serialization implementation of link
bool link::serialize(std::ostream& out) const
{
  out << "<a ";

  if (attribute("target"))
    out << "href=\"" << attribute("target") << '"';

  if (attribute("frame"))
    out << " target=\"" << attribute("frame") << '"';
  
  out << "\">" << payload() << "</a>"; 
  
  return element::serialize(out);
}

////////////////////////////////////////////////////////////////////////
// paragraph method defintions
////////////////////////////////////////////////////////////////////////

paragraph::paragraph()
  : element()
{
}

// Serialization implementation of paragraph
bool paragraph::serialize(std::ostream& out) const
{
  out << "<p>" << payload() << "</p>" << std::endl;
  return element::serialize(out);
}
 
////////////////////////////////////////////////////////////////////////
// serialization main entry method 
////////////////////////////////////////////////////////////////////////
bool html::serialize(std::ostream& out , shared_ptr<element> elem)
{
  return elem->serialize(out);
}
