//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// mtrace analyzing tool that will analayze the allocation database
// created from ps3mtrace
// ---------------------------------------------------------------------------
// History:
// - June 13 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <sstream>
#include <set>
#include <time.h>
#include <windows.h> 
#include <tchar.h>
#include <stdio.h> 
#include <cstdlib>
#include <strsafe.h>
#include <boost/filesystem.hpp>
#include <mtracedb/mtracedb.h>
#include <mtracedb/process.h>
#include "html_gen.h"

namespace fs = boost::filesystem; 

#define GENERATE_TABULAR_OVERVIEW 1

namespace 
{
  static signed g_snapshot_begin = 0; 
  static signed g_snapshot_end = 0; 
  static size_t g_section_entry_count = 4096;
  static size_t g_cs_depth = 32;
  static std::string g_output_dir = ".\\";
  static mtracedb::memory_t s_active_mem;


  std::string strcrc( const std::string& str ) 
  {
    boost::crc_optimal<32, 0x04C11DB7> crc;
    crc.process_bytes( 
      &(*str.begin()), str.length() * sizeof(char) );
    return (boost::format("0x%x") % crc.checksum()).str();
  }
  

  // Print a callstack 
  std::string print_callstack(const mtracedb::callstack_info_t& cs) 
  { 
    mtracedb::address_resolver& resolver =
      mtracedb::address_resolver::instance(); 
    std::stringstream s; 
    for (std::size_t i=0,c=0; i<cs.addresses.size() && c<g_cs_depth; ++i) 
    {
      std::string fn_name = resolver.resolve(cs.addresses[i]); 
#if 0
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find(".h") != std::string::npos)
        continue;
      if (fn_name.find("/STLPORT/") != std::string::npos)
        continue;
      if (fn_name.find("CryMemoryManager") != std::string::npos) 
        continue;
#endif
      ++c;
      std::size_t slashDot = fn_name.find("./"); 
      if (slashDot != std::string::npos) 
        fn_name.erase(slashDot, 2);
      if (i!=0)
				s << '\t';
      s << fn_name;
      if (i != cs.addresses.size()-1)
				s << ",<br>" << std::endl; 
    }
    return s.str(); 
  } 

  // Print a callstack 
  std::string print_callstack_linear(const mtracedb::callstack_info_t& cs) 
  { 
    mtracedb::address_resolver& resolver =
      mtracedb::address_resolver::instance(); 
    std::stringstream s; 
    for (std::size_t i=0,c=0; i<cs.addresses.size() && c<g_cs_depth; ++i) 
    {
      std::string& fn_name = resolver.resolve(cs.addresses[i]); 
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find(".h") != std::string::npos)
        continue;
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find("PS3MemoryInterface") != std::string::npos)
        continue;
      if (fn_name.find("/STLPORT/") != std::string::npos)
        continue;
      if (fn_name.find("CryMemoryManager") != std::string::npos) 
        continue;
      if (fn_name.find("CryMTrace") != std::string::npos) 
        continue;
      ++c;
      std::size_t slashDot = fn_name.find("./"); 
      if (slashDot != std::string::npos) 
        fn_name.erase(slashDot, 2);
      s << fn_name;
      if (i != cs.addresses.size()-1)
				s << ","; 
    }
    return s.str(); 
  } 

  // Print a callstack 
  std::string brief_callstack_linear(const mtracedb::callstack_info_t& cs) 
  { 
    mtracedb::address_resolver& resolver =
      mtracedb::address_resolver::instance(); 
    std::stringstream s; 
    for (std::size_t i=0,c=0; i<cs.addresses.size() && c<3; ++i) 
    {
      std::string& fn_name = resolver.resolve(cs.addresses[i]); 
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find(".h") != std::string::npos)
        continue;
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find("PS3MemoryInterface") != std::string::npos)
        continue;
      if (fn_name.find("/STLPORT/") != std::string::npos)
        continue;
      if (fn_name.find("CryMemoryManager") != std::string::npos) 
        continue;
      if (fn_name.find("CryMTrace") != std::string::npos) 
        continue;
      ++c;
      std::size_t slashDot = fn_name.find("./"); 
      if (slashDot != std::string::npos) 
        fn_name.erase(slashDot, 2);
      s << fn_name;
      if (i != cs.addresses.size()-1)
				s << "<br>"; 
    }
    return s.str(); 
  } 

  void serialize_callgraph(
    shared_ptr<mtracedb::function_info> fn, 
    html::list* list, 
    mtracedb::call_graph& cg, 
    std::vector<bool>& visited) 
  { 
    if (visited[fn->id])
      return; 
    visited[fn->id] = true; 

    html::list_item& item = list->create_list_item(); 
    boost::format header(
      "%s total allocated: %d kb total freed: %d kb "
      "self allocated: %d kb self freed: %d kb");
    header
      % fn->name 
      % (fn->total_allocated / 1024) 
      % (fn->total_freed / 1024)
      % (fn->self_allocated / 1024)
      % (fn->self_freed / 1024); 
    item << header.str();

    html::list& childs = item.create_list(); 

    const std::map<std::string, mtracedb::call_info>& calls = 
      fn->calls; 
    for (std::map<std::string,
           mtracedb::call_info>::const_iterator it = calls.begin(); it
           != calls.end(); ++it)
    {
      const mtracedb::call_info& info = it->second; 
      const shared_ptr<mtracedb::function_info> callee = 
        cg[info.target_id];
      html::list_item& call = childs.create_list_item(); 
      const float pct_allocated = (fn->total_allocated>0) 
        ? (((1.f*info.allocated) / (1.f*fn->total_allocated)) * 100.0f) : 0.0f;
      const float pct_freed = (fn->total_freed>0) 
        ? (((1.f*info.freed) / (1.f*fn->total_freed)) * 100.0f) : 0.0f; 
      boost::format callinfo(
        "<a href=\"#%s\">%s</a> allocated: %3.3f %% freed: %3.3f %% (%s)");
      callinfo
        % strcrc(callee->name)
        % callee->name 
        % pct_allocated
        % pct_freed
        % info.source_info;
      call << callinfo.str();
    }
  } 

  static std::set<uint32_t> s_used_cs_hashes;

  // Generate the index 
  signed generate_index()
  {
    try 
    { 
      // Create the Document and fill the links 
      shared_ptr<html::document> index(
        new html::document("PS3 runtime memory analysis")); 
      // Link to the overview frame 
      index->create_link("Overview").
        set_attribute("target","mtracedb_overview.html").
        set_attribute("frame","main");
      *index << "";

      index->create_link("Flat Profile").
        set_attribute("target","mtracedb_flat_profile_overview_all_modules.html").
        set_attribute("frame","main");
      *index << "";

      html::table& modules = index->create_table(); 
      modules.set_attribute("border", "1");
      html::row& module_headers = modules.create_row(); 
      module_headers.create_column("module name", true);
      module_headers.create_column("unique", true);
      module_headers.create_column("largest", true);
      module_headers.create_column("still live", true);
      module_headers.create_column("frames", true);

      for(int i=mtracedb::eCryM_Num; i>=0; --i)
      {
        mtracedb::ECryModule module = (mtracedb::ECryModule)i;
        std::string module_name = (module!=mtracedb::eCryM_Num)
          ? mtracedb::to_string(module) : "all_modules";

        html::row& module_row = modules.create_row(); 
        module_row.create_column(module_name);

        module_row.create_column().create_link("unique").
          set_attribute(
            "target",
            boost::format("mtracedb_unique_cs_%s.html")
            % module_name).
          set_attribute("frame","main");

        module_row.create_column().create_link("largest live").
          set_attribute(
            "target",
            boost::format("mtracedb_largest_allocs_live_%s.html")
            % module_name).
          set_attribute("frame","main");

        module_row.create_column().create_link("largest rec.").
          set_attribute(
            "target",
            boost::format("mtracedb_largest_allocs_%s.html")
            % module_name).
          set_attribute("frame","main");	

        module_row.create_column().create_link("frames.").
          set_attribute(
            "target",
            boost::format("mtracedb_snapshots_%s.html")
            % module_name).
          set_attribute("frame","main");	
      }

      const std::string out_file_name =
        g_output_dir+"\\mtracedb_index.html"; 
      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      if (!html::serialize(out_file, index))
      {
        std::cerr << "manalyze: could not serialize html document!" <<
          std::endl; 
        return -1; 
      }
    }
    catch (std::exception& e) 
    {
      std::cerr << "manalyze generate_index() error: " << e.what() <<
        std::endl;
      return -1; 
    }
    return 0; 
  }

  // Generate the overall overview over the memory analysis
  signed generate_overview(mtracedb::storage_t& storage)
  {
    const mtracedb::options& opts = mtracedb::get_options(); 
    const mtracedb::segment_stats &stats =
      mtracedb::segment_stats::instance();

    std::cout << "generating general overview statistics " << std::endl; 

    try 
    { 
      shared_ptr<html::document> document(new html::document(""));
      html::section& overview = document->create_section("Overview");

      // Elf section size summary
      overview << "<b>Elf section size summary</b>"; 
      html::table& sections = overview.create_table(); 
      sections.set_attribute("border", "1");
      html::row& section_headers = sections.create_row(); 
      section_headers.create_column("section name", true);
      section_headers.create_column("size in kb", true);
      html::row& text_segment = sections.create_row(); 
      text_segment.create_column(".text");
      text_segment.create_column(
        (boost::format("%8d kb") % (stats.text_size()/1024)).str());
      html::row& data_segment = sections.create_row(); 
      data_segment.create_column(".data");
      data_segment.create_column(
        (boost::format("%8d kb") % (stats.data_size()/1024)).str());
      html::row& bss_segment = sections.create_row(); 
      bss_segment.create_column(".bss");
      bss_segment.create_column(
        (boost::format("%8d kb") % (stats.bss_size()/1024)).str());


      overview << "<b>Allocation by Module</b>"; 
      html::table& alloc_sizes = overview.create_table();
      alloc_sizes.set_attribute("border", "1");
      html::row& alloc_headers = alloc_sizes.create_row();
      alloc_headers.create_column("name", true);
      alloc_headers.create_column("live", true);
      alloc_headers.create_column("total allocated",true);
      alloc_headers.create_column("total freed",true);
      alloc_headers.create_column("total operations",true);
      for(int i=0; i<mtracedb::eCryM_Num; ++i)
      {
        mtracedb::ECryModule module = (mtracedb::ECryModule)i;
        const mtracedb::ModuleStats &stats = 
          s_active_mem.module_stats(module);

        html::row& row = alloc_sizes.create_row();
        row.create_column(mtracedb::to_string(module));
        row.create_column(boost::format("%-8d kb") % ((stats.allocated - stats.freed)/1024));
        row.create_column(boost::format("%-8d kb") % (stats.allocated/1024));
        row.create_column(boost::format("%-8d kb") % (stats.freed/1024));
        row.create_column(boost::format("%-12d ops") % (stats.num_allocations));
      }

      const std::string out_file_name =
        g_output_dir+"\\mtracedb_overview.html"; 
      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      if (!html::serialize(out_file, document))
      {
        std::cerr << "manalyze: could not serialize html document!" <<
          std::endl; 
        return -1; 
      }
    }
    catch (std::exception& e) 
    { 
      std::cerr << "manalyze generate_index() error: " << e.what() <<
        std::endl;
      return -1; 
    } 
 
    return 0;
  }

  typedef std::vector< std::pair<size_t, uint64_t> > IdPairT; 
  static IdPairT s_largest_ops[mtracedb::eCryM_Num+1]; 

  signed presort_ops(const mtracedb::storage_t& storage)
  {
    std::cout << "presorting memory operations per module " << std::endl; 

    for (uint64_t i=0; i<storage.max_id(); ++i)
    {
      clock_t start = clock(); 
      mtracedb::memory_operation_t op; 
      if (storage.get(i, op) != 0) 
      { 
        std::cerr << "manalyze: could not retrieve mem op " << i 
                  << " from storage" << std::endl; 
        continue; 
      } 
      
      mtracedb::ECryModule module = s_active_mem.module(op);
      if (module != mtracedb::eCryM_Num) 
      {
        IdPairT& recorded_ops = s_largest_ops[module]; 
      
        std::pair<size_t,uint64_t> value = 
          std::make_pair(op.size, op.id);
      
        if (!recorded_ops.size())
          recorded_ops.push_back(value);
        else
        {
          for (size_t j=0; j<recorded_ops.size(); ++j)
          {
            if (value.first >= recorded_ops[j].first)
            {
              recorded_ops.insert(recorded_ops.begin()+j,value);
              break;
            }
          }
        }
        if (recorded_ops.size()>g_section_entry_count)
          recorded_ops.pop_back();
      }

      IdPairT& recorded_ops = s_largest_ops[mtracedb::eCryM_Num]; 

      std::pair<size_t,uint64_t> value = 
        std::make_pair(op.size, op.id);
      
      if (!recorded_ops.size())
        recorded_ops.push_back(value);
      else
      {
        for (size_t j=0; j<recorded_ops.size(); ++j)
        {
          if (value.first >= recorded_ops[j].first)
          {
            recorded_ops.insert(recorded_ops.begin()+j,value);
            break;
          }
        }
      }
      if (recorded_ops.size()>g_section_entry_count)
        recorded_ops.pop_back();
    }

    return 0; 
  }
  

  signed generate_largest_ops(
    const mtracedb::storage_t& storage, bool live, 
    mtracedb::ECryModule module)
  {
    const mtracedb::options& opts = mtracedb::get_options(); 
    const mtracedb::segment_stats &stats =
      mtracedb::segment_stats::instance();

    std::string module_name = (module!=mtracedb::eCryM_Num)
      ? mtracedb::to_string(module) : "all_modules";

    std::cout << "searching for largest "; 
    if (live) std::cout << "live ";
    std::cout << "memory operations for " << 
      module_name << std::endl; 

    try 
    { 
      shared_ptr<html::document> document(
        new html::document("Largest memory operations"));
      if (live)
        *document << 
          "A list of largest allocations still alive for " 
                  << module_name; 
      else
        *document << 
          "A list of largest recorded memory operations for"
                  << module_name; 
      
      std::vector< std::pair<size_t, uint64_t> > recorded_ops; 
      clock_t current_time = 0; 

      html::table& cs_table = document->create_table();
      cs_table.set_attribute("border", "1");
      html::row& headers = cs_table.create_row();
      headers.create_column("size kb", true);
      headers.create_column("callstack", true);

      if (live)
      {
        std::vector<mtracedb::alloc_info_t> callstack_results; 
        s_active_mem.callstack(callstack_results); 
        for (std::size_t i=0; 
             i<callstack_results.size() && i<g_section_entry_count; 
             ++i)
        {
          const mtracedb::alloc_info_t& ai = callstack_results[i]; 
          mtracedb::callstack_info_t callstack; 
          if (!storage.get(ai.cs_hash, callstack) == 0) 
          {
            std::cerr << 
              "manalyze: could not retrieve callstack from storage"
                      << std::endl;
            continue; 
          }

          if (module != mtracedb::eCryM_Num && 
              s_active_mem.module(ai)!= module)
            continue;

          html::row& data = cs_table.create_row();
          data.create_column(boost::format("%8d") % (ai.size/1024));
          data.create_column().
            create_link(brief_callstack_linear(callstack)).
            set_attribute(
              "target",
              boost::format("./mtracedb_callstacks.html#cs_%x") % ai.cs_hash).
            set_attribute("frame","detail");
          s_used_cs_hashes.insert(ai.cs_hash);
        }
      }
      else
      {
        IdPairT& recorded_ops = s_largest_ops[module]; 
        for (size_t i=0; i<recorded_ops.size(); ++i)
        {
          mtracedb::callstack_info_t callstack; 
          mtracedb::memory_operation_t op; 
          uint64_t opid = recorded_ops[i].second;

          if (!storage.get(opid, op) == 0) 
          {
            std::cerr << 
              "manalyze: could not retrieve memory op from storage"
                      << std::endl;
            continue; 
          }

          if (!storage.get(op.callstack, callstack) == 0) 
          {
            std::cerr << 
              "manalyze: could not retrieve callstack from storage"
                      << std::endl;
            continue;
          }

          html::row& data = cs_table.create_row();
          data.create_column(boost::format("%8d") % (op.size/1024));
          data.create_column().
            create_link(brief_callstack_linear(callstack)).
            set_attribute(
              "target",
              boost::format("./mtracedb_callstacks.html#cs_%x") % op.callstack).
            set_attribute("frame","detail");
          s_used_cs_hashes.insert(op.callstack);
         
        }
      }

      std::string out_file_name = g_output_dir;
      out_file_name += "\\mtracedb_largest_allocs_" ;
      if (live) out_file_name += "live_"; 
      out_file_name += module_name + ".html"; 

      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      if (!html::serialize(out_file, document))
      {
        std::cerr << "manalyze: could not serialize html document!" <<
          std::endl; 
        return -1; 
      }

    }
    catch(std::exception& e) 
    { 
      std::cerr << "manalyze generate_largest_ops() error: " <<
        e.what() << std::endl;
      return -1; 
    } 

    return 0; 
  }

  signed generate_unique_callstacks(mtracedb::storage_t& storage,
                                    mtracedb::ECryModule module)
  {
    const mtracedb::options& opts = mtracedb::get_options(); 
    const mtracedb::segment_stats &stats =
      mtracedb::segment_stats::instance();


    std::string module_name = (module!=mtracedb::eCryM_Num)
      ? mtracedb::to_string(module) : "all_modules";

    std::cout << "analyzing callstacks " << module_name <<
      std::endl; 

    try 
    { 
      shared_ptr<html::document> document(
        new html::document("Unique callstacks"));
      *document << 
        "A summary of the largest allocations identified "
        "by a unique callstack for " << module_name; 

      html::table& cs_table = document->create_table();
      cs_table.set_attribute("border", "1");
      html::row& headers = cs_table.create_row();
      headers.create_column("size kb", true);
      headers.create_column("no. of allocations", true);
      headers.create_column("callstack", true);

      std::vector<mtracedb::alloc_info_t> callstack_results; 
      s_active_mem.callstack(callstack_results); 
      for (std::size_t i=0; 
           i<callstack_results.size() && i<g_section_entry_count; 
           ++i)
      {
        const mtracedb::alloc_info_t& ai = callstack_results[i]; 
        mtracedb::callstack_info_t callstack; 
        if (!storage.get(ai.cs_hash, callstack) == 0) 
        {
          std::cerr << 
            "manalyze: could not retrieve callstack from storage"
                    << std::endl;
          continue; 
        }

        if (module != mtracedb::eCryM_Num && 
            s_active_mem.module(ai) != module)
          continue;
	
        html::row& data = cs_table.create_row();
        data.create_column(boost::format("%8d") % (ai.size/1024));
        data.create_column(boost::format("%8d") % (ai.num_allocs));
        data.create_column().
          create_link(brief_callstack_linear(callstack)).
          set_attribute(
            "target",
            boost::format("./mtracedb_callstacks.html#cs_%x") % ai.cs_hash).
          set_attribute("frame","detail");
        s_used_cs_hashes.insert(ai.cs_hash);
      }

      const std::string out_file_name =
        g_output_dir+"\\mtracedb_unique_cs_"+module_name+".html"; 

      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      if (!html::serialize(out_file, document))
      {
        std::cerr << "manalyze: could not serialize html document!" <<
          std::endl; 
        return -1; 
      }

    }
    catch (std::exception& e) 
    { 
      std::cerr << "manalyze generate_unique_callstacks() error: " << e.what() <<
        std::endl;
      return -1; 
    } 

    return 0; 
  }

  signed generate_callstack_details(
    const mtracedb::storage_t& storage)
  {
    const mtracedb::options& opts = mtracedb::get_options(); 
    const mtracedb::segment_stats &stats =
      mtracedb::segment_stats::instance();

    std::cout << "generating used callstacks file " <<
      std::endl; 

    try 
    { 
      shared_ptr<html::document> document(
        new html::document("Callstack details"));
      *document << "All callstacks used in the analysis";

      for (std::set<uint32_t>::const_iterator it =
             s_used_cs_hashes.begin(); it != s_used_cs_hashes.end();
           ++it)
      {
        uint32_t cs_hash = *it;
        mtracedb::callstack_info_t callstack; 
        if (!storage.get(cs_hash, callstack) == 0) 
        {
          std::cerr << 
            "manalyze: could not retrieve callstack from storage"
                    << std::endl;
          continue; 
        }

        boost::format fmt("cs_%x"); fmt  % cs_hash;
        html::section& cs = document->create_section(fmt.str(),false);
        cs.set_attribute("ref_name", fmt.str());
        cs << print_callstack(callstack);
      }

      const std::string out_file_name =
        g_output_dir+"\\mtracedb_callstacks.html"; 
      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      if (!html::serialize(out_file, document))
      {
        std::cerr << "manalyze: could not serialize html document!" <<
          std::endl; 
        return -1; 
      }

    } 
    catch (std::exception& e) 
    { 
      std::cerr << "manalyze generate_callstack_details() error: " << e.what() <<
        std::endl;
      return -1; 
    } 
    return 0;
  }

  signed generate_snapshots(const mtracedb::storage_t& storage, 
                            mtracedb::ECryModule module)
  {
    const mtracedb::options& opts = mtracedb::get_options(); 
    const std::vector<mtracedb::storage_t::snapshot_t>& snapshots =
      storage.snapshots();

    std::string module_name = (module!=mtracedb::eCryM_Num)
      ? mtracedb::to_string(module) : "all_modules";

    std::cout << "generating snapshot summary for " << 
      module_name << std::endl;

    shared_ptr<html::document> document(
      new html::document("per frame analysis"));
    *document << "a detailed per frame analysis";

    html::table& t = document->create_table(); 
    t.set_attribute("border", "1");
    html::row headers = t.create_row();
    headers.create_column("id", true);
    headers.create_column("delta kb", true);
    headers.create_column("memory op count", true);
    headers.create_column("link", true);

    mtracedb::memory_t active_mem(const_cast<mtracedb::storage_t*>(&storage)); 
    uint64_t alloc_id = 0; 
    uint64_t last_size = 0;
    for (std::size_t i=0; i<snapshots.size(); ++i) 
    { 
      if (opts.debug || opts.verbose)
        std::cout << "processing snapshot " << i << " of " <<
          snapshots.size() << std::endl; 

      std::vector<mtracedb::memory_operation_t> ops;
      std::multimap<mtracedb::address_t, mtracedb::memory_operation_t>
        allocs;
      std::multimap<mtracedb::address_t, mtracedb::memory_operation_t>
        deallocs;
      const mtracedb::storage_t::snapshot_t& snapshot = snapshots[i]; 

      std::stringstream snapshot_filename; 
      snapshot_filename << "snapshot_" << module_name << "_" << i << ".txt";

      std::ofstream snap_out(
        (g_output_dir + ".\\" + snapshot_filename.str()).c_str());
      if (!snap_out.good())
      {
        std::cerr << "ps3manalyze: error! could not open file " << 
          snapshot_filename.str() << " for writing!"; 
        return -1; 
      }

      while (alloc_id <= snapshot.second) 
      { 
        mtracedb::memory_operation_t op; 
        if (!(alloc_id<storage.max_id()))
          goto trace_done; 
        if (storage.get(alloc_id++, op) != 0) 
        { 
          std::cerr << "manalyze: could not retrieve mem op " << i 
                    << " from	storage" << std::endl; 
          continue; 
        } 
        active_mem(op);
        if (module != mtracedb::eCryM_Num && 
            active_mem.module(op) != module)
          continue; 
        ops.push_back(op);
      } 

      const std::string &name = snapshot.first; 
      uint64_t current_size = active_mem.allocated();
      int64_t delta = current_size - (int64_t)last_size;
      last_size = current_size;  
      size_t num_ops = ops.size(); 

      if (delta == 0 && num_ops == 0) 
        continue;

      html::row& data = t.create_row();
      data.create_column(boost::format("%d") % i);
      data.create_column(boost::format("%d") % (delta/1024));
      data.create_column(boost::format("%d") % num_ops);
      data.create_column().create_link(name).
        set_attribute("target",snapshot_filename.str()).
        set_attribute("frame","detail");
			
      snap_out << "list of operations sorted by size: " << std::endl; 
      std::sort(ops.begin(), ops.end()); 
      for (
        std::size_t j=0; j<ops.size() && j<g_section_entry_count*4; ++j) 
      {
        const mtracedb::memory_operation_t& op = ops[j];
        snap_out << "\top: " << mtracedb::util::to_string(op.type); 
        if ((op.type == 	mtracedb::MO_CALLOC || op.type ==
             mtracedb::MO_MALLOC || op.type == mtracedb::MO_REALLOC ||
             op.type == mtracedb::MO_MEMALIGN))
        {
          snap_out << " " << op.size << " bytes ";
        }
        mtracedb::callstack_info_t callstack; 
        if (!storage.get(op.callstack, callstack) == 0) 
        {
          std::cerr << 
            "manalyze: could not retrieve callstack from storage"
                    << std::endl;
          continue; 
        }
        snap_out << "(" << print_callstack_linear(callstack) << ")"; 
        snap_out << std::endl; 
      }


      snap_out << std::endl << "list of potential fragementations: "
               << std::endl; 

#if 0
      for (std::size_t j=0; j<ops.size(); ++j) 
      {
        const mtracedb::memory_operation_t& op = ops[j];
        if ((op.type == 	mtracedb::MO_CALLOC || op.type ==
             mtracedb::MO_MALLOC || op.type == mtracedb::MO_REALLOC ||
             op.type == mtracedb::MO_MEMALIGN))
        {
          allocs.insert(std::make_pair(op.primary, op));
        }
        if ((op.type == 	mtracedb::MO_FREE || op.type ==
             mtracedb::MO_REALLOC))
        {
          deallocs.insert(std::make_pair(op.primary, op));
          deallocs.insert(std::make_pair(op.secondary, op));
        }
      }

      for (std::multimap<mtracedb::address_t,
             mtracedb::memory_operation_t>::iterator it =
             allocs.begin(); it != allocs.end(); ++it) 
      { 
        const mtracedb::address_t addr = it->first; 
        std::multimap<mtracedb::address_t,
          mtracedb::memory_operation_t>::iterator found =
          deallocs.find(addr);
        if (found != deallocs.end())
        {
          mtracedb::callstack_info_t callstack_alloc; 
          if (!storage.get(it->second.callstack, callstack_alloc) ==
              0) 
          {
            std::cerr << 
              "manalyze: could not retrieve callstack from storage"
                      << std::endl;
            continue; 
          }
          mtracedb::callstack_info_t callstack_dealloc; 
          if (!storage.get(found->second.callstack, callstack_dealloc) ==
              0) 
          {
            std::cerr << 
              "manalyze: could not retrieve callstack from storage"
                      << std::endl;
            continue; 
          }
          snap_out << "\t bytes:" << it->second.size << std::endl;
          snap_out << "\t callstack alloc: " << 
            print_callstack_linear(callstack_alloc) << std::endl;
          snap_out << "\t callstack free: " << 
            print_callstack_linear(callstack_dealloc) << std::endl;
          snap_out << "\t --- " << std::endl; 
        }
      } 
#endif
      snap_out << std::endl; 

    trace_done:
      if(!(alloc_id<storage.max_id()))
        break;
    } 

    const std::string out_file_name =
      g_output_dir+"\\mtracedb_snapshots_"+module_name+".html"; 
    std::ofstream
      out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
    if (!out_file.good()) 
    { 
      std::cerr << "manalyze: could not open output file!" << std::endl; 
      return -1; 
    } 
    if (!html::serialize(out_file, document))
    {
      std::cerr << "manalyze: could not serialize html document!" <<
        std::endl; 
      return -1; 
    }

    return 0; 
  }

  // Generate the flat profile header
  signed generate_flat_profile(
    const mtracedb::storage_t& storage, 
    mtracedb::ECryModule module)
  {
    const mtracedb::options& opts = mtracedb::get_options(); 
    mtracedb::call_graph cg; 

    std::string module_name = (module!=mtracedb::eCryM_Num)
      ? mtracedb::to_string(module) : "all_modules";


    std::cout << "generating flat profile statistics " 
              << module_name << std::endl; 

    clock_t current_time = 0; 
    for (uint64_t i=0; i<storage.max_id(); ++i)
    {
      clock_t start = clock(); 
      mtracedb::memory_operation_t op; 
      if (storage.get(i, op) != 0) 
      { 
        std::cerr << "manalyze: could not retrieve mem op " << i 
                  << " from	storage" << std::endl; 
        continue; 
      } 

      if (module != mtracedb::eCryM_Num && 
          s_active_mem.module(op) != module) 
        continue;

      mtracedb::callstack_info_t cs; 
      if (storage.get(op.callstack, cs) != 0)
      {
        std::cerr << "manalyze: could not retrieve callstack " << op.callstack
                  << " from	storage" << std::endl; 
        continue; 
      }

      if (cg(op, cs) != 0) 
      { 
        std::cerr << "manalyze: could not insert function into "
          " callgraph  " << op.callstack
                  << " from	storage" << std::endl; 
        continue; 
      } 

      current_time += clock() - start; 
      if (current_time > CLOCKS_PER_SEC) 
      { 
        if (opts.verbose || opts.debug)
          std::cout << "\r" << i << " out of " << storage.max_id() <<
            " operations processed"; 
        current_time = 0 ; 
      } 
    }

    if (cg.analyze(storage) != 0)
    {
      std::cerr << "manalyze: error performing callgraph analysis" <<
        std::endl; 
      return -1; 
    }

    // Find the function that contains the total allocations
    typedef std::vector< shared_ptr<mtracedb::function_info> >
      fn_array;
    fn_array total_alloc; 

    for (size_t i=0; i<cg.size(); ++i)
      total_alloc.push_back(cg[i]);
    std::sort(
      total_alloc.begin(), total_alloc.end(), 
      mtracedb::cmp_by_total_alloc());

    // Generate data-file  
    try
    {
      shared_ptr<html::document> document(new html::document("flat_profile"));
      html::section& flat_profile = document->create_section(
        "Flat Profile");

      for (size_t i=0; i < total_alloc.size(); ++i) 
      {
        boost::shared_ptr<mtracedb::function_info> fn =
          total_alloc[i];

        html::section& fn_desc =
          document->create_section(fn->name,false);
        fn_desc.set_attribute("ref_name", strcrc(fn->name));
        fn_desc.depth() = 6; 
        
        html::list& calls = fn_desc.create_list(); 
        std::vector<bool> visited(cg.size(), false);
        serialize_callgraph(fn, &calls, cg, visited); 
      }

      const std::string out_file_name =
        g_output_dir+"\\mtracedb_flat_profile_data_"+module_name+".html"; 
      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      if (!html::serialize(out_file, document))
      {
        std::cerr << "manalyze: could not serialize html document!" <<
          std::endl; 
        return -1; 
      }
    }
    catch (std::exception& e) 
    { 
      std::cerr << "manalyze: could not generate flat profile data! "
                << "reason: " << e.what() << std::endl; 
      return -1; 
    } 

    // Generate dot-file 
#if defined(GENERATE_TABULAR_OVERVIEW)
    try
    {
      shared_ptr<html::document> document(new html::document("flat_profile_overview"));
      html::section& flat_profile = document->create_section(
        "Flat Profile Overview");

      html::table& funcs = flat_profile.create_table(); 
      funcs.set_attribute("border", "1");
      html::row& headers = funcs.create_row();
      headers.create_column("function", true);
      headers.create_column("allocated kb", true);
      headers.create_column("freed kb", true);
      headers.create_column("self allocated kb", true);
      headers.create_column("self freed kb", true);

      for (size_t i=0; i < total_alloc.size(); ++i) 
      {
        boost::shared_ptr<mtracedb::function_info> fn =
          total_alloc[i];

        std::string fnName = fn->name; 
        if (fnName.length() > 32) 
        {
          fnName.resize(32);
          fnName += "...";
        }
        html::row& data = funcs.create_row();
        data.create_column().
          create_link(fnName).
          set_attribute(
            "target",
            boost::format("./mtracedb_flat_profile_data_%s.html#%x") 
            % module_name 
            % strcrc(fn->name)).
          set_attribute("frame","detail");
        data.create_column(boost::format("%8d") % (fn->total_allocated / 1024));
        data.create_column(boost::format("%8d") % (fn->total_freed / 1024));
        data.create_column(boost::format("%8d") % (fn->self_allocated / 1024));
        data.create_column(boost::format("%8d") % (fn->self_freed / 1024));
      }

      const std::string out_file_name =
        g_output_dir+"\\mtracedb_flat_profile_overview_"+module_name+".html"; 
      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      if (!html::serialize(out_file, document))
      {
        std::cerr << "manalyze: could not serialize html document!" <<
          std::endl; 
        return -1; 
      }
    }
    catch (std::exception& e) 
    { 
      std::cerr << "manalyze: could not tabular overview file!" <<
        std::endl; 
      return -1; 
    } 
#else
    try
    {
      const std::string out_file_name =
        g_output_dir+"\\mtracedb_flat_profile.dot"; 
      const std::string svg_file_name =
        g_output_dir+"\\mtracedb_flat_profile.svg"; 
      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 

      out_file << "digraph {" << std::endl; 

      std::vector<std::size_t> queue(cg.size(), std::size_t());
      for (std::size_t i=0; i<cg.size(); ++i) 
        queue[i] = total_alloc[i]->id;
      std::vector<bool> visited(cg.size(), false); 
      while (!queue.empty())
      {
        std::size_t index = queue.front(); 
        queue.erase(queue.begin()); 

        if (visited[index]) 
          continue; 
        visited[index] = true; 

        shared_ptr<mtracedb::function_info> fn = cg[index]; 
        std::string fnName = fn->name; 
        if (fnName.length() > 32) 
          fnName.resize(32);

        out_file << "node_" << index << " [";
        if (fn->name.length()) 
          out_file << "URL=\"./mtracedb_flat_profile_data.html#" 
                   << strcrc(fn->name) << "\",target=\"detail\""; 
        out_file << "shape=box, label=\"" << fnName << "\"];" <<
          std::endl; 

        const std::map<mtracedb::address_t, mtracedb::call_info>& calls = 
          fn->calls; 
        for (std::map<mtracedb::address_t,
               mtracedb::call_info>::const_iterator it = calls.begin(); it
               != calls.end(); ++it)
        {
          const mtracedb::call_info& info = it->second; 

          out_file << "node_" << index << " -> node_" <<
            info.target_id << std::endl; 

          DEBUG_CHECK (info.target_id < cg.size());
          queue.push_back(info.target_id);
        }
      }

      out_file << "}" << std::endl; 
      out_file.close(); 

      mtracedb::process_t dot_process; 
      const char* argv[] = {
        opts.dot_exe.c_str(),
        "-Tsvg",
        out_file_name.c_str(), 
        "-o",
        svg_file_name.c_str()
      }; 
      if (!dot_process.run(sizeof(argv)/sizeof(const char*), argv))
        throw std::exception("can not run dot on dot file"); 
    }
    catch (std::exception& e) 
    { 
      std::cerr << "manalyze: could not generate and convert dot file!" <<
        std::endl; 
      return -1; 
    } 
#endif     

    return 0;
  }

  // Generate the frameset 
  signed generate_frameset()
  {
    try 
    { 
      const std::string out_file_name =
        g_output_dir+"\\mtracedb.html"; 
      std::ofstream
        out_file(out_file_name.c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      out_file << 
        "<html><head><title>PS3 memory trace analysis</title></head>"
        "<frameset cols=\"25%,75%\">"
        "<frame src=\"./mtracedb_index.html\" name=\"index\">"
        "<frameset rows=\"80%,20%\">"
        "<frame src=\"./mtracedb_overview.html\" name=\"main\">"
        "<frame src=\"./mtracedb_callstacks.html\" name=\"detail\">"
        "</frameset>"
        "</frameset>"
        "</html"; 
    }
    catch (std::exception& e) 
    {
      std::cerr << "manalyze generate_index() error: " << e.what() <<
        std::endl;
      return -1; 
    }
    return 0; 
  }

}

// Main entry method 
int main(int argc, char* argv[]) 
{
  mtracedb::config_map_t& config = mtracedb::config();
  mtracedb::options& opts = mtracedb::get_options(); 
  program_options::options_description opt_desc("manalyze options");
  opt_desc.add_options()
    ("help", "this help message")
    ("callstack-depth",
     program_options::value<size_t>(&g_cs_depth),
     "a callstack with a depth larger than this value will be"
     " truncated")
    ("section-entries",
     program_options::value<size_t>(&g_section_entry_count),
     "the number of entries in each output section")
    ("output-directory,o",
     program_options::value<std::string>(&g_output_dir),
     "the output directory into which the html files should be placed");
  mtracedb::parse_command_line(argc, argv, opt_desc);

  fs::path dir = opts.data_directory;
  if (!fs::exists(dir)) 
  { 
    std::cerr << "mtrace: data directory does not exist " << dir <<
      std::endl; 
    return -1; 
  } 

  fs::path elf_file = opts.elf_file;
  fs::path new_elf = dir / elf_file.filename();
  if (!fs::exists(new_elf))
  {
    std::cerr << "mtrace: file " << new_elf << " does not exist" <<
      std::endl; 
    return -1; 
  }
  opts.elf_file = new_elf.file_string();

  shared_ptr<std::ofstream> ofile; 
  if (opts.log_file.size())
  {
    ofile = shared_ptr<std::ofstream>(
      new std::ofstream(opts.log_file.c_str()));
    mtracedb::util::set_output_streams(*ofile, *ofile); 
  }

  // Create active memory object and storage backend
  mtracedb::storage_t storage(opts.data_directory.c_str(), false);
  s_active_mem.set_storage(&storage);
  // Retrieve the maximum id and loop over all memory optimizations
  if (opts.verbose || opts.debug)
    std::cout << "playing back memory trace " << std::endl; 
  clock_t current_time = 0; 
  for (uint64_t i=0; i<storage.max_id(); ++i)
  {
    clock_t start = clock(); 
    mtracedb::memory_operation_t op; 
    if (storage.get(i, op) != 0) 
    { 
      std::cerr << "manalyze: could not retrieve mem op " << i 
                << " from	storage" << std::endl; 
      continue; 
    } 
    s_active_mem(op); 
    current_time += clock() - start; 
    if (current_time > CLOCKS_PER_SEC) 
    { 
      if (opts.verbose || opts.debug)
        std::cout << "\r" << i << " out of " << storage.max_id() <<
          " operations processed"; 
      current_time = 0 ; 
    } 
  }

  if (generate_index() != 0)
  {
    std::cerr << "manalyze: generating index failed!" << std::endl;
  }

  if (generate_overview(storage) != 0)
  {
    std::cerr << "manalyze: generating overview failed!" << std::endl;
  }

  if (generate_unique_callstacks(storage,mtracedb::eCryM_Num) != 0)
  {
    std::cerr << 
      "manalyze: generating unique callstack analysis failed!" << std::endl;
  }
  for(int i=0; i<mtracedb::eCryM_Num; ++i)
  {

    if (generate_unique_callstacks(storage,(mtracedb::ECryModule)i) != 0)
    {
      std::cerr << 
        "manalyze: generating unique callstack analysis failed!" << std::endl;
    }
  }

  if (presort_ops(storage) != 0)
  {
    std::cerr << 
      "manalyze: could not presort memory ops!" << std::endl;
  }; 

  if (generate_largest_ops(storage,true,mtracedb::eCryM_Num) != 0)
  {
    std::cerr << 
      "manalyze: generating largest allocation summary failed!" << std::endl;
  }
  for(int i=0; i<mtracedb::eCryM_Num; ++i)
  {

    if (generate_largest_ops(storage,true,(mtracedb::ECryModule)i) != 0)
    {
      std::cerr << 
        "manalyze: generating largest allocation summary failed!" << std::endl;
    }
  }

  if (generate_largest_ops(storage,false,mtracedb::eCryM_Num) != 0)
  {
    std::cerr << 
      "manalyze: generating largest allocation summary failed!" << std::endl;
  }
  for(int i=0; i<mtracedb::eCryM_Num; ++i)
  {

    if (generate_largest_ops(storage,false,(mtracedb::ECryModule)i) != 0)
    {
      std::cerr << 
        "manalyze: generating largest allocation summary failed!" << std::endl;
    }
  }

  if (generate_snapshots(storage,mtracedb::eCryM_Num) != 0)
  {
    std::cerr << 
      "manalyze: generating largest allocation summary failed!" << std::endl;
  }
  for(int i=0; i<mtracedb::eCryM_Num; ++i)
  {
    if (generate_snapshots(storage,(mtracedb::ECryModule)i) != 0)
    {
      std::cerr << 
        "manalyze: generating largest allocation summary failed!" << std::endl;
    }
  }


  if (generate_callstack_details(storage) != 0) 
  {
    std::cerr << "manalyze: generating callstack details failed!" <<
      std::endl;
  }

  if (generate_flat_profile(storage,mtracedb::eCryM_Num) != 0)
  {
    std::cerr << "manalyze: generating overview failed!" << std::endl;
  }

  if (generate_frameset() != 0)
  {
    std::cerr << "manalyze: generating index failed!" << std::endl;
  }

  return 0; 
}
