//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// mtrace server for recieving allocation information from cryengine3
// ps3 applications
// ---------------------------------------------------------------------------
// History:
// - June 13 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////

#include <mtracedb/mtracedb.h>
#include <mtracedb/process.h>
#include <mtracedb/html_gen.h>

#include <time.h>
#include <list>
#include <iostream>
#include <algorithm>
#include <numeric>
#include <boost/filesystem.hpp>
#include <boost/format.hpp>

//Win32 specific?
#include <signal.h>

//#define ENABLE_STREAM_DEBUGGING 1
#if defined(ENABLE_STREAM_DEBUGGING)
# define STREAM_DEBUG_TAG (0xdeadbeefU)
#endif

//#define TRACE_RECIEVE_OPERATIONS
#undef TRACE_RECIEVE_OPERATIONS

namespace fs = boost::filesystem; 
using mtracedb::cerr;
using mtracedb::out;


#     if 0      
      std::stringstream plotscript;
      plotscript << "reset" << std::endl;
      plotscript << "set term png font arial 6 size 1280,600" << std::endl;
      plotscript << "set output " << '\'' << snapimg << '\'' << std::endl; 
      plotscript << "set xtics ("; 
      for (int i=0; i<(int)eCryM_Num+1; ++i)
      {
        plotscript << '"' << to_string((ECryModule)i) << '"'; 
        plotscript << ' ' << i; 
        if (i<(int)eCryM_Num)
          plotscript << ','; 
      }
      plotscript << ")" << std::endl; 
      plotscript << "plot '-' using 1:2 with boxes" << std::endl;
      for (int i=0; i<(int)eCryM_Num+1; ++i)
      {
        plotscript << i << ' ' << ops[i] << std::endl;
      }
      plotscript << "e" << std::endl; 
      plotter::instance().write(plotscript.str());
#     endif 

namespace mtracedb
{
	const char* application_name = "ps3mtrace"; 
};

namespace 
{ 
  ////////////////////////////////////////////////////////////////////////////////
  // Helper methods for callstack printing 
  std::string print_callstack(const mtracedb::callstack_info_t& cs) 
  { 
    mtracedb::address_resolver& resolver =
      mtracedb::address_resolver::instance(); 
    std::stringstream s; 
    for (std::size_t i=0,c=0; i<cs.addresses.size(); ++i) 
    {
      std::string fn_name = resolver.resolve(cs.addresses[i]); 
      ++c;
      std::size_t slashDot = fn_name.find("./"); 
      if (slashDot != std::string::npos) 
        fn_name.erase(slashDot, 2);
      if (i!=0)
				s << '\t';
      s << fn_name;
      if (i != cs.addresses.size()-1)
				s << ",<br>" << std::endl; 
    }
    return s.str(); 
  } 

  // Print a callstack 
  std::string print_callstack_linear(const mtracedb::callstack_info_t& cs) 
  { 
    mtracedb::address_resolver& resolver =
      mtracedb::address_resolver::instance(); 
    std::stringstream s; 
    for (std::size_t i=0,c=0; i<cs.addresses.size(); ++i) 
    {
      std::string& fn_name = resolver.resolve(cs.addresses[i]); 
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find(".h") != std::string::npos)
        continue;
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find("PS3MemoryInterface") != std::string::npos)
        continue;
      if (fn_name.find("/STLPORT/") != std::string::npos)
        continue;
      if (fn_name.find("CryMemoryManager") != std::string::npos) 
        continue;
      if (fn_name.find("CryMTrace") != std::string::npos) 
        continue;
      ++c;
      std::size_t slashDot = fn_name.find("./"); 
      if (slashDot != std::string::npos) 
        fn_name.erase(slashDot, 2);
      s << fn_name;
      if (i != cs.addresses.size()-1)
				s << ","; 
    }
    return s.str(); 
  } 

  // Print a callstack 
  std::string brief_callstack_linear(const mtracedb::callstack_info_t& cs) 
  { 
    mtracedb::address_resolver& resolver =
      mtracedb::address_resolver::instance(); 
    std::stringstream s; 
    for (std::size_t i=0,c=0; i<cs.addresses.size() && c<3; ++i) 
    {
      std::string& fn_name = resolver.resolve(cs.addresses[i]); 
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find(".h") != std::string::npos)
        continue;
      if (fn_name.find("/CryCommon/") != std::string::npos &&
          fn_name.find("PS3MemoryInterface") != std::string::npos)
        continue;
      if (fn_name.find("/STLPORT/") != std::string::npos)
        continue;
      if (fn_name.find("CryMemoryManager") != std::string::npos) 
        continue;
      if (fn_name.find("CryMTrace") != std::string::npos) 
        continue;
      ++c;
      std::size_t slashDot = fn_name.find("./"); 
      if (slashDot != std::string::npos) 
        fn_name.erase(slashDot, 2);
      s << fn_name;
      if (i != cs.addresses.size()-1)
				s << "<br>"; 
    }
    return s.str(); 
  } 

  // Compute the crc of a string 
  std::string strcrc( const std::string& str ) 
  {
    boost::crc_optimal<32, 0x04C11DB7> crc;
    crc.process_bytes( 
      &(*str.begin()), str.length() * sizeof(char) );
    return (boost::format("0x%x") % crc.checksum()).str();
  }

  // Mtrace specific options 
  struct options
  {
    // input path 
    std::string input; 
    // analyze at the end of receving the stream 
    bool analyze;
    // save the mtrace stream
    bool save;
  }; 
  // Static options instance 
  static options g_options; 

  // Class that can plot figures onto a canvas 
  class plotter
  {
    // Constructor 
    plotter();
    
    // Inialize the plotter 
    bool init(); 

  public:

    void write(const std::string&); 

    // Retrieve the global instance of the plotter 
    static plotter& instance(); 
  };

  // The address to line process
  static mtracedb::process_t s_gnuplot_process;

  // Constructor 
  plotter::plotter()
  {
    if (!init())
    {
      cerr() << "could not initialze the gnuplot" << std::endl; 
      EXIT();
    }
  }

  bool plotter::init()
  {
    const mtracedb::options& o = mtracedb::get_options(); 
    const char* argv[] = {
      o.gnuplot_exe.c_str()
    }; 
    return s_gnuplot_process.run(sizeof(argv)/sizeof(const char*), argv);
  }

  void plotter::write(const std::string& s)
  {
    using namespace mtracedb;

    size_t written = s_gnuplot_process.write(
      process_t::STREAM_INPUT, 
      s.c_str(), 
      s.length());
    if (written == 0) 
    {
      cerr() << "write to gnuplot failed!" << std::endl;
      EXIT();
    }
  }

  plotter& plotter::instance()
  {
    static plotter _plotter; 
    return _plotter; 
  }

  // Analyzer holds information about the overall process of analyzing the
  // analyzis
  class analysis
  {
    // The monitor containing the memory operations 
    mtracedb::memory_monitor_t* m_monitor; 

    // The state of the memory at any given time during the capture
    mtracedb::memory_state_t* m_snapshot_state; 

    // The memory state object
    mtracedb::memory_state_t m_state; 

    // The snapshot data
    struct snapshot_data
    { 
      std::string name; 
      uint64_t total_ops; 
    }; 
    std::vector<snapshot_data> m_snapshots; 
    std::vector<const mtracedb::memory_op_t*> m_errors;

    // The output directory 
    fs::path m_output_dir; 
    fs::path m_image_dir;
    std::string m_index_file;
    std::string m_frameset_file;
    std::string m_overview_file; 
    std::string m_callstack_file; 
    std::string m_snapshots_file;
    std::string m_uniquecs_file;

    signed generate_callstacks();
    signed generate_snapshots();
    signed generate_overview();
    signed generate_index();
    signed generate_frameset();
    signed create_img_dir();

  public:

    // Constructor 
    analysis(mtracedb::memory_monitor_t* monitor);

    // Destructor 
    ~analysis();

    // Generate the results of the analyzis 
    signed generate();
  }; 

  // Generate the callstacks file 
  signed analysis::generate_callstacks()
  {
    using namespace mtracedb; 
    out(LOG_INFO) << "generating callstack details" << std::endl;
    shared_ptr<html::document> document(
      new html::document("Callstack details"));
    *document << "callstacks encountered during the analysis";

    foreach(const memory_monitor_t::callstack_item_t& cs, m_monitor->callstacks())
    {
      boost::format fmt("cs_%x"); fmt  % cs.first;
      html::section& csection = document->create_section(fmt.str(),false);
      csection.set_attribute("ref_name", fmt.str());
      csection << print_callstack(cs.second.first);
    }

    fs::path  out_path = fs::path(m_output_dir) / m_callstack_file;
    std::ofstream out_file(
      out_path.file_string().c_str(),std::ios::out|std::ios::trunc);

    if (!out_file.good()) 
    { 
      std::cerr << "manalyze: could not open output file!" << std::endl; 
      return -1; 
    } 

    if (!html::serialize(out_file, document))
    {
      std::cerr << "manalyze: could not serialize html document!" <<
        std::endl; 
      return -1; 
    }
    return 0;
  }

  struct memop_cmp_by_size
  { 
    bool operator() (
        const mtracedb::memory_op_t* a, 
        const mtracedb::memory_op_t* b) const 
    { 
      if (a == b) return false; 
      return a->size > b->size; 
    } 
  }; 

  // Generate the snapshots
  signed analysis::generate_snapshots()
  {
    using namespace mtracedb; 
    out(LOG_INFO) << "generating frames " << std::endl;

    shared_ptr<html::document> document(
      new html::document("Snapshots"));
    *document << "snapshots taken during the capture";

    uint64_t allocated[eCryM_Num+1]; 
    uint64_t freed[eCryM_Num+1]; 
    uint64_t ops[eCryM_Num+1]; 

    std::memset(allocated, 0x0, sizeof (allocated));
    std::memset(freed, 0x0, sizeof (freed));
    std::memset(ops, 0x0, sizeof (ops));

    typedef std::vector<const memory_op_t*> ops_t; 
    ops_t sorted_ops; 
    
    size_t last_id = 0; 
    memory_state_t &state = m_state; 
    foreach(const snapshot_t& snapshot, m_monitor->snapshots())
    {
      out(LOG_INFO) << "processing snapshot " << snapshot.name << std::endl;
      memory_op_range range(m_monitor, last_id, snapshot.id);

      html::section& cs = document->create_section(snapshot.name,false);
      cs.set_attribute("ref_name", snapshot.name);

      // Sort the operations within the range by size and module 
      foreach(const memory_op_t& op, range)
      {
        switch (op.type)
        {
          case MO_MALLOC:
          case MO_CALLOC:
          case MO_MEMALIGN:
          {
            allocated[op.module] += op.size;

            for (size_t i=0; i<sorted_ops.size(); ++i) 
            { 
              if (op.size > sorted_ops[i]->size)
              {
                sorted_ops.insert(
                  sorted_ops.begin() + i,
                  &op);
                break;
              }
            }
            if (sorted_ops.size() > 0xff) 
              sorted_ops.erase(sorted_ops.end()-1, sorted_ops.end());
          }
          break;

          case MO_REALLOC:
          {
            size_t prev_size = 0; 
            ECryModule prev_module = eCryM_Num; 
            if (state.size(op.secondary, prev_size, prev_module))
            {
              freed[prev_module] += prev_size;
              allocated[op.module] += op.size; 
              for (size_t i=0; i<sorted_ops.size(); ++i) 
              { 
                if (op.size > sorted_ops[i]->size)
                {
                  sorted_ops.insert(
                    sorted_ops.begin() + i,
                    &op);
                  break;
                }
              }
              if (sorted_ops.size() > 0xff) 
                sorted_ops.erase(sorted_ops.end()-1, sorted_ops.end());
            }
          }
          break;

          case MO_FREE:
          {
            size_t size = 0; 
            ECryModule module = eCryM_Num; 
            if (state.size(op.primary, size, module))
            {
              freed[module] += size;
              for (size_t i=0; i<sorted_ops.size(); ++i) 
              { 
                if (op.size > sorted_ops[i]->size)
                {
                  sorted_ops.insert(
                    sorted_ops.begin() + i,
                    &op);
                  break;
                }
              }
              if (sorted_ops.size() > 0xff) 
                sorted_ops.erase(sorted_ops.end()-1, sorted_ops.end());
            }
          }
          break;
        }
        ++ops[op.module]; 

        if (state.apply(op) != 0)
          m_errors.push_back(&op);
      }

      html::section& overview = cs.create_section("Breakdown of operations");
      html::table& ov_table = overview.create_table();
      ov_table.set_attribute("border", "1");
      html::row& headers = ov_table.create_row();
      headers.create_column("module", true);
      headers.create_column("memory allocated (kb)", true);
      headers.create_column("memory freed (kb)", true);
      headers.create_column("number of ops", true);

      const uint64_t total_ops = std::accumulate(
        ops, ops + sizeof(ops)/sizeof(ops[0]), uint64_t());
      const float total_allocated = (float) std::accumulate(
        allocated, allocated + sizeof(allocated)/sizeof(allocated[0]),
        uint64_t()) 
        / 1024.0f;
      const float total_freed = (float) std::accumulate(
        freed, freed + sizeof(freed)/sizeof(freed[0]), 
        uint64_t()) 
        / 1024.0f;

      html::row& total = ov_table.create_row();
      total.create_column("total");
      total.create_column(boost::format("%3.3f") % (total_allocated));
      total.create_column(boost::format("%3.3f") % (total_freed));
      total.create_column(boost::format("%8d") % (total_ops));

      for (signed i=0; i<(int)eCryM_Num+1; ++i)
      {
        html::row& total = ov_table.create_row();
        total.create_column(to_string((ECryModule)i));
        total.create_column(boost::format("%3.3f") % ((float)allocated[i]/1024.0f));
        total.create_column(boost::format("%3.3f") % ((float)freed[i]/1024.0f));
        total.create_column(boost::format("%8d") % (ops[i]));
      }

      html::section& list = cs.create_section("List of operations");
      html::table& op_table = list.create_table();
      op_table.set_attribute("border", "1");
      html::row& op_hdrs = op_table.create_row();
      op_hdrs.create_column("type", true);
      op_hdrs.create_column("size (bytes)", true);
      op_hdrs.create_column("module", true);
      op_hdrs.create_column("callstack", true);

      for (size_t i=0; i<sorted_ops.size() && i<0xff; ++i)
      {
        const memory_op_t &sop = *sorted_ops[i];
        const callstack_info_t& callstack = m_monitor->callstack(sop.callstack)->first;

        html::row& data = op_table.create_row();
        data.create_column(util::to_string(sop.type));
        data.create_column(boost::format("%8d") % (sop.size));
        data.create_column(to_string(sop.module));
        data.create_column().
          create_link(brief_callstack_linear(callstack)).
          set_attribute(
            "target",
            boost::format("./%s#cs_%x") % m_callstack_file % sop.callstack).
          set_attribute("frame","detail");
      }

      // Insert the snapshot into the snapshot data form 
      snapshot_data sd;
      sd.name = snapshot.name;
      sd.total_ops = total_ops;
      m_snapshots.push_back(sd);

      last_id = snapshot.id;
      sorted_ops.clear();
    }
    
    fs::path  out_path = fs::path(m_output_dir) / m_snapshots_file;
    std::ofstream out_file(
      out_path.file_string().c_str(),std::ios::out|std::ios::trunc);
    if (!out_file.good()) 
    { 
      cerr() << "manalyze: could not open output file!" << std::endl; 
      return -1; 
    } 
    if (!html::serialize(out_file, document))
    {
      cerr() << "manalyze: could not serialize html document!" <<
        std::endl; 
      return -1; 
    }
    return 0; 
  }

  signed analysis::generate_overview()
  {
    using namespace mtracedb; 
    shared_ptr<html::document> document(new html::document("mtrace analysis"));
    out(LOG_INFO) << "processing overview " << std::endl;

    // Module Breakdown 
    html::section& modules = document->create_section("Modules");
    html::table& mod_table = modules.create_table();
    mod_table.set_attribute("border", "1");
    html::row& modhdrs = mod_table.create_row();
    modhdrs.create_column("module", true);
    modhdrs.create_column("memory allocated (kb)", true);
    modhdrs.create_column("number of ops", true);

    SystemStats stats; 
    m_state.fill_stats(stats); 
    
    html::row& total = mod_table.create_row();
    total.create_column("total");
    total.create_column(boost::format("%3.3f") 
      % ((float)(stats.allocations.allocated_memory-stats.allocations.freed_memory)/1024.f));
    total.create_column(
      boost::format("%8d") % 
      (stats.allocations.malloc_calls + 
        stats.allocations.calloc_calls + 
        stats.allocations.realloc_calls +
        stats.allocations.memalign_calls +
        stats.allocations.free_calls));

    for (signed i=0; i<(signed)eCryM_Num; ++i)
    {
      html::row& data = mod_table.create_row();
      data.create_column(to_string((ECryModule)i));
      data.create_column(boost::format("%3.3f") 
        % ((float)(stats.modules[i].allocated-stats.modules[i].freed)/1024.f));
      data.create_column(
        boost::format("%8d") % 
        (stats.modules[i].num_allocations));
    }
    html::row& unknown = mod_table.create_row();
    unknown.create_column("unknown");
    unknown.create_column(boost::format("%3.3f") 
      % ((float)(stats.unknown.allocated-stats.unknown.freed)/1024.f));
    unknown.create_column(
      boost::format("%8d") % 
      (stats.unknown.num_allocations));

    // Snapshot Breakdown 
    html::section &snapshots = document->create_section("snapshots", false);
    html::table &snap_table = document->create_table();
    snap_table.set_attribute("border", "1");
    html::row& snap_hdrs = snap_table.create_row();
    snap_hdrs.create_column("snapshot", true);
    snap_hdrs.create_column("no. of ops", true);
    foreach(const snapshot_data& snapshot, m_snapshots)
    {
      html::row& data = snap_table.create_row();
      data.create_column().create_link(snapshot.name).
          set_attribute(
            "target",
            boost::format("%s#%s")
            % m_snapshots_file % snapshot.name).
          set_attribute("frame","main");	
      data.create_column(boost::format("%8d") % snapshot.total_ops);
    }    

    // Errors encountered  
    html::section &errors = document->create_section("errors", false);
    foreach(const memory_op_t* op, m_errors)
    {
      const callstack_info_t& callstack =
        m_monitor->callstack(op->callstack)->first;
      std::string opstring = (boost::format("operation %d: %s : %s")
        % op->id 
        % util::to_string(op->type) 
        % print_callstack(callstack)).str();
      errors << opstring << "\n";
    }    

    const fs::path out_path = m_output_dir / m_overview_file;
    std::ofstream
      out_file(out_path.file_string().c_str(),std::ios::out|std::ios::trunc);
    if (!out_file.good()) 
    { 
      std::cerr << "manalyze: could not open output file!" << std::endl; 
      return -1; 
    } 
    if (!html::serialize(out_file, document))
    {
      std::cerr << "manalyze: could not serialize html document!" <<
        std::endl; 
      return -1; 
    }

    return 0; 
  }

  signed analysis::generate_index()
  {
    shared_ptr<html::document> document(new html::document("Index"));
    
    const fs::path out_path = m_output_dir / m_index_file;
    std::ofstream
      out_file(out_path.file_string().c_str(),std::ios::out|std::ios::trunc);
    if (!out_file.good()) 
    { 
      std::cerr << "manalyze: could not open output file!" << std::endl; 
      return -1; 
    } 
    if (!html::serialize(out_file, document))
    {
      std::cerr << "manalyze: could not serialize html document!" <<
        std::endl; 
      return -1; 
    }
    return 0; 
  }

    signed analysis::generate_frameset()
    {
      const fs::path out_path = m_output_dir / m_frameset_file;
      std::ofstream
        out_file(out_path.file_string().c_str(),std::ios::out|std::ios::trunc);
      if (!out_file.good()) 
      { 
        std::cerr << "manalyze: could not open output file!" << std::endl; 
        return -1; 
      } 
      out_file << 
        "<html><head><title>PS3 memory trace analysis</title></head>"
        "<frameset cols=\"25%,75%\">"
        "<frame src=\"./mtracedb_index.html\" name=\"index\">"
        "<frameset rows=\"80%,20%\">"
        "<frame src=\"./mtracedb_overview.html\" name=\"main\">"
        "<frame src=\"./mtracedb_callstacks.html\" name=\"detail\">"
        "</frameset>"
        "</frameset>"
        "</html"; 
      return 0; 
    }

    signed analysis::create_img_dir()
    {
      if (!fs::exists(m_image_dir))
      {
        if (!fs::create_directory(m_image_dir))
        {
          cerr() << "could not create image director" << std::endl;
          return -1; 
        }
      }
      return 0;
    }


    // The analysis of the mtrace run 
    analysis::analysis(mtracedb::memory_monitor_t* monitor)
      : m_monitor(monitor), 
        m_snapshot_state(NULL)
    {
      m_output_dir = mtracedb::get_options().data_directory; 
      m_image_dir = m_output_dir / "images";
      m_frameset_file = "mtracedb.html";
      m_index_file = "mtracedb_index.html";
      m_overview_file = "mtracedb_overview.html";
      m_callstack_file = "mtracedb_callstacks.html";
      m_snapshots_file = "mtracedb_snapshots.html";
    }

    analysis::~analysis()
    {
    }

    signed analysis::generate()
    {
      signed result = 0;

      if (create_img_dir() != 0) 
      { 
        cerr() << "could not generate image directory" << std::endl; 
        result = -1;
      } 

      if (generate_callstacks() == -1)
      {
        cerr() << "callstack generation failed" << std::endl; 
        result = -1;
      }

      if (generate_snapshots() == -1)
      {
        cerr() << "snapshot generation failed" << std::endl; 
        result = -1;
      }

      if (generate_overview() == -1)
      {
        cerr() << "overview generation failed" << std::endl; 
        result = -1;
      }

      if (generate_index() == -1)
      {
        cerr() << "overview generation failed" << std::endl; 
        result = -1;
      }

      if (generate_frameset() == -1)
      {
        cerr() << "frameset generation failed" << std::endl; 
        result = -1;
      }

      return result;
    }
  }; 

int main(int argc, char* argv[]) 
{
  // Parse the command line 
  program_options::options_description desc("mtrace options");
  desc.add_options()
    ("help", 
      "this help message")
    ("input",
      program_options::value<std::string>(&g_options.input)->default_value("network"),
      "provide a path to the input source. If not given or \"network\", tracing"
      "over the network will be performed")
    ("save",
      program_options::value<bool>(&g_options.save)->default_value(false),
      "save the networking stream")
    ("analyze",
      program_options::value<bool>(&g_options.analyze)->default_value(false),
      "analyze a previously recorded file instead of tracing the operations"
      "over the network");
  mtracedb::parse_command_line(argc, argv, desc);
  mtracedb::options& opts = mtracedb::get_options(); 

  // Create the active memory status object
  mtracedb::memory_monitor_t monitor; 

  ////////////////////////////////////////////////////////////////////////
  // Trace over the network 
  if (g_options.input == "network")
  {
    fs::path dir = opts.data_directory;
    if (!fs::exists(dir)) 
    { 
      if (!fs::create_directory(dir))
      {
        cerr() << " could not create directory" << dir <<
          std::endl; 
        return -1; 
      }
    } 

    fs::path elf_file = opts.elf_file;
    if (!fs::exists(elf_file))
    {
      cerr() << " elf file does not exist " << elf_file <<
        std::endl; 
      return -1; 
    }
    fs::path new_elf = dir / elf_file.filename();
    if (fs::exists(new_elf))
      fs::remove(new_elf);
    fs::copy_file(elf_file, new_elf); 
    opts.elf_file = new_elf.file_string();

    // Create the storage object
    std::ostream* os = NULL;
    if (g_options.save)
    {
      const std::string& storage_path = opts.elf_file + ".mtrace";
      os = new std::ofstream(storage_path.c_str(), std::ios_base::binary);
      if (!os->good())
      {
        cerr() << "could not open memory operation file " << storage_path << 
          " for reading" << std::endl; 
        EXIT();
       }
    }
    else 
      os = new mtracedb::null_stream();

    mtracedb::file_writer_t writer(*os); 

    // Receive the network stream of memory operations 
    if (mtracedb::netpump(writer, monitor) != 0)
    {
      cerr() << "netpump exited with an error" << std::endl; 
    }

    delete os;
  }
  ////////////////////////////////////////////////////////////////////////
  // Read from file 
  else
  {
    fs::path file = g_options.input;
    if (!fs::exists(file)) 
    { 
      cerr() << " input trace file does not exist" << file <<
      std::endl; 
      return -1; 
    } 

    fs::path elf_file = opts.elf_file;
    if (!fs::exists(elf_file))
    {
      cerr() << " elf file does not exist " << elf_file <<
        std::endl; 
      return -1; 
    }

    // Create the file reader 
    mtracedb::file_reader_t reader(file.file_string()); 

    if (mtracedb::filepump(reader, monitor) != 0) 
    { 
      cerr() << "filepump exited with an error" << std::endl; 
    } 
  }

  ////////////////////////////////////////////////////////////////////////
  // produce output 
  if (g_options.analyze) 
  { 
    analysis context(&monitor); 

    if (context.generate() != 0) 
    { 
      cerr() << "error generating analysis" << std::endl; 
      return -1; 
    } 
  }

  return 0; 
}
