//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// mtrace database for storing allocation information from cryengine3
// ps3 applications
// ---------------------------------------------------------------------------
// History:
// - June 14 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////

#ifndef _SOCKET_H
#define _SOCKET_H

#define MTRACE_MAX_CHUNKSIZE (4*256)

namespace mtrace
{
  // A stack based small utility buffer. Chunks can accumulate writes
  // on the stack so that the gathered data can get written in one
  // large chunk.
  template<size_t Size> 
  class CChunk_t
  {
    // The current position within the chunk
    size_t m_CurrentPosition; 

    // The actual chunk buffer 
    char m_Buffer[Size]; 

    // Ensure that the write operation fits into the buffer 
    inline void CheckSize(size_t size) const
    {
      if (m_CurrentPosition + size >= Size) 
        std::abort();
    }
  public:
    // Constructor
    CChunk_t() : m_CurrentPosition() {}; 

    // Write a value to the buffer, flush if the buffer is full.
    template<typename T> 
    inline void Write(T value)
    {
      CheckSize(sizeof value);
      memcpy(m_Buffer + m_CurrentPosition, &value, sizeof(value));
      m_CurrentPosition += sizeof(value);
    }

    // Retrieve a pointer to the data in the chunk buffer 
    const char* GetData() const { return m_Buffer; }

    // Retrieve the size of the data 
    size_t GetSize() const { return m_CurrentPosition; }
  }; 
  typedef CChunk_t<MTRACE_MAX_CHUNKSIZE> CChunk;


	// A simple networking layer. Can create a tcp connection that acts
	// as a server for a ps3 application. Note: This implementation is
	// very rudimentary and probably should be refactored!
	namespace net 
	{
		// Forward declaration of the internal connection structure
		struct connection_t; 

    // Determine if the connection is active 
    bool is_active(connection_t*);

		// Create a connection to listen to a given port 
		connection_t* create_connection(); 

		// Destroy the connection
		void destroy_connection(connection_t*); 

		// Listen and accept a single connection 
		signed listen(connection_t*); 

		// Send data over the network to the attached client 
		size_t send(connection_t*, const void*, size_t); 

		// Recieve data from the attached client 
		size_t recv(connection_t*, void*, size_t); 

		// Shutdown the connection 
		signed shutdown(connection_t*); 
	}
}

#endif // _SOCKET_H
