//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// mtrace database for storing allocation information from cryengine3
// ps3 applications
// ---------------------------------------------------------------------------
// History:
// - June 13 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////
#include "getopt.h"
#include <mtracedb/process.h>
#include <mtracedb/mtracedb.h>
#include <iostream>
#include <iosfwd>                          
#include <cstring>
#include <cstdarg>
#include <boost/iostreams/categories.hpp>
#include <boost/iostreams/device/file.hpp>  
#include <boost/iostreams/operations.hpp> 
#include <boost/iostreams/filter/line.hpp>
#include <boost/iostreams/concepts.hpp> 
#include <boost/iostreams/filtering_stream.hpp>
#include <boost/thread.hpp>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#include <Windows.h>


using namespace mtracedb; 

//#define DEBUG_MODULEINFO_RESOLVING 1
#undef DEBUG_MODULEINFO_RESOLVING

//#define TRACE_MEMORY_OPERATIONS
#undef TRACE_MEMORY_OPERATIONS

//#define TRACE_STORAGE_OPERATIONS
#undef TRACE_MEMORY_OPERATIONS

#define VERBOSE 1

namespace 
{
  // Static mutex for logging 
  static boost::mutex &log_mutex() 
  { 
    static boost::mutex _mutex; 
    return _mutex;
  } 
  #define LOG_LOCK boost::unique_lock<boost::mutex> lock(log_mutex())

  // registered log call backs 
  static std::vector<mtracedb::log_callback*> s_log_callbacks; 

  // Add a callback from the callback storage 
  static void register_log_callback(log_callback* cb) 
  { 
    LOG_LOCK; 
    s_log_callbacks.push_back(cb); 
  }

  // Remove a callback from the callback storage
  static void remove_log_callback(const log_callback* cb)
  {
    LOG_LOCK; 

    s_log_callbacks.erase(
      std::remove(s_log_callbacks.begin(), s_log_callbacks.end(), cb),
      s_log_callbacks.end());
  }


  // Line based filter that outputs to ::OutputDebugString as well 
  class line_filter : public boost::iostreams::line_filter
  {
  private:
    std::string do_filter(const std::string& line)
    {
      LOG_LOCK; 
      const std::string &newline = line + '\n';

      foreach (mtracedb::log_callback* cb, s_log_callbacks) 
        cb->on_log_msg(newline); 

      ::OutputDebugString(newline.c_str());

      return line; 
    }
  };

  // Output DebugString 
  static std::ostream &output_stream() 
  {
    static boost::iostreams::filtering_ostream os; 
    static bool initialized = false; 

    if (!initialized) 
    {
      os.push(line_filter());
      os.push(std::cout);
      initialized = true; 
    }

    return os;
  }

  // The name of the memory operation file 
  static const char* s_mem_op_name = "ps3mtrace.memops";
  // The name of the callstack file 
  static const char* s_callstack_name  = "ps3mtrace.callstacks";
  // The name of the snapshits 
  static const char* s_snapshot_name  = "ps3mtrace.snapshots";
  // The address to line process
  static process_t s_addr2line_process;
  // The gdb to line process
  static process_t s_function_process;
  // The address to line process
  static process_t s_size_process;
  // The gdb to line process
  static process_t s_ps3bin_process;

  // Simple class that manages the ps3bin address 2 line resolving 
  class ps3bin_a2l 
  {
    // Function and File pair 
    typedef std::pair<std::string, std::string> value_type; 

    // Mapping of instruction pointer address to [function, filename] 
    typedef std::map<address_t, value_type > a2l_map; 

    // The cache of previously resolved addresses 
    a2l_map m_resolve_cache; 

    ps3bin_a2l()
    {
      if (!init())
      {
        cerr() << "ps3bin_a2l() init failed" << std::endl; 
        EXIT();
      }
    } 

    bool init() 
    { 
      const options& o = get_options(); 
      const char* argv[] = {
        o.ps3bin_exe.c_str(),
        o.elf_file.c_str(),
        "-gnu",
        "-a2l", 
      }; 
      return s_ps3bin_process.run(sizeof(argv)/sizeof(const char*), argv);
    } 

    // Resolve the function of the address 
    value_type& resolve(address_t addr)
    {
      a2l_map::iterator it = m_resolve_cache.find(addr); 
      if (it != m_resolve_cache.end()) 
        return it->second; 

      const options& o = get_options(); 

      char hex_addr[64];
      sprintf(hex_addr, "0x%x\n", addr); 

      size_t written = s_ps3bin_process.write(
        process_t::STREAM_INPUT, 
        hex_addr, 
        std::strlen(hex_addr));
      if (written == 0) 
      {
        cerr() << "write to ps3bin failed!" << std::endl;
        EXIT();
      }

      char token[16<<10];
      memset(token, 0, sizeof(token)); 
      size_t read = s_ps3bin_process.read(
        process_t::STREAM_OUTPUT,
        token,
        sizeof(token));
      if (read == 0) 
      {
        cerr() << "read from ps3bin.exe failed!" << std::endl;
        EXIT();
      }

      size_t functionEnd=~0U, fileBegin=~0U, fileEnd=~0U; 
      for (size_t i=0; i<read; ++i) 
      { 
        if (token[i] == '\\') token[i] = '/'; 

        if (token[i] == '\n' || token[i] == '\r')
        {
          token[i] = '\0';
          if (functionEnd == ~0U) 
            functionEnd = i; 
          else 
          {
            if (fileBegin == ~0U) 
              fileBegin = i + 1; 
            else if (fileEnd == ~0U) 
            {
              fileEnd = i; 
              break; 
            }
          }
        }
      }
      if (functionEnd == ~0U || fileEnd == ~0U || fileBegin == ~0U) 
      { 
        cerr() << " error in ps3bin output!" << std::endl;
        cerr() << " recieved this for address " << hex_addr << std::endl; 
        cerr() << token << std::endl; 
      } 

      const std::string &functionName = std::string(token, functionEnd);
      const std::string &fileName = std::string(token + fileBegin, fileEnd);
      std::pair<a2l_map::iterator, bool> inserted =  m_resolve_cache.insert(
        std::make_pair(addr, value_type(functionName, fileName))
        );
      if (inserted.second) 
        return inserted.first->second; 
      static value_type _value = value_type("-a2l-error-", "-a2l-error-");
      return _value;
    }

  public:

    // Resolve the function name of a given instruction address 
    std::string& resolve_function(mtracedb::address_t addr)
    {
      return resolve(addr).first; 
    }

    // Resolve the filename of a given instruction address 
    std::string& resolve_filename(mtracedb::address_t addr)
    {
      return resolve(addr).second; 
    }

    static ps3bin_a2l& instance() 
    {
      static ps3bin_a2l _instance; 
      return _instance; 
    }
  }; 
}

namespace mtracedb
{
  ELogLevel from_string(const char* string) 
  {
    if (string == NULL) std::abort();
    if (std::strcmp(string, "Info") == 0) return LOG_INFO;
    if (std::strcmp(string, "Warning") == 0) return LOG_WARNING;
    if (std::strcmp(string, "Debug") == 0) return LOG_DEBUG;
    if (std::strcmp(string, "Noise") == 0) return LOG_NOISE;
    if (std::strcmp(string, "Error") == 0) return LOG_ERROR;
    std::abort(); 
    return LOG_ERROR;
  }; 
  const char* to_string(ELogLevel level) 
  { 
    switch (level) 
    { 
      case LOG_INFO: return "Info"; 
      case LOG_WARNING: return "Warning"; 
      case LOG_DEBUG: return "Debug"; 
      case LOG_NOISE: return "Noise"; 
      case LOG_ERROR: return "Error"; 
      default: break;
    }; 
    return NULL; 
  }; 

  const char* to_string(ECryModule module)
  {
    switch(module)
    {
      case eCryM_3DEngine: return "eCryM_3DEngine";
      case eCryM_Action: return "eCryM_Action";
      case eCryM_AISystem: return "eCryM_AISystem";
      case eCryM_Animation: return "eCryM_Animation";
      case eCryM_EntitySystem: return "eCryM_EntitySystem";
      case eCryM_Font: return "eCryM_Font";
      case eCryM_Input: return "eCryM_Input";
      case eCryM_Movie: return "eCryM_Movie";
      case eCryM_Network: return "eCryM_Network";
      case eCryM_Physics: return "eCryM_Physics";
      case eCryM_ScriptSystem: return "eCryM_ScriptSystem";
      case eCryM_SoundSystem: return "eCryM_SoundSystem";
      case eCryM_System: return "eCryM_System";
      case eCryM_Game: return "eCryM_Game";
      case eCryM_Render: return "eCryM_Render";
      case eCryM_Launcher: return "eCryM_Launcher";
      case eCryM_Num: return "unknown module";
    }
    return NULL; 
  }

  void log(ELogLevel level, const char* fmt, ...)
  {
    std::va_list argptr;
    va_start( argptr, fmt );

    char buffer[512]; 
    
    std::vsprintf(buffer, fmt, argptr); 
    out(level) << "mtrace " << to_string(level) << ": " 
               << buffer << std::endl; 
  }

  std::ostream &out(ELogLevel level)
  {
    static mtracedb::null_stream _stream; 
    if (level >= get_options().log_level)
      return output_stream(); 
    return _stream; 
  }
  std::ostream &cerr() 
  { 
    return out(LOG_ERROR) << "mtrace " << to_string(LOG_ERROR) << ':'; 
  }

  // Log callback construction/destruction 
  log_callback::log_callback() { register_log_callback(this); }
  log_callback::~log_callback() { remove_log_callback(this); }

  // Retrieve the options 
  options& get_options()
  {
    static options _options;
    return _options;
  }; 

  // Retrieve the config instance 
  config_map_t& config() 
  { 
    static config_map_t s_config;
    return s_config; 
  }

  // Retrieve the options specific to the mtrace database
  void option_descriptions(
    program_options::options_description& parent)
  { 
    program_options::options_description general("general options"); 
    general.add_options()
      ("log-level",
        program_options::value<std::string>()->default_value("Info"), 
        "sets the loglevel to 'info, warning, error, debug, noise'")
      ("config-file,C", 
        program_options::value<std::string>(&get_options().config_file),
        "set a configuaration variable\nexample: -c range=(1,2)")
      ("log-file,l", 
        program_options::value<std::string>(&get_options().log_file),
        "the path to the log file: --log-file=mtrace.log.txt")
      ("config-var,c", 
        program_options::value< std::vector<std::string> >(),
        "set a configuaration variable\nexample: -c range=(1,2)");
    parent.add(general);

    program_options::options_description database("database options"); 
    database.add_options()
      ("ppu-lv2-addr2line", 
        program_options::value<std::string>(&get_options().addr2line_exe)->default_value("ppu-lv2-addr2line.exe"),
        "the path to the ppu-lv2-addr2line executable")
      ("ppu-lv2-size", 
        program_options::value<std::string>(&get_options().size_exe)->default_value("ppu-lv2-size.exe"),
        "the path to the ppu-lv2-size executable")
      ("ppu-lv2-gdb", 
        program_options::value<std::string>(&get_options().gdb_exe)->default_value("ppu-lv2-gdb.exe"), 
        "the path to the ppu-lv2-gdb executable")
      ("ps3bin", 
        program_options::value<std::string>(&get_options().ps3bin_exe)->default_value("ps3bin.exe"),
        "the path to the ps3bin.exe executable")
      ("use-ps3-bin,p", 
        program_options::value< bool >(&get_options().use_ps3bin)->default_value(true),
        "use the ps3bin executable instead of the ppu-lv2-* toolchain. Note: The"
        "ps3bin executable can work on self files as well")
      ("gnuplot", 
        program_options::value<std::string>(&get_options().gnuplot_exe)->default_value("gnuplot.exe"),
        "the path to the gnuplot executable")
      ("elf-file,e", 
        program_options::value<std::string>(&get_options().elf_file), 
        "the path to the ps3 elf file")
      ("dot", 
        program_options::value<std::string>(&get_options().dot_exe), 
        "the path to the dot executable for generating svg graphs")
      ("data-dir,d", 
        program_options::value<std::string>(&get_options().data_directory), 
        "the path of the data directory into which the database will "
        "store it's files.")
      ("ipconfig", 
        program_options::value<std::string>(&get_options().ipconfig_file),
        "the path to the ipconfig file into which mtrace will write the host address");
    parent.add(database); 
  } 

  void parse_command_line(int argc, char* argv[],
    program_options::options_description& app_options)
  {
    // Parse the command line and retrieve the variables map 
    program_options::variables_map vm; 
    program_options::options_description cmdline_options;
    program_options::options_description config_options;
    option_descriptions(app_options);
    cmdline_options.add(app_options);
    config_options.add(app_options);
    try
    {
      program_options::store(
        program_options::parse_command_line(argc, argv, cmdline_options), vm);
      program_options::notify(vm);     
      // find the log level first so error reporting works
      if (vm.count("log-level")) 
      {
        const std::string& logLevel = vm["log-level"].as<std::string>();
        get_options().log_level = from_string(logLevel);
      }
      // Display help message and exit
      if (vm.count("help"))
      {
        out(LOG_INFO) << app_options << std::endl; 
        std::exit(EXIT_SUCCESS); 
      }
      // Parse the configuration file 
      if (vm.count("config-file"))
      {
        const std::string& filename =
          vm["config-file"].as<std::string>();
        std::ifstream config_file(filename.c_str()); 
        if (!config_file)
        {
          cerr() << "error: could not parse config file " << filename <<
            std::endl;
          std::exit(EXIT_FAILURE);
        }
        out(LOG_DEBUG) << "parsing config file " << filename << std::endl;
        program_options::store(
          program_options::parse_config_file(config_file, config_options), vm);
        program_options::notify(vm);
      }
      else
      {
        // No configuration file was given on the command line. Try to
        // parse the default configuration file. 
        const std::string filename(".//.mtracerc");
        std::ifstream config_file(filename.c_str());
        if (config_file)
        {
          out(LOG_DEBUG) << "parsing config file " << filename << std::endl;
          program_options::store(
            program_options::parse_config_file(config_file, config_options), vm);
          program_options::notify(vm);
        }
      }
      // Extract the configuration variables 
      if(vm.count("config-var"))
      {
        const std::vector<std::string>& cvars = 
          vm["config-var"].as< std::vector<std::string> >();
        for (std::size_t i=0; i<cvars.size(); ++i)
        {
          const std::string &cvar = cvars[i]; 
          std::string key, value; 
          std::size_t eq = cvar.find('='); 
          if (eq != std::string::npos)
          {
            key.assign(*cvar.begin(), *(cvar.begin()+eq));
            value.assign(*cvar.begin()+eq+1,
              *cvar.begin()+cvar.length());
          }
          else
            key = cvar; 
          config()[key] = value; 
          out(LOG_DEBUG) << "parsed config variable " << key << " = " << value << std::endl;
        }
      }
      // Ensure that the path to the database directory has been set
      if (!vm.count("data-dir"))
      {
        cerr() << "error: database directory has not been set!" << std::endl;
        cerr() << "please supply it with '-d' on the command line" << std::endl;
        std::exit(EXIT_FAILURE); 
      }
    }
    catch(std::exception& e)
    {
      cerr() << "error: " << e.what() << std::endl;
      std::exit(EXIT_FAILURE);
    }
  }
}

address_resolver::address_resolver() 
{
}

bool address_resolver::init()
{
  const options& o = get_options(); 
  if (!o.use_ps3bin)
  {
    const char* argv[] = {
      o.addr2line_exe.c_str(),
      "-i", 
      "-e",
      o.elf_file.c_str()
    }; 
    return s_addr2line_process.run(
      sizeof(argv)/sizeof(const char*), argv);
  }
  return true; 
}

std::string& address_resolver::resolve_ps3bin(mtracedb::address_t addr)
{
  return ps3bin_a2l::instance().resolve_filename(addr);
}

std::string address_resolver::resolve_addr2line(mtracedb::address_t addr)
{
  char hex_addr[64];
  sprintf(hex_addr, "%x\n", addr); 
		
  size_t written = s_addr2line_process.write(
    process_t::STREAM_INPUT, 
    hex_addr, 
    std::strlen(hex_addr));
  if (written == 0) 
  {
    cerr() << "write to ppu-lv2-addr2line.exe failed!" << std::endl;
    EXIT();
  }

  char token[4096];
  memset(token, 0, sizeof(token)); 
  size_t read = s_addr2line_process.read(
    process_t::STREAM_OUTPUT,
    token,
    sizeof(token));
  if (read == 0) 
  {
    cerr() << "read from ppu-lv2-addr2line.exe failed!" << std::endl;
    EXIT();
  }

  for (size_t i=0; i<read; ++i)
  {
    if (token[i] == '\\') token[i] = '/'; 
    if (token[i] == '\n' || token[i] == '\r')
    {
      token[i] = '\0'; read = i; break; 
    }
  }	
		
  return std::string(token,read); 
}

std::string& address_resolver::resolve(mtracedb::address_t addr)
{
  addr_map_t::iterator it = m_resolved_cache.find(addr);
  if (it == m_resolved_cache.end())
  {
    const options& o = get_options(); 
    std::string resolved; 

    if (!o.use_ps3bin) 
      resolved = resolve_addr2line(addr);
    else 
      resolved = resolve_ps3bin(addr);

    return m_resolved_cache.insert(
      std::make_pair(addr, resolved)).first->second; 
  }
  return it->second;
}

address_resolver& address_resolver::instance()
{
  static address_resolver _resolver; 
  return _resolver; 
}

function_resolver::function_resolver()
{
  if (!init())
  {
    cerr() << "function_resolver() init failed" << std::endl; 
    EXIT();
  }
}

function_resolver::~function_resolver()
{
}

bool function_resolver::init()
{
  const options& o = get_options(); 
  if (!o.use_ps3bin) 
  { 
    const char* argv[] = {
      o.addr2line_exe.c_str(),
      "--demangle",
      "-i", 
      "-f",
      "-e",
      o.elf_file.c_str()
    }; 
    return s_function_process.run(
      sizeof(argv)/sizeof(const char*), argv);
  }
  return true; 
}

std::string& function_resolver::resolve(mtracedb::address_t addr)
{
  addr_map_t::iterator it = m_resolved_cache.find(addr);
  if (it == m_resolved_cache.end())
  {
    const options& o = get_options(); 
    std::string resolved; 

    if (!o.use_ps3bin) 
    { 
      char hex_addr[64];
      sprintf(hex_addr, "%x\n", addr); 
		
      size_t written = s_function_process.write(
        process_t::STREAM_INPUT, 
        hex_addr, 
        std::strlen(hex_addr));
      if (written == 0) 
      {
        cerr() << "write to ppu-lv2-addr2line.exe failed!"
               << std::endl;
        EXIT();
      }

      char token[4096];
      memset(token, 0, sizeof(token)); 
      size_t read = s_function_process.read(
        process_t::STREAM_OUTPUT,
        token,
        sizeof(token));
      if (read == 0) 
      {
        cerr() << "read from ppu-lv2-addr2line.exe failed!"
               << std::endl;
        EXIT();
      }

      for (size_t i=0; i<read; ++i)
      {
        if (token[i] == '\n' || token[i] == '\r')
        {
          token[i] = '\0'; read = i; break; 
        }
      }	
		
      resolved = std::string(token,read); 
    }
    else 
      resolved = ps3bin_a2l::instance().resolve_function(addr);
    
    return m_resolved_cache.insert(
      std::make_pair(addr, resolved)).first->second; 
  }
  return it->second;
}

function_resolver& function_resolver::instance()
{
  static function_resolver _instance;
  return _instance; 
}


bool segment_stats::init()
{
  const options& o = get_options(); 
  char token[4096];

  // Execute the command!
  if (!o.use_ps3bin) 
  { 
    const char* argv[] = {
      o.size_exe.c_str(),
      "-d", 
      o.elf_file.c_str()
    }; 
  
    if (!s_size_process.run(
        sizeof(argv)/sizeof(const char*), argv))
    {
      cerr() << "could not spawn child process!" <<
        std::endl; 
      return false; 
    }
    // Read the results from the command
    memset(token, 0, sizeof(token)); 
    size_t read = s_size_process.read(
      process_t::STREAM_OUTPUT,
      token,
      sizeof(token));
    if (read == 0) 
    {
      cerr() << "read from ppu-lv2-size.exe failed!"
             << std::endl;
      return false; 
    }
  }
  else
  { 
    const char* argv[] = {
      o.ps3bin_exe.c_str(),
      o.elf_file.c_str(),
      "-dsi"
    }; 
  
    if (!s_size_process.run(
        sizeof(argv)/sizeof(const char*), argv))
    {
      cerr() << "could not spawn child process!" <<
        std::endl; 
      return false; 
    }

    // Read the results from the command
    memset(token, 0, sizeof(token)); 
    size_t read = s_size_process.read(
      process_t::STREAM_OUTPUT,
      token,
      sizeof(token));
    if (read == 0) 
    {
      cerr() << "read from ps3bin.exe failed!"
             << std::endl;
      return false; 
    }
  } 

  // Parse the result string. Skip the first line of output containing
  // the stupid output header, then parse the first 3 integers.
  char *begin, *end = token; 
  while (*end && *end != '\n') ++end;
  begin = end;
  
  m_text_size = (uint64_t) std::strtol(begin, &end, 10);

  begin = end; 
  m_data_size = (uint64_t) std::strtol(begin, &end, 10);

  begin = end; 
  m_bss_size = (uint64_t) std::strtol(begin, &end, 10);
    
  return true; 
}

segment_stats::segment_stats()
{
  if (!init())
  {
    cerr() << "segment_stats() init failed" 
                 << std::endl; 
    EXIT();
  }
}

const segment_stats& segment_stats::instance()
{
  static segment_stats _stats;
  return _stats;
}

file_writer_t::file_writer_t(std::ostream& out)
  : m_mem_op_file(out)
{}

bool file_writer_t::write(const memory_stream_element_t &op)
{
  size_t callstacks = op.callstack.addresses.size();
  
  m_mem_op_file.write((const char*)&op.type, sizeof(op.type));
  m_mem_op_file.write((const char*)&op.id, sizeof(op.id));
  m_mem_op_file.write((const char*)&op.thread_id, sizeof(op.thread_id));
  m_mem_op_file.write((const char*)&op.primary, sizeof(op.primary));
  m_mem_op_file.write((const char*)&op.secondary, sizeof(op.secondary));
  m_mem_op_file.write((const char*)&op.size, sizeof(op.size));
    
  m_mem_op_file.write((const char*)&callstacks, sizeof(callstacks));
  for (size_t i=0; i<callstacks; ++i) 
  {
    const address_t addr = op.callstack.addresses[i]; 
    m_mem_op_file.write((const char*)&addr, sizeof(addr));
  }
  
  m_mem_op_file.write(op.snapshot, sizeof(op.snapshot));
  
  return m_mem_op_file.good();
}

file_reader_t::file_reader_t(const std::string& path)
{
  m_mem_op_file.open(path.c_str(), std::ios_base::binary);
  if (!m_mem_op_file.good())
  {
    cerr() << "could not open memory operation file " << path << 
      " for reading" << std::endl; 
    EXIT();
 }
}

bool file_reader_t::read(memory_stream_element_t &op)
{
  if (m_mem_op_file)
  {
    size_t callstacks = 0;

    m_mem_op_file.read((char*)&op.type, sizeof(op.type));
    m_mem_op_file.read((char*)&op.id, sizeof(op.id));
    m_mem_op_file.read((char*)&op.thread_id, sizeof(op.thread_id));
    m_mem_op_file.read((char*)&op.primary, sizeof(op.primary));
    m_mem_op_file.read((char*)&op.secondary, sizeof(op.secondary));
    m_mem_op_file.read((char*)&op.size, sizeof(op.size));
    
    m_mem_op_file.read((char*)&callstacks, sizeof(callstacks));
    op.callstack.addresses.resize(callstacks, ~0U);
    for (size_t i=0; i<callstacks; ++i) 
    {
      address_t addr; 
      m_mem_op_file.read((char*)&addr, sizeof(addr));
      op.callstack.addresses[i] = addr;
    }

    m_mem_op_file.read((char*)op.snapshot, sizeof(op.snapshot));

    return m_mem_op_file.good();
  }
  return false; 
}

std::streampos file_reader_t::file_size()
{
  std::streampos prev, length;

  prev = m_mem_op_file.tellg();
  m_mem_op_file.seekg(0, std::ios::end);
  length = m_mem_op_file.tellg();
  m_mem_op_file.seekg(prev, std::ios::beg);
  
  return length;
}

std::streampos file_reader_t::current()
{
  std::streampos current = m_mem_op_file.tellg();
  return current; 
}


signed memory_state_t::consume(const memory_op_t& op)
{
  signed result = 0;
  ECryModule prevModule = eCryM_Num; 
  ECryModule moduleInfo = eCryM_Num; 
  size_t prev_size = 0; 
  // Update the active memory of the process 
  switch(op.type)
  {
    if (false) { case MO_MEMALIGN: ++m_memalign_calls; }
    if (false) { case MO_CALLOC: ++m_calloc_calls; }
    if (false) { case MO_MALLOC: ++m_malloc_calls; }
    moduleInfo = op.module; 
    if (allocate(op.primary, op.id, op.size, moduleInfo) != 0)
    {
#     if defined(VERBOSE)
      cerr() << util::to_string(op.type) << 
        ": trying to allocate memory twice!" <<
        std::endl;
#     endif 
      return -1;
    }
    if (moduleInfo != eCryM_Num) 
    { 
      ModuleStats &stats = m_module_stats[moduleInfo];
      stats.current_memory += op.size;
      stats.allocated += op.size;
      ++stats.num_allocations; 
      if (stats.peak_memory < stats.current_memory)
        stats.peak_memory = stats.current_memory;
    } 
    else
    {
      ModuleStats &stats = m_unknown;
      stats.current_memory += op.size;
      stats.allocated += op.size;
      ++stats.num_allocations; 
      if (stats.peak_memory < stats.current_memory)
        stats.peak_memory = stats.current_memory;
    }
    break; 
    case MO_FREE:
      ++ m_free_calls;
      if (!size(op.primary, prev_size, prevModule))
      {
#if defined(VERBOSE)
        cerr() << "free ==): previous size unknown?!" <<
          std::endl;
#endif
        result = -1;
      }
      if (deallocate(op.primary) != 0)
      {
#if defined(VERBOSE)
        cerr() << "" << util::to_string(op.type) <<
          ": could not deallocate memory!" <<
          std::endl;
#endif
        return -1;
      }
      DEBUG_CHECK(prevModule != eCryM_Num); 
      DEBUG_CHECK(prev_size != 0);
      if (prevModule != eCryM_Num) 
      { 
        ModuleStats &stats = m_module_stats[prevModule];
        stats.current_memory -= prev_size;
        stats.freed += prev_size;
      } 
      else
      {
        ModuleStats &stats = m_unknown;
        stats.current_memory -= prev_size;
        stats.freed += prev_size;
      }
      break;
    case MO_REALLOC:
      moduleInfo = op.module;
      ++m_realloc_calls;
      if (op.secondary == op.primary)
      {
        if (!allocated(op.primary, prevModule))
        {
#         if defined(VERBOSE)
          cerr() << "(realloc ==): could not update memory !" <<
            std::endl;
#         endif
          return -1;
        }
        size_t prev_size = 0; 
        if (!size(op.secondary, prev_size, prevModule))
        {
#         if defined(VERBOSE)
          cerr() << "(realloc ==): previous size unknown?!" <<
            std::endl;
#         endif
          result = -1;
        }
        if (update(op.primary, op.id, op.size) != 0)
        {
#         if defined(VERBOSE)
          cerr() << "(realloc ==): could not update memory! (realloc)" <<
            std::endl;
#         endif
          return -1;
        }
        DEBUG_CHECK(prevModule != eCryM_Num && prev_size != 0);
        if (moduleInfo != eCryM_Num) 
        { 
          ModuleStats &stats = m_module_stats[moduleInfo];
          stats.current_memory += op.size;
          stats.allocated += op.size;
          ++stats.num_allocations;
          if (stats.peak_memory < stats.current_memory)
            stats.peak_memory = stats.current_memory;
        } 
        else
        {
          ModuleStats &stats = m_unknown;
          stats.current_memory += op.size;
          stats.allocated += op.size;
          ++stats.num_allocations;
          if (stats.peak_memory < stats.current_memory)
            stats.peak_memory = stats.current_memory;
        }
        if (prevModule != eCryM_Num) 
        { 
          ModuleStats &stats = m_module_stats[prevModule];
          stats.current_memory -= prev_size;
          stats.freed += prev_size;
          ++stats.num_allocations;
        } 
        else
        {
          ModuleStats &stats = m_unknown;
          stats.current_memory -= prev_size;
          stats.freed += prev_size;
          ++stats.num_allocations;
        }
      }
      else
      {
        size_t prev_size = 0; 
        if (!size(op.secondary, prev_size, prevModule))
        {
#         if defined(VERBOSE)
          cerr() << "(realloc ==): previous size unknown?!" <<
            std::endl;
#         endif
          result = -1;
        }
        if (deallocate(op.secondary) != 0)
        {
#         if defined(VERBOSE)
          cerr() << "(realloc !=): could not deallocate memory!" <<
            std::endl;
#         endif
          return -1;
        }
        if (allocate(op.primary, op.id, op.size, moduleInfo) != 0)
        {
#         if defined(VERBOSE)
          cerr() << "(realloc !=): could not allocate memory!" <<
            std::endl;
#         endif
          return -1;
        }
        DEBUG_CHECK(prevModule != eCryM_Num && prev_size != 0);
        if (moduleInfo != eCryM_Num) 
        { 
          ModuleStats &stats = m_module_stats[moduleInfo];
          stats.current_memory += op.size;
          stats.allocated += op.size;
          ++stats.num_allocations;
          if (stats.peak_memory < stats.current_memory)
            stats.peak_memory = stats.current_memory;
        } 
        else
        {
          ModuleStats &stats = m_unknown;
          stats.current_memory += op.size;
          stats.allocated += op.size;
          ++stats.num_allocations;
          if (stats.peak_memory < stats.current_memory)
            stats.peak_memory = stats.current_memory;
        }
        if (prevModule != eCryM_Num) 
        { 
          ModuleStats &stats = m_module_stats[prevModule];
          stats.current_memory -= prev_size;
          stats.freed += prev_size;
          ++stats.num_allocations;
        } 
        else
        {
          ModuleStats &stats = m_unknown;
          stats.current_memory -= prev_size;
          stats.freed += prev_size;
          ++stats.num_allocations;
        }
      }
      break;
  }; 
  return result; 
}


signed memory_state_t::apply(const memory_op_range& range)
{
  signed result = 0;
  foreach(const memory_op_t& op, range) 
  {
    if (consume(op) != 0)
     result = -1;
  }
  return result;
}
signed memory_state_t::apply(const memory_op_t& op)
{
  return consume(op);
}

signed memory_state_t::remove(const memory_op_range& range)
{
  // TODO 
  return -1; 
}

memory_monitor_t::memory_monitor_t()
  : m_mutex(),
    m_num_memops(), 
    m_callstacks(),
    m_memory_ops(),
    m_snapshots()
{}

memory_monitor_t::~memory_monitor_t()
{
  foreach(memory_op_t* op, m_memory_ops)
    delete op; 
}

ECryModule memory_monitor_t::find_module(const callstack_info_t& callstack)
{
  // If a storage object has been attached to this instance of active
  // memory, we try to associate the allocation request to a cry
  // module if possible.
  ECryModule moduleInfo = eCryM_Num;
  ECryModule prevModule = eCryM_Num; 
  address_resolver& resolver = address_resolver::instance(); 
  for (size_t i=0; i<callstack.addresses.size(); ++i) 
  { 
    const address_t addr = callstack.addresses[i];
    const std::string& symbol = resolver.resolve(addr);
    // Skip the memory manager (as it will get called for each and
    // every allocation)
    if (symbol.find("CryMemoryManager") != std::string::npos)
      continue;
    if (symbol.find("STLPORT") != std::string::npos)
      continue;
    if (symbol.find("/CryCommon/") != std::string::npos &&
      symbol.find(".cpp") == std::string::npos)
      continue;
    // Does the operation originate from CrySystem?
    if (symbol.find("Code/CryEngine/Cry3DEngine/") != std::string::npos)
    {
      moduleInfo = eCryM_3DEngine; 
      break;
    }
    // Does the operation originate from CryAction?
    if (symbol.find("Code/CryEngine/CryAction/") != std::string::npos)
    {
      moduleInfo = eCryM_Action; 
      break;
    }
    // Does the operation originate from CryAISystem?
    if (symbol.find("Code/CryEngine/CryAISystem/") != std::string::npos)
    {
      moduleInfo = eCryM_AISystem; 
      break;
    }
    // Does the operation originate from CryAnimation?
    if (symbol.find("Code/CryEngine/CryAnimation/") != std::string::npos)
    {
      moduleInfo = eCryM_Animation; 
      break;
    }
    // Does the operation originate from CryEntitySystem?
    if (symbol.find("Code/CryEngine/CryEntitySystem/") != std::string::npos)
    {
      moduleInfo = eCryM_EntitySystem; 
      break;
    }
    // Does the operation originate from CryFont?
    if (symbol.find("Code/CryEngine/CryFont/") != std::string::npos)
    {
      moduleInfo = eCryM_Font; 
      break;
    }
    // Does the operation originate from CryFont?
    if (symbol.find("Code/CryEngine/CryInput/") != std::string::npos)
    {
      moduleInfo = eCryM_Input; 
      break;
    }
    // Does the operation originate from CryMovie?
    if (symbol.find("Code/CryEngine/CryMovie/") != std::string::npos)
    {
      moduleInfo = eCryM_Movie; 
      break;
    }
    // Does the operation originate from CryNetwork?
    if (symbol.find("Code/CryEngine/CryNetwork/") != std::string::npos)
    {
      moduleInfo = eCryM_Network; 
      break;
    }
    // Does the operation originate from CryPhysics?
    if (symbol.find("Code/CryEngine/CryPhysics/") != std::string::npos)
    {
      moduleInfo = eCryM_Physics; 
      break;
    }
    // Does the operation originate from CryScriptSystem?
    if (symbol.find("Code/CryEngine/CryScriptSystem") !=
      std::string::npos)
    {
      moduleInfo = eCryM_ScriptSystem; 
      break;
    }
    // Does the operation originate from Renderer?
    if (symbol.find("Code/CryEngine/RenderDll") !=
      std::string::npos)
    {
      moduleInfo = eCryM_Render;
      break; 
    }
    // Does the operation originate from CrySystem?
    if (symbol.find("Code/CryEngine/CrySystem") !=
      std::string::npos)
    {
      moduleInfo = eCryM_System; 
      break;
    }
    // Does the operation originate from PS3Launcher?
    if (symbol.find("PS3Launcher") != std::string::npos && 
      symbol.find("PS3MemoryInterface") == std::string::npos)
    {
      moduleInfo = eCryM_Launcher;
      break;
    }
    // Does the operation originate from GameDll?
    if ((symbol.find("Code/Game02") != std::string::npos 
        || symbol.find("Code/Game03") != std::string::npos)
      && symbol.find("GameDll") != std::string::npos)
    {
      moduleInfo = eCryM_Game; 
      break;
    }
  } 
  if (moduleInfo == eCryM_Num)
  {
    static std::set<unsigned> already_analyzed;
    unsigned cs = callstack.hash();
    if (already_analyzed.find(cs) == already_analyzed.end())
    {
      out(LOG_DEBUG) << "callstack associated to module: " 
                     << to_string(moduleInfo) << std::endl; 
      for (size_t i=0; i<callstack.addresses.size(); ++i) 
      { 
        const address_t addr = callstack.addresses[i];
        const std::string& symbol = resolver.resolve(addr);
        out(LOG_DEBUG) << "\t " <<  symbol << std::endl; 
      }
      out(LOG_DEBUG) << "\t " <<  std::endl; 
      already_analyzed.insert(cs);
    }
  }
  return moduleInfo;
}

bool memory_monitor_t::process(
  const std::vector<memory_stream_element_t*>& elements)
{
  // Consume elements as they are coming 
  foreach (memory_stream_element_t* e, elements)
  { 
    memory_stream_element_t& element = *e; 
    if (process(element) == false)
      cerr() << "memory_monitor_t::process error processing element" <<
        std::endl;
  } 
  return true;
}

bool memory_monitor_t::process(const memory_stream_element_t& element)
{
  switch (element.type)
  {
    // Handle regular memory allocations  
    case MO_CALLOC:
    case MO_MALLOC:
    case MO_FREE:
    case MO_REALLOC:
    case MO_MEMALIGN:
    {
      boost::unique_lock<boost::mutex> lock(m_mutex); 
      const uint32_t cs_hash = element.callstack.hash();

      callstack_map_t::iterator it = m_callstacks.find(cs_hash); 
      if (it == m_callstacks.end())
      {
        it = m_callstacks.insert(
          it, 
          std::make_pair(cs_hash, 
            std::make_pair(
              element.callstack, 
              find_module(element.callstack))));
      }

      memory_op_t* operation = new memory_op_t;
      operation->type = element.type; 
      operation->id = m_num_memops++;
      operation->thread_id = element.id; 
      operation->primary = element.primary; 
      operation->secondary = element.secondary; 
      operation->size = element.size; 
      operation->callstack = cs_hash;
      operation->module = it->second.second;

      m_memory_ops.push_back(operation);
    }
    break; 

    // Handle snapshots 
    case MO_SNAPSHOT:
    {
      boost::unique_lock<boost::mutex> lock(m_mutex); 
      snapshot_t snapshot; 
      snapshot.name = element.snapshot; 
      snapshot.id = m_num_memops;
      m_snapshots.push_back(snapshot);
    }
    break;

    // Skip the stat requests, they are handled differently 
    case MO_STAT_REQUEST:
      break;
  }

  return true; 
}

// Constructor
call_graph::call_graph()
  : m_next_function_id()
{
}

signed call_graph::analyze(memory_monitor_t* monitor)
{
  function_resolver &fn_resolve =
    function_resolver::instance();
  address_resolver &addr_resolve =
    address_resolver::instance();

  memory_op_range range(monitor);
  
  foreach(const memory_op_t& op, range)
  {
    const callstack_info_t &cs  = monitor->callstack(op.callstack)->first;
    shared_ptr<function_info> last;
    address_t last_address = ~0U; 
    for (size_t i=cs.addresses.size()-1; i>0; --i)
    {
      const address_t addr = cs.addresses[i];
      const std::string& fn_name = fn_resolve.resolve(addr); 
      std::map< std::string, std::size_t >::iterator it =
        m_function_names.find(fn_name);
      shared_ptr<function_info> fn;
      if (it == m_function_names.end())
      {
        fn = shared_ptr<function_info>(new function_info);
        fn->name = fn_name;
        fn->id = m_next_function_id++;
        m_vertices.push_back(fn);
        m_function_names.insert(std::make_pair(fn_name, fn->id));
      }
      else
        fn = m_vertices[it->second];
      if (last) 
      {
        call_info ci;
        ci.target_id = fn->id;
        ci.source_info = addr_resolve.resolve(last_address);
        last->calls.insert(std::make_pair(fn_name, ci));
      }
      last = fn; 
      last_address = addr; 
    }
  }

  foreach(const memory_op_t& op, range)
  {
    const callstack_info_t &cs  = monitor->callstack(op.callstack)->first;
    shared_ptr<function_info> last;
    address_t last_address = ~0U; 
    for (size_t i=cs.addresses.size()-1; i>0; --i)
    {
      const address_t addr = cs.addresses[i];
      const std::string& fn_name = fn_resolve.resolve(addr); 
      const std::size_t fn_id = m_function_names[fn_name]; 
      shared_ptr<function_info> fn = m_vertices[fn_id]; 

      switch(op.type)
      {
        case MO_CALLOC: 	  // call to calloc
        case MO_MALLOC:     // call to malloc
        case MO_MEMALIGN: 	// call to memalign 
          fn->total_allocated += op.size;
          if (i==1)
          {
            fn->self_allocated += op.size; 
          }
          if (last && last_address != ~0U) 
          { 
            DEBUG_CHECK( last->calls.find(fn->name)!=last->calls.end() );
            if (last->calls.find(fn->name)!=last->calls.end())
            {
              call_info& ci = last->calls[fn->name]; 
              ci.allocated += op.size; 
            }
          }
          break;

        case MO_FREE:      // call to free
        {
          std::size_t size = ~0U; 
          fn->total_freed += size;
          if (i==1)
          {
            fn->self_freed += size;
          }
          if (last && last_address != ~0U) 
          { 
            DEBUG_CHECK( last->calls.find(fn->name)!=last->calls.end() );
            if (last->calls.find(fn->name)!=last->calls.end())
            {
              call_info& ci = last->calls[fn->name]; 
              ci.freed += size; 
            }
          } 
        }
        break;

        case MO_REALLOC: 	// call to realloc
        {
          std::size_t size = ~0U; 
          fn->total_freed += size;
          fn->total_allocated += op.size;
          if (i==1)
          {
            fn->self_freed += size;
            fn->self_allocated += op.size;
          }
          if (last && last_address != ~0U) 
          { 
            DEBUG_CHECK( last->calls.find(fn->name)!=last->calls.end() );
            if (last->calls.find(fn->name)!=last->calls.end())
            {
              call_info& ci = last->calls[fn->name]; 
              ci.allocated += op.size; 
              ci.freed += size; 
            }
          } 
        }
        break;

        default:
          goto skip_op;
          break; 
      }

    skip_op:
      last = fn; 
      last_address = addr; 
    }
  }

  return 0; 
}

#if 0
// Retrieve all allocations summed by callstack 
std::size_t memory_state_t::callstack(std::vector<alloc_info_t>& result) const
{
  alloc_info_t ai; 
  std::map<uint32_t, alloc_info_t> map;
  for (allocation_map_t::const_iterator it = m_address_map.begin();
       it != m_address_map.end(); ++it) 
  { 
    const allocation_t& alloc = it->second;
    alloc_info_t& ai = map[alloc.callstack];
    ai.cs_hash = alloc.callstack; 
    ai.num_allocs += 1;
    ai.size += alloc.size; 
  } 
  for(std::map<uint32_t, alloc_info_t>::iterator it = map.begin(); 
      it != map.end(); ++it) 
  { 
    result.push_back(it->second);
  } 
  std::sort(result.begin(), result.end());
  return result.size();
}

// Retrieve all allocations by size 
std::size_t memory_state_t::size(std::vector<alloc_info_t>& result) const
{
  alloc_info_t ai; 
  for (allocation_map_t::const_iterator it=m_address_map.begin(); 
       it != m_address_map.end(); ++it) 
  { 
    const allocation_t& alloc = it->second;
    ai.size = alloc.size; 
    ai.cs_hash = alloc.callstack;
    result.push_back(ai);
  } 
  std::sort(result.begin(), result.end());
  return result.size();
}

// Retrieve all allocations by age 
std::size_t memory_state_t::age(std::vector<alloc_info_t>& result) const
{
  alloc_info_t ai; 
  uint64_t last_id = 0;
  for (allocation_map_t::const_iterator it = m_address_map.begin(); 
       it != m_address_map.end(); ++it) 
  { 
    const allocation_t& alloc = it->second;
    last_id = alloc.id;
    ai.size = alloc.size; 
    ai.cs_hash = alloc.callstack;
    ai.id = alloc.id;
    result.push_back(ai);
  } 
  std::sort(result.begin(), result.end(), by_age());
  return result.size();	
}

void memory_state_t::clear()
{
  m_allocation_count = 0;
  m_malloc_calls = 0;
  m_calloc_calls = 0;
  m_realloc_calls = 0;
  m_reallocalign_calls = 0;
  m_memalign_calls = 0;
  m_free_calls = 0;
  m_allocated = 0;
  m_peak_memory = 0; 
  m_allocated = 0; 
  for (size_t i=0; i<eCryM_Num; ++i) 
    memset(&m_module_stats[i], 0, sizeof(m_module_stats[i]));
  memset(&m_unknown, 0, sizeof(m_unknown));

  m_address_map.clear();
}
#endif


