//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// mtrace server for recieving allocation information from cryengine3
// ps3 applications
// ---------------------------------------------------------------------------
// History:
// - June 13 2009 - Created by Christopher Raine 
//
//////////////////////////////////////////////////////////////////////////////

#include "socket.h"

#include <mtracedb/mtracedb.h>

#include <time.h>
#include <list>
#include <iostream>
//Win32 specific?
#include <signal.h>

//#define ENABLE_STREAM_DEBUGGING 1
#if defined(ENABLE_STREAM_DEBUGGING)
# define STREAM_DEBUG_TAG (0xdeadbeefU)
#endif

//#define TRACE_RECIEVE_OPERATIONS
#undef TRACE_RECIEVE_OPERATIONS

using mtracedb::cerr;
using mtracedb::out;

namespace 
{
	// The memory operation structure as sent from the client 
	struct client_memory_operation
	{
		// The operation id of the memory operation 
		uint32_t opid;
		// The type of the memory operation 
		uint16_t fn_type; 
		// The type of the memory operation 
		uint16_t thread_id; 
		// The primary address
		mtracedb::address_t primary; 
		// The secondary address
		mtracedb::address_t secondary; 
		// The size of the operation 
		size_t size; 
	};

  // Win32 && Win64 specific functionality 
#if defined(WIN32) || defined(WIN64)
  // Handles the case when SIGINT is sent to the application 
	extern "C" void sigint_handler(int)
	{
		::signal(SIGINT,  sigint_handler);
		cerr() << " SIGINT recieved" << std::endl;
    if (mtracedb::s_main_loop && !mtracedb::s_exit_request) 
		{
			cerr() << " wants exit flag set" << std::endl;
			mtracedb::s_exit_request = true; 
		}
    else 
    { 
      ::signal(SIGINT,  SIG_DFL);
      ::raise(SIGINT);
    } 
	}

  // Handles the case when SIGINT is sent to the application 
	extern "C" void sigabrt_handler(int)
	{
		cerr() << " SIGABRT recieved" << std::endl;
		if (mtracedb::s_main_loop && !mtracedb::s_exit_request) 
		{
			cerr() << " wants exit flag set" << std::endl;
			mtracedb::s_exit_request = true; 
		}
    else
    { 
      ::signal(SIGABRT,  SIG_DFL);
      ::raise(SIGABRT);
    } 
	}

  // Handles the case when SIGINT is sent to the application 
	extern "C" void sigterm_handler(int)
	{
		cerr() << " SIGTERM recieved" << std::endl;
		if (mtracedb::s_main_loop && !mtracedb::s_exit_request) 
		{
			cerr() << " wants exit flag set" << std::endl;
			mtracedb::s_exit_request = true; 
		}
    else 
    { 
      ::signal(SIGTERM,  SIG_DFL);
      ::raise(SIGTERM);
    } 
	}
	
	// Installs the signal handler to catch SIGINTS 
	void install_signal_handlers()
	{
		// Install signal handlers 
		::signal(SIGINT,  sigint_handler);
		::signal(SIGABRT, sigabrt_handler);
		::signal(SIGTERM, sigterm_handler);
	}
#endif

  // The operation buffer 
  typedef std::vector<mtracedb::memory_stream_element_t*> op_buffer_t; 

  // Simple multithreaded stream queue
  class op_stream 
  { 
    // The mutex
    boost::mutex m_mutex; 

    // The condition variable
    boost::condition_variable_any m_condition;

    // The thread pumping the elements to the memory monitor
    boost::thread m_thread; 

    // The actual buffer of elements 
    op_buffer_t m_stream_ops; 

    // The monitor receiving the data 
    mtracedb::memory_monitor_t* m_monitor;

    // The file writer receiving the operation data
    mtracedb::file_writer_t* m_file_writer;

    // Volatile flag depicting if processing should complete
    volatile bool m_done; 

    // Consume elements produced by the "produce function"
    void consume() 
    { 
      op_buffer_t queue; 
      boost::unique_lock<boost::mutex> guard(m_mutex, boost::defer_lock_t()); 

      do 
      { 
        guard.lock(); 
        while (m_stream_ops.size() == 0)
        {
          m_condition.wait(guard);
          if (m_done) 
            goto done;
        }
        done:      

        queue.resize(m_stream_ops.size(), NULL); 
        std::copy(m_stream_ops.begin(), m_stream_ops.end(), queue.begin());
        m_stream_ops.clear();
        guard.unlock();
        
        m_monitor->process(queue); 

        foreach (mtracedb::memory_stream_element_t* e, queue)
        {
          m_file_writer->write(*e);
          delete e; 
        }
        
        queue.clear(); 
        
       } while(!m_done);
    } 

  public: 

    // Constructor 
    op_stream(
      mtracedb::memory_monitor_t* monitor,
      mtracedb::file_writer_t* file_writer
      )
      : m_mutex(),
        m_condition(),
        m_thread(),
        m_stream_ops(),
        m_done(false),
        m_monitor(monitor),
        m_file_writer(file_writer)
    {
      // Start the thread 
      m_thread = boost::thread(boost::mem_fn(&op_stream::consume), this);
    }

    // Destructor 
    ~op_stream()
    {
      m_done = true; 
      m_condition.notify_all();
      m_thread.join();
    }
      
    // Lock the stream 
    void lock() 
    { m_mutex.lock(); }

    // Unlock and notify the stream variable
    void unlock_and_notify() 
    { 
      m_mutex.unlock(); 
      if (m_stream_ops.size()) m_condition.notify_all();
    }

    // Produce elements to the stream 
    // Note: the buffer has to be 
    void produce(mtracedb::memory_stream_element_t* element)
    {
      m_stream_ops.push_back(element);
    }
  }; 


  // A running state 
  class running_memory_state_t
  { 
    mtracedb::memory_state_t m_memory_state;

    mtracedb::memory_monitor_t* m_memory_monitor;

    // The mutex guarding the memory state
    boost::mutex m_mutex; 

    // The thread pumping the elements to the memory monitor
    boost::thread m_thread; 

    // Volatile flag depicting if processing should complete
    volatile bool m_done; 

    void pump() 
    { 
      size_t begin = 0; 
      while (!m_done) 
      { 
        size_t end = m_memory_monitor->num_ops();
        mtracedb::memory_op_range range(m_memory_monitor, begin, end); 

        if (end - begin > 0)
        {
          boost::unique_lock<boost::mutex> lock(m_mutex);
          m_memory_state.apply(range);
        }

        begin = end; 

        boost::thread::yield();
      } 
    } 

  public:

    // Constructor 
    running_memory_state_t(mtracedb::memory_monitor_t* monitor)
      : m_memory_state(),
        m_memory_monitor(monitor),
        m_done(false)
    {
      // Start the thread 
      m_thread = boost::thread(
        boost::mem_fn(&running_memory_state_t::pump), this);
    }

    // Destructor 
    ~running_memory_state_t()
    {
      m_done = true; 
      m_thread.join();
    }
    
    // Fill in the statistics for the memory fields
    void fill_stats(mtracedb::SystemStats& stats)
    {
      boost::unique_lock<boost::mutex> lock(m_mutex);
      m_memory_state.fill_stats(stats);
    }
  }; 
  
  inline size_t flush_buffer(
    op_buffer_t& op_buffer, 
    op_stream& op_stream, 
    uint64_t &last_memop, 
    bool force = false)
  {
    size_t num_ops = 0;
    op_stream.lock();
    while (!op_buffer.empty())
    {
      const uint64_t id = op_buffer.front()->id;
      if ((id-last_memop > 1 && op_buffer.size() < 0x7ff) && !force)
        break; 
      last_memop = id;
      ++num_ops; 
      op_stream.produce(op_buffer.front());
      op_buffer.erase(op_buffer.begin()); 
    }
    op_stream.unlock_and_notify();
    return num_ops;
  }
}

#if defined(ENABLE_STREAM_DEBUGGING)
#define CHECK_FOR_DEBUG_TAG(connection)                                 \
  {                                                                     \
    uint32_t tag = mtracedb::util::endian_swap(STREAM_DEBUG_TAG);       \
    std::size_t sent = mtrace::net::recv(                               \
      connection, &tag, sizeof(tag));                                   \
    if (sent == ~0U || sent < 1)                                        \
    {                                                                   \
      cerr() << " error while sending data to client!"        \
                << std::endl;                                           \
      std::exit(EXIT_FAILURE);                                          \
    }                                                                   \
    tag = mtracedb::util::endian_swap(tag);                             \
    if (tag != STREAM_DEBUG_TAG)                                        \
    {                                                                   \
      cerr() << " " << __FILE__ << ':' <<  __LINE__ <<        \
        " stream debug tag not present (got "                           \
                << tag << ")!" << std::endl;                            \
      std::exit(EXIT_FAILURE);                                          \
    }                                                                   \
  }
#define SEND_DEBUG_TAG(connection)                                      \
  {                                                                     \
    uint32_t tag = mtracedb::util::endian_swap(STREAM_DEBUG_TAG);       \
    std::size_t sent = mtrace::net::send(                               \
      connection, &tag, sizeof(tag));                                   \
    if (sent == ~0U || sent < 1)                                        \
    {                                                                   \
      cerr() << " error while sending data to client!"        \
                << std::endl;                                           \
      std::exit(EXIT_FAILURE);                                          \
    }                                                                   \
  }
#else
#define CHECK_FOR_DEBUG_TAG(connection) 
#define SEND_DEBUG_TAG(connection)                                      
#endif 

// Has the application recieved a request to exit?
volatile bool mtracedb::s_exit_request = false; 

// Is the application in the main loop?
volatile bool mtracedb::s_main_loop = false; 

int mtracedb::netpump(
  mtracedb::file_writer_t& file_writer, 
  mtracedb::memory_monitor_t& monitor) 
{
	// install the signal handlers 
	install_signal_handlers(); 
  mtracedb::options& opts = mtracedb::get_options(); 

  // The async stream that pumps the input into the monitor and the file
  op_stream stream(&monitor, &file_writer);

  // The running memory state that will be queried for module info stats 
  running_memory_state_t active_state(&monitor); 

  // Starting the address resolving instance 
  mtracedb::address_resolver& resolve = 
    mtracedb::address_resolver::instance(); 

  // Write the ipconfig file if not empty 
  if (!opts.ipconfig_file.empty())
  {
    const char* ipaddr = mtrace::net::ipconfig();
    if (!ipaddr) 
    { 
      cerr() << "could not retrieve host ip address" << std::endl; 
      return -1; 
    } 
    std::ofstream os(opts.ipconfig_file.c_str()); 
    if (!os) 
    { 
      cerr() << "could not write to ipconfig file " << 
        opts.ipconfig_file << std::endl; 
      return -1; 
    } 
    os << ipaddr << std::endl;
  }

	// Create the server connection 
	mtrace::net::connection_t* connection = 
		mtrace::net::create_connection(); 
	if (connection == NULL) 
	{ 
		cerr() << " could not create server" << std::endl;
		return -1; 
	} 

	// Listen for a client and accept it's connection 
  out(mtracedb::LOG_INFO) << " waiting for client to connect" << std::endl;
	if (mtrace::net::listen(connection) == -1) 
	{
		cerr() << " could accept client connection" <<
			std::endl;
		return -1; 
	}

	// Recieve allocation information from the client 
  out(mtracedb::LOG_INFO) << " client connected, recieving data" <<
    std::endl; 
	mtracedb::callstack_info_t cstack; 
	std::size_t recieved = 0;
	clock_t last_tick = 0;
  uint64_t last_memop = 0;  
  op_buffer_t op_buffer; 
	size_t num_snapshots = 0, num_ops = 0, num_stats = 0; 
	clock_t update_ops = 0, update_stats = 0; 
	while(!s_exit_request && mtrace::net::is_active(connection))
	{
		if (!s_main_loop) s_main_loop = true; 
		// Recieve a few bytes 
		char cmd = '\0'; 
		recieved = mtrace::net::recv(connection, &cmd, sizeof(cmd)); 
		if (recieved == ~0U || recieved < 1)
		{
			cerr() << " error while recieving data from client!"
								<< std::endl; 
			return -1; 
		}

		// Extract the operation from the buffer
		switch (cmd) 
		{

    ////////////////////////////////////////////////////////////////////
    // Client has requested to recieve an update of the local statistics 
    case 'I': 
    {
      ++num_stats; 
      clock_t start = clock(); 
      CHECK_FOR_DEBUG_TAG(connection);

      // Retrieve the memory statistics
      SystemStats s;
      active_state.fill_stats(s);

      // Endian swap the statistics 
      s.allocations.malloc_calls  = 
        mtracedb::util::endian_swap(s.allocations.malloc_calls);
      s.allocations.calloc_calls  = 
        mtracedb::util::endian_swap(s.allocations.calloc_calls);
      s.allocations.realloc_calls = 
        mtracedb::util::endian_swap(s.allocations.realloc_calls);
      s.allocations.memalign_calls = 
        mtracedb::util::endian_swap(s.allocations.memalign_calls);
      s.allocations.free_calls = 
        mtracedb::util::endian_swap(s.allocations.free_calls);
      s.allocations.allocated_memory =
        mtracedb::util::endian_swap(s.allocations.allocated_memory);
      s.allocations.freed_memory = 
        mtracedb::util::endian_swap(s.allocations.freed_memory);
      for (size_t i=0; i<mtracedb::eCryM_Num; ++i)
      {
        s.modules[i].current_memory = 
          mtracedb::util::endian_swap(s.modules[i].current_memory);
        s.modules[i].peak_memory = 
          mtracedb::util::endian_swap(s.modules[i].peak_memory);
        s.modules[i].allocated = 
          mtracedb::util::endian_swap(s.modules[i].allocated);
        s.modules[i].freed = 
          mtracedb::util::endian_swap(s.modules[i].freed);
        s.modules[i].num_allocations = 
          mtracedb::util::endian_swap(s.modules[i].num_allocations);
      }
      s.unknown.current_memory = 
        mtracedb::util::endian_swap(s.unknown.current_memory);
      s.unknown.peak_memory = 
        mtracedb::util::endian_swap(s.unknown.peak_memory);
      s.unknown.allocated = 
        mtracedb::util::endian_swap(s.unknown.allocated);
      s.unknown.freed = 
        mtracedb::util::endian_swap(s.unknown.freed);
      s.unknown.num_allocations = 
        mtracedb::util::endian_swap(s.unknown.num_allocations);

      mtrace::CChunk chunk; 
      chunk.Write('I'); 
      chunk.Write(s);
#if defined(ENABLE_STREAM_DEBUGGING)
      chunk.Write(mtracedb::util::endian_swap(STREAM_DEBUG_TAG));
#endif

      std::size_t sent = mtrace::net::send(
        connection, chunk.GetData(), chunk.GetSize());
      if (sent == ~0U || sent < 1) 
      {
				cerr() << " error while sending data to client!"
									<< std::endl; 
				return -1; 
      }

      update_stats += clock() - start; 
    }
    break;

    ////////////////////////////////////////////////////////////////////
    // Client has performed a memory operation 
		case 'M': 
		{
      clock_t start = clock(); 
			// Extract the memory operation from the client;
      client_memory_operation cmo;
			recieved = mtrace::net::recv(connection,
				&cmo, sizeof(client_memory_operation)); 
			if (recieved == ~0U || recieved < 1)
			{
				cerr() << " error while recieving data from client!"
									<< std::endl; 
				return -1; 
			}
			// Swap the endianess of the data 
			cmo.opid = mtracedb::util::endian_swap(cmo.opid); 
			cmo.fn_type = mtracedb::util::endian_swap(cmo.fn_type); 
			cmo.thread_id = mtracedb::util::endian_swap(cmo.thread_id); 
			cmo.primary = mtracedb::util::endian_swap(cmo.primary); 
			cmo.secondary = mtracedb::util::endian_swap(cmo.secondary); 
			cmo.size = mtracedb::util::endian_swap(cmo.size); 
      uint32_t memop_id   = cmo.opid;
#if defined(ENABLE_STREAM_DEBUGGING)
			if (cmo.fn_type == mtracedb::MO_UNKNOWN)
			{
				char debug_token[256]; 
				std::sprintf(debug_token, "unknown type token 0x%x", cmo.fn_type);
				cerr() << debug_token << std::endl;
				return -1;
			}
#endif 

      CHECK_FOR_DEBUG_TAG(connection);

			// Extract the callstack from the stream
			while(true) 
			{ 
				mtracedb::address_t callstack_addr = 0; 
				recieved = mtrace::net::recv(connection,
					&callstack_addr, sizeof(callstack_addr)); 
				if (recieved == ~0U || recieved < 1)
				{
					cerr() << " error while recieving data from client!"
										<< std::endl; 
					return -1; 
				}
				callstack_addr = mtracedb::util::endian_swap(callstack_addr);
				if (callstack_addr == 0xffffffffU)
					break;
				cstack.addresses.push_back(callstack_addr);
			} 

      CHECK_FOR_DEBUG_TAG(connection);

      mtracedb::memory_stream_element_t* element = new 
        mtracedb::memory_stream_element_t; 
      element->type = mtracedb::util::from_func(cmo.fn_type); 
      element->id = cmo.opid; 
      element->thread_id = cmo.thread_id;
      element->primary = cmo.primary;
      element->secondary = cmo.secondary;
      element->size = cmo.size; 
      element->callstack = cstack; 
      std::memset(element->snapshot, 0x0, sizeof(element->snapshot));

      bool inserted = false; 
      uint64_t smallest_id = last_memop; 
      for (op_buffer_t::iterator it=op_buffer.begin(); 
           it!=op_buffer.end(); ++it)
      {
        if (memop_id <= (*it)->id)
        {
          op_buffer.insert(it, element);
          inserted = true;
          break;
        }
      }
      if (!inserted)
        op_buffer.push_back(element);

      num_ops += flush_buffer(
        op_buffer, stream, last_memop);

      // clear the addresses 
      cstack.addresses.clear(); 
      update_ops += clock() - start; 
		}
		break;

    ////////////////////////////////////////////////////////////////////
    // Client has requests a snapshot to be made 
		case 'S': 
		{
			++num_snapshots;
			// Extract the actual characters
			std::string name; 
			char c = '\0';
			do 
			{
				recieved = mtrace::net::recv(connection, &c, sizeof(c)); 
				if (recieved == ~0U || recieved < 1)
				{
					cerr() << " error while recieving data from client!"
										<< std::endl; 
					return -1; 
				}
				if (c != '\0')
					name.append(1,c);
			} while(c != '\0');

      mtracedb::memory_stream_element_t* element = new 
        mtracedb::memory_stream_element_t; 
      element->type = mtracedb::MO_SNAPSHOT; 
      element->id = last_memop; 
      element->thread_id = 0;
      element->primary = 0;
      element->secondary = 0;
      element->size = 0; 
      const size_t len = std::min(name.length(),sizeof(element->snapshot));
      std::strncpy(element->snapshot, name.c_str(), len);

      CHECK_FOR_DEBUG_TAG(connection);

      uint64_t memop_id = element->id;
      bool inserted = false; 
      for (op_buffer_t::iterator it=op_buffer.begin(); 
           it!=op_buffer.end(); ++it)
      {
        if (memop_id <= (*it)->id)
        {
          op_buffer.insert(it, element);
          inserted = true;
          break;
        }
      }
      if (!inserted)
        op_buffer.push_back(element);

      num_ops += flush_buffer(
        op_buffer, stream, last_memop);
		}
		break;

		default: // unknown command recieved from client
			cerr() << " unknown command recieved from client" <<
				std::endl; 
			cerr() << cmd << std::endl;
			break; 
		}

		clock_t tick = clock();
		if (tick - last_tick > (CLOCKS_PER_SEC))
		{
			out(mtracedb::LOG_INFO) << " recieved " << num_ops << 
				" mem ops, " << 
				num_snapshots << " snapshots, " << 
        num_stats << " stats " << std::endl;
      update_ops = update_stats = 0;
      num_ops = num_stats = 0;
			last_tick = tick; 
		}
	}

  // Flush all not completed memory operations so they definetly land 
  // in the storage space  
  flush_buffer(
    op_buffer, stream, last_memop, true);

  mtrace::net::destroy_connection(connection);

	return 0; 
}
