//////////////////////////////////////////////////////////////////////////////
//
// Crytek Source File.
// Copyright (C), Crytek Studios, 2007.
// ---------------------------------------------------------------------------
// Description:
// Process communication helpers
// ---------------------------------------------------------------------------
// History:
// - June 30 2009 - Created by Christopher Raine 
//////////////////////////////////////////////////////////////////////////////

#include <sstream>
#include <iostream> 


#if defined(_MSC_VER)
#include <Windows.h> 
#include <tchar.h>
#include <cstdio> 
#include <strsafe.h>
#endif

#include <mtracedb/mtracedb.h>
#include <mtracedb/process.h>
using namespace mtracedb; 


class process_t::internals
{
	// A pipe are handles to streams from which data can be read or written to.
	struct Pipe { HANDLE in, out;  }; 

	// The stream pipes
	Pipe m_pipes[3]; 

	// The handle to the child process descriptor
	HANDLE m_child_handle; 

	// The handle to the child's main thread 
	HANDLE m_child_thread; 

	// Create the pipe 
	bool create_pipe(Pipe& pipe, bool inheritIn)
	{
		SECURITY_ATTRIBUTES saAttr; 

		saAttr.nLength = sizeof(SECURITY_ATTRIBUTES); 
		saAttr.bInheritHandle = TRUE; 
		saAttr.lpSecurityDescriptor = NULL; 

		// Create the pipes for input, output and others
		if ( !CreatePipe(&pipe.in, &pipe.out, &saAttr, 0) ) 
		{
			std::cerr << "mtracedb: CreatePipe failed!" << std::endl; 
			return false; 
		}
		
		if (inheritIn) 
		{ 
			if ( !SetHandleInformation(pipe.in, HANDLE_FLAG_INHERIT, 0) )
			{
				std::cerr << "mtracedb: SetHandleInformation failed!" <<
					std::endl; 
				return false;
			}
		} 
		else 
		{ 
			if ( !SetHandleInformation(pipe.out, HANDLE_FLAG_INHERIT, 0) )
			{
				std::cerr << "mtracedb: SetHandleInformation failed!" <<
					std::endl; 
				return false;
			}
		} 
			
		return true; 
	}

  bool closehandle(HANDLE handle)
  {
    if (handle != INVALID_HANDLE_VALUE)
      return CloseHandle(handle) != 0;
    return true; 
  }

public:

	// Constructor 
	internals() 
		: m_child_handle(INVALID_HANDLE_VALUE),
			m_child_thread(INVALID_HANDLE_VALUE)
	{ 
		m_pipes[STREAM_INPUT].in = 
			m_pipes[STREAM_INPUT].out = INVALID_HANDLE_VALUE;
		m_pipes[STREAM_OUTPUT].in = 
			m_pipes[STREAM_OUTPUT].out = INVALID_HANDLE_VALUE;
		m_pipes[STREAM_ERROR].in = 
			m_pipes[STREAM_ERROR].out = INVALID_HANDLE_VALUE;
	}

	// Destructor
	~internals()
	{	close(); }

	// Spawns a process 
	bool run(int argc, const char* argv[])
	{
		if (!create_pipe(m_pipes[STREAM_INPUT],false))
			return false; 
		if (!create_pipe(m_pipes[STREAM_OUTPUT],true))
			return false; 
		if (!create_pipe(m_pipes[STREAM_ERROR],true))
			return false; 

		std::stringstream scmd; 
		for (int i=0; i<argc; ++i)
			scmd << argv[i] << ' ';
 		const std::string cmdLine = scmd.str(); 

		PROCESS_INFORMATION procInfo; 
		STARTUPINFO startInfo;
		BOOL bSuccess = FALSE; 

		// Set up members of the PROCESS_INFORMATION structure. 
		ZeroMemory( &procInfo, sizeof(PROCESS_INFORMATION) );
		
		// Set up members of the STARTUPINFO structure. 
		// This structure specifies the STDIN and STDOUT handles for redirection.
		ZeroMemory( &startInfo, sizeof(STARTUPINFO) );
		startInfo.cb = sizeof(STARTUPINFO); 
		startInfo.hStdError = m_pipes[STREAM_ERROR].out;
		startInfo.hStdOutput = m_pipes[STREAM_OUTPUT].out;
		startInfo.hStdInput = m_pipes[STREAM_INPUT].in;
		startInfo.dwFlags |= STARTF_USESTDHANDLES;
		
		// Create the child process. 
		bSuccess = CreateProcess(
			NULL, (LPSTR)cmdLine.c_str(), NULL, NULL, 
			TRUE, 0, NULL, NULL, &startInfo, &procInfo);
		
		// If an error occurs, exit the application. 
		if (!bSuccess) 
		{
			std::cerr << "mtracedb: could not spawn process " <<
				scmd.str() << std::endl; 
			EXIT();
		}

		// Wait for the process to have initializaed itself 
		// might be unessary!
		DWORD waitInput = WaitForInputIdle(procInfo.hProcess, INFINITE);
		if (waitInput == WAIT_FAILED && (get_options().log_level <= LOG_DEBUG))
		{
			std::cerr << "warning: WaitForInputIdle() failed for child process" <<
				std::endl; 
		}

		// Save the handles to the child process 
		m_child_handle = procInfo.hProcess;
		m_child_thread = procInfo.hThread; 

		// The child shouldn't have exited here.
		if (has_exited())
		{
			std::cerr << "error: child has exited!" <<
				std::endl; 
			return false;
		}

		return true;
	}

	// Close 
	bool close()
	{
	  // Close the pipe handles to the child process 
		if (!(closehandle(m_pipes[STREAM_INPUT].in)
        && closehandle(m_pipes[STREAM_INPUT].out)
        && closehandle(m_pipes[STREAM_OUTPUT].in)
        && closehandle(m_pipes[STREAM_OUTPUT].out)
        && closehandle(m_pipes[STREAM_ERROR].in)
        && closehandle(m_pipes[STREAM_ERROR].out)))
		{
			std::cerr << "mtracedb: closing handles to child process failed"
								<< std::endl;
			return false;
		}
		
		// Close the handles to the child process descriptor and the
		// child's main thread 
    if ( !(closehandle(m_child_handle) && closehandle(m_child_thread)))
		{
			std::cerr << "mtracedb: closing handles to the child process"
				" failed" << std::endl; 
			return false; 
		}
				
		return true;
	}

	// Determines if the spawned process  has exited
	bool has_exited() const
	{
		DWORD exitCode = 0; 
		DWORD exitCodeRetrieved = 0;
		exitCodeRetrieved = GetExitCodeProcess(
			m_child_handle, &exitCode); 
		if (exitCodeRetrieved != STILL_ACTIVE)
 			return false;
		return false;
	}

	// Determines if the spawned process has emitted an error
	bool has_error() const
	{
		return false;
	}

	// Write data to a stream 
	size_t write(STREAM_KIND kind, const void* data, size_t len)
	{
		HANDLE stream = m_pipes[kind].out;
		if (stream == NULL) 
		{
			std::cerr << "mtracedb: could not write to invalid stream handle"
								<< std::endl; 
			return 0; 
		}

    // Check if the child process has terminated 
		if (has_exited())
		{
			std::cerr << "mtracedb: child process has exited!" << std::endl;
			return 0; 
		}

		// Perform the actual write
		DWORD written; 
		BOOL bSuccess = WriteFile(
			stream, 
			data, 
			(DWORD)len, 
			&written, 
			NULL);
		if (!bSuccess || written != len) 
		{
			std::cerr << "mtracedb: write to stream handle failed"
								<< std::endl; 
			return 0; 
		}

#if 0
		const std::string& s = std::string((const char*)data, len);
		std::cerr << "mtracedb: write(" << s << ')' << std::endl;
#endif

		return written;
	}
	
	// Read data from a stream 
	size_t read(STREAM_KIND kind, void* data, size_t len)
	{
		HANDLE stream = m_pipes[kind].in;
		if (stream == NULL) 
		{
			std::cerr << "mtracedb: could not from invalid stream handle"
								<< std::endl; 
			return 0; 
		}

    // Check if the child process has terminated 
		if (has_exited())
		{
			std::cerr << "mtracedb: child process has exited!" << std::endl;
			return 0; 
		}

		DWORD read = 0; 
		BOOL bSuccess = ReadFile(
			stream, 
			data, 
			(DWORD)len, 
			&read, 
			NULL);
		if (!bSuccess || read == 0) 
		{
			std::cerr << "mtracedb: read to stream handle failed"
								<< std::endl; 
			return 0; 
		}

#if 0
		const std::string& s = std::string((const char*)data, read);
		std::cerr << "mtracedb: read(" << s << ')' << std::endl;
#endif

		return read; 
	}

	// Read data from a stream 
	size_t peak(STREAM_KIND kind)
	{
		HANDLE stream = m_pipes[kind].in;
		if (stream == NULL) 
		{
			std::cerr << "mtracedb: could not from invalid stream handle"
								<< std::endl; 
			return 0; 
		}

    // Check if the child process has terminated 
		if (has_exited())
		{
			std::cerr << "mtracedb: child process has exited!" << std::endl;
			return 0; 
		}

		DWORD read = 0; 
		BOOL bSuccess = ::PeekNamedPipe(
			stream, 
			NULL, 
			0,
			&read, 
			NULL,
      NULL);
		if (!bSuccess) 
		{
			std::cerr << "mtracedb: read to stream handle failed"
								<< std::endl; 
			return 0; 
		}

		return read; 
	}

};

process_t::process_t()
	: m_process_internals(new internals())
{
}

process_t::~process_t()
{
	delete m_process_internals;
}

bool process_t::run(int argc, const char* argv[])
{
	const options& o = get_options(); 
  if (o.log_level <= LOG_DEBUG) 
  {
    std::cerr << "mtracedb: spawning child process (cmdline: ";
    for (size_t i = 0; i<(size_t)argc; ++i)
      std::cerr << argv[i] << ' ';
    std::cerr << ')' << std::endl;
  }
	return m_process_internals->run(argc,argv);
}

bool process_t::close()
{
	return m_process_internals->close();
}

bool process_t::has_exited() const
{
	return m_process_internals->has_exited();
}

bool process_t::has_error() const
{
	return m_process_internals->has_error();
}

std::size_t process_t::write(
	STREAM_KIND kind, 
	const void* data, 
	std::size_t len)
{
	if (m_process_internals->write(kind, data, len) != len)
	{
		std::cerr << "write to child process failed!" << std::endl;
		EXIT();
	}
	return len; 
}

std::size_t process_t::read(
  STREAM_KIND kind, 
  void* data, 
  std::size_t len)
{
	if (m_process_internals->read(kind, data, len) == 0)
	{
		std::cerr << "read from child process failed!" << std::endl;
		EXIT();
	}
	return len; 
}

std::size_t process_t::peak(
  STREAM_KIND kind)
{
	return m_process_internals->peak(kind); 
}

