#include "stdafx.h"
#include "WebKitHud.h"

#ifdef BROWSER_GUI_ENABLED

#include "WebCoreProxy.h"

#include "ISystem.h"
#include "IRenderer.h"
#include "IGame.h"
#include "IGameFramework.h"
#include "IActionMapManager.h"

#define JAVASCRIPT_LOADING_FUNCTION_NAME "loading"

/************************************************************************/
CWebKitHud* CWebKitHud::ms_instance( NULL );

/************************************************************************/
void CWebKitHud::Init()
{
	assert( ms_instance == NULL );
	if ( ms_instance != NULL )
	{
		ShutDown();
	}

	ms_instance = new CWebKitHud();
}

/************************************************************************/
void CWebKitHud::ShutDown()
{
	assert( ms_instance != NULL );
	if ( ms_instance == NULL )
	{
		return;
	}

	delete ms_instance;
	ms_instance = NULL;
}

/************************************************************************/
CWebKitHud* CWebKitHud::GetInstance()
{
	assert( ms_instance != NULL );
	if ( ms_instance == NULL )
	{
		Init();
	}

	return ms_instance;
}

/************************************************************************/
CWebKitHud::CWebKitHud()
: m_pRenderer( NULL )
, m_pWebCore( NULL )
, m_pWebView( NULL )
, m_visible( false )
, m_transparent( false )
, m_hudInputEnabled( false )
, m_cursorVisible( false )
, m_loadingMap( false )
{
	assert( gEnv != NULL );

	assert( gEnv->pRenderer != NULL );
	m_pRenderer = gEnv->pRenderer;

	m_pWebCore = new CWebCoreProxy();
	
	int width = m_webKitTexture.GetWidth();
	int height = m_webKitTexture.GetHeight();
	m_pWebView = m_pWebCore->CreateWebView( width, height );

	assert( gEnv->pHardwareMouse != NULL );
	gEnv->pHardwareMouse->AddListener( this );

	assert( gEnv->pGame );
	assert( gEnv->pGame->GetIGameFramework() );
	gEnv->pGame->GetIGameFramework()->RegisterListener( this, "WebKitHud", FRAMEWORKLISTENERPRIORITY_HUD );

	assert( gEnv->pGame->GetIGameFramework()->GetILevelSystem() );
	gEnv->pGame->GetIGameFramework()->GetILevelSystem()->AddListener( this );

	m_webKitCallbacks.Init( GetWebView() );

	SetStartStateProperties();
}

/************************************************************************/
CWebKitHud::~CWebKitHud()
{
	gEnv->pGame->GetIGameFramework()->GetILevelSystem()->RemoveListener( this );

	gEnv->pGame->GetIGameFramework()->UnregisterListener( this );
	
	gEnv->pHardwareMouse->RemoveListener( this );

	delete m_pWebCore;
}

/************************************************************************/
bool CWebKitHud::IsVisible() const
{
	return m_visible;
}

/************************************************************************/
void CWebKitHud::SetVisible( bool visible )
{
	m_visible = visible;

	if ( ! IsVisible() )
	{
		LoadBlankPage();
	}
}

/************************************************************************/
void CWebKitHud::ToggleVisible()
{
	SetVisible( ! IsVisible() );
}

/************************************************************************/
void CWebKitHud::Show()
{
	SetVisible( true );
}

/************************************************************************/
void CWebKitHud::Hide()
{
	SetVisible( false );
}

/************************************************************************/
void CWebKitHud::SetTransparent( bool transparent )
{
	m_transparent = transparent;
	GetWebView()->SetTransparentBackground( transparent );
}

/************************************************************************/
bool CWebKitHud::IsTransparent() const
{
	return m_transparent;
}

/************************************************************************/
void CWebKitHud::SetHudInputEnabled( bool inputEnabled )
{
	m_hudInputEnabled = inputEnabled;
}

/************************************************************************/
bool CWebKitHud::IsHudInputEnabled() const
{
	return m_hudInputEnabled;
}

/************************************************************************/
void CWebKitHud::SetCursorVisible( bool cursorVisible )
{
	if ( cursorVisible == IsCursorVisible() )
	{
		return;
	}

	if ( cursorVisible )
	{
		gEnv->pHardwareMouse->IncrementCounter();
		gEnv->pInput->ClearKeyState();
	}
	else
	{
		gEnv->pHardwareMouse->DecrementCounter();
	}

	m_cursorVisible = cursorVisible;
}

/************************************************************************/
bool CWebKitHud::IsCursorVisible() const
{
	return m_cursorVisible;
}

/************************************************************************/
void CWebKitHud::SetPlayerInputEnabled( bool playerInputEnabled )
{
	IActionMapManager* pActionMapManager = gEnv->pGame->GetIGameFramework()->GetIActionMapManager();
	if( ! pActionMapManager )
	{
		return;
	}

	pActionMapManager->Enable( playerInputEnabled );
}

/************************************************************************/
bool CWebKitHud::CanInjectInput() const
{
	return ( IsVisible() && IsHudInputEnabled() );
}

/************************************************************************/
void CWebKitHud::OnHardwareMouseEvent( int mouseX, int mouseY, EHARDWAREMOUSEEVENT hardwareMouseEvent, int wheelDelta )
{
	if ( ! CanInjectInput() )
	{
		return;
	}

	switch ( hardwareMouseEvent )
	{
	case HARDWAREMOUSEEVENT_MOVE:
		GetWebView()->InjectMouseMove( mouseX, mouseY );
		break;

	case HARDWAREMOUSEEVENT_LBUTTONDOWN:
		GetWebView()->InjectLeftMouseDown();
		break;

	case HARDWAREMOUSEEVENT_LBUTTONUP:
		GetWebView()->InjectLeftMouseUp();
		break;

	case HARDWAREMOUSEEVENT_WHEEL:
		GetWebView()->InjectMouseWheel( wheelDelta );
		break;

	case HARDWAREMOUSEEVENT_RBUTTONDOWN:
		GetWebView()->InjectRightMouseDown();
		break;

	case HARDWAREMOUSEEVENT_RBUTTONUP:
		GetWebView()->InjectRightMouseUp();
		break;

	case HARDWAREMOUSEEVENT_MBUTTONDOWN:
		GetWebView()->InjectMiddleMouseDown();
		break;

	case HARDWAREMOUSEEVENT_MBUTTONUP:
		GetWebView()->InjectMiddleMouseUp();
		break;
	}
}

/************************************************************************/
void CWebKitHud::InjectKeyboardKeyUp( UINT_PTR wParam, LONG_PTR lParam )
{
	if ( CanInjectInput() )
	{
		HWND hWnd = ( HWND ) m_pRenderer->GetHWND();
		GetWebView()->InjectKeyboardKeyUp( hWnd, wParam, lParam );
	}
}

/************************************************************************/
void CWebKitHud::InjectKeyboardKeyDown( UINT_PTR wParam, LONG_PTR lParam )
{
	if ( CanInjectInput() )
	{
		HWND hWnd = ( HWND ) m_pRenderer->GetHWND();
		GetWebView()->InjectKeyboardKeyUp( hWnd, wParam, lParam );
	}
}

/************************************************************************/
void CWebKitHud::InjectKeyboardChar( UINT_PTR wParam, LONG_PTR lParam )
{
	if ( CanInjectInput() )
	{
		HWND hWnd = ( HWND ) m_pRenderer->GetHWND();
		GetWebView()->InjectKeyboardChar( hWnd, wParam, lParam );
	}
}

/************************************************************************/
void CWebKitHud::LoadURL( const string url )
{
	GetWebView()->LoadURL( url.c_str() );
}

/************************************************************************/
void CWebKitHud::LoadBlankPage()
{
	GetWebView()->LoadHTML( "" );
}

/************************************************************************/
void CWebKitHud::Refresh()
{
	GetWebView()->Refresh();
}

/************************************************************************/
void CWebKitHud::SetProperty( const string& name, const string& value )
{
	GetWebView()->SetProperty( name.c_str(), value.c_str() );
}

/************************************************************************/
void CWebKitHud::SetProperty( const string& name, float value )
{
	GetWebView()->SetProperty( name.c_str(), value );
}

/************************************************************************/
void CWebKitHud::SetProperty( const string& name, int value )
{
	GetWebView()->SetProperty( name.c_str(), value );
}

/************************************************************************/
void CWebKitHud::SetProperty( const string& name, bool value )
{
	GetWebView()->SetProperty( name.c_str(), value );
}

/************************************************************************/
void CWebKitHud::RegisterCallbackName( const string& callbackName )
{
	GetWebView()->RegisterCallback( callbackName.c_str() );
}

/************************************************************************/
void CWebKitHud::AddJavascriptListener( IJavascriptCallbackListener* listener )
{
	GetWebView()->AddJavascriptListener( listener );
}

/************************************************************************/
void CWebKitHud::RemoveJavascriptListener( IJavascriptCallbackListener* listener )
{
	GetWebView()->RemoveJavascriptListener( listener );
}

/************************************************************************/
void CWebKitHud::OnPostUpdate( float deltaTime )
{
	GetWebCore()->Update();
	Render();
}

/************************************************************************/
void CWebKitHud::Render()
{
	if ( ! IsVisible() )
	{
		return;
	}

	m_webKitTexture.RenderToFillViewport( GetWebView(), IsTransparent() );
}

/************************************************************************/
void CWebKitHud::RenderLoading()
{
	int width = m_pRenderer->GetWidth();
	int height = m_pRenderer->GetHeight();

	gEnv->pSystem->RenderBegin();
	m_pRenderer->Set2DMode( true, width, height );

	ColorF clearColor( Col_Black );
	m_pRenderer->ClearBuffer( FRT_CLEAR | FRT_CLEAR_IMMEDIATE, &clearColor );

	m_pRenderer->SetWireframeMode( R_SOLID_MODE );

	Render();	

	m_pRenderer->Set2DMode( false, 0, 0 );
	gEnv->pSystem->RenderEnd();
}

/************************************************************************/
CWebCoreProxy* CWebKitHud::GetWebCore()
{
	return m_pWebCore;
}

/************************************************************************/
const CWebCoreProxy* CWebKitHud::GetWebCore() const
{
	return m_pWebCore;
}

/************************************************************************/
CWebViewProxy* CWebKitHud::GetWebView()
{
	return m_pWebView;
}

/************************************************************************/
const CWebViewProxy* CWebKitHud::GetWebView() const
{
	return m_pWebView;
}

/************************************************************************/
void CWebKitHud::OnLevelNotFound( const char* levelName )
{
	m_loadingMap = false;
}

/************************************************************************/
void CWebKitHud::OnLoadingStart( ILevelInfo* pLevel )
{
	if ( m_loadingMap )
	{
		return;
	}

	m_loadingMap = true;

	
	UpdateLoadingProgress( pLevel, 0 );

	SetLoadingStateProperties();
}

/************************************************************************/
void CWebKitHud::OnLoadingComplete( ILevel* pLevel )
{
	m_loadingMap = false;

	SetGameStateProperties();
}

/************************************************************************/
void CWebKitHud::OnLoadingError( ILevelInfo* pLevel, const char* error )
{
	m_loadingMap = false;

	bool someLevelLoaded = gEnv->pGame->GetIGameFramework()->GetILevelSystem()->IsLevelLoaded();
	if ( someLevelLoaded )
	{
		SetGameStateProperties();
	}
	else
	{
		SetStartStateProperties();
	}
}

/************************************************************************/
void CWebKitHud::OnLoadingProgress( ILevelInfo* pLevel, int progressAmount )
{
	if ( ! m_loadingMap )
	{
		return;
	}

	UpdateLoadingProgress( pLevel, progressAmount );

	RenderLoading();
}

/************************************************************************/
void CWebKitHud::UpdateLoadingProgress( ILevelInfo* pLevel, int progressAmount )
{
	if ( pLevel == NULL )
	{
		return;
	}

	int maxLoadingProgress = ( ( pLevel->GetDefaultGameType() != NULL ) ? pLevel->GetDefaultGameType()->cgfCount : 100 );
	if ( maxLoadingProgress == 0 )
	{
		return;
	}

	float progress = CLAMP( ( progressAmount * 100.f ) / maxLoadingProgress, 0, 100 );
	
	string progressFunction;
	progressFunction.Format( JAVASCRIPT_LOADING_FUNCTION_NAME "('%f');", progress );
	
	GetWebView()->ExecuteJavascript( progressFunction.c_str() );
}

/************************************************************************/
void CWebKitHud::SetStartStateProperties()
{
	const char* startUrl = m_webKitConfig.GetStartUrl().c_str();
	LoadURL( startUrl );
	Show();
	SetTransparent( false );
	SetHudInputEnabled( true );
	SetPlayerInputEnabled( false );
	SetCursorVisible( true );
}

/************************************************************************/
void CWebKitHud::SetLoadingStateProperties()
{
	const char* loadingUrl = m_webKitConfig.GetLoadingUrl().c_str();
	LoadURL( loadingUrl );
	Show();
	SetTransparent( false );
	SetHudInputEnabled( false );
	SetPlayerInputEnabled( false );
	SetCursorVisible( true );
}

/************************************************************************/
void CWebKitHud::SetGameStateProperties()
{
	LoadBlankPage();
	Hide();
	SetTransparent( true );
	SetHudInputEnabled( false );
	SetPlayerInputEnabled( true );
	SetCursorVisible( false );
}

#endif // BROWSER_GUI_ENABLED