#include "StdAfx.h"
#include "WebKitTexture.h"
#include "WebViewProxy.h"

/************************************************************************/
CWebKitTexture::CWebKitTexture()
: m_pRenderer( NULL )
, m_pTexture( NULL )
, m_viewportWidth( 0 )
, m_viewportHeight( 0 )
{
	assert( gEnv != NULL );

	assert( gEnv->pRenderer != NULL );
	m_pRenderer = gEnv->pRenderer;

	UpdateStoredViewportSize();
	CreateTexture( GetViewportWidth(), GetViewportHeight() );
}

/************************************************************************/
CWebKitTexture::~CWebKitTexture()
{
	if ( m_pTexture != NULL )
	{
		m_pTexture->Release();
	}
}

/************************************************************************/
void CWebKitTexture::RenderToFillViewport( CWebViewProxy* pWebView, bool transparent )
{
	int oldViewportWidth = m_viewportWidth;
	int oldViewportHeight = m_viewportHeight;
	UpdateStoredViewportSize();

	bool viewportHasChanged = ( oldViewportWidth != m_viewportWidth || oldViewportHeight != m_viewportHeight );
	if ( viewportHasChanged )
	{
		pWebView->Resize( GetViewportWidth(), GetViewportHeight() );
	}

	bool viewportBiggerThanTexture = ( GetWidth() < GetViewportWidth() || GetHeight() < GetViewportHeight() );
	if ( viewportBiggerThanTexture )
	{
		CreateTexture( GetViewportWidth(), GetViewportHeight() );
	}

	UpdateTextureContents( pWebView );

	float x = 0;
	float y = 0;
	float width = float( GetViewportWidth() );
	float height = float( GetViewportHeight() );

	int textureId = GetTextureId();

	float relativeWidth = width / GetWidth();
	float relativeHeight = height / GetHeight();

	// For more information on the offset for u and v: http://msdn.microsoft.com/en-us/library/bb219690(VS.85).aspx
	float offsetU = 0.5f / GetWidth();
	float offsetV = 0.5f / GetHeight();

	float u1 = 0.f + offsetU;
	float v1 = 1.f + offsetV;
	float u2 = relativeWidth + offsetU;
	float v2 = 1.f - relativeHeight + offsetV;

	m_pRenderer->Set2DMode( true, GetViewportWidth(), GetViewportHeight() );

	const int TRANSPARENT_STATE = ( GS_NODEPTHTEST | GS_BLSRC_SRCALPHA | GS_BLDST_ONEMINUSSRCALPHA );
	const int OPAQUE_STATE = GS_NODEPTHTEST;
	int state = ( transparent ? TRANSPARENT_STATE : OPAQUE_STATE );
	m_pRenderer->SetState( state );

	m_pRenderer->DrawImage( x, y, width, height, textureId, u1, v1, u2, v2, 1, 1, 1, 1 );

	m_pRenderer->Set2DMode( false, 0, 0 );
}

/************************************************************************/
int CWebKitTexture::GetTextureId() const
{
	return m_pTexture->GetTextureID();
}

/************************************************************************/
void CWebKitTexture::CreateTexture( int textureWidth, int textureHeight )
{
	if ( m_pTexture != NULL )
	{
		m_pTexture->Release();
		m_pTexture = NULL;
	}

	int textureId = m_pRenderer->SF_CreateTexture( textureWidth, textureHeight, 0, NULL, eTF_A8R8G8B8 );
	m_pTexture = m_pRenderer->EF_GetTextureByID( textureId );
}

/************************************************************************/
void CWebKitTexture::UpdateTextureContents( CWebViewProxy* pWebView )
{
	bool updateNeeded = pWebView->IsDirty();
	if ( ! updateNeeded )
	{
		return;
	}

	const int TEXTURE_DEPTH = 4;
	int pitch = 0;
	byte* textureData = m_pTexture->LockData( pitch );
	pWebView->Render( textureData, pitch, TEXTURE_DEPTH );
	m_pTexture->UnlockData();
}

/************************************************************************/
int CWebKitTexture::GetWidth() const
{
	return m_pTexture->GetWidth();
}

/************************************************************************/
int CWebKitTexture::GetHeight() const
{
	return m_pTexture->GetHeight();
}

/************************************************************************/
void CWebKitTexture::UpdateStoredViewportSize()
{
	int x = 0;
	int y = 0;

	m_pRenderer->GetViewport( &x, &y, &m_viewportWidth, &m_viewportHeight );
}

/************************************************************************/
int CWebKitTexture::GetViewportWidth() const
{
	return m_viewportWidth;
}

/************************************************************************/
int CWebKitTexture::GetViewportHeight() const
{
	return m_viewportHeight;
}