////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   GameStateRecorder.cpp
//  Version:     v1.00
//  Created:     3/2008 by Luciano Morpurgo.
//  Compilers:   Visual Studio.NET
//  Description: Checks the player and other game specific objects' states and communicate them to the TestManager
//							 Implements IGameStateRecorder
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "IGame.h"
#include "IGameFramework.h"
#include "IActorSystem.h"
#include "GameStateRecorder.h"
#include "CryAction.h"
#include "NanoSuit.h"
#include "Player.h"
#include "Item.h"
#include "Accessory.h"
#include "Inventory.h"

// avoid fake mistakes when health is raising progressively 
// there can be small mismatches due to time-dependent health regeneration
#define CHECK_MISMATCH(cur,rec,thr) (cur < rec-thr || cur > rec+thr)


///////////////////////////////////////////////////////////////////////
const char* CGameStateRecorder::m_SuitModeName[NANOMODE_LAST] = 
{
	"Speed",
	"Strength",
	"Cloak",
	"Armor",
	"Invulnerability",
	"Armor hit reaction"
};

///////////////////////////////////////////////////////////////////////
bool IsAmmoPickup(CItem* pItem) 
{
	return strcmp(pItem->GetEntity()->GetClass()->GetName(), "CustomAmmoPickup")==0;
}

///////////////////////////////////////////////////////////////////////
bool IsAmmoPickup(const char* className) 
{
	return strcmp(className, "CustomAmmoPickup")==0;
}

///////////////////////////////////////////////////////////////////////
bool IsGrenade(const char* className)
{
	return className && (!strcmp(className,"OffhandGrenade") ||
				!strcmp(className,"OffhandSmoke") ||
				!strcmp(className,"OffhandFlashbang") ||
				!strcmp(className,"OffhandNanoDistuptor"));
}

///////////////////////////////////////////////////////////////////////
bool IsAmmoGrenade(const char* className)
{
	return className && (!strcmp(className,"explosivegrenade") ||
			!strcmp(className,"flashbang") ||
			!strcmp(className,"smokegrenade") ||
			!strcmp(className,"empgrenade"));
}

///////////////////////////////////////////////////////////////////////
CGameStateRecorder::CGameStateRecorder() 
{
	m_mode = GPM_Disabled;
	m_bRecording = false;
	m_bEnable = false;
	m_bLogWarning = true;
	m_currentFrame = 0;
	m_pSingleActor = NULL;
	//m_pRecordGameEventFtor = new RecordGameEventFtor(this);
	m_demo_actorInfo = REGISTER_STRING( "demo_actor_info","player",0,"name of actor which game state info is displayed" );
	m_demo_actorFilter = REGISTER_STRING( "demo_actor_filter","player",0,"name of actor which game state is recorded ('player','all',<entity name>" );
	REGISTER_CVAR2( "demo_force_game_state",&m_demo_forceGameState,2,0,"Forces game state values into game while playing timedemo: only health and suit energy (1) or all (2)" );

}

////////////////////////////////////////////////////////////////////////////
CGameStateRecorder::~CGameStateRecorder()
{
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::Release()
{
//	if(m_pRecordGameEventFtor)
		//delete m_pRecordGameEventFtor;
	IConsole* pConsole = gEnv->pConsole;
	if(pConsole)
	{
		pConsole->UnregisterVariable( "demo_force_game_state", true );
		pConsole->UnregisterVariable( "demo_actor_info", true );
		pConsole->UnregisterVariable( "demo_actor_filter", true );
	}
	delete this;
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::Enable(bool bEnable, bool bRecording)
{
	m_bRecording = bRecording;
	m_bEnable = bEnable;
	//m_mode = mode;
	if(!bEnable)
		m_mode = GPM_Disabled;

	if(m_bEnable && bRecording)
	{
		gEnv->pGame->GetIGameFramework()->GetIGameplayRecorder()->RegisterListener(this);

		//CActor *pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
		//if(pActor && !pActor->GetSpectatorMode() && pActor->IsPlayer() && ((CPlayer*)pActor)->GetNanoSuit())
			//((CPlayer*)pActor)->GetNanoSuit()->AddListener(this);

	}
	else
	{
		gEnv->pGame->GetIGameFramework()->GetIGameplayRecorder()->UnregisterListener(this);

		//CActor *pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
		//if(pActor && !pActor->GetSpectatorMode() && pActor->IsPlayer() && ((CPlayer*)pActor)->GetNanoSuit())
			//((CPlayer*)pActor)->GetNanoSuit()->RemoveListener(this);
	}
	
	if(m_bEnable)
		StartSession();
}


////////////////////////////////////////////////////////////////////////////
float CGameStateRecorder::GetSuitEnergy(CActor* pActor)
{
	CNanoSuit *pSuit = ((CPlayer*)pActor)->GetNanoSuit();
	if(pSuit)
		return pSuit->GetSuitEnergy();
	return -1;
}

////////////////////////////////////////////////////////////////////////////
int CGameStateRecorder::GetSuitMode(CActor* pActor)
{
	CNanoSuit *pSuit = ((CPlayer*)pActor)->GetNanoSuit();
	if(pSuit)
		return pSuit->GetMode();
	return -1;
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::SetSuitMode(CActor* pActor, int mode)
{
	CNanoSuit *pSuit = ((CPlayer*)pActor)->GetNanoSuit();
	if(pSuit)
		pSuit->SetMode(ENanoMode(mode));
}

/////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::AddActorToStats(const CActor* pActor)
{
	if(m_bRecording)
		DumpWholeGameState(pActor);
	else
	{
		IEntity* pEntity = pActor->GetEntity();
		EntityId id;
		if(pEntity && (id=pEntity->GetId()))
		{
			m_GameStates.insert(std::make_pair(id,SActorGameState()));
		}
	}
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::StartSession()
{
	m_GameStates.clear();
	m_itSingleActorGameState = m_GameStates.end();
	m_IgnoredEvents.clear();

	const char* filterName = m_demo_actorFilter->GetString();
	// send game events to record the initial game state
/*	if(m_mode)
	{
		CActor *pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
*/
		
	m_pSingleActor = GetActorOfName(filterName);
	if(m_pSingleActor)// && !pActor->GetSpectatorMode() && pActor->IsPlayer())
	{
		m_mode = GPM_SingleActor;
		AddActorToStats(m_pSingleActor);
		m_itSingleActorGameState = m_GameStates.begin();// position of requested actor's id (player by default)
	}
//	}
	else if (!strcmpi(filterName,"all"))
	{
		m_mode = GPM_AllActors;
		AutoAIObjectIter it(gEnv->pAISystem->GetFirstAIObject(IAISystem::OBJFILTER_TYPE, AIOBJECT_PUPPET));
		for(; it->GetObject(); it->Next())
		{
			IAIObject* pObject = it->GetObject();
			if(pObject)
			{
				CActor* pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(pObject->GetEntityID()));
				if(pActor)
					AddActorToStats(pActor);
			}
		}
		
		it = gEnv->pAISystem->GetFirstAIObject(IAISystem::OBJFILTER_TYPE, AIOBJECT_VEHICLE);
		for(; it->GetObject(); it->Next())
		{
			IAIObject* pObject = it->GetObject();
			if(pObject)
			{
				CActor* pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(pObject->GetEntityID()));
				if(pActor)
					AddActorToStats(pActor);
			}
		}

		it->Release();
	}
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::DumpWholeGameState(const CActor* pActor)
{
	GameplayEvent event;
	IEntity* pEntity = pActor->GetEntity();
	
	// health
	float health = (float)pActor->GetHealth();
	event.event = eGE_HealthChanged;
	event.value = health;
	SendGamePlayEvent(pEntity,event);
	
	//nanosuit
	if(pActor->IsPlayer())
	{
		const CPlayer* pPlayer = static_cast<const CPlayer *>(pActor);
		if(CNanoSuit *pSuit=pPlayer->GetNanoSuit())
		{
			event.event = eGE_SuitEnergyChanged;
			event.value = pSuit->GetSuitEnergy();
			SendGamePlayEvent(pEntity,event);

			event.event = eGE_SuitModeChanged;
			event.value = (float)pSuit->GetMode();
			SendGamePlayEvent(pEntity,event);
		}
	}

	// Inventory
	CInventory *pInventory = (CInventory*)(pActor->GetInventory());
	if(!pInventory)
		return;

	int count = pInventory->GetCount();
	for(int i=0; i<count; ++i)
	{
		EntityId itemId = pInventory->GetItem(i);
		
		CItem* pItem=NULL;
		TItemName itemName = GetItemName(itemId,&pItem);
		if(pItem && itemName)	
		{
			event.event = eGE_ItemPickedUp;
			event.description = itemName;
			SendGamePlayEvent(pEntity,event);

			if(pActor->GetCurrentItem() == pItem)
			{
				event.event = eGE_ItemSelected;
				event.description = itemName;
				event.value = 1; // for initialization
				SendGamePlayEvent(pEntity,event);
			}
		}
	}

	count = pInventory->GetAccessoryCount();

	for(int i=0; i< count; i++)
	{
		const char* accessory = pInventory->GetAccessory(i);
		if(accessory && strlen(accessory))	
		{
			IEntityClass* pClass = gEnv->pEntitySystem->GetClassRegistry()->FindClass(accessory);
			if(pClass)
			{
				TItemName classItem = pClass->GetName();
				event.event = eGE_AccessoryPickedUp;
				//event.value = classIdx;
				event.description = classItem;
				SendGamePlayEvent(pEntity,event);
			}
		}
	}

	int nInvAmmo = pInventory->GetAmmoPackCount();
	pInventory->AmmoIteratorFirst();
	for(int j=0 ; !pInventory->AmmoIteratorEnd(); j++, pInventory->AmmoIteratorNext())
	{
		const IEntityClass* pAmmoClass = pInventory->AmmoIteratorGetClass();
		if(pAmmoClass)
		{
			const char* ammoClassName = pAmmoClass->GetName();
			int ammoCount = pInventory->AmmoIteratorGetCount();
			GameplayEvent event;
			event.event = eGE_AmmoPickedUp;
			event.description = ammoClassName;
			event.value = (float)ammoCount;
			SendGamePlayEvent(pEntity,event);
		}
	}

}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::Update()
{
	if(m_bRecording)
	{
		switch(m_mode)
		{
			case GPM_SingleActor:
				{
					bool bEvent = false;
					//CActor *pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
					//if(pActor && !pActor->GetSpectatorMode() && pActor->IsPlayer())
					if(m_pSingleActor)
					{
						float health = (float)m_pSingleActor->GetHealth();
						if(m_itSingleActorGameState != m_GameStates.end())
						{
							SActorGameState& playerState = m_itSingleActorGameState->second;
							//if( health!= playerState.health)
							{
//								playerState.health = health;
								GameplayEvent event;
								event.event = eGE_HealthChanged;
								event.value = health;
								OnGameplayEvent(m_pSingleActor->GetEntity(),event);
							}

							float energy = GetSuitEnergy(m_pSingleActor);
							//if(energy!= playerState.suitEnergy)
							{
							//	playerState.suitEnergy = energy;
								GameplayEvent event;
								event.event = eGE_SuitEnergyChanged;
								event.value = energy;
								OnGameplayEvent(m_pSingleActor->GetEntity(),event);
							}
							
						}
					}
				}
				break;

			case GPM_AllActors:
			default:
				break;
		}
	}
}


////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::OnRecordedGameplayEvent(IEntity *pEntity, const GameplayEvent &event, int currentFrame, bool bRecording)
{
	EntityId id;
	m_currentFrame = currentFrame;

	int demo_forceGameState = 0;
	if(!bRecording)
	{
		ICVar *pVar = gEnv->pConsole->GetCVar( "demo_force_game_state" );
		if(pVar)
			demo_forceGameState = pVar->GetIVal();
	}
	
	if(!pEntity || !(id = pEntity->GetId()))
		return;

	if(m_IgnoredEvents.size())
		if(event.event == m_IgnoredEvents[0])
		{
			m_IgnoredEvents.erase(m_IgnoredEvents.begin());
			return;
		}

		TGameStates::iterator itActor = m_GameStates.find(id);
	if(itActor == m_GameStates.end())
	{
		m_GameStates.insert(std::make_pair(id,SActorGameState()));
		itActor = m_GameStates.find(id);
	}
	if(itActor == m_GameStates.end())
	{
		GameWarning("TimeDemo:GameState: actor %s not found in records",pEntity->GetName());
		return;
	}

	SActorGameState& gstate = itActor->second;

	switch(event.event)
	{
		case eGE_HealthChanged: 
			{
				gstate.health = (int)event.value;

				if(!m_bRecording)
				{
					CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
					if(pActor)
					{
						if(m_bLogWarning)
						{
							if(CHECK_MISMATCH(pActor->GetHealth(), gstate.health,10))
							{
								if(!gstate.bHealthDifferent)
								{
									GameWarning("TimeDemo:GameState: Frame %d - Actor %s - HEALTH mismatch (%d, %d)",m_currentFrame,pEntity->GetName(), pActor->GetHealth(),gstate.health);
									gstate.bHealthDifferent = true;
								}
							}
							else
								gstate.bHealthDifferent = false;
						}

						if( demo_forceGameState)
							pActor->SetHealth(gstate.health);
					}
				}
			}
			break;

		case eGE_SuitEnergyChanged:
			{
				gstate.suitEnergy = event.value;
				
				if(!m_bRecording)
				{
					CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));

					if(pActor)
					{
						if(m_bLogWarning)
						{
							float curEnergy =GetSuitEnergy(pActor);
							if(CHECK_MISMATCH(curEnergy, gstate.suitEnergy,1))
							{
								if(!gstate.bSuitEnergyDifferent)
								{
									GameWarning("TimeDemo:GameState: Frame %d - Actor %s - NANOSUIT ENERGY mismatch (%4.1f, %4.1f)",m_currentFrame,pEntity->GetName(), curEnergy,gstate.suitEnergy);
									gstate.bSuitEnergyDifferent = true;
								}
							}
							else
								gstate.bSuitEnergyDifferent = false;
						}

						if(demo_forceGameState)
						{
							CNanoSuit *pSuit = ((CPlayer*)pActor)->GetNanoSuit();
							if(pSuit)
								pSuit->SetSuitEnergy(gstate.suitEnergy);
						}
					}
				}
			}
			break;

		case eGE_SuitModeChanged:
			{
				gstate.suitMode = uint8(event.value);

				CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
				if(pActor)
				{
					if(m_bLogWarning)
						CheckDifference(gstate.suitMode,GetSuitMode(pActor),"TimeDemo:GameState: Frame %d - Actor %s - NANOSUIT MODE mismatch (%d, %d)",pEntity);
					if(demo_forceGameState==2)
						SetSuitMode(pActor,gstate.suitMode);
				}
			}
			break;

		case eGE_WeaponFireModeChanged:
			break;

		case eGE_WeaponReload:
			break;

		case eGE_ItemSelected:
			{
				TItemContainer& Items = gstate.Items;
				TItemName itemName = event.description;
				gstate.itemSelected = itemName;
				if(itemName)
				{
					if( !bRecording && (event.value > 0 || demo_forceGameState==2)) // EVENT.VALUE > 0 means initialization
					{
						CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
						if(pActor && pActor->GetInventory())
						{
							IItem* pItem = pActor->GetInventory()->GetItemByName(itemName);
							if(pItem)
								pActor->SelectItem(pItem->GetEntityId(),false);
						}
					}
				}

				if(m_bLogWarning)
				{	
					CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
					if(pActor)
					{
						IItem* pItem = pActor->GetCurrentItem();
						const char* curItemName= pItem && pItem->GetEntity() ? pItem->GetEntity()->GetName(): NULL;
						CheckDifferenceString(itemName,curItemName,"TimeDemo:GameState: Frame %d - Actor %s - SELECTED ITEM mismatch (rec:%s, cur:%s)",pEntity);
					}
				}
				break;
			}
			break;

		case eGE_ItemExchanged:
			{
				// prevent unwanted record/play mismatch error with current item during item exchanging
				// two unneeded game events are sent (selecting fists)
				m_IgnoredEvents.push_back(eGE_ItemSelected);
				m_IgnoredEvents.push_back(eGE_ItemSelected);
			}
			break;

		case eGE_ItemPickedUp:
			{
				TItemName sel = (TItemName)event.description;
//				gstate.itemSelected = sel;
				TItemContainer& Items = gstate.Items;
				TItemContainer::iterator it = Items.find(sel);
				if(it == Items.end())
				{
					Items.insert(std::make_pair(sel,SItemProperties()));
					it = Items.find(sel);
				}

				if(it != Items.end())
				{
					it->second.count++;

					CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
					if(pActor && !m_bRecording)
					{
						CInventory* pInventory = (CInventory*)(pActor->GetInventory());
						if(pInventory)
						{
							// just check if the item is the inventory
							if(m_bLogWarning && !pInventory->GetItemByName(sel))
								GameWarning("TimeDemo:GameState: Frame %d - Actor %s - Recorded PICKED UP ITEM class '%s' not found in current inventory",m_currentFrame,pEntity->GetName(),sel);

							if(demo_forceGameState == 2)
							{
								IEntity* pItemEntity = gEnv->pEntitySystem->FindEntityByName(sel);
								if(pItemEntity)
									pInventory->AddItem(pItemEntity->GetId());
								else
									GameWarning("TimeDemo:GameState: Frame %d - Actor %s - PICKED UP ITEM entity %s not found in level",m_currentFrame,pEntity->GetName(),sel);
							}
						}
					}
				}
				else if(m_bLogWarning)
				{
					if(!sel)
						sel = "(null)";
					GameWarning("TimeDemo:GameState: Frame %d - Actor %s - PICKED UP ITEM %s not found in recorded inventory",m_currentFrame,pEntity->GetName(),sel);
				}

			}
			break;

		case eGE_AmmoPickedUp:
			break;

		case eGE_AccessoryPickedUp:
			{
				TItemName sel = (TItemName)event.description;
				//				gstate.itemSelected = sel;
				TAccessoryContainer& Accessories = gstate.Accessories;
				TAccessoryContainer::iterator it = Accessories.find(sel);
				if(it == Accessories.end())
				{
					Accessories.insert(std::make_pair(sel,1));
				}
				else
					it->second++;

				CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
				if(pActor)
				{
					CInventory* pInventory = (CInventory*)(pActor->GetInventory());
					if(pInventory)
					{
						IEntityClass* pClass = gEnv->pEntitySystem->GetClassRegistry()->FindClass(sel);
						
						if(m_bLogWarning && !pInventory->HasAccessory(pClass))
							GameWarning("TimeDemo:GameState: Frame %d - Actor %s - ACCESSORY PICKEDUP %s not found in current inventory", m_currentFrame, pEntity->GetName(),sel ? sel:"(null)");

						if(demo_forceGameState == 2 && pClass)					
							pInventory->AddItemByClass(pClass);// doesn't actually add it if it's there already

					}
				}
			}
			break;

		case eGE_ItemDropped:
			{
				TItemName sel = (TItemName)event.description;
				//gstate.itemSelected = sel;
				TItemContainer& Items = gstate.Items;
				TItemContainer::iterator it = Items.find(sel);
				if(it != Items.end())
				{
					it->second.count--;
					if(it->second.count<=0)
						Items.erase(it);

					if(demo_forceGameState == 2)
					{
						CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
						if(pActor)
						{
							CInventory* pInventory = (CInventory*)(pActor->GetInventory());
							if(pInventory)
							{
								IEntityClass* pClass = gEnv->pEntitySystem->GetClassRegistry()->FindClass(sel);
								if(pClass)
								{
									EntityId itemId = pInventory->GetItemByClass(pClass);
									if(itemId)
										pInventory->RemoveItem(itemId);
								}
							}
						}
					}
				}
				else
					GameWarning("TimeDemo:GameState: Frame %d - Actor %s - ITEM DROPPED, wrong item selected (%s)",m_currentFrame,pEntity->GetName(),sel);
			}
			break;

		case eGE_AmmoCount: 
			break;

		case eGE_AttachedAccessory: 
			break;

		case eGE_EntityGrabbed:
			break;

		default:
			break;
	}
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::OnGameplayEvent(IEntity *pEntity, const GameplayEvent &event)
{
	EntityId id;
	if(!pEntity || !(id = pEntity->GetId()))
	{
		GameWarning("TimeDemo:GameState::OnGamePlayEvent: Entity not found");
		return;
	}
	CActor *pActor = (CActor*)(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(id));
	if(!pActor)
	{
		GameWarning("TimeDemo:GameState::OnGamePlayEvent: Entity %s has no actor", pEntity->GetName());
		return;
	}

	GameplayEvent  event2 = event;
	event2.extra = 0;// event2 is the forwarded event, extra either will be not used by the listener or re-set as a string
	uint8 eType = event.event;

	bool bPlayer = (pActor->IsPlayer() && m_mode);
	if(bPlayer || m_mode==GPM_AllActors)
	{
		//items
		switch(eType)
		{
			case eGE_ItemPickedUp:
				{
					CheckInventory(pActor,0);//,*m_pRecordGameEventFtor);
				}
				break;

			case eGE_ItemDropped:
				{
					TItemName itemName = GetItemName(EntityId(event.extra));
					if(!itemName ) //if(itemIdx < 0)
						break;
					event2.description = itemName;
					SendGamePlayEvent(pEntity,event2);
					IEntity* pItemEntity = gEnv->pEntitySystem->FindEntityByName(itemName);
					if(!pItemEntity)
						break;
					IItem* pItem = g_pGame->GetIGameFramework()->GetIItemSystem()->GetItem(pItemEntity->GetId());
					if(!pItem)
						break;

					IEntityClass* pItemClass = pItem->GetEntity()->GetClass();
					if(pItemClass && !strcmpi(pItemClass->GetName(),"SOCOM"))
					{
						IItem* pCurrentItem = pActor->GetCurrentItem();
						if(pCurrentItem)
						{
							IEntityClass* pCurrentItemClass = pCurrentItem->GetEntity()->GetClass();
							if(pCurrentItemClass && !strcmpi(pCurrentItemClass->GetName(),"SOCOM"))
							{
								GameplayEvent event3;
								event3.event = eGE_ItemSelected;
								TItemName itemName = GetItemName(pCurrentItem->GetEntity()->GetId());
								if(!itemName)
									break;
								event3.value = 0;
								event3.description = (const char*)itemName;
								SendGamePlayEvent(pEntity,event3);
							}
						}
					}
				}
				break;

			case eGE_WeaponFireModeChanged:
				{
					TItemName itemIdx = GetItemName(EntityId(event.extra));
					if(!itemIdx)//if(itemIdx < 0)
						break;
					event2.description = (const char*)itemIdx;
					SendGamePlayEvent(pEntity,event2);
				}
				break;

			case eGE_ItemSelected:
				{
					EntityId itemId = EntityId(event.extra);
					TItemName itemIdx = GetItemName(itemId);
					if(itemId && !itemIdx)
						break;
					event2.value = 0;
					event2.description = (const char*)itemIdx;
					SendGamePlayEvent(pEntity,event2);
				}
				break;

			case eGE_AttachedAccessory:
				{
					if(!IsGrenade(event.description)) // NOT OffHandGrenade
						SendGamePlayEvent(pEntity,event2);
				}
				break;

			case eGE_AmmoCount:
				break;

			case eGE_SuitEnergyChanged:
				{
					if(pActor->GetEntity() )
					{
						EntityId id = pActor->GetEntity()->GetId();
						TGameStates::iterator itActor = m_GameStates.find(id);
						if(itActor == m_GameStates.end())
						{
							m_GameStates.insert(std::make_pair(id,SActorGameState()));
							itActor = m_GameStates.find(id);
						}
						if(itActor == m_GameStates.end())
							GameWarning("TimeDemo:GameState: actor %s not found in records",pEntity->GetName());
/*
						SActorGameState& gstate = itActor->second;
						float now = gEnv->pTimer->GetFrameStartTime().GetMilliSeconds();
						float duration =  now - gstate.lastEnergyModTime;
						if(gstate.lastEnergyModTime>0 && duration<1000)
						{
							event2.extrai = (int32)duration;
						}
						gstate.lastEnergyModTime = now;
*/
					}
					SendGamePlayEvent(pEntity,event2);
				}
				break;

			case eGE_EntityGrabbed:
				{
					EntityId entityId = EntityId(event.extra);
					IEntity * pGrabbedEntity = gEnv->pEntitySystem->GetEntity(entityId);
					if(pGrabbedEntity)
					{
						event2.description = pGrabbedEntity->GetName();
						SendGamePlayEvent(pEntity,event2);
					}
				}
				break;

			case eGE_WeaponReload:
			case eGE_ZoomedIn:
			case eGE_ZoomedOut:
			case eGE_SuitModeChanged:
			case eGE_HealthChanged:
			case eGE_ItemExchanged:
				SendGamePlayEvent(pEntity,event2);
				break;

			default:
				break;
		}

	}
}

void CGameStateRecorder::SendGamePlayEvent(IEntity *pEntity, const GameplayEvent &event)
{
	for(TListeners::iterator it = m_listeners.begin(), itEnd = m_listeners.end(); it!=itEnd; ++it)
		(*it)->OnGameplayEvent(pEntity,event);

	OnRecordedGameplayEvent(pEntity,event,0,true); //updates actor game state during recording
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::GetMemoryStatistics(ICrySizer * s)
{
	SIZER_SUBCOMPONENT_NAME(s, "GameStateRecorder");
	s->Add(*this);
	s->AddContainer(m_listeners);
	s->AddContainer(m_GameStates);
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::RegisterListener(IGameplayListener* pL)
{
	if(pL)
		stl::push_back_unique(m_listeners,pL);
}

////////////////////////////////////////////////////////////////////////////
void CGameStateRecorder::UnRegisterListener(IGameplayListener* pL)
{
	stl::find_and_erase(m_listeners, pL);
}


//////////////////////////////////////////////////////////////////////////
float CGameStateRecorder::RenderInfo(float y, bool bRecording)
{
	float retY = 0;

	IRenderer *pRenderer = gEnv->pRenderer;

	if (bRecording)
	{
		float fColor[4] = {1,0,0,1};
		pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," Recording game state");
		retY +=15;
	}
	else 
	{
		const float xp = 300;
		const float xr = 400;
		const float xri = 600;

		float fColor[4] = {0,1,0,1};
		float fColorWarning[4] = {1,1,0,1};

		const char* actorName = m_demo_actorInfo->GetString();
		CActor *pActor = GetActorOfName(actorName);
		if(pActor)
		{
			pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," Game state - Actor: %s --------------------------------------------------",pActor->GetEntity()? pActor->GetEntity()->GetName(): "(no entity)");
			retY +=15;

			if(m_itSingleActorGameState != m_GameStates.end() && pActor->GetEntity() && m_itSingleActorGameState->first == pActor->GetEntity()->GetId())
			{
				ICVar *pVar = gEnv->pConsole->GetCVar( "demo_force_game_state" );
				if(pVar)
				{
					int demo_forceGameState = pVar->GetIVal();
					if(demo_forceGameState)
					{
						pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false,demo_forceGameState==1 ? 
							" Override mode = (health, suit energy)" : " Override mode = (all)");
						retY +=15;
					}
				}

				pRenderer->Draw2dLabel( xp,y+retY, 1.3f, fColor,false,"Current");
				pRenderer->Draw2dLabel( xr,y+retY, 1.3f, fColor,false,"Recorded");
				retY +=15;

				SActorGameState& gstate = m_itSingleActorGameState->second;
				float recordedHealth = (float)gstate.health;
				float health = (float)pActor->GetHealth();
				bool bError = CHECK_MISMATCH(health, recordedHealth,10);

				pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," Health:");
				pRenderer->Draw2dLabel( xp,y+retY, 1.3f, bError? fColorWarning : fColor, false,"%d",(int)health);
				pRenderer->Draw2dLabel( xr,y+retY, 1.3f, bError? fColorWarning : fColor, false,"%d",(int)recordedHealth);
				retY +=15;

				float suitEnergy = GetSuitEnergy(pActor);
				if(suitEnergy>=0) // has actually a suit
				{
					float recordedEnergy = gstate.suitEnergy;
					bool bEnergyError = CHECK_MISMATCH(suitEnergy, recordedEnergy,1);//(suitEnergy!= recordedEnergy);
					int8 sCurMode = GetSuitMode(pActor);
					bool bcurModeORError = (sCurMode<0 || sCurMode>= NANOMODE_LAST);
					int8 recordedMode = gstate.suitMode;
					bool brecModeORError = (recordedMode<0 || recordedMode >= NANOMODE_LAST);
					bool bModeError =  sCurMode != recordedMode || bcurModeORError || brecModeORError;


					pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," Suit Mode:");
					if(bcurModeORError)
						pRenderer->Draw2dLabel( xp,y+retY, 1.3f, fColorWarning , false,"Out of range (%d)",sCurMode);
					else
						pRenderer->Draw2dLabel( xp,y+retY, 1.3f, bModeError? fColorWarning : fColor, false,"%s",m_SuitModeName[sCurMode]);
					if(brecModeORError)
						pRenderer->Draw2dLabel( xr,y+retY, 1.3f, fColorWarning , false,"Out of range (%d)",recordedMode);
					else
						pRenderer->Draw2dLabel( xr,y+retY, 1.3f, bModeError? fColorWarning : fColor, false,"%s",m_SuitModeName[recordedMode]);
					retY +=15;

					pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," Suit Energy:");
					pRenderer->Draw2dLabel( xp,y+retY, 1.3f, bEnergyError? fColorWarning : fColor, false,"%d",(int)suitEnergy);
					pRenderer->Draw2dLabel( xr,y+retY, 1.3f, bEnergyError? fColorWarning : fColor, false,"%d",(int)recordedEnergy);
					retY +=15;
				}
				// items
				pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," Inventory ---------------------------------------------------------------------------------------");
				retY +=15;
				pRenderer->Draw2dLabel( xp,y+retY, 1.3f, fColor,false,"Current");
				pRenderer->Draw2dLabel( xri,y+retY, 1.3f, fColor,false,"Recorded");
				retY +=15;

				CInventory *pInventory = (CInventory*)(pActor->GetInventory());
				if(pInventory)
				{
					int nInvItems = pInventory->GetCount();
						
					TItemContainer& Items = gstate.Items;
					int nRecItems = Items.size();
					int maxItems = max(nRecItems,nInvItems);

					int i=0;
					EntityId curSelectedId = pActor->GetCurrentItemId();
					TItemName curSelClass = GetItemName(curSelectedId);
					bool bSelectedError = !equal_strings(gstate.itemSelected,curSelClass);

					for(; i< nInvItems; i++)
					{
						IItem *pItem = g_pGame->GetIGameFramework()->GetIItemSystem()->GetItem(pInventory->GetItem(i));
						if(pItem)	
						{
							TItemName szItemName = GetItemName(pItem->GetEntityId());

							TItemContainer::iterator it = Items.find(szItemName);
							bError = it==Items.end();
							pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," %2d)",i+1);

							EntityId curId = pItem->GetEntityId();
							TItemName curClass = GetItemName(curId);
							if(equal_strings(curClass,curSelClass) )
								pRenderer->Draw2dLabel( xp-16,y+retY, 1.3f,bSelectedError ? fColorWarning:fColor, false, "[]");

							if(equal_strings(szItemName, gstate.itemSelected))
								pRenderer->Draw2dLabel( xri-16,y+retY, 1.3f,bSelectedError ? fColorWarning:fColor, false, "[]");

							char itemName[32];
							const char* originalItemName = pItem->GetEntity() ? pItem->GetEntity()->GetName():"(null)";
							int length = strlen(originalItemName);
							length = min(length,31);
							strncpy(itemName,originalItemName,length);
							itemName[length]=0;
							pRenderer->Draw2dLabel( 1,y+retY, 1.3f, bError? fColorWarning : fColor, false,"     %s",itemName);

							if(bError)
								pRenderer->Draw2dLabel( xp,y+retY, 1.3f, fColorWarning, false, "Missing");
							else
							{								
								pRenderer->Draw2dLabel( xp,y+retY, 1.3f, fColor, false, "Ok");
							}

						}
						retY +=15;
					}

					/// Accessories

					int nInvAccessories = pInventory->GetAccessoryCount();

					TAccessoryContainer& Accessories = gstate.Accessories;
					int nRecAccessories = Accessories.size();
					if(nRecAccessories)
					{
						pRenderer->Draw2dLabel( 1,y+retY, 1.3f, bError? fColorWarning : fColor, false," Accessories");
						retY +=15;
					}

					for(int j=0 ; j< nInvAccessories; j++,i++)
					{
						const char* accessory = pInventory->GetAccessory(j);
						if(accessory && strlen(accessory))	
						{

							IEntityClass* pClass = gEnv->pEntitySystem->GetClassRegistry()->FindClass(accessory);
							if(pClass)
							{
								TItemName szItemName = pClass->GetName();
								TAccessoryContainer::iterator it = Accessories.find(szItemName);
								bError = it==Accessories.end();
								pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false," %2d)",i+1);

								char itemName[32];
								int length = strlen(accessory);
								length = min(length,31);
								strncpy(itemName,accessory,length);
								itemName[length]=0;
								pRenderer->Draw2dLabel( 1,y+retY, 1.3f, bError? fColorWarning : fColor, false,"     %s",itemName);

								if(bError)
									pRenderer->Draw2dLabel( xp,y+retY, 1.3f, fColorWarning, false, "Missing");
								else
									pRenderer->Draw2dLabel( xp,y+retY, 1.3f, fColor, false, "Ok");

								retY +=15;
							}
						}

					}
					/// Ammo Mags
					TAmmoContainer& Ammo = gstate.AmmoMags;
					int nRecAmmo = Ammo.size();
					int nInvAmmo = pInventory->GetAmmoPackCount();
					if(nInvAmmo)
					{
						pRenderer->Draw2dLabel( 1,y+retY, 1.3f, bError? fColorWarning : fColor, false," Ammo Packs");
						retY +=15;
					}

					pInventory->AmmoIteratorFirst();
					for(int j=0 ; !pInventory->AmmoIteratorEnd(); j++, pInventory->AmmoIteratorNext())
					{
						TAmmoContainer& Mags = gstate.AmmoMags;
						const IEntityClass* pAmmoClass = pInventory->AmmoIteratorGetClass();
						if(pAmmoClass)
						{
							int invAmmoCount = pInventory->AmmoIteratorGetCount();
							const char* ammoClassName = pAmmoClass->GetName();
							bool bNotFound = Mags.find(ammoClassName) == Mags.end();
							int recAmmoCount = bNotFound ? 0 :Mags[ammoClassName];
							bool bError =  bNotFound || invAmmoCount!= recAmmoCount;

							pRenderer->Draw2dLabel( 1,y+retY, 1.3f, bError? fColorWarning : fColor, false,"  %s:",ammoClassName);
							pRenderer->Draw2dLabel( xp,y+retY, 1.3f, bError? fColorWarning : fColor, false,"%d",invAmmoCount);
							if(bNotFound)
								pRenderer->Draw2dLabel( xr,y+retY, 1.3f, fColorWarning, false,"NotRecd");
							else
								pRenderer->Draw2dLabel( xr,y+retY, 1.3f, bError? fColorWarning : fColor, false,"%d",recAmmoCount);
							retY +=15;

						}
					}
				}
			}
			else // m_itSingleActorGameState != m_GameStates.end()
			{
				pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false, "<<Not Recorded>>");
				retY +=15;
			}
		}
		else // pActor
		{
			pRenderer->Draw2dLabel( 1,y+retY, 1.3f, fColor,false, "<<Actor %s not in the map>>",actorName ? actorName:"(no name)");
			retY +=15;		
		}

	}
	return retY;
}

///////////////////////////////////////////////////////////////////////
TItemName CGameStateRecorder::GetItemName(EntityId id, CItem** pItemOut)
{
	if(id)
	{
		CItem* pItem = (CItem*)(gEnv->pGame->GetIGameFramework()->GetIItemSystem()->GetItem(id));
		if(pItem && pItem->GetEntity())
		{
			if(pItemOut)
				*pItemOut = pItem;
			return pItem->GetEntity()->GetName();
		}
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////
CItem* CGameStateRecorder::GetItemOfName(CActor* pActor, TItemName itemName)
{
	CInventory* pInventory= (CInventory*)(pActor->GetInventory());
	if(pInventory && itemName)
		return (CItem*)(pInventory->GetItemByName(itemName));

	return 0;
}


///////////////////////////////////////////////////////////////////////
/*template <class EventHandlerFunc> */void CGameStateRecorder::CheckInventory(CActor* pActor, IItem *pItem)//, EventHandlerFunc eventHandler)
{

	if(pActor)
	{
		TGameStates::iterator itGS;
		if(pActor->IsPlayer())
			itGS = m_itSingleActorGameState;
		else if(pActor->GetEntity())
			itGS = m_GameStates.find(pActor->GetEntity()->GetId());
		else
			return;

		if(itGS != m_GameStates.end())
		{
			SActorGameState& gstate = itGS->second;

			// items

			CInventory *pInventory = (CInventory*)(pActor->GetInventory());
			if(pInventory)
			{
				bool bSingleItem = (pItem!=0);
				int nInvItems = (bSingleItem ? 1 : pInventory->GetCount());
				TItemContainer& Items = gstate.Items;
				
				for(int i=0; i< nInvItems; i++)
				{
					if(!bSingleItem) 
						pItem = g_pGame->GetIGameFramework()->GetIItemSystem()->GetItem(pInventory->GetItem(i));

					if(pItem)
					{
						TItemName szItemName = GetItemName(pItem->GetEntityId());

						TItemContainer::iterator it = Items.find(szItemName);
						if(it==Items.end())
						{
							it = (Items.insert(std::make_pair(szItemName,SItemProperties()))).first;

							GameplayEvent event;
							event.event = eGE_ItemPickedUp;
							event.description = szItemName;
							//eventHandler(pActor,event);
							SendGamePlayEvent(pActor->GetEntity(),event);
						}
					}
				}

				/// Accessories

				int nInvAccessories = pInventory->GetAccessoryCount();

				TAccessoryContainer& Accessories = gstate.Accessories;
				int nRecAccessories = Accessories.size();

				for(int j=0 ; j< nInvAccessories; j++)
				{
					const char* accessory = pInventory->GetAccessory(j);
					if(accessory && strlen(accessory))	
					{
						IEntityClass* pClass = gEnv->pEntitySystem->GetClassRegistry()->FindClass(accessory);
						if(pClass)
						{
							TItemName itemClass = pClass->GetName();
							TAccessoryContainer::iterator it = Accessories.find(itemClass);
							if(it==Accessories.end())
							{
								GameplayEvent event;
								event.event = eGE_AccessoryPickedUp;
								//event.value = accIdx;
								event.description = itemClass;
//								eventHandler(pActor,event);
								SendGamePlayEvent(pActor->GetEntity(),event);
							}
						}
					}
				}

				/// Ammo

				TAmmoContainer& Ammo = gstate.AmmoMags;
				int nRecAmmo = Ammo.size();
				int nInvAmmo = pInventory->GetAmmoPackCount();
				pInventory->AmmoIteratorFirst();
				for(int j=0 ; !pInventory->AmmoIteratorEnd(); j++, pInventory->AmmoIteratorNext())
				{
					TAmmoContainer& Mags = gstate.AmmoMags;
					const IEntityClass* pAmmoClass = pInventory->AmmoIteratorGetClass();
					if(pAmmoClass)
					{
						const char* ammoClassName = pAmmoClass->GetName();
						int ammoCount = pInventory->AmmoIteratorGetCount();
						if(Mags.find(ammoClassName) == Mags.end() || ammoCount!= Mags[ammoClassName])
						{
							GameplayEvent event;
							event.event = eGE_AmmoPickedUp;
							event.description = ammoClassName;
							event.value = (float)ammoCount;
//							eventHandler(pActor,event);
							SendGamePlayEvent(pActor->GetEntity(),event);
						}
					}
				}
			}
		}
	}
}

/*
///////////////////////////////////////////////////////////////////////////////////////
// Nanosuit listener callbacks
// This can work only for the player
// if other actors have nanosuit and their state is recorded, this INanoSuitListener method
// must be added the actor parameter

void CGameStateRecorder::ModeChanged(ENanoMode mode)
{
	CActor *pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
	if(pActor && !pActor->GetSpectatorMode() && pActor->IsPlayer() && m_itSingleActorGameState != m_GameStates.end())
	{
		SActorGameState& playerState = m_itSingleActorGameState->second;
		GameplayEvent event;
		event.event = eGE_SuitModeChanged;
		event.value = mode;
		//playerState.suitEnergy = mode;
		SendGamePlayEvent(pActor->GetEntity(),event);
	}
}


///////////////////////////////////////////////////////////////////////////////////////
// This can work only for the player
// if other actors have nanosuit and their state is recorded, this INanoSuitListener method
// must be added the actor parameter
void CGameStateRecorder::EnergyChanged(float energy)
{
	CActor *pActor = static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
	if(pActor && !pActor->GetSpectatorMode() && pActor->IsPlayer() && m_itSingleActorGameState != m_GameStates.end())
	{
		SActorGameState& playerState = m_itSingleActorGameState->second;
//		playerState.suitEnergy = energy;
		GameplayEvent event;
		event.event = eGE_SuitEnergyChanged;
		event.value = energy;
		//SendGamePlayEvent(pActor->GetEntity(),event);
	}
}
*/
///////////////////////////////////////////////////////////////////////////////////////
CActor* CGameStateRecorder::GetActorOfName( const char* name)
{
	if(!strcmpi(name,"all"))
		return NULL;

	if(!strcmpi(name,"player"))
		return static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetClientActor());
	else
	{
		IEntity* pEntity = gEnv->pEntitySystem->FindEntityByName(name);
		if(pEntity)
			return static_cast<CActor *>(gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(pEntity->GetId()));
	}
	return NULL;
}

