/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id: RealtimeRemoveUpdate.cpp,v 1.1 2009/01/03 10:45:15 PauloZaffari Exp wwwrun $
$DateTime$
Description:  This is the source file for the module Realtime remote 
update. The purpose of this module is to allow data update to happen 
remotely so that you can, for example, edit the terrain and see the changes
in the console.
-------------------------------------------------------------------------
History:
- 03:01:2009   10:45: Created by Paulo Zaffari
*************************************************************************/

#include "StdAfx.h"
#include "RealtimeRemoteUpdate.h"
#include "ISystem.h"
#include "I3DEngine.h"
#include <IEntitySystem.h>
#include "IGame.h"
#include "IGameFramework.h"
#include "IViewSystem.h"
#include "IEntitySystem.h"
#include "IGameFramework.h"
#include "Game.h"
#include "IGameRulesSystem.h"

#include "Player.h"

#ifdef XENON
#include "Xtl.h"
#endif //XENON

//////////////////////////////////////////////////////////////////////////
CRealtimeRemoteUpdateListener&	CRealtimeRemoteUpdateListener::GetRealtimeRemoteUpdateListener()
{
	static CRealtimeRemoteUpdateListener oRealtimeUpdateListener;
	return oRealtimeUpdateListener;
}
//////////////////////////////////////////////////////////////////////////
bool CRealtimeRemoteUpdateListener::Enable(bool boEnable)
{
	if (!gEnv)
	{
		return false;
	}

	if (!gEnv->pSystem)
	{
		return false;
	}

	INotificationNetwork*	piNotificationNetwork=gEnv->pSystem->GetINotificationNetwork();
	if (!piNotificationNetwork)
	{
		return false;
	}

	if (boEnable)
	{
		m_boIsEnabled=piNotificationNetwork->ListenerBind("RealtimeUpdate",this);
	}
	else
	{
		piNotificationNetwork->ListenerRemove(this);
		m_boIsEnabled=false;
	}

	return m_boIsEnabled;
}
//////////////////////////////////////////////////////////////////////////
bool	CRealtimeRemoteUpdateListener::IsEnabled()
{
	if (!gEnv)
	{
		return false;
	}

	if (!gEnv->pSystem)
	{
		return false;
	}

	INotificationNetwork*	piNotificationNetwork=gEnv->pSystem->GetINotificationNetwork();
	if (!piNotificationNetwork)
	{
		return false;
	}

	// We should instead query the notification network here.
	return m_boIsEnabled;	
}
//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::OnNotificationNetworkReceive(const void *pBuffer, size_t length)
{
	char*							szBuffer((char*)pBuffer);
	size_t						nStringLenght(0);
	unsigned char*		chBinaryBuffer(NULL);
	XmlNodeRef				oXmlNode=gEnv->pSystem->LoadXmlFromString(szBuffer);
	size_t						nBinaryDataSize(0);

	// Currently, if we have no XML node this is not a well formed message and
	// thus we stop processing.
	if (!oXmlNode)
	{
		return;
	}

	nStringLenght=strlen(szBuffer)+1;
	chBinaryBuffer=(unsigned char*)(szBuffer+nStringLenght);

	if (strcmp(oXmlNode->getTag(),"SyncMessage")!=0)
	{
		return;
	}

	string oSyncType = oXmlNode->getAttr("Type");
	if (oSyncType.empty())
	{
		return;
	}

	if (!oXmlNode->getAttr("BinaryDataSize",nBinaryDataSize))
	{
		return;
	}

#ifdef XENON
	// We are, this way, reseting the timer for the screensaver.
	XEnableScreenSaver(FALSE);
	XEnableScreenSaver(TRUE);
#endif //XENON

  static std::vector<struct CStatObj*> * pStatObjTable = NULL;
  static std::vector<IMaterial*> * pMatTable = NULL;

	if (oSyncType.compare("EngineTerrainData")==0)
	{
    gEnv->p3DEngine->LockCGFResources();

		if (nBinaryDataSize>0)
		{
			size_t					nUncompressedBinarySize(nBinaryDataSize);
			unsigned char*	szData=new unsigned char[nBinaryDataSize];

      gEnv->pSystem->DecompressDataBlock(chBinaryBuffer,length-nStringLenght,szData,nUncompressedBinarySize);

			if(ITerrain * piTerrain = gEnv->p3DEngine->GetITerrain())
      {
        SHotUpdateInfo * pExportInfo = (SHotUpdateInfo *)szData;

        if(piTerrain)
          piTerrain->SetCompiledData((uchar*)szData+sizeof(SHotUpdateInfo),nBinaryDataSize-sizeof(SHotUpdateInfo),&pStatObjTable,&pMatTable,true,pExportInfo);

        SAFE_DELETE_ARRAY(szData);
      }
		}
	}
	else if (oSyncType.compare("EngineIndoorData")==0)
	{
    if (nBinaryDataSize>0)
    {
      size_t					nUncompressedBinarySize(nBinaryDataSize);
      unsigned char*	szData=new unsigned char[nBinaryDataSize];

      gEnv->pSystem->DecompressDataBlock(chBinaryBuffer,length-nStringLenght,szData,nUncompressedBinarySize);

      if(IVisAreaManager * piIVisAreaManager = gEnv->p3DEngine->GetIVisAreaManager())
      {
        SHotUpdateInfo * pExportInfo = (SHotUpdateInfo *)szData;

        if(piIVisAreaManager)
          piIVisAreaManager->SetCompiledData((uchar*)szData+sizeof(SHotUpdateInfo),nBinaryDataSize-sizeof(SHotUpdateInfo),&pStatObjTable,&pMatTable,true,pExportInfo);

        SAFE_DELETE_ARRAY(szData);
      }
    }
    
    gEnv->p3DEngine->UnlockCGFResources();
    
    pStatObjTable = NULL;
    pMatTable = NULL;
	}
	else if (oSyncType.compare("Voxel")==0)
	{
		if (nBinaryDataSize>0)
		{
			SVoxTerrainInfo		* pHeader		= (SVoxTerrainInfo*)chBinaryBuffer;
			IVoxTerrain*			pVoxTerrain = gEnv->p3DEngine->CreateVoxTerrain(*pHeader);
			pVoxTerrain->SetCompiledData(chBinaryBuffer,nBinaryDataSize,true,GetPlatformEndian(),NULL);
		}
	}
	else if (oSyncType.compare("Vegetation")==0)
	{
		XmlNodeRef	oCurrentNode=oXmlNode->findChild("Vegetation");


	}
	else if (oSyncType.compare("DetailLayers")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("SurfaceTypes");
		if (oChildRootNode)
		{
			gEnv->p3DEngine->LoadTerrainSurfacesFromXML(oChildRootNode,true);
		}
	}
	else if (oSyncType.compare("Environment")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("Environment");
		if (oChildRootNode)
		{
			gEnv->p3DEngine->LoadEnvironmentSettingsFromXML( oChildRootNode );	
		}
	}
	else if (oSyncType.compare("Camera")==0)
	{		
		Vec3						oPosition;
		Vec3						oViewDirection;
		int							nStartSync(0);

		if (!oXmlNode->getAttr("Position",oPosition))
		{
			return;
		}
		if (!oXmlNode->getAttr("Direction",oViewDirection))
		{
			return;
		}

		//Matrix33	oDirectionMatrix(Matrix33::CreateRotationVDir(oViewDirection));

		//Matrix34	oNewMatrix(oDirectionMatrix);

		// complicated but maybe the best Entity we can move to the given spot
		IGame *pGame = gEnv->pGame;																								if(!pGame)return;
		IGameFramework *pGameFramework=pGame->GetIGameFramework();								if(!pGameFramework)return;
		IViewSystem *pViewSystem=pGameFramework->GetIViewSystem();								if(!pViewSystem)return;
		IView *pView=pViewSystem->GetActiveView();																if(!pView)return;
		IEntity *pEntity = gEnv->pEntitySystem->GetEntity(pView->GetLinkedId());	if(!pEntity)return;

		CPlayer *pPlayer=static_cast<CPlayer *>(pGameFramework->GetClientActor());
		IEntity *pPlayerEntity(NULL);
		IPhysicalEntity*	piPlayerPhysics(NULL);

		pe_player_dimensions dim;

		if (pPlayer)
		{
			pPlayerEntity=pPlayer->GetEntity();
		}

		if (pPlayerEntity)
		{
			piPlayerPhysics=pPlayerEntity->GetPhysics();
		}

		if (piPlayerPhysics)
		{
			piPlayerPhysics->GetParams( &dim );
			oPosition.z = oPosition.z - dim.heightHead;
		}


		pEntity->SetPos(oPosition);

		//////////////////////////////////////////////////////////////////////////
		// Should be removed later.
		// When we rotate the entity it is not rotated around 0,0,0, but around
		// it's Pivot (only when it's invisible), we simply set by force the right camera rotation.
		pViewSystem->SetOverrideCameraRotation(true,Quat::CreateRotationVDir(oViewDirection));
		//////////////////////////////////////////////////////////////////////////


		if (oXmlNode->getAttr("StartSync",nStartSync))
		{
			if (pPlayer)
			{
				if (nStartSync)
				{
					if (!m_boIsSyncingCamera)
					{
						m_boIsSyncingCamera=true;
						m_nPreviousFlyMode=pPlayer->GetFlyMode();
						pPlayer->SetFlyMode(2);
						if (pPlayerEntity)
						{
							pPlayerEntity->Hide(true);
							pViewSystem->SetOverrideCameraRotation(true,Quat::CreateRotationVDir(oViewDirection));
						}
					}
				}
				else
				{
					if (m_boIsSyncingCamera)
					{
						m_boIsSyncingCamera=false;
						pPlayer->SetFlyMode(m_nPreviousFlyMode);
						if (pPlayerEntity)
						{
							pPlayerEntity->Hide(false);
							pViewSystem->SetOverrideCameraRotation(false,Quat::CreateRotationVDir(oViewDirection));
							pEntity->SetRotation(Quat::CreateRotationVDir(oViewDirection));
						}
					}
				}
			}
		}
	}
	else if (oSyncType.compare("Entities")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("Entities");
		if (oChildRootNode)
			LoadEntities( oChildRootNode );
	}
	else if (oSyncType.compare("TimeOfDay")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("TimeOfDay");
		if (oChildRootNode)
			LoadTimeOfDay( oChildRootNode );
	}
	else if (oSyncType.compare("Materials")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("Materials");
		if (oChildRootNode)
			LoadMaterials( oChildRootNode );
	}
	else if (oSyncType.compare("EntityArchetype")==0)
	{
		XmlNodeRef oChildRootNode=oXmlNode->findChild("EntityPrototypes");
		if (oChildRootNode)
			LoadArchetypes( oChildRootNode );
	}
	else if (oSyncType.compare("ConsoleVariables")==0)
	{
		LoadConsoleVariables(oXmlNode);
	}
	else if (oSyncType.compare("Particles")==0)
	{		
		LoadParticles(oXmlNode);
	}
	else if (oSyncType.compare("LayerTexture")==0)
	{
		if (nBinaryDataSize>0)
		{
			size_t					nUncompressedBinarySize(nBinaryDataSize);
			unsigned char*	szData=new unsigned char[nBinaryDataSize];

			if (!szData)
			{
				return;
			}

			if (!gEnv->pSystem->DecompressDataBlock(chBinaryBuffer,length-nStringLenght,szData,nUncompressedBinarySize))
			{
				SAFE_DELETE_ARRAY(szData);
				return;
			}

			LoadTerrainLayer(oXmlNode,szData);

			SAFE_DELETE_ARRAY(szData);
		}
	}
}
//////////////////////////////////////////////////////////////////////////
CRealtimeRemoteUpdateListener::CRealtimeRemoteUpdateListener():
m_boIsEnabled(false),
m_nPreviousFlyMode(0),
m_boIsSyncingCamera(false)
{

}
//////////////////////////////////////////////////////////////////////////
CRealtimeRemoteUpdateListener::~CRealtimeRemoteUpdateListener()
{

}
//////////////////////////////////////////////////////////////////////////
void	CRealtimeRemoteUpdateListener::LoadArchetypes(XmlNodeRef &root)
{
	IEntitySystem		*pEntitySystem	= gEnv->pEntitySystem;

	// Remove Entities with ID`s from the list.
	for (int i = 0; i < root->getChildCount(); i++)
	{
		XmlNodeRef entityNode = root->getChild(i);
		if (entityNode->isTag("EntityPrototype"))
		{
			pEntitySystem->LoadEntityArchetype(entityNode);
		}
	}	
}
//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::LoadEntities( XmlNodeRef &root )
{
	IEntitySystem *pEntitySystem = gEnv->pEntitySystem;

	bool bTransformOnly(false);
	bool bDeleteOnly = false;
	bool bRemoveAllOld = true;

	if (root->haveAttr("PartialUpdate"))
	{
		bRemoveAllOld = false;
	}
	if (root->haveAttr("Delete"))
	{
		bDeleteOnly = true;
	}

	//bRemoveAllOld = false;
	
	//////////////////////////////////////////////////////////////////////////
	// Delete all entities except the unremovable ones and the local player.
	if (bRemoveAllOld)
	{
		IEntityItPtr pIt = pEntitySystem->GetEntityIterator();

		IGameFramework		*piGameFramework(g_pGame->GetIGameFramework());
		IEntity						*piRulesEntity(NULL);
		if (piGameFramework)
		{
			IGameRulesSystem	*piGameRulesSystem(piGameFramework->GetIGameRulesSystem());
			if (piGameRulesSystem)
			{
				piRulesEntity=piGameRulesSystem->GetCurrentGameRulesEntity();
			}
		}

		pIt->MoveFirst();
		while (!pIt->IsEnd())
		{
			IEntity * pEntity = pIt->Next();
			uint32 nEntityFlags = pEntity->GetFlags();

			// Local player must not be deleted.
			if (nEntityFlags & ENTITY_FLAG_LOCAL_PLAYER)
				continue;

			// Rules should not be deleted as well.
			if (piRulesEntity)
			{
				if (pEntity->GetId()==piRulesEntity->GetId())
				{
					continue;
				}
			}
			
			pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
			pEntitySystem->RemoveEntity( pEntity->GetId() );
		}
		// Force deletion of removed entities.
		pEntitySystem->DeletePendingEntities();
		//////////////////////////////////////////////////////////////////////////
	}
	else
	{
		// Remove Entities with ID`s from the list.
		for (int i = 0; i < root->getChildCount(); i++)
		{
			XmlNodeRef objectNode = root->getChild(i);
			if (objectNode->isTag("Entity"))
			{
				// reserve the id
				EntityId id;
				if (objectNode->getAttr( "EntityId", id ))
				{
					IEntity * pEntity = pEntitySystem->GetEntity(id);
					if (!pEntity)
					{
						pEntitySystem->RemoveEntity(id,true);
						continue;
					}

					if (!objectNode->getAttr("TransformOnly",bTransformOnly ))
					{
						pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
						pEntitySystem->RemoveEntity(id,true);
						continue;
					}

					if (!bTransformOnly)
					{
						pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
						pEntitySystem->RemoveEntity(id,true);
						continue;
					}

					//////////////////////////////////////////////////////////////////////////
					//TEMPORARY, should be removed after GDC09 Presentation
					XmlString strEntityClass;
					if (objectNode->getAttr("EntityClass",strEntityClass))
					{
						if (stricmp(strEntityClass,"simplelight")!=0)
						{
							pEntity->ClearFlags(ENTITY_FLAG_UNREMOVABLE);
							pEntitySystem->RemoveEntity(id,true);
							continue;					
						}
					}
					//////////////////////////////////////////////////////////////////////////

					Vec3 oPos(0,0,0);
					Vec3 oScale(0,0,0);
					Quat oRotate(1,0,0,0);

					objectNode->getAttr("Pos",oPos );
					objectNode->getAttr("Rotate",oRotate );
					objectNode->getAttr("Scale",oScale );

					pEntity->SetPosRotScale(oPos,oRotate,oScale);

				}
			}
		}
		// Force deletion of removed entities.
		pEntitySystem->DeletePendingEntities();
	}

	if (!bDeleteOnly)
	{
		pEntitySystem->LoadEntities(root);
	}
}

//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::LoadTimeOfDay( XmlNodeRef &root )
{
	gEnv->p3DEngine->GetTimeOfDay()->Serialize( root,true );
}

//////////////////////////////////////////////////////////////////////////
void CRealtimeRemoteUpdateListener::LoadMaterials( XmlNodeRef &root )
{
	// Remove Entities with ID`s from the list.
	for (int i = 0; i < root->getChildCount(); i++)
	{
		XmlNodeRef mtlNode = root->getChild(i);
		if (mtlNode->isTag("Material"))
		{
			const char *mtlName = mtlNode->getAttr( "name" );
			gEnv->p3DEngine->GetMaterialManager()->LoadMaterialFromXml( mtlName,mtlNode );
		}
	}
}
//////////////////////////////////////////////////////////////////////////
void  CRealtimeRemoteUpdateListener::LoadConsoleVariables(XmlNodeRef &root )
{
	IConsole*	piConsole(NULL);
	char*			szKey(NULL);
	char*			szValue(NULL);
	ICVar*		piCVar(NULL);

	piConsole=gEnv->pConsole;
	if (!piConsole)
	{
		return;
	}

	// Remove Entities with ID`s from the list.
	for (int i = 0; i < root->getNumAttributes(); i++)
	{
		root->getAttributeByIndex(i,(const char**)&szKey,(const char**)&szValue);
		piCVar=piConsole->GetCVar(szKey);
		if (!piCVar)
		{
			continue;
		}
		piCVar->Set(szValue);
	}
}
//////////////////////////////////////////////////////////////////////////
void  CRealtimeRemoteUpdateListener::LoadParticles(XmlNodeRef &root )
{
	XmlNodeRef		oParticlesLibrary;
	XmlNodeRef		oLibrary;
	XmlString			strLibraryName;

	int						nCurrentChild(0);
	int						nNumberOfChildren(0);

	oParticlesLibrary=root->findChild("ParticlesLibrary");
	if (!oParticlesLibrary)
	{
		return;
	}

	nNumberOfChildren=oParticlesLibrary->getChildCount();
	for (nCurrentChild=0;nCurrentChild<nNumberOfChildren;++nCurrentChild)
	{
		oLibrary=oParticlesLibrary->getChild(nCurrentChild);
		if (oLibrary->isTag("Library"))
		{
			continue;
		}
		if (!oLibrary->getAttr("name",strLibraryName))
		{
			continue;
		}
		gEnv->p3DEngine->LoadParticleLibrary((const char*)strLibraryName,oLibrary,true);
	}
}
//////////////////////////////////////////////////////////////////////////
void  CRealtimeRemoteUpdateListener::LoadTerrainLayer(XmlNodeRef &root, unsigned char*	uchData)
{
	int texId(0);
	int posx(0),posy(0);
	int w(0),h(0);
	int					nSourceFormat(0);
	ETEX_Format	eTFSrc(eTF_R8G8B8);

	if (!root->getAttr("Posx",posx))
	{
		return;
	}

	if (!root->getAttr("Posy",posy))
	{
		return;
	}

	if (!root->getAttr("w",w))
	{
		return;
	}

	if (!root->getAttr("h",h))
	{
		return;
	}

	if (!root->getAttr("ETEX_Format",nSourceFormat))
	{
		return;
	}

	eTFSrc=(ETEX_Format)nSourceFormat;

	if (gEnv->pRenderer&&gEnv->p3DEngine)
	{

		//texId = gEnv->pRenderer->DownLoadToVideoMemory(uchData,w,h, 
		//	eTFSrc,eTF_DXT5, 0, false, FILTER_LINEAR, 0, 0, FT_USAGE_ALLOWREADSRGB|FT_TEX_WAS_NOT_PRE_TILED);
#if defined(XENON)
		texId = gEnv->pRenderer->DownLoadToVideoMemory(uchData,w,h, 
			eTFSrc,eTF_DXT1, 0, false, FILTER_LINEAR, 0, 0, FT_USAGE_ALLOWREADSRGB|FT_TEX_WAS_NOT_PRE_TILED);
#endif
#if defined(PS3)
		SwapEndian((uint32*)uchData,w*h);
		for(uint32 a=0;a!=w*h;++a)
		{
			uint32& rColor	=	reinterpret_cast<uint32*>(uchData)[a];
			rColor	=	(rColor&0xff00ff00)|
								((rColor&0xff)<<16)|
								((rColor&0xff0000)>>16);
		}
		texId = gEnv->pRenderer->DownLoadToVideoMemory(uchData,w,h, 
			eTFSrc,eTFSrc, 0, false, FILTER_LINEAR, 0, 0, FT_USAGE_ALLOWREADSRGB|FT_TEX_WAS_NOT_PRE_TILED);
#endif

		// Swapped x & y for historical reasons.
		gEnv->p3DEngine->SetTerrainSectorTexture(posy,posx,texId);		
	}
}
//////////////////////////////////////////////////////////////////////////
