/*************************************************************************
  Crytek Source File.
  Copyright (C), Crytek Studios, 2001-2004.
 -------------------------------------------------------------------------
  $Id$
  $DateTime$
  
 -------------------------------------------------------------------------
  History:
  - 6:12:2004: Created by Filippo De Luca

*************************************************************************/
#include "StdAfx.h"
#include "Game.h"
#include "Alien.h"
#include "GameUtils.h"
#include "GameActions.h"

#include <IViewSystem.h>
#include <IItemSystem.h>
#include <IPhysics.h>
#include <ICryAnimation.h>
#include <ISerialize.h>
#include <IRenderAuxGeom.h>
#include <IEffectSystem.h>

#include "CompatibilityAlienMovementController.h"


//--------------------
//this function will be called from the engine at the right time, since bones editing must be placed at the right time.
int AlienProcessBones(ICharacterInstance *pCharacter,void *pAlien)
{
	//FIXME: do something to remove GetISystem()->GetITimer()->GetFrameTime()
	//process bones specific stuff (IK, torso rotation, etc)
	((CAlien *)pAlien)->ProcessBonesRotation(pCharacter,GetISystem()->GetITimer()->GetFrameTime());

	return 1;
}
//--------------------

CAlien::CAlien() : 
	m_pItemSystem(0),
	m_curSpeed(0),
	m_forceOrient(false),
	m_endOfThePathTime(-1.0f),
	m_roll(0),
	m_pGroundEffect(NULL)
{	
	m_tentaclesProxy.clear();
	m_tentaclesProxyFullAnimation.clear();
}

CAlien::~CAlien()
{
	m_tentaclesProxy.clear();
	m_tentaclesProxyFullAnimation.clear();

	GetGameObject()->ReleaseActions( this );

	ICharacterInstance *pCharacter = GetEntity()->GetCharacter(0);
	if(pCharacter)
		pCharacter->GetISkeleton()->SetProceduralAnimCallback(0,0);

	delete m_pGroundEffect;
}

void CAlien::BindInputs( IAnimationGraph * pGraph )
{
	CActor::BindInputs(pGraph);

	if (pGraph)
	{
		m_inputSpeed = pGraph->LookupInputId("Speed");
		m_inputDesiredSpeed = pGraph->LookupInputId("DesiredSpeed");
		m_inputAiming = pGraph->LookupInputId("Aiming");
	}
}

void CAlien::ProcessEvent(SEntityEvent& event)
{
	if (event.event == ENTITY_EVENT_HIDE || event.event == ENTITY_EVENT_UNHIDE)
	{
		ICharacterInstance *pCharacter = GetEntity()->GetCharacter(0);	

		if (pCharacter)
		{
			pe_params_flags pf;
			if (event.event == ENTITY_EVENT_HIDE)
				pf.flags = 0;
			else
			{
				pf.flagsOR = rope_findiff_attached_vel | rope_no_solver | pef_traceable;
				if (m_params.tentaclesCollide)
					pf.flagsOR |= rope_ignore_attachments | rope_collides_with_terrain | rope_collides;
			}

			int tNum(0);
			IPhysicalEntity *pTentacle;

			while(pTentacle = pCharacter->GetCharacterPhysics(tNum++))
				pTentacle->SetParams(&pf);
		}
	}
	else
		CActor::ProcessEvent(event);
}

//FIXME:testing
ICVar * pPhysicsAnimRatio = 0;

bool CAlien::Init( IGameObject * pGameObject )
{
	//FIXME:testing
	if (!pPhysicsAnimRatio)
		pPhysicsAnimRatio = GetISystem()->GetIConsole()->RegisterFloat( "g_alienPhysicsAnimRatio", 0.0f, VF_CHEAT );

	if (!CActor::Init(pGameObject))
		return false;

	if (!pGameObject->CaptureActions( this ))
		return false;

	m_pItemSystem = g_pGame->GetIGameFramework()->GetIItemSystem();

	Revive();

	return true;
}

void CAlien::PostPhysicalize()
{
	CActor::PostPhysicalize();

	ICharacterInstance *pCharacter = GetEntity()->GetCharacter(0);

	if (!pCharacter)
		return;

	m_tentaclesProxy.clear();
	m_tentaclesProxyFullAnimation.clear();

	pCharacter->GetISkeleton()->SetProceduralAnimCallback(AlienProcessBones,this);
	pCharacter->EnableStartAnimation(true);

	//collect all the tentacle proxies from the attachments
  IAttachmentManager* pIAttachmentManager = pCharacter->GetIAttachmentManager();
	uint32 numAttachmnets = pIAttachmentManager ? pIAttachmentManager->GetAttachmentCount() : 0;

  for (uint32 i=0; i<numAttachmnets; ++i) 
  {
		IAttachment* pIAttachment = pIAttachmentManager->GetInterfaceByIndex(i);			
    IAttachmentObject* pIAttachmentObject = pIAttachment->GetIAttachmentObject();

    if (pIAttachmentObject) 
    {
			ICharacterInstance* pITentacleInstance = pIAttachmentObject->GetICharacterInstance();
			PushCharacterTentacles(pITentacleInstance);
		}
	}

	//and from the main character
	PushCharacterTentacles(pCharacter);

	//CryLogAlways("%s has %i tentacles",m_pEntity->GetName(),m_tentaclesProxy.size());

	//set tentacles params
	pe_params_flags pf;
	pf.flagsOR = rope_findiff_attached_vel | rope_no_solver | pef_traceable;
	if (m_params.tentaclesCollide)
		pf.flagsOR |= rope_ignore_attachments | rope_collides_with_terrain | rope_collides;

	pe_params_rope pRope;							
	pRope.bTargetPoseActive = 1;
	pRope.stiffnessAnim = 1.0f;
	pRope.stiffnessDecayAnim = 0.0f;
	pRope.collDist = m_params.tentaclesRadius;
	pRope.surface_idx = GetISystem()->GetI3DEngine()->GetMaterialManager()->GetSurfaceTypeIdByName( m_params.tentaclesMaterial,"alien tentacles" );
	
	float jlim = GetISystem()->GetIConsole()->GetCVar("g_tentacle_joint_limit")->GetFVal();
	if (jlim>=0)
		pRope.jointLimit = DEG2RAD(jlim);
	else
		pRope.jointLimit = DEG2RAD(m_params.tentaclesJointLimit);

	std::vector<IPhysicalEntity *>::iterator it;
	for (it = m_tentaclesProxy.begin(); it != m_tentaclesProxy.end(); it++)
	{
		IPhysicalEntity *pT = *it;
		if (pT)
		{
			pT->SetParams(&pRope);
 			pT->SetParams(&pf);
		}
	}

	//FIXME:this disable the impulse, remove it
	IPhysicalEntity *pPhysEnt = pCharacter->GetCharacterPhysics(-1);

	if (pPhysEnt)
	{
		pe_params_flags pFlags;
		pFlags.flagsOR = pef_monitor_impulses;
		//pPhysEnt->SetParams(&pFlags);
	}

	//set a default offset for the character, so in the editor the bbox is correct
	m_charLocalMtx.SetIdentity();
	m_charLocalMtx.SetTranslation(GetStanceInfo(STANCE_STAND)->modelOffset);

	GetEntity()->SetSlotLocalTM(0,m_charLocalMtx);
}

void CAlien::DetachTentacle(ICharacterInstance *pCharacter,const char *tentacle)
{
  IAttachmentManager* pIAttachmentManager = pCharacter->GetIAttachmentManager();
  uint32 numAttachmnets = pIAttachmentManager->GetAttachmentCount();

  IAttachment* pIAttachment = pIAttachmentManager->GetInterfaceByName(tentacle);
	IAttachmentObject* pIAttachmentObject = pIAttachment->GetIAttachmentObject();

  if (pIAttachmentObject) 
	{
	  //get character-instance of a tentacle 
    ICharacterInstance *pITentacleInstance = pIAttachmentObject->GetICharacterInstance();

		//detach tentacle from methagen 
		pIAttachment->ClearBinding();
	}
}

void CAlien::PushCharacterTentacles(ICharacterInstance *pCharacter)
{
	if (!pCharacter)
		return;

	int tNum(0);
	IPhysicalEntity *pTentacle = pCharacter->GetCharacterPhysics(tNum);

	while(pTentacle)
	{
		m_tentaclesProxy.push_back(pTentacle);
		pTentacle = pCharacter->GetCharacterPhysics(++tNum);
	}

	if (m_params.fullAnimTentacles[0])
	{
		char *pBone;
		char boneList[256];

		strcpy(boneList,m_params.fullAnimTentacles);
		pBone = boneList;
		pBone = strtok(pBone,";");

		while (pBone != NULL && *pBone)
		{
			pTentacle = pCharacter->GetCharacterPhysics(pBone);
			if (pTentacle) 
				m_tentaclesProxyFullAnimation.push_back(pTentacle);

			pBone = strtok(NULL,";");
		}
	}
}

void CAlien::UpdateAnimGraph( IAnimationGraphState * pState )
{
	CActor::UpdateAnimGraph(pState);

	if (pState)
	{
		pState->SetInput(m_inputSpeed, m_stats.speedModule);
		pState->SetInput(m_inputDesiredSpeed, m_stats.desiredSpeed);
	}
}

/*void CAlien::OnPhysicsPreStep(float frameTime)
{

}*/

void CAlien::Update(SEntityUpdateContext& ctx, int updateSlot)
{
	FUNCTION_PROFILER(GetISystem(), PROFILE_GAME);

	CActor::Update(ctx,updateSlot);

	const float frameTime = ctx.fFrameTime;
  IEntity* pEnt = GetEntity();

	//offset the character so its hip is at entity's origin
	ICharacterInstance *pCharacter = pEnt->GetCharacter(0);
		
	if (pCharacter)
	{
		Vec3 goal = (m_stats.isRagDoll?Vec3(0,0,0):GetStanceInfo(m_stance)->modelOffset);
		Interpolate(m_modelOffset,goal,5.0f,frameTime);
		
		m_charLocalMtx.SetTranslation(m_modelOffset+m_modelOffsetAdd);

		pEnt->SetSlotLocalTM(0,m_charLocalMtx);
	}
	//

	if (!m_stats.isRagDoll && GetHealth()>0)
	{
		//FIXME:maybe find a better position for this?
		//when the player is supposed to reach some precise position/direction
		if (m_input.posTarget.len2()>0.0f)
		{ 
			Vec3 desiredMovement(m_input.posTarget - pEnt->GetWorldPos());

			float distance = desiredMovement.len();
	    
			desiredMovement *= 0.1f;
			float movelen(distance * 0.1f);

			if (movelen>1.0f)
				desiredMovement /= movelen;
	    
			desiredMovement *= m_input.speedTarget;

			//desiredMovement.NormalizeSafe();
	    
			SetDesiredSpeed(desiredMovement);        		
		}

		if (m_input.dirTarget.len2()>0.0f)
		{
			Vec3 desiredDir(m_input.dirTarget - GetEntity()->GetWorldPos());
			desiredDir.NormalizeSafe();

			SetDesiredDirection(desiredDir);
		}
		//

		bool client(IsClient());
		
		if (m_input.actions & ACTION_CROUCH)
			SetStance(STANCE_CROUCH);
		else if (m_input.actions & ACTION_PRONE)
			SetStance(STANCE_PRONE);
		else if (m_input.actions & ACTION_STEALTH)
			SetStance(STANCE_STEALTH);
		else
			SetStance(STANCE_STAND);

		//
		UpdateStats(frameTime);

		if (m_pMovementController)
		{
			SActorFrameMovementParams params;
			m_pMovementController->Update(frameTime, params);
		}

		//rotation processing
		ProcessRotation(frameTime);

		//movement processing
		ProcessMovement(frameTime);

		//animation processing
		ProcessAnimation(pCharacter,frameTime);

		//reset the input for the next frame
		if (client)
			m_input.ResetDeltas();

		//update tentacles blending
		Vec3 refVec(-m_viewMtx.GetColumn(1)*max(0.1f,m_params.forceView) + -m_desiredVelocity);
		refVec.NormalizeSafe();

		float directionDot = min(1.0f,fabs(refVec * m_baseMtx.GetColumn(0)) * 3.0f);
		float animStiff = 0.0f;

		if (m_params.blendingRatio>0.001f)
		{
			float ratio((GetStanceInfo(m_stance)->maxSpeed - m_stats.speedModule * directionDot) / GetStanceInfo(m_stance)->maxSpeed);
			Interpolate(m_tentacleBlendRatio,m_params.blendingRatio,20.0f,frameTime);
			animStiff = 1.0f + (ratio) * m_tentacleBlendRatio;
		}

		//SetTentacles(pCharacter,animStiff);
		//CryLogAlways("%.1f",animStiff);

		//update ground effects, if any
		if (m_pGroundEffect)
			m_pGroundEffect->Update();
	}
}

void CAlien::UpdateView(SViewParams &viewParams)
{
	viewParams.nearplane = 0.0f;
	viewParams.fov = 90.0f*gf_PI/180.0f;

	viewParams.position = GetEntity()->GetPos();

	Matrix33 viewMtx(m_viewMtx);
	
	if (m_stats.isThirdPerson)
	{
		//FIXME:dont hook up the cvar each frame, just store the Cvar *
		float thirdPersonDistance(GetISystem()->GetIConsole()->GetCVar("cl_tpvDist")->GetFVal());
		float thirdPersonYaw(GetISystem()->GetIConsole()->GetCVar("cl_tpvYaw")->GetFVal());

		if (thirdPersonYaw>0.001f)
			viewMtx *= Matrix33::CreateRotationZ(thirdPersonYaw * gf_PI/180.0f);

		viewParams.position += viewMtx.GetColumn(1) * -thirdPersonDistance;
	}

	viewParams.rotation = GetQuatFromMat33(viewMtx);
}

bool CAlien::IsZeroG()
{
	return m_stats.inZeroG;
}

void CAlien::UpdateStats(float frameTime)
{
	IPhysicalEntity *pPhysEnt = GetEntity()->GetPhysics();

	if (!pPhysEnt)
		return;

	//retrieve some information about the status of the player
	pe_status_dynamics dynStat;
	pe_status_living livStat;
	pe_player_dynamics simPar;

	Vec3 ZAxis;
	m_stats.inZeroG = CheckZeroG(ZAxis);
	m_stats.isFloating = false;//used only by trooper for now
	
	//FIXME: temporary
	simPar.bSwimming = IsZeroG() || IsFlying();
	pPhysEnt->SetParams(&simPar);
	//

	pPhysEnt->GetStatus(&dynStat);
	pPhysEnt->GetStatus(&livStat);
	pPhysEnt->GetParams(&simPar);

	//update status table
	if (livStat.bFlying)
	{
		m_stats.inAir += frameTime;
		m_stats.onGround = 0.0f;
	}
	else
	{
		m_stats.onGround += frameTime;
		m_stats.inAir = 0.0f;
	}
	
	m_stats.gravity = simPar.gravity;
	m_stats.velocity = dynStat.v;
	m_stats.speedModule = m_stats.velocity.len();

	m_stats.mass = dynStat.mass;

	//the alien is able to sprint for a bit right after being standing
	if (m_stats.speedModule > m_stats.sprintTreshold)
	{
		m_stats.sprintLeft = max(0.0f,m_stats.sprintLeft - frameTime);
		m_stats.sprintMaxSpeed = min(m_stats.speedModule, GetStanceInfo( m_stance )->maxSpeed );
	}
	else
	{
		// If the speed slowsdown to 80% of the last max speed, allow to sprint again.
		const float slowdownPercent = 0.8f;
		if( m_stats.speedModule < m_stats.sprintMaxSpeed * slowdownPercent )
			m_stats.sprintTreshold = m_stats.sprintMaxSpeed * slowdownPercent;
		m_stats.sprintLeft = m_params.sprintDuration;
	}

	//misc things
	Vec3 lookTarget(ZERO);//GetAIAttentionPos());
	if (lookTarget.len2()<0.01f)
		lookTarget = GetEntity()->GetSlotWorldTM(0) * GetLocalEyePos() + m_eyeMtx.GetColumn(1) * 10.0f;

	if (m_stats.lookTargetSmooth.len2()<0.01f)
		m_stats.lookTargetSmooth = lookTarget;
	else
		Interpolate(m_stats.lookTargetSmooth,lookTarget,3.0f,frameTime);

	//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(255,0,0,255), m_stats.lookTargetSmooth, ColorB(255,0,0,255));

	//
	UpdateFiringDir(frameTime);
	//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos() + Vec3(0,0,5), ColorB(255,0,0,255), GetEntity()->GetWorldPos() + Vec3(0,0,5) + m_stats.fireDir * 20.0f, ColorB(255,0,0,255));

	//update some timers
	m_stats.inFiring = max(0.0f,m_stats.inFiring - frameTime);

	if (m_endOfThePathTime > 0.0f)
		m_endOfThePathTime -= frameTime;	
}

void CAlien::ProcessRotation(float frameTime)
{
	IPhysicalEntity *pPhysEnt = GetEntity()->GetPhysics();

	if (!pPhysEnt)
		return;

	if (frameTime > 0.1f)
			frameTime = 0.1f;

	//rotation
	//6 degree of freedom
	//FIXME:put mouse sensitivity here!
	//TODO:use radians
	float rotSpeed(0.5f);

//	if (m_stats.inAir && IsZeroG())
	{		

		// Mikko: Separated the look and movement directions. This is a HACK! The reason is below (moved from the SetActorMovement):
		// >> Danny - old code had desired direction using vLookDir but this caused spinning behaviour
		// >> when it was significantly different to vMoveDir

		if (m_input.viewVector.len2()>0.0f)
		{
//			m_eyeMtx.SetRotationVDir(m_input.viewVector.GetNormalizedSafe());
			Matrix33	eyeTarget;
			eyeTarget.SetRotationVDir(m_input.viewVector.GetNormalizedSafe());
			Quat	eyeTargetQuat(eyeTarget);
			Quat	currQuat(m_eyeMtx);
			m_eyeMtx = Matrix33(Quat::CreateSlerp( currQuat.GetNormalized(), eyeTargetQuat, min(frameTime * 12.0f, 1.0f)));
		}

		if (m_input.viewVector.len2()>0.0f)
		{
			Vec3	lookat = m_eyeMtx.GetColumn(1);
			Vec3	orient = m_viewMtx.GetColumn(1);
			if( lookat.Dot( orient ) < cosf( DEG2RAD( 65.0f ) ) || m_forceOrient )
				m_viewMtx.SetRotationVDir(m_input.viewVector.GetNormalizedSafe());
		}
		else //if (m_input.deltaRotation.len2()>0.001f)
		{
			Ang3 desiredAngVel(m_input.deltaRotation.x * rotSpeed,0,m_input.deltaRotation.z * rotSpeed);
					
			//rollage
			if (m_input.actions & ACTION_LEANLEFT)
				desiredAngVel.y -= 10.0f * rotSpeed;
			if (m_input.actions & ACTION_LEANRIGHT)
				desiredAngVel.y += 10.0f * rotSpeed;

			Interpolate(m_angularVel,desiredAngVel,3.5f,frameTime);

			Matrix33 yawMtx;
			Matrix33 pitchMtx;
			Matrix33 rollMtx;

			//yaw
			yawMtx.SetRotationZ(m_angularVel.z * gf_PI/180.0f);
			//pitch
			pitchMtx.SetRotationX(m_angularVel.x * gf_PI/180.0f);
			//roll
			if (fabs(m_angularVel.y) > 0.001f)
				rollMtx.SetRotationY(m_angularVel.y * gf_PI/180.0f);
			else
				rollMtx.SetIdentity();
			//
			
			m_viewMtx = m_viewMtx * yawMtx * pitchMtx * rollMtx;
			m_viewMtx.NoScale();
			m_eyeMtx = m_viewMtx;
		}
	}
}

void CAlien::ProcessMovement(float frameTime)
{
	//FIXME:dont remove this yet
	//ProcessMovement2(frameTime);
	//return;

	IPhysicalEntity *pPhysEnt = GetEntity()->GetPhysics();

	if (!pPhysEnt)
		return;

	if (frameTime > 0.1f)
			frameTime = 0.1f;

	//movement
	Vec3 move(m_input.movementVector);//usually 0, except AIs

/*	if (m_input.actions & ACTION_JUMP)
		m_input.deltaMovement.z += 1.0f; 
	if (m_input.actions & ACTION_CROUCH)
		m_input.deltaMovement.z -= 1.0f;*/

	move += m_viewMtx.GetColumn(0) * m_input.deltaMovement.x;
	move += m_viewMtx.GetColumn(1) * m_input.deltaMovement.y;
	move += m_viewMtx.GetColumn(2) * m_input.deltaMovement.z;
	
	move += m_viewMtx.GetColumn(1) * m_params.approachLookat;

	//cap the movement vector to max 1
	float moveModule(move.len());

	if (moveModule > 1.0f)
		move /= moveModule;

	move *= GetStanceInfo( m_stance )->maxSpeed;

	if (m_stats.sprintLeft)
		move *= m_params.sprintMultiplier;

	//FIXME:testing
	Interpolate(m_stats.physicsAnimationRatio,min(moveModule*(pPhysicsAnimRatio?pPhysicsAnimRatio->GetFVal():0.0f),1.0f),3.3f,frameTime,1.0f);

	Matrix33 velMtx( m_baseMtx );
	Vec3 vecRefRoll( move );

	// A little bit more hacky, need to aling the alien to some up vector too.
	Vec3 up, right, forward;

	if (m_input.upTarget.len2()>0.0f)
	{
		Vec3	diff = m_input.upTarget - m_input.posTarget;
		diff.NormalizeSafe();
		up = diff;
	}
	else
		up = m_viewMtx.GetColumn(2);

	if( move.len2() > 0 )
	{
		forward = move.GetNormalizedSafe();
	}
	else
	{
		forward = m_viewMtx.GetColumn(1);

		if (!m_forceOrient)
		{
			float dot(forward * m_baseMtx.GetColumn(1));
			if (dot>0.2f)
				forward = m_baseMtx.GetColumn(1);
			else
				m_followEyesTime = 1.0f;
		}
	}

	if ((m_followEyesTime-=frameTime)>0.001)
		forward = m_viewMtx.GetColumn(1);

	right = (forward % up).GetNormalizedSafe();
	velMtx.SetFromVectors(right,forward,right % forward);

	//rollage
	if (m_input.upTarget.len2()>0.0f)
	{
		// No rollage, when the up vector is forced!
	}
	else
	{
		float	rollAmt = 0.0f;
		float dotRoll(vecRefRoll * m_baseMtx.GetColumn(0));
		if (fabs(dotRoll)>0.001f)
			rollAmt = max(min(gf_PI*0.49f,dotRoll*m_params.rollAmount),-gf_PI*0.49f);

		Interpolate(m_roll,rollAmt,m_params.rollSpeed,frameTime);

		velMtx *= Matrix33::CreateRotationY(m_roll);
	}

	m_desiredVeloctyQuat = GetQuatFromMat33(velMtx);
	float rotSpeed = m_params.rotSpeed_min + (1.0f - (max(GetStanceInfo( m_stance )->maxSpeed - max(m_stats.speedModule - m_params.speed_min,0.0f),0.0f) / GetStanceInfo( m_stance )->maxSpeed)) * (m_params.rotSpeed_max - m_params.rotSpeed_min);
	Interpolate(m_turnSpeed,rotSpeed,3.0f,frameTime);

	float turnSlowDown = min(1.0f,max(0.0f,1.0f-m_followEyesTime));
	Quat currQuat(m_baseMtx);
	m_baseMtx = Matrix33(Quat::CreateSlerp( currQuat.GetNormalized(), m_desiredVeloctyQuat, min(frameTime * m_turnSpeed * turnSlowDown, 1.0f)));
	m_baseMtx.NoScale();


	Vec3 desiredVel;

	//a bit hacky: needed when the alien is forced to move in some position
/*	if (m_input.posTarget.len2()>0.0f)
		move = m_desiredVelocity;
	else*/
	{
		// Cap the speed to the ideal speed.
		float	moveLen( move.len() );
		if( moveLen > 0 && m_params.idealSpeed >= 0 && moveLen > m_params.idealSpeed )
			move *= m_params.idealSpeed / moveLen;
	}

	//counter gravity
	//move.z += m_velocity.z - m_stats.velocity.z;

	// make sure alien reaches destination, ignore speedInertia
	if (m_input.posTarget.len2()>0.0f)
		m_velocity = move;
	else
		Interpolate(m_velocity,move,m_params.speedInertia,frameTime);

	// Slow down if the forward direction differs from the movement direction.
	// A side effect of this is that the velocity after turning will be higher (=good).
	float speed = m_velocity.len();
	float	velScale( speed );

	if(velScale > 0.0001f)
	{
		Vec3	move = m_velocity / velScale;
		Vec3	forw = -GetEntity()->GetRotation().GetColumn1(); //m_viewMtx.GetColumn(1);

		float	dot = forw.Dot( move );

		const float treshold = cosf(DEG2RAD(15.0));
		if( dot > treshold )
			velScale = 1.0;
		else
		{
			if( dot < 0 ) dot = 0;
			velScale = (dot / treshold) * 0.99f + 0.01f;
		}
	}

	// Accelerate faster than slowdown.
	float	target = speed * velScale;
	float	s = 5.0f;
//	if( target > m_curSpeed || bExactPositioning )
	if( m_input.posTarget.len2()>0.0f )
		s *= 2.0f;
	Interpolate( m_curSpeed, target, s, frameTime );

	if(speed > 0.0001f)
		velScale *= m_curSpeed / speed;

	SCharacterMoveRequest moveRequest;
	moveRequest.movement = m_velocity * velScale;
	moveRequest.type = eCMT_Fly;

	//FIXME:sometime
	m_stats.desiredSpeed = m_stats.speedModule;

	if (m_pAnimatedCharacter)
		m_pAnimatedCharacter->AddMovement( moveRequest );
}

void CAlien::ProcessMovement2(float frameTime)
{
	IPhysicalEntity *pPhysEnt = GetEntity()->GetPhysics();

	if (!pPhysEnt)
		return;

	if (frameTime > 0.1f)
			frameTime = 0.1f;

	//movement
	Vec3 move(m_input.movementVector);

	move += m_viewMtx.GetColumn(0) * m_input.deltaMovement.x;
	move += m_viewMtx.GetColumn(1) * m_input.deltaMovement.y;
	move += m_viewMtx.GetColumn(2) * m_input.deltaMovement.z;
	
	move += m_viewMtx.GetColumn(1) * m_params.approachLookat;

	//cap the movement vector to max 1
	float moveModule(move.len());

	if (moveModule > 1.0f)
		move /= moveModule;

	move *= GetStanceInfo( m_stance )->maxSpeed;

	if (m_stats.sprintLeft)
		move *= m_params.sprintMultiplier;

	//FIXME:testing
	Interpolate(m_stats.physicsAnimationRatio,min(moveModule*(pPhysicsAnimRatio?pPhysicsAnimRatio->GetFVal():0.0f),1.0f),3.3f,frameTime,1.0f);
		
	float color[] = {1,1,1,0.5f};
	GetISystem()->GetIRenderer()->Draw2dLabel(100,100,2,color,false,"moveModule:%f,physicsAnimationRatio:%f",moveModule,m_stats.physicsAnimationRatio);
	//

	Matrix33 velMtx;
	Vec3 vecRefRoll;

	Vec3 tempVel;
	//a bit hacky: needed when the alien is forced to look in some direction
	if (m_input.dirTarget.len2()>0.0f)
		tempVel = m_viewMtx.GetColumn(1);
	else
		tempVel = m_viewMtx.GetColumn(1)*max(0.1f,m_params.forceView) + m_stats.velocity;//move;

	// A little bit more hacky, need to aling the alien to some up vector too.
	Vec3 up;
	if (m_input.upTarget.len2()>0.0f)
	{
		Vec3	diff = m_input.upTarget - m_input.posTarget;
		diff.NormalizeSafe();
		up = diff;
	}
	else
		up = m_viewMtx.GetColumn(2);

	Vec3 forward = tempVel.GetNormalized();
	Vec3 right = (forward % up).GetNormalized();

	velMtx.SetFromVectors(right,forward,right % forward);
	vecRefRoll = tempVel;

	/*GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(255,0,0,255), GetEntity()->GetWorldPos() + velMtx.GetColumn(0), ColorB(255,0,0,255));
	GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(0,255,0,255), GetEntity()->GetWorldPos() + velMtx.GetColumn(1), ColorB(0,255,0,255));
	GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(0,0,255,255), GetEntity()->GetWorldPos() + velMtx.GetColumn(2), ColorB(0,0,255,255));
	*/

	//rollage
	if (m_input.upTarget.len2()>0.0f)
	{
		// No rollage, when the up vector is forced!
	}
	else
	{
		float	rollAmt = 0.0f;
		float dotRoll(vecRefRoll * m_baseMtx.GetColumn(0));
		if (fabs(dotRoll)>0.001f)
			rollAmt = max(min(gf_PI*0.49f,dotRoll*m_params.rollAmount),-gf_PI*0.49f);

		Interpolate(m_roll,rollAmt,m_params.rollSpeed,frameTime);

		velMtx *= Matrix33::CreateRotationY(m_roll);
	}

	m_desiredVeloctyQuat = GetQuatFromMat33(velMtx);
	float rotSpeed = m_params.rotSpeed_min + (1.0f - (max(GetStanceInfo( m_stance )->maxSpeed - max(m_stats.speedModule - m_params.speed_min,0.0f),0.0f) / GetStanceInfo( m_stance )->maxSpeed)) * (m_params.rotSpeed_max - m_params.rotSpeed_min);
	Interpolate(m_turnSpeed,rotSpeed,3.0f,frameTime);

	Quat currQuat(m_baseMtx);
	m_baseMtx = Matrix33(Quat::CreateSlerp( currQuat.GetNormalized(), m_desiredVeloctyQuat, min(frameTime * m_turnSpeed, 1.0f)));
	m_baseMtx.NoScale();

	//a bit hacky: needed when the alien is forced to move in some position
/*	if (m_input.posTarget.len2()>0.0f)
		move = m_desiredVelocity;
	else*/
	{
		// Cap the speed to the ideal speed.
		float	moveLen( move.len() );
		if( moveLen > 0 && m_params.idealSpeed >= 0 && moveLen > m_params.idealSpeed )
			move *= m_params.idealSpeed / moveLen;
	}

	// make sure alien reaches destination, ignore speedInertia
	if (m_input.posTarget.len2()>0.0f)
		m_velocity = move;
	else
		Interpolate(m_velocity,move,m_params.speedInertia,frameTime);

	// Slow down if the forward direction differs from the movement direction.
	// A side effect of this is that the velocity after turning will be higher (=good).
	float speed = m_velocity.len();
	float	velScale( speed );
	if(velScale > 0 )
	{
		Vec3	move = m_velocity / velScale;
		Vec3	forw = -GetEntity()->GetRotation().GetColumn1(); //m_viewMtx.GetColumn(1);
		float	dot = forw.Dot( move );
		const float treshold = cosf(DEG2RAD(15.0));
		if( dot > treshold )
			velScale = 1.0;
		else
		{
			if( dot < 0 ) dot = 0;
			velScale = (dot / treshold) * 0.99f + 0.01f;
		}
	}

	// Accelerate faster than slowdown.
	float	target = speed * velScale;
	float	s = 5.0f;
//	if( target > m_curSpeed || bExactPositioning )
	if( m_input.posTarget.len2()>0.0f )
		s *= 2.0f;
	Interpolate( m_curSpeed, target, s, frameTime );

	if( speed > 0 )
		velScale *= m_curSpeed / speed;

	pe_action_move actionMove;
	//FIXME:wip
	actionMove.dir = m_velocity + m_stats.animationSpeedVec;// * velScale;
	actionMove.iJump = 3;

	//FIXME:sometime
	m_stats.desiredSpeed = m_stats.speedModule;
//	m_stats.xDelta = m_stats.zDelta = 0.0f;

	pPhysEnt->Action(&actionMove);
}

void CAlien::ProcessAnimation(ICharacterInstance *pCharacter,float frameTime)
{
	ISkeleton *pSkeleton = pCharacter ? pCharacter->GetISkeleton() : NULL;
	if (pSkeleton)
	{
		static const float customBlend[5] = { 0.04f, 0.06f, 0.08f, 0.6f, 0.6f };
		//update look ik	
		pSkeleton->SetLookIK(true,gf_PI*0.9f,m_stats.lookTargetSmooth,customBlend);
	}

	//update the model rotation
	Quat modelRot(m_baseMtx);
	modelRot *= Quat::CreateRotationZ(gf_PI);
	modelRot = Quat::CreateSlerp(GetEntity()->GetRotation().GetNormalized(), modelRot, min(frameTime * 6.6f/*m_turnSpeed*/ /** (m_stats.speedModule/GetStanceInfo(m_stance)->maxSpeed)*/, 1.0f));

	if (m_pAnimatedCharacter)
	{
		SCharacterMoveRequest moveRequest;
		moveRequest.type = eCMT_JustRotation;
		//FIXME:remove the * rotationZ
		moveRequest.turn = GetEntity()->GetRotation().GetInverted() * modelRot * Quat::CreateRotationZ(gf_PI);

		m_pAnimatedCharacter->AddMovement(moveRequest);
	}
}

void CAlien::ProcessBonesRotation(ICharacterInstance *pCharacter,float frameTime)
{
	CActor::ProcessBonesRotation(pCharacter,frameTime);

	//FIXME:testing
	if (pCharacter)
	{
		if (m_stats.physicsAnimationRatio>0.001f)
		{
			int32 idx = pCharacter->GetISkeleton()->GetIDByName("root");
			if (idx>-1)
			{
				Vec3 rootPos(pCharacter->GetISkeleton()->GetAbsJPositionByID(idx)*m_stats.physicsAnimationRatio);
				m_stats.animationSpeedVec = Matrix33(GetEntity()->GetSlotWorldTM(0)) * ((rootPos - m_stats.lastRootPos) / frameTime);
				m_stats.lastRootPos = rootPos;

				m_charLocalMtx.SetTranslation(m_charLocalMtx.GetTranslation()-rootPos);
				GetEntity()->SetSlotLocalTM(0,m_charLocalMtx);
			}
		}
		else
		{
			m_stats.animationSpeedVec.zero();
			m_stats.lastRootPos.zero();
		}
	}
	//

	return;

	//flat desired view direction
	Matrix33 modelMtx(GetEntity()->GetRotation()/* * Matrix33::CreateRotationZ(-gf_PI * 0.5)*/);

	Vec3 viewFlat(m_viewMtx.GetColumn(1) - m_viewMtx.GetColumn(1) * (modelMtx * Matrix33::CreateScale(Vec3(0,0,1))));
	viewFlat.NormalizeSafe();

	float dotForward(viewFlat * modelMtx.GetColumn(1));
	float dotSide(viewFlat * modelMtx.GetColumn(0));
	float yawDiff(cry_atan2f(-dotSide,-dotForward));

	float pitchDiff(0);
	/*modelMtx = modelMtx * Matrix33::CreateRotationZ(yawDiff);

	dotForward = m_viewMtx.GetColumn(1) * modelMtx.GetColumn(1);
	float dotUp(m_viewMtx.GetColumn(1) * modelMtx.GetColumn(2));
	float pitchDiff(cry_atan2f(dotUp,-dotForward));*/

	//CryLogAlways("y:%.1f | p:%.1f",RAD2DEG(yawDiff),RAD2DEG(pitchDiff));

	if (yawDiff>0.52f)
		yawDiff = 0.52f;

	if (yawDiff<-0.52f)
		yawDiff = -0.52f;

	if (pitchDiff>1.04f)
		pitchDiff = 1.04f;

	if (pitchDiff<-1.04f)
		pitchDiff = -1.04f;

	//IJoint *pBones[2];
	//pBones[0] = pCharacter->GetISkeleton()->GetIJointByName("head");
	//pBones[1] = pCharacter->GetISkeleton()->GetIJointByName("neck");

	int16 id[2];
	id[0] = pCharacter->GetISkeleton()->GetIDByName("head");
	id[1] = pCharacter->GetISkeleton()->GetIDByName("neck");

	pitchDiff /= 2.0f;
	yawDiff /= 2.0f;

	Quat qtH;
	Quat qtV;
	Quat qtR;
	Quat qtTotal;
	Quat qtParent;
	Quat qtParentCnj;

	for (int i=0;i<2;++i)
	{
		if (id[i])
		{
			qtH.SetRotationAA( yawDiff, Vec3(0.0f, 0.0f, 1.0f) );//yaw
			qtV.SetRotationAA( pitchDiff, Vec3(1.0f, 0.0f, 0.0f) );//pitch
			qtR.SetRotationAA( 0.0f,  Vec3(0.0f, 1.0f, 0.0f) );//roll
			
		//	IJoint* pIJoint = pBones[i]->GetParent();
			int16 parentID = pCharacter->GetISkeleton()->GetParentIDByID(id[i]);
			Quat wquat(IDENTITY);
			if (parentID>=0)
				//wquat=!Quat(pIJoint->GetAbsoluteMatrix());
				wquat=!Quat(pCharacter->GetISkeleton()->GetAbsJMatrixByID(parentID));

			qtParent = wquat;//pBones[i]->GetParentWQuat();
			qtParentCnj = qtParent;
			qtParentCnj.w = -qtParentCnj.w;
			qtTotal = qtParent*qtR*qtV*qtH*qtParentCnj;

		//	pBones[i]->SetPlusRotation( qtTotal );
			//pCharacter->GetISkeleton()->SetPlusRotation( id[i], qtTotal );
		}
	}
}

//FIXME:tentacle testing
void CAlien::SetTentacles(ICharacterInstance *pCharacter,float animStiffness,float mass,float damping,bool bRagdolize)
{
	//TODO:use the correct number, not an hardcoded "8", and make it faster by holding pointers and such.
	pe_params_rope pRope;
	pe_simulation_params sp;
	pe_action_target_vtx atv;
	pe_params_flags pf;

	pRope.stiffnessAnim = animStiffness;
	
	float jlim = GetISystem()->GetIConsole()->GetCVar("g_tentacle_joint_limit")->GetFVal();
	if (jlim>=0)
		pRope.jointLimit = DEG2RAD(jlim);
	else
		pRope.jointLimit = DEG2RAD(m_params.tentaclesJointLimit);

	//pRope.stiffnessDecayAnim = 10.1f;
	if (bRagdolize)
	{
		pRope.bTargetPoseActive=2, pRope.collDist=m_params.tentaclesRadius;
		pf.flagsOR = rope_target_vtx_rel0|rope_no_stiffness_when_colliding;
		pf.flagsAND = ~(rope_findiff_attached_vel | rope_no_solver);
		sp.minEnergy = sqr(0.03f);
	}

	if (damping > 0.001f)
		sp.damping = damping;

	if (mass > 0.001f)
		pRope.mass = mass;

	std::vector<IPhysicalEntity *>::iterator it;
	for (it = m_tentaclesProxy.begin(); it != m_tentaclesProxy.end(); it++)
	{
		IPhysicalEntity *pT = *it;
		if (pT)
		{
			pT->SetParams(&pRope);
			pT->SetParams(&sp);
			pT->SetParams(&pf);
			pT->Action(&atv);
		}
	}

	if (m_params.fullAnimationTentaclesBlendMult>0.001)
		pRope.stiffnessAnim = animStiffness * m_params.fullAnimationTentaclesBlendMult;
	else
		pRope.stiffnessAnim = 0;

	for (it = m_tentaclesProxyFullAnimation.begin(); it != m_tentaclesProxyFullAnimation.end(); it++)
	{
		IPhysicalEntity *pT = *it;
		if (pT)
			pT->SetParams(&pRope);
	}
}

void CAlien::Draw(bool draw)
{
	uint32 slotFlags = GetEntity()->GetSlotFlags(0);

	if (draw)
		slotFlags |= ENTITY_SLOT_RENDER;
	else
		slotFlags &= ~ENTITY_SLOT_RENDER;

	GetEntity()->SetSlotFlags(0,slotFlags);
}

void CAlien::ResetAnimations()
{
	ICharacterInstance *character = GetEntity()->GetCharacter(0);

	if (character)
	{
		if (m_pAnimatedCharacter)
		{
			m_pAnimatedCharacter->ClearForcedStates();
			//m_pAnimatedCharacter->GetAnimationGraphState()->Pause(true, eAGP_StartGame);
		}

		character->GetISkeleton()->StopAnimationsAllLayers();
		character->GetISkeleton()->SetLookIK(false,gf_PI*0.9f,m_stats.lookTargetSmooth);
	}
}

void CAlien::Kill()
{
	CActor::Kill();

	ResetAnimations();

	//FIXME:this stops the ground effect, maybe its better to directly remove the effect instead?
	if (m_pGroundEffect)
		m_pGroundEffect->Stop(true);
}

void CAlien::Revive(bool fromInit)
{
	CActor::Revive(fromInit);

	m_stats = SAlienStats();
	
	m_modelOffset = Vec3(0,0,0);
	m_modelOffsetAdd = Vec3(0,0,0);
	m_velocity = Vec3(0,0,0);
	m_desiredVelocity = Vec3(0,0,0);

	m_turnSpeed = 0.0f;
	m_turnSpeedGoal = 0.0f;
	m_roll = 0.0f;

	m_curSpeed = 0;

	m_endOfThePathTime = -1.0f;

	m_tentacleBlendRatio = 1.0f;

	m_baseMtx = Matrix33(GetEntity()->GetRotation() * Quat::CreateRotationZ(gf_PI));
	m_modelQuat = GetEntity()->GetRotation();
	SetDesiredDirection(-GetEntity()->GetRotation().GetColumn1());

	m_angularVel = Ang3(0,0,0);

	m_charLocalMtx.SetIdentity();

	m_isFiring = false;

  m_input.posTarget.zero();
  m_input.dirTarget.zero();

	m_forceOrient = false;

	ResetAnimations();

	//initialize the ground effect
	if (m_params.groundEffect[0])
	{
		if (!m_pGroundEffect)
			m_pGroundEffect = g_pGame->GetIGameFramework()->GetIEffectSystem()->CreateGroundEffect(GetEntity());

		if (m_pGroundEffect)
		{
			m_pGroundEffect->SetParticleEffect(m_params.groundEffect);
			m_pGroundEffect->SetHeight(m_params.groundEffectHeight);
			m_pGroundEffect->SetFlags(m_pGroundEffect->GetFlags() | IGroundEffect::eGEF_StickOnGround);
		}
	}

	{
		// hackity hack hack hack
		SAnimatedCharacterParams params = m_pAnimatedCharacter->GetParams();
		params.flags |= eACF_NoTransRot2k;
		m_pAnimatedCharacter->SetParams(params);
	}
}

void CAlien::RagDollize()
{
	FUNCTION_PROFILER(GetISystem(), PROFILE_GAME);

	ResetAnimations();

	CActor::RagDollize();

	IPhysicalEntity *pPhysEnt = GetEntity()->GetPhysics();

	if (pPhysEnt)
	{
		pe_params_flags flags;
		flags.flagsOR = pef_log_collisions;
		pPhysEnt->SetParams(&flags);

		pe_simulation_params sp;
		sp.damping = 1.0f;
		sp.dampingFreefall = 0.0f;
		sp.mass = m_stats.mass;
		pPhysEnt->SetParams(&sp);

		pe_params_articulated_body pa;
		pa.dampingLyingMode = 5.5f;
		//pa.scaleBounceResponse = 0.1f;
		pPhysEnt->SetParams(&pa);
	}

	ICharacterInstance *pCharacter = GetEntity()->GetCharacter(0);	

	if (pCharacter)
	{
		pCharacter->EnableStartAnimation(false);
		SetTentacles(pCharacter,8.5f,0,2.25f,true);
	}
}

//this will convert the input into a structure the player will use to process movement & input
void CAlien::OnAction(const ActionId& actionId, int activationMode, float value)
{
	GetGameObject()->ChangedNetworkState( eEA_GameServerStatic | eEA_GameServerDynamic | eEA_GameClientStatic | eEA_GameClientDynamic );

	//this tell if OnAction have to be forwarded to scripts
	bool filterOut(true);
	const SGameActions& actions = g_pGame->Actions();

	if (actions.rotateyaw == actionId)
	{
		m_input.deltaRotation.z -= value;
		filterOut = false;
	}
	else if (actions.rotatepitch == actionId)
	{
		m_input.deltaRotation.x -= value;
		filterOut = false;
	}
	else if (actions.moveright == actionId)
	{
		m_input.deltaMovement.x += value;
		filterOut = false;
	}
	else if (actions.moveleft == actionId)
	{
		m_input.deltaMovement.x -= value;
		filterOut = false;
	}
	else if (actions.moveforward == actionId)
	{
		m_input.deltaMovement.y += value;
		filterOut = false;
	}
	else if (actions.moveback == actionId)
	{
		m_input.deltaMovement.y -= value;
		filterOut = false;
	}
	else if (actions.jump == actionId)
	{
		m_input.actions |= ACTION_JUMP;		
	}
	else if (actions.crouch == actionId)
	{
		m_input.actions |= ACTION_CROUCH;
	}
	else if (actions.prone == actionId)
	{
		if (!(m_input.actions & ACTION_PRONE))
			m_input.actions |= ACTION_PRONE;
		else
			m_input.actions &= ~ACTION_PRONE;
	}
	else if (actions.sprint == actionId)
	{
		m_input.actions |= ACTION_SPRINT;
	}
	else if (actions.leanleft == actionId)
	{
		m_input.actions |= ACTION_LEANLEFT;
	}
	else if (actions.leanright == actionId)
	{
		m_input.actions |= ACTION_LEANRIGHT;
	}
	else if (actions.thirdperson == actionId)
	{ 
		m_stats.isThirdPerson = !m_stats.isThirdPerson;
	}

	//FIXME: this is duplicated from CPlayer.cpp, put it just once in CActor.
	//send the onAction to scripts, after filter the range of actions. for now just use and hold
	if (filterOut)
	{
		HSCRIPTFUNCTION scriptOnAction(NULL);

		IScriptTable *scriptTbl = GetEntity()->GetScriptTable();

		if (scriptTbl)
		{
			scriptTbl->GetValue("OnAction", scriptOnAction);

			if (scriptOnAction)
			{
				char *activation = 0;

				switch(activationMode)
				{
				case eAAM_OnHold:
					activation = "hold";
					break;
				case eAAM_OnPress:
					activation = "press";
					break;
				case eAAM_OnRelease:
					activation = "release";
					break;
				default:
					activation = "";
					break;
				}

				Script::Call(GetISystem()->GetIScriptSystem(),scriptOnAction,scriptTbl,actionId.c_str(),activation, value);
			}
		}
	}

	CActor::OnAction(actionId, activationMode, value);
}

void CAlien::SetStats(SmartScriptTable &rTable)
{
	CActor::SetStats(rTable);
}

//fill the status table for the scripts
bool CAlien::GetStats(SmartScriptTable &rTable)
{
	rTable->SetValue("inAir",m_stats.inAir);
	rTable->SetValue("onGround",m_stats.onGround);
	rTable->SetValue("speedModule",m_stats.speedModule);
	rTable->SetValue("thirdPerson",m_stats.isThirdPerson);
	rTable->SetValue("velocity",m_stats.velocity);
	rTable->SetValue("inFiring",m_stats.inFiring);

	switch(m_stance)
	{
	case STANCE_STAND: rTable->SetValue("stance","stand"); break;
	case STANCE_PRONE: rTable->SetValue("stance","prone"); break;
	case STANCE_CROUCH: rTable->SetValue("stance","crouch"); break;
	case STANCE_RELAXED: rTable->SetValue("stance","relaxed"); break;
	case STANCE_STEALTH: rTable->SetValue("stance","stealth"); break;
	}
	
	return true;
}

void CAlien::SetParams(SmartScriptTable &rTable,bool resetFirst)
{
	FUNCTION_PROFILER(GetISystem(), PROFILE_GAME);

	//not sure about this
	if (resetFirst)
	{
		m_params = SAlienParams();
	}

	CActor::SetParams(rTable,resetFirst);

	rTable->GetValue("speedInertia",m_params.speedInertia);
	rTable->GetValue("rollAmount",m_params.rollAmount);
	rTable->GetValue("rollSpeed",m_params.rollSpeed);

	rTable->GetValue("sprintMultiplier",m_params.sprintMultiplier);
	rTable->GetValue("sprintDuration",m_params.sprintDuration);
			
	rTable->GetValue("rotSpeed_min",m_params.rotSpeed_min);
	rTable->GetValue("rotSpeed_max",m_params.rotSpeed_max);

	rTable->GetValue("speed_min",m_params.speed_min);
	rTable->GetValue("movingBend",m_params.movingBend);

	rTable->GetValue("idealSpeed",m_params.idealSpeed);
	rTable->GetValue("blendingRatio",m_params.blendingRatio);
	rTable->GetValue("approachLookat",m_params.approachLookat);

	rTable->GetValue("forceView",m_params.forceView);

	const char *str;
	if (rTable->GetValue("fullAnimationTentacles",str))
		strncpy(m_params.fullAnimTentacles,str,256);
	else
		m_params.fullAnimTentacles[0] = 0;

	rTable->GetValue("fullAnimationTentaclesBlendMult",m_params.fullAnimationTentaclesBlendMult);

	int tentaclesCollide(0);
	if (rTable->GetValue("tentaclesCollide",tentaclesCollide))
		m_params.tentaclesCollide = tentaclesCollide;

	//
	rTable->GetValue("jumpTo",m_params.jumpTo);

	//
	rTable->GetValue("tentaclesRadius",m_params.tentaclesRadius);
	rTable->GetValue("tentaclesJointLimit",m_params.tentaclesJointLimit);
	if (rTable->GetValue("tentaclesMaterial",str))
	{
		strncpy(m_params.tentaclesMaterial,str,64);
		m_params.tentaclesMaterial[63] = 0;
	}

	//
	rTable->GetValue("cameraShakeRange",m_params.cameraShakeRange);
	rTable->GetValue("cameraShakeMultiplier",m_params.cameraShakeMultiplier);

	//
	if (rTable->GetValue("groundEffect",str))
	{
		strncpy(m_params.groundEffect,str,128);
		m_params.groundEffect[127] = 0;
	}
	else
		m_params.groundEffect[0] = 0;

	rTable->GetValue("groundEffectHeight",m_params.groundEffectHeight);
}

void CAlien::SetDesiredSpeed(const Vec3 &desiredSpeed)
{
	m_input.movementVector = desiredSpeed;
}

void CAlien::SetDesiredDirection(const Vec3 &desiredDir)
{
	if (desiredDir.len2()>0.001f)
	{
		m_viewMtx.SetRotationVDir(desiredDir.GetNormalizedSafe());
		m_eyeMtx = m_viewMtx;
	}

	//m_input.viewVector = desiredDir;
}

// common functionality can go in here and called from subclasses that override SetActorMovement
void CAlien::SetActorMovementCommon(SOBJECTSTATE& control)
{
  // added bExactPos property for easier testing
  SmartScriptTable props;
  int bExactPos = 0;  
  GetEntity()->GetScriptTable()->GetValue("Properties", props);
  props->GetValue("bExactPos", bExactPos);

  int nPoints = control.remainingPath.size();
  bool exactActive = false;

	control.bGameInPositionReached = false;
	if (control.bExactPositioning || bExactPos) 
	{
		// activate exact positioning    
		// todo: determine a better threshold
		if (nPoints && (nPoints == 1 || GetEntity()->GetWorldPos().GetSquaredDistance( control.remainingPath[0].vPos ) < 2.5f))
		{      
			// todo: determine a better threshold
			float dist((GetEntity()->GetWorldPos() - control.remainingPath[0].vPos).GetLengthSquared());
			if(dist<.3f*.3f)
				// tell AI the position is reached, approach/trace should be finished
				control.bGameInPositionReached = true;
			m_input.posTarget = control.remainingPath[0].vPos;
			m_input.dirTarget = control.remainingPath[0].vDir;
			exactActive = true;
		}    
	}
  
  if (!exactActive)
  {
    m_input.posTarget.zero();
    m_input.dirTarget.zero();
  }

	m_stats.fireDirGoal = control.vFireDir.GetNormalizedSafe();

	if (m_pAnimatedCharacter)
		m_pAnimatedCharacter->GetAnimationGraphState()->SetInput( m_inputAiming, control.aimLook? 1 : 0 );

  // draw pathpoints
  /*if (nPoints)
  { 
    IRenderAuxGeom *pGeom = GetISystem()->GetIRenderer()->GetIRenderAuxGeom();
    float size = 0.25f;

    for (PATHPOINTVECTOR::iterator it = control.path.begin(); it != control.path.end(); ++it)
    {        
      pGeom->DrawSphere((*it).vPos, size, ColorB(0,255,0,128));
      size += 0.1f;
    }
  }*/
}

//---------------------------------
//AI Specific
void CAlien::SetActorMovement(SOBJECTSTATE &control)
{
	//if (IsClient())
	//	return;

  SetActorMovementCommon(control);

	SetDesiredDirection(control.vLookDir);
	SetDesiredSpeed(control.vMoveDir * (0.15f + (float)pow( control.fDesiredSpeed * control.fDesiredSpeedMod, 2 ) * 0.85f));

//	m_input.actions = control.m_desiredActions;
	int actions;

	switch(control.bodystate)
	{
	case 1:
		actions = ACTION_CROUCH;
		break;
	case 2:
		actions = ACTION_PRONE;
		break;
	case 3:
		actions = ACTION_RELAXED;
		break;
	case 4:
		actions = ACTION_STEALTH;
		break;
	default:
		actions = 0;
		break;
	}

	// When firing, force orientation towards the target.
	if( control.vFireDir.len2() > 0.01f )
		m_forceOrient = true;
	else
		m_forceOrient = false;

	// Override the stance based on special behavior.
	SetActorStance(control, actions);

	m_input.actions = actions;


//	GetEntity()->GetScriptTable()->SetValue( "fireDir", control.vFireDir );

//	CryLog( "mv: (%f, %f, %f)", m_input.movementVector.x, m_input.movementVector.y, m_input.movementVector.z );
//	CryLog( "desiredSpeed: %f (mv.len=%f)", control.m_desiredSpeed, control.m_movementVector.GetLength() );

	//SetFiring(control.fire);
}

void CAlien::SetFiring(bool fire)
{
	//weapons
	//FIXME:remove this weapon specific ASAP
	if (fire)
	{
		if (!m_stats.isFiring)
			OnAction("attack1", eAAM_OnPress, 1.0f);
	}
	else
	{
		if (m_stats.isFiring)
			OnAction("attack1", eAAM_OnRelease, 1.0f);
	}

	m_stats.isFiring = fire;
}

void CAlien::SetActorStance(SOBJECTSTATE &control, int& actions)
{
	bool	atEndOfPath(false);
	float	curSpeed(m_velocity.len());
	if (curSpeed < 0.1f)
		curSpeed = 0.1f;

	float	timeEstimate = 0.25f;	// if we are the point in timeEstime seconds, start to go vert already.

//	if (GetStance() == STANCE_STEALTH || (actions & ACTION_STEALTH))
//		timeEstimate *= 1.0f; //0.5f;

	if (GetStance() == STANCE_PRONE && control.fDistanceToPathEnd > 0 && control.fDistanceToPathEnd < curSpeed * timeEstimate )
	{
		if (m_endOfThePathTime < 0.0f)
			m_endOfThePathTime = timeEstimate;
//		atEndOfPath = true;
		m_forceOrient = true;
	}

	if (m_endOfThePathTime > 0.0f)
	{
		atEndOfPath = true;
		m_forceOrient = true;
	}

	// Set stance based on the desired speed.
	if( !atEndOfPath && control.vMoveDir.GetLengthSquared() > 0.1f && control.fDesiredSpeed > 0.001f )
	{
//		CryLog( "ACTION_PRONE %s modeDirLen=%f desiredSpeed=%f m_endOfThePathTime=%f dist=%f", atEndOfPath ? "atEndOfPath" : "", control.vMoveDir.GetLengthSquared(), control.fDesiredSpeed, m_endOfThePathTime, control.fDistanceToPathEnd);
		actions = ACTION_PRONE;
	}
	else
	{
		if (actions == ACTION_STEALTH)
		{
//			CryLog( "ACTION_STEALTH %s modeDirLen=%f desiredSpeed=%f m_endOfThePathTime=%f dist=%f", atEndOfPath ? "atEndOfPath" : "", control.vMoveDir.GetLengthSquared(), control.fDesiredSpeed, m_endOfThePathTime, control.fDistanceToPathEnd);
			actions = ACTION_STEALTH;
		}
		else if (actions == ACTION_PRONE)
		{
			actions = ACTION_PRONE;
		}
		else
		{
//			CryLog( "ACTION_STAND %s modeDirLen=%f desiredSpeed=%f m_endOfThePathTime=%f dist=%f", atEndOfPath ? "atEndOfPath" : "", control.vMoveDir.GetLengthSquared(), control.fDesiredSpeed, m_endOfThePathTime, control.fDistanceToPathEnd);
			actions = 0;
		}
	}
}


void CAlien::GetActorInfo(SAIBodyInfo& bodyInfo)
{
	Vec3 lEyePos(GetLocalEyePos());
	bodyInfo.vEyePos = GetEntity()->GetSlotWorldTM(0) * lEyePos;

	int headBoneID = GetBoneID(BONE_HEAD);
	if (headBoneID>-1 && GetEntity()->GetCharacter(0))
	{
		Matrix33 HeadMat(GetEntity()->GetCharacter(0)->GetISkeleton()->GetAbsJMatrixByID(headBoneID));
		bodyInfo.vEyeDir = Matrix33(GetEntity()->GetSlotWorldTM(0) * HeadMat).GetColumn(1);
	}
	else
	{
		bodyInfo.vEyeDir = m_viewMtx.GetColumn(1);//m_eyeMtx.GetColumn(1);
	}

	//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(bodyInfo.vEyePos, ColorB(0,255,0,100), bodyInfo.vEyePos + bodyInfo.vEyeDir * 10.0f, ColorB(255,0,0,100));

	bodyInfo.vFwdDir = -GetEntity()->GetRotation().GetColumn1();//m_viewMtx.GetColumn(1);
	bodyInfo.vUpDir = m_viewMtx.GetColumn(2);
	bodyInfo.vFireDir = m_stats.fireDir;
	//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(0,255,0,100), GetEntity()->GetWorldPos() + bodyInfo.vFwdDir * 10, ColorB(255,0,0,100));

	const SStanceInfo * pStanceInfo = GetStanceInfo(m_stance);
	bodyInfo.minSpeed = min(m_params.speed_min, pStanceInfo->maxSpeed*0.01f);
	bodyInfo.normalSpeed = pStanceInfo->maxSpeed*0.5f;
	bodyInfo.maxSpeed = pStanceInfo->maxSpeed;
// if you want to change the .eyeHeight for aliens to something different from 0 - talk to Kirill. It might cause problems in
// CPuppet::AddToVisibleList when calculating time-out increase using target's eyeHeight.
	bodyInfo.eyeHeight = lEyePos.z;
	bodyInfo.stance = m_stance;
}

void CAlien::SetAngles(const Ang3 &angles) 
{
	Matrix33 rot(Matrix33::CreateRotationXYZ(angles));
	SetDesiredDirection(rot.GetColumn(1));
}

Ang3 CAlien::GetAngles() 
{
	Ang3 angles;
	angles.SetAnglesXYZ(m_viewMtx);

	return angles;
}

void CAlien::StanceChanged(EStance last)
{
	float delta(GetStanceInfo(last)->modelOffset.z - GetStanceInfo(m_stance)->modelOffset.z);
	if (delta>0.0f)
		m_modelOffset.z -= delta;
}

void CAlien::Serialize( TSerialize ser, unsigned aspects )
{
	CActor::Serialize(ser, aspects);

	ser.BeginGroup("CAlien");		//this serialization has to be redesigned to work properly in MP
	ser.Value("modelQuat", m_modelQuat, NSerPolicy::AC_Orientation(12));
	ser.Value("health", m_health, NSerPolicy::AC_SimpleFloat(0, 4095, 12, 32, 1));
	// skip matrices
	ser.Value("velocity", m_velocity, NSerPolicy::AC_Velocity(100, 12));
	ser.Value("desiredVelocity", m_desiredVelocity, NSerPolicy::AC_Velocity(100, 8));
	ser.Value("desiredVelocityQuat", m_desiredVeloctyQuat, NSerPolicy::AC_Orientation(12));
	ser.Value("turnSpeed", m_turnSpeed, NSerPolicy::AC_SimpleFloat(0, 1024, 12));
	ser.Value("turnSpeedGoal", m_turnSpeedGoal, NSerPolicy::AC_SimpleFloat(0, 1024, 8));
	ser.Value("angularVel", m_angularVel);
	ser.Value("tentacleBlendRatio", m_tentacleBlendRatio, NSerPolicy::AC_Float(0, 1, 12));
	ser.Value("isFiring", m_isFiring, NSerPolicy::A_Bool());
	ser.EnumValue("stance", m_stance, STANCE_NULL, STANCE_LAST);
	ser.EndGroup();

	m_input.Serialize(ser, aspects);
	m_stats.Serialize(ser, aspects);
	m_params.Serialize(ser, aspects);
}

void CAlien::SerializeXML( XmlNodeRef& node, bool bLoading )
{
}

void CAlien::SetAuthority( bool auth )
{

}

IActorMovementController * CAlien::CreateMovementController()
{
	return new CCompatibilityAlienMovementController(this);
}

void SAlienInput::Serialize( TSerialize ser, unsigned aspects )
{
	ser.BeginGroup("SAlienInput");
	ser.Value("deltaMovement", deltaMovement);
	ser.Value("deltaRotation", deltaRotation);
	ser.Value("actions", actions);
	ser.Value("movementVector", movementVector);
	ser.Value("viewVector", viewVector);
	ser.Value("posTarget", posTarget);
	ser.Value("dirTarget", dirTarget);
	ser.Value("upTarget", upTarget);
	ser.Value("speedTarget", speedTarget);
	ser.EndGroup();
}

void SAlienParams::Serialize( TSerialize ser, unsigned aspects )
{
	ser.BeginGroup("SAlienParams");
	ser.Value("speedInertia", speedInertia);
	ser.Value("rollAmount", rollAmount);
	ser.Value("sprintMultiplier", sprintMultiplier);
	ser.Value("sprintDuration", sprintDuration);
	ser.Value("rotSpeed_min", rotSpeed_min);
	ser.Value("rotSpeed_max", rotSpeed_max);
	ser.Value("speed_min", speed_min);
	ser.Value("forceView", forceView);
	ser.Value("movingBend", movingBend);
	ser.Value("idealSpeed", idealSpeed);
	ser.Value("blendingRatio", blendingRatio);
	ser.Value("approachLookAt", approachLookat);
	// skip string
	ser.Value("fullAnimationTentaclesBlendMult", fullAnimationTentaclesBlendMult);
	ser.Value("tentaclesCollide", tentaclesCollide);
	ser.EndGroup();
}

void SAlienStats::Serialize( TSerialize ser, unsigned aspects )
{
	ser.BeginGroup("SAlienStats");
	ser.Value("inAir", inAir);
	ser.Value("onGround", onGround);
	ser.Value("speedModule", speedModule);
	ser.Value("mass", mass);
	ser.Value("bobCycle", bobCycle);
	ser.Value("sprintLeft", sprintLeft);
	ser.Value("sprintThreshold", sprintTreshold);
	ser.Value("sprintMaxSpeed", sprintMaxSpeed);
	ser.Value("isThirdPerson", isThirdPerson);
	ser.Value("isRagdoll", isRagDoll);
	ser.Value("velocity", velocity);
	ser.Value("eyePos", eyePos);
	ser.Value("eyeAngles", eyeAngles);
	ser.EndGroup();
}
