/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: Directs camera shots, as requested by the CFilmDirector.
-------------------------------------------------------------------------
History:
- 30:6:2005: Created by Nick Hesketh

*************************************************************************/

#pragma once

#include <list>

#include "G4PlayerView.h"
class CFilmDirector;
class CGame;
class CPlayer;
class CPlayerG4;
struct SViewParams;
struct IView;
class CFilmEvent;

const int FilmShot_MaxActors = 4;   // upto 4 actors considered in a shot

enum eSlowMoRate
{
  eSlowMo_None,               // normal speed, no slow mo effect
  eSlowMo_Stretch,            // action takes slightly longer, not noticeable as slow mo
  eSlowMo_StretchMore,        // action takes longer still, may seem a little drawn out
  eSlowMo_Med,                // definate slow mo : half speed
  eSlowMo_High,               // quarter speed
  eSlowMo_VHigh,              // eighth speed
  eSlowMo_Extreme,            // 20 times longer. ie a 3 sec event takes 1 min, sound would only travel 16 metres per sec!
  eSlowMo_Compress,           // Things happen a little quicker, not noticeable
  eSlowMo_CompressMore,       // Things happen quicker still, may seem slightly fast
  eSlowMo_QuickTime,          // Noticeably fast (Kung Fu fast)
  eSlowMo_DoubleSpeed,        // Double speed, ** cartoony **
};

enum eSlowMoEase
{
  eSlowMoEase_Hold,                 // same slowmo rate through shot, no ease in/out
  eSlowMoEase_SlideIn,              // ease in to the slowmo from real time
  eSlowMoEase_SlideInFast,          // fast ease in from real time
  eSlowMoEase_SlideOut,             // ease out from slowmo to real time
  eSlowMoEase_SlideOutFast,         // fast ease out from slowmo to real time
  eSlowMoEase_SlideInHoldOut,       // ease in, hold, then ease out
  eSlowMoEase_SlideInHoldOutFast,   // fast ease in, hold, then fast ease out
  eSlowMoEase_SlideOutHoldIn,       // ease out, hold, then ease in
};



struct CFilmShotCast
{
  EntityId aIdActors[FilmShot_MaxActors]; // upto 4 actors in a shot
  bool bInterestPoint;
  Vec3 vInterestPoint;                    // can also have an interest point
  bool bInterestDirection;
  Vec3 vInterestDir;                      // can also have an interest direction
};


// shot sequences
struct SFilmShotSeqNode
{

  const char *psShot;
  const char **apsShot;
  int iShot;
  float time;
  float timeDelta;
  eSlowMoRate slowMoRate;
  eSlowMoEase slowMoEase;
};

struct SFilmShotSeq
{
  const char *psName;
  float intensity;
  float frequency;
  int count;
  SFilmShotSeqNode aSeqShot[8];
};


enum EFramePos
{
  eFrameNone=		0x000,
  // 9 main positions
  eFrameTopLeft=	0x001,
  eFrameMidLeft=	0x002,
  eFrameBotLeft=	0x004,
  eFrameTopRight=	0x008,
  eFrameMidRight=	0x010,
  eFrameBotRight=	0x020,
  eFrameTopMid=	0x040,
  eFrameMidMid=	0x080,
  eFrameBotMid=	0x100,
  // Left/Right/Top/Bottom of frame
  eFrameLeft=		(eFrameTopLeft|eFrameMidLeft|eFrameBotLeft),
  eFrameRight=	(eFrameTopRight|eFrameMidRight|eFrameBotRight),	
  eFrameTop=		(eFrameTopLeft|eFrameTopMid|eFrameTopRight),
  eFrameBot=		(eFrameBotLeft|eFrameBotMid|eFrameBotRight),
  // H/V Centre
  eFrameHorzCentre=(eFrameMidLeft|eFrameMidMid|eFrameMidRight),
  eFrameVertCentre=(eFrameTopMid|eFrameMidMid|eFrameBotMid),
  // Any
  eFrameAny=		0xffff

};

enum EShotType
{
  eShotECU=		0x001,
  eShotCU=		0x002,
  eShotMCU=		0x004,
  eShotM=			0x008,
  eShotML=		0x010,
  eShotL=			0x020,
  eShotW=			0x040,

  eShotClose=		(eShotCU|eShotMCU),
  eShotMedium=	(eShotMCU|eShotM),
  eShotLong=		(eShotM|eShotML)
};

enum EShotPOV
{
  eShotPovELow=	0x001,
  eShotPovWorm=	eShotPovELow,
  eShotPovLow=	0x002,
  eShotPovMidLow=	0x004,
  eShotPovMidMid=	0x008,
  eShotPovMidHigh=0x010,
  eShotPovMid=	eShotPovMidLow|eShotPovMidMid|eShotPovMidHigh,
  eShotPovHigh=	0x020,
  eShotPovEHigh=	0x040,
  eShotPovBird=	eShotPovEHigh
};

enum EShotDOF
{
  eShotDofNone=		0x001,
  eShotDofLow=		0x002,
  eShotDofMid=		0x004,
  eShotDofMidHigh=	0x008,
  eShotDofHigh=		0x010,
  eShotDofEHigh=		0x020
};



class CFilmCamShot;
struct SFilmCamShotNode
{
  char acShotName[64];
  CFilmCamShot *pShotHandler;
};

#if 0
struct SCamPos
{
	Vec3 vPos;
	float dist;		// spherical coords of camera relative to player/reference object
	float yaw;		// spherical coords of camera relative to player/reference object
	float pitch;	// spherical coords of camera relative to player/reference object

	float horzFraming;		// horizontal framing offset. Normally -1/3, 0, or +1/3 (left third, middle, right third)
	float vertFraming;		// vertical framing offset. Normally -1/3, 0, or +1/3 (top third, middle, bottom third)

	float angDamping;
	float posDamping;
};

struct SCamLook
{
	Quat rotation;
	float yaw;		// where the camera is looking
	float pitch;	// where the camera is looking
	float roll;		// where the camera is looking

	float focalDist;		// These are used by the DOF effect
	float focalWidth;		// These are used by the DOF effect
	float focalAmount;	// These are used by the DOF effect

	float angDamping;
	float focalDistDamping;
	float focalWidthDamping;
	float focalAmountDamping;

};

struct SCamEffect
{
	float dofAmount;	// depth of field. calc'd from focal amount
	float dofNear;		// depth of field. calc'd from focal dist, width
	float dofFar;			// depth of field. calc'd from focal dist, width
	float fov;				// field of view
	float fovDamping;
};

struct SCamState
{
	SCamPos pos;
	SCamLook look;
	SCamEffect effect;
};

inline Vec3 SphericalToCartesian(float yaw,float pitch,float dist)
{
	Vec3 vPos;
	vPos.x=dist*cos(yaw)*sin(pitch);
	vPos.y=dist*sin(yaw)*sin(pitch);
	vPos.z=dist*cos(pitch);

	return vPos;
}

inline void CartesianToSpherical(Vec3 vPos,float &yaw,float &pitch,float &dist)
{
	yaw=atan2(vPos.y,vPos.x);
	dist=vPos.len();
	float distPitch= abs(dist)> 0.0001f ? dist : dist >=0 ? 0.0001f : -0.0001f;
	pitch=acos(vPos.z/distPitch);
}
#endif
typedef std::list<SFilmCamShotNode> TCamShotNodes;
typedef std::list<SFilmCamShotNode>::iterator TCamShotNodesIter;

typedef std::list<CPlayerG4*> TDeadPlayers;
typedef std::list<CPlayerG4*>::iterator TDeadPlayersIter;
typedef std::list<IEntity*> TDeadPlayerEntities;
typedef std::list<IEntity*>::iterator TDeadPlayerEntitiesIter;

//! Directs camera shots, as requested by the CFilmDirector.
class CFilmCamera
{
public:
  //-------------------------------
  //--- API for Director ----------
  //-------------------------------

  CFilmCamera(CFilmDirector *pDirector,CGame *pGame);

  void Update(float frameTime);

  bool TickCamera(CPlayerG4 *pPlayer,SViewParams &viewParams);

  const SFilmShotSeq  *GetShotSeqForEvent(CFilmEvent &event);

  void SetCast(CFilmShotCast &rCast);
  void SetCast(CFilmEvent &event);

  CFilmShotCast &GetCast() { return m_cast; }
  bool SetShot(const char *psShotName);
  void RequestShot(const char *psShotName);
  bool TestShot(const char *psShotName);

  void DumpShots();

  void EditKey(const char *psName,float val);

  void NotifyBeginShot();
  void NotifyEndShot();
  //-------------------------------
  //--- API for CamShots ----------
  //-------------------------------

  void CalcLineOfAction();

  Vec3 GetLineOfAction()
  {
    if(!m_bLineOfAction)
      CalcLineOfAction();
    return m_vLineOfAction;
  }

  float GetActionDist()
  {
    if(!m_bLineOfAction)
      CalcLineOfAction();
    return m_actionDist;
  }

  IView *GetView();

  void EnableEffects(bool bOn);

  void PlayerDeleted(CPlayerG4 *pPlayer);

  bool IsValidPlayer(CPlayerG4 *pPlayer);
  bool IsValidPlayer(IEntity *pEntPlayer);

  void OnEditorSetGameMode(bool bGameMode);
private:
  CFilmDirector *m_pDirector;
  CFilmShotCast m_cast;
  CGame *m_pGame;
  bool m_bLineOfAction;
  Vec3 m_vLineOfAction;
  float m_actionDist;
  bool m_bInit;
  TCamShotNodes m_shotTable;
  SFilmCamShotNode *m_pActiveNode;
  SFilmCamShotNode *m_pRequestedNode;
  float m_desat;
  float m_dt;
  float m_editSavedTimeStep;
  TDeadPlayers m_deadPlayers;
  TDeadPlayerEntities m_deadPlayerEntities;

	void UVCollision(CPlayerG4 *pPlayer,SViewParams &viewParams);
	void UVGetDesiredCameraState(CPlayerG4 *pPlayer,SViewParams &viewParams,SCamState &state);
	//void UVGetDesiredCameraState(CPlayerG4 *pPlayer,SViewParams &viewParams,Vec3 &vPosOut,Quat &rotOut,float &fovOut,float &posDamping,float &angDamping);
  void UpdateViewThirdPersonDamped(CPlayerG4 *pPlayer,SViewParams &viewParams);
  bool UpdateViewCameraShot(CPlayerG4 *pPlayer,SViewParams &viewParams);
  bool UpdateView2(CPlayerG4 *pPlayer,SViewParams &viewParams);

  void UpdateCinematicIntensity(float intensity);

  bool CFilmCamera::EvalSeq(SFilmShotSeq *pSeq);
  const SFilmShotSeq  *CFilmCamera::PickSeqFromList(SFilmShotSeq *paSeqList);
};

