/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: Monitors the world for the CFilmDirector.
-------------------------------------------------------------------------
History:
- 30:6:2005: Created by Nick Hesketh

*************************************************************************/

#include "StdAfx.h"

#include ".\FilmWorld.h"

#include <IActorSystem.h>
#include <IEntitySystem.h>
#include <IScriptSystem.h>
#include "Player.h"
#include "G4Player.h"
#include "Game.h"
#include "Game04.h"

typedef std::vector<EntityId> TEntityIds;
typedef std::vector<EntityId>::iterator TEntityIdIter;

struct SFilmEntityInfo;
//--------------------------------------------------------------------------------------------------------
//--- CFilmBubble
//--------------------------------------------------------------------------------------------------------
// An action region of cinematic interest.
//--------------------------------------------------------------------------------------------------------
// ActionBubble, SceneBubble, LevelBubble etc. A position, rough radius, cast list, interesting object list
//--------------------------------------------------------------------------------------------------------
class CFilmBubble
{
public:
  CFilmBubble(CFilmWorld *pWorld);
  virtual ~CFilmBubble();

  void Init(Vec3 &rPos,float radius=10.0f,int id=-1);
  void DeInit();  

  void Update(float frameTime,Vec3 &vPlayer, Ang3 &vAng);


  // returns 0..1 NB: don't rely on ever getting a 1.0 .... the player may escape the bubble early!
  float CalcProgress();


  int GetNumLiveActors() { return m_nLiveActors; }
  int GetNumStartActors() { return m_nActors; }
  int GetNumObjects() { return m_nLiveObjects; }
  int GetNumStartObjects() { return m_nObjects; }

protected:
  bool m_bInit;
  CFilmWorld *m_pWorld;
  Vec3 m_vPos;
  float m_radius;

  // Actors aka enemies in the bubble
  TEntityIds m_actorEntities;
  int m_nActors;
  int m_nLiveActors;

  // Interesting objects - i.e. explosive/breakable objects
  TEntityIds m_objects;
  int m_nObjects;
  int m_nLiveObjects;
  float m_lastPreogress;

  void Clear();

  void PopulateByRadius();        // Use physics - Entiies in box...
  void PopulateById(int id);        // Entiies in box...

  bool GetEntityObjectInfo(EntityId id,IPhysicalEntity * pPhys,SFilmEntityInfo & rInfo);

};

void CFilmBubble::Update(float frameTime,Vec3 &vPlayer,Ang3 &vAng)
{
  IActor *pActor;
  EntityId idEntity;
  
  if(m_bInit)
  {
    if(m_nLiveActors)
    {
      int nLive,iActor;

      for(iActor=0,nLive=0;iActor<m_nActors;++iActor)
      {
        idEntity=m_actorEntities[iActor];
        pActor=m_pWorld->GetActorForEntityId(idEntity);
        if(pActor && pActor->GetHealth()>0)  // Not a dead body
        {
          ++nLive;
        }
      }

      m_nLiveActors=nLive;
    }
  }
}

void CFilmBubble::Init(Vec3 &rPos,float radius,int id)
{
    DeInit();

    m_vPos=rPos;
    m_radius=radius;

    m_bInit=true;

    PopulateByRadius();

    if(m_nLiveActors>0)
    {
      CryLogAlways( "CFilmBubble:Init: Pos(%0.2f,%0.2f,%0.2f), r:%0.2f, nLiveAct:%d nLiveObj:%d", rPos.x,rPos.y,rPos.z,radius,m_nLiveActors,m_nLiveObjects);
    }
}

void CFilmBubble::DeInit()
{
  if(m_bInit)
  {
    Clear();
    m_bInit=false;
    m_vPos=Vec3(0.0f,0.0f,0.0f);
    m_radius=0.0f;
    m_nActors=0;
    m_nLiveActors=0;
    m_nObjects=0;
    m_nLiveObjects=0;
    m_lastPreogress=0.0f;
  }
}

void CFilmBubble::Clear()
{
  TEntityIds a,b; // These are destroyed on return, clearing their members, and freeing associated memory.

  // Use the stl swap-to-clear trick to properly clear out the vectors.
  m_actorEntities.swap(a);
  m_objects.swap(b);
}

CFilmBubble::CFilmBubble(CFilmWorld *pWorld) : m_bInit(false), m_vPos(0.0f,0.0f,0.0f), m_radius(0.0f), m_nActors(0),m_nLiveActors(0),m_nObjects(0), m_nLiveObjects(0), m_lastPreogress(0.0f)
{
  m_pWorld=pWorld;
}

CFilmBubble::~CFilmBubble()
{
}

struct SFilmExplodeableInfo
{
  float effectRating;

  float rMin;
  float rMax;
  float radius;
  float impulse;
  float damage;

  //float reqDamage;
};

struct SFilmBreakableInfo
{
  float effectRating;
  // bounding box info.
  // required damage.
};

struct SFilmBurnInfo
{
  float effectRating;
  // bounding box info.
  // required damage.
};

struct SFilmSparkInfo
{
  float effectRating;
  // bounding box info.
  // required damage.
};

struct SFilmDebrisInfo
{
  float effectRating;
  // bounding box info.
  // required damage.
};

struct SFilmEntityInfo
{
  EntityId id;

  float effectRating;

  bool bExplodes;
  bool bBreaks;
  bool bBurns;
  bool bSparks;
  bool bDebris;

  SFilmExplodeableInfo explodeInfo;
  SFilmBreakableInfo breakInfo;
  SFilmBurnInfo burnInfo;
  SFilmSparkInfo sparkInfo;
  SFilmDebrisInfo debrisInfo;
};

// Cache these queries for performance improvement ...
bool CFilmBubble::GetEntityObjectInfo(EntityId id,IPhysicalEntity *pPhys,SFilmEntityInfo & rInfo)
{
  bool bRet=false;
  bool bExplosive=false;
  float damage,impulse,radius,rMax,rMin;

  IEntity *pEnt = GetISystem()->GetIEntitySystem()->GetEntity(id);

  if(pEnt)
  {
    IScriptSystem *pSS=GetISystem()->GetIScriptSystem();
    IScriptTable *pScript = pEnt->GetScriptTable();

		if(pScript)
		{
    SmartScriptTable properties(pSS, false);
    SmartScriptTable explosion(pSS, false);

    
    if (pScript->GetValue("Properties", properties))
    {
      int iExplode;
      IScriptTable *pScriptProp=properties.GetPtr();
      if(pScriptProp->GetValue("bExplodeOnHit",iExplode))
      {
        bExplosive=true;
        float pressure;
        if(!pScriptProp->GetValue("ExplosionDamage",damage)) damage=1000.0f;
        if(!pScriptProp->GetValue("ExplosionRadius",radius)) radius=10.0f;
        if(!pScriptProp->GetValue("ExplosionPressure",pressure)) pressure=1000.0f;
        if(!pScriptProp->GetValue("ExplosionDamage",damage)) damage=1.0f;
        if(!pScriptProp->GetValue("ExplosionDamage",damage)) damage=1.0f;

        impulse=pressure;
        rMin=1;
        rMax=max(radius,rMin);
      }
    }
    if(!bExplosive)
    {
      if(pScript->GetValue("explosion_params",properties))
        bExplosive=true;
    }

    // "Properties.bExplodeOnHit"
    // " .ExplosionDamage"
    // " .ExplosionRadius"
    // " .ExplosionPressure"
    // ":Explode" func
    if(bExplosive)
    {
      {
        rInfo.id=id;
        rInfo.effectRating=0.75f;
        rInfo.bExplodes=true;
        rInfo.explodeInfo.effectRating=0.75f;
        rInfo.explodeInfo.damage=damage;
        rInfo.explodeInfo.impulse=impulse;
        rInfo.explodeInfo.radius=radius;
        rInfo.explodeInfo.rMax=rMax;
        rInfo.explodeInfo.rMin=rMin;
      }
    }
  }
	}

  // explodes? => check the script for an explosion_params sub table or ExplosionEffect member
  // or :Event_Explode( ) member

  // :OnHit()

  // :OnDamage()

  // :SpawnDamageParticles()

  // Particles.explosion ""

  // table .Dead

  // add a cinematicDamageRating to objects along with hints as per FilmEntityInfo

  return bRet;
}

void CFilmBubble::PopulateByRadius()        // Use physics - Entiies in box...
{

  CPlayerG4 *pPlayer=g_pGame04->GetClientPlayer();
  IEntity *pPlayerEnt=pPlayer->GetEntity();

  // note: this list is shared by physics for ray intersection etc.
  // so entities must be saved into another list before ray checks for visibilty
  IPhysicalEntity **pPhysArray=NULL,*pPhys;
  Vec3 vMin,vMax,vRadius(m_radius,m_radius,m_radius);
  int nEntities;

  vMin=m_vPos-vRadius;
  vMax=m_vPos+vRadius;

	//	int flags=ent_static|ent_sleeping_rigid|ent_rigid|ent_living|ent_independent;
	// ent_independent => these are entities like particles (bullets, glass shards), live character skeletons, cloth
	int flags=ent_static|ent_sleeping_rigid|ent_rigid|ent_living;
  nEntities=GetISystem()->GetIPhysicalWorld()->GetEntitiesInBox(vMin, vMax, pPhysArray,flags);

  int iEntity,idEntity;
  IActor *pActor;
  for (iEntity=0; iEntity<nEntities; iEntity++)
  {
    pPhys=pPhysArray[iEntity];
    IEntity *pEnt=m_pWorld->GetEntitySystem()->GetEntityFromPhysics(pPhys);  // NB: sometimes returns null pointers.
    if(pEnt && pEnt!=pPlayerEnt)
    {
      idEntity=pEnt->GetId();
      pActor=m_pWorld->GetActorForEntityId(idEntity);
      if (pActor) // An actor
      {
        if (pActor->GetHealth()>0)  // Not a dead body
        {
          //--- Check for duplicate idEntity's here, since we seem to be getting some ......
          ++m_nActors;
          ++m_nLiveActors;
          m_actorEntities.push_back(idEntity);
        }
      }
      else  // An object
      {
        // Is it an interesting object?
        //if(pPhys->)
        SFilmEntityInfo info;
        memset(&info,0,sizeof(info));
        GetEntityObjectInfo(idEntity,pPhys,info);
        if(info.bExplodes || info.bBurns || info.bBreaks || info.bSparks)
        {
          ++m_nObjects;
          ++m_nLiveObjects;
          m_objects.push_back(idEntity);
        }
      }
    }
  }
}

void CFilmBubble::PopulateById(int id)        // Entiies in box...
{
}

// returns 0..1 NB: don't rely on ever getting a 1.0 .... the player may escape the bubble early!
float CFilmBubble::CalcProgress()
{
  float progress=0.25f;
  
  if(m_nActors>0)
    progress=(m_nActors-m_nLiveActors)/(float)m_nActors;

  return progress;
}

IActor *CFilmWorld::GetActorForEntityId(EntityId idEntity)
{
  return m_pActorSystem->GetActor(idEntity);
}

float CFilmWorld::GetBubbleProgress()
{
  return m_pBubble->CalcProgress();
}

CFilmWorld::CFilmWorld(CFilmDirector *pDirector,CGame *pGame)
{
  m_pDirector=pDirector;
  m_pGame=pGame;
  m_pBubble=new CFilmBubble(this);
  m_iBubble=0;
  m_pActorSystem=0;
  m_pEntitySystem=0;
}

CFilmWorld::~CFilmWorld()
{
  delete m_pBubble;
}

void CFilmWorld::Update(float frameTime)
{
  // Need an init func.
  if(!m_pEntitySystem || !m_pActorSystem)
  {
    m_pActorSystem=m_pGame->GetIGameFramework()->GetIActorSystem();
    m_pEntitySystem=m_pGame->GetIGameFramework()->GetISystem()->GetIEntitySystem();
  }

  CPlayerG4 *pPlayer=g_pGame04->GetClientPlayer();
  IEntity *pPlayerEnt=0;
  Vec3 vPlayer(0.0f,0.0f,0.0f);
  Ang3 vAng(0.0f,0.0f,0.0f);

  if(pPlayer)
  {
    pPlayerEnt=pPlayer->GetEntity();
    vPlayer=pPlayerEnt->GetWorldPos();
    vAng=pPlayer->GetAngles();
  }

  if(m_pBubble->GetNumLiveActors()<=0)
  {
    m_pBubble->Init(vPlayer,20.0f);  // 100 m default bubble radius.
    if(m_pBubble->GetNumLiveActors()>0)
      ++m_iBubble;
  }

  m_pBubble->Update(frameTime,vPlayer,vAng);
}

