//--------------------------------------------------------------------------
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2001-2005.
// -------------------------------------------------------------------------
//  File name:   G4FlowNodes_GameLogic.cpp
//  Version:     v1.00
//  Created:     21/11/2005		Nick Hesketh
//  Compilers:   Visual Studio.NET 2003
//  Description: Flow nodes required for Game04's logic.
//--------------------------------------------------------------------------
//
// CG4FlowNode_SetInputActionMap:
//		G4Game:SetInputActionMap
//		Lets designers enable/disable controller input action maps. see G4Game.cpp
//
// CG4FlowNode_OnAction:
//		G4Game:OnAction
//		Lets designers respond to individual input actions e.g. jump, sprint etc.
//		See G4Game.cpp for action mappings.
//
//
// See comment in G4FlowBaseNode.h (CG4AutoRegFlowNode) for a description of the
// registration process.
//--------------------------------------------------------------------------
//  History:
//
//--------------------------------------------------------------------------


#include "stdafx.h"

#include "G4FlowBaseNode.h"

#include "G4Game.h"
#include "G4Player.h"

#include <IActionMapManager.h>

//#include <time.h>

CG4AutoRegFlowNodeBase *CG4AutoRegFlowNodeBase::m_pFirst=0;
CG4AutoRegFlowNodeBase *CG4AutoRegFlowNodeBase::m_pLast=0;


//--------------------------------------------------------------
//-- Enable/Disable user input via action map filters
//--------------------------------------------------------------
// NOT USED at present
class CG4FlowNode_EnableUserInput : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_ENABLED
	};

	CG4FlowNode_EnableUserInput( SActivationInfo * pActInfo )
	{
		//pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, true );
	}
	virtual void GetConfiguration( SFlowNodeConfig& config )
	{
		static const SInputPortConfig in_config[] = {
			InputPortConfig<bool>("enabled", true, _HELP("Enables user input.")),
			{0}
		};
		static const SOutputPortConfig out_config[] = {
			{0}
		};

		config.pInputPorts = in_config;
		config.pOutputPorts = out_config;
		config.SetCategory(EFLN_WIP);
	}
	virtual void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch (event)
		{
		case eFE_Update:
			{
				bool bEnabled = GetPortBool(pActInfo,IN_ENABLED);
				g_pGame->GetIGameFramework()->GetIActionMapManager()->EnableFilter("no_player_action",bEnabled);
			}
			break;
		case eFE_Activate:
		case eFE_Initialize:
			if (IsPortActive(pActInfo,IN_ENABLED))
			{
				bool bEnabled = GetPortBool(pActInfo,IN_ENABLED);
				g_pGame->GetIGameFramework()->GetIActionMapManager()->EnableFilter("no_player_action",bEnabled);
			}
			break;
		}
	}
};

//--------------------------------------------------------------
//-- Set user input action map filters
//--------------------------------------------------------------
// NOT USED at present
class CG4FlowNode_EnableInputFilter : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_SET,
		IN_ENABLED,
		IN_FILTER
	};

	CG4FlowNode_EnableInputFilter( SActivationInfo * pActInfo )
	{
		//pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, true );
	}
	virtual void GetConfiguration( SFlowNodeConfig& config )
	{
		static const SInputPortConfig in_config[] = {
			InputPortConfig<bool>("set", false, _HELP("Trigger to enable/disable the user input filter.")),
			InputPortConfig<bool>("enabled", true, _HELP("true will enable the user input filter when set is triggered.")),
			InputPortConfig<string>("filter", _HELP("User input filter to enable/disable. eg no_follow, no_attack, no_move")),
			{0}
		};
		static const SOutputPortConfig out_config[] = {
			{0}
		};

		config.pInputPorts = in_config;
		config.pOutputPorts = out_config;
		config.SetCategory(EFLN_WIP);
	}
	virtual void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch (event)
		{
		case eFE_Update:
			break;
		case eFE_Activate:
			{
				if(IsPortActive(pActInfo,IN_SET))
				{
					bool bEnabled = GetPortBool(pActInfo,IN_ENABLED);
					string strFilter=GetPortString(pActInfo,IN_FILTER);
					g_pGame->GetIGameFramework()->GetIActionMapManager()->EnableFilter(strFilter.c_str(),bEnabled);
				}
			}
			break;
		case eFE_Initialize:
			break;
		}
	}
};

//--------------------------------------------------------------
//-- Set user input action map
//--------------------------------------------------------------
class CG4FlowNode_SetInputActionMap : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_SET,
		IN_ENABLED,
		IN_ACTIONMAP
	};

	CG4FlowNode_SetInputActionMap( SActivationInfo * pActInfo )
	{
		//pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, true );
	}
	virtual void GetConfiguration( SFlowNodeConfig& config )
	{
		static const SInputPortConfig in_config[] = {
			InputPortConfig<bool>("set", false, _HELP("Sets user action map.")),
			InputPortConfig<bool>("enabled", true, _HELP("true enables the user input filter.")),
			InputPortConfig<string>("map", _HELP("User input action map to enable/disable. eg 'default'")),
			{0}
		};
		static const SOutputPortConfig out_config[] = {
			{0}
		};

		config.pInputPorts = in_config;
		config.pOutputPorts = out_config;
		config.SetCategory(EFLN_WIP);
	}
	virtual void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch (event)
		{
		case eFE_Update:
			break;
		case eFE_Activate:
			{
				if(IsPortActive(pActInfo,IN_SET))
				{
					bool bEnabled = GetPortBool(pActInfo,IN_ENABLED);
					string strActionMap=GetPortString(pActInfo,IN_ACTIONMAP);
					g_pGame->GetIGameFramework()->GetIActionMapManager()->EnableActionMap(strActionMap.c_str(),bEnabled);
				}
			}
			break;
		case eFE_Initialize:
			break;
		}
	}
};


//--------------------------------------------------------------
//-- Test for specific OnAction msgs
//--------------------------------------------------------------
class CG4FlowNode_OnInputAction : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_PAUSED,
		IN_ACTION
	};
	enum EOutputs
	{
		OUT_ACTIVE,
		OUT_VALUE,
		OUT_STARTED,
		//OUT_STARTED_TRIGGER,
		//OUT_OFF	// Not implementing because we don't want the stuck key logic problem migrating up to the flow graph.
	};

	virtual void Serialize(SActivationInfo *, TSerialize ser)
	{
		if (ser.IsReading())
		{
			m_bWasActive=false;
			m_pActionState=0;
		}
	}

	void AutoUpdate(SActivationInfo * pActInfo,bool bOn,char *psMsg=0)
	{
		pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, bOn );
		GetISystem()->GetILog()->Log("OnAction:%s: Set auto update to %c",psMsg ? psMsg : "", bOn ? 'T' : 'F');
	}
	CG4FlowNode_OnInputAction( SActivationInfo * pActInfo )
	{
		//pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, true );
		AutoUpdate(pActInfo,true,"Constructor");
	}
	
	virtual void GetConfiguration( SFlowNodeConfig &config )
	{
		static const SInputPortConfig in_config[] = {
			InputPortConfig<bool>("paused",true,_HELP("pause testing for this action")),
			InputPortConfig<string>("action", _HELP("Action to test for eg jump, draw, follow")),
			{0}
		};

		static const SOutputPortConfig out_config[] = {
			OutputPortConfig<bool>("active",_HELP("Action is occurring. May be have just started, or may have been happening continuously for some time.")),
			OutputPortConfig<float>("value",_HELP("Value associated with ths action. Not all actions have a valid value.")),
			OutputPortConfig<bool>("started",_HELP("Action has just started, wasn't happening last frame (true/false). Note: you need to check whether this is true or false.")),
			//OutputPortConfig<bool>("started_trigger",_HELP("Action has just started, wasn't happening last frame. trigger")),

			// Not implementing because we don't want the stuck key logic problem migrating up to the flow graph.
			//OutputPortConfig<bool>("off",_HELP("Action just stopped")),
			{0}
		};

		config.pInputPorts = in_config;
		config.pOutputPorts = out_config;
		config.SetCategory(EFLN_WIP);
	}
	void UpdateActionStatePtr(SActivationInfo *pActInfo,const char *psMsg=0)
	{
		const char *psActionName=GetPortString(pActInfo,IN_ACTION).c_str();
		if(psActionName && psActionName!=0)
		{
			CPlayerActionMap *pMap=CPlayerActionMap::GetPlayerActionMap();
			m_pActionState=pMap->GetActionState(psActionName);
		}
		else
		{
			m_pActionState=0;
		}
		if(m_pActionState)
			GetISystem()->GetILog()->Log("OnAction:%s: Got action state ptr for '%s'",psMsg ? psMsg : "",psActionName);
		else
			GetISystem()->GetILog()->Log("OnAction:%s: Got action state ptr for '%s'",psMsg ? psMsg : "",psActionName ? psActionName : "null");
	}
	virtual void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch (event)
		{
		case eFE_Activate:
			if(IsPortActive(pActInfo,IN_PAUSED))
			{
				bool bPaused=GetPortBool(pActInfo,IN_PAUSED);
				AutoUpdate(pActInfo,!bPaused,"Activate");		
				//pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, !bPaused );
				m_bWasActive=false;
			}
			if(IsPortActive(pActInfo,IN_ACTION))
			{
				UpdateActionStatePtr(pActInfo,"IN_ACTION");
				m_bWasActive=false;
			}
			break;
		case eFE_Initialize:
			m_bWasActive=false;
			UpdateActionStatePtr(pActInfo,"Initialize");
			ActivateOutput( pActInfo,OUT_ACTIVE,false);
			ActivateOutput( pActInfo,OUT_STARTED,false);
			ActivateOutput(pActInfo,OUT_VALUE,0);
			break;
		case eFE_Update:
			{
				bool bPaused=GetPortBool(pActInfo,IN_PAUSED);

				if(!bPaused)
				{
					bool bActive=false;

					// test if GetPortString(pActInfo,IN_ACTION) is active
					if(!m_pActionState)
					{
						UpdateActionStatePtr(pActInfo,"Update(wasActive=>false)");
						m_bWasActive=false;
					}
					if(m_pActionState)
						bActive=m_pActionState->bThisFrame;

					if(bActive)
					{
						ActivateOutput( pActInfo,OUT_ACTIVE,bActive);
						ActivateOutput( pActInfo,OUT_STARTED,!m_bWasActive);
						if(m_bWasActive!=bActive)
						{
							
							//ActivateOutput( pActInfo,OUT_STARTED_TRIGGER,bActive);
						}
						ActivateOutput(pActInfo,OUT_VALUE,m_pActionState->value);
					}
					else
					{
						ActivateOutput( pActInfo,OUT_ACTIVE,false);
						ActivateOutput( pActInfo,OUT_STARTED,false);
					}
					m_bWasActive=bActive;
				}
				else
				{
					GetISystem()->GetILog()->Log("OnAction: Update: paused!");
				}
			}
			break;
		}
	}
private:
	bool m_bWasActive;
	SPlayerActionState *m_pActionState;
};

// Display a message
class CG4FlowNode_DisplayMessage : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_DISPLAY,
		IN_MSG,
		IN_POSX,
		IN_POSY,
		IN_FONTSIZE,
		IN_CENTER,
		//IN_COLOR,
	};
	//enum EOutputs
	//{
	//};

	CG4FlowNode_DisplayMessage( SActivationInfo * pActInfo )
	{
		//pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, true );
	}

	void Serialize(SActivationInfo *pActInfo, TSerialize ser)
	{
		// CryLogAlways("CG4FlowNode_DisplayMessage: Message='%s'", GetPortString(pActInfo, 0));
	}

	void GetConfiguration( SFlowNodeConfig& config )
	{
		// declare input ports
		static const SInputPortConfig in_ports[] = 
		{
				InputPortConfig<bool>( "display",false, _HELP("Enable display of this message" )),
				InputPortConfig<string>( "message", _HELP("The message to display" )),
				InputPortConfig<float>( "posX",	400.0f, _HELP("Input x text position" )),
				InputPortConfig<float>( "posY",	450.0f, _HELP("Input y text position" )),
				InputPortConfig<float>( "fontSize",	2.0f, _HELP("Input font size" )),
				InputPortConfig<bool>( "center",false, _HELP("Center the message at posX,posY" )),
				//InputPortConfig<Vec3>( "color", _HELP("Text color. RGB 0.0 to 1.0 per component." )),
			{0}
		};
		config.pInputPorts = in_ports;
		config.pOutputPorts = 0;
		config.SetCategory(EFLN_WIP);
	}

	void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch(event)
		{
		case eFE_Initialize:
			//if(IsPortActive(pActInfo,IN_DISPLAY))
			//pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, GetPortBool(pActInfo,IN_DISPLAY) );
			pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, true );
			break;
		case eFE_Activate:
			if(IsPortActive(pActInfo,IN_DISPLAY))
			{
				bool bUpdate=GetPortBool(pActInfo,IN_DISPLAY);
				pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, bUpdate );
			}
			break;
		case eFE_Update:
			if(GetPortBool(pActInfo,IN_DISPLAY))
			{
				IRenderer *pRenderer = GetISystem()->GetIRenderer();
				float colour[4]={1,1,1,1};
				pRenderer->Draw2dLabel(GetPortFloat(pActInfo,IN_POSX), GetPortFloat(pActInfo,IN_POSY), 
					GetPortFloat(pActInfo,IN_FONTSIZE), colour, GetPortBool(pActInfo,IN_CENTER), 
					GetPortString(pActInfo,IN_MSG).c_str() );
			}
			else
				pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, false );

			break;
		}
	}
};

// Output a true or false depending on which input was last triggered
class CG4FlowNode_Switch_2_1 : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_ON,
		IN_OFF,
	};
	enum EOutputs
	{
		OUT_ONOFF,
	};

	CG4FlowNode_Switch_2_1( SActivationInfo * pActInfo )
	{
		m_bOn=false;
	}

	void Serialize(SActivationInfo *pActInfo, TSerialize ser)
	{
		ser.BeginGroup("Local");
			ser.Value("m_bOn", m_bOn);
		ser.EndGroup();
	}

	void GetConfiguration( SFlowNodeConfig& config )
	{
		// declare input ports
		static const SInputPortConfig in_ports[] = 
		{
			InputPortConfig<bool>( "on", _HELP("Trigger this to turn output on" )),
			InputPortConfig<bool>( "off", _HELP("Trigger this to turn output off" )),
			{0}
		};
		static const SOutputPortConfig out_ports[] = {
			OutputPortConfig<bool>("on_off",_HELP("True if last input was to on, otherwise false.")),
				// Not implementing because we don't want the stuck key logic problem migrating up to the flow graph.
				//OutputPortConfig<bool>("off",_HELP("Action just stopped")),
			{0}
		};

		config.pInputPorts = in_ports;
		config.pOutputPorts = out_ports;
		config.SetCategory(EFLN_WIP);
	}

	void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch(event)
		{
		case eFE_Initialize:
			ActivateOutput( pActInfo,OUT_ONOFF,m_bOn);
			break;
		case eFE_Activate:
			if(IsPortActive(pActInfo,IN_ON))
			{
				m_bOn=true;
				ActivateOutput( pActInfo,OUT_ONOFF,m_bOn);
			}
			else if(IsPortActive(pActInfo,IN_OFF))
			{
				m_bOn=false;
				ActivateOutput( pActInfo,OUT_ONOFF,m_bOn);
			}
			break;
		}
	}

private:
	bool m_bOn;
};

// splits a single true false input into either a true signal or a flase signal.
class CG4FlowNode_Switch_1_2 : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_ONOFF,
	};
	enum EOutputs
	{
		OUT_ON,
		OUT_OFF
	};

	CG4FlowNode_Switch_1_2( SActivationInfo * pActInfo )
	{
	}

	void Serialize(SActivationInfo *pActInfo, TSerialize ser)
	{
	}

	void GetConfiguration( SFlowNodeConfig& config )
	{
		// declare input ports
		static const SInputPortConfig in_ports[] = 
		{
			InputPortConfig<bool>( "onoff", _HELP("True or false input" )),
			{0}
		};
		static const SOutputPortConfig out_ports[] = {
			OutputPortConfig<bool>("on",_HELP("Triggered if onoff is true.")),
			OutputPortConfig<bool>("off",_HELP("Triggered if onoff is false.")),
				// Not implementing because we don't want the stuck key logic problem migrating up to the flow graph.
				//OutputPortConfig<bool>("off",_HELP("Action just stopped")),
			{0}
		};

		config.pInputPorts = in_ports;
		config.pOutputPorts = out_ports;
		config.SetCategory(EFLN_WIP);
	}

	void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch(event)
		{
		case eFE_Initialize:
			break;
		case eFE_Activate:
			if(IsPortActive(pActInfo,IN_ONOFF))
			{
				if(GetPortBool(pActInfo,IN_ONOFF))
					ActivateOutput( pActInfo,OUT_ON,true);
				else
					ActivateOutput( pActInfo,OUT_OFF,true);
			}
			break;
		}
	}
};


//--------------------------------------------------------------
//-- Send an OnAction msg
//--------------------------------------------------------------
class CG4FlowNode_SendInputAction : public CFlowBaseNode
{
public:
	enum EInputs
	{
		IN_SEND,
		IN_ACTION,
		IN_MODE,
		IN_VALUE
	};
	enum EOutputs
	{
	};

	virtual void Serialize(SActivationInfo *, TSerialize ser)
	{
		if (ser.IsReading())
		{
		}
	}

	CG4FlowNode_SendInputAction( SActivationInfo * pActInfo )
	{
	}
// actor->GetActionListener()->OnAction("attack1", eAAM_OnRelease, 1.0f);
	virtual void GetConfiguration( SFlowNodeConfig &config )
	{
		static const SInputPortConfig in_config[] = {
			InputPortConfig<bool>("send",true,_HELP("send the action")),
			InputPortConfig<string>("action", _HELP("Action to send")),
			InputPortConfig<int>("mode",1, _HELP("0: none, 1: press, 2: release, 3: held")),
			InputPortConfig<float>("value",1.0f, _HELP("Value to send with action")),
			{0}
		};

		static const SOutputPortConfig out_config[] = {
			{0}
		};

		config.pInputPorts = in_config;
		config.pOutputPorts = out_config;
		config.SetCategory(EFLN_WIP);
	}
	virtual void ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
	{
		switch (event)
		{
		case eFE_Activate:
			if(IsPortActive(pActInfo,IN_SEND))
			{
				int mode;
				switch (GetPortInt(pActInfo,IN_MODE))
				{
				default:
				case 0:
					mode=0;
					break;
				case 1:
					mode=eAAM_OnPress;
					break;
				case 2:
					mode=eAAM_OnRelease;
					break;
				case 3:
					mode=eAAM_OnHold;
					break;
				}

				IGameFramework *pFramework=GetISystem()->GetIGame()->GetIGameFramework();
				if(pFramework)
				{
					IActor *pActor;
					pActor=pFramework->GetClientActor();
					if(pActor)
					{
						CPlayerG4 *pPlayer=static_cast<CPlayerG4*>(pActor);
						const char *psAction=GetPortString(pActInfo,IN_ACTION).c_str();
						float val=GetPortFloat(pActInfo,IN_VALUE);
						pPlayer->PushActionQueue( psAction, mode, val);
					}
				}
			}
			break;
		case eFE_Initialize:
			break;
		}
	}
private:
};

REGISTER_FLOW_NODE( "G4Game:EnableInputFilter",CG4FlowNode_EnableInputFilter );
REGISTER_FLOW_NODE( "G4Game:SetInputActionMap",CG4FlowNode_SetInputActionMap );
//REGISTER_FLOW_NODE( "G4Game:EnableUserInput",CG4FlowNode_EnableUserInput );
REGISTER_FLOW_NODE( "G4Game:OnInputAction",CG4FlowNode_OnInputAction );
REGISTER_FLOW_NODE( "G4Game:SendInputAction",CG4FlowNode_SendInputAction );
REGISTER_FLOW_NODE( "G4Game:DisplayMessage",CG4FlowNode_DisplayMessage);
REGISTER_FLOW_NODE( "G4Game:Switch_2_1",CG4FlowNode_Switch_2_1);
REGISTER_FLOW_NODE( "G4Game:Switch_1_2",CG4FlowNode_Switch_1_2);
