/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id$
$DateTime$

Description:
	The view specific code (First person, Third person, followCharacterHead etc).


-------------------------------------------------------------------------
History:
- 21:02:2006: Re-factored from Filippo's CPlayerG4 code by Nick Hesketh

*************************************************************************/

#include "StdAfx.h"

#include "Game.h"
#include "G4Player.h"
#include "G4PlayerView.h"
#include "GameUtils.h"
#include "GameActions.h"

#include "Weapon.h"
#include "WeaponSystem.h"

#include <IViewSystem.h>
#include <IItemSystem.h>
#include <IPhysics.h>
#include <ICryAnimation.h>
#include "IAISystem.h"
#include "IAgent.h"
#include <IVehicleSystem.h>
#include <ISerialize.h>

#include <IRenderAuxGeom.h>

#include <IGameTokens.h>

#include "HUD/HUD.h"

static int iDrawAngles=0;
inline void DrawAngles(const char *psMsg,Quat &rQ)
{
	float yaw,pitch,dummy;
	Vec3 vSmoothDir=rQ*Vec3(0,1,0);
	float clrGreen[4]={0,1,0,1};
	CartesianToSpherical(vSmoothDir,yaw,pitch,dummy);
	++iDrawAngles;
	IRenderer *pRend=GetISystem()->GetIRenderer();
	//pRend->Draw2dLabel(10, 30*iDrawAngles, 2.0f, clrGreen, false, "%s: smoothdir: (%0.3f,%0.3f)", psMsg,yaw*180/gf_PI,pitch*180/gf_PI);
}
inline void DrawAnglesDirVec(const char *psMsg,Vec3 dirVec)
{
	Quat qDir=Quat::CreateRotationVDir(dirVec.GetNormalizedSafe(Vec3(0,1,0)));
	DrawAngles(psMsg,qDir);
}


//CPlayerViewG4::SViewStateIn CPlayerViewG4::m_in;
//CPlayerViewG4::SViewStateInOut CPlayerViewG4::m_io;

CPlayerViewG4::CPlayerViewG4(const CPlayerG4 &rPlayer,SViewParams &viewParams) : CPlayerView(rPlayer,viewParams),m_in(m_viewStateIn_private)
{
	ViewPreProcess(rPlayer,viewParams,m_viewStateIn_private);
}

void CPlayerViewG4::Process(SViewParams &viewParams)
{
	ViewProcess(viewParams);
}

void CPlayerViewG4::Commit(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
	ViewPostProcess(rPlayer,viewParams);
	DrawAngles("OutFinal",viewParams.rotation);
}

//-----------------------------------------------------------------------------------------------------
//-----------------------------------------------------------------------------------------------------
//-----------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------------
//--- ViewPreProcess
//--------------------------------------------------------------------------
// Fetch all the data we need to work on.
//--------------------------------------------------------------------------
void CPlayerViewG4::ViewPreProcess(const CPlayerG4 &rPlayer,SViewParams &viewParams,SViewStateIn &m_in)
{

	// SViewStateIn --- the constants
	{
		m_in.lastPos=viewParams.position;
		m_in.lastQuat=viewParams.rotation;
		m_in.defaultFov=GetISystem()->GetIConsole()->GetCVar("cl_fov")->GetFVal();
		
		m_in.frameTime=min(GetISystem()->GetITimer()->GetFrameTime(),0.1f);
		
		m_in.pCharacter = rPlayer.GetEntity()->GetCharacter(0);
		m_in.pVehicle=rPlayer.GetLinkedVehicle();
		
		m_in.bIsGrabbing=false;
		if (rPlayer.m_grabStats.grabId>0)
		{
			m_in.bIsGrabbing=true;
			// *****
			viewParams.nearplane = 0.1f;
			// *****
		}
		
		m_in.stats_isRagDoll=rPlayer.m_stats.isRagDoll;
		m_in.stats_followCharacterHead=rPlayer.m_stats.followCharacterHead;
		m_in.stats_flatSpeed=rPlayer.m_stats.flatSpeed;	
		m_in.stats_leanAmount=rPlayer.m_stats.leanAmount;
		m_in.stats_inAir=rPlayer.m_stats.inAir;	
		m_io.stats_jumped=rPlayer.m_stats.jumped;
		m_in.stats_firstPersonBody=rPlayer.m_stats.firstPersonBody;	
		m_in.stats_onGround=rPlayer.m_stats.onGround;
		m_io.stats_landed=rPlayer.m_stats.landed;
		m_in.stats_velocity=rPlayer.m_stats.velocity;
	
		m_in.params_viewFoVScale=rPlayer.m_params.viewFoVScale;
		m_in.params_viewPivot=rPlayer.m_params.viewPivot;
		m_in.params_headBobbingMultiplier=rPlayer.m_params.headBobbingMultiplier;
		m_in.params_viewDistance=rPlayer.m_params.viewDistance;
		m_in.params_weaponInertiaMultiplier=rPlayer.m_params.weaponInertiaMultiplier;
		m_in.params_hudAngleOffset=rPlayer.m_params.hudAngleOffset;
		m_in.params_hudOffset=rPlayer.m_params.hudOffset;	
		m_in.params_viewHeightOffset=rPlayer.m_params.viewHeightOffset;
		m_in.params_weaponBobbingMultiplier=rPlayer.m_params.weaponBobbingMultiplier;
		
		m_in.cl_bob=GetISystem()->GetIConsole()->GetCVar("cl_bob")->GetFVal();
		
		m_in.bIsThirdPerson=rPlayer.IsThirdPerson();
		m_in.vEntityWorldPos=rPlayer.GetEntity()->GetWorldPos();
		m_in.entityId=rPlayer.GetEntityId();	
		m_in.entityWorldMatrix=Matrix34(rPlayer.GetEntity()->GetSlotWorldTM(0));
		m_in.entityWorldMatrix2=Matrix34(rPlayer.GetEntity()->GetSlotWorldTM(0));	
		m_in.localEyePos=rPlayer.GetLocalEyePos();
		m_in.worldEyePos=m_in.entityWorldMatrix*m_in.localEyePos;
		
		{
			int16 joint_id = rPlayer.GetBoneID(BONE_HEAD);
	
			if (joint_id>=0)
			{
				m_in.headMtxLocal=Matrix33(m_in.pCharacter->GetISkeleton()->GetAbsJMatrixByID(joint_id));
	//			viewParams.rotation = Quat(rPlayer.GetEntity()->GetSlotWorldTM(0) * m_in.headMtxLocal * Matrix33::CreateRotationY(gf_PI*0.5f));
			}
			else
			{
				m_in.headMtxLocal=Matrix33::CreateIdentity();
			}
		}
	
		//FIXME:dont hook up the cvar each frame, just store the Cvar *
		m_in.thirdPersonDistance=GetISystem()->GetIConsole()->GetCVar("cl_tpvDist")->GetFVal();
		m_in.thirdPersonYaw=GetISystem()->GetIConsole()->GetCVar("cl_tpvYaw")->GetFVal();
		
		m_in.stand_MaxSpeed=rPlayer.GetStanceInfo(STANCE_STAND)->maxSpeed;
	
		m_in.health=rPlayer.GetHealth();
		m_in.stance=rPlayer.m_stance;
		m_in.shake=GetISystem()->GetIConsole()->GetCVar("cl_sprintShake")->GetFVal();
		m_in.standSpeed=rPlayer.GetStanceMaxSpeed(STANCE_STAND);	
	}
	
	// SViewStateInOut -- temporaries and output
	{
		//--- temporaries - FirstThird shared
		
		//m_io.bUsePivot
		//m_io.bobMul
		//m_io.viewMtxForWeapon
		//m_io.eyeOffsetGoal
		
		//--- temporaries - view shared
		m_io.wAngles=Ang3(0,0,0);
		//--- I/O
		//m_io.m_lastPos=rPlayer.m_lastPos;
		// Integ
		//if (m_io.m_lastPos.len2()>0.01f)
		//{
		//	m_io.blendViewOffset = m_io.m_lastPos - rPlayer.GetEntity()->GetWorldPos(); 
		//	m_io.m_lastPos.Set(0,0,0);
		//}
	}
//m_io.blendViewOffset=rPlayer.m_blendViewOffset;
	// *****
	viewParams.fov = m_in.defaultFov*rPlayer.m_params.viewFoVScale*(gf_PI/180.0f);
	viewParams.nearplane = 0.0f;
	// *****
		
//m_io.stats_viewID=rPlayer.m_stats.viewID;
	m_io.stats_HUDPos=rPlayer.m_stats.HUDPos;

	m_io.eyeOffsetGoal=rPlayer.GetStanceViewOffset(rPlayer.m_stance);

	m_io.viewMtxFinal=rPlayer.m_viewMtxFinal;
	m_io.viewMtx=rPlayer.m_viewMtx;

//m_io.blendViewRotOffset=rPlayer.m_blendViewRotOffset;
	

	m_io.stats_FPWeaponAngles=rPlayer.m_stats.FPWeaponAngles;
	m_io.stats_FPWeaponPos=rPlayer.m_stats.FPWeaponPos;

	m_io.vFPWeaponOffset=rPlayer.m_FPWeaponOffset;

	m_io.stats_FPSecWeaponAngles=rPlayer.m_stats.FPSecWeaponAngles;
	m_io.stats_FPSecWeaponPos=rPlayer.m_stats.FPSecWeaponPos;

	//m_io.viewShake=rPlayer.m_viewShake;

	m_io.eyeOffset=rPlayer.m_eyeOffset;
	m_io.baseMtx=rPlayer.m_baseMtx;

	m_io.vFPWeaponAngleOffset=rPlayer.m_FPWeaponAngleOffset;
	m_io.stats_bobCycle=rPlayer.m_stats.bobCycle;

	m_io.vFPWeaponLastDirVec=rPlayer.m_FPWeaponLastDirVec;
	m_io.bobOffset=rPlayer.m_bobOffset;
	m_io.angleOffset=rPlayer.m_angleOffset;
	m_io.viewAngleOffset = rPlayer.m_viewAnglesOffset;

//m_io.stats_viewIDLast=rPlayer.m_stats.viewIDLast;
	m_in.stats_flyMode=rPlayer.m_stats.flyMode;

	m_io.stats_smoothViewZ=rPlayer.m_stats.smoothViewZ;
	m_io.stats_smoothZType=rPlayer.m_stats.smoothZType;

	//-- Any individual PreProcess handlers should be called here

	m_in.iCamMode=GetISystem()->GetIConsole()->GetCVar("cl_cm")->GetIVal();
	m_in.bThirdPersonDamped=false;
	if(m_in.bIsThirdPerson && !m_in.pVehicle)
	{
		if(0==m_in.iCamMode)
		{
			m_in.bThirdPersonDamped=true;
			ViewThirdPersonDampedPre(rPlayer,viewParams,m_in);
		}
	}
}

//--------------------------------------------------------------------------
//--- ViewProcess
//--------------------------------------------------------------------------
// Process our local copy of the data.
// (This is where all the actual work should occur)
//--------------------------------------------------------------------------
void CPlayerViewG4::ViewProcess(SViewParams &viewParams)
{
	FUNCTION_PROFILER(GetISystem(), PROFILE_GAME);

	// Externally controlled first person view e.g. by animation
	if (m_in.stats_isRagDoll && !m_in.bIsThirdPerson && m_in.pCharacter && (m_in.stats_firstPersonBody==1 || m_in.stats_followCharacterHead))
	{
		ViewFollowCharacterFirstPerson(viewParams);
	}
	else if (m_in.pVehicle)
	{
		ViewVehicle(viewParams);
	}
	else
	{
		if(m_in.bThirdPersonDamped)
		{
				ViewThirdPersonDamped(viewParams);
		}
		else
		{
			ViewFirstThirdSharedPre(viewParams);

			if (m_in.bIsThirdPerson || m_io.bUsePivot)
				ViewThirdPerson(viewParams);
			else
				ViewFirstPerson(viewParams);

			ViewFirstThirdSharedPost(viewParams);
		}
	}
}

//--------------------------------------------------------------------------
//--- ViewPostProcess
//--------------------------------------------------------------------------
// Commit all the changes
//--------------------------------------------------------------------------
void CPlayerViewG4::ViewPostProcess(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
	//return;
	// If the view ID changed then blend the position/rotation changes over a few frames
	//if(0!=m_in.iCamMode)
	//	ViewBlendPostProcess(rPlayer,viewParams);

	// update the player rotation if view control is taken from somewhere else (e.g. animation or vehicle)
	if(!m_in.bThirdPersonDamped)
		ViewExternalControlPostProcess(rPlayer,viewParams);

	//set first person weapon position/rotation
	if(!m_in.bThirdPersonDamped)
		FirstPersonWeaponPostProcess(rPlayer,viewParams);

	if(!m_in.bThirdPersonDamped)
		ViewShakePostProcess(rPlayer,viewParams);


	//--------------------------
	// Output changed temporaries - debugging.
	//--------------------------

	//--------------------------
	// Output changed state.
	//--------------------------
	//rPlayer.m_lastPos=m_io.m_lastPos;
//rPlayer.m_blendViewOffset=m_io.blendViewOffset;
//rPlayer.m_stats.viewID=m_io.stats_viewID;
//rPlayer.m_stats.viewIDLast=m_io.stats_viewIDLast;
	rPlayer.m_stats.HUDPos=m_io.stats_HUDPos;
	rPlayer.m_viewMtxFinal=m_io.viewMtxFinal;
	rPlayer.m_viewMtx=m_io.viewMtx;
	//FIXME:updating the baseMatrix due being in a vehicle or having a first person animations playing has to be moved somewhere else
	rPlayer.m_baseMtx=m_io.baseMtx;
//rPlayer.m_blendViewRotOffset=m_io.blendViewRotOffset;
	rPlayer.m_stats.FPWeaponAngles=m_io.stats_FPWeaponAngles;
	rPlayer.m_stats.FPWeaponPos=m_io.stats_FPWeaponPos;
	rPlayer.m_stats.FPSecWeaponAngles=m_io.stats_FPSecWeaponAngles;
	rPlayer.m_stats.FPSecWeaponPos=m_io.stats_FPSecWeaponPos;
	//rPlayer.m_viewShake=m_io.viewShake;

	rPlayer.m_eyeOffset=m_io.eyeOffset;
	rPlayer.m_stats.bobCycle=m_io.stats_bobCycle;

	rPlayer.m_FPWeaponAngleOffset=m_io.vFPWeaponAngleOffset;
	rPlayer.m_FPWeaponLastDirVec=m_io.vFPWeaponLastDirVec;
	rPlayer.m_FPWeaponOffset=m_io.vFPWeaponOffset;
	rPlayer.m_bobOffset=m_io.bobOffset;
	rPlayer.m_angleOffset=m_io.angleOffset;
	rPlayer.m_viewAnglesOffset = m_io.viewAngleOffset;
	rPlayer.m_stats.landed=m_io.stats_landed;
	rPlayer.m_stats.jumped=m_io.stats_jumped;

	//--------------------------
	rPlayer.m_camViewMtxFinal=m_io.camViewMtxFinal;
	//--------------------------

	if(!m_in.bThirdPersonDamped)
		HudPostProcess(rPlayer,viewParams);

	rPlayer.m_stats.shakeAmount = viewParams.shakingRatio;

	if(!m_in.bThirdPersonDamped)
		HandsPostProcess(rPlayer,viewParams);
}


void CPlayerViewG4::ViewFirstThirdSharedPre(SViewParams &viewParams)
{
	viewParams.viewID = 0;

	m_io.bUsePivot = (m_in.params_viewPivot.len2() > 0) && !m_in.bIsThirdPerson;
	m_io.bobMul=m_in.cl_bob * m_in.params_headBobbingMultiplier;

	viewParams.position = m_io.bUsePivot ? m_in.params_viewPivot : m_in.vEntityWorldPos;

	m_io.viewMtxForWeapon=m_io.viewMtxFinal;

	m_io.viewMtxFinal *= Matrix33::CreateRotationXYZ(m_io.angleOffset * max(0.25f,m_io.bobMul) * gf_PI/180.0f);

	viewParams.rotation = GetQuatFromMat33(m_io.viewMtxFinal);

	//Vec3 eyeOffsetGoal(GetStanceViewOffset(m_stance));
}

void CPlayerViewG4::ViewFirstThirdSharedPost(SViewParams &viewParams)
{
	//--- Update the eye offset and apply 
	{
		// Blend towards the goal eye offset
		Interpolate(m_io.eyeOffset,m_io.eyeOffsetGoal,10.0f,m_in.frameTime);

		//apply eye offset
		if (!m_io.bUsePivot)
		{
			//m_eyeOffset represent the shift on the side/forward/up (x,y,z)
			viewParams.position += m_io.baseMtx * m_io.eyeOffset;//m_baseMtx.GetColumn(0) * m_eyeOffset.x + m_baseMtx.GetColumn(1) * m_eyeOffset.y + m_baseMtx.GetColumn(2) * m_eyeOffset.z;

			if (m_in.stats_firstPersonBody==2)
			{
				float lookDown(m_io.viewMtxFinal.GetColumn(1) * m_io.baseMtx.GetColumn(2));
				float forwardOffset(0.0f);
				if (m_in.stance == STANCE_STAND)
					forwardOffset = 0.15f;

				viewParams.position += m_io.viewMtxFinal.GetColumn(1) * max(-lookDown,0.0f) * forwardOffset + m_io.viewMtxFinal.GetColumn(2) * max(-lookDown,0.0f) * 0.0f;
			}
		}
	}

	//--- Weapon orientation
	//FIXME: this should be done in the player update anyway.
	//And all the view position update. (because the game may need to know other players eyepos and such)
	//update first person weapon model position/angles
	{
		Quat wQuat(m_io.viewMtxForWeapon * Matrix33::CreateRotationXYZ(m_io.vFPWeaponAngleOffset * gf_PI/180.0f));
		//wQuat *= Quat::CreateSlerp(viewParams.shakeQuat,IDENTITY,0.5f);
		wQuat *= Quat::CreateSlerp(viewParams.currentShakeQuat,IDENTITY,0.5f);
		wQuat.Normalize();

		m_io.wAngles = Ang3(wQuat);
	}

	//smooth out the view elevation		
	if (m_in.stats_inAir < 0.1f && !m_in.stats_flyMode && !m_io.bUsePivot)
	{
		if (m_io.stats_smoothZType!=1)
		{
			m_io.stats_smoothViewZ = viewParams.position.z;
			m_io.stats_smoothZType = 1;
		}

		Interpolate(m_io.stats_smoothViewZ,viewParams.position.z,9.9f,m_in.frameTime);
		viewParams.position.z = m_io.stats_smoothViewZ;
	}
	else
	{
		if (m_io.stats_smoothZType==1)
		{
			m_io.stats_smoothViewZ = m_in.lastPos.z - m_io.stats_smoothViewZ;
			m_io.stats_smoothZType = 2;
		}

		Interpolate(m_io.stats_smoothViewZ,0.0f,9.9f,m_in.frameTime);
		viewParams.position.z += m_io.stats_smoothViewZ;
	}
}

void CPlayerViewG4::ViewThirdPerson(SViewParams &viewParams)
{
	if (m_in.thirdPersonYaw>0.001f)
	{
		viewParams.rotation *= Quat::CreateRotationXYZ(Ang3(0,0,m_in.thirdPersonYaw * gf_PI/180.0f));
		m_io.viewMtxFinal = Matrix33(viewParams.rotation);
	}

	if (m_io.bUsePivot)			
		viewParams.position += m_io.viewMtxFinal.GetColumn(1) * m_in.params_viewDistance + m_io.viewMtxFinal.GetColumn(2) * (0.25f + m_in.params_viewHeightOffset);
	else
		viewParams.position += m_io.viewMtxFinal.GetColumn(1) * -m_in.thirdPersonDistance + m_io.viewMtxFinal.GetColumn(2) * (0.25f + m_in.params_viewHeightOffset);
}

#if 0
void CPlayerViewG4::ViewThirdPersonDirected(SViewParams &viewParams)
{
	if (m_in.thirdPersonYaw>0.001f)
	{
		viewParams.rotation *= Quat::CreateRotationXYZ(Ang3(0,0,m_in.thirdPersonYaw * gf_PI/180.0f));
		m_io.viewMtxFinal = Matrix33(viewParams.rotation);
	}

	if (m_io.bUsePivot)			
		viewParams.position += m_io.viewMtxFinal.GetColumn(1) * m_in.params_viewDistance + m_io.viewMtxFinal.GetColumn(2) * (0.25f + m_in.params_viewHeightOffset);
	else
		viewParams.position += m_io.viewMtxFinal.GetColumn(1) * -m_in.thirdPersonDistance + m_io.viewMtxFinal.GetColumn(2) * (0.25f + m_in.params_viewHeightOffset);
}
#endif

// jump/land spring effect. Adjust the eye and weapon pos as required.
void CPlayerViewG4::FirstPersonJump(SViewParams &viewParams,Vec3 &weaponOffset)
{
	if (m_in.stats_inAir)
	{
		if (m_io.stats_jumped)
			m_io.eyeOffsetGoal.z -= 0.7f;

		if (m_in.stats_inAir>0.13f)
			m_io.stats_jumped = false;
	}
	else if (m_in.stats_onGround)
	{
		if (m_io.stats_landed)
		{
			m_io.eyeOffsetGoal.z -= (m_in.stance == STANCE_PRONE)?0.1f:0.3f;
			weaponOffset -= m_io.baseMtx.GetColumn(2) * 0.05f;
		}

		if (m_in.stats_onGround>0.1f)
			m_io.stats_landed = false;
	}
}

//
// Extract the various modifiers out into their own function.
//
// Aim is to make the code easier to understand and modify, as 
// well as to ease the addition of new modifiers.
//
void CPlayerViewG4::ViewFirstPerson(SViewParams &viewParams)
{
		//headbob
		Vec3 bobGoal(0,0,0);
		Ang3 angOffset(0,0,0);
		Vec3 weaponOffset(0,0,0);
		Ang3 weaponAngleOffset(0,0,0);

		// jump/land spring effect. Adjust the eye and weapon pos as required.
		FirstPersonJump(viewParams,weaponOffset);

		//float standSpeed(GetStanceMaxSpeed(STANCE_STAND));

		Vec3 vSpeed(0,0,0);
		if (m_in.standSpeed>0.001f)
			vSpeed = (m_in.stats_velocity / m_in.standSpeed);

		float vSpeedLen(vSpeed.len());
		if (vSpeedLen>1.5f)
			vSpeed = vSpeed / vSpeedLen * 1.5f;

		// On the ground.
		if (m_in.stats_inAir < 0.1f)
		{
			float speedMul(0);
			if (m_in.standSpeed>0.001f)
				speedMul=(m_in.stats_flatSpeed / m_in.standSpeed * 1.1f);

			speedMul = min(1.5f,speedMul);


			//--- Bobbing.
			// bobCycle is a speed varying time step running (looping) from 0 to 1 
			// this feeds into a sin eqn creating a double horizontal figure of 8.
			// ( a lissajous figure with the vertical freq twice the horz freq ).

			// To tweak the total speed of the curve:

			// To tweak the effect speed has on the curve:
			const float kSpeedToBobFactor=1.25f;
			// To tweak the width of the bob:
			const float kBobWidth=0.1f;
			// To tweak the height of the bob:
			const float kBobHeight=0.05f;
			// To tweak the scale of strafing lag: (may need to manually adjust the strafing angle offsets as well.)
			const float kStrafeHorzScale=0.05f;


			m_io.stats_bobCycle += m_in.frameTime * kSpeedToBobFactor * speedMul;

			//if player is standing set the bob to rest. (bobCycle reaches 1.0f within 1 second)
			if (speedMul < 0.1f)
				m_io.stats_bobCycle = min(m_io.stats_bobCycle + m_in.frameTime * 1.0f,1.0f);

			// bobCycle loops between 0 and 1
			if (m_io.stats_bobCycle>1.0f)
				m_io.stats_bobCycle = m_io.stats_bobCycle - 1.0f;

			//set the bob offset
			Vec3 bobDir(cry_sinf(m_io.stats_bobCycle*gf_PI*2.0f)*kBobWidth*speedMul,0,cry_sinf(m_io.stats_bobCycle*gf_PI*4.0f)*kBobHeight*speedMul);
			bobGoal = m_io.viewMtxFinal * bobDir;

			//not the bob offset for the weapon
			bobDir *= 0.25f;

			//if player is strafing shift a bit the weapon on left/right
			if (speedMul > 0.01f)
			{
				// right vector dot speed vector
				float dot(m_io.viewMtxFinal.GetColumn(0) * vSpeed);

				bobDir.x -= dot * kStrafeHorzScale;	// the faster we move right, the more the gun lags to the left and vice versa

				//tweak the right strafe for weapon laser
				if (dot>0.0f)
					weaponAngleOffset.z += dot * 1.5f;	// kStrafeHorzScale
				else
					weaponAngleOffset.z -= dot * 2.0f;	// kStrafeHorzScale

				weaponAngleOffset.y += dot * 5.0f;		// kStrafeHorzScale
			}

			weaponOffset += m_io.viewMtxFinal * bobDir;
			weaponOffset -= m_io.baseMtx.GetColumn(2) * 0.035f * speedMul;

			weaponAngleOffset.y += cry_sinf(m_io.stats_bobCycle*gf_PI*2.0f) * speedMul * 1.5f;
			weaponAngleOffset.x += speedMul * 1.5f;

			//FIXME: viewAngles must include all the view offsets, otherwise aiming wont be precise.
			angOffset.x += cry_sinf(m_io.stats_bobCycle*gf_PI*4.0f)*0.7f*speedMul;
		}
		else
		{
			m_io.stats_bobCycle = 0;

			//while flying offset a bit the weapon model by the player speed	
			if (m_in.stats_velocity.len2()>0.001f)
			{				
				float dotFwd(m_io.viewMtxFinal.GetColumn(1) * vSpeed);
				float dotSide(m_io.viewMtxFinal.GetColumn(0) * vSpeed);
				float dotUp(m_io.viewMtxFinal.GetColumn(2) * vSpeed);

				weaponOffset += m_io.viewMtxFinal * Vec3(dotSide * -0.05f,dotFwd * -0.035f,dotUp * -0.05f);

				weaponAngleOffset.x += dotUp * 2.0f;
				weaponAngleOffset.y += dotSide * 5.0f;
				weaponAngleOffset.z -= dotSide * 2.0f;
			}
		}

		//add some inertia to weapon due view direction change.
		float deltaDotSide(m_io.vFPWeaponLastDirVec * m_io.viewMtxFinal.GetColumn(0));
		float deltaDotUp(m_io.vFPWeaponLastDirVec * m_io.viewMtxFinal.GetColumn(2));

		weaponOffset += m_io.viewMtxFinal * Vec3(deltaDotSide * 0.1f + m_in.stats_leanAmount * 0.05f,0,deltaDotUp * 0.1f - fabs(m_in.stats_leanAmount) * 0.05f) * m_in.params_weaponInertiaMultiplier;

		weaponAngleOffset.x -= deltaDotUp * 5.0f * m_in.params_weaponInertiaMultiplier;
		weaponAngleOffset.z += deltaDotSide * 5.0f * m_in.params_weaponInertiaMultiplier;
		weaponAngleOffset.y += deltaDotSide * 5.0f * m_in.params_weaponInertiaMultiplier;

		weaponAngleOffset.y += m_in.stats_leanAmount * 5.0f;
		//

		//apply some multipliers
		weaponOffset *= m_in.params_weaponBobbingMultiplier;
		bobGoal *= m_io.bobMul;

		Interpolate(m_io.vFPWeaponOffset,weaponOffset,7.0f,m_in.frameTime);
		Interpolate(m_io.vFPWeaponAngleOffset,weaponAngleOffset,7.0f,m_in.frameTime);
		Interpolate(m_io.vFPWeaponLastDirVec,m_io.viewMtxFinal.GetColumn(1),5.0f,m_in.frameTime);

		Interpolate(m_io.angleOffset,angOffset,10.0f,m_in.frameTime);
		Interpolate(m_io.bobOffset,bobGoal,7.0f,m_in.frameTime);

		viewParams.position += m_io.bobOffset;
}

void CPlayerViewG4::ViewVehicle(SViewParams &viewParams)
{
	if (m_in.pVehicle)
	{
		m_in.pVehicle->UpdateView(viewParams, m_in.entityId);
		viewParams.viewID = 2;

		Ang3 offset = m_io.viewAngleOffset;
		Vec3 viewDir = viewParams.rotation.GetColumn1();
		Matrix33 viewMat = Matrix33::CreateRotationVDir(viewDir);
		Matrix33 offmat = Matrix33::CreateRotationXYZ(offset);
		viewMat *= offmat;
		Quat rot = Quat::CreateRotationXYZ(Ang3::GetAnglesXYZ(viewMat));
		viewParams.rotation = rot;

	}
}

//FIXME:use the animated character view filter for this
void CPlayerViewG4::ViewFollowCharacterFirstPerson(SViewParams &viewParams)
{
	viewParams.viewID = 3;

	//to avoid clipping
	viewParams.nearplane = 0.1f;
	viewParams.position = m_in.entityWorldMatrix *m_in.localEyePos;

	if (m_in.stats_isRagDoll)
		viewParams.position.z += 0.05f;

	if (m_in.stats_followCharacterHead)
	{
		viewParams.rotation = Quat(Matrix33(m_in.entityWorldMatrix)*m_in.headMtxLocal* Matrix33::CreateRotationY(gf_PI*0.5f));
	}
}

#if 0
// If the view type (viewID) has changed then blend the position/rotation changes over a few frames.
void CPlayerViewG4::ViewBlendPostProcess(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
	//if necessary blend the view
	if (m_io.stats_viewIDLast != m_io.stats_viewID)
	{
		m_io.blendViewOffset = m_in.lastPos - viewParams.position;

		m_io.blendViewRotOffset = (m_in.lastQuat / viewParams.rotation).GetNormalized();

		m_io.stats_viewIDLast = m_io.stats_viewID;
	}
	else
	{
		Interpolate(m_io.blendViewOffset,ZERO,10.0f,m_in.frameTime);
		m_io.blendViewRotOffset = Quat::CreateSlerp( m_io.blendViewRotOffset, IDENTITY, m_in.frameTime * 5.0f );
	}

	viewParams.position += m_io.blendViewOffset;
	viewParams.rotation *= m_io.blendViewRotOffset;
}
#endif


// View was set from an external source such as a vehicle, or followCharacterHead, so update the players rotation to match.
void CPlayerViewG4::ViewExternalControlPostProcess(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
	// update the player rotation if view control is taken from somewhere else
	if (m_in.health>0 && (m_in.pVehicle || m_in.stats_followCharacterHead))
	{
		Vec3 forward = viewParams.rotation.GetColumn1().GetNormalizedSafe();
		Vec3 up = Vec3(0,0,1);

		m_io.baseMtx.SetFromVectors(forward % up,forward,up);
		//m_baseMtx.NoScale();

		m_io.viewMtx = m_io.viewMtxFinal = Matrix33(viewParams.rotation);
		//m_input.viewVector = viewParams.rotation.GetColumn1();
	}
}

// Position the first person weapons
void CPlayerViewG4::FirstPersonWeaponPostProcess(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
	//set first person weapon position/rotation
	Vec3 shakeVec(viewParams.currentShakeShift *0.85f);
	m_io.stats_FPWeaponAngles = m_io.wAngles;
	m_io.stats_FPWeaponPos = viewParams.position + m_io.vFPWeaponOffset + shakeVec;

	m_io.stats_FPSecWeaponAngles = m_io.wAngles;
	m_io.stats_FPSecWeaponPos = viewParams.position + m_io.vFPWeaponOffset + shakeVec;
}

// Shake the view as requested
void CPlayerViewG4::ViewShakePostProcess(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
#if 0
	if (m_io.viewShake.amount>0.001f)
	{
		viewParams.shakeDuration = m_io.viewShake.duration;
		viewParams.shakeFrequency = m_io.viewShake.frequency;
		viewParams.shakeAmount = m_io.viewShake.angle;
		viewParams.shakeID = 1;

		memset(&m_io.viewShake,0,sizeof(SViewShake));
	}
	/*else if (IsZeroG() && m_stats.flatSpeed > 10.0f)
	{
	viewParams.shakeDuration = 0.1f;
	viewParams.shakeFrequency = 0.1f;
	viewParams.shakeAmount = Ang3(m_stats.flatSpeed*0.001f,0,0);
	}*/
	else if (m_in.stats_inAir < 0.1f && m_in.stats_flatSpeed > m_in.stand_MaxSpeed * 1.1f)
	{
		viewParams.shakeDuration = 0.1f ;/// (shake*shake);
		viewParams.shakeFrequency = 0.1f ;/// (shake*shake);
		viewParams.shakeAmount = Ang3(m_in.stats_flatSpeed*0.0015f,m_in.stats_flatSpeed*0.00035f,m_in.stats_flatSpeed*0.00035f) * m_in.shake;
		viewParams.shakeID = 0;
	}
#endif
	//testing code
	/*if (m_stats.inAir < 0.1f && m_stats.flatSpeed > GetStanceInfo(STANCE_STAND)->maxSpeed * 1.1f)
	{
	float shake(GetISystem()->GetIConsole()->GetCVar("cl_sprintShake")->GetFVal());

	IView *pView = g_pGame->GetIGameFramework()->GetIViewSystem()->GetViewByEntityId(GetEntityId());
	if (pView)
	pView->SetViewShake(ZERO,Vec3(m_stats.flatSpeed*0.0035f,0,m_stats.flatSpeed*0.0035f) * shake,0.1f,0.05f,0.5f,1);
	}*/
}

// Locate the HUD in 3d space so that it appears correctly on screen.
void CPlayerViewG4::HudPostProcess(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
	Quat hudQuat(Matrix33(viewParams.rotation) * Matrix33::CreateRotationXYZ(m_in.params_hudAngleOffset));
	hudQuat *= Quat::CreateSlerp(viewParams.currentShakeQuat,IDENTITY,0.05f);
	hudQuat.Normalize();

	Matrix33 hudMtx(hudQuat);

	rPlayer.m_stats.HUDAngles = Ang3::GetAnglesXYZ(hudMtx);
	rPlayer.m_stats.HUDPos = viewParams.position - rPlayer.m_FPWeaponOffset * 0.3f;
	rPlayer.m_stats.HUDPos += hudMtx.GetColumn(2)*rPlayer.m_params.hudOffset.z;
	rPlayer.m_stats.HUDPos += hudMtx.GetColumn(1)*rPlayer.m_params.hudOffset.y;
	rPlayer.m_stats.HUDPos += hudMtx.GetColumn(0)*rPlayer.m_params.hudOffset.x;

	Script::CallMethod(rPlayer.GetEntity()->GetScriptTable(), "OnUpdateView", m_in.frameTime);

	GetISystem()->GetIGame()->GetIGameFramework()->GetIGameTokenSystem()->SetOrCreateToken("hud.offsetx",TFlowInputData(rPlayer.m_FPWeaponOffset.x,true));
	GetISystem()->GetIGame()->GetIGameFramework()->GetIGameTokenSystem()->SetOrCreateToken("hud.offsety",TFlowInputData(rPlayer.m_FPWeaponOffset.z,true));
}

// If there are first person hands present, then position and orient them relative to the view
void CPlayerViewG4::HandsPostProcess(CPlayerG4 &rPlayer,SViewParams &viewParams)
{
	//for testing sake
	float nearPlaneOverride(GetISystem()->GetIConsole()->GetCVar("cl_nearPlane")->GetFVal());
	if (nearPlaneOverride > 0.001f)
		viewParams.nearplane = nearPlaneOverride;

	//FIXME:find a better place?
	// Convention: if Player has a renderable object in slot 2 then it's the first person hands.
	if (rPlayer.GetEntity()->GetSlotFlags(2) & ENTITY_SLOT_RENDER)
	{
		Matrix34 handsMtx(rPlayer.m_viewMtxFinal * Matrix33::CreateRotationZ(gf_PI),viewParams.position);
		rPlayer.GetEntity()->SetSlotLocalTM(2,rPlayer.GetEntity()->GetWorldTM().GetInverted() * handsMtx);
		ICharacterInstance *pHands = rPlayer.GetEntity()->GetCharacter(2);
		if (pHands)
		{
			pHands->GetISkeleton()->SetAlwaysUpdate(1);
			pHands->Update(handsMtx, GetISystem()->GetViewCamera(), 0);
		}
	}
}

#if G4_VIEW

#define RWI             (rwi_ignore_noncolliding |  rwi_stop_at_pierceable)

struct TCamCut
{
	float Dist;

	float vOff;
	float hOff;

	float roll;
	float pitch;
	float yaw;
	float intensity;
	int iSubject;
	int iDof;
	int iType;
};

// Helper so nums can be entered in same order they come from the console when editing.
#define CAM_DHPRVY(d,h,p,r,v,y,i,tgt,dof,typ)   { d,v,h,r,p,y,i,tgt,dof,typ }

#define CAM_ILOW      0.25f
#define CAM_IMED      0.5f
#define CAM_IHIGH     0.75f

#define CAM_PLAYER      1
#define CAM_PLAYERHEAD  2
#define CAM_PLAYERWEAP  3

#define CAM_GIRL				10
#define CAM_GIRLHEAD		11
#define CAM_GIRLWEAP		12

#define CAM_PLAYERGIRL			20
#define CAM_PLAYERGIRLCLOSE	21


#define CAM_TARGET      30
#define CAM_TARGETHEAD  31
#define CAM_TARGETWEAP  32


#define CAM_STATIC      1
#define CAM_PAN         2
#define CAM_FOLLOW      3

#define NAV_CAM     0
#define COM_CAM     1
#define COM_CAM2    2


#define CAM_DOFNONE     0
#define CAM_DOFLOW      1
#define CAM_DOFMED      2
#define CAM_DOFHIGH     3
#define CAM_DOFEXTREME  4
#define CAM_DOFAUTO     5

TCamCut aCamCutsFollow[]=
{
	// Nav cam - behind player
	CAM_DHPRVY(1.3f,-0.0632f,13.78f,0,-0.154f,13.19f,CAM_ILOW,CAM_PLAYER,CAM_DOFNONE,CAM_FOLLOW),
	// Combat cam - behind and to right of player
	CAM_DHPRVY(1.3f,-0.374f,0.949f,0,-0.0854f,17.597f,CAM_ILOW,CAM_PLAYER,CAM_DOFNONE,CAM_FOLLOW),
	// Extreme Close over shoulder
	CAM_DHPRVY(-0.511f,-0.2425f,-19.972f,0,-0.1239f,-157.68f,CAM_IMED,CAM_PLAYER,CAM_DOFNONE,CAM_FOLLOW)
};

int nCamCutsFollow=sizeof(aCamCutsFollow)/sizeof(aCamCutsFollow[0]);

#define  SET_CamCut(pCut)                         \
{ thirdPersonDistance=pCut->Dist;                 \
	thirdPersonRoll=pCut->roll*(gf_PI/180.0f);      \
	thirdPersonPitch=pCut->pitch*(gf_PI/180.0f);    \
	thirdPersonYaw=pCut->yaw*(gf_PI/180.0f);        \
	thirdPersonVOff=pCut->vOff;                     \
	thirdPersonHOff=pCut->hOff; }

//-------------------------------------------------------------------------------
//Bone names:
// "Bip01"
//		This is a special one that is on the ground, and linearly interpolates over the animation.
// ----------------------------
// In order from hips up to head:
// "Bip01 Pelvis"
// "Bip01 Spine"
// "Bip01 Spine1"
// "Bip01 Spine2"
// "Bip01 Spine3"
// "Bip01 Neck"
// "Bip01 Head"
// ----------------------------
// "eye_left_bone"
// "eye_right_bone"
//
// "Bip01 R Clavicle"	(shoulder joint)
// "Bip01 R UpperArm"
// "Bip01 R ForeArm"
// "Bip01 R Hand"
//
// "Bip01 L Clavicle"	(shoulder joint)
// "Bip01 L UpperArm"
// "Bip01 L ForeArm"
// "Bip01 L Hand"
//
// "weapon_bone"		( at the right hand )
//
// "Bip01 L Thigh"
// "Bip01 L Calf"
// "Bip01 L Foot"
// "Bip01 L Heel"
// "Bip01 L Toe0"			( Ball of the foot )
// "Bip01 L Toe0Nub"	( The toes (big toe))
//
// "Bip01 R Thigh"
// "Bip01 R Calf"
// "Bip01 R Foot"
// "Bip01 R Heel"
// "Bip01 R Toe0"			( Ball of the foot )
// "Bip01 R Toe0Nub"	( The toes (big toe))


inline Vec3 GetSmoothMoveDir(EntityId id)
{
	IEntity *pEnt=GetISystem()->GetIEntitySystem()->GetEntity(id);

	Vec3 vSmoothDir=Vec3(0,1,0);
	if(pEnt)
	{
		ICharacterInstance * pCharacter = pEnt->GetCharacter(0);
		if (pCharacter)
		{			
			ISkeleton * pSkeleton = pCharacter->GetISkeleton();
			if(pSkeleton)
			{
				int idJoint=pSkeleton->GetIDByName("Bip01 Spine3");
				if(-1!=idJoint)
				{
					// right= Matrix.GetColumn(0), forward=Matrix.GetColumn(1), up=Matrix.GetColumn(2)
					//Vec3 vNeckDir0=pSkeleton->GetAbsJMatrixByID(idJoint).GetColumn(0);
					//Vec3 vSmoothDir=pSkeleton->GetAbsJMatrixByID(idJoint).GetColumn(1);
					Vec3 vSmoothDir=pSkeleton->GetAbsJMatrixByID(idJoint).GetColumn(1);//pSkeleton->GetMoveDirection();
					//vSmoothDir=pEnt->GetRotation()*vSmoothDir;
				}
			}
		}
	}
	
	//
		float yaw,pitch,dummy;
		CartesianToSpherical(vSmoothDir,yaw,pitch,dummy);
	//

	vSmoothDir=pEnt->GetRotation()*vSmoothDir;

	//
		static float green[4] = {0,1,0,1};
		float yaw2,pitch2;
		CartesianToSpherical(vSmoothDir,yaw2,pitch2,dummy);
		IRenderer *pRend=GetISystem()->GetIRenderer();
		//pRend->Draw2dLabel( 10, 10, 2.0f, green, false, "%s: smoothdir: (%0.3f,%0.3f) (%0.3f,%0.3f)", pEnt->GetName(),yaw*180/gf_PI,pitch*180/gf_PI,yaw2*180/gf_PI,pitch2*180/gf_PI );
	//
	return vSmoothDir;
}


void CPlayerViewG4::ViewThirdPersonDampedPre(const CPlayerG4 &rPlayer,SViewParams &viewParams,SViewStateIn & m_in)
{
	iDrawAngles=0;
	//Vec3 posRef,posDesired;
	//Quat rotRef,rotDesired;
	//-- 1.1 Apply player entity angle offsets and bob.

	//--- Get reference position and orientation:
	// reference pos is player pos
	m_in.posRef=m_in.vEntityWorldPos;
	m_in.vCoverDir=rPlayer.m_cover.vCoverDir;
	m_in.modes=rPlayer.m_modes;
	m_io.rotRef=Quat::CreateIdentity();
	m_in.eMode=rPlayer.m_pAutoCombat->GetCurrentMode();

	// if not in cover, then reference orientation is player movement orientation
	if(!m_in.modes.bCoverMode)
	{
		//m_io.rotRef=Quat::CreateRotationVDir(m_in.entityWorldMatrix.GetColumn(1));
		m_io.rotRef=Quat::CreateRotationVDir(GetSmoothMoveDir(m_in.entityId));
	}
	else	// otherwise is defined by cover (orientation when entering cover).
	{
		m_io.rotRef=Quat::CreateRotationVDir(m_in.vCoverDir);
	}

	//-- 1.2 Apply third person cam parameters:- angles, distance, and horizontal and vertical offsets.    
	float thirdPersonDistance;
	float thirdPersonYaw;
	float thirdPersonPitch;
	float thirdPersonRoll;
	float thirdPersonHOff;
	float thirdPersonVOff;

	float thirdPersonYaw2=0;
	float thirdPersonPitch2=0;

	//--- WIP
	eGamePlayMode eMode=m_in.eMode;

	{
		thirdPersonYaw2=GetISystem()->GetIConsole()->GetCVar("cl_tpvYawNav")->GetFVal();

		thirdPersonYaw2*=gf_PI/180.0f;

		thirdPersonPitch2=GetISystem()->GetIConsole()->GetCVar("cl_tpvPitchNav")->GetFVal();

		if(abs(thirdPersonPitch2)>65)
		{
			if(thirdPersonPitch2>65)
				thirdPersonPitch2=65;
			else
				thirdPersonPitch2=-65;
		}

		thirdPersonPitch2*=gf_PI/180.0f;

		TCamCut *pCut;
		const char *psName;
		float distCam;

		switch (eMode)
		{
		case eCombatMode:
			pCut=&aCamCutsFollow[COM_CAM];
			psName="Combat";
			distCam=GetISystem()->GetIConsole()->GetCVar("cl_tpvDistCombat")->GetFVal();
			break;
		default:
		case eNavigationMode:
			pCut=&aCamCutsFollow[NAV_CAM];
			psName="Follow";
			distCam=GetISystem()->GetIConsole()->GetCVar("cl_tpvDistNav")->GetFVal();
			break;
		}
		SET_CamCut(pCut);
		thirdPersonDistance=distCam;
		//pEditActiveCut=pCut;
		//psEditActiveCutName=psName;
	}

	float thirdPersonPosDamping;
	float thirdPersonAngDamping;

	float kVel;
	float kDist;
	float desiredFOV;

	if(eNavigationMode == eMode)
	{
		desiredFOV=GetISystem()->GetIConsole()->GetCVar("cl_tpvFOVNav")->GetFVal();
		thirdPersonPosDamping=GetISystem()->GetIConsole()->GetCVar("cl_tpvPosDampingNav")->GetFVal();
		thirdPersonAngDamping=GetISystem()->GetIConsole()->GetCVar("cl_tpvAngDampingNav")->GetFVal();
		thirdPersonHOff+=GetISystem()->GetIConsole()->GetCVar("cl_tpvHOffNav")->GetFVal();
		thirdPersonVOff+=GetISystem()->GetIConsole()->GetCVar("cl_tpvVOffNav")->GetFVal();
		kVel=GetISystem()->GetIConsole()->GetCVar("cl_tpvDeltaVelNav")->GetFVal();
		kDist=GetISystem()->GetIConsole()->GetCVar("cl_tpvDeltaDistNav")->GetFVal();

		thirdPersonYaw=0;
		thirdPersonPitch=0;
	}
	else
	{
		desiredFOV=GetISystem()->GetIConsole()->GetCVar("cl_tpvFOVCombat")->GetFVal();

		thirdPersonYaw2=0;

		//--- Nick: add pitch control back in for now.
		thirdPersonPitch=0;	//thirdPersonPitch2=0;

		thirdPersonPosDamping=GetISystem()->GetIConsole()->GetCVar("cl_tpvPosDamping")->GetFVal();
		thirdPersonAngDamping=GetISystem()->GetIConsole()->GetCVar("cl_tpvAngDamping")->GetFVal();
		thirdPersonHOff+=GetISystem()->GetIConsole()->GetCVar("cl_tpvHOffCombat")->GetFVal();
		thirdPersonVOff+=GetISystem()->GetIConsole()->GetCVar("cl_tpvVOffCombat")->GetFVal();
		kVel=GetISystem()->GetIConsole()->GetCVar("cl_tpvDeltaVel")->GetFVal();
		kDist=GetISystem()->GetIConsole()->GetCVar("cl_tpvDeltaDist")->GetFVal();
	}
	if(desiredFOV < 5.0f) desiredFOV=5.0f;
	if(desiredFOV > 178.0f) desiredFOV=178.0f;
	desiredFOV*=g_PI/180.0f;	// was Nav: 60 degrees, Combat: 45 degrees

	thirdPersonYaw+=thirdPersonYaw2;
	//thirdPersonPitch=0;//+=thirdPersonPitch2;
	thirdPersonPitch+=thirdPersonPitch2;


	SCamState desired;

	desired.pos.dist=thirdPersonDistance;
	desired.pos.pitch=thirdPersonPitch;
	desired.pos.yaw=thirdPersonYaw;

	desired.look.pitch=thirdPersonPitch;
	desired.look.yaw=thirdPersonYaw;
	desired.look.roll=thirdPersonRoll;

	desired.look.focalDist=5000;
	desired.look.focalWidth=5000;
	desired.look.focalAmount=0;

	desired.effect.fov=viewParams.fov;
	desired.effect.dofAmount=desired.look.focalAmount;
	desired.effect.dofFar=desired.look.focalDist+desired.look.focalWidth;
	desired.effect.dofNear=desired.look.focalDist-desired.look.focalWidth;

	if(desired.effect.dofNear<0.01f)
		desired.effect.dofNear=0.01f;

	m_io.rotDesired=m_io.rotRef;
	Quat rotPos=m_io.rotRef;

	float yawSph,pitchSph,distSph;
	Vec3 vDir=m_io.rotDesired*Vec3(0,-1,0);
	CartesianToSpherical(vDir,yawSph,pitchSph,distSph);
	static bool bAddDesired=true;
	if(bAddDesired)
	{
		yawSph+=desired.look.yaw;
		pitchSph+=desired.look.pitch;
		distSph=thirdPersonDistance;
	}

	if(distSph<0.1f)	distSph=0.1f;
	//vDir=SphericalToCartesian(yawSph,g_PI/2.0f,1.0f);
	vDir=SphericalToCartesian(yawSph,pitchSph,1.0f);

	m_io.rotDesired=Quat::CreateRotationVDir(vDir.GetNormalized());
	rotPos=Quat::CreateRotationVDir(vDir.GetNormalized());

	m_io.camViewMtxFinal = Matrix33(m_io.rotDesired);
	Matrix33 matPos=Matrix33(rotPos);

	m_io.posDesired=m_in.posRef;
	m_io.posDesired += matPos.GetColumn(1) * -thirdPersonDistance + matPos.GetColumn(2) * (rPlayer.m_params.viewHeightOffset + 0.25f);//(0.25f + m_params.viewHeightOffset + thirdPersonVOff);
	//posDesired += matPos.GetColumn(2) * (rPlayer.m_params.viewHeightOffset + 0.25f);//(0.25f + m_params.viewHeightOffset + thirdPersonVOff);


	//--------------------------------------
	m_io.state=desired;
	m_io.state.pos.vPos=m_io.posDesired;
	m_io.state.pos.posDamping=thirdPersonPosDamping;
	m_io.state.pos.angDamping=thirdPersonAngDamping;
	m_io.state.pos.horzFraming=thirdPersonHOff;
	m_io.state.pos.vertFraming=thirdPersonVOff;

	m_io.state.look.angDamping=thirdPersonAngDamping;
// try m_io.rotRef;
	m_io.state.look.rotation=m_io.rotDesired;
	m_io.state.look.focalAmountDamping=thirdPersonAngDamping;	// temp, needs own value
	m_io.state.look.focalDistDamping=thirdPersonPosDamping;		// temp, needs own value
	m_io.state.look.focalWidthDamping=thirdPersonPosDamping;		// temp, needs own value
	m_io.state.effect.fovDamping=thirdPersonAngDamping;				// temp, needs own value
	m_io.state.effect.fov=desiredFOV;//viewParams.fov;

}
//-------------------------------------------------------------------------------
// dampTime affects the rate of convergence. 
template <typename TVal,typename TFloat>
inline TVal InterpolateDamped(TFloat dt,TFloat dampTime,const TVal & v0, const TVal & v1, TVal & velInOut)
{
	if(dampTime<TFloat(0.001)) dampTime=TFloat(0.001);

	TFloat damping=TFloat(2)/dampTime;
	TFloat e = TFloat(exp(-damping*dt));	// exp(-x) == 1/exp(x)
	TVal dV=v0-v1;
	TVal temp=(velInOut+dV*damping)*dt;
	velInOut=(velInOut-temp*damping)*e;
	return (v1+(dV+temp)*e);
}

// Adjusts a pair of angles to maintain relative numerical/directional distance as they
// cross into/out of negative angle space.
// i.e after being adjusted they should be ready for interpolation. Assumes less than 180 degree increment per call
inline void AdjustAnglePair(float &alpha,float &alphaPrev,float &beta,float &betaPrev,int &nAlphaWraps,int &nBetaWraps)
{
	// Track which direction each angle entered/left negative angle space
	if((alpha<0) && (alphaPrev>=0))
	{
		if((alphaPrev-alpha)<180.0f*g_PI/180.0f)	// clockwise transition past 0
		{
		}
		else													// anticlockwise transition past +/- 180
		{
			++nAlphaWraps;
		}
	}
	else if((alpha>=0) && (alphaPrev<0))
	{
		if((alphaPrev+360*g_PI/180.0f-alpha)<180.0f*g_PI/180.0f)	// clockwise transition past +/- 180
			--nAlphaWraps;
	}

	// Track which direction each angle entered/left negative angle space
	if((beta<0) && (betaPrev>=0))
	{
		if((betaPrev-beta)<180.0f*g_PI/180.0f)	// clockwise transition past 0
		{
		}
		else													// anticlockwise transition past +/- 180
		{
			++nBetaWraps;
		}
	}
	else if((beta>=0) && (betaPrev<0))
	{
		if((betaPrev+360*g_PI/180.0f-beta)<180.0f*g_PI/180.0f)	// clockwise transition past +/- 180
			--nBetaWraps;
	}

	betaPrev=beta;
	alphaPrev=alpha;

	if((alpha+nAlphaWraps*360*g_PI/180)>(beta+nBetaWraps*360*g_PI/180))
	{
		if(alpha<beta)
			alpha+=360*g_PI/180;
	}
	else
	{
		if(beta<alpha)
			beta+=360*g_PI/180;
	}
}

void CPlayerViewG4::ViewThirdPersonDamped(SViewParams &viewParams)
{
	Vec3 targetPos=m_in.vEntityWorldPos;

	targetPos.z+=0.25f;

	Vec3 prevTargetPos=viewParams.targetPos;
	EntityId idPrevTarget=viewParams.idTarget;
	EntityId idTarget=m_in.entityId;
	Vec3 prevCamPos=viewParams.position;
	Quat prevCamRot=viewParams.rotation;
	float pos_kStiffness=15.0f;
	float pos_kDamping=0.0f;
	float ang_kStiffness=20.0f;
	float ang_kDamping=0.0f;
	float ang_maxRate=gf_PI/0.5f;   // 0.5 sec to turn cam by 180
	float prevDist=viewParams.dist;
	Vec3 vSavedTargetPos=targetPos;
	float thirdPersonPosDamping;
	float thirdPersonAngDamping;
	ang_maxRate=gf_PI/1.5f;

	eGamePlayMode eMode= m_in.eMode;

	float dt=viewParams.frameTime;
	// View is system called by CCryAction::PreUpdate which currently passes in a max frame time of 0.1 sec
	// we'll clamp to max of 1 sec here, and base our clamping of the damping factors upon that.
	if(dt>1.0f)
		dt=1.0f;

	//
	//--- 1. Calculate the desired (undamped/instantaneous) view position and angle
	//
	//SCamState m_io.state;
	//UVGetDesiredCameraState(pPlayer,viewParams,viewParams.position,viewParams.rotation,fov,thirdPersonPosDamping,thirdPersonAngDamping);
	//UVGetDesiredCameraStatePre(pPlayer,viewParams);

	viewParams.position=m_io.state.pos.vPos;
	viewParams.rotation=m_io.state.look.rotation;
	//viewParams.rotation=Quat::CreateIdentity(); //m_io.state.look.rotation;

DrawAngles("in",viewParams.rotation);
DrawAngles("prev",prevCamRot);

	static float fovVel=0.0f;
	float fovPrev=viewParams.fov;

	float kFovDamp=GetISystem()->GetIConsole()->GetCVar("cl_tpvFOVDamping")->GetFVal();
	float fovDamped;

	if(kFovDamp>=0.01f)
		fovDamped=InterpolateDamped(dt,kFovDamp,fovPrev*1000,m_io.state.effect.fov*1000,fovVel)/1000;
	else
		fovDamped=m_io.state.effect.fov;

	//viewParams.fov=m_io.state.effect.fov;
	viewParams.fov=fovDamped;

	thirdPersonPosDamping=m_io.state.pos.posDamping;
	thirdPersonAngDamping=m_io.state.pos.angDamping;
	float thirdPersonHOff = m_io.state.pos.horzFraming;
	float thirdPersonVOff = m_io.state.pos.vertFraming;

	//viewParams.fov=m_io.state.effect.fov;

	// Hack for build
	float kVel=GetISystem()->GetIConsole()->GetCVar("cl_tpvDeltaVelNav")->GetFVal();
	float kDist=GetISystem()->GetIConsole()->GetCVar("cl_tpvDeltaDistNav")->GetFVal();


	//--- Collision ----------------------------------------------------------------
	ray_hit rayHit;
	Vec3 vDirLen=(viewParams.position-targetPos);

DrawAnglesDirVec("vDirLen",vDirLen);
	float distLen = vDirLen.len();
	const float safetyDistUp=0.05f;
	const float safetyDist=0.7f;
	const float safetyDistLook=0.7f;
	const float safetySideDist=1.0f;

// WIP
if(0) //distLen>0.01f)
	{
		Vec3 vLook=vDirLen.GetNormalizedSafe(Vec3(0,1,0));
		Vec3 vRight=-vLook.Cross(Vec3(0,0,1));
		vDirLen+=vDirLen.GetNormalized()*safetyDist;
		float tL=1,tR=1,tM=1,tMR=1,tML=1;
		//vHitDir/=distHit;

		// Obstruction between target and camera?
		//if(0)
		// centre
		IEntity *pEnt=GetISystem()->GetIEntitySystem()->GetEntity(m_in.entityId);
		IPhysicalEntity *pPhys=pEnt ? pEnt->GetPhysics() : 0;
		if(pPhys)
		{
			if (GetISystem()->GetIPhysicalWorld()->RayWorldIntersection(targetPos, vDirLen, 
				ent_terrain|ent_static,RWI, &rayHit, 1,pPhys, 0))
			{
				float newDist=rayHit.dist-safetyDistLook;   // move closer to target to get past obstruction
				if(newDist<=0.1f)
					newDist=0.1f;
				float t=newDist/distLen;
				if(t>1) t=1;
				tM=t;
				//float dHgt=(1.0f-t)*safetyDistUp;
				//viewParams.position=targetPos+Vec3(0,0,dHgt)+newDist*vDirLen.GetNormalized();
				//targetPos.z+=dHgt;
			}
		}
		{
			tM=min(tM,min(tL,tR));
		}
		//else
		{
			tM=min(tM,min(tML,tMR));
		}
		float dHgt=(1.0f-tM)*safetyDistUp;
		viewParams.position=targetPos+Vec3(0,0,dHgt)+tM*vDirLen;
		targetPos.z+=dHgt;
				// Now from right of camera =>  flag for move to left if obstr within 5 m
				// Now from left of camera => flag for move to right if obstr within 5 m
				// if bLeft, or bRight then move left or right, else move up, and point down more
	}

	//------------------------------------------------------------------------------

	Vec3 dir = targetPos - viewParams.position;
	float dist=dir.len();
DrawAnglesDirVec("dir",dir);
	//if(v.bWasCut || (idPrevTarget!=idTarget))  // New Camera or target ?
//WIP
if(0)//	if((idPrevTarget!=idTarget))  // New Camera or target ?
	{
		prevTargetPos=targetPos;
		prevCamPos=viewParams.position;
		prevCamRot=viewParams.rotation;
		prevDist=dist;
		viewParams.angleVel=0;
		viewParams.vel=0;
		//v.bWasCut=false;
	}

	Vec3 vMove=targetPos-prevTargetPos;
	float moveDist=vMove.len();
	float desiredDist=dist;
	if( ((moveDist*moveDist)>0.0001f) && ((dist*dist)>0.0001f) )
	{
		Vec3 vDir=dir/dist;
		float alignedDist;
		vMove.Normalize();
		alignedDist= vMove.Dot(vDir) * moveDist;

		// used normalised vMove so we can play with the mix of dist(cam) and dist(target movement)
		if(alignedDist*dist >= 0 )  // same sign
			desiredDist+=alignedDist;
		else
			desiredDist=dist;
	}
	else
	{
		vMove=Vec3(0,1,0);
	}


	//-- viewParams position and rotation now have the desired undamped pos and rotation.

	//
	//--- 2. Apply damping to smooth the changes in position and angle
	//

	//-- 2.1. Get/Calc damping parameters.

	// Sqrt gives finer curve control for lower damping values.
	pos_kDamping = 2*(float)(cry_sqrtf(thirdPersonPosDamping));   // default: 2*sqrt(15)
	ang_kDamping = 2*(float)(cry_sqrtf(thirdPersonAngDamping));   // default: 2*sqrt(20)

	// These need to be clamped to prevent range errors further on, since they are used in an exp() expression
	// dt is clamped to a max of 1 second above.
	// e=(float) cry_expf(-ang_kDamping*0.5f*dt); // e=(float) cry_expf(-pos_kDamping*0.5f*dt);
	// limiting the input to expf to -/+40 limits our output to approx:  4e-18 .. 2e+17

	ang_kDamping = ang_kDamping<-80 ? -80 : ang_kDamping>80 ? 80 : ang_kDamping;
	pos_kDamping = pos_kDamping<-80 ? -80 : pos_kDamping>80 ? 80 : pos_kDamping;



	//-- 2.2. Calculate the desired angle and distance changes in a form suitable for damping.

	// Calc the change in angle(dangle) and the plane it moves on.
	Vec3 camdir;
	camdir = prevCamRot*Vec3(0,1,0);//viewParams.rotation * Vec3(0,1,0);
	camdir.normalize();
	Vec3 rotax;
	bool bDampAngle=false;
	Vec3 camdiry;

	if( (dist>0.001f) )
	{
		dir/=dist;
		//if( fabs(dir.Dot(camdir))<0.9999f ) // make sure the vectors aren't near_parallel
		{
			rotax = (camdir^dir);
			rotax.NormalizeSafe(Vec3(0,1,0));
			bDampAngle=true;
			camdiry = (rotax^camdir);
		}
		//else
		{
			//rotax=Vec3(0,0,1);
			//bDampAngle=false;
			//camdiry = (rotax^dir);
		}
	}
	else
	{
		dir=camdir;
		rotax=Vec3(0,0,1);
		camdiry = (rotax^camdir);
	}

	float yawPrev,pitchPrev,distPrev;
	float yawNew,pitchNew,distNew;
	float yawDamped,pitchDamped;

	float kDamp=ang_kDamping;//4/(0.1f+abs(ang_kDamping));
	// Move these
	static float yawVel=0;
	static float pitchVel=0;
	static float yawNewPrev=0;
	static float yawPrevPrev=0;
	static int nYawWrap=0;
	static int nNewWrap=0;

	CartesianToSpherical(camdir,yawPrev,pitchPrev,distPrev);
	Vec3 vTest=SphericalToCartesian(yawPrev,pitchPrev,distPrev);

	CartesianToSpherical(dir.GetNormalized(),yawNew,pitchNew,distNew);

	// Adjust angles so that interpolation will work.
	AdjustAnglePair(yawNew,yawNewPrev,yawPrev,yawPrevPrev,nNewWrap,nYawWrap);


	yawDamped=InterpolateDamped(dt,kDamp,yawPrev,yawNew,yawVel);
	pitchDamped=InterpolateDamped(dt,kDamp,pitchPrev,pitchNew,pitchVel);

	// clamp angles to max separation of 1 wrap (360+diff(yawNew,yawPrev))
	if(nNewWrap<nYawWrap)
	{
		nYawWrap-=nNewWrap;
		nNewWrap=0;
		if(nYawWrap>1)
			nYawWrap=1;
	}
	else
	{
		nNewWrap-=nYawWrap;
		nYawWrap=0;
		if(nNewWrap>1)
			nNewWrap=1;
	}

	{
		char acBuf[1024];
		Vec3 vDiff=camdir-vTest;
		sprintf(acBuf,"Yaw:(%0.3f(%d) .. %0.3f(%d) => %0.3f vel:%0.3f)\n",
			yawPrev*180.0f/g_PI,nYawWrap,
			yawNew*180.0f/g_PI,nNewWrap,
			yawDamped*180.0f/g_PI,yawVel*180.0f/g_PI);
		//CryLogAlways(acBuf);
		//pPlayer->CreateActorEvent("printhud",0,acBuf);
	}

	Vec3 vDirDamped=SphericalToCartesian(yawDamped,pitchDamped,1.0f);


	if(bDampAngle)
	{
		camdiry.NormalizeSafe(Vec3(0,1,0));
		//float dangle=0;
		float dangle = cry_acosf(min(1.0f,max(-1.0f,camdir*dir.GetNormalized())));
		Vec3 camdirTmp = camdir*(float)(cry_cosf(dangle))+camdiry*(float)(cry_sinf(dangle));
		camdirTmp.Normalize();
		Vec3 camposTmp = targetPos+camdirTmp*-dist;

		float thetaTmp=camdirTmp*camdir;
		//thetaTmp*=180.0f/gf_PI;
		float thetaDir=camdir*dir;
		Vec3 vUp;

		//-----------------------------------------------

		//-- 2.3. Damp the change of angle

		float e,v0,C1,C2;

		// Angle damping.
		// in: dAngle = desired angle change. viewParams.angleVel = prev rate of change of angle.

		// Rate of turn no higher than ang_maxRate
		if((abs(dangle)/viewParams.frameTime)>ang_maxRate)
		{
			//      if(dangle<0)
			//        dangle=-ang_maxRate*viewParams.frameTime;
			//      else
			//        dangle=ang_maxRate*viewParams.frameTime;
		}

		if (dangle>gf_PI/2)
		{
			//    dangle = gf_PI/2;
		}
		else if (dangle<-gf_PI/2)
		{
			//    dangle=-gf_PI/2;
		}

		float dangleDesired=0;
		dangleDesired=dangle;
		if (ang_kDamping==0)
		{
			//dangle = 0;
			e=1.0f;
		}
		else
		{
			//dangle=InterpolateDamped(dt,4/ang_kDamping,dangle,(float)0,viewParams.angleVel);

			// This is subject to float range errors, see the clamping of dt and
			// ang_kDamping above if you change this expression.
			e=(float) cry_expf(-ang_kDamping*0.5f*dt);
			v0 = viewParams.angleVel*0.0;
			C1 = dangle;
			C2 = v0 + C1*ang_kDamping*0.5f;
			dangle = 0 + C1*e + C2*dt*e;		// desired angle is zero
			viewParams.angleVel = -C1*ang_kDamping*0.5f*e + C2*(e-ang_kDamping*0.5f*dt*e);
		}
		dangle=dangleDesired;
		if(dangle<0)
			dangle=max(dangle,(float)-g_PI/2.01f);
		else
			dangle=min(dangle,(float)g_PI/2.01f);

		{
			char acBuf[1024];
			sprintf(acBuf,"dangle:%0.3f (%0.3f)  (e:%f) (k:%0.3f)\n",dangle,dangleDesired,e,ang_kDamping);
			//			CryLogAlways(acBuf);
			//pPlayer->CreateActorEvent("printhud",0,acBuf);
		}
// WIP
//gDanglePrev2=gDanglePrev;
//gDanglePrev=dangle;

		//-- dangle is output angle from new desired angle towards old angle. i.e. (dangle==0) => at target

		//-- 2.4. Damp the change in position

		// Position damping.

		// in: dist = desired dist. dDist = prevDist-desiredDist. viewParams.vel = prev rate of change of dist.
		{
			float dDistVel;
			float vel=(vSavedTargetPos-prevTargetPos).len()/dt;
			if(kVel<1.0f) kVel=1.0f;
			float tVel=vel/kVel;
			tVel=min(1.0f,tVel);
			dDistVel=tVel*kDist;  // damping proveded below

			// This changes the desired dist from player/target based on player/target speed
			desiredDist+=dDistVel;

			//newDist=InterpolateDamped(dt,4/pos_kDamping,prevDist,desiredDist,viewParams.vel);

			// This is subject to float range errors, see the clamping of dt and
			// ang_kDamping above if you change this expression.
			e = (float) cry_expf(-pos_kDamping*0.5f*dt);
			v0 = viewParams.vel;// - (dist-m_sParam.m_cur_cam_dist)/dt;
			//prevDist=(targetPos - prevCamPos).len();
			//C1 = prevDist-dist; //dist-cam_dist;
			C1 = prevDist - desiredDist;
			C2 = v0 + C1*pos_kDamping*0.5f;
			//float prevDist=0;
			float newDist;
			//dist;
			//newDist = dist + C1*e + C2*dt*e;
			newDist = desiredDist + C1*e + C2*dt*e;

			//float dDist=fabs(newDist-dist);
			float dDist=abs(newDist-desiredDist);
			//   if( dDist < 10)
			{
				dist=newDist;
				viewParams.vel = -C1*pos_kDamping*0.5f*e + C2*(e-pos_kDamping*0.5f*dt*e);
			}
		}
		//-- dist is output dist from target.

		//-- 2.5. convert the damped angle and position back into the form required for output.

		//if(bDampAngle)
		{
			Quat q;
			//camdir = camdir*(float)(cry_cosf(dangle))+camdiry*(float)(cry_sinf(dangle));

			camdir=vDirDamped;
			//camdir=dir.GetNormalized();

			q.SetRotationVDir(camdir.GetNormalized());

			//prevCamRot*=q;
			viewParams.rotation=q;
			viewParams.position = targetPos+camdir*-dist;
			viewParams.dist=dist;
		}

	}

	//--- Screen offsets. Not part of the damping.

	m_io.camViewMtxFinal = Matrix33(viewParams.rotation);

	// Put this back..
	viewParams.position += m_io.camViewMtxFinal.GetColumn(0)*(-thirdPersonHOff) + m_io.camViewMtxFinal.GetColumn(2) * (thirdPersonVOff);

	#if 0
		if(gbFlattenViewDir)
		{
			if(eNavigationMode == eMode)
			{
				float pitchOff=GetISystem()->GetIConsole()->GetCVar("cl_tpvPitchNavOffset")->GetFVal();

				// Flatten rotation back onto Z plane, then apply pitch
				pPlayer->m_camViewMtxFinal=Matrix33(viewParams.rotation);
				//  FlattenRotationMatrixToZPlane(pPlayer->m_viewMtxFinal);
				viewParams.rotation=GetQuatFromMat33(pPlayer->m_camViewMtxFinal);
				viewParams.rotation*=Quat::CreateRotationXYZ(Ang3(pitchOff*(gf_PI/180.0f),0,0));
			}
		}
	#endif
//viewParams.rotation=Quat::CreateRotationZ(gf_PI*180/180);
	// Where the camera's looking
	m_io.camViewMtxFinal=Matrix33(viewParams.rotation);
	// Where the player's looking
	m_io.viewMtxFinal = Matrix33(viewParams.rotation);

	if(!m_in.modes.bCoverMode)
	{
		//pPlayer->m_viewMtxFinal = Matrix33(viewParams.rotation);
	}
	//

	//vOldDir=viewParams.rotation*(vOldDir*prevCamRot);//vOldDir;
	viewParams.targetPos=targetPos;
	viewParams.idTarget=idTarget;

//viewParams.rotation=Quat::CreateRotationZ(gf_PI*180/180);
//viewParams.position=m_in.vEntityWorldPos+viewParams.rotation*Vec3(0,-3,0);
//viewParams.targetPos=m_in.vEntityWorldPos;
DrawAngles("out",viewParams.rotation);
}
#endif
