/*************************************************************************
	Crytek Source File.
	Copyright (C), Crytek Studios, 2001-2004.
	-------------------------------------------------------------------------
	$Id$
	$DateTime$

	-------------------------------------------------------------------------
	History:
		- 7:2:2006   15:38 : Created by Mrcio Martins

*************************************************************************/
#include "StdAfx.h"
#include "ScriptBind_GameRules.h"
#include "GameRules.h"
#include "Game.h"
#include "G4Game.h"
#include "Actor.h"

#include "IVehicleSystem.h"
#include "IItemSystem.h"


//------------------------------------------------------------------------
CGameRules::CGameRules()
: m_pGameFramework(0),
	m_pSystem(0),
	m_pActorSystem(0),
	m_pEntitySystem(0),
	m_pScriptSystem(0),
	m_pMaterialManager(0),
	m_teamIdGen(0),
	m_hitMaterialIdGen(0),
	m_hitTypeIdGen(0)
{
}

//------------------------------------------------------------------------
CGameRules::~CGameRules()
{
	m_pGameFramework->GetIGameRulesSystem()->SetCurrentGameRules(0);
}

//------------------------------------------------------------------------
bool CGameRules::Init( IGameObject * pGameObject )
{
	SetGameObject(pGameObject);

	if (!GetGameObject()->BindToNetwork())
		return false;

	GetGameObject()->EnablePostUpdates(this);

	m_pGameFramework = g_pGame->GetIGameFramework();
	m_pSystem = m_pGameFramework->GetISystem();
	m_pActorSystem = m_pGameFramework->GetIActorSystem();
	m_pEntitySystem = m_pSystem->GetIEntitySystem();
	m_pScriptSystem = m_pSystem->GetIScriptSystem();
	m_pMaterialManager = m_pSystem->GetI3DEngine()->GetMaterialManager();
	
	m_script = GetEntity()->GetScriptTable();
	m_script->GetValue("Client", m_clientScript);
	m_script->GetValue("Server", m_serverScript);

	m_scriptHitInfo.Create(m_pGameFramework->GetISystem()->GetIScriptSystem());
	m_scriptExplosionInfo.Create(m_pGameFramework->GetISystem()->GetIScriptSystem());

	m_pGameFramework->GetIGameRulesSystem()->SetCurrentGameRules(this);
	g_pGame->GetGameRulesScriptBind()->AttachTo(this);

	return true;
}

//------------------------------------------------------------------------
void CGameRules::PostInit( IGameObject * pGameObject )
{
	pGameObject->EnableUpdateSlot(this, 0);
	pGameObject->EnablePostUpdates(this);
}

//------------------------------------------------------------------------
void CGameRules::InitClient(int channelId)
{
	// update team status on the clients
	for (TPlayerTeamIdMap::iterator it=m_playerteams.begin(); it!=m_playerteams.end(); ++it)
		GetGameObject()->InvokeRMI(ClSetTeam(), SetTeamParams(it->first, it->second), eRMI_ToClientChannel, channelId);
}

//------------------------------------------------------------------------
void CGameRules::PostInitClient(int channelId)
{
}

//------------------------------------------------------------------------
void CGameRules::Release()
{
	delete this;
}

//------------------------------------------------------------------------
void CGameRules::Serialize( TSerialize ser, unsigned aspects )
{
}

//------------------------------------------------------------------------
void CGameRules::Update( SEntityUpdateContext& ctx, int updateSlot )
{
	// Here is the best place to call anything that needs to be done at the start of an update cycle.
	// (This is the first entity updated).
}

//------------------------------------------------------------------------
void CGameRules::HandleEvent( const SGameObjectEvent& )
{
}

//------------------------------------------------------------------------
void CGameRules::ProcessEvent( SEntityEvent& event )
{
	switch(event.event) {
		case ENTITY_EVENT_RESET: 
			break;
		case ENTITY_EVENT_START_LEVEL:
			// Here is the best place to call anything that needs to be initialised at the start of a level.
			
			// Initialise the Tweak action map
			CGameG4 *game = (CGameG4*) GetISystem()->GetIGame();
			game->RegisterActionMaps();
			break;
	}
}

//------------------------------------------------------------------------
void CGameRules::SetAuthority( bool auth )
{
}

//------------------------------------------------------------------------
void CGameRules::PostUpdate( float frameTime )
{
}

//------------------------------------------------------------------------
CActor *CGameRules::GetActorByChannelId(int channelId)
{
	return static_cast<CActor *>(m_pGameFramework->GetIActorSystem()->GetActorByChannelId(channelId));
}

//------------------------------------------------------------------------
CActor *CGameRules::GetActorByEntityId(EntityId entityId)
{
	return static_cast<CActor *>(m_pGameFramework->GetIActorSystem()->GetActor(entityId));
}

//------------------------------------------------------------------------
int CGameRules::GetChannelId(EntityId entityId)
{
	CActor *pActor = static_cast<CActor *>(m_pGameFramework->GetIActorSystem()->GetActor(entityId));
	if (pActor)
		return pActor->GetChannelId();

	return 0;
}

//------------------------------------------------------------------------
void CGameRules::OnClientConnect(int channelId)
{
}

//------------------------------------------------------------------------
void CGameRules::OnClientDisconnect(int channelId)
{
}

//------------------------------------------------------------------------
void CGameRules::OnClientEnteredGame(int channelId)
{
}

//------------------------------------------------------------------------
void CGameRules::OnItemDropped(EntityId itemId, EntityId actorId)
{
	ScriptHandle itemIdHandle(itemId);
	ScriptHandle actorIdHandle(actorId);
	CallScript(m_serverScript, "OnItemDropped", itemIdHandle, actorIdHandle);
}

//------------------------------------------------------------------------
void CGameRules::OnItemPickedUp(EntityId itemId, EntityId actorId)
{
	ScriptHandle itemIdHandle(itemId);
	ScriptHandle actorIdHandle(actorId);
	CallScript(m_serverScript, "OnItemPickedUp", itemIdHandle, actorIdHandle);
}

//------------------------------------------------------------------------
void CGameRules::OnTextMessage(ETextMessageType type, const char *msg)
{
	//send message to hud (actually it is broadcasted right now!)
	SGameObjectEvent evt("HUD_TextMessage",eGOEF_ToAll, IGameObjectSystem::InvalidExtensionID, (void*)msg);
	m_pGameFramework->GetIGameObjectSystem()->BroadcastEvent(evt);
}

//------------------------------------------------------------------------
void CGameRules::OnChatMessage(EChatMessageType type, EntityId sourceId, EntityId targetId, const char *msg)
{
	//send message to hud (actually it is broadcasted right now!)
	string completeMsg(GetActorNameByEntityId(sourceId));
	completeMsg.append("$o: ");
	completeMsg.append(msg);
	SGameObjectEvent evt("MultiplayerChatMessage",eGOEF_ToAll, IGameObjectSystem::InvalidExtensionID, (void*)completeMsg.c_str());
	m_pGameFramework->GetIGameObjectSystem()->BroadcastEvent(evt);
}

//------------------------------------------------------------------------
void CGameRules::OnKillMessage(EntityId targetId, EntityId shooterId, EntityId weaponId, float damage, int material)
{
	//send message to hud (actually it is broadcasted right now!)
	string completeMsg("$8");
	completeMsg.append(GetActorNameByEntityId(shooterId));
	completeMsg.append("$o killed $8");
	completeMsg.append(GetActorNameByEntityId(targetId));
	IEntity *pWeapon=m_pEntitySystem->GetEntity(weaponId);
	if (pWeapon)
	{
		completeMsg.append(" with ");
		completeMsg.append(pWeapon->GetClass()->GetName());
	}
	completeMsg.append("!");
	SGameObjectEvent evt("HUD_TextMessage",eGOEF_ToAll, IGameObjectSystem::InvalidExtensionID, (void*)completeMsg.c_str());
	m_pGameFramework->GetIGameObjectSystem()->BroadcastEvent(evt);
}

//------------------------------------------------------------------------
CActor *CGameRules::SpawnPlayer(int channelId, const char *name, const char *className, const Vec3 &pos, const Ang3 &angles)
{
	if (!IsServer())
		return 0;

	CActor *pActor = static_cast<CActor *>(m_pActorSystem->CreateActor(channelId, VerifyName(name).c_str(), className, pos, Quat::CreateRotationXYZ(angles), Vec3(1, 1, 1)));
	if (pActor)
		MovePlayer(pActor, pos, angles);

	return pActor;
}

//------------------------------------------------------------------------
CActor *CGameRules::ChangePlayerClass(int channelId, const char *className)
{
	if (!IsServer())
		return 0;

	CActor *pOldActor = GetActorByChannelId(channelId);
	if (!pOldActor)
		return 0;

	if (!strcmp(pOldActor->GetEntity()->GetClass()->GetName(), className))
		return pOldActor;

	EntityId oldEntityId = pOldActor->GetEntityId();
	string oldName = pOldActor->GetEntity()->GetName();
	Ang3 oldAngles=pOldActor->GetAngles();
	Vec3 oldPos = pOldActor->GetEntity()->GetWorldPos();

	m_pEntitySystem->RemoveEntity(pOldActor->GetEntityId(), true);

	CActor *pActor = static_cast<CActor *>(m_pActorSystem->CreateActor(channelId, oldName.c_str(), className, oldPos, Quat::CreateRotationXYZ(oldAngles), Vec3(1, 1, 1), oldEntityId));
	if (pActor)
		MovePlayer(pActor, oldPos, oldAngles);

	return pActor;
}

//------------------------------------------------------------------------
void CGameRules::RevivePlayer(CActor *pActor, const Vec3 &pos, const Ang3 &angles)
{
	pActor->SetHealth(100);
	pActor->SetMaxHealth(100);
	pActor->GetGameObject()->SetPhysicalizationProfile(eAP_Alive);

	pActor->Revive(false);

	Matrix34 tm(pActor->GetEntity()->GetWorldTM());
	tm.SetTranslation(pos);

	pActor->GetEntity()->SetWorldTM(tm);
	pActor->SetAngles(angles);
	pActor->GetGameObject()->InvokeRMI(CActor::ClRevive(), CActor::ReviveParams(pos, angles), eRMI_ToOtherClients);
}

//------------------------------------------------------------------------
void CGameRules::RenamePlayer(CActor *pActor, const char *name)
{
	string fixed=VerifyName(name);
	RenameEntityParams params(pActor->GetEntityId(), fixed.c_str());
	if (fixed == pActor->GetEntity()->GetName())
		return;

	if (IsServer())
	{
		if (!IsClient())
			pActor->GetEntity()->SetName(fixed.c_str());
		GetGameObject()->InvokeRMI(ClRenameEntity(), params, eRMI_ToAllClients);
	}
	else if (pActor->GetEntityId() == m_pGameFramework->GetClientActor()->GetEntityId())
		GetGameObject()->InvokeRMI(SvRequestRename(), params, eRMI_ToServer);
}

//------------------------------------------------------------------------
string CGameRules::VerifyName(const char *name)
{
	return string(name);
}

//------------------------------------------------------------------------
void CGameRules::KillPlayer(CActor *pActor, bool dropItem, bool ragdoll, EntityId shooterId, EntityId weaponId, float damage, int material, const Vec3 &impulse)
{
	IInventory *pInventory=pActor->GetInventory();
	if (dropItem && pInventory && pInventory->GetCurrentItem())
		pActor->DropItem(pInventory->GetCurrentItem());

	pActor->GetInventory()->Destroy();

	IVehicle *pVehicle=pActor->GetLinkedVehicle();
	
	if (pVehicle)
	{
		pVehicle->OnPassengerKilled(pActor->GetEntityId());
		ragdoll=false;
	}

	if (ragdoll)
		pActor->GetGameObject()->SetPhysicalizationProfile(eAP_Ragdoll);

	pActor->GetGameObject()->InvokeRMI(CActor::ClKill(),
		CActor::KillParams(true, ragdoll, shooterId, weaponId, damage, material, impulse),
		eRMI_ToAllClients);
}

//------------------------------------------------------------------------
void CGameRules::MovePlayer(CActor *pActor, const Vec3 &pos, const Ang3 &angles)
{
	CActor::MoveParams params(pos, Quat::CreateRotationXYZ(angles));
	pActor->GetGameObject()->InvokeRMI(CActor::ClMoveTo(), params, eRMI_ToClientChannel, pActor->GetChannelId());
	pActor->GetEntity()->SetWorldTM(Matrix34::Create(Vec3(1,1,1), params.rot, params.pos));
}

//------------------------------------------------------------------------
int CGameRules::CreateTeam(const char *name)
{
	TTeamIdMap::iterator it = m_teams.find(name);
	if (it != m_teams.end())
		return it->second;

	m_teams.insert(TTeamIdMap::value_type(name, ++m_teamIdGen));

	return m_teamIdGen;
}

//------------------------------------------------------------------------
void CGameRules::RemoveTeam(int teamId)
{
	TTeamIdMap::iterator it = m_teams.find(GetTeamName(teamId));
	if (it == m_teams.end())
		return;

	m_teams.erase(it);

	for (TPlayerTeamIdMap::iterator pit=m_playerteams.begin(); pit != m_playerteams.end(); ++it)
	{
		if (pit->second == teamId)
			pit->second = 0; // 0 is no team
	}
}

//------------------------------------------------------------------------
const char *CGameRules::GetTeamName(int teamId) const
{
	for (TTeamIdMap::const_iterator it = m_teams.begin(); it!=m_teams.end(); ++it)
	{
		if (teamId == it->second)
			return it->first;
	}

	return 0;
}

//------------------------------------------------------------------------
void CGameRules::SetTeam(int teamId, EntityId id)
{
	m_playerteams.insert(TPlayerTeamIdMap::value_type(id, teamId));

	GetGameObject()->InvokeRMI(ClSetTeam(), SetTeamParams(id, teamId), eRMI_ToRemoteClients);
}

//------------------------------------------------------------------------
int CGameRules::GetTeam(EntityId playerId) const
{
	TPlayerTeamIdMap::const_iterator it = m_playerteams.find(playerId);
	if (it != m_playerteams.end())
		return it->second;

	return 0;
}

//------------------------------------------------------------------------
int CGameRules::RegisterHitMaterial(const char *materialName)
{
	if (int id=GetHitMaterialId(materialName))
		return id;

	ISurfaceType *pSurfaceType=m_pMaterialManager->GetSurfaceTypeByName(materialName);
	if (pSurfaceType)
	{
		m_hitMaterials.insert(THitMaterialMap::value_type(++m_hitMaterialIdGen, pSurfaceType->GetId()));
		return m_hitMaterialIdGen;
	}
	return 0;
}

//------------------------------------------------------------------------
int CGameRules::GetHitMaterialId(const char *materialName) const
{
	ISurfaceType *pSurfaceType=m_pMaterialManager->GetSurfaceTypeByName(materialName);
	if (!pSurfaceType)
		return 0;

	int id=pSurfaceType->GetId();

	for (THitMaterialMap::const_iterator it=m_hitMaterials.begin(); it!=m_hitMaterials.end(); ++it)
	{
		if (it->second==id)
			return it->first;
	}

	return 0;
}

//------------------------------------------------------------------------
int CGameRules::GetHitMaterialIdFromSurfaceId(int surfaceId) const
{
	for (THitMaterialMap::const_iterator it=m_hitMaterials.begin(); it!=m_hitMaterials.end(); ++it)
	{
		if (it->second==surfaceId)
			return it->first;
	}

	return 0;
}

//------------------------------------------------------------------------
ISurfaceType *CGameRules::GetHitMaterial(int id) const
{
	THitMaterialMap::const_iterator it=m_hitMaterials.find(id);
	if (it==m_hitMaterials.end())
		return 0;

	ISurfaceType *pSurfaceType=m_pMaterialManager->GetSurfaceType(it->second);
	
	return pSurfaceType;
}

//------------------------------------------------------------------------
void CGameRules::ResetHitMaterials()
{
	m_hitMaterials.clear();
	m_hitMaterialIdGen=0;
}

//------------------------------------------------------------------------
int CGameRules::RegisterHitType(const char *type)
{
	if (int id=GetHitTypeId(type))
		return id;

	m_hitTypes.insert(THitTypeMap::value_type(++m_hitTypeIdGen, type));
	return m_hitTypeIdGen;
}

//------------------------------------------------------------------------
int CGameRules::GetHitTypeId(const char *type) const
{
	for (THitTypeMap::const_iterator it=m_hitTypes.begin(); it!=m_hitTypes.end(); ++it)
	{
		if (it->second==type)
			return it->first;
	}

	return 0;
}

//------------------------------------------------------------------------
const char *CGameRules::GetHitType(int id) const
{
	THitTypeMap::const_iterator it=m_hitTypes.find(id);
	if (it==m_hitTypes.end())
		return 0;

	return it->second.c_str();
}

//------------------------------------------------------------------------
void CGameRules::ResetHitTypes()
{
	m_hitTypes.clear();
	m_hitTypeIdGen=0;
}

//------------------------------------------------------------------------
void CGameRules::ClientHit(const HitInfo &hitInfo)
{
	CreateScriptHitInfo(m_scriptHitInfo, hitInfo);
	CallScript(m_clientScript, "OnHit", m_scriptHitInfo);

	bool backface = hitInfo.dir.Dot(hitInfo.normal)>0;
	if (!hitInfo.remote && hitInfo.targetId && !backface)
	{
		if (!IsServer())
			GetGameObject()->InvokeRMI(SvRequestHit(), hitInfo, eRMI_ToServer);
		else
			ServerHit(hitInfo);
	}
}

//------------------------------------------------------------------------
void CGameRules::ServerHit(const HitInfo &hitInfo)
{
	bool ok=true;
	// check if shooter is alive
	if (hitInfo.shooterId)
	{
		CActor *pShooter=GetActorByEntityId(hitInfo.shooterId);
		if (!pShooter || pShooter->GetHealth()<=0)
			ok=false;
	}

	if (ok)
	{
		CreateScriptHitInfo(m_scriptHitInfo, hitInfo);
		CallScript(m_serverScript, "OnHit", m_scriptHitInfo);
	}
}

//------------------------------------------------------------------------
void CGameRules::ServerExplosion(const ExplosionInfo &explosionInfo)
{
	GetGameObject()->InvokeRMI(ClExplosion(), explosionInfo, eRMI_ToRemoteClients);

	CreateScriptExplosionInfo(m_scriptExplosionInfo, explosionInfo);
	CallScript(m_serverScript, "OnExplosion", m_scriptExplosionInfo);

	ClientExplosion(explosionInfo);
}

//------------------------------------------------------------------------
void CGameRules::ClientExplosion(const ExplosionInfo &explosionInfo)
{
	GetISystem()->GetI3DEngine()->OnExplosion(explosionInfo.pos, explosionInfo.dir, explosionInfo.hole_size*0.5f, 0, 0, true);

	pe_explosion explosion;
	explosion.epicenter = explosionInfo.pos;
	explosion.holeSize = explosionInfo.hole_size;
	explosion.r = explosionInfo.radius;
	explosion.rmin = 1.0f;
	explosion.rmax = explosionInfo.radius;
	explosion.impulsivePressureAtR = explosionInfo.pressure;
	explosion.epicenterImp = explosionInfo.pos;
	explosion.explDir = explosionInfo.dir;
	GetISystem()->GetIPhysicalWorld()->SimulateExplosion(&explosion);

	if (IsClient())
	{
		IParticleEffect *pParticleEffect = GetISystem()->GetI3DEngine()->FindParticleEffect(explosionInfo.effect.c_str());
		if (pParticleEffect)
			pParticleEffect->Spawn(true, IParticleEffect::ParticleLoc(explosionInfo.pos, explosionInfo.dir, explosionInfo.effectScale));

		CreateScriptExplosionInfo(m_scriptExplosionInfo, explosionInfo);
		CallScript(m_clientScript, "OnExplosion", m_scriptExplosionInfo);
	}

	// HACKCKCKCKCKCKC
	GetISystem()->GetAISystem()->SoundEvent(GetEntity()->GetWorldPos(),5*explosionInfo.radius,1,0, NULL);
}

//------------------------------------------------------------------------
void CGameRules::SendTextMessage(ETextMessageType type, const char *msg, unsigned int to, int channelId)
{
	GetGameObject()->InvokeRMI(ClTextMessage(), TextMessageParams(type, msg), to, channelId);
}

//------------------------------------------------------------------------
void CGameRules::SendChatMessage(EChatMessageType type, EntityId sourceId, EntityId targetId, const char *msg)
{
	ChatMessageParams params(type, sourceId, targetId, msg);

	if (IsServer())
	{
		switch(type)
		{
		case eChatToAll:
			GetGameObject()->InvokeRMI(ClChatMessage(), params, eRMI_ToAllClients);
			break;
		case eChatToTarget:
			GetGameObject()->InvokeRMI(ClChatMessage(), params, eRMI_ToClientChannel, GetChannelId(targetId));
			break;
		case eChatToTeam:
			{
				int teamId = GetTeam(sourceId);
				if (teamId)
				{
					for (TPlayerTeamIdMap::iterator it=m_playerteams.begin(); it!=m_playerteams.end(); ++it)
					{
						if (it->second == teamId)
							GetGameObject()->InvokeRMI(ClChatMessage(), params, eRMI_ToClientChannel, GetChannelId(it->first));
					}
				}
			}
		}
	}
	else
		GetGameObject()->InvokeRMI(SvRequestChatMessage(), params, eRMI_ToServer);
}

//------------------------------------------------------------------------
void CGameRules::CreateScriptHitInfo(SmartScriptTable &scriptHitInfo, const HitInfo &hitInfo)
{
	CScriptSetGetChain hit(scriptHitInfo);
	{
		hit.SetValue("normal", hitInfo.normal);
		hit.SetValue("pos", hitInfo.pos);
		hit.SetValue("dir", hitInfo.dir);
		hit.SetValue("partId", hitInfo.partId);
		hit.SetValue("backface", hitInfo.normal.Dot(hitInfo.dir)>=0.0f);

		hit.SetValue("targetId", ScriptHandle(hitInfo.targetId));		
		hit.SetValue("shooterId", ScriptHandle(hitInfo.shooterId));
		hit.SetValue("weaponId", ScriptHandle(hitInfo.weaponId));

		IEntity *pTarget=m_pEntitySystem->GetEntity(hitInfo.targetId);
		IEntity *pShooter=m_pEntitySystem->GetEntity(hitInfo.shooterId);
		IEntity *pWeapon=m_pEntitySystem->GetEntity(hitInfo.weaponId);
		hit.SetValue("target", pTarget?pTarget->GetScriptTable():(IScriptTable *)0);
		hit.SetValue("shooter", pShooter?pShooter->GetScriptTable():(IScriptTable *)0);
		hit.SetValue("weapon", pWeapon?pWeapon->GetScriptTable():(IScriptTable *)0);

		hit.SetValue("materialId", hitInfo.material);
		ISurfaceType *pSurfaceType=GetHitMaterial(hitInfo.material);
		hit.SetValue("material", pSurfaceType?pSurfaceType->GetScriptTable():(IScriptTable *)0);

		hit.SetValue("damage", hitInfo.damage);
		hit.SetValue("radius", hitInfo.radius);
		
		hit.SetValue("typeId", hitInfo.type);
		const char *type=GetHitType(hitInfo.type);
		hit.SetValue("type", type);
		hit.SetValue("remote", hitInfo.remote);
	}
}

//------------------------------------------------------------------------
void CGameRules::CreateScriptExplosionInfo(SmartScriptTable &scriptExplosionInfo, const ExplosionInfo &explosionInfo)
{
	CScriptSetGetChain explosion(scriptExplosionInfo);
	{
		explosion.SetValue("pos", explosionInfo.pos);
		explosion.SetValue("dir", explosionInfo.dir);

		explosion.SetValue("shooterId", ScriptHandle(explosionInfo.shooterId));
		explosion.SetValue("weaponId", ScriptHandle(explosionInfo.weaponId));
		IEntity *pShooter=m_pEntitySystem->GetEntity(explosionInfo.shooterId);
		IEntity *pWeapon=m_pEntitySystem->GetEntity(explosionInfo.weaponId);
		explosion.SetValue("shooter", pShooter?pShooter->GetScriptTable():(IScriptTable *)0);
		explosion.SetValue("weapon", pWeapon?pWeapon->GetScriptTable():(IScriptTable *)0);
		explosion.SetValue("materialId", 0);
		explosion.SetValue("damage", explosionInfo.damage);
		explosion.SetValue("radius", explosionInfo.radius);
		explosion.SetValue("pressure", explosionInfo.pressure);
		explosion.SetValue("hole_size", explosionInfo.hole_size);
		explosion.SetValue("effect", explosionInfo.effect.c_str());
		explosion.SetValue("effectScale", explosionInfo.effectScale);
	}
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, SvRequestRename)
{
	CActor *pActor = GetActorByEntityId(params.entityId);
	if (!pActor)
		return true;

	RenamePlayer(pActor, params.name.c_str());

	return true;
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, ClRenameEntity)
{
	IEntity *pEntity=GetISystem()->GetIEntitySystem()->GetEntity(params.entityId);
	if (pEntity)
	{
		string old=pEntity->GetName();
		pEntity->SetName(params.name.c_str());

		CryLogAlways("$8%s$o renamed to $8%s", old.c_str(), params.name.c_str());
	}

	return true;
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, SvRequestChatMessage)
{
	//SendChatMessage(params.type, params.sourceId, params.targetId, params.msg.c_str());
	SendChatMessage(eChatToAll, params.sourceId, params.targetId, params.msg.c_str());

	return true;
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, ClChatMessage)
{
	OnChatMessage(params.type, params.sourceId, params.targetId, params.msg.c_str());

	return true;
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, ClSetTeam)
{
	m_playerteams.insert(TPlayerTeamIdMap::value_type(params.playerId, params.teamId));
	
	return true;
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, ClTextMessage)
{
	OnTextMessage(params.type, params.msg.c_str());

	return true;
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, SvRequestHit)
{
	ServerHit(params);

	return true;
}

//------------------------------------------------------------------------
IMPLEMENT_RMI(CGameRules, ClExplosion)
{
	ClientExplosion(params);

	return true;
}
