/*************************************************************************
	Crytek Source File.
	Copyright (C), Crytek Studios, 2001-2004.
	-------------------------------------------------------------------------
	$Id$
	$DateTime$
	Description: 

	-------------------------------------------------------------------------
	History:
	- 7:2:2006   15:38 : Created by Mrcio Martins

*************************************************************************/
#ifndef __GAMERULES_H__
#define __GAMERULES_H__

#if _MSC_VER > 1000
# pragma once
#endif


#include <IGameObject.h>
#include <IGameRulesSystem.h>
#include "Actor.h"

class CActor;

struct IGameObject;
struct IActorSystem;


class CGameRules : public CGameObjectExtensionHelper<CGameRules, IGameRules>
{
public:
	CGameRules();
	virtual ~CGameRules();
	//IGameObjectExtension
	virtual bool Init( IGameObject * pGameObject );
	virtual void PostInit( IGameObject * pGameObject );
	virtual void InitClient(int channelId);
	virtual void PostInitClient(int channelId);
	virtual void Release();
	virtual void Serialize( TSerialize ser, unsigned aspects );
	virtual void Update( SEntityUpdateContext& ctx, int updateSlot );
	virtual void HandleEvent( const SGameObjectEvent& );
	virtual void ProcessEvent( SEntityEvent& );
	virtual void SetChannelId(uint16 id) {};
	virtual void SetAuthority( bool auth );
	virtual void PostUpdate( float frameTime );
	//~IGameObjectExtension

	//IGameRules
	virtual void OnClientConnect(int channelId);
	virtual void OnClientDisconnect(int channelId);
	virtual void OnClientEnteredGame(int channelId);

	virtual void OnItemDropped(EntityId itemId, EntityId actorId);
	virtual void OnItemPickedUp(EntityId itemId, EntityId actorId);

	virtual void SendTextMessage(ETextMessageType type, const char *msg, uint to=eRMI_ToAllClients, int channelId=-1);
	virtual void SendChatMessage(EChatMessageType type, EntityId sourceId, EntityId targetId, const char *msg);
	//~IGameRules

	virtual void OnTextMessage(ETextMessageType type, const char *msg);
	virtual void OnChatMessage(EChatMessageType type, EntityId sourceId, EntityId targetId, const char *msg);
	virtual void OnKillMessage(EntityId targetId, EntityId shooterId, EntityId weaponId, float damage, int material);


	CActor *GetActorByChannelId(int channelId);
	CActor *GetActorByEntityId(EntityId entityId);
	ILINE const char *GetActorNameByEntityId(EntityId entityId)
	{
		CActor *pActor=GetActorByEntityId(entityId);
		if (pActor)
			return pActor->GetEntity()->GetName();
		return 0;
	}
	ILINE const char *GetActorName(CActor *pActor) { return pActor->GetEntity()->GetName(); };
	int GetChannelId(EntityId entityId);

	//------------------------------------------------------------------------
	// player
	virtual CActor *SpawnPlayer(int channelId, const char *name, const char *className, const Vec3 &pos, const Ang3 &angles);
	virtual CActor *ChangePlayerClass(int channelId, const char *className);
	virtual void RevivePlayer(CActor *pActor, const Vec3 &pos, const Ang3 &angles);
	virtual void RenamePlayer(CActor *pActor, const char *name);
	virtual string VerifyName(const char *name);
	virtual void KillPlayer(CActor *pActor, bool dropItem, bool ragdoll, EntityId shooterId, EntityId weaponId, float damage, int material, const Vec3 &impulse);
	virtual void MovePlayer(CActor *pActor, const Vec3 &pos, const Ang3 &angles);

	//------------------------------------------------------------------------
	// teams
	virtual int CreateTeam(const char *name);
	virtual void RemoveTeam(int teamId);
	virtual const char *GetTeamName(int teamId) const;

	virtual void SetTeam(int teamId, EntityId playerId);
	virtual int GetTeam(EntityId playerId) const;

	//------------------------------------------------------------------------
	// materials
	virtual int RegisterHitMaterial(const char *materialName);
	virtual int GetHitMaterialId(const char *materialName) const;
	virtual ISurfaceType *GetHitMaterial(int id) const;
	virtual int GetHitMaterialIdFromSurfaceId(int surfaceId) const;
	virtual void ResetHitMaterials();

	//------------------------------------------------------------------------
	// hit type
	virtual int RegisterHitType(const char *type);
	virtual int GetHitTypeId(const char *type) const;
	virtual const char *GetHitType(int id) const;
	virtual void ResetHitTypes();

	//------------------------------------------------------------------------
	// game
	typedef struct HitInfo
	{
		EntityId shooterId;
		EntityId targetId;
		EntityId weaponId;

		float		damage;
		float		radius;
		int			material;
		int			type;

		int			partId;
		
		Vec3		pos;
		Vec3		dir;
		Vec3		normal;

		bool		remote;

		HitInfo()
		: shooterId(0),
			targetId(0),
			weaponId(0),
			damage(0),
			radius(0),
			material(-1),
			partId(-1),
			type(0),
			pos(0,0,0),
			dir(FORWARD_DIRECTION),
			normal(-FORWARD_DIRECTION),
			remote(false)
		{}

		HitInfo(EntityId shtId, EntityId trgId, EntityId wpnId, float dmg, float rd, int mat, int part, int typ)
			: shooterId(shtId),
			targetId(trgId),
			weaponId(wpnId),
			damage(dmg),
			radius(rd),
			material(mat),
			partId(part),
			type(typ),
			pos(0,0,0),
			dir(FORWARD_DIRECTION),
			normal(-FORWARD_DIRECTION),
			remote(false)
		{}

		HitInfo(EntityId shtId, EntityId trgId, EntityId wpnId, float dmg, float rd, int mat, int part, int typ, const Vec3 &p, const Vec3 &d, const Vec3 &n)
			: shooterId(shtId),
			targetId(trgId),
			weaponId(wpnId),
			damage(dmg),
			radius(rd),
			material(mat),
			partId(part),
			type(typ),
			pos(p),
			dir(d),
			normal(n),
			remote(false)
		{}

		void SerializeWith(TSerialize ser)
		{
			ser.Value("shooterId", shooterId, NSerPolicy::AC_EntityId());
			ser.Value("targetId", targetId, NSerPolicy::AC_EntityId());
			ser.Value("weaponId", weaponId, NSerPolicy::AC_EntityId());
			ser.Value("damage", damage, NSerPolicy::AC_Float(0.0f, 1000.0f, 12));
			ser.Value("radius", radius, NSerPolicy::AC_Float(0.0f, 100.0f, 10));
			ser.Value("material", material, NSerPolicy::AS_RangedInt32(-1, 255));
			ser.Value("partId", material, NSerPolicy::AS_RangedInt32(-1, 255));
			ser.Value("type", type, NSerPolicy::AS_RangedInt32(0, 31));
			ser.Value("pos", pos, NSerPolicy::AC_WorldPos());
			ser.Value("dir", dir, NSerPolicy::AC_Vec3Norm(12));
			ser.Value("normal", normal, NSerPolicy::AC_Vec3Norm(12));
		}
	};

	virtual void ClientHit(const HitInfo &hitInfo);
	virtual void ServerHit(const HitInfo &hitInfo);

	typedef struct ExplosionInfo
	{
		EntityId shooterId;
		EntityId weaponId;

		float		damage;

		Vec3		pos;
		Vec3		dir;
		float		radius;
		float		pressure;
		float		hole_size;
		string	effect;
		float		effectScale;

		ExplosionInfo()
			: shooterId(0),
			weaponId(0),
			damage(0),
			pos(0,0,0),
			dir(FORWARD_DIRECTION),
			radius(5.0f),
			pressure(200.0f),
			hole_size(5.0f),
			effectScale(1.0f)
		{}

		ExplosionInfo(EntityId shtId, EntityId wpnId, float dmg, const Vec3 &p, const Vec3 &d, float r, float press, float holesize)
			: shooterId(shtId),
			weaponId(wpnId),
			damage(dmg),
			pos(p),
			dir(d),
			radius(r),
			pressure(press),
			hole_size(holesize)
		{}

		void SerializeWith(TSerialize ser)
		{
			ser.Value("shooterId", shooterId, NSerPolicy::AC_EntityId());
			ser.Value("weaponId", weaponId, NSerPolicy::AC_EntityId());
			ser.Value("damage", damage, NSerPolicy::AC_Float(0.0f, 1000.0f, 12));
			ser.Value("pos", pos, NSerPolicy::AC_WorldPos());
			ser.Value("dir", dir, NSerPolicy::AC_Vec3Norm(12));
			ser.Value("radius", radius, NSerPolicy::AC_Float(0.0f, 50.0f, 12));
			ser.Value("pressure", pressure, NSerPolicy::AC_Float(0.0f, 10000.0f, 16));
			ser.Value("hole_size", hole_size, NSerPolicy::AC_Float(0.0f, 50.0f, 12));
			ser.Value("effect", effect);
			if (!effect.empty())
				ser.Value("effectScale", effectScale, NSerPolicy::AC_FloatWithUsualValue(0.0f, 10.0f, 1.0f, 8));
		}
	};

	virtual void ServerExplosion(const ExplosionInfo &explosionInfo);
	virtual void ClientExplosion(const ExplosionInfo &explosionInfo);

	struct ChatMessageParams
	{
		EChatMessageType type;
		EntityId sourceId;
		EntityId targetId;
		string msg;

		ChatMessageParams() {};
		ChatMessageParams(EChatMessageType _type, EntityId src, EntityId trg, const char *_msg)
		: type(_type),
			sourceId(src),
			targetId(trg),
			msg(_msg)
		{
		};

		void SerializeWith(TSerialize ser)
		{
			ser.EnumValue("type", type, eChatToTarget, eChatToAll);
			ser.Value("source", sourceId, NSerPolicy::AC_EntityId());
			if (type == eChatToTarget)
				ser.Value("target", targetId, NSerPolicy::AC_EntityId());
			ser.Value("message", msg);
		}
	};

	struct TextMessageParams
	{
		ETextMessageType type;
		string msg;

		TextMessageParams() {};
		TextMessageParams(ETextMessageType _type, const char *_msg)
			: type(_type),
			msg(_msg)
		{
		};

		void SerializeWith(TSerialize ser)
		{
			ser.EnumValue("type", type, eTextMessageCenter, eTextMessageInfo);
			ser.Value("message", msg);
		}
	};

	struct SetTeamParams
	{
		int				teamId;
		EntityId	playerId;

		SetTeamParams() {};
		SetTeamParams(EntityId _playerId, int _teamId)
		: playerId(_playerId),
			teamId(_teamId)
		{
		}

		void SerializeWith(TSerialize ser)
		{
			ser.Value("playerId", playerId, NSerPolicy::AC_EntityId());
			ser.Value("teamId", teamId, NSerPolicy::AS_RangedInt32(0, 7));
		}
	};

	struct RenameEntityParams
	{
		EntityId	entityId;
		string		name;

		RenameEntityParams() {};
		RenameEntityParams(EntityId _entityId, const char *name)
			: entityId(_entityId),
				name(name)
		{
		}

		void SerializeWith(TSerialize ser)
		{
			ser.Value("entityId", entityId, NSerPolicy::AC_EntityId());
			ser.Value("name", name);
		}
	};

	DECLARE_SERVER_RMI_PREATTACH(SvRequestHit, HitInfo, eNRT_ReliableOrdered);
	DECLARE_CLIENT_RMI_PREATTACH(ClExplosion, ExplosionInfo, eNRT_ReliableUnordered);

	DECLARE_SERVER_RMI_POSTATTACH(SvRequestChatMessage, ChatMessageParams, eNRT_ReliableUnordered);
	DECLARE_CLIENT_RMI_POSTATTACH(ClChatMessage, ChatMessageParams, eNRT_ReliableUnordered);

	DECLARE_SERVER_RMI_POSTATTACH(SvRequestRename, RenameEntityParams, eNRT_ReliableOrdered);
	DECLARE_CLIENT_RMI_PREATTACH(ClRenameEntity, RenameEntityParams, eNRT_ReliableOrdered);

	DECLARE_CLIENT_RMI_PREATTACH(ClSetTeam, SetTeamParams, eNRT_ReliableOrdered);
	DECLARE_CLIENT_RMI_POSTATTACH(ClTextMessage, TextMessageParams, eNRT_ReliableUnordered);
	
	// chat message
	// kill message
	// rename entity message
	ILINE bool IsClient() const
	{
		return m_pGameFramework->IsClient();
	}
	ILINE bool IsServer() const
	{
		return m_pGameFramework->IsServer();
	}

	typedef std::map<EntityId, int>	TPlayerTeamIdMap;
	typedef std::map<string, int>		TTeamIdMap;

	typedef std::map<int, int>			THitMaterialMap;
	typedef std::map<int, string>		THitTypeMap;

protected:
	virtual void CreateScriptHitInfo(SmartScriptTable &scriptHitInfo, const HitInfo &hitInfo);
	virtual void CreateScriptExplosionInfo(SmartScriptTable &scriptExplosionInfo, const ExplosionInfo &explosionInfo);

	void CallScript(IScriptTable *pScript, const char *name)
	{
		if (pScript->GetValueType(name) != svtFunction)
			return;
		m_pScriptSystem->BeginCall(pScript, name); m_pScriptSystem->PushFuncParam(m_script);
		m_pScriptSystem->EndCall();
	};
	template<typename P1>
	void CallScript(IScriptTable *pScript, const char *name, P1 &p1)
	{
		if (pScript->GetValueType(name) != svtFunction)
			return;
		m_pScriptSystem->BeginCall(pScript, name); m_pScriptSystem->PushFuncParam(m_script);
		m_pScriptSystem->PushFuncParam(p1);
		m_pScriptSystem->EndCall();
	};
	template<typename P1, typename P2>
	void CallScript(IScriptTable *pScript, const char *name, P1 &p1, P2 &p2)
	{
		if (pScript->GetValueType(name) != svtFunction)
			return;
		m_pScriptSystem->BeginCall(pScript, name); m_pScriptSystem->PushFuncParam(m_script);
		m_pScriptSystem->PushFuncParam(p1); m_pScriptSystem->PushFuncParam(p2);
		m_pScriptSystem->EndCall();
	};
	template<typename P1, typename P2, typename P3, typename P4>
	void CallScript(IScriptTable *pScript, const char *name, P1 &p1, P2 &p2, P3 &p3)
	{
		if (pScript->GetValueType(name) != svtFunction)
			return;
		m_pScriptSystem->BeginCall(pScript, name); m_pScriptSystem->PushFuncParam(m_script);
		m_pScriptSystem->PushFuncParam(p1); m_pScriptSystem->PushFuncParam(p2); m_pScriptSystem->PushFuncParam(p3);
		m_pScriptSystem->EndCall();
	};

	IGameFramework			*m_pGameFramework;
	ISystem							*m_pSystem;
	IActorSystem				*m_pActorSystem;
	IEntitySystem				*m_pEntitySystem;
	IScriptSystem				*m_pScriptSystem;
	IMaterialManager		*m_pMaterialManager;
	SmartScriptTable		m_script;
	SmartScriptTable		m_clientScript;
	SmartScriptTable		m_serverScript;
	

	TTeamIdMap					m_teams;
	TPlayerTeamIdMap		m_playerteams;
	int									m_teamIdGen;

	THitMaterialMap			m_hitMaterials;
	int									m_hitMaterialIdGen;

	THitTypeMap					m_hitTypes;
	int									m_hitTypeIdGen;

	SmartScriptTable		m_scriptHitInfo;
	SmartScriptTable		m_scriptExplosionInfo;
};


#endif //__GAMERULES_H__