/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2005.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: HUD draw functions

-------------------------------------------------------------------------
History:
- 07:11:2005: Created by Julien Darre

*************************************************************************/
#include "StdAfx.h"
#include "HUDDraw.h"

//-----------------------------------------------------------------------------------------------------

CHUDDraw::CHUDDraw()
{
	m_pRenderer = GetISystem()->GetIRenderer();

	m_pFont = GetISystem()->GetICryFont()->GetFont("default");
	assert(m_pFont);
	m_pFont->UseRealPixels(true);
	m_pFont->SetSize(vector2f(20.0f,20.0f));
}

//-----------------------------------------------------------------------------------------------------

CHUDDraw::~CHUDDraw()
{
	for(TTexturesMap::iterator iter=m_texturesMap.begin(); iter!=m_texturesMap.end(); iter++)
	{
		SAFE_RELEASE((*iter).second);
	}
}

//-----------------------------------------------------------------------------------------------------

void CHUDDraw::PreRender()
{
	m_pRenderer->SetState(GS_BLSRC_SRCALPHA|GS_BLDST_ONE|GS_NODEPTHTEST);
	m_pRenderer->SetColorOp(255,255,eCA_Texture|(eCA_Diffuse<<3),eCA_Texture|(eCA_Diffuse<<3));
	m_pRenderer->SelectTMU(1);
	m_pRenderer->EnableTMU(false); 
	m_pRenderer->SelectTMU(0);
	m_pRenderer->EnableTMU(true);  
 	m_pRenderer->SetCullMode(R_CULL_DISABLE);
	m_pRenderer->Set2DMode(true,m_pRenderer->GetWidth(),m_pRenderer->GetHeight());
}

//-----------------------------------------------------------------------------------------------------

void CHUDDraw::PostRender()
{
	m_pRenderer->Set2DMode(false,0,0);
}

//-----------------------------------------------------------------------------------------------------

uint CHUDDraw::GetColorARGB(uint8 ucAlpha,uint8 ucRed,uint8 ucGreen,uint8 ucBlue)
{
	int iRGB = (m_pRenderer->GetFeatures() & RFT_RGBA);
	return (iRGB ? RGBA8(ucRed,ucGreen,ucBlue,ucAlpha) : RGBA8(ucBlue,ucGreen,ucRed,ucAlpha));
}

//-----------------------------------------------------------------------------------------------------

int CHUDDraw::CreateTexture(const char *strName)
{
	for(TTexturesMap::iterator iter=m_texturesMap.begin(); iter!=m_texturesMap.end(); iter++)
	{
		if(0 == strcmpi((*iter).second->GetName(),strName))
		{
			// Two textures with same name !
			assert(0);
			return (*iter).first;
		}
	}
	ITexture *pTexture = m_pRenderer->EF_LoadTexture(strName,FT_DONT_RELEASE,eTT_2D);
	pTexture->SetClamp(true);
	int iTextureID = pTexture->GetTextureID();
	m_texturesMap.insert(std::make_pair(iTextureID,pTexture));
	return iTextureID;
}

//-----------------------------------------------------------------------------------------------------

void CHUDDraw::DrawTriangle(float fX0,float fY0,float fX1,float fY1,float fX2,float fY2,uint uiColor)
{
	struct_VERTEX_FORMAT_P3F_COL4UB_TEX2F aVertices[3];

	aVertices[0].color.dcolor = uiColor;
	aVertices[0].xyz[0] = m_pRenderer->ScaleCoordX(fX0);
	aVertices[0].xyz[1] = m_pRenderer->ScaleCoordY(fY0);
	aVertices[0].xyz[2] = 0.0f;
	aVertices[0].st[0] = 0.0f;
	aVertices[0].st[1] = 0.0f;

	aVertices[1].color.dcolor = uiColor;
	aVertices[1].xyz[0] = m_pRenderer->ScaleCoordX(fX1);
	aVertices[1].xyz[1] = m_pRenderer->ScaleCoordY(fY1);
	aVertices[1].xyz[2] = 0.0f;
	aVertices[1].st[0] = 0.0f;
	aVertices[1].st[1] = 0.0f;

	aVertices[2].color.dcolor = uiColor;
	aVertices[2].xyz[0] = m_pRenderer->ScaleCoordX(fX2);
	aVertices[2].xyz[1] = m_pRenderer->ScaleCoordY(fY2);
	aVertices[2].xyz[2] = 0.0f;
	aVertices[2].st[0] = 0.0f;
	aVertices[2].st[1] = 0.0f;

	ushort ausIndices[] = {0,1,2};

	m_pRenderer->SetWhiteTexture();
	m_pRenderer->DrawDynVB(aVertices,ausIndices,3,sizeof(ausIndices)/sizeof(ausIndices[0]),R_PRIMV_TRIANGLES);
}

//-----------------------------------------------------------------------------------------------------

void CHUDDraw::DrawQuad(	float fX,
													float fY,
													float fSizeX,
													float fSizeY,
													uint uiDiffuse,
													uint uiDiffuseTL,
													uint uiDiffuseTR,
													uint uiDiffuseDL,
													uint uiDiffuseDR,
													uint uiTextureID,
													float fUTexCoordsTL,
													float fVTexCoordsTL,
													float fUTexCoordsTR,
													float fVTexCoordsTR,
													float fUTexCoordsDL,
													float fVTexCoordsDL,
													float fUTexCoordsDR,
													float fVTexCoordsDR)
{
	struct_VERTEX_FORMAT_P3F_COL4UB_TEX2F aVertices[4];

	float fWidth43 = m_pRenderer->GetHeight()*4.0f/3.0f;

	float scale = fWidth43/(float)m_pRenderer->GetWidth();
	float offset = (fSizeX - fSizeX*scale);
	fX += 0.5f*offset;
	fSizeX -=offset;

	aVertices[0].color.dcolor = uiDiffuse ? uiDiffuse : uiDiffuseTL;
	aVertices[0].xyz[0] = m_pRenderer->ScaleCoordX(fX);
	aVertices[0].xyz[1] = m_pRenderer->ScaleCoordY(fY);
	aVertices[0].xyz[2] = 0.0f;
	aVertices[0].st[0] = fUTexCoordsTL;
	aVertices[0].st[1] = fVTexCoordsTL;

	aVertices[1].color.dcolor = uiDiffuse ? uiDiffuse : uiDiffuseTR;
	aVertices[1].xyz[0] = m_pRenderer->ScaleCoordX(fX+fSizeX-1.0f);
	aVertices[1].xyz[1] = m_pRenderer->ScaleCoordY(fY);
	aVertices[1].xyz[2] = 0.0f;
	aVertices[1].st[0] = fUTexCoordsTR;
	aVertices[1].st[1] = fVTexCoordsTR;

	aVertices[2].color.dcolor = uiDiffuse ? uiDiffuse : uiDiffuseDL;
	aVertices[2].xyz[0] = m_pRenderer->ScaleCoordX(fX);
	aVertices[2].xyz[1] = m_pRenderer->ScaleCoordY(fY+fSizeY-1.0f);
	aVertices[2].xyz[2] = 0.0f;
	aVertices[2].st[0] = fUTexCoordsDL;
	aVertices[2].st[1] = fVTexCoordsDL;

	aVertices[3].color.dcolor = uiDiffuse ? uiDiffuse : uiDiffuseDR;
	aVertices[3].xyz[0] = m_pRenderer->ScaleCoordX(fX+fSizeX-1.0f);
	aVertices[3].xyz[1] = m_pRenderer->ScaleCoordY(fY+fSizeY-1.0f);
	aVertices[3].xyz[2] = 0.0f;
	aVertices[3].st[0] = fUTexCoordsDR;
	aVertices[3].st[1] = fVTexCoordsDR;

	if(uiTextureID)
	{
		m_pRenderer->SetTexture(uiTextureID);
	}
	else
	{
		m_pRenderer->SetWhiteTexture();
	}

	ushort ausIndices[] = {0,1,2,3};

	m_pRenderer->DrawDynVB(aVertices,ausIndices,4,4,R_PRIMV_TRIANGLE_STRIP);
}

//-----------------------------------------------------------------------------------------------------

void CHUDDraw::DrawImage(int iTextureID,	float fX,
																					float fY,
																					float fSizeX,
																					float fSizeY,
																					float fAngleInDegrees,
																					float fRed,
																					float fGreen,
																					float fBlue,
																					float fAlpha,
																					float fS0,
																					float fT0,
																					float fS1,
																					float fT1)
{
	float fWidth43 = m_pRenderer->GetHeight()*4.0f/3.0f;

	float scale = fWidth43/(float)m_pRenderer->GetWidth();
	float offset = (fSizeX - fSizeX*scale);
	fX += 0.5f*offset;
	fSizeX -=offset;

	m_pRenderer->Draw2dImage(	fX,
														fY+fSizeY,
														fSizeX,
														-fSizeY,
														iTextureID,
														fS0,fT0,fS1,fT1,
														fAngleInDegrees,
														fRed,
														fGreen,
														fBlue,
														fAlpha);
}

//-----------------------------------------------------------------------------------------------------

void CHUDDraw::DrawImageCentered(int iTextureID,	float fX,
																									float fY,
																									float fSizeX,
																									float fSizeY,
																									float fAngleInDegrees,
																									float fRed,
																									float fGreen,
																									float fBlue,
																									float fAlpha)
{
	float fImageX = fX - 0.5f * fSizeX;
	float fImageY = fY - 0.5f * fSizeY;

	DrawImage(iTextureID,fImageX,fImageY,fSizeX,fSizeY,fAngleInDegrees,fRed,fGreen,fBlue,fAlpha);
}

//-----------------------------------------------------------------------------------------------------

void CHUDDraw::DrawText(float fX,float fY,const char *strText,float fAlpha,float fRed,float fGreen,float fBlue,EHUDDRAWHORIZONTAL eHUDDrawHorizontal)
{
	m_pFont->SetColor(ColorF(fRed,fGreen,fBlue,fAlpha));
	// Default is HUDDRAWHORIZONTAL_LEFT and HUDDRAWVERTICAL_TOP

	float fTextX = m_pRenderer->ScaleCoordX(fX);
	float fTextY = m_pRenderer->ScaleCoordY(fY);

	vector2f vDim = m_pFont->GetTextSize(strText);

	if(HUDDRAWHORIZONTAL_CENTER == eHUDDrawHorizontal)
	{
		fTextX -= vDim.x * 0.5f;
	}
	else if(HUDDRAWHORIZONTAL_RIGHT == eHUDDrawHorizontal)
	{
		fTextX -= vDim.x;
	}

/*	if(HUDDRAWVERTICAL_CENTER == _eHUDDrawVertical)
	{
		fTextY -= vDim.y * 0.5f;
	}
	else if(HUDDRAWVERTICAL_BOTTOM == _eHUDDrawVertical)
	{
		fTextY -= vDim.y;
	}*/

	m_pFont->DrawString(fTextX,fTextY,strText);
}

//-----------------------------------------------------------------------------------------------------