/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2006.
-------------------------------------------------------------------------

Description: 
	A HUD object that displays a menu for Tweaking gameplay variables

-------------------------------------------------------------------------
History:
- 28:02:2006  : Created by Matthew Jack

*************************************************************************/


#include "StdAfx.h"
#include "HUDTweakMenu.h"
#include "..\HUDDraw.h"
#include "IScriptSystem.h"
#include "ScriptUtils.h"
#include "ICryPak.h"

//-----------------------------------------------------------------------------------------------------

CHUDTweakMenu::CHUDTweakMenu(CHUDDraw *pHUDDraw, IScriptSystem *pScriptSystem) : CHUDObject(pHUDDraw)//, m_menu(pScriptSystem)
{
	m_fX = 100.0f;
	m_fY = 100.0f;

	m_fWidth = 200;
	m_fHeight = 200;
	
	m_bActive = false;

	m_menu = new CTweakMenu(pScriptSystem);
	m_traverser = m_menu->GetTraverser();
	m_traverser.First();
}

//-----------------------------------------------------------------------------------------------------

void CHUDTweakMenu::Init( IScriptSystem *pScriptSystem ) 
{
	// Initialise menu items, for instance noting their default state
	m_menu->Init();

	// Load any saved LUA tweaks
	LoadChanges();
}

//-----------------------------------------------------------------------------------------------------

void CHUDTweakMenu::OnUpdate(float fDeltaTime,float fFadeValue) 
{
	IScriptSystem *pScript = GetISystem()->GetIScriptSystem();
	SmartScriptTable tweakTable;
	pScript->GetGlobalValue("Tweaks", tweakTable);

	// Check it the menu needs reloading
	bool bReload = false;
	tweakTable->GetValue("RELOAD",bReload);
	if (bReload) {
		SAFE_DELETE (m_menu);
		m_menu = new CTweakMenu(pScript);
		m_traverser = m_menu->GetTraverser();
		m_traverser.First();
		bReload = false;
		tweakTable->SetValue("RELOAD",bReload);
	}

	// Check if we should save the LUA Tweak changes
	bool bSaveChanges = false;
	tweakTable->GetValue("SAVECHANGES",bSaveChanges);
	if (bSaveChanges) {
		bSaveChanges = false;
		tweakTable->SetValue("SAVECHANGES",bSaveChanges);
		WriteChanges();
	}

	// Check if it has been initialised
	if (! m_menu->IsInitialized()) {
		Init(pScript);
	}

	// Draw the menu
	if (m_bActive) DrawMenu();
}

//-----------------------------------------------------------------------------------------------------

void CHUDTweakMenu::DrawMenu() 
{
	// Set up renderer state
	GetISystem()->GetIRenderer()->SetState(GS_BLSRC_SRCALPHA|GS_BLDST_ONE|GS_NODEPTHTEST);

	ResetMenuPane();

	PrintToMenuPane("Tweak Menu", eTC_White );

	// Display the menu path
	PrintToMenuPane( GetMenuPath().c_str(), eTC_Blue );
	
	// Get a Traverser pointing at the top of this menu
	CTweakTraverser itemIter = m_traverser;
	itemIter.Top();

	while (itemIter.Next()) {
		CTweakCommon * item = itemIter.GetItem();
		string text = item->GetName();

		// Accessorize by type
		ETextColor colour = eTC_White;
		switch (item->GetType()) {
			case CTweakCommon::eTT_Menu:
				colour = eTC_Yellow; 
				break;
			case CTweakCommon::eTT_Metadata:
				colour = eTC_Green; 
				text += " " + ((CTweakMetadata*)(item))->GetValue();
				break;
		}
		 
		// Is this the currently selected item?
		if (itemIter == m_traverser) colour = eTC_Red;

		// Display it
		PrintToMenuPane( text.c_str(), colour );
	}

	// Set back renderer state
	GetISystem()->GetIRenderer()->SetState(GS_BLSRC_SRCALPHA|GS_BLDST_ONEMINUSSRCALPHA|GS_NODEPTHTEST);
}

//-----------------------------------------------------------------------------------------------------

void CHUDTweakMenu::ResetMenuPane(void) 
{
	m_nLineCount = 0;
}

//-----------------------------------------------------------------------------------------------------

void CHUDTweakMenu::PrintToMenuPane( const char * line,  ETextColor color ) 
{
	float fVertSpace = 30;
	float fR = 0.0f, fG =  0.0f , fB = 0.0f;
	float fI = 0.8f;
	switch (color) {
		case eTC_Red:				
			fR = fI; break;		// Selection
		case eTC_Green:
			fG = fI; break;		// Tweaks
		case eTC_Blue:
			fB = fI; break;		// Path
		case eTC_Yellow:
			fR = fG = fI; break;// Submenus
		case eTC_White:
			fR = fG = fB = fI; break; // Default
	}

	m_pHUDDraw->DrawText(m_fX, m_fY + fVertSpace * m_nLineCount, line, 0.8f, fR, fG, fB, HUDDRAWHORIZONTAL_LEFT);
	m_nLineCount++;
}


//-----------------------------------------------------------------------------------------------------

void CHUDTweakMenu::OnActionTweak(const char *actionName, int activationMode, float value) 
{
	if (0 == strcmp(actionName,"tweakenter")) {

		m_bActive = true;

	} else if (0 == strcmp(actionName,"tweakexit")) {

		m_bActive = false;

	} else if (0 == strcmp(actionName,"tweakleft")) {

		m_traverser.Back();

	} else if (0 == strcmp(actionName,"tweakright")) {
		
		m_traverser.Forward();

	} else if (0 == strcmp(actionName,"tweakup")) {
		
		if (!m_traverser.Previous())
			m_traverser.First();

	} else if (0 == strcmp(actionName,"tweakdown")) {

		if (!m_traverser.Next())
			m_traverser.Last();
		
	} else if (0 == strcmp(actionName,"tweakincrement")) {

		CTweakCommon *item = m_traverser.GetItem();
		if (item && item->GetType() == CTweakCommon::eTT_Metadata) 
			((CTweakMetadata*)(item))->IncreaseValue();
		
	}else if (0 == strcmp(actionName,"tweakdecrement")) {

		CTweakCommon *item = m_traverser.GetItem();
		if (item && item->GetType() == CTweakCommon::eTT_Metadata) 
			((CTweakMetadata*)(item))->DecreaseValue();

	}
}


//-----------------------------------------------------------------------------------------------------

string CHUDTweakMenu::GetMenuPath(void) const 
{
	// Check validity
	if (!m_traverser.IsRegistered()) return "No valid menu";

	// Create a copy of the traverser we can use
	CTweakTraverser backTracker = m_traverser;

	// Form string to display of menu position
	string sPathText;
	do {
		sPathText = backTracker.GetMenu()->GetName() + "->" + sPathText;
	}	while (backTracker.Back());
	return sPathText;
}

//-----------------------------------------------------------------------------------------------------

CHUDTweakMenu::~CHUDTweakMenu(void)
{
	SAFE_DELETE(m_menu);
}

//-----------------------------------------------------------------------------------------------------


SmartScriptTable CHUDTweakMenu::FetchSaveTable( void ) {
	// Could do this much more neatly with GetLuaVarRecursive

	IScriptSystem * pScripts = GetISystem()->GetIScriptSystem();
	SmartScriptTable tweakTable;
	if (! pScripts->GetGlobalValue("Tweaks", tweakTable)) {
		GetISystem()->GetILog()->Log("Can't find Tweak table");
	}

	SmartScriptTable saveTable;
	if (!tweakTable->GetValue("TweaksSave", saveTable)) {
		GetISystem()->GetILog()->Log("Can't find Tweak save tables");
	}

	return saveTable;
} 


//-----------------------------------------------------------------------------------------------------


void CHUDTweakMenu::WriteChanges( void ) {
	
	SmartScriptTable saveTable = FetchSaveTable();
	if (! saveTable.GetPtr()) return;

	m_menu->StoreChanges( saveTable.GetPtr() );


	//FILE *file=GetISystem()->GetIPak()->GetAlias("Game\\Scripts",true);		
	const char *szFolder=GetISystem()->GetIPak()->GetAlias("Scripts",true);		
	char szScriptFolder[512];	
	_snprintf(szScriptFolder,512,"Game\\%s\\TweaksSave.lua",szFolder);
	FILE *file=fxopen(szScriptFolder,"wt");
	if (file) {
		string luaText;
		DumpLuaTable(saveTable.GetPtr(), file, luaText);
		fprintf(file, "%s", luaText.c_str());
		fclose(file);
	} else {
		GetISystem()->GetILog()->Log("No save - failed to open Tweak save file");
	}
}

//-----------------------------------------------------------------------------------------------------

void CHUDTweakMenu::LoadChanges( void ) {

	// We should probably reload the script first

	SmartScriptTable saveTable = FetchSaveTable();
	if (! saveTable.GetPtr()) return;

	// Go through and apply these LUA values
	// Identify and recurse on each element of the table
	IScriptTable::Iterator iter = saveTable->BeginIteration();
	while (saveTable->MoveNext(iter)) {

		SetLuaVarRecursive(iter.sKey, iter.value);
	}
	saveTable->EndIteration(iter);
}
