/*************************************************************************
  Crytek Source File.
  Copyright (C), Crytek Studios, 2001-2004.
 -------------------------------------------------------------------------
  $Id$
  $DateTime$
  Description: C++ Item Implementation
  
 -------------------------------------------------------------------------
  History:
  - 27:10:2004   11:25 : Created by Mrcio Martins

*************************************************************************/
#ifndef __ITEM_H__
#define __ITEM_H__

#if _MSC_VER > 1000
# pragma once
#endif


#include <IScriptSystem.h>
#include <IGameFramework.h>
#include <IItemSystem.h>
#include <ICryAnimation.h>
#include <IActionMapManager.h>
#include <map>
#include <list>


#include "ItemScheduler.h"


#define ITEM_ARMS_ATTACHMENT_NAME		"arms_fp"

#define ITEM_DESELECT_POSE					"nw"

#define ITEM_FIRST_PERSON_TOKEN			"_fp"
#define ITEM_THIRD_PERSON_TOKEN			""


enum EItemUpdateSlots
{
  eIUS_General = 0,
	eIUS_Zooming = 1,
	eIUS_FireMode = 2,  
	eIUS_Scheduler = 3,  
};


class CActor;
class CItem :
	public ICharInstanceSink,
	public IGameObjectPhysics,
	public CGameObjectExtensionHelper<CItem, IItem>
{
	friend class CScriptBind_Item;
public:
	//------------------------------------------------------------------------
	// Typedefs
	//------------------------------------------------------------------------
	enum eGeometrySlot
	{
		eIGS_FirstPerson = 0,
		eIGS_ThirdPerson,
		eIGS_Arms,
		eIGS_Aux0,
		eIGS_Owner,
		eIGS_Last,
	};

	enum ePhysicalization
	{
		eIPhys_Physicalized,
		eIPhys_NotPhysicalized,
	};

	enum ETimer
	{
		eIT_Flying = 0,
		eIT_Last,
	};

	enum ePlayActionFlags
	{
		eIPAF_FirstPerson				= 1 << eIGS_FirstPerson,
		eIPAF_ThirdPerson				= 1 << eIGS_ThirdPerson,
		eIPAF_Arms							= 1 << eIGS_Arms,
		eIPAF_Aux0							= 1 << eIGS_Aux0,
		eIPAF_Owner							= 1 << eIGS_Owner,
		eIPAF_ForceFirstPerson	= 1 << 15,
		eIPAF_ForceThirdPerson	= 1 << 16,
		eIPAF_NoBlend						= 1 << 17,
		eIPAF_CleanBlending			= 1 << 18,
		eIPAF_Animation					= 1 << 19,
		eIPAF_Sound							= 1 << 20,
	  eIPAF_SoundLooped				= 1 << 21,
		eIPAF_SoundStartPaused	= 1 << 22,
		eIPAF_RestartAnimation	= 1 << 23,
		eIPAF_RepeatLastFrame	= 1 << 24,
		eIPAF_Default						= eIPAF_FirstPerson|eIPAF_ThirdPerson|eIPAF_Owner|eIPAF_Sound|eIPAF_Animation,
	};

	enum eViewMode
	{
		eIVM_FirstPerson = 1,
		eIVM_ThirdPerson = 2,
	};

	enum eScriptEventTable
	{
		eISET_Server				= 1<<1,
		eISET_Client				= 1<<2,
		eISET_Root					= 1<<3,
		eISET_All						= eISET_Client|eISET_Server|eISET_Root,
		eISET_ClientServer	= eISET_Client|eISET_Server
	};

	struct SStats
	{
		SStats()
		:	fp(false),
			mounted(false),
			mount_dir(Vec3(0.0f,0.0f,0.0f)),
			pickable(true),
			selectable(true),
			dropped(false),
			flying(false),
			viewmode(0),
			used(false),
			sound_enabled(true),
			hand(eIH_Right)
		{};

		bool	fp;
		bool	mounted;
		Vec3	mount_dir;
		bool	pickable;
		bool	selectable;
		bool	dropped;
		bool	flying;
		int		viewmode;
		bool	used;
		bool	sound_enabled;
		int		hand;
	};

	struct SParams
	{
		SParams()
		: selectable(true),
			droppable(true),
			pickable(true),
			mountable(true),
			usable(true),
			giveable(true),
			unique(true),
			arms(true),
			mass(3.5f),
			fly_timer(750),
			drop_impulse(5),
			drop_impulse_pos(0.005f,0,0.005f),
			drop_angles(0,0,0),

			pose("nw"),
			attachment("item_attachment")
		{};

		bool		selectable;
		bool		droppable;
		bool		pickable;
		bool		mountable;
		bool		usable;
		bool		giveable;
		bool		unique;
		bool		arms;
		float		mass;
		float		drop_impulse;
		Vec3		drop_impulse_pos;
		Vec3		drop_angles;
		int			fly_timer;
		string	pose;
		string	attachment;
		string	dual_wield_suffix;
	};


	struct SMountParams
	{
		SMountParams()
		:	min_pitch(0.0f),
			max_pitch(80.0f),
			yaw_range(0.0f),
			eye_distance(0.85f),
			eye_height(0.0f)
		{};

		string	pivot;
		float		min_pitch;
		float		max_pitch;
		float		yaw_range;
		float		eye_distance;
		float		eye_height;
		string	left_hand_helper;
		string	right_hand_helper;
	};

	struct SAccessoryParams
	{
		string		attach_helper;
		string		attach_action;
		string		attach_layer;
		string		detach_action;
		string    firemode;
		bool      exclusive;
		string		zoommode;
		const IItemParamsNode *params;
	};

	struct SAudio
	{
		string		name;
		string		static_name;
		float			airadius;
		bool			isstatic;
		tSoundID	id;
	};

	struct SLayer
	{
		std::vector<string>	bones;
		string	name[eIGS_Last];
		int			id[eIGS_Last];
		bool		isstatic;
	};

	struct SAnimation
	{
		string	name;
		float		speed;
		float		blend;
	};

	struct SAction
	{
		SAnimation	animation[eIGS_Last];
		SAudio			sound;		
	};

	struct SAttachmentHelper
	{
		string	name;
		string	bone;
		int			slot;
	};

	struct SEffectInfo
	{
		int			characterSlot;
		string	helper;
		int			slot;
	};

	struct SGeometry
	{
		string	name;
		Vec3		position;
		Vec3		angles;
		float		scale;
	};

	typedef std::map<string, SAction>					TActionMap;
	typedef std::map<string, SLayer>					TLayerMap;
	typedef std::map<string, int>							TActiveLayerMap;
	typedef std::vector<SAttachmentHelper>		THelperVector;
	typedef std::map<string, bool>						TDualWieldSupportMap;
	typedef	std::map<uint, SEffectInfo>				TEffectInfoMap;
	typedef std::map<string, EntityId>				TAccessoryMap;
	typedef std::map<string, SAccessoryParams>TAccessoryParamsMap;
	typedef std::vector<string>								TInitialSetup;

public:
	CItem();
	virtual ~CItem();

	// IItem, IGameObjectExtension
	virtual bool Init( IGameObject * pGameObject );
	virtual void InitClient(int channelId);
	virtual void PostInitClient(int channelId);
	virtual void PostInit( IGameObject * pGameObject );
	virtual void Release();
	virtual void Serialize( TSerialize ser, unsigned aspects );
	virtual void Update( SEntityUpdateContext& ctx, int );
	virtual void PostUpdate( float frameTime ) {};
	virtual void HandleEvent( const SGameObjectEvent& ) {}
	virtual void ProcessEvent(SEntityEvent& );
	virtual void SetChannelId(uint16 id) {};
	virtual void SetAuthority(bool auth) {}

	virtual void OnAction(EntityId actorId, const char* actionId, int activationMode, float value);
	virtual void UpdateFPView(float frameTime);

	virtual IWeapon *GetIWeapon() { return 0; };

	virtual void SetOwnerId(EntityId ownerId);
	virtual EntityId GetOwnerId() const;

	virtual void PreCache();

	virtual void SetParentId(EntityId parentId);
	virtual EntityId GetParentId() const;

	virtual void SetHand(int hand);
	virtual void Use(EntityId userId);
	virtual void Select(bool select);
	virtual void Drop();
	virtual void PickUp(EntityId pickerId, bool sound, bool select=true, bool keepHistory=true);
	virtual void Physicalize(bool enable);
	virtual void Pickalize(bool enable, bool dropped);
	virtual void AttachArms(bool attach);
	virtual void Impulse(const Vec3 &position, const Vec3 &direction, float impulse);

	virtual bool CanPickUp(EntityId userId);
	virtual bool CanDrop();
	virtual bool CanUse(EntityId userId);
	virtual bool IsMounted();
	virtual bool IsUsed();

	virtual bool SupportsDualWield(const char *itemName) const;
	virtual void ResetDualWield();
	virtual IItem *GetDualWieldSlave() const;
	virtual EntityId GetDualWieldSlaveId() const;
	virtual IItem *GetDualWieldMaster() const;
	virtual EntityId GetDualWieldMasterId() const;
	virtual void SetDualWieldMaster(EntityId masterId);
	virtual void SetDualWieldSlave(EntityId slaveId);
	virtual bool IsDualWield() const;
	virtual bool IsDualWieldMaster() const;
	virtual bool IsDualWieldSlave() const;

	virtual Vec3 GetMountedAngleLimits() const;

	virtual void EnableSelect(bool enable);	
	virtual bool CanSelect() const;
	virtual bool IsSelected() const;

	virtual void MountAt(const Vec3 &pos);
	virtual void MountAtEntity(EntityId entityId, const Vec3 &pos, const Ang3 &angles);
	virtual void StartUse(EntityId userId) ;
	virtual void StopUse(EntityId  userId);

	virtual void EnableSound(bool enable);
	virtual bool IsSoundEnabled() const;
	virtual bool IsModifying() { return m_bModifying || m_bTransitioning; }
	// ~IItem

	// ICharInstanceSink
	virtual void OnStartAnimation(const char *animation) {};
	virtual void OnAnimationEvent(const char *animation, AnimSinkEventData data) {};
	virtual void OnEndAnimation(const char *animation) {};
	// ~ICharInstanceSink

	// IGameObjectPhysics
	virtual bool SetProfile( uint8 profile );
	virtual bool SerializeProfile( TSerialize ser, uint8 profile );
	// ~IGameObjectPhysics
  
	// Events
	virtual void OnStartUsing();
	virtual void OnStopUsing();
	virtual void OnSelect(bool select);
	virtual void OnSelected(bool selected);
	virtual void OnEnterFirstPerson();
	virtual void OnEnterThirdPerson();
  virtual void OnReset();
  
	virtual const SStats &GetStats() const { return m_stats; };
	virtual const SParams &GetParams() const { return m_params; };

public:

	// params
	virtual bool ReadItemParams(const IItemParamsNode *root);
	virtual bool ReadParams(const IItemParamsNode *params);
	virtual bool ReadGeometry(const IItemParamsNode *geometry);
	virtual bool ReadActions(const IItemParamsNode *actions);
	virtual bool ReadAction(const IItemParamsNode *action, SAction *pAction);
	virtual bool ReadLayers(const IItemParamsNode *layers);
	virtual bool ReadLayer(const IItemParamsNode *layer, SLayer *pLayer);
	virtual bool ReadAccessories(const IItemParamsNode *accessories);
	virtual bool ReadAccessoryParams(const IItemParamsNode *accessory, SAccessoryParams *params);
	virtual bool SetGeometryFromParams(int slot, const IItemParamsNode *geometry);
	virtual int TargetToSlot(const char *name);

	// accessories
	virtual CItem *AddAccessory(const char *name);
	virtual void RemoveAccessory(const char *name);
	virtual void AttachAccessory(const char *name, bool attach, bool noanim);
	virtual void AttachAccessoryPlaceHolder(const char *name, bool attach);
	virtual CItem *GetAccessoryPlaceHolder(const char *name);
	virtual CItem *GetAccessory(const char *name);
	virtual SAccessoryParams *CItem::GetAccessoryParams(const char *name);
	virtual bool IsAccessoryHelperFree(const char *helper);
	virtual void InitialSetup();
	virtual void ReAttachAccessories();
	virtual void ReAttachAccessory(const char *name);
	virtual void AccessoriesChanged();
	virtual void FixAccessories(SAccessoryParams *newParams, bool attach) {};

	// effects
	uint AttachEffect(int slot, uint id, bool attach, const char *effectName=0, const char *helper=0,
		const Vec3 &offset=Vec3(0,0,0), const Vec3 &dir=Vec3(1,0,0), float scale=1.0f, bool prime=true);
	uint AttachLight(int slot, uint id, bool attach, float radius=5.0f, const Vec3 &color=Vec3(1,1,1),
		const Vec3 &specular=Vec3(1,1,1), const char *projectTexture=0, float projectFov=0, const char *helper=0,
		const Vec3 &offset=Vec3(0,0,0), const Vec3 &dir=Vec3(0,1,0));
	void SpawnEffect(int slot, const char *effectName, const char *helper, const Vec3 &offset=Vec3(0,0,0),
		const Vec3 &dir=Vec3(0,1,0), float scale=1.0f);
	IParticleEmitter *GetEffectEmitter(uint id) const;
	void SetEffectWorldTM(uint id, const Matrix34 &tm);
	Matrix34 GetEffectWorldTM(uint it);

	// misc
	void AttachToHand(bool attach);
	void RequireUpdate(bool enable);
	void EnableUpdate(bool enable);
	void Hide(bool hide);
	void HideArms(bool hide);
	void HideItem(bool hide);
	void SetBusy(bool busy) { m_scheduler.SetBusy(busy); };
	bool IsBusy() { return m_scheduler.IsBusy(); };
	CItemScheduler *GetScheduler() { return &m_scheduler; };
	IItemSystem *GetIItemSystem() { return m_pItemSystem; };

	IEntity *GetOwner() const;
	CActor *GetOwnerActor() const;
	CActor *GetActor(EntityId actorId) const;
	IInventory *GetActorInventory(IActor *pActor) const;

	// view
	bool IsOwnerFP();
	bool IsCurrentItem();
	void UpdateMounted(float frameTime);
	void CheckViewChange();
	void SetViewMode(int mode);
	void CopyRenderFlags(IEntity *pOwner);
	void ResetRenderFlags();

	// character attachments
	bool CreateCharacterAttachment(int slot, const char *name, int type, const char *bone);
	void DestroyCharacterAttachment(int slot, const char *name);
	void ResetCharacterAttachment(int slot, const char *name);
	const char *GetCharacterAttachmentBone(int slot, const char *name);
	void SetCharacterAttachment(int slot, const char *name, IEntity *pEntity, int flags);
	void SetCharacterAttachment(int slot, const char *name, IStatObj *pObj, int flags);
	void SetCharacterAttachment(int slot, const char *name, ICharacterInstance *pCharacter, int flags);
	void SetCharacterAttachment(int slot, const char *name, CDLight &light, int flags);
	void SetCharacterAttachment(int slot, const char *name, IEntity *pEntity, int objSlot, int flags);
	void SetCharacterAttachmentLocalTM(int slot, const char *name, const Matrix34 &tm);
	void SetCharacterAttachmentWorldTM(int slot, const char *name, const Matrix34 &tm);
	Matrix34 GetCharacterAttachmentLocalTM(int slot, const char *name);
	Matrix34 GetCharacterAttachmentWorldTM(int slot, const char *name);
	void HideCharacterAttachment(int slot, const char *name, bool hide);
	void HideCharacterAttachmentMaster(int slot, const char *name, bool hide);

	void CreateAttachmentHelpers(int slot);
	void DestroyAttachmentHelpers(int slot);


	// HACK
	virtual void SetDummyOffset(const Vec3 &dummyOffset) {m_dummyOffset=dummyOffset;};
	// ~HACK


	// freeze
	void Freeze(bool freeze);

	// resource
	bool SetGeometry(int slot, const char *name, const Vec3 &poffset=Vec3(0,0,0), const Vec3 &aoffset=Vec3(0,0,0), float scale=1.0f, bool forceReload=false);
	void SetDefaultIdleAnimation(int slot, const char *actionName);
	void ForceSkinning(bool always);
	void EnableHiddenSkinning(bool force);
	void FixResourceName(string &name, int flags, const char *hand=0, const char *suffix=0, const char *pose=0, const char *pov=0, const char *env=0);
	tSoundID PlayAction(const char *actionName, int layer=0, bool loop=false, uint flags = eIPAF_Default);
	void PlayAnimation(const char *animationName, int layer=0, bool loop=false, uint flags = eIPAF_Default);
	void PlayAnimationEx(const char *animationName, int slot=eIGS_FirstPerson, int layer=0, bool loop=false, float blend=0.175f, float speed=1.0f, uint flags = eIPAF_Default);
	void PlayLayer(const char *name, int flags = eIPAF_Default, bool record=true);
	void StopLayer(const char *name, int flags = eIPAF_Default, bool record=true);
	void RestoreLayers();
	void ResetAnimation(int layer=0, uint flags = eIPAF_Default);
	uint GetCurrentAnimationTime(int slot);
	uint GetCurrentAnimationEnd(int slot);
	uint GetCurrentAnimationStart(int slot);
	void DrawSlot(int slot, bool draw, bool near=false);
	Vec3 GetSlotHelperPos(int slot, const char *helper, bool worldSpace);
	void StopSound(tSoundID id);
	void Quiet();
	ISound *GetISound(tSoundID id);
	void ReleaseStaticSound(SAudio *sound);
	void ReleaseStaticSounds();

	IEntitySoundProxy *GetSoundProxy(bool create=false);
	IEntityRenderProxy *GetRenderProxy(bool create=false);
	IEntityPhysicalProxy *GetPhysicalProxy(bool create=false);
		
	// net
	virtual bool IsServer() { return m_pGameFramework->IsServer(); };
	virtual bool IsClient() { return m_pGameFramework->IsClient(); };

	EntityId NetGetOwnerId() const;
	void NetSetOwnerId(EntityId id);

	// properties
	template<typename T>bool GetEntityProperty(const char *name, T &value)
	{
		SmartScriptTable props;
		if (GetEntity()->GetScriptTable()->GetValue("Properties", props))
			return props->GetValue(name, value);
		return false;
	}

	// script
	bool CallScriptEvent(IScriptTable *pScriptTable, const char *name)
	{
		IScriptSystem *pSS = pScriptTable->GetScriptSystem();
		if (pScriptTable->GetValueType(name) != svtFunction)
			return false;
		pSS->BeginCall(pScriptTable, name); pSS->PushFuncParam(GetEntity()->GetScriptTable());
		bool result = false;
		pSS->EndCall(result);
		return result;
	}
	template<typename P1>
		bool CallScriptEvent(IScriptTable *pScriptTable, const char *name, P1 &p1)
	{
		IScriptSystem *pSS = pScriptTable->GetScriptSystem();
		if (pScriptTable->GetValueType(name) != svtFunction)
			return false;
		pSS->BeginCall(pScriptTable, name); pSS->PushFuncParam(GetEntity()->GetScriptTable());
		pSS->PushFuncParam(p1);
		bool result = false;
		pSS->EndCall(result);
		return result;
	}
	template<typename P1, typename P2>
		bool CallScriptEvent(IScriptTable *pScriptTable, const char *name, P1 &p1, P2 &p2)
	{
		IScriptSystem *pSS = pScriptTable->GetScriptSystem();
		if (pScriptTable->GetValueType(name) != svtFunction)
			return false;
		pSS->BeginCall(pScriptTable, name); pSS->PushFuncParam(GetEntity()->GetScriptTable());
		pSS->PushFuncParam(p1); pSS->PushFuncParam(p2);
		bool result = false;
		pSS->EndCall(result);
		return result;
	}
	template<typename P1, typename P2, typename P3, typename P4>
		bool CallScriptEvent(IScriptTable *pScriptTable, const char *name, P1 &p1, P2 &p2, P3 &p3)
	{
		IScriptSystem *pSS = pScriptTable->GetScriptSystem();
		if (pScriptTable->GetValueType(name) != svtFunction)
			return false;
		pSS->BeginCall(pScriptTable, name); pSS->PushFuncParam(GetEntity()->GetScriptTable());
		pSS->PushFuncParam(p1); pSS->PushFuncParam(p2); pSS->PushFuncParam(p3);
		bool result = false;
		pSS->EndCall(result);
		return result;
	}
	template<typename P1, typename P2, typename P3, typename P4>
		bool CallScriptEvent(IScriptTable *pScriptTable, const char *name, P1 &p1, P2 &p2, P3 &p3, P4 &p4)
	{
		IScriptSystem *pSS = pScriptTable->GetScriptSystem();
		if (pScriptTable->GetValueType(name) != svtFunction)
			return false;
		pSS->BeginCall(pScriptTable, name); pSS->PushFuncParam(GetEntity()->GetScriptTable());
		pSS->PushFuncParam(p1); pSS->PushFuncParam(p2); pSS->PushFuncParam(p3); pSS->PushFuncParam(p4);
		bool result = false;
		pSS->EndCall(result);
		return result;
	}
	void CallScriptEvent(int tables, const char *name, bool *rserver, bool *rclient, bool *rroot)
	{
		if ((tables&eISET_Root) && !!m_stateTable[0])
		{
			bool r = CallScriptEvent(m_stateTable[0], name);
			if(rroot) *rroot = r;
		}
		if ((tables&eISET_Server) && !!m_stateTable[1] && IsServer())
		{
			bool s = CallScriptEvent(m_stateTable[1], name);
			if(rserver) *rserver = s;
		}
		if ((tables&eISET_Client) && !!m_stateTable[2] && IsClient())
		{
			bool c = CallScriptEvent(m_stateTable[2], name);
			if(rclient) *rclient = c;
		}
	}

	template<typename P1>
		void CallScriptEvent(int tables, const char *name, P1 &p1, bool *rserver, bool *rclient, bool *rroot)
	{
		if ((tables&eISET_Root) && !!m_stateTable[0])
		{
			bool r = CallScriptEvent(m_stateTable[0], name, p1);
			if(rroot) *rroot = r;
		}
		if ((tables&eISET_Server) && !!m_stateTable[1] && IsServer())
		{
			bool s = CallScriptEvent(m_stateTable[1], name, p1);
			if(rserver) *rserver = s;
		}
		if ((tables&eISET_Client) && !!m_stateTable[2] && IsClient())
		{
			bool c = CallScriptEvent(m_stateTable[2], name, p1);
			if(rclient) *rclient = c;
		}
	}
	template<typename P1, typename P2>
		void CallScriptEvent(int tables, const char *name, P1 &p1, P2 &p2, bool *rserver, bool *rclient, bool *rroot)
	{
		if ((tables&eISET_Root) && !!m_stateTable[0])
		{
			bool r = CallScriptEvent(m_stateTable[0], name, p1, p2);
			if(rroot) *rroot = r;
		}
		if ((tables&eISET_Server) && !!m_stateTable[1] && IsServer())
		{
			bool s = CallScriptEvent(m_stateTable[1], name, p1, p2);
			if(rserver) *rserver = s;
		}
		if ((tables&eISET_Client) && !!m_stateTable[2] && IsClient())
		{
			bool c = CallScriptEvent(m_stateTable[2], name, p1, p2);
			if(rclient) *rclient = c;
		}
	}
	template<typename P1, typename P2, typename P3>
		void CallScriptEvent(int tables, const char *name, P1 &p1, P2 &p2, P3 &p3, bool *rserver, bool *rclient, bool *rroot)
	{
		if ((tables&eISET_Root) && !!m_stateTable[0])
		{
			bool r = CallScriptEvent(m_stateTable[0], name, p1, p2, p3);
			if(rroot) *rroot = r;
		}
		if ((tables&eISET_Server) && !!m_stateTable[1] && IsServer())
		{
			bool s = CallScriptEvent(m_stateTable[1], name, p1, p2, p3);
			if(rserver) *rserver = s;
		}
		if ((tables&eISET_Client) && !!m_stateTable[2] && IsClient())
		{
			bool c = CallScriptEvent(m_stateTable[2], name, p1, p2, p3);
			if(rclient) *rclient = c;
		}
	}
	template<typename P1, typename P2, typename P3, typename P4>
		void CallScriptEvent(int tables, const char *name, P1 &p1, P2 &p2, P3 &p3, P4 &p4, bool *rserver, bool *rclient, bool *rroot)
	{
		if ((tables&eISET_Root) && !!m_stateTable[0])
		{
			bool r = CallScriptEvent(m_stateTable[0], name, p1, p2, p3, p4);
			if(rroot) *rroot = r;
		}
		if ((tables&eISET_Server) && !!m_stateTable[1] && IsServer())
		{
			bool s = CallScriptEvent(m_stateTable[1], name, p1, p2, p3, p4);
			if(rserver) *rserver = s;
		}
		if ((tables&eISET_Client) && !!m_stateTable[2] && IsClient())
		{
			bool c = CallScriptEvent(m_stateTable[2], name, p1, p2, p3, p4);
			if(rclient) *rclient = c;
		}
	};

protected:


	// data
	TActionMap						m_actions;
	SParams								m_params;
	SMountParams					m_mountparams;
	SStats								m_stats;
	THelperVector					m_helpers;
	TEffectInfoMap				m_effects;
	TLayerMap							m_layers;
	TActiveLayerMap				m_activelayers;
	TAccessoryMap					m_accessories;
	TAccessoryParamsMap		m_accessoryparams;

	float									m_fVolume;	//non-physics volume for player pick-up check
	
	TInitialSetup					m_initialSetup;

	uint									m_effectGenId;

	TDualWieldSupportMap	m_dualWieldSupport;
	EntityId							m_dualWieldMasterId;
	EntityId							m_dualWieldSlaveId;

	bool									m_requireUpdate;

	CItemScheduler				m_scheduler;

	EntityId							m_ownerId;
	EntityId							m_parentId;

	SmartScriptTable			m_stateTable[3];		// root, server, client

	string								m_idleAnimation[eIGS_Last];
	uint									m_animationTime[eIGS_Last];
	uint									m_animationEnd[eIGS_Last];
	float									m_animationSpeed[eIGS_Last];


	SGeometry							m_fpgeometry[3];		// have to save the first person geometry
																				// to support dynamic change of hand
	EntityId							m_hostId;

	IScriptTable					*m_pEntityScript;
	IEntitySystem					*m_pEntitySystem;
	IItemSystem						*m_pItemSystem;
	IGameFramework				*m_pGameFramework;

	protected:
	bool									m_bModifying;
	bool									m_bTransitioning;

	Vec3									m_dummyOffset;
	bool									m_frozen;

public:
	static ICVar					*i_staticfiresounds;
	static ICVar					*i_offset_front;
	static ICVar					*i_offset_up;
	static ICVar					*i_offset_right;
	static ICVar					*i_unlimitedammo;
	static ICVar					*i_lighteffects;
	static ICVar					*i_particleeffects;
};


#endif //__ITEM_H__
