/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id$
$DateTime$

-------------------------------------------------------------------------
History:
- 30:8:2005   12:33 : Created by Mrcio Martins

*************************************************************************/
#include "StdAfx.h"
#include "Item.h"
#include "ItemParamReader.h"
#include <ISound.h>


//------------------------------------------------------------------------
bool CItem::ReadItemParams(const IItemParamsNode *root)
{
	const IItemParamsNode *params = root->GetChild("params");
	const IItemParamsNode *geometry = root->GetChild("geometry");
	const IItemParamsNode *actions = root->GetChild("actions");
	const IItemParamsNode *layers = root->GetChild("layers");
	const IItemParamsNode *accessories = root->GetChild("accessories");

	if (params) ReadParams(params);
	if (actions) ReadActions(actions);
	if (geometry) ReadGeometry(geometry);
	if (layers) ReadLayers(layers);
	if (accessories) ReadAccessories(accessories);

	return true;
}

#define ReadValue(hold, param)	reader.Read(#param, hold.param)

//------------------------------------------------------------------------			
bool CItem::ReadParams(const IItemParamsNode *params)
{
	{
		CItemParamReader reader(params);
		ReadValue(m_params, selectable);
		ReadValue(m_params, droppable);
		ReadValue(m_params, pickable);
		ReadValue(m_params, mountable);
		ReadValue(m_params, usable);
		ReadValue(m_params, giveable);
		ReadValue(m_params, unique);
		ReadValue(m_params, arms);
		ReadValue(m_params, mass);
		ReadValue(m_params, fly_timer);
		ReadValue(m_params, drop_impulse);
		ReadValue(m_params, drop_impulse_pos);
		ReadValue(m_params, drop_angles);
		ReadValue(m_params, pose);
		ReadValue(m_params, attachment);
		ReadValue(m_params, dual_wield_suffix);
	}

	const IItemParamsNode *dw = params->GetChild("dualWield");
	if (dw)
	{
		int n = dw->GetChildCount();
		for (int i=0; i<n; i++)
		{
			const IItemParamsNode *item = dw->GetChild(i);
			if (!stricmp(dw->GetChildName(i), "item"))
			{
				const char *name = item->GetAttribute("value");
				if (name && name[0])
					m_dualWieldSupport.insert(TDualWieldSupportMap::value_type(name, true));
			}
			else if (!stricmp(dw->GetChildName(i), "suffix"))
			{
				const char *suffix = item->GetAttribute("value");
				if (suffix)
					m_params.dual_wield_suffix = suffix;
			}
		}
	}

	const IItemParamsNode *mp = params->GetChild("mount");
	if (mp)
	{
		CItemParamReader reader(mp);
		ReadValue(m_mountparams, pivot);
		ReadValue(m_mountparams, eye_distance);
		ReadValue(m_mountparams, eye_height);
		ReadValue(m_mountparams, min_pitch);
		ReadValue(m_mountparams, max_pitch);
		ReadValue(m_mountparams, yaw_range);
		ReadValue(m_mountparams, left_hand_helper);
		ReadValue(m_mountparams, right_hand_helper);
	}

	return true;
}

#undef ReadValue

//------------------------------------------------------------------------
bool CItem::ReadGeometry(const IItemParamsNode *geometry)
{
	// read teh helpers
	m_helpers.clear();

	const IItemParamsNode *attachments = geometry->GetChild("boneattachments");
	if (attachments)
	{

		int n = attachments->GetChildCount();
		SAttachmentHelper helper;
		for (int i=0; i<n; i++)
		{
			const IItemParamsNode *attachment = attachments->GetChild(i);
			const char *slot = attachment->GetAttribute("target");
			const char *name = attachment->GetAttribute("name");
			const char *bone = attachment->GetAttribute("bone");

			int islot = TargetToSlot(slot);
			if (islot == eIGS_Last)
			{
				GameWarning("Invalid attachment helper target for item '%s'! Skipping...", GetEntity()->GetName());
				continue;
			}

			if (!name || !bone)
			{
				GameWarning("Invalid attachment helper specification for item '%s'! Skipping...", GetEntity()->GetName());
				continue;
			}

			helper.name = name;
			helper.bone = bone;
			helper.slot = islot;

			m_helpers.push_back(helper);
		}
	}

	bool result = true;
	int n = geometry->GetChildCount();
	for (int i=0; i<n; i++)
	{
		const IItemParamsNode *child = geometry->GetChild(i);
		if (!stricmp(geometry->GetChildName(i), "firstperson"))
			result = result && SetGeometryFromParams(eIGS_FirstPerson, child);
		else if (!stricmp(geometry->GetChildName(i), "thirdperson"))
			result = result && SetGeometryFromParams(eIGS_ThirdPerson, child);
		if (result)
		{
			PlayAction("idle", 0, true, eIPAF_Default|eIPAF_NoBlend);
			ForceSkinning(true);
		}
	}

	return result;
}

//------------------------------------------------------------------------
bool CItem::ReadActions(const IItemParamsNode *actions)
{
	int n = actions->GetChildCount();
	for (int i=0; i<n; i++)
	{
		const IItemParamsNode *actionparams = actions->GetChild(i);
		if (actionparams)
		{
			SAction action;
			if (ReadAction(actionparams, &action))
			{	
				const char *name = actionparams->GetAttribute("name");
				m_actions.insert(TActionMap::value_type(name, action));
			}
		}
	}

	return true;
}

//------------------------------------------------------------------------
bool CItem::ReadAction(const IItemParamsNode *actionparams, SAction *pAction)
{
	const char *actionName = actionparams->GetAttribute("name");
	if (!actionName)
	{
		GameWarning("Missing action name for item '%s'! Skipping...", GetEntity()->GetName());
		return false;
	}

	int n = actionparams->GetChildCount();
	for (int i=0; i<n; i++)
	{
		const IItemParamsNode *child = actionparams->GetChild(i);
		const char *childName = actionparams->GetChildName(i);
		if (!stricmp(childName, "sound"))
		{
			const char *name = child->GetAttribute("name");
			if (!name)
			{
				GameWarning("Missing name of sound for action '%s' in item '%s'! Skipping...", actionName, GetEntity()->GetName());
				return false;
			}
			float radius = 1.0f; child->GetAttribute("radius", radius);
			int isstatic = 0; child->GetAttribute("static", isstatic);
			pAction->sound.name = name;
			pAction->sound.airadius = radius;
			pAction->sound.isstatic = isstatic!=0;
			pAction->sound.id = INVALID_SOUNDID;
		}
		else if (!stricmp(childName, "animation"))
		{
			const char *name = child->GetAttribute("name");
			if (!name)
			{
				GameWarning("Missing name of animation for action '%s' in item '%s'! Skipping...", actionName, GetEntity()->GetName());
				return false;
			}

			const char *slot = child->GetAttribute("target");
			int islot = TargetToSlot(slot);
			
			if (islot == eIGS_Last)
			{
				GameWarning("Invalid animation target '%s' of for action '%s' in item '%s'! Skipping...", slot, actionName, GetEntity()->GetName());
				return false;
			}

			float speed = 1.0f; child->GetAttribute("speed", speed);
			float blend = 0.125f; child->GetAttribute("blendTime", blend);
			pAction->animation[islot].name = name;
			pAction->animation[islot].speed = speed;
			pAction->animation[islot].blend = blend;
		}
		else
		{
			GameWarning("Unknown param '%s' for action '%s' in item '%s'! Skipping...", childName, actionName, GetEntity()->GetName());
			return false;
		}
	}

	return true;
}

//------------------------------------------------------------------------
bool CItem::ReadLayers(const IItemParamsNode *layers)
{
	int n = layers->GetChildCount();
	for (int i=0; i<n; i++)
	{
		const IItemParamsNode *layer = layers->GetChild(i);
		if (layer)
		{
			SLayer lyr;
			if (ReadLayer(layer, &lyr))
			{	
				const char *name = layer->GetAttribute("name");
				m_layers.insert(TLayerMap::value_type(name, lyr));
			}
		}
	}

	return true;
}

//------------------------------------------------------------------------
bool CItem::ReadLayer(const IItemParamsNode *layer, SLayer *pLayer)
{
	const char *layerName = layer->GetAttribute("name");
	if (!layerName)
	{
		GameWarning("Missing layer name for item '%s'! Skipping...", GetEntity()->GetName());
		return false;
	}


	pLayer->isstatic=false;
	int isstatic;
	if (layer->GetAttribute("static", isstatic))
		pLayer->isstatic = isstatic!=0;

	int n = layer->GetChildCount();
	for (int i=0; i<n; i++)
	{
		const IItemParamsNode *child = layer->GetChild(i);
		if (!stricmp(layer->GetChildName(i), "animation"))
		{
			const char *name = child->GetAttribute("name");
			if (!name)
			{
				GameWarning("Missing name of animation for layer '%s' in item '%s'! Skipping...", layerName, GetEntity()->GetName());
				return false;
			}

			const char *slot = child->GetAttribute("target");
			int islot = TargetToSlot(slot);

			if (islot == eIGS_Last)
			{
				GameWarning("Invalid animation target '%s' of for layer '%s' in item '%s'! Skipping...", slot, layerName, GetEntity()->GetName());
				return false;
			}

			pLayer->name[islot] = name;
			pLayer->id[islot] = 0; child->GetAttribute("layerId", pLayer->id[islot]);
		}
		else if (!stricmp(layer->GetChildName(i), "bones"))
		{
			int nb = child->GetChildCount();
			for (int b=0; b<nb; b++)
			{
				const IItemParamsNode *bone = child->GetChild(b);
				if (!stricmp(child->GetChildName(b), "bone"))
				{
					const char *name = bone->GetAttribute("name");
					if (!name)
					{
						GameWarning("Missing name of bone for layer '%s' in item '%s'! Skipping...", layerName, GetEntity()->GetName());
						return false;
					}

					pLayer->bones.push_back(name);
				}
			}
		}
		else
		{
			GameWarning("Unknown param '%s' for layer '%s' in item '%s'! Skipping...", layer->GetChildName(i), layerName, GetEntity()->GetName());
			return false;
		}
	}

	return true;
}

//------------------------------------------------------------------------
bool CItem::ReadAccessories(const IItemParamsNode *accessories)
{
	int n = accessories->GetChildCount();
	for (int i=0; i<n; i++)
	{
		const IItemParamsNode *child = accessories->GetChild(i);
		if (!stricmp(accessories->GetChildName(i), "accessory"))
		{
			SAccessoryParams params;
			if (!ReadAccessoryParams(child, &params))
				continue;

			const char *name = child->GetAttribute("name");
			m_accessoryparams.insert(TAccessoryParamsMap::value_type(name, params));
		}
		else if (!stricmp(accessories->GetChildName(i), "initialsetup"))
		{
			int na = child->GetChildCount();
			for (int k=0; k<na; k++)
			{
				const IItemParamsNode *accessory = child->GetChild(k);
				if (!stricmp(child->GetChildName(k), "accessory"))
				{
					const char *name = accessory->GetAttribute("name");
					if (!name || !name[0])
					{
						GameWarning("Missing accessory name for initial setup in item '%s'! Skipping...", GetEntity()->GetName());
						continue;
					}

					m_initialSetup.push_back(name);
				}
				else
				{
					GameWarning("Unknown param '%s' in initial setup for item '%s'! Skipping...", child->GetChildName(k), GetEntity()->GetName());
					continue;
				}
			}
		}
	}

	return true;
}

//------------------------------------------------------------------------
bool CItem::ReadAccessoryParams(const IItemParamsNode *accessory, SAccessoryParams *params)
{
	const char *name = accessory->GetAttribute("name");
	if (!name || !name[0])
	{
		GameWarning("Missing accessory name for item '%s'! Skipping...", GetEntity()->GetName());
		return false;
	}

	const IItemParamsNode *attach = accessory->GetChild("attach");
	const IItemParamsNode *detach = accessory->GetChild("detach");

	if (!attach || !detach)
	{
		GameWarning("Missing attach/detach details for accessory '%s' in item '%s'! Skipping...", name, GetEntity()->GetName());
		return false;
	}

	params->attach_action = attach->GetAttribute("action");
	params->attach_helper = attach->GetAttribute("helper");
	params->attach_layer = attach->GetAttribute("layer");
	params->detach_action = detach->GetAttribute("action");
	params->firemode = accessory->GetAttribute("firemode");
	params->zoommode = accessory->GetAttribute("zoommode");
	
	int exclusive = 0;
	accessory->GetAttribute("exclusive", exclusive);
	params->exclusive = exclusive!=0;
	params->params = accessory->GetChild("params");

	return true;
}

//------------------------------------------------------------------------
bool CItem::SetGeometryFromParams(int slot, const IItemParamsNode *geometry)
{
	const char *name = geometry->GetAttribute("name");
	if (!name || !name[0])
	{
		GameWarning("Missing geometry name for loading item '%s'!", GetEntity()->GetName());
		return false;
	}

	Vec3 position(0,0,0); geometry->GetAttribute("position", position);
	Vec3 angles(0,0,0);		geometry->GetAttribute("angles", angles);
	float scale=1.0f;			geometry->GetAttribute("scale", scale);

	if (slot == eIGS_FirstPerson)
	{
		const char *hand = geometry->GetAttribute("hand");
		int idx = 0;
		if (hand && hand[0])
		{
			if (!stricmp(hand, "right"))
				idx = 1;
			else if (!stricmp(hand, "left"))
				idx = 2;
			else
			{
				GameWarning("Invalid hand '%s' loading item '%s'!", hand, GetEntity()->GetName());
				return false;
			}
		}

		m_fpgeometry[idx].name = name;
		m_fpgeometry[idx].position = position;
		m_fpgeometry[idx].angles = DEG2RAD(angles);
		m_fpgeometry[idx].scale = scale;		

		bool doit = false;
		if (((idx<2) && (m_stats.hand == eIH_Right)) || ((idx==2) && (m_stats.hand == eIH_Left)))
			SetGeometry(slot, name, position, DEG2RAD(angles), scale);
	}
	else
		SetGeometry(slot, name, position, DEG2RAD(angles), scale);

	return true;
}

//------------------------------------------------------------------------
int CItem::TargetToSlot(const char *slot)
{
	int islot = eIGS_Last;
	if (slot)
	{
		if (!stricmp(slot, "firstperson"))
			islot = eIGS_FirstPerson;
		else if (!stricmp(slot, "thirdperson"))
			islot = eIGS_ThirdPerson;
		else if (!stricmp(slot, "arms"))
			islot = eIGS_Arms;
		else if (!stricmp(slot, "aux0"))
			islot = eIGS_Aux0;
		else if (!stricmp(slot, "owner"))
			islot = eIGS_Owner;
	}

	return islot;
}