#include "StdAfx.h"
#include "PlayerInput.h"
#include "Player.h"
#include "Game.h"
#include "GameActions.h"
#include "Weapon.h"
#include "WeaponSystem.h"
#include "IVehicleSystem.h"
#include "HUD/HUD.h"


CPlayerInput::CPlayerInput( CPlayer * pPlayer ) : 
m_actions(ACTION_GYROSCOPE), 
m_pPlayer(pPlayer), 
m_pStats(&pPlayer->m_stats),
m_deltaRotation(0,0,0), 
m_deltaMovement(0,0,0), 
m_xi_deltaRotation(0,0,0),
m_filteredDeltaMovement(0,0,0)
{
	m_pPlayer->GetGameObject()->CaptureActions(this);
}

CPlayerInput::~CPlayerInput()
{
	m_pPlayer->GetGameObject()->ReleaseActions(this);
}

void CPlayerInput::Reset()
{
	m_actions = ACTION_GYROSCOPE;//gyroscope is on by default

	m_deltaMovement.zero();
	m_filteredDeltaMovement.zero();
	m_deltaRotation.Set(0,0,0);
	m_xi_deltaRotation.Set(0,0,0);

	m_moveKeysPressed = 0;
}

void CPlayerInput::ApplyMovement(Vec3 delta)
{
	//m_deltaMovement += delta;
	m_deltaMovement.x = clamp_tpl(m_deltaMovement.x+delta.x,-1.0f,1.0f);
	m_deltaMovement.y = clamp_tpl(m_deltaMovement.y+delta.y,-1.0f,1.0f);
	m_deltaMovement.z = 0;
}

void CPlayerInput::OnAction( const ActionId& actionId, int activationMode, float value )
{
	FUNCTION_PROFILER(GetISystem(), PROFILE_GAME);

	m_pPlayer->GetGameObject()->ChangedNetworkState( INPUT_ASPECT );

	//this tell if OnAction have to be forwarded to scripts, now its true by default, only high framerate actions are ignored
	bool filterOut = true;
	bool checkZoom = false;
	const SGameActions& actions = g_pGame->Actions();
	CHUD * pHUD = m_pPlayer->GetHUD();

	IVehicle *pVehicle = m_pPlayer->GetLinkedVehicle();

	//FIXME:on vehicles use cannot be used
	if ((actions.use == actionId) && pVehicle)
	{
		filterOut = false;
	}
	else if (actions.rotateyaw == actionId)
	{
		m_deltaRotation.z -= value;
		filterOut = false;
	}
	else if (actions.rotatepitch == actionId)
	{
		m_deltaRotation.x -= value;
		filterOut = false;
	}
	else if (actions.xi_rotateyaw == actionId)
	{
		// scale rotation speed if we strafe in opposite direction
		float scale = 0.0f;
		/*
		if ((m_input.desiredDeltaMovement.x < -0.6 && value >  0.6) ||
		(m_input.desiredDeltaMovement.x >  0.6 && value < -0.6))
		{
		float movScale = 1.0f - (1.0f - fabs(m_input.desiredDeltaMovement.x))/0.4f;
		float rotScale = 1.0f - (1.0f - fabs(value))/0.4f;
		scale = movScale*rotScale;
		if (value < 0.0f)
		scale = -scale;
		}
		*/
		m_xi_deltaRotation.z = -20.0f*(value + 0.6f*scale);
		filterOut = false;
	}
	else if (actions.xi_rotatepitch == actionId)
	{
		m_xi_deltaRotation.x = 20.0f*value;
		filterOut = false;
	}
	else if (actions.moveright == actionId)
	{
		ApplyMovement(Vec3(value*2.0f - 1.0f,0,0));

		filterOut = false;
		checkZoom = true;
		AdjustMoveKeysPressed(activationMode);
	}
	else if (actions.moveleft == actionId)
	{
		ApplyMovement(Vec3(-(value*2.0f - 1.0f),0,0));

		filterOut = false;
		checkZoom = true;
		AdjustMoveKeysPressed(activationMode);
	}
	else if (actions.moveforward == actionId)
	{
		ApplyMovement(Vec3(0,value*2.0f - 1.0f,0));

		filterOut = false;
		checkZoom = true;
		AdjustMoveKeysPressed(activationMode);
	}
	else if (actions.moveback == actionId)
	{
		ApplyMovement(Vec3(0,-(value*2.0f - 1.0f),0));

		filterOut = false;
		checkZoom = true;
		AdjustMoveKeysPressed(activationMode);
	}
	else if (actions.xi_movex == actionId)
	{
		m_deltaMovement.y = value;
		filterOut = false;
	}
	else if (actions.xi_movey == actionId)
	{
		m_deltaMovement.x = value;
		filterOut = false;
	}
	else if (actions.jump == actionId)
	{
		if (value > 0.0f)
		{
			//if (m_pPlayer->m_params.speedMultiplier > 0.99f)
			m_actions |= ACTION_JUMP;
		}
		else
		{
			m_actions &= ~ACTION_JUMP;
		}
	}
	else if (actions.crouch == actionId)
	{
		if (value > 0.0f)
		{
			//if (m_pPlayer->m_params.speedMultiplier > 0.99f)
			m_actions |= ACTION_CROUCH;
		}
		else
		{
			m_actions &= ~ACTION_CROUCH;
		}
	}
	else if (actions.prone == actionId)
	{
		if (!(m_actions & ACTION_PRONE))
			m_actions |= ACTION_PRONE;
		else
			m_actions &= ~ACTION_PRONE;
	}
	else if (actions.gyroscope == actionId)
	{
		//FIXME:makes more sense a ExosuitActive()
		if (m_pPlayer->IsZeroG())
		{
			if (m_actions & ACTION_GYROSCOPE)
				m_actions &= ~ACTION_GYROSCOPE;
			else
				m_actions |= ACTION_GYROSCOPE;

			m_pPlayer->CreateScriptEvent("gyroscope",(m_actions & ACTION_GYROSCOPE)?1.0f:0.0f);
		}
	}
	else if (actions.gboots == actionId)
	{
		//FIXME:makes more sense a ExosuitActive()
		if (m_pPlayer->IsZeroG())
		{
			if (m_actions & ACTION_GRAVITYBOOTS)
				m_actions &= ~ACTION_GRAVITYBOOTS;
			else
				m_actions |= ACTION_GRAVITYBOOTS;

			m_pPlayer->CreateScriptEvent("gravityboots",(m_actions & ACTION_GRAVITYBOOTS)?1.0f:0.0f);

			if(m_actions & ACTION_GRAVITYBOOTS)
			{
				SGameObjectEvent evt("HUD_TextMessage",eGOEF_ToAll, IGameObjectSystem::InvalidExtensionID, (void*)("gravity_boots_on"));
				GetISystem()->GetIGame()->GetIGameFramework()->GetIGameObjectSystem()->BroadcastEvent(evt);
			}
			else
			{
				SGameObjectEvent evt("HUD_TextMessage",eGOEF_ToAll, IGameObjectSystem::InvalidExtensionID, (void*)("gravity_boots_off"));
				GetISystem()->GetIGame()->GetIGameFramework()->GetIGameObjectSystem()->BroadcastEvent(evt);
			}

		}
	}
	else if (actions.sprint == actionId)
	{
		if (value > 0.0f)
		{
			if (m_pPlayer->m_params.speedMultiplier > 0.99f)
				m_actions |= ACTION_SPRINT;
		}
		else
		{
			m_actions &= ~ACTION_SPRINT;
		}
	}
	else if (actions.leanleft==actionId)
	{
		m_actions |= ACTION_LEANLEFT;
		//not sure about this, its for zeroG
		m_deltaRotation.y -= 30.0f;
	}
	else if (actions.leanright==actionId)
	{
		m_actions |= ACTION_LEANRIGHT;
		//not sure about this, its for zeroG
		m_deltaRotation.y += 30.0f;
	}
	else if (actions.thirdperson==actionId)
	{ 
		if (!pVehicle)
			m_pPlayer->ToggleThirdPerson();
	}
	else if (actions.flymode==actionId)
	{
		++m_pPlayer->m_stats.flyMode;

		if (m_pPlayer->m_stats.flyMode>2)
			m_pPlayer->m_stats.flyMode = 0;

		IPhysicalEntity *pPhys = m_pPlayer->GetEntity()->GetPhysics();
		if (pPhys)
		{
			pe_player_dynamics pD;
			pD.bActive = (m_pPlayer->m_stats.flyMode==2)?false:true;
			pPhys->SetParams(&pD);
		}

		switch(m_pPlayer->m_stats.flyMode)
		{
		case 0:m_pPlayer->CreateScriptEvent("printhud",0,"FlyMode/NoClip OFF");break;
		case 1:m_pPlayer->CreateScriptEvent("printhud",0,"FlyMode ON");break;
		case 2:m_pPlayer->CreateScriptEvent("printhud",0,"NoClip ON");break;
		}
	}
	else if (actions.godmode==actionId)
	{
		++m_pPlayer->m_stats.godMode;

		if (m_pPlayer->m_stats.godMode>2)
			m_pPlayer->m_stats.godMode = 0;

		if (pHUD)
		{
			pHUD->SetGODMode(m_pPlayer->m_stats.godMode);
		}
	}
	else if (actions.debuggun == actionId)
	{
		if (g_pGame)
			g_pGame->GetWeaponSystem()->DebugGun(0);
	}
	else if (actions.refgun == actionId)
	{
		if (g_pGame)
			g_pGame->GetWeaponSystem()->RefGun(0);
	}
	else if (actions.speedmode == actionId)		//the following buttons are quick-keys for the menu functions *****
	{
		m_pPlayer->GetHUD()->HandleFSCommand("QuickMenuSpeedPreset",0);
		return;
	}
	else if (actions.strengthmode == actionId)
	{
		m_pPlayer->GetHUD()->HandleFSCommand("QuickMenuStrengthPreset",0);
		return;
	}
	else if (actions.defensemode == actionId)
	{
		m_pPlayer->GetHUD()->HandleFSCommand("QuickMenuDefensePreset",0);
		return;
	}
	else if (actions.defaultmode == actionId)
	{
		m_pPlayer->GetHUD()->HandleFSCommand("QuickMenuDefault",0);
		return;
	}
	else if (actions.suitcloak == actionId)
	{
		m_pPlayer->GetHUD()->HandleFSCommand("Cloak",0);
		return;
	}					// end of quickkeys *******************************************************************************
	else if (actions.debug_ag_step == actionId)
	{
		GetISystem()->GetIConsole()->ExecuteString("ag_step");
	}
	else if (actions.suitsave == actionId)
	{
		m_pPlayer->GetNanoSuit().SaveConfig();
		SGameObjectEvent evt("HUD_TextMessage",eGOEF_ToAll, IGameObjectSystem::InvalidExtensionID, (void*)("suit_config_saved"));
		GetISystem()->GetIGame()->GetIGameFramework()->GetIGameObjectSystem()->BroadcastEvent(evt);
	}
	else if (actions.suitload == actionId)
	{
		m_pPlayer->GetNanoSuit().LoadConfig();
		SGameObjectEvent evt("HUD_TextMessage",eGOEF_ToAll, IGameObjectSystem::InvalidExtensionID, (void*)("suit_config_loaded"));
		GetISystem()->GetIGame()->GetIGameFramework()->GetIGameObjectSystem()->BroadcastEvent(evt);
	}

	if (pVehicle)
	{
		pVehicle->OnAction(actionId.c_str(),activationMode,value,m_pPlayer->GetEntityId());

		// if thirdperson, check if view has changed
		if (actions.thirdperson==actionId)
		{
			IVehicleSeat* pSeat = pVehicle->GetSeat(m_pPlayer->GetEntity()->GetId());

			if (pSeat && pSeat->GetCurrentView() && pSeat->GetCurrentView()->IsThirdPerson() != m_pPlayer->m_stats.isThirdPerson)      
				m_pPlayer->ToggleThirdPerson();
			else
				filterOut = false;
		}

		//FIXME:not really good
		m_actions = 0;
		m_deltaRotation.Set(0,0,0);
		m_deltaMovement.Set(0,0,0);
	}
	else if (!m_pPlayer->m_stats.isFrozen)
	{
		m_pPlayer->CActor::OnAction(actionId, activationMode, value);

		IInventory *pInventory = static_cast<IInventory *>(m_pPlayer->GetGameObject()->QueryExtension("Inventory"));
		if (!pInventory)
			return;

		bool binoculars = false;
		EntityId itemId = pInventory->GetCurrentItem();
		if (itemId)
		{
			CWeapon *pWeapon = m_pPlayer->GetWeapon(itemId);
			if (pWeapon && !strcmp(pWeapon->GetEntity()->GetClass()->GetName(), "Binoculars"))
				binoculars = true;
		}
		if ((actions.nextitem == actionId) && !binoculars)
			m_pPlayer->SelectNextItem(1, true);
		else if ((actions.previtem == actionId) && !binoculars)
			m_pPlayer->SelectNextItem(-1, true);
		else if ((actions.drop == actionId) && !binoculars && itemId)
			m_pPlayer->DropItem(itemId);
		else if (actions.binoculars == actionId)
		{
			if (binoculars)
				m_pPlayer->SelectLastItem(false);
			else
				m_pPlayer->SelectItemByName("Binoculars", true);
		}
	}

	if (m_moveKeysPressed > 0 && checkZoom)
	{
		IWeapon *wep = NULL;
		if(m_pPlayer->GetCurrentItem())
			wep = m_pPlayer->GetCurrentItem()->GetIWeapon();
		if (wep)
		{
			IZoomMode *zm = wep->GetZoomMode(wep->GetCurrentZoomMode());
			if (zm && !zm->IsZooming() && !zm->IsZoomed())
			{
				m_pPlayer->m_intensityMonitor->Enable(true);
				m_pPlayer->m_intensityMonitor->ChangeFOV(1.0f, m_pPlayer->m_pMoveZoomTime->GetFVal(), true, 1.0f - m_pPlayer->m_pZoomAmount->GetFVal());
				m_pPlayer->m_intensityMonitor->Enable(false);
			}
		}

		checkZoom = false;
	}
	if (checkZoom)
	{
		IWeapon *wep = NULL;
		if(m_pPlayer->GetCurrentItem())
			wep = m_pPlayer->GetCurrentItem()->GetIWeapon();
		if (wep)
		{
			IZoomMode *zm = wep->GetZoomMode(wep->GetCurrentZoomMode());
			if (zm && !zm->IsZooming() && !zm->IsZoomed())
			{
				m_pPlayer->m_intensityMonitor->Enable(true);
			}
		}
		checkZoom = false;
	}

	//send the onAction to scripts, after filter the range of actions. for now just use and hold
	if (filterOut)
	{
		HSCRIPTFUNCTION scriptOnAction(NULL);

		IScriptTable *scriptTbl = m_pPlayer->GetEntity()->GetScriptTable();

		if (scriptTbl)
		{
			scriptTbl->GetValue("OnAction", scriptOnAction);

			if (scriptOnAction)
			{
				char *activation = 0;

				switch(activationMode)
				{
				case eAAM_OnHold:
					activation = "hold";
					break;
				case eAAM_OnPress:
					activation = "press";
					break;
				case eAAM_OnRelease:
					activation = "release";
					break;
				default:
					activation = "";
					break;
				}

				Script::Call(GetISystem()->GetIScriptSystem(),scriptOnAction,scriptTbl,actionId.c_str(),activation, value);
			}
		}
	}

	// FIXME: temporary method to dispatch Actions to HUD (it's not yet possible to register)
	if(pHUD)
	{
		pHUD->OnAction(actionId,activationMode,value);
	}
}

//this function basically returns a smoothed movement vector, for better movement responsivness in small spaces
const Vec3 &CPlayerInput::FilterMovement(const Vec3 &desired)
{
	float frameTimeCap(min(GetISystem()->GetITimer()->GetFrameTime(),0.033f));
	float inputAccel(g_pGame->GetCVars()->pl_inputAccel->GetFVal());

	if (desired.len2()<0.01f)
	{
		m_filteredDeltaMovement.zero();
	}
	else if (inputAccel<=0.0f)
	{
		m_filteredDeltaMovement = desired;
	}
	else
	{
		Vec3 delta(desired - m_filteredDeltaMovement);

		float len(delta.len());
		if (len<=1.0f)
			delta = delta * (1.0f - len*0.55f);

		m_filteredDeltaMovement += delta * min(frameTimeCap * inputAccel,1.0f);
	}

	return m_filteredDeltaMovement;
}

void CPlayerInput::PreUpdate()
{
	CMovementRequest request;

	
	// get rotation into a manageable form
	float mouseSensitivity;
	if (m_pPlayer->IsZeroG())
		mouseSensitivity = 0.01f*MAX(1.0f, g_pGame->GetCVars()->cl_sensitivityZeroG->GetFVal());
	else
		mouseSensitivity = 0.01f*MAX(1.0f, g_pGame->GetCVars()->cl_sensitivity->GetFVal());

	mouseSensitivity *= gf_PI / 180.0f;//doesnt make much sense, but after all helps to keep reasonable values for the sensitivity cvars
	//these 2 could be moved to CPlayerRotation
	mouseSensitivity *= m_pPlayer->m_params.viewSensitivity;
	mouseSensitivity *= m_pPlayer->GetMassFactor();

	// apply rotation from xinput controller
	m_deltaRotation += m_xi_deltaRotation;

	Ang3 deltaRotation(m_deltaRotation * mouseSensitivity);
	if (g_pGame->GetCVars()->cl_invertMouse->GetIVal())
		deltaRotation.x *= -1.0f;

  if (m_pStats->isFrozen)
  { 
    float sMin = GetISystem()->GetIConsole()->GetCVar("cl_frozenSensMin")->GetFVal();
    float sMax = GetISystem()->GetIConsole()->GetCVar("cl_frozenSensMax")->GetFVal();
    float mult = sMin + (sMax-sMin)*(1.f-m_pPlayer->GetFrozenAmount());    
    deltaRotation *= mult;

    static ICVar* pDebugFreezeShake = GetISystem()->GetIConsole()->GetCVar("cl_debugFreezeShake");
    if (pDebugFreezeShake->GetIVal())
    {
      static float color[] = {1,1,1,1};    
      GetISystem()->GetIRenderer()->Draw2dLabel(100,50,1.5,color,false,"frozenAmount: %f (actual: %f)", m_pPlayer->GetFrozenAmount(), m_pPlayer->m_frozenAmount);
      GetISystem()->GetIRenderer()->Draw2dLabel(100,100,1.5,color,false,"deltaRotation: %f (freeze mult: %f)", deltaRotation.z, mult);
    }
  }

	bool animControlled(m_pPlayer->m_stats.animationControlled);

	if (!animControlled)
		request.AddDeltaRotation( deltaRotation );

	// add some movement...
	if (!m_pStats->isFrozen && !animControlled)
		request.AddDeltaMovement( FilterMovement(m_deltaMovement) );


	// handle actions
	if (m_actions & ACTION_JUMP)
		request.SetJump();
	if (m_pPlayer->m_stats.isOnLadder)
	{
		m_actions &= ~ACTION_PRONE;
		m_actions &= ~ACTION_CROUCH;
	}
	else if (m_actions & ACTION_JUMP && m_pPlayer->GetStance() == STANCE_PRONE)
	{
		m_actions &= ~ACTION_PRONE;
//		m_actionsPressed &= ~ACTION_JUMP;
	}

	request.SetStance(FigureOutStance());

	// handle leaning
	if (m_actions & ACTION_LEANLEFT)
		request.SetLean(-1.0f);
	else if (m_actions & ACTION_LEANRIGHT)
		request.SetLean(1.0f);
	else
		request.ClearLean();


	// send the movement request to the appropriate spot!
	m_pPlayer->m_pMovementController->RequestMovement( request );
	m_pPlayer->m_actions = m_actions;

	// reset things for next frame that need to be
	m_deltaRotation = Ang3(0,0,0);

	//static float color[] = {1,1,1,1};    
  //GetISystem()->GetIRenderer()->Draw2dLabel(100,50,1.5,color,false,"deltaMovement:%f,%f", m_deltaMovement.x,m_deltaMovement.y);
}

EStance CPlayerInput::FigureOutStance()
{
	if (m_actions & ACTION_CROUCH)
		return STANCE_CROUCH;
	else if (m_actions & ACTION_PRONE)
		return STANCE_PRONE;
	else if (m_actions & ACTION_RELAXED)
		return STANCE_RELAXED;
	else if (m_actions & ACTION_STEALTH)
		return STANCE_STEALTH;
	else if (m_pPlayer->GetStance() == STANCE_NULL)
		return STANCE_STAND;
	return STANCE_STAND;
}

void CPlayerInput::Update()
{
}

void CPlayerInput::PostUpdate()
{
	m_actions &= ~(ACTION_LEANLEFT | ACTION_LEANRIGHT);
}

void CPlayerInput::GetState( SSerializedPlayerInput& input )
{
	SMovementState movementState;
	m_pPlayer->GetMovementController()->GetMovementState( movementState );

	input.stance = FigureOutStance();
	input.desiredDirection = movementState.movementDirection.GetNormalizedSafe(FORWARD_DIRECTION);
	input.desiredSpeed = movementState.desiredSpeed;
	Vec3 playerPos = m_pPlayer->GetEntity()->GetWorldPos();
	input.lookDirection = (movementState.eyePosition + movementState.eyeDirection - playerPos).GetNormalized();
	input.bodyDirection = movementState.bodyDirection;
}

void CPlayerInput::SetState( const SSerializedPlayerInput& input )
{
	assert(false);
	GameWarning("CPlayerInput::SetState called: should never happen");
}

void CPlayerInput::AdjustMoveKeysPressed( int activationMode )
{
	if (activationMode == eAAM_OnPress)
		m_moveKeysPressed ++;
	else if (activationMode == eAAM_OnRelease)
		m_moveKeysPressed --;
}
