#include "StdAfx.h"
#include "Game.h"
#include "PlayerMovementController.h"
#include "GameUtils.h"
#include "ITimer.h"

#include "IVehicleSystem.h"

// minimum desired speed accepted
static const float MIN_DESIRED_SPEED = 0.2f;
// distance from target that is accepted as "there"
static const float TARGET_REACHED_DISTANCE = 0.4f;
// distance to start slowing down before reaching target
static const float TARGET_SLOWDOWN_DISTANCE = 1.0f;

// maximum head turn angle. This is really low, should be nearer 90, but the helmet intersects with the suit for larger values.
static const float MAX_HEAD_TURN_ANGLE = DEG2RAD(45);

// maximum aim angle
static const float MAX_AIM_TURN_ANGLE = DEG2RAD(45);
// anticipation look ik
static const float ANTICIPATION_COSINE_ANGLE = DEG2RAD(45);
// amount of time to aim for
static const CTimeValue AIM_TIME = 50000000.0f; // VS2: make guys aim for a very long time (look ik/aim ik change over is broken)
// amount of time to look for
static const CTimeValue LOOK_TIME = 2.0f;
// maximum angular velocity for look-at motion (radians/sec)
static const float MAX_LOOK_AT_ANGULAR_VELOCITY = DEG2RAD(75.0f);
// maximum angular velocity for aim-at motion (radians/sec)
static const float MAX_AIM_AT_ANGULAR_VELOCITY = DEG2RAD(75.0f);

static ICVar * pDebugAimLook = 0;

CPlayerMovementController::CPlayerMovementController( CPlayer * pPlayer ) : m_pPlayer(pPlayer)
{
	if (!pDebugAimLook)
		pDebugAimLook = GetISystem()->GetIConsole()->RegisterInt("g_debugaimlook", 0, VF_CHEAT);
	Reset();
}

void CPlayerMovementController::Reset()
{
	m_state = CMovementRequest();
	m_atTarget = false;
	m_desiredSpeed = 0.0f;
	m_usingAimIK = m_usingLookIK = false;
	m_aimInterpolator.Reset();
	m_lookInterpolator.Reset();
	m_updateFunc = &CPlayerMovementController::UpdateNormal;

	UpdateMovementState( m_currentMovementState );
}

bool CPlayerMovementController::RequestMovement( CMovementRequest& request )
{
	if (IVehicle* pVehicle = m_pPlayer->GetLinkedVehicle())
		{
		if (IMovementController* pController = 
		pVehicle->GetPassengerMovementController(m_pPlayer->GetEntityId()))
		{
			pController->RequestMovement(request);
 		}
 	}
	// because we're not allowed to "commit" values here, we make a backup,
	// perform modifications, and then copy the modifications make if everything
	// was successful
	CMovementRequest state = m_state;
	// we have to process right through and not early out, because otherwise we
	// won't modify request to a "close but correct" state
	bool ok = true;

	if (request.HasMoveTarget())
	{
		// TODO: check validity of getting to that target
		state.SetMoveTarget( request.GetMoveTarget() );
		m_atTarget = false;

		float distanceToEnd(request.GetDistanceToPathEnd());
		if (distanceToEnd>0.001f)
			state.SetDistanceToPathEnd(distanceToEnd);
	}
	else if (request.RemoveMoveTarget())
	{
		state.ClearMoveTarget();
		state.ClearDesiredBodyDirectionAtTarget();
		state.ClearDesiredSpeedAtTarget();
		state.ClearDesiredSpeed();
		state.ClearDistanceToPathEnd();
	}

	if (request.HasStance())
	{
		state.SetStance( request.GetStance() );
	}
	else if (request.RemoveStance())
	{
		state.RemoveStance();
	}

	if (request.HasDesiredSpeedAtTarget())
	{
		// if we do not have a place to move to, then we can't accept a target
		// speed when we get there
		if (!state.HasMoveTarget())
		{
			request.SetMoveTarget( m_pPlayer->GetEntity()->GetWorldPos() );
			ok = false;
		}
		else
		{
			state.SetDesiredSpeedAtTarget( request.GetDesiredSpeedAtTarget() );
		}
	}

	if (request.HasDesiredBodyDirectionAtTarget())
	{
		if (!state.HasMoveTarget())
		{
			request.SetMoveTarget( m_pPlayer->GetEntity()->GetWorldPos() );
			ok = false;
		}
		else
		{
			state.SetDesiredBodyDirectionAtTarget( request.GetDesiredBodyDirectionAtTarget() );
		}
	}

	if (request.HasDesiredSpeed())
	{
		if (!state.HasMoveTarget() && request.GetDesiredSpeed() > 0.0f)
		{
			request.SetDesiredSpeed(0.0f);
			ok = false;
		}
		else
		{
			state.SetDesiredSpeed( request.GetDesiredSpeed() );
		}
	}
	else if (request.RemoveDesiredSpeed())
	{
		state.RemoveDesiredSpeed();
	}

	if (request.HasAimTarget())
	{
		state.SetAimTarget( request.GetAimTarget() );
	}
	else if (request.RemoveAimTarget())
	{
		state.ClearAimTarget();
	}

	if (request.HasLookTarget())
	{
		// TODO: check against move direction to validate request
		state.SetLookTarget( request.GetLookTarget(), request.GetLookImportance() );
	}
	else if (request.RemoveLookTarget())
	{
		state.ClearLookTarget();
	}

	if (request.HasStance())
	{
		state.SetStance( request.GetStance() );
	}
	else if (request.RemoveStance())
	{
		state.ClearStance();
	}

	if (request.HasLean())
	{
		state.SetLean( request.GetLean() );
	}
	else if (request.RemoveLean())
	{
		state.ClearLean();
	}

	if (request.ShouldJump())
		state.SetJump();

	if (request.HasNoAiming())
		state.SetNoAiming();

	if (request.HasDeltaMovement())
		state.AddDeltaMovement( request.GetDeltaMovement() );
	if (request.HasDeltaRotation())
		state.AddDeltaRotation( request.GetDeltaRotation() );

	// commit modifications
	if (ok)
	{
		m_state = state;
	}

	return ok;
}

ILINE static f32 GetYaw( const Vec3& v0, const Vec3& v1 )
{
	Vec3 cross  = v0%v1;
	f32 sign = (cross.z<0) ? -1.0f : 1.0f;
	return atan2f( sign*cross.GetLength(), v0|v1 );
}

ILINE static Quat GetTargetRotation( Vec3 oldTarget, Vec3 newTarget, Vec3 origin )
{
	Vec3 oldDir = (oldTarget - origin).GetNormalizedSafe(FORWARD_DIRECTION);
	Vec3 newDir = (newTarget - origin).GetNormalizedSafe(ZERO);
	if (newDir.GetLength() < 0.001f)
		return Quat::CreateIdentity();
	else
		return Quat::CreateRotationV0V1( oldDir, newDir );
}

static void DrawArrow( Vec3 from, Vec3 to, float length, float * clr )
{
	float r = clr[0];
	float g = clr[1];
	float b = clr[2];
	float color[] = {r,g,b,1};
	GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine( from, ColorF(r,g,b,1), to, ColorF(r,g,b,1) );
	GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawSphere( to, 0.2f, ColorF(r,g,b,1) );
}

static float blue[4] = {0,0,1,1};
static float red[4] = {1,0,0,1};
static float yellow[4] = {1,1,0,1};
static float green[4] = {0,1,0,1};
static CTimeValue lastTime;
static int y = 100;

bool CPlayerMovementController::Update( float frameTime, SActorFrameMovementParams& params )
{
	FUNCTION_PROFILER(GetISystem(), PROFILE_GAME);

	bool ok = false;
	if (m_updateFunc)
		ok = (this->*m_updateFunc)(frameTime, params);

	// GetMovementState is called 4-5 times per entity per frame, and calculating this is expensive
	// => it's better to cache it and just copy when we need it... it's constant after the update
	UpdateMovementState( m_currentMovementState );

//	m_state.RemoveDeltaMovement();
//	m_state.RemoveDeltaRotation();

	return ok;
}

bool CPlayerMovementController::UpdateNormal( float frameTime, SActorFrameMovementParams& params )
{
	ITimer * pTimer = GetISystem()->GetITimer();

	IRenderer * pRend = GetISystem()->GetIRenderer();
	if (pDebugAimLook->GetIVal())
	{
		if (pTimer->GetFrameStartTime()!=lastTime)
		{
			y = 100;
			lastTime = pTimer->GetFrameStartTime();
		}
	}

	IEntity * pEntity = m_pPlayer->GetEntity();
	Vec3 playerPos = pEntity->GetWorldPos();

	params.desiredVelocity = Vec3(0,0,0);
	params.lookTarget = m_currentMovementState.eyePosition + m_currentMovementState.eyeDirection;
	params.aimTarget = m_currentMovementState.handPosition + m_currentMovementState.aimDirection;
	params.lookIK = false;
	params.aimIK = false;
	params.deltaAngles = Ang3(0,0,0);


	ICharacterInstance * pCharacter = pEntity->GetCharacter(0);
	if (!pCharacter)
		return true;

	ISkeleton * pSkeleton = pCharacter->GetISkeleton();

	CTimeValue now = pTimer->GetFrameStartTime();

	Vec3 moveDirection = pEntity->GetRotation() * pSkeleton->GetCurrentBodyDirection();
	if (m_pPlayer->GetLastRequestedVelocity().len2() > 0.01f)
		moveDirection = m_pPlayer->GetLastRequestedVelocity().GetNormalized();

	Vec3 viewFollowMovement(ZERO);
 	// speed control
	if (m_state.HasMoveTarget())
	{
		Vec3 desiredMovement = m_state.GetMoveTarget() - playerPos;
		float distance = desiredMovement.len();
		if (m_state.HasDesiredSpeedAtTarget() && m_state.GetDesiredSpeedAtTarget() < MIN_DESIRED_SPEED)
		{
			m_atTarget = distance < TARGET_REACHED_DISTANCE;
		}
		else
		{
			m_atTarget = false;
		}
		if (distance > 0.01f) // need to have somewhere to move, to actually move :)
		{
			// speed at which things are "guaranteed" to stop... 0.03f is a fudge factor to ensure this
			static const float STOP_SPEED = MIN_DESIRED_SPEED-0.03f;

			desiredMovement /= distance;
			float desiredSpeed = 0.0f;
			if (m_state.HasDesiredSpeed())
			{
				desiredSpeed = m_state.GetDesiredSpeed();
			}
			if (m_state.HasDesiredSpeedAtTarget() && distance < TARGET_SLOWDOWN_DISTANCE)
			{
				// we're close to the target, speed becomes a matter of interpolation...
				float targetSpeed = m_state.GetDesiredSpeedAtTarget();
				if (targetSpeed < STOP_SPEED)
				{
					targetSpeed = STOP_SPEED;
				}
				float blendFactor = distance / TARGET_SLOWDOWN_DISTANCE;
				desiredSpeed = targetSpeed * (1.0f - blendFactor) + desiredSpeed * blendFactor;
			}
			if (!m_state.HasDesiredSpeed() && m_state.HasDesiredSpeedAtTarget())
			{
				// if we don't have a desired speed, but we do have one at the target, just use that
				desiredSpeed = m_state.GetDesiredSpeedAtTarget();
			}
			if (desiredSpeed > MIN_DESIRED_SPEED)
			{
				//pRend->GetIRenderAuxGeom()->DrawLine( playerPos, ColorF(1,1,1,1), playerPos + desiredMovement, ColorF(1,1,1,1) );

				//calculate the desired speed amount (0-1 length) in world space
				params.desiredVelocity = desiredMovement * desiredSpeed / m_pPlayer->GetStanceMaxSpeed(m_pPlayer->GetStance());
				//
				viewFollowMovement = params.desiredVelocity * 5.0f;
				//and now, after used it for the viewFollowMovement convert it to the Actor local space
				params.desiredVelocity = m_pPlayer->GetBaseMatrix().GetInverted() * params.desiredVelocity;
			}
		}
	}

	Vec3 bodyTarget = playerPos + (m_pPlayer->GetBaseMatrix()*params.desiredVelocity);

	// look and aim direction
	bool upd;
	Vec3 tgt;

	upd = false;
	tgt = playerPos + moveDirection;
	const char * lookType = "previous";
	if (m_state.HasAimTarget())
	{
		upd = true;
		tgt = m_state.GetAimTarget();
		lookType = "aim";
	}
	else if (m_state.HasLookTarget())
	{
		upd = true;
		tgt = m_state.GetLookTarget();
		lookType = "look";
	}
	else if (m_state.HasMoveTarget())
	{
		Vec3 desiredMoveDir = (m_state.GetMoveTarget() - playerPos).GetNormalizedSafe(ZERO);
		if (moveDirection.Dot(desiredMoveDir) < ANTICIPATION_COSINE_ANGLE)
		{
			upd = true;
			tgt = m_state.GetMoveTarget();
			lookType = "anticipate";
		}
	}
	else if (m_lookInterpolator.HasTarget(now, LOOK_TIME))
	{
		tgt = m_lookInterpolator.GetTarget();
		lookType = "lastLook";
	}
	m_lookInterpolator.TargetValue( tgt, now, LOOK_TIME, frameTime, upd, playerPos, MAX_LOOK_AT_ANGULAR_VELOCITY );

	upd = false;
	const char * aimType = lookType;
	if (m_state.HasAimTarget())
	{
		upd = true;
		tgt = m_state.GetAimTarget();
		aimType = "aim";
	}
	m_aimInterpolator.TargetValue( tgt, now, AIM_TIME, frameTime, upd, playerPos, MAX_AIM_AT_ANGULAR_VELOCITY );

	const char * ikType = "none";
	ColorB dbgClr(0,255,0,255);
	ColorB * pDbgClr = pDebugAimLook->GetIVal()? &dbgClr : NULL;
	if (!m_state.HasNoAiming() && m_aimInterpolator.HasTarget( now, AIM_TIME ))	// AIM IK
	{
		//m_aimInterpolator.GetTarget( params.aimTarget, bodyTarget, m_currentMovementState.handPosition, moveDirection, MAX_AIM_TURN_ANGLE, pDbgClr );
		params.aimIK = true;
		ikType = "aim";
	}
	else if (m_lookInterpolator.HasTarget( now, LOOK_TIME ))	// Look IK
	{
		int idJoint=pSkeleton->GetIDByName("Bip01 Neck");
		Vec3 vNeckDir1=pSkeleton->GetAbsJMatrixByID(idJoint).GetColumn(1);
		vNeckDir1 = pEntity->GetRotation() * vNeckDir1;

		//m_lookInterpolator.GetTarget( params.lookTarget, bodyTarget, m_currentMovementState.eyePosition, moveDirection, MAX_HEAD_TURN_ANGLE, pDbgClr );
		//m_lookInterpolator.GetTarget( params.lookTarget, bodyTarget, m_currentMovementState.eyePosition, vNeckDir1, MAX_HEAD_TURN_ANGLE, pDbgClr );
		params.lookIK = true;
		ikType = "look";
	}

	Vec3 viewDir(ZERO);
	if (m_state.HasAimTarget())
	{
		//here, the close the AI is to the target the more it will look at the firing direction while moving (move backwards/strafe)
		Vec3 deltaAim(m_state.GetAimTarget() - m_currentMovementState.handPosition);
		viewDir += deltaAim.GetNormalizedSafe(ZERO);

		float deltaLen(deltaAim.len2());
		float aimLimit(9.0f*9.0f);
		if (deltaLen<aimLimit)
		{
			float mult(deltaLen/aimLimit);
			viewFollowMovement *= mult * mult;
		}
	}
	if (m_state.HasLookTarget())
	{
		viewDir += (m_state.GetLookTarget() - m_currentMovementState.eyePosition).GetNormalizedSafe(ZERO);
	}
	float distanceToEnd(m_state.GetDistanceToPathEnd());
	if (distanceToEnd>0.001f)
	{
		float lookOnPathLimit(7.0f);
		if (distanceToEnd<lookOnPathLimit)
		{
			float mult(distanceToEnd/lookOnPathLimit);
			viewFollowMovement *= mult * mult;
			//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawSphere( playerPos, 0.5f, ColorF(1,1,1,1) );
		}
		//pRend->Draw2dLabel( 100, 100, 1.0f, yellow, false, "distance:%.1f", distanceToEnd);
	}

	//if the AI is running, the view will
	viewDir += viewFollowMovement;
	//pRend->GetIRenderAuxGeom()->DrawLine( m_currentMovementState.eyePosition, ColorF(0.5f,0.5f,1,1), m_currentMovementState.eyePosition + viewDir, ColorF(0.5f,0.5f,1,1) );
	viewDir.NormalizeSafe(ZERO);

	//viewDir = ((bodyTarget - playerPos).GetNormalizedSafe(ZERO));
	if (viewDir.len2() > 0.01f)
	{		
		Vec3 localVDir(m_pPlayer->GetViewMatrixFinal().GetInverted() * viewDir);

		params.deltaAngles.x += asin(localVDir.z);
		params.deltaAngles.z += cry_atan2f(-localVDir.x,localVDir.y);

		//smooth out the view direction change
		for (int i=0; i<3; i++)
			params.deltaAngles[i] *= min(1.0f,frameTime*6.6f);
	}

	if (m_state.HasDeltaRotation())
	{
		params.deltaAngles += m_state.GetDeltaRotation();
		ikType = "mouse";
	}


	if (pDebugAimLook->GetIVal())
	{
		pRend->Draw2dLabel( 10, y, 1.0f, green, false, "%s: look:%s aim:%s ik:%s (%f,%f,%f)", pEntity->GetName(), aimType, lookType, ikType, params.deltaAngles.x, params.deltaAngles.y, params.deltaAngles.z );
		y += 10;
	}

	// process incremental movement
	if (m_state.HasDeltaMovement())
	{
		params.desiredVelocity += m_state.GetDeltaMovement();
	}

	// stance control
	if (m_state.HasStance())
	{
		params.stance = m_state.GetStance();
	}

	// leaning
	if (m_state.HasLean())
		params.desiredLean = m_state.GetLean();
	else
		params.desiredLean = 0.0f;

	params.jump = m_state.ShouldJump();
	m_state.ClearJump();

	m_usingAimIK = params.aimIK;
	m_usingLookIK = params.lookIK;

	m_desiredSpeed = params.desiredVelocity.GetLength() * m_pPlayer->GetStanceMaxSpeed( m_pPlayer->GetStance() );

	m_state.RemoveDeltaRotation();
	m_state.RemoveDeltaMovement();

	return true;
}

// G04 Find camera target, so that player 3rd person model can aim at it
inline Vec3 CalcCameraTarget()
{
	float range=2000.0f;
	Vec3 tgt;
	CCamera &cam = GetISystem()->GetViewCamera();
	Vec3 aimDir = cam.GetViewdir();
	Vec3 aimPos = cam.GetPosition();
	ray_hit hit;

	tgt=aimPos+aimDir*range;

	IPhysicalEntity *pPhys=0;
	IEntity *pEnt=0;
	IActor *pAct=GetISystem()->GetIGame()->GetIGameFramework()->GetClientActor();

	if(pAct)
	{
		pEnt=pAct->GetEntity();
		if(pEnt)
			pPhys=pEnt->GetPhysics();
	}
	int result = GetISystem()->GetIPhysicalWorld()->RayWorldIntersection(aimPos, aimDir * range, 
							(ent_all ), rwi_stop_at_pierceable|rwi_colltype_any|rwi_any_hit, &hit, 1, pPhys);
	if (result)
	{
		tgt=hit.pt;
	}

	if(pEnt)
		DrawArrow(pEnt->GetWorldPos(),tgt,8,red);
	DrawArrow(tgt-Vec3(0,0,8),tgt,8,yellow);
	return tgt;
}


void CPlayerMovementController::UpdateMovementState( SMovementState& state )
{
	IEntity * pEntity = m_pPlayer->GetEntity();
	ICharacterInstance * pCharacter = pEntity->GetCharacter(0);
	if (!pCharacter)
		return;
	ISkeleton * pSkeleton = pCharacter->GetISkeleton();

	int boneEyeL = m_pPlayer->GetBoneID(BONE_EYE_L);
	int boneEyeR = m_pPlayer->GetBoneID(BONE_EYE_R);
	int boneHead = m_pPlayer->GetBoneID(BONE_HEAD);
	int boneWeapon = m_pPlayer->GetBoneID(BONE_WEAPON);

	if (m_pPlayer->IsPlayer())
	{
		Vec3 viewOfs(m_pPlayer->GetStanceViewOffset(m_pPlayer->GetStance()));
		state.eyePosition = pEntity->GetWorldPos() + m_pPlayer->GetBaseMatrix() * viewOfs;
		state.eyeDirection = m_pPlayer->GetViewMatrixFinal().GetColumn(1);
		state.handPosition = state.eyePosition;

		// G04 if third person then aim at point camera is looking at.
		if(m_pPlayer->IsThirdPerson())
		{
			//--- Dbg
			//Vec3 posEyeL = pSkeleton->GetAbsJPositionByID(boneEyeL);
			//Vec3 posEyeR = pSkeleton->GetAbsJPositionByID(boneEyeR);
			//Vec3 posHead = pSkeleton->GetAbsJPositionByID(boneHead);
			//Vec3 lEyePos = (0.5f * (posEyeL + posEyeR));
			//state.eyePosition=pEntity->GetSlotWorldTM(0) * lEyePos;
			//state.eyeHeight = lEyePos.z;
			//
			Vec3 posWeapon = pSkeleton->GetAbsJPositionByID(boneWeapon);
			state.handPosition = state.handPosition = pEntity->GetSlotWorldTM(0) * posWeapon; // state.eyePosition;

			Vec3 camTgt=CalcCameraTarget();
			state.aimDirection = (camTgt-state.handPosition);//.GetNormalizedSafe(Vec3(0,1,0));
			//DrawArrow(state.handPosition,camTgt,8,blue);

			state.eyeDirection=state.aimDirection;
		}
		else
		{
			state.aimDirection = state.eyeDirection;
		}
	}
	else
	{
		Vec3 posEyeL = pSkeleton->GetAbsJPositionByID(boneEyeL);
		Vec3 posEyeR = pSkeleton->GetAbsJPositionByID(boneEyeR);
		Vec3 posHead = pSkeleton->GetAbsJPositionByID(boneHead);
		Vec3 posWeapon = pSkeleton->GetAbsJPositionByID(boneWeapon);
		Vec3 lEyePos = (0.5f * (posEyeL + posEyeR));

		Vec3 dirWeapon = pSkeleton->GetAbsJMatrixByID(boneWeapon).GetColumn(0);

		state.eyePosition = pEntity->GetSlotWorldTM(0) * lEyePos;
		state.handPosition = /*state.eyePosition;//*/pEntity->GetSlotWorldTM(0) * posWeapon;

		/*Vec3 dirEye;
		const char * eyeDirType = "none";
		if (pCharacter->GetISkeleton()->GetAimIKStatus())
		{
			Vec3 posWeaponWorld = pEntity->GetSlotWorldTM(0) * posWeapon;
			Vec3 aimTgt, temp;
			m_aimInterpolator.GetTarget(aimTgt, temp, state.handPosition, pSkeleton->GetCurrentBodyDirection(), MAX_AIM_TURN_ANGLE);
			float distanceToTarget = (aimTgt - posWeaponWorld).len();
			Vec3 aimTarget = pEntity->GetSlotWorldTM(0) * (posWeapon + dirWeapon.GetNormalizedSafe() * distanceToTarget);

			ICVar *pg_aimDebug = GetISystem()->GetIConsole()->GetCVar("g_aimdebug");
			if (pg_aimDebug && pg_aimDebug->GetIVal()!=0)
			{
				GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawSphere(aimTarget, 0.5f, ColorB(255,255,0,255));
				GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawSphere(m_aimInterpolator.GetTarget(), 0.5f, ColorB(255,255,0,255));
			}

			dirEye = (aimTarget - state.eyePosition).GetNormalized();
			eyeDirType = "aim";
		}
		else
		{
			Vec3 dirEyeL = pSkeleton->GetAbsJMatrixByID(boneEyeL).GetColumn(1);
			Vec3 dirEyeR = pSkeleton->GetAbsJMatrixByID(boneEyeR).GetColumn(1);
			dirEye = (dirEyeL + dirEyeR)*0.5f;
			dirEye = dirEye.GetNormalized();
			dirEye = pEntity->GetRotation() * dirEye;
			if (!m_usingLookIK)
				dirEye.z = 0;
			dirEye = dirEye.GetNormalized();
			eyeDirType = "mid";
		}
		if (m_pPlayer->GetLinkedVehicle()&&m_state.HasLookTarget())
			dirEye = (m_state.GetLookTarget() - state.eyePosition).GetNormalizedSafe(ZERO);

		float dp = 0.0f;
		if (m_state.HasLookTarget())
			dp = (m_state.GetLookTarget() - state.eyePosition).GetNormalizedSafe(ZERO).Dot(dirEye);
		float dpa = 0.0f;
		if (m_state.HasAimTarget())
			dpa = (m_state.GetAimTarget() - state.eyePosition).GetNormalizedSafe(ZERO).Dot(dirEye);*/

		//state.eyeDirection = dirEye; //(lEyePos - posHead).GetNormalizedSafe(FORWARD_DIRECTION);
		//state.aimDirection = pEntity->GetRotation() * dirWeapon;
		
		//filippo: relying on assets for this sort of thing is evil/unreliable, so I'm using game internal directions instead.
		state.eyeDirection = m_pPlayer->GetViewMatrixFinal().GetColumn(1);
		state.aimDirection = state.eyeDirection;

		state.eyeHeight = lEyePos.z;

		/*
		GetISystem()->GetIRenderer()->Draw2dLabel( 10, y, 1.0f, green, false, "%s: %s %f %f", m_pPlayer->GetEntity()->GetName(), eyeDirType, dp, dpa );
		y += 10;
		if (m_state.HasAimTarget())
		{
			float dist = (m_state.GetAimTarget() - state.eyePosition).len();
			GetISystem()->GetIRenderer()->Draw2dLabel( 10, y, 1.0f, green, false, "%s: %.3f %.3f %.3f", pEntity->GetName(), m_state.GetAimTarget().x, m_state.GetAimTarget().y, m_state.GetAimTarget().z );
			y += 10;
			Vec3 tgt = state.eyePosition + dirEye * dist;
			GetISystem()->GetIRenderer()->Draw2dLabel( 10, y, 1.0f, green, false, "%s: %.3f %.3f %.3f", pEntity->GetName(), tgt.x, tgt.y, tgt.z );
			y += 10;
		}
		else if (m_state.HasLookTarget())
		{
			float dist = (m_state.GetLookTarget() - state.eyePosition).len();
			GetISystem()->GetIRenderer()->Draw2dLabel( 10, y, 1.0f, yellow, false, "%s: %.3f %.3f %.3f", pEntity->GetName(), m_state.GetLookTarget().x, m_state.GetLookTarget().y, m_state.GetLookTarget().z );
			y += 10;
			Vec3 tgt = state.eyePosition + dirEye * dist;
			GetISystem()->GetIRenderer()->Draw2dLabel( 10, y, 1.0f, yellow, false, "%s: %.3f %.3f %.3f", pEntity->GetName(), tgt.x, tgt.y, tgt.z );
			y += 10;
		}
		*/
	}

  state.movementDirection = pEntity->GetRotation() * pSkeleton->GetCurrentBodyDirection();
	if (m_pPlayer->GetLastRequestedVelocity().len2() > 0.01f)
		state.movementDirection = m_pPlayer->GetLastRequestedVelocity().GetNormalized();
	state.bodyDirection = pEntity->GetRotation() * pSkeleton->GetCurrentBodyDirection();
	state.atMoveTarget = m_atTarget;
	state.desiredSpeed = m_desiredSpeed;
	state.stance = m_pPlayer->GetStance();
	state.upDirection = m_pPlayer->GetBaseMatrix().GetColumn(2);
	state.minSpeed = MIN_DESIRED_SPEED;
	state.maxSpeed = m_pPlayer->GetStanceMaxSpeed(state.stance);
	if (state.maxSpeed < state.minSpeed)
	{
//		assert(state.stance == STANCE_NULL);
		state.maxSpeed = state.minSpeed;
		/*if (!g_pGame->GetIGameFramework()->IsEditing())
			GameWarning("%s In STANCE_NULL - movement speed is clamped", pEntity->GetName());*/
	}
	state.normalSpeed = 0.5f * (state.minSpeed + state.maxSpeed);
	state.stanceSize = m_pPlayer->GetStanceInfo(state.stance)->size;
	state.pos = pEntity->GetWorldPos();

	// TODO: remove this
	//if (m_state.HasAimTarget())
	//	state.aimDirection = (m_state.GetAimTarget() - state.handPosition).GetNormalizedSafe();


	state.isAlive = (m_pPlayer->GetHealth()>0);
}

void CPlayerMovementController::Release()
{
	delete this;
}
