/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id$
$DateTime$

-------------------------------------------------------------------------
History:
- 17:1:2006   11:18 : Created by Mrcio Martins

*************************************************************************/
#include "StdAfx.h"
#include "TracerManager.h"
#include "Game.h"
#include "Actor.h"
#include <I3DEngine.h>



//------------------------------------------------------------------------
CTracerPath::CTracerPath()
: m_current(0)
{
	m_points.reserve(3);
}

//------------------------------------------------------------------------
CTracerPath::~CTracerPath()
{
}

//------------------------------------------------------------------------
void CTracerPath::Reset()
{
	m_points.resize(0);
}

//------------------------------------------------------------------------
void CTracerPath::AddPoint(const Vec3 &pt)
{
	m_points.push_back(pt);
}

//------------------------------------------------------------------------
Vec3 CTracerPath::GetPoint() const
{
	if (m_current<m_points.size())
		return m_points[m_current];

	return Vec3(0,0,0);
}

//------------------------------------------------------------------------
void CTracerPath::First()
{
	m_current=0;
}

//------------------------------------------------------------------------
void CTracerPath::Next()
{
	if (!IsEnd())
		++m_current;
}

//------------------------------------------------------------------------
bool CTracerPath::IsEnd() const
{
	return m_current>=m_points.size();
}

//------------------------------------------------------------------------
CTracer::CTracer(const Vec3 &pos)
: m_pStatObj(0),
	m_pRenderNode(0),
	m_pEmitter(0),
	m_pos(0,0,0),
	m_startingpos(pos),
	m_length(2.0f),
	m_speed(90.0f),
	m_scale(1.0f),
	m_age(0.0f),
	m_lifeTime(1.5f)
{
}

//------------------------------------------------------------------------
CTracer::~CTracer()
{
	if (m_pRenderNode)
		m_pRenderNode->Release();

	if (m_pEmitter)
		GetISystem()->GetI3DEngine()->DeleteParticleEmitter(m_pEmitter);
}

//------------------------------------------------------------------------
void CTracer::SetGeometry(const char *name, float scale)
{
	m_pRenderNode = GetISystem()->GetI3DEngine()->CreateRenderNode(eERType_Brush);
	m_pRenderNode->SetViewDistRatio(255);
	m_pRenderNode->SetLodRatio(255);
	m_pStatObj = GetISystem()->GetI3DEngine()->LoadStatObj(name);
}

//------------------------------------------------------------------------
void CTracer::SetEffect(const char *name, float scale)
{
	IParticleEffect *pEffect = GetISystem()->GetI3DEngine()->FindParticleEffect(name);
	if (!pEffect)
		return;

	m_pEmitter = pEffect->Spawn( false, Matrix34::CreateTranslationMat(m_startingpos));
}

//------------------------------------------------------------------------
bool CTracer::Update(float frameTime, const Vec3 &campos)
{
	m_age += frameTime;

	if (m_age >= m_lifeTime)
		return false;

	Vec3 end(m_path.GetPoint());

	if ((m_pos-end).len2() <= 0.5f*0.5f)
	{
		m_path.Next();
		if (m_path.IsEnd())
			return false;
		else
		{
			end=m_path.GetPoint();
		}
	}

	Vec3 dp = end-m_pos;
	float dist = dp.len();
	Vec3 dir = dp/dist;

	m_pos = m_pos+dir*MIN(m_speed*frameTime, dist);

	float scaleMult=1.0f;

	if (m_pRenderNode)
	{
		float minDistance = 2.0f;
		float maxDistance = 200.0f;
		float minLength = 1.0f;
		float maxLength = 100.0f;
		float minScale = 0.5f;
		float maxScale = 30.0f;

		float cameraDistance = (m_pos-campos).len2();
		if (cameraDistance<=minDistance*minDistance)
		{
			m_length=minLength;
			scaleMult=minScale;
		}
		else if (cameraDistance>=maxDistance*maxDistance)
		{
			m_length=maxLength;
			scaleMult=maxScale;
		}
		else
		{
			float t=(sqrtf(cameraDistance)-minDistance)/(maxDistance-minDistance);
			m_length=minLength+t*(maxLength-minLength);
			scaleMult=minScale+t*(maxScale-minScale);
		}

		float cosine = dir.Dot((m_pos-campos).normalized());
		m_length = minLength+(m_length-minLength)*fabsf(cosine);

		if ((m_pos-end).len2() > 0.5f*0.5f)
		{
			if ((m_pos-m_startingpos).len2()<m_length*m_length)
				m_pos = m_startingpos+dir*m_length;
		}

	}

	UpdateVisual(m_pos, dir, m_scale*scaleMult, m_length);

	return true;
}

//------------------------------------------------------------------------
void CTracer::UpdateVisual(const Vec3 &pos, const Vec3 &dir, float scale, float length)
{
	Matrix34 tm(Matrix33::CreateRotationVDir(dir));
	tm.AddTranslation(pos);

	if (m_pRenderNode)
	{
		// marcok: made this uniform again ... it was spamming the console
		tm.Scale(Vec3(scale, length, scale));
		m_pRenderNode->SetEntityStatObj(0, m_pStatObj, &tm);
	}

	if (m_pEmitter)
	{
		tm.Scale(Vec3(m_scale, m_scale, m_scale));
		m_pEmitter->SetMatrix(tm);
	}
}

//------------------------------------------------------------------------
void CTracer::SetLifeTime(float lifeTime)
{
	m_lifeTime = lifeTime;
}

//------------------------------------------------------------------------
CTracerManager::CTracerManager()
{
}

//------------------------------------------------------------------------
CTracerManager::~CTracerManager()
{
}

//------------------------------------------------------------------------
void CTracerManager::EmitTracer(const STracerParams &params)
{
	m_actives.push_back(new CTracer(params.position));

	CTracer *tracer = *m_actives.rbegin();

	if (params.geometry && params.geometry[0])
		tracer->SetGeometry(params.geometry, params.scale);
	if (params.effect && params.effect[0])
		tracer->SetEffect(params.effect, params.scale);
	tracer->SetLifeTime(params.lifetime);

	tracer->m_speed = params.speed;
	tracer->m_pos = params.position;
	tracer->m_path = *params.path;
}

//------------------------------------------------------------------------
void CTracerManager::Update(float frameTime)
{
	int c=0;
	int r=0;

	IActor *pActor=g_pGame->GetIGameFramework()->GetClientActor();
	if (!pActor)
		return;

	SMovementState state;
	if (!pActor->GetMovementController())
		return;
	
	pActor->GetMovementController()->GetMovementState(state);

	m_actives.swap(m_updating);
	for (TTracerVector::iterator it = m_updating.begin(); it!=m_updating.end(); ++it)
	{
		++c;
		CTracer *tracer = *it;

		if (tracer->Update(frameTime, state.eyePosition))
			m_actives.push_back(tracer);
		else
		{
			delete tracer;
			++r;
		}
	}

	m_updating.resize(0);

	float color[4] = {1, 1, 1, 1};

	//GetISystem()->GetIRenderer()->Draw2dLabel(100, 400, 1, color, false, "tracers: %d // removed: %d", c, r);
}