/*************************************************************************
  Crytek Source File.
  Copyright (C), Crytek Studios, 2001-2004.
 -------------------------------------------------------------------------
  $Id$
  $DateTime$
  
 -------------------------------------------------------------------------
  History:
  - 21:7:2005: Created by Mikko Mononen

*************************************************************************/
#include "StdAfx.h"
#include "Game.h"
#include "Trooper.h"
#include "GameUtils.h"

#include <IViewSystem.h>
#include <IItemSystem.h>
#include <IPhysics.h>
#include <ICryAnimation.h>
#include <ISerialize.h>
#include <IRenderAuxGeom.h>

void STrooperBeam::CreateBeam(IEntity *owner,const char *effect,EntityId targetId)
{
	if (!owner)
		return;

	IParticleEffect *pEffect = GetISystem()->GetI3DEngine()->FindParticleEffect(effect);
	if (pEffect)
	{
		active = true;
		beamTargetId = targetId;
		
		SpawnParams params;
		params.fPulsePeriod = 0.0f;
		params.fSizeScale = 0.5f;
		params.fCountScale = 1.0f;
		params.bCountPerUnit = 0;

		effectSlot = owner->LoadParticleEmitter( 3, pEffect, &params, -1, false );
		owner->SetSlotLocalTM(effectSlot,Matrix33::CreateRotationZ(gf_PI));
	}	
}

void STrooperBeam::RemoveBeam(IEntity *owner)
{
	if (!owner)
		return;

	IParticleEmitter *pEmitter = owner->GetParticleEmitter(effectSlot);
	if (pEmitter)
		GetISystem()->GetI3DEngine()->DeleteParticleEmitter(pEmitter);

	effectSlot = 0;
	beamTargetId = 0;
	active = false;
}

void STrooperBeam::Update(IEntity *owner,float frameTime)
{
	if (!active || !owner)
		return;

	IParticleEmitter *pEmitter = owner->GetParticleEmitter(effectSlot);
	if (pEmitter)
	{
		IEntity *pTarget = GetISystem()->GetIEntitySystem()->GetEntity(beamTargetId);
		if (pTarget)
		{
			ParticleTarget targetOptions;	
			targetOptions.bTarget = true;
			targetOptions.bExtendCount = true;
			targetOptions.bExtendLife = false;
			targetOptions.bExtendSpeed = true;
			targetOptions.bPriority = true;
			targetOptions.vTarget = pTarget->GetWorldPos();

			pEmitter->SetTarget(targetOptions);
		}
	}
}

void CTrooper::Kill()
{
	CAlien::Kill();
	
	m_beamEffect.RemoveBeam(GetEntity());
}

void CTrooper::Revive(bool fromInit)
{
	CAlien::Revive(fromInit);

	m_beamEffect.RemoveBeam(GetEntity());
}

void CTrooper::UpdateStats(float frameTime)
{
	m_beamEffect.Update(GetEntity(),frameTime);
	//if (m_beamEffect.active)
	//	m_stats.inFiring = 10.0f;

	CAlien::UpdateStats(frameTime);

	IPhysicalEntity *pPhysEnt = GetEntity()->GetPhysics();
	if (!pPhysEnt)
		return;

	if (IsZeroG())
	{
		ray_hit hit;
		int rayFlags = (COLLISION_RAY_PIERCABILITY & rwi_pierceability_mask);

		if (GetISystem()->GetIPhysicalWorld()->RayWorldIntersection(GetEntity()->GetWorldPos(), m_baseMtx.GetColumn(2)*-5.0f, ent_terrain|ent_static|ent_rigid, rayFlags, &hit, 1, &pPhysEnt, 1))
		{
			pe_player_dynamics newGravity;
			newGravity.gravity = m_baseMtx.GetColumn(2) * -9.81f;
			pPhysEnt->SetParams(&newGravity);
		}
		else
			m_stats.isFloating = true;
	}
	else
	{
		pe_player_dynamics newGravity;
		m_stats.gravity *= 3.0f;
		newGravity.gravity = m_stats.gravity;
		pPhysEnt->SetParams(&newGravity);
	}
}

void CTrooper::ProcessRotation(float frameTime)
{
	IPhysicalEntity *pPhysEnt = GetEntity()->GetPhysics();

	if (!pPhysEnt)
		return;

	if (frameTime > 0.1f)
			frameTime = 0.1f;

	float rotSpeed(0.3f);

	if (m_input.viewVector.len2()>0.0f)
	{
		m_eyeMtx.SetRotationVDir(m_input.viewVector.GetNormalizedSafe());
	}

	if (m_input.viewVector.len2()>0.0f)
	{
		m_viewMtx.SetRotationVDir(m_input.viewVector.GetNormalizedSafe());
	}
	else
	{
		Ang3 desiredAngVel(m_input.deltaRotation.x * rotSpeed,0,m_input.deltaRotation.z * rotSpeed);
				
		//rollage
		if (m_input.actions & ACTION_LEANLEFT)
			desiredAngVel.y -= 10.0f * rotSpeed;
		if (m_input.actions & ACTION_LEANRIGHT)
			desiredAngVel.y += 10.0f * rotSpeed;

		Interpolate(m_angularVel,desiredAngVel,3.5f,frameTime);

		Matrix33 yawMtx;
		Matrix33 pitchMtx;
		Matrix33 rollMtx;

		//yaw
		yawMtx.SetRotationZ(m_angularVel.z * gf_PI/180.0f);
		//pitch
		pitchMtx.SetRotationX(m_angularVel.x * gf_PI/180.0f);
		//roll
		if (fabs(m_angularVel.y) > 0.001f)
			rollMtx.SetRotationY(m_angularVel.y * gf_PI/180.0f);
		else
			rollMtx.SetIdentity();
		//
		
		m_viewMtx = m_viewMtx * yawMtx * pitchMtx * rollMtx;
		m_viewMtx.NoScale();
	}

	//now build the base matrix
	Vec3 forward(m_viewMtx.GetColumn(1));
	
	if (Vec3(forward.x,forward.y,0).len2()>0.001f)
	{
		//force the up vector to be 0,0,1 for now
		Vec3 up(0,0,1);
		Vec3 right = (up % forward).GetNormalized();
		
		Quat goalQuat(Matrix33::CreateFromVectors(right % up,right,up)*Matrix33::CreateRotationZ(gf_PI*-0.5f));
		Quat currQuat(m_baseMtx);

		float rotSpeed = m_params.rotSpeed_min + (1.0f - (max(GetStanceInfo( m_stance )->maxSpeed - max(m_stats.speedModule - m_params.speed_min,0.0f),0.0f) / GetStanceInfo( m_stance )->maxSpeed)) * (m_params.rotSpeed_max - m_params.rotSpeed_min);
		Interpolate(m_turnSpeed,rotSpeed,3.0f,frameTime);

		m_baseMtx = Matrix33(Quat::CreateSlerp( currQuat.GetNormalized(), goalQuat, frameTime * m_turnSpeed ));
		m_baseMtx.NoScale();
	}

}

void CTrooper::ProcessMovement(float frameTime)
{
	if (frameTime > 0.1f)
			frameTime = 0.1f;

	//movement
	Vec3 move(m_input.movementVector);//usually 0, except AIs
	
	if (!m_stats.isFloating)
		move -= move * (m_baseMtx * Matrix33::CreateScale(Vec3(0,0,1)));//make it flat

/*	if (m_input.actions & ACTION_JUMP)
		m_input.deltaMovement.z += 1.0f; 
	if (m_input.actions & ACTION_CROUCH)
		m_input.deltaMovement.z -= 1.0f;*/

	move += m_baseMtx.GetColumn(0) * m_input.deltaMovement.x;
	move += m_baseMtx.GetColumn(1) * m_input.deltaMovement.y;
	move += m_baseMtx.GetColumn(2) * m_input.deltaMovement.z;

	//cap the movement vector to max 1
	float moveModule(move.len());

	if (moveModule > 1.0f)
	{
		move /= moveModule;
	}

	move *= GetStanceInfo( m_stance )->maxSpeed;

	if (m_stats.sprintLeft)
		move *= m_params.sprintMultiplier;

	SCharacterMoveRequest moveRequest;
	moveRequest.type = eCMT_Normal;

	float jumpToLen2(m_params.jumpTo.len2());
	if ((m_input.actions & ACTION_JUMP || jumpToLen2>0.01f) && m_stats.onGround)
	{
		if (jumpToLen2)
		{
			//TODO:make this calculation more accurate/precise
			Vec3 delta = m_params.jumpTo - GetEntity()->GetWorldPos();
			float deltaZ(delta.z);
			delta.z = 0;
			move = delta * 0.75f;

			float jumpLen(delta.len());
			float gravityLen(m_stats.gravity.len());
			//jumpLen = delta.len();

			move += m_baseMtx.GetColumn(2) * cry_sqrtf( 2.0f * gravityLen * ((1.0f) + max(0.0f,deltaZ*1.0f)) );
			
			m_params.jumpTo.Set(0,0,0);
		}
		else
		{
			move += m_baseMtx.GetColumn(2) * cry_sqrtf( 2.0f * m_stats.gravity.len() * 1.5f );
		}

		moveRequest.type = eCMT_JumpAccumulate;
	}
	
	//CryLogAlways("ground %.1f, inair %.1f",m_stats.onGround,m_stats.inAir);

	//LM related
	ICharacterInstance *pCharacter = GetEntity()->GetCharacter(0);

#if !PLAYER_REFACTORING
	if (m_locomanized && pCharacter)
	{
		m_stats.desiredSpeed = moveRequest.movement.len();

		Matrix33 modelMtx(GetEntity()->GetRotation());

		Vec3 up = modelMtx.GetColumn(2);
		Vec3 forward = modelMtx * -pCharacter->GetISkeleton()->GetMoveDirection();
				
		modelMtx.SetFromVectors(forward % up,forward,up);
		modelMtx.Invert();

		Vec3 refDir(m_viewMtx.GetColumn(1) + actionMove.dir * 10);
		refDir.NormalizeSafe(ZERO);

		//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(0,255,0,100), GetEntity()->GetWorldPos() + refDir * 2.0f, ColorB(255,0,0,100));

		Vec3 localDir(modelMtx * refDir);
		
		float goalX = -asin(localDir.z);
		float goalZ = cry_atan2f(-localDir.x,-localDir.y);

		Interpolate(m_stats.xDelta,goalX,5.0f,frameTime,0.5f);
		Interpolate(m_stats.zDelta,goalZ,5.0f,frameTime,0.5f);
	}
	else
#endif
	{
		//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(0,255,0,255), GetEntity()->GetWorldPos() + move, ColorB(255,0,0,255));

		m_stats.desiredSpeed = m_stats.speedModule;
#if !PLAYER_REFACTORING
		m_stats.xDelta = m_stats.zDelta = 0.0f;
#endif

		moveRequest.movement = move;
	}

	if (m_pAnimatedCharacter)
		m_pAnimatedCharacter->AddMovement(moveRequest);
}

void CTrooper::ProcessAnimation(ICharacterInstance *pCharacter,float frameTime)
{
#if !PLAYER_REFACTORING
	if (!m_locomanized)
#endif
	{
		Vec3 forward(m_viewMtx.GetColumn(1) + m_stats.velocity * 10);
		forward.NormalizeSafe();
		Vec3 up(m_baseMtx.GetColumn(2));

		/*if (m_stats.inAir>0.2f)
		{
			up += m_stats.velocity.GetNormalizedSafe(ZERO)*0.75f;
			up.NormalizeSafe();
		}*/

		if ((forward-up).len2()>0.001f)
		{
			float dotY(forward * m_viewMtx.GetColumn(1));
			float dotX(forward * m_viewMtx.GetColumn(0));

			//char dirStr[32];
			
			if (dotY<-0.3f)
			{
				m_stats.movementDir = 1;
				//strcpy(dirStr,"back");
			}
			else if (dotX>0.6f)
			{
				m_stats.movementDir = 2;
				//strcpy(dirStr,"right");
			}
			else if (dotX<-0.6f)
			{
				m_stats.movementDir = 3;
				//strcpy(dirStr,"left");
			}
			else
			{
				m_stats.movementDir = 0;
				//strcpy(dirStr,"front");
			}

			//float color[4] = {1,1,1,1};
			//GetISystem()->GetIRenderer()->Draw2dLabel(300,300, 1.5f, color, false,"dotY:%.1f, dotX:%.1f (%s)", dotY, dotX, dirStr);

			if (m_stats.inAir<0.2f)
				forward = m_viewMtx.GetColumn(1);

			forward *= -1.0f;

			Vec3 right = -(up % forward).GetNormalizedSafe(Vec3(1,0,0));

			//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(255,0,0,255), GetEntity()->GetWorldPos() + (up%right) * 1.0f, ColorB(255,0,0,255));
			//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(0,255,0,255), GetEntity()->GetWorldPos() + right * 1.0f, ColorB(0,255,0,255));
			//GetISystem()->GetIRenderer()->GetIRenderAuxGeom()->DrawLine(GetEntity()->GetWorldPos(), ColorB(0,0,255,255), GetEntity()->GetWorldPos() + up * 1.0f, ColorB(0,0,255,255));

			//moving backward?
			Vec3 viewDir(m_viewMtx.GetColumn(1));
			viewDir -= (m_baseMtx * Matrix33::CreateScale(Vec3(0,0,1))) * viewDir;

			viewDir.NormalizeSafe();
			forward.NormalizeSafe();

			float roll = forward * Matrix33(GetEntity()->GetRotation()).GetColumn(0);
			if (m_stance != STANCE_PRONE)
				roll *= 0.1f;

			roll = min(max(-DEG2RAD(45.0f),roll),DEG2RAD(45.0f));

			Quat goalQuat(Matrix33::CreateFromVectors(right,up%right,up) * Matrix33::CreateRotationY(roll));

			float rotSpeed = m_params.rotSpeed_min + (1.0f - (max(GetStanceInfo( m_stance )->maxSpeed - max(m_stats.speedModule - m_params.speed_min,0.0f),0.0f) / GetStanceInfo( m_stance )->maxSpeed)) * (m_params.rotSpeed_max - m_params.rotSpeed_min);

			m_modelQuat = Quat::CreateSlerp(GetEntity()->GetRotation().GetNormalized(), goalQuat, min(1.0f,frameTime * rotSpeed));
		}

		if (m_pAnimatedCharacter)
		{
			SCharacterMoveRequest moveRequest;
			moveRequest.type = eCMT_JustRotation;
			//FIXME:remove the * rotationZ
			moveRequest.turn = GetEntity()->GetRotation().GetInverted() * m_modelQuat * Quat::CreateRotationZ(gf_PI);

			m_pAnimatedCharacter->AddMovement(moveRequest);
		}
	}

	if (pCharacter)
	{
		pCharacter->GetISkeleton()->SetLookIK(true,gf_PI*0.9f,m_stats.lookTargetSmooth);
	}
}

void CTrooper::CreateCodeEvent(SmartScriptTable &rTable)
{
	const char *event = NULL;
	rTable->GetValue("event",event);

	if (event && !strcmp(event,"beamStart"))
	{
		const char *effect = NULL;
		ScriptHandle id;
		id.n = 0;

		if (rTable->GetValue("effect",effect) && rTable->GetValue("targetId",id))
			m_beamEffect.CreateBeam(GetEntity(),effect,id.n);
	}
	else if (event && !strcmp(event,"beamStop"))
	{
		m_beamEffect.RemoveBeam(GetEntity());
	}
	else
		CActor::CreateCodeEvent(rTable);
}

//---------------------------------
//AI Specific
void CTrooper::SetActorMovement(SOBJECTSTATE &control)
{
	if (IsClient())
		return;

  CAlien::SetActorMovementCommon(control);
	SetDesiredDirection(control.vLookDir);
	SetDesiredSpeed(control.vMoveDir*control.fDesiredSpeed);

  //	m_input.actions = control.m_desiredActions;
	int actions;
	switch(control.bodystate)
	{
	case 1:
		actions = ACTION_CROUCH;
		break;
	case 2:
		actions = ACTION_PRONE;
		break;
	case 3:
		actions = ACTION_RELAXED;
		break;
	case 4:
		actions = ACTION_STEALTH;
		break;
	default:
		actions = 0;
		break;
	}

	// Override the stance based on special behavior.
	SetActorStance(control, actions);

	m_input.actions = actions;

	GetEntity()->GetScriptTable()->SetValue( "fireDir", control.vFireDir );

//	CryLog( "mv: (%f, %f, %f)", m_input.movementVector.x, m_input.movementVector.y, m_input.movementVector.z );
//	CryLog( "desiredSpeed: %f (mv.len=%f)", control.m_desiredSpeed, control.m_movementVector.GetLength() );

	//weapons
	// trooper doesn't fire if it moves
	bool bFire = control.fire && (control.fDesiredSpeed<0.1f);

	if (bFire)
	{
		if (!m_stats.isFiring)
			OnAction("attack1", eAAM_OnPress, 1.0f);
	}
	else
	{
		if (m_stats.isFiring)
			OnAction("attack1", eAAM_OnRelease, 1.0f);
	}

	m_stats.isFiring = bFire;
}

void CTrooper::SetActorStance(SOBJECTSTATE &control, int& actions)
{
	// Set stance based on the desired speed and distance
	if( control.vMoveDir.GetLengthSquared() > 0.1f && control.fDesiredSpeed > 0.25f )
	{
		if(m_stance == STANCE_PRONE) //trooper is prone and moving, remain prone
			actions = ACTION_PRONE;
		else
		{// check distance to run, go prone only if there's more than a (threshold) meters to run
			IPuppet* pPuppet;
			if(GetEntity() && GetEntity()->GetAI() && GetEntity()->GetAI()->CanBeConvertedTo(AIOBJECT_PUPPET,(void**)&pPuppet))
			{
				float distance = control.fDistanceToPathEnd;
				if(distance > 3)
					actions = ACTION_PRONE;
			}
		}
	}
	else if(control.fDesiredSpeed<0.1f && m_stance == STANCE_PRONE)
		actions=ACTION_RELAXED;
}


