/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2005.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: Implements a base class for vehicle movements

-------------------------------------------------------------------------
History:
- 09:28:2005: Created by Mathieu Pinard

*************************************************************************/
#include "StdAfx.h"
#include "Game.h"

#include "IVehicleSystem.h"
#include "VehicleMovementBase.h"
#include <IGameTokens.h>
#include "IMaterialEffects.h"
#include <IEffectSystem.h>
#include "GameUtils.h"


XmlNodeRef CVehicleMovementBase::m_sfxNode = 0;


//------------------------------------------------------------------------
CVehicleMovementBase::CVehicleMovementBase()
: m_actor(0),
  m_pVehicle(0),
  m_pEntity(0),
  m_pPhysics(0),
  m_pGameTokenSystem(0),
  m_pVehicleSystem(0),
  m_maxSpeed(0.f),
  m_pSilentStart(0),
  m_enginePos(ZERO),
  m_runSoundDelay(0.f),
  m_rpmScale(0.f),
  m_pRunFadeInTime(0),
  m_pRunFadeOutTime(0),
  m_rpmPitchSpeed(0.f)
{ 
}

//------------------------------------------------------------------------
bool CVehicleMovementBase::Init(IVehicle* pVehicle, const SmartScriptTable &table)
{
	m_pVehicle = pVehicle;
	m_pEntity = pVehicle->GetEntity();
	m_pPhysics = m_pEntity->GetPhysics();
  m_p3DEngine = GetISystem()->GetI3DEngine();
  m_pGameTokenSystem = g_pGame->GetIGameFramework()->GetIGameTokenSystem();
  m_pVehicleSystem = g_pGame->GetIGameFramework()->GetIVehicleSystem();
  m_pSilentStart = GetISystem()->GetIConsole()->GetCVar("v_runSoundsSilentStart");
  m_pRunFadeInTime = GetISystem()->GetIConsole()->GetCVar("v_runSoundFadeInTime");
  m_pRunFadeOutTime = GetISystem()->GetIConsole()->GetCVar("v_runSoundFadeOutTime");
  
  m_soundNames[eSID_Start] = "start";
  m_soundNames[eSID_Run] = "run";
  m_soundNames[eSID_Stop] = "stop";
  m_soundNames[eSID_Ambience] = "ambience";

  // init particles
  m_pPaParams = m_pVehicle->GetParticleParams();  
  InitExhaust();    
  
	m_pEntitySoundsProxy = (IEntitySoundProxy*) m_pEntity->CreateProxy(ENTITY_PROXY_SOUND);
	assert(m_pEntitySoundsProxy);
	if (GetISystem()->GetISoundSystem())
	{
		//string soundName = string("Sounds/Vehicles:") + string(m_pEntity->GetClass()->GetName()).MakeLower() + string(":run");
		//GetISystem()->GetISoundSystem()->Precache(soundName, FLAG_SOUND_DEFAULT_3D, true);
	}

  for (int i=0; i<eSID_Max; ++i)
    m_soundIds[i] = INVALID_SOUNDID;

	m_isEnginePowered = false;

	m_isEngineStarting = false;
	m_isEngineGoingOff = false;
	m_engineStartup = 0.0f;
  m_crossFade = 0.f;
	m_damage = 0.0f;

	if (!table->GetValue("engineIgnitionTime", m_engineIgnitionTime))
		m_engineIgnitionTime = 1.6f;

  SmartScriptTable soundParams;
  if (table->GetValue("SoundParams", soundParams))
  {
    const char* pEngineSoundPosName;
    if (soundParams->GetValue("engineSoundPosition", pEngineSoundPosName))
    {
      m_enginePos = m_pVehicle->GetHelperLocalTM(pEngineSoundPosName).GetTranslation();		      
    }

    soundParams->GetValue("rpmPitchSpeed", m_rpmPitchSpeed);
    soundParams->GetValue("runSoundDelay", m_runSoundDelay);
    
    if (m_runSoundDelay > 0.f)
    {
      // if runDelay set, it determines the engineIgnitionTime
      m_engineIgnitionTime = m_runSoundDelay + m_pRunFadeInTime->GetFVal();
    }
  }
		
	return true;
}


//------------------------------------------------------------------------
void CVehicleMovementBase::Physicalize()
{
	m_pPhysics=m_pEntity->GetPhysics();
}

//------------------------------------------------------------------------
void CVehicleMovementBase::InitExhaust()
{ 
  for (int i=0; i<m_pPaParams->GetExhaustParams()->GetExhaustCount(); ++i)
  {
    SExhaustStatus stat;
    stat.helper = m_pPaParams->GetExhaustParams()->GetHelper(i);
    m_paStats.exhaustStats.push_back(stat);    
  }
  
  m_paStats.pExhaustCVar = GetISystem()->GetIConsole()->GetCVar("v_pa_exhaust");
  assert(m_paStats.pExhaustCVar);
}

//------------------------------------------------------------------------
void CVehicleMovementBase::Release()
{

}

//------------------------------------------------------------------------
void CVehicleMovementBase::Reset()
{
	m_damage = 0.0f;

	m_isEngineGoingOff = false;
	m_isEngineStarting = false;
	m_engineStartup = 0.0f;
  m_rpmScale = 0.f;
  m_crossFade = 0.f;

	if (m_isEnginePowered)
	{
		m_isEnginePowered = false;
		OnEngineCompletelyStopped();
		m_pVehicle->GetGameObject()->DisableUpdateSlot(m_pVehicle, IVehicle::eVUS_EnginePowered);
	}

	StopSounds();
  ResetParticles();

  m_movementAction.brake = true;
}

//------------------------------------------------------------------------
void CVehicleMovementBase::ProcessMovement(const float deltaTime)
{
	assert(m_pEntity);

	m_pPhysics = m_pEntity->GetPhysics();
	assert(m_pPhysics);

	if (m_actor && m_actor->IsPlayer())
		ProcessActions(deltaTime);
	else
		ProcessAI(deltaTime);
}

//------------------------------------------------------------------------
void CVehicleMovementBase::RequestActions(const SVehicleMovementAction& movementAction)
{ 
	m_movementAction = movementAction;

	for (TVehicleMovementActionFilterList::iterator ite = m_actionFilters.begin(); ite != m_actionFilters.end(); ++ite)
	{
		IVehicleMovementActionFilter* pActionFilter = *ite;
		pActionFilter->OnProcessActions(m_movementAction);
	}
}

//------------------------------------------------------------------------
void CVehicleMovementBase::Update(const float deltaTime)
{  
  FUNCTION_PROFILER( GetISystem(), PROFILE_GAME );

  m_pPhysics->GetStatus(&m_statusDyn); 
    
  UpdateExhaust(deltaTime);
  UpdateEnvParticles(deltaTime);
  DebugDrawSounds();
  
	if (m_isEngineStarting)
	{
		m_engineStartup += deltaTime;

    if (m_runSoundDelay>0.f && m_engineStartup >= m_runSoundDelay)
    {
      ISound* pRunSound = GetSound(eSID_Run);
      if (!pRunSound)
      {
        // start run
        string runSound = GetSoundName(eSID_Run);
        m_soundIds[eSID_Run] = m_pEntitySoundsProxy->PlaySoundEx(runSound.c_str(), m_enginePos, FORWARD_DIRECTION, FLAG_SOUND_DEFAULT_3D|FLAG_SOUND_LOOP, 255, 0, ENGINESOUND_MAX_DIST);      

        pRunSound = GetSound(eSID_Run);        
      }

      // "fade" in run and ambience
      float fadeInRatio = min(1.f, (m_engineStartup-m_runSoundDelay)/m_pRunFadeInTime->GetFVal());
      m_rpmScale = fadeInRatio * ENGINESOUND_IDLE_RATIO;

      SetSoundParam(eSID_Run, "rpm_scale", m_rpmScale);
      SetSoundParam(eSID_Ambience, "speed", m_rpmScale);
    }

		if (m_engineStartup >= m_engineIgnitionTime)		
		{
			m_isEngineStarting = false;
			m_isEnginePowered = true;
		}
	}
	else if (m_isEngineGoingOff)
	{
		m_engineStartup -= deltaTime;
  
    if (m_rpmScale <= ENGINESOUND_IDLE_RATIO && GetSoundId(eSID_Stop) == INVALID_SOUNDID)
    {
      // start stop sound and stop start sound, for now without fading
      StopSound(eSID_Start);

      string stopSound = GetSoundName(eSID_Stop);
      m_soundIds[eSID_Stop] = m_pEntitySoundsProxy->PlaySoundEx(stopSound.c_str(), m_enginePos, FORWARD_DIRECTION, FLAG_SOUND_DEFAULT_3D, 255, 0, ENGINESOUND_MAX_DIST);
    }
    
    // handle run sound 
    if (ISound* pRunSound = GetSound(eSID_Run))
    {
      if (m_rpmScale <= 0.f)
        StopSound(eSID_Run);
      else      
        pRunSound->SetParam("rpm_scale", m_rpmScale, false);      
    }
    
    m_rpmScale = max(0.f, m_rpmScale - 1.f/m_pRunFadeOutTime->GetFVal()*deltaTime);  
            
		if (m_engineStartup <= 0.0f)		
		{
			m_isEngineGoingOff = false;
			m_isEnginePowered = false;

			OnEngineCompletelyStopped();
			m_pVehicle->GetGameObject()->DisableUpdateSlot(m_pVehicle, IVehicle::eVUS_EnginePowered);
		}
	}
  else if (m_isEnginePowered)
  {
    float speedRatio = 0.2f + 0.8f * min(1.f, m_statusDyn.v.len2()/sqr(m_maxSpeed));
    
    SetSoundParam(eSID_Run, "speed", speedRatio);
    SetSoundParam(eSID_Ambience, "speed", speedRatio);

    if (m_rpmPitchSpeed>0.f)
    {
      // pitch rpm with pedal      
      float delta = abs(GetEnginePedal()) - m_rpmScale;            
      m_rpmScale = max(0.2f, min(1.f, m_rpmScale + sgn(delta)*min(abs(delta), m_rpmPitchSpeed*deltaTime)));
      
      SetSoundParam(eSID_Run, "rpm_scale", m_rpmScale);
    }
  }

  UpdateGameTokens(deltaTime);
}


//------------------------------------------------------------------------
void CVehicleMovementBase::UpdateGameTokens(const float deltaTime)
{
  if (m_pVehicle->IsPlayerDriving())
  {
    float speedNorm = CLAMP(m_statusDyn.v.len()/(m_maxSpeed?m_maxSpeed:1.f), 0.f, 1.f);
    m_pGameTokenSystem->SetOrCreateToken("vehicle.speedNorm", TFlowInputData(speedNorm, true));
  }    
}

//------------------------------------------------------------------------
void CVehicleMovementBase::OnSoundEvent(ESoundCallbackEvent event,ISound *pSound)
{
}


//------------------------------------------------------------------------
bool CVehicleMovementBase::StartEngine(EntityId driverId)
{  
	if (m_isEnginePowered)
	{
		if (!m_isEngineGoingOff)
			return false;
	}

	if (m_damage >= 1.0f)
		return false;

  // check bDisableEngine property
  IScriptTable* sT = m_pEntity->GetScriptTable();
  SmartScriptTable props;
   
  if (sT && sT->GetValue("Properties", props))
  {
    bool bDisable = false;
    props->GetValue("bDisableEngine", bDisable);
    if (bDisable)    
      return false;    
  }

	m_isEngineGoingOff = false;
	m_isEngineStarting = true;
	m_engineStartup = 0.0f;
  m_rpmScale = 0.f;
  m_engineIgnitionTime = m_runSoundDelay + m_pRunFadeInTime->GetFVal();
	
  StopSound(eSID_Stop);

  string startSound = GetSoundName(eSID_Start);	
  m_soundIds[eSID_Start] = m_pEntitySoundsProxy->PlaySoundEx(startSound.c_str(), m_enginePos, FORWARD_DIRECTION, FLAG_SOUND_DEFAULT_3D, 255, 0, ENGINESOUND_MAX_DIST);

  StartExhaust();  

  m_actor = g_pGame->GetIGameFramework()->GetIActorSystem()->GetActor(driverId);

	m_pVehicle->GetGameObject()->EnableUpdateSlot(m_pVehicle, IVehicle::eVUS_EnginePowered);

	m_movementAction.brake = false;
	return true;
}

//------------------------------------------------------------------------
void CVehicleMovementBase::StopEngine()
{
  m_actor = NULL;

  if (!m_isEngineStarting && !m_isEnginePowered)
	{		
		return;
	}
  
	//if (m_isEngineStarting)
		//m_engineStartup = m_engineIgnitionTime - m_engineStartup;
	
  if (!m_isEngineStarting)
		m_engineStartup = m_engineIgnitionTime;

	m_isEngineStarting = false;
	m_isEngineGoingOff = true;
  
  m_movementAction.Clear();
  m_movementAction.brake = true;

  StopExhaust();
  
  m_pGameTokenSystem->SetOrCreateToken("vehicle.speedNorm", TFlowInputData(0.f, true));
}

//------------------------------------------------------------------------
void CVehicleMovementBase::OnEvent(EVehicleMovementEvent event, float value)
{
	if (event == eVME_Damage || event == eVME_Freeze)
	{
		m_damage = value;

		if (value >= 1.0f)
		{
      if (m_isEnginePowered)
      { 
        m_pVehicle->GetGameObject()->DisableUpdateSlot(m_pVehicle, IVehicle::eVUS_EnginePowered);    
      }
			m_isEnginePowered = false;
			m_isEngineStarting = false;
			m_isEngineGoingOff = false;

			m_movementAction.Clear();

			StopExhaust();
			
			// TODO: something nicer would be better
			StopSounds();
		}
	}
	else if (event == eVME_PlayerIsDrivingFP)
	{
		if (value == 1.f)
		{ 
			if (GetSoundId(eSID_Ambience) == INVALID_SOUNDID)
			{        
				string ambienceSoundName = GetSoundName(eSID_Ambience);
				m_soundIds[eSID_Ambience] = m_pEntitySoundsProxy->PlaySound(ambienceSoundName.c_str(), Vec3(ZERO), FORWARD_DIRECTION, FLAG_SOUND_2D|FLAG_SOUND_LOOP);
			}

      // fade in ambience with rpmscale
      SetSoundParam(eSID_Ambience, "speed", m_rpmScale);       
		}
		else if (value == 0.f)
		{
			StopSound(eSID_Ambience);
		}
	}  
}

//------------------------------------------------------------------------
ISound* CVehicleMovementBase::GetSound(int eSID)
{
  assert(eSID>=0 && eSID<eSID_Max);

  if (m_soundIds[eSID] == INVALID_SOUNDID)
    return 0;

  return m_pEntitySoundsProxy->GetSound(m_soundIds[eSID]);
}

//------------------------------------------------------------------------
void CVehicleMovementBase::StopSounds()
{
  for (int i=0; i<eSID_Max; ++i)
  {
    StopSound(i);
  }	
}

//------------------------------------------------------------------------
void CVehicleMovementBase::StopSound(int eSID)
{
  assert(eSID>=0 && eSID<eSID_Max);

  if (m_soundIds[eSID] != INVALID_SOUNDID)
  {
    m_pEntitySoundsProxy->StopSound(m_soundIds[eSID]); 
    m_soundIds[eSID] = INVALID_SOUNDID;
  }
}

//------------------------------------------------------------------------
void CVehicleMovementBase::StartExhaust()
{
  if (m_paStats.pExhaustCVar->GetIVal() == 0)
    return;

  SExhaustParams* exParams = m_pPaParams->GetExhaustParams();

  if (!exParams->hasExhaust)
    return;
  
  // start effect
  string startEff = exParams->GetStartEffect();
  if (startEff.length())
  {
    IParticleEffect* pEff = m_p3DEngine->FindParticleEffect(startEff.c_str());

    if (pEff)    
    {
      for (std::vector<SExhaustStatus>::iterator it = m_paStats.exhaustStats.begin(); it != m_paStats.exhaustStats.end(); ++it)
      {
        if (GetWaterMod(*it))
        {
          //CryLogAlways("[VehicleMovement::StartExhaust]: Loading emitter to slot %i", it->startStopSlot);
          it->startStopSlot = m_pEntity->LoadParticleEmitter(it->startStopSlot, pEff);
          m_pEntity->SetSlotLocalTM(it->startStopSlot, m_pVehicle->GetHelperLocalTM(it->helper));
          //CryLogAlways("[VehicleMovement::StartExhaust]: Loaded to slot %i..", it->startStopSlot);
        }
      }
    }
  }

  // load emitter for exhaust running effect (this can be overidden)
  string runEff = exParams->GetRunEffect();
  if (runEff.length())
  {
    IParticleEffect* pEff = m_p3DEngine->FindParticleEffect(runEff.c_str());

    if (pEff)    
    {
      SEntitySlotInfo slotInfo;
      SpawnParams spawnParams;

      for (std::vector<SExhaustStatus>::iterator it = m_paStats.exhaustStats.begin(); it != m_paStats.exhaustStats.end(); ++it)
      { 
        //CryLogAlways("[VehicleMovement::StartExhaust]: Loading emitter to slot %i..", it->runSlot);
        it->runSlot = m_pEntity->LoadParticleEmitter(it->runSlot, pEff);
        m_pEntity->SetSlotLocalTM(it->runSlot, m_pVehicle->GetHelperLocalTM(it->helper));
        //CryLogAlways("[VehicleMovement::StartExhaust]: Loaded to slot %i..", it->runSlot);
        
        slotInfo.pParticleEmitter = 0;
        if (m_pEntity->GetSlotInfo(it->runSlot, slotInfo) && slotInfo.pParticleEmitter)
        { 
          spawnParams.fSizeScale = exParams->runBaseSizeScale;
          slotInfo.pParticleEmitter->SetSpawnParams(spawnParams);
        }
      }
    }
  }  
}

//------------------------------------------------------------------------
void CVehicleMovementBase::FreeEmitterSlot(int& slot)
{
  if (slot > -1)
  {
    //CryLogAlways("[VehicleMovement]: Freeing slot %i", slot);
    m_pEntity->FreeSlot(slot);
    slot = -1;
  }
}

//------------------------------------------------------------------------
void CVehicleMovementBase::FreeEmitterSlot(const int& slot)
{
  if (slot > -1)  
    m_pEntity->FreeSlot(slot);     
}


//------------------------------------------------------------------------
void CVehicleMovementBase::StopExhaust()
{
  if (m_paStats.pExhaustCVar->GetIVal() == 0)
    return;

  SExhaustParams* exParams = m_pPaParams->GetExhaustParams();

  if (!exParams->hasExhaust) 
    return;

  for (std::vector<SExhaustStatus>::iterator it = m_paStats.exhaustStats.begin(); it != m_paStats.exhaustStats.end(); ++it)
  {
    FreeEmitterSlot(it->runSlot);
    FreeEmitterSlot(it->startStopSlot);
  }

  // trigger stop effect if available
  if (0 != strlen(exParams->GetStopEffect()))
  {
    IParticleEffect* pEff = m_p3DEngine->FindParticleEffect(exParams->GetStopEffect());

    if (pEff)    
    { 
      for (std::vector<SExhaustStatus>::iterator it = m_paStats.exhaustStats.begin(); it != m_paStats.exhaustStats.end(); ++it)
      {
        if (GetWaterMod(*it))
        {
          //CryLogAlways("[VehicleMovement::StopExhaust]: Loading emitter to slot %i", it->startStopSlot);
          it->startStopSlot = m_pEntity->LoadParticleEmitter(it->startStopSlot, pEff);
          m_pEntity->SetSlotLocalTM(it->startStopSlot, m_pVehicle->GetHelperLocalTM(it->helper));        
          //CryLogAlways("[VehicleMovement::StopExhaust]: Loaded to slot %i..", it->startStopSlot);
        }
      }
    }
  }
}

//------------------------------------------------------------------------
void CVehicleMovementBase::ResetParticles()
{   
  // exhausts
  for (std::vector<SExhaustStatus>::iterator it = m_paStats.exhaustStats.begin(); it != m_paStats.exhaustStats.end(); ++it)
  {
    FreeEmitterSlot(it->startStopSlot);
    FreeEmitterSlot(it->runSlot);  
  }
  
  // environmental
  for (SEnvParticleStatus::TEnvEmitters::iterator it=m_paStats.envStats.emitters.begin(); it!=m_paStats.envStats.emitters.end(); ++it)
  {
    FreeEmitterSlot(it->slot);
    
    if (it->pGroundEffect)
      it->pGroundEffect->Stop(true);
      
  }
  //m_paStats.envStats.emitters.clear();
  //m_paStats.envStats.initalized = false;
}


//------------------------------------------------------------------------
void CVehicleMovementBase::UpdateExhaust(const float deltaTime)
{ 
  FUNCTION_PROFILER( GetISystem(), PROFILE_GAME );

  if (m_paStats.pExhaustCVar->GetIVal() == 0)
    return;

  SExhaustParams* exParams = m_pPaParams->GetExhaustParams();
  
  if (exParams->hasExhaust && m_paStats.exhaustStats[0].runSlot > -1)
  {
    SEntitySlotInfo info;
    SpawnParams sp;
    float countScale = 1;
    float sizeScale = 1;
    
    float vel = m_statusDyn.v.GetLength();
    float absPower = abs(m_movementAction.power);

    // disable running exhaust if requirements aren't met
    if (vel < exParams->runMinSpeed || vel > exParams->runMaxSpeed || absPower < exParams->runMinPower || absPower > exParams->runMaxPower)         
    {   
      countScale = 0;          
    }
    else
    { 
      // scale with engine power
      float powerDiff = max(0.f, exParams->runMaxPower - exParams->runMinPower);        

      if (powerDiff)
      {
        float powerNorm = CLAMP(absPower, exParams->runMinPower, exParams->runMaxPower) / powerDiff;                                

        float countDiff = max(0.f, exParams->runMaxPowerCountScale - exParams->runMinPowerCountScale);          
        if (countDiff)          
          countScale *= (powerNorm * countDiff) + exParams->runMinPowerCountScale;

        float sizeDiff  = max(0.f, exParams->runMaxPowerSizeScale - exParams->runMinPowerSizeScale);
        if (sizeDiff)                      
          sizeScale *= (powerNorm * sizeDiff) + exParams->runMinPowerSizeScale;                  
      }

      // scale with vehicle speed
      float speedDiff = max(0.f, exParams->runMaxSpeed - exParams->runMinSpeed);        

      if (speedDiff)
      {
        float speedNorm = CLAMP(vel, exParams->runMinSpeed, exParams->runMaxSpeed) / speedDiff;                                

        float countDiff = max(0.f, exParams->runMaxSpeedCountScale - exParams->runMinSpeedCountScale);          
        if (countDiff)          
          countScale *= (speedNorm * countDiff) + exParams->runMinSpeedCountScale;

        float sizeDiff  = max(0.f, exParams->runMaxSpeedSizeScale - exParams->runMinSpeedSizeScale);
        if (sizeDiff)                      
          sizeScale *= (speedNorm * sizeDiff) + exParams->runMinSpeedSizeScale;                  
      }
    }

    sp.fSizeScale = sizeScale;
    
    for (std::vector<SExhaustStatus>::iterator it = m_paStats.exhaustStats.begin(); it != m_paStats.exhaustStats.end(); ++it)
    {
      sp.fCountScale = countScale * GetWaterMod(*it);

      info.pParticleEmitter = 0;
      if (m_pEntity->GetSlotInfo(it->runSlot, info) && info.pParticleEmitter)
      { 
        info.pParticleEmitter->SetSpawnParams(sp);
      }      
    }
    
    if (DebugParticles())
    {
      IRenderer* pRenderer = GetISystem()->GetIRenderer();
      float color[4] = {1,1,1,1};
      float x = 200.f;

      pRenderer->Draw2dLabel(x,  80.0f, 1.5f, color, false, "Exhaust:");
      pRenderer->Draw2dLabel(x,  105.0f, 1.5f, color, false, "countScale: %.2f", sp.fCountScale);
      pRenderer->Draw2dLabel(x,  120.0f, 1.5f, color, false, "sizeScale: %.2f", sp.fSizeScale);
    }
  }
}

//------------------------------------------------------------------------
float CVehicleMovementBase::GetWaterMod(SExhaustStatus& exStatus)
{  
  // check water flag
  bool inWater = m_p3DEngine->IsPointInWater( m_pVehicle->GetHelperWorldTM(exStatus.helper).GetTranslation() );
  if ((inWater && !m_pPaParams->GetExhaustParams()->insideWater) || (!inWater && !m_pPaParams->GetExhaustParams()->outsideWater))
    return 0;

  return 1;
}

//------------------------------------------------------------------------
void CVehicleMovementBase::RegisterActionFilter(IVehicleMovementActionFilter* pActionFilter)
{
	if (pActionFilter)
		m_actionFilters.push_back(pActionFilter);
}

//------------------------------------------------------------------------
void CVehicleMovementBase::UnregisterActionFilter(IVehicleMovementActionFilter* pActionFilter)
{
	if (pActionFilter)
		m_actionFilters.remove(pActionFilter);
}

//------------------------------------------------------------------------
void CVehicleMovementBase::GetMovementState(SMovementState& movementState)
{
	movementState.minSpeed = 0.0f;
	movementState.normalSpeed = 15.0f;
	movementState.maxSpeed = 30.0f;
}

//------------------------------------------------------------------------
bool CVehicleMovementBase::IsProfilingMovement()
{
  if (!m_pVehicle || !m_pVehicle->GetEntity())
    return false;

  static ICVar* pProfile = GetISystem()->GetIConsole()->GetCVar("v_profileMovement");
  static ICVar* pDebugVehicle = GetISystem()->GetIConsole()->GetCVar("v_debugVehicle");

  if ((pProfile->GetIVal()==1 && m_pVehicle->IsPlayerDriving()) || 0 == strcmp(pDebugVehicle->GetString(), m_pVehicle->GetEntity()->GetName()))
    return true;

  return false;
}

//------------------------------------------------------------------------
XmlNodeRef CVehicleMovementBase::GetEffectNode(int matId)
{
  // load sfx xml once (until the api supports custom lookup)
  if (m_sfxNode == 0)
  {
    m_sfxNode = GetISystem()->LoadXmlFile("libs/materialeffects/fxlibs/vehicles.xml");
    
    if (m_sfxNode == 0)
    {
      CryLog("Failed to load fxlibs/vehicles.xml");
      return 0;
    }
  }

  if (matId <= 0)
    return 0;

  IMaterialEffects *mfx = g_pGame->GetIGameFramework()->GetIMaterialEffects();

  // maybe cache this
  const static string prefix = "vfx_";
  string effCol = prefix + m_pEntity->GetClass()->GetName();
  const char* effectId = mfx->GetEffectString(effCol.MakeLower().c_str(), matId);
  
  if (effectId)
  {
    string effName(effectId);    
    int idx;

    if ((idx = effName.find(":")) != string::npos)
    {
      effName = effName.substr(idx+1);
    }

    for (int i=0; i<m_sfxNode->getChildCount(); ++i)
    {
      XmlNodeRef child = m_sfxNode->getChild(i);
      if (!strcmp(child->getTag(), "Effect") && child->haveAttr("name") && !strcmp(child->getAttr("name"), effName.c_str()))
      {
        return child;
      }
    }
    if (DebugParticles())
      CryLog("matfx lookup for %s failed", effName.c_str());
  }
  else
  {
    if (DebugParticles())
      CryLog("GetEffectString for %s -> %i failed", effCol.c_str(), matId);
  }
  
  return 0;
}

//------------------------------------------------------------------------
const char* CVehicleMovementBase::GetEffectByIndex(const XmlNodeRef& effectNode, int idx)
{
  if (effectNode == 0 || idx >= effectNode->getChildCount())
    return 0;

  int num = 0;

  for (int i=0; i<effectNode->getChildCount(); ++i)
  {
    XmlNodeRef child = effectNode->getChild(i);
    if (!strcmp(child->getTag(), "Particle"))
    {
      if (idx == num)
      {
        XmlNodeRef name = child->findChild("Name");
        return (name != 0 && strlen(name->getContent())>0) ? name->getContent() : 0;
      } 
      ++num;
    }
  }
  return 0;  
}


//------------------------------------------------------------------------
void CVehicleMovementBase::InitEnvParticles()
{
  m_paStats.envStats.emitters.clear();
    
  SEnvironmentParticles* envParams = m_pPaParams->GetEnvironmentParticles();

  for (int iLayer=0; iLayer<envParams->GetLayerCount(); ++iLayer)
  {    
    const SEnvironmentLayer& layer = envParams->GetLayer(iLayer);
    
    int cnt = (layer.alignGroundHeight>0 || layer.alignToWater) ? 1 : layer.GetHelperCount();
    m_paStats.envStats.emitters.reserve( m_paStats.envStats.emitters.size() + cnt);    
    
    for (int i=0; i<cnt; ++i)
    { 
      TEnvEmitter emitter;
      emitter.layer = iLayer;   

      // dummy effect for loading
      IParticleEffect* pEff = m_p3DEngine->FindParticleEffect("vehicle_fx.vehicles_surface_fx.soil_dust");

      if (layer.alignGroundHeight>0)      
      { 
        // create ground effect if height specified
        IGroundEffect* pGroundEffect = g_pGame->GetIGameFramework()->GetIEffectSystem()->CreateGroundEffect(m_pEntity);

        if (pGroundEffect)
        {
          pGroundEffect->SetParticleEffect(pEff->GetName());
          pGroundEffect->SetHeight(layer.alignGroundHeight);
          pGroundEffect->SetHeightScale(layer.maxHeightSizeScale, layer.maxHeightCountScale);
          pGroundEffect->SetFlags(pGroundEffect->GetFlags() | IGroundEffect::eGEF_StickOnGround);
          
          string interaction("vfx_");
          interaction.append(m_pEntity->GetClass()->GetName()).MakeLower();
          
          pGroundEffect->SetInteraction(interaction.c_str());

          emitter.pGroundEffect = pGroundEffect;

          if (DebugParticles())
            CryLog("Ground effect loaded with height %f", layer.alignGroundHeight);
        }
      }      
      else if (layer.alignToWater)
      {
        // else load emitter in slot
        SpawnParams sp;         
        sp.fCountScale = 0.f;      

        int slot = m_pEntity->LoadParticleEmitter(-1, pEff, &sp);
        if (slot > -1) 
        { 
          Matrix34 tm;
          tm.SetIdentity();
          
          if (layer.GetHelperCount()>i)
            tm = m_pVehicle->GetHelperLocalTM(layer.GetHelper(i), false, false);
            
          m_pEntity->SetSlotLocalTM(slot, tm);
          emitter.slot = slot;        

          SEntitySlotInfo info;
          if (m_pEntity->GetSlotInfo(slot, info) && info.pParticleEmitter)
          {
            info.pParticleEmitter->SetSpawnParams(sp);
          }

          if (DebugParticles())
          {
            const Vec3 loc = tm.GetTranslation();
            CryLog("env emitter %i local pos: %.1f %.1f %.1f", i, loc.x, loc.y, loc.z);
            CryLog("env emitter %i loaded emitter in slot %i", i, slot);
          }      
        }
      }
      
      if (emitter.pGroundEffect || emitter.slot > -1)
        m_paStats.envStats.emitters.push_back(emitter);
    }
  }

  m_paStats.envStats.initalized = true;  
}

//------------------------------------------------------------------------
void CVehicleMovementBase::UpdateEnvParticles(const float deltaTime)
{
  FUNCTION_PROFILER( GetISystem(), PROFILE_GAME );

  if (!m_paStats.envStats.initalized)
    InitEnvParticles();

  SEnvironmentParticles* envParams = m_pPaParams->GetEnvironmentParticles();
  SEnvParticleStatus::TEnvEmitters::iterator emitterIt = m_paStats.envStats.emitters.begin();

  for (; emitterIt!=m_paStats.envStats.emitters.end(); emitterIt++)  
  { 
    if (emitterIt->layer < 0)
    {
      assert(0);
      continue;
    }
    
    const SEnvironmentLayer& layer = envParams->GetLayer(emitterIt->layer);
    
    float countScale = 1;
    float sizeScale = 1;
    float vel = m_statusDyn.v.len();
    float powerNorm = CLAMP(abs(m_movementAction.power), 0.f, 1.f);

    if (vel < layer.minSpeed || (emitterIt->pGroundEffect && !(m_isEngineGoingOff|m_isEnginePowered)))
    {   
      countScale = 0;          
    }
    else
    { 
      float speedDiff = max(0.f, layer.maxSpeed - layer.minSpeed);        

      if (speedDiff)
      {
        float speedNorm = CLAMP(vel, layer.minSpeed, layer.maxSpeed) / speedDiff;                                

        float countDiff = max(0.f, layer.maxSpeedCountScale - layer.minSpeedCountScale);          
        if (countDiff)          
          countScale *= (speedNorm * countDiff) + layer.minSpeedCountScale;

        float sizeDiff  = max(0.f, layer.maxSpeedSizeScale - layer.minSpeedSizeScale);
        if (sizeDiff)                      
          sizeScale *= (speedNorm * sizeDiff) + layer.minSpeedSizeScale;                  
      }
      
      float countDiff = max(0.f, layer.maxPowerCountScale - layer.minPowerCountScale);          
      if (countDiff)          
        countScale *= (powerNorm * countDiff) + layer.minPowerCountScale;

      float sizeDiff  = max(0.f, layer.maxPowerSizeScale - layer.minPowerSizeScale);
      if (sizeDiff)                      
        sizeScale *= (powerNorm * sizeDiff) + layer.minPowerSizeScale;    
    }
    
    if (emitterIt->pGroundEffect)
    {
      emitterIt->pGroundEffect->Stop(false);
      emitterIt->pGroundEffect->SetBaseScale(sizeScale, countScale);
      emitterIt->pGroundEffect->Update();
    }    
    else
    {
      if (layer.alignToWater)
      {
        SEntitySlotInfo info;        
        if (m_pEntity->GetSlotInfo(emitterIt->slot, info) && info.pParticleEmitter)
        { 
          // check if helper position is beneath water level
          static int matIdWater = GetISystem()->GetI3DEngine()->GetMaterialManager()->GetSurfaceTypeIdByName("mat_water");
          Vec3 pos;
          float matId = 0;          
          
          if (layer.GetHelperCount()>0)
            pos = m_pVehicle->GetHelperWorldTM(layer.GetHelper(0)).GetTranslation();
          else
            pos = m_pVehicle->GetEntity()->GetWorldPos();
          
          if (pos.z <= GetISystem()->GetI3DEngine()->GetWaterLevel(&pos))
            matId = matIdWater;
          
          if (matId == 0)
          {
            countScale = 0.f;
          }
          else if (matId != emitterIt->matId)
          {
            // change effect       
            IParticleEffect* pEff = 0;    
            const char* effect = GetEffectByIndex( GetEffectNode(int(matId)), emitterIt->layer );
            
            if (effect && (pEff = m_p3DEngine->FindParticleEffect(effect)))
            {           
              if (DebugParticles())              
                CryLog("%s changes sfx to %s (slot %i)", m_pEntity->GetName(), effect, emitterIt->slot);
              
              info.pParticleEmitter->SetEffect(pEff);              
            }
            else
              countScale = 0.f;
          }
          
          emitterIt->matId = int(matId);

          SpawnParams sp;
          sp.fSizeScale = sizeScale;
          sp.fCountScale = countScale;
          info.pParticleEmitter->SetSpawnParams(sp);
        }
      }
    }
    
    if (DebugParticles())
    {
      float color[] = {1,1,1,1};
      GetISystem()->GetIRenderer()->Draw2dLabel(200, 300+25*emitterIt->group, 1.5f, color, false, "group %i: sizeScale %f, countScale %f", emitterIt->group, sizeScale, countScale);
    }               
  }
}

//------------------------------------------------------------------------
string CVehicleMovementBase::GetSoundName(int eSID)
{
  assert(eSID>=0 && eSID<eSID_Max);

  if (m_pVehicleSystem->IsSoundEnabled(m_soundNames[eSID].c_str()) <= 0)
    return "";

  string sound("Sounds/Vehicles:");
  string entityClass(m_pEntity->GetClass()->GetName());
  
  entityClass = entityClass.substr(0, entityClass.find("_new"));
  entityClass = entityClass.substr(0, entityClass.find("_old"));
  
  sound.append(entityClass);
  sound.MakeLower();   

  sound.append(":");
  sound.append(m_soundNames[eSID]);
  
  return sound;
}

//------------------------------------------------------------------------
void CVehicleMovementBase::SetSoundParam(int eSID, const char* param, float value)
{
  if (ISound* pSound = GetSound(eSID))
  { 
    pSound->SetParam(param, value, false);
  }		
}

//------------------------------------------------------------------------
void CVehicleMovementBase::DebugDrawSounds()
{
  static float color[] = {1,1,1,1};
  static ICVar* pVar = GetISystem()->GetIConsole()->GetCVar("v_debugdraw");
  float val = 0.f;

  if (pVar->GetIVal() != eVDB_Sounds)
    return;

  GetISystem()->GetIRenderer()->Draw2dLabel(500,100,1.5f,color,false,"vehicle rpm: %.2f", m_rpmScale);

  if (ISound* pSound = GetSound(eSID_Run))
  { 
    if (pSound->GetParam("rpm_scale", &val, false))
      GetISystem()->GetIRenderer()->Draw2dLabel(500,125,1.5f,color,false,":run rpm_scale: %.2f", val);

    if (pSound->GetParam("speed", &val, false))
      GetISystem()->GetIRenderer()->Draw2dLabel(500,150,1.5f,color,false,":run speed: %.2f", val);
  }

  if (ISound* pSound = GetSound(eSID_Ambience))
  { 
    if (pSound->GetParam("speed", &val, false))
      GetISystem()->GetIRenderer()->Draw2dLabel(500,175,1.5f,color,false,":ambient speed: %.2f", val);
  }
}

//------------------------------------------------------------------------