/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2005.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: Implements a standard helicopter movements

-------------------------------------------------------------------------
History:
- 04:04:2005: Created by Mathieu Pinard

*************************************************************************/
#include "StdAfx.h"
#include "Game.h"

#include "IMovementController.h"
#include "IVehicleSystem.h"
#include "VehicleMovementHelicopter.h"
#include "ICryAnimation.h"

#include "IRenderAuxGeom.h"

//------------------------------------------------------------------------
CVehicleMovementHelicopter::CVehicleMovementHelicopter()
{	
	m_isDestroyed = false;

	m_netActionSync.PublishActions( CNetworkMovementHelicopter(this) );
}

//------------------------------------------------------------------------
bool CVehicleMovementHelicopter::Init(IVehicle* pVehicle, const SmartScriptTable &table)
{
	if (!CVehicleMovementBase::Init(pVehicle, table))
	{
		assert(0);
	}

	table->GetValue("rotationDamping", m_rotationDamping);
	table->GetValue("dampInertia", m_dampInertia);
	table->GetValue("dampInertiaIdle", m_dampInertiaIdle);
	table->GetValue("rollSpeed", m_rollSpeed);
	table->GetValue("pitchSpeed", m_pitchSpeed);
	table->GetValue("yawSpeed", m_yawSpeed);
	table->GetValue("rollWYawAngle", m_rollWYawAngle);
	table->GetValue("rollRelax", m_rollRelax);
	table->GetValue("rollSpeed", m_rollSpeed);
	table->GetValue("pitchRelax", m_pitchRelax);
	table->GetValue("enginePower", m_enginePower);
	table->GetValue("enginePowerMax", m_enginePowerMax);
	table->GetValue("engineForce", m_engineForce);
	table->GetValue("engineForceDeceleration", m_engineForceDeceleration);
	table->GetValue("engineAngle", m_engineAngle);
	table->GetValue("engineAngleForward", m_engineAngleForward);
	table->GetValue("engineAltAngle", m_engineAltAngle);
	table->GetValue("engineActualAngle", m_engineActualAngle);
	table->GetValue("movingSpeed", m_movingSpeed);

	m_engineAngle = DEG2RAD(m_engineAngle);
  m_engineAngleForward = DEG2RAD(m_engineAngleForward);
	m_engineAltAngle = DEG2RAD(m_engineAltAngle);
	m_engineActualAngle = DEG2RAD(m_engineActualAngle);

	m_enginePower = 0.0f;
	m_engineGoalThrottle = 0.0f;

	if (!table->GetValue("enginePowerMax", m_enginePowerMax))
		m_enginePowerMax = 0.0f;

	// Initialise the power PID.
	m_powerPID.Reset();
	m_powerPID.m_kP = 0.2f;
	m_powerPID.m_kD = 0.01f;
	m_powerPID.m_kI = 0.0001f;

	m_steeringDamage = 0.0f;

  m_maxSpeed = 40.f; // empirically determined

	m_pitchActionMul = 1.0f;

	ResetActions();

	m_pRotorAnim = NULL;
	m_pLandingGearsAnim = NULL;

	return true;
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::PostInit()
{
	m_pRotorAnim = m_pVehicle->GetAnimation("rotor");
	m_pLandingGearsAnim = m_pVehicle->GetAnimation("landingGears");
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::Release()
{
	CVehicleMovementBase::Release();
	delete this;
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::Physicalize()
{
	CVehicleMovementBase::Physicalize();
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::Reset()
{
	CVehicleMovementBase::Reset();

	m_isEnginePowered = false;
	m_engineGoalThrottle = 0.0f;

	m_isDestroyed = false;

  m_powerPID.Reset();

	m_steeringDamage = 0.0f;
	ResetActions();
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::ResetActions()
{
	m_liftAction = 0.0f;
	m_forwardAction = 0.0f;
	m_afterburnerAction = 0.0f;

	m_pitchAction = 0.0f;
	m_rollAction = 0.0f;
	m_yawAction = 0.0f;
}

//------------------------------------------------------------------------
bool CVehicleMovementHelicopter::StartEngine(EntityId driverId)
{
	if (!CVehicleMovementBase::StartEngine(driverId))
	{
		return false;
	}

	m_isEngineStarting = false;
	m_isEnginePowered = true;

  m_powerPID.Reset();

	if (m_pRotorAnim)
		m_pRotorAnim->StartAnimation();

	return true;
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::StopEngine()
{
	CVehicleMovementBase::StopEngine();
	ResetActions();
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::OnEvent(EVehicleMovementEvent event, float value)
{
	CVehicleMovementBase::OnEvent(event, value);

	if (event == eVME_DamageSteering)
	{
		float newSteeringDamage = (((float(rand()) / float(RAND_MAX)) * 2.5f ) + 0.5f);
		m_steeringDamage = max(m_steeringDamage, newSteeringDamage);
	}
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::OnAction(const char *actionName, int activationMode, float value)
{
	CVehicleMovementBase::OnAction(actionName, activationMode, value);

	if (!strcmp("rotatepitch", actionName))
		m_pitchAction += value;
	else if (!strcmp("rotateyaw", actionName))
		m_yawAction += value;
	else if (!strcmp("moveforward", actionName))
	{
		if (activationMode == eAAM_OnPress)
			m_forwardAction += value;
		else if (activationMode == eAAM_OnRelease)
			m_forwardAction = min(0.0f, m_forwardAction);
	}
	else if (!strcmp("moveback", actionName))
	{
		if (activationMode == eAAM_OnPress)
			m_forwardAction -= value;
		else if (activationMode == eAAM_OnRelease)
			m_forwardAction = max(0.0f, m_forwardAction);
	}
	else if (!strcmp("leanleft", actionName))
	{
		/*
		if (activationMode == eAAM_OnPress)
		m_rollAction -= value;
		else if (activationMode == eAAM_OnRelease)
		m_rollAction = max(0.0f, m_rollAction);
		*/
		if (activationMode == eAAM_OnHold)
			m_rollAction -= value;
	}
	else if (!strcmp("leanright", actionName))
	{
		/*
		if (activationMode == eAAM_OnPress)
		m_rollAction += value;
		else if (activationMode == eAAM_OnRelease)
		m_rollAction = min(0.0f, m_rollAction);
		*/
		if (activationMode == eAAM_OnHold)
			m_rollAction += value;
	}
	else if (!strcmp("jump", actionName))
	{
		if (activationMode == eAAM_OnPress)
			m_liftAction += value;
		else if (activationMode == eAAM_OnRelease)
			m_liftAction = min(0.0f, m_liftAction);
	}
	else if (!strcmp("crouch", actionName))
	{
		if (activationMode == eAAM_OnPress)
			m_liftAction -= value;
		else if (activationMode == eAAM_OnRelease)
			m_liftAction = max(0.0f, m_liftAction);
	}
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::ProcessActions(const float deltaTime)
{
}

//===================================================================
// ProcessAI
// This treats the helicopter as able to move in any horizontal direction
// by tilting in any direction. Yaw control is thus secondary. Throttle
// control is also secondary since it is adjusted to maintain or change
// the height, and the amount needed depends on the tilt.
//===================================================================
void CVehicleMovementHelicopter::ProcessAI(const float deltaTime)
{
	// it's useless to progress further if the engine has yet to be turned on
	if (!m_isEnginePowered)
		return;

	m_movementAction.Clear();
	m_movementAction.isAI = true;
	ResetActions();

  // Our current state
	const Vec3 worldPos = m_pEntity->GetWorldPos();
	const Ang3 worldAngles = m_pEntity->GetWorldAngles();
  const Matrix33 worldMat(m_pEntity->GetRotation());

	pe_status_dynamics	status;
	IPhysicalEntity*	pPhysics = m_pEntity->GetPhysics();
	pPhysics->GetStatus(&status);
	const Vec3 currentVel = status.v;
  const Vec3 currentVel2D(currentVel.x, currentVel.y, 0.0f);

  // Our inputs
  const float desiredSpeed = m_aiRequest.HasDesiredSpeed() ? m_aiRequest.GetDesiredSpeed() : 0.0f;
  const Vec3 desiredMoveDir = m_aiRequest.HasMoveTarget() ? (m_aiRequest.GetMoveTarget() - worldPos).GetNormalizedSafe() : ZERO;
  const Vec3 desiredMoveDir2D = Vec3(desiredMoveDir.x, desiredMoveDir.y, 0.0f).GetNormalizedSafe();
	const Vec3 desiredVel = desiredMoveDir * desiredSpeed; 
	const Vec3 desiredVel2D(desiredVel.x, desiredVel.y, 0.0f);
  const Vec3 desiredLookDir = m_aiRequest.HasLookTarget() ? (m_aiRequest.GetLookTarget() - worldPos).GetNormalizedSafe() : desiredMoveDir;
  const Vec3 desiredLookDir2D = Vec3(desiredLookDir.x, desiredLookDir.y, 0.0f).GetNormalizedSafe();

  // Calculate the desired 2D velocity change
  Vec3 desiredVelChange2D = desiredVel2D - currentVel2D;

  static float anglePerVelDifference = 0.5f;
  float desiredTiltAngle = anglePerVelDifference * desiredVelChange2D.GetLength();
  static float maxAngle = DEG2RAD(60.0f);
  Limit(desiredTiltAngle, -maxAngle, maxAngle);

  if (desiredTiltAngle > 0.001f)
  {
    const Vec3 desiredWorldTiltAxis = Vec3(-desiredVelChange2D.y, desiredVelChange2D.x, 0.0f).GetNormalizedSafe();
    const Vec3 desiredLocalTiltAxis = worldMat.GetTransposed() * desiredWorldTiltAxis;

    float desiredPitch = desiredTiltAngle * desiredLocalTiltAxis.x;
    float desiredRoll = desiredTiltAngle * desiredLocalTiltAxis.y;

    static float scalePitch = 10.0f;
    static float scaleRoll  = 1.0f;

    m_pitchAction = scalePitch * (desiredPitch - worldAngles.x);
    m_rollAction = scaleRoll * (desiredRoll - worldAngles.y);
  }

  float currentYawAngle = worldAngles.z;
  currentYawAngle += DEG2RAD(90.0f);
  float desiredYawAngle = atan2(desiredLookDir2D.y, desiredLookDir2D.x);
  float deltaYaw = (desiredYawAngle - currentYawAngle);
  if (deltaYaw > DEG2RAD(180.0f)) deltaYaw -= DEG2RAD(360.0f); else if (deltaYaw < -DEG2RAD(180.0f)) deltaYaw += DEG2RAD(360.0f);
  Limit(deltaYaw, -DEG2RAD(90.0f), DEG2RAD(90.0f));
  static float yawScale = -1.0f;
  m_yawAction = yawScale * deltaYaw;

	m_liftAction = m_powerPID.Update(currentVel.z, desiredVel.z, -1.0f, 1.0f);
}


//------------------------------------------------------------------------
void CVehicleMovementHelicopter::ProcessMovement(const float deltaTime)
{
  FUNCTION_PROFILER( GetISystem(), PROFILE_GAME );

	CVehicleMovementBase::ProcessMovement(deltaTime);

	assert(m_pEntity);

	m_netActionSync.UpdateObject( this );

	// warmup or stop the engine
	UpdateEngine(deltaTime);

	if (!m_isEnginePowered)
		return;

	m_pPhysics = m_pEntity->GetPhysics();
	assert(m_pPhysics);

	Vec3 worldPos = m_pEntity->GetWorldPos();

	pe_action_impulse control;
	Vec3& impulse = control.impulse;
	Vec3& angImpulse = control.angImpulse;

	pe_status_dynamics dyn;
	m_pPhysics->GetStatus(&dyn);
	float& mass = dyn.mass;

	pe_simulation_params paramsSim;
	m_pPhysics->GetParams( &paramsSim );
	float gravity = abs(paramsSim.gravity.z);

	// update the engine
	m_engineGoalThrottle += m_enginePower * mass * m_liftAction * deltaTime;
	m_engineGoalThrottle = min(m_enginePower * mass, max(0.0f, m_engineGoalThrottle));

	Ang3 angles = m_pEntity->GetWorldAngles();
	angles.Snap180();

	Matrix33 tm;
	tm.SetRotationXYZ((angles));

	Vec3 forward = tm * Vec3(0.0f, -1.0f , 0.0f);
	Vec3 right = tm * Vec3(1.0f, 0.0f, 0.0f);
	Vec3 up = tm * Vec3(0.0f, 0.0f, 1.0f);

	float deltaAngle = 0.0f;
	float deltaX, deltaY;

	deltaX = 0.0f - angles.x;
	deltaAngle = m_engineAngle - m_engineActualAngle;

	m_engineActualAngle += deltaAngle * min(1.0f, deltaTime * 5.0f);

	m_engineDir.x = 0;
	m_engineDir.y = cos((m_engineActualAngle));
	m_engineDir.z = sin((m_engineActualAngle));
	m_engineDir = tm * m_engineDir;

	impulse += Vec3(0.0f, 0.0f, max(0.0f, 1.0f)) * (gravity * mass);
	impulse += Vec3(0.0f, 0.0f, up.z) * mass * /*m_liftForce*/ 1.5f * m_liftAction;

	float movingSpeed = m_movingSpeed * (1.0f + m_forwardAction);
	
	impulse += Vec3(m_engineDir.x * movingSpeed, m_engineDir.y * movingSpeed, 0) 
		* m_engineGoalThrottle * m_enginePower;

	if (m_dampInertia > 0.001f)
	{
		Vec3 fwdTemp(forward.x, forward.y, 0.0f);
		Vec3 dirTemp(dyn.v.x, dyn.v.y, 0.0f);
		Vec3 dirFixed(dirTemp);

		fwdTemp.normalize();
		dirTemp.normalize();

		float dotSide = 1.0f - fabs(fwdTemp.x*dirTemp.x + fwdTemp.y*dirTemp.y + fwdTemp.z*dirTemp.z);
		Vec3 energyAbsorb = dirFixed * dyn.mass * min(dotSide * m_dampInertia, 1.0f);
		float AbsorbModule = energyAbsorb.len();

		//impulse -= fwdTemp * AbsorbModule;
		//impulse -= energyAbsorb;
	}

	if (m_dampInertiaIdle > 0.001f)
	{
		float inertiaMul = 1.0f - min(fabs(deltaX) + fabs(deltaX),1.0f);
		impulse -= Vec3(dyn.v.x,dyn.v.y,0) * dyn.mass * min(1.0f, m_dampInertiaIdle * inertiaMul);
	}

	angImpulse = right * mass * m_pitchSpeed * (-m_pitchAction * m_pitchActionMul);
	angImpulse += up * mass * m_yawSpeed * -(m_yawAction + m_steeringDamage);
	angImpulse += forward * mass * m_rollSpeed * -m_rollAction;
	//angImpulse += right * mass * deltaX * m_pitchRelax;

	if (m_rollRelax > 0.0f)
	{
		deltaY = 0.0f - angles.y;
		angImpulse -= forward * deltaY * m_rollRelax * dyn.mass;
	}

	angImpulse -= dyn.w * dyn.mass * min(1.0f, m_rotationDamping);

	impulse *= deltaTime;
	angImpulse *= deltaTime;

	control.iSource = 3;
	m_pPhysics->Action(&control);

	float heightDelta = 0.0f;
	float deltaImpulse = 0.0f;

	ICVar* profileVar = GetISystem()->GetIConsole()->GetCVar("v_profileMovement");
	if ((profileVar->GetIVal() == 1 && m_actor && m_actor->IsClient()) || profileVar->GetIVal() == 2)
	{
		IRenderer* pRenderer = GetISystem()->GetIRenderer();
		float color[4] = {1,1,1,1};

		Vec3 localAngles = m_pEntity->GetWorldAngles();

		pRenderer->Draw2dLabel(5.0f,   0.0f, 2.0f, color, false, "Helicopter movement");
		pRenderer->Draw2dLabel(5.0f,  25.0f, 1.5f, color, false, "actionPower: %f", m_liftAction);
		pRenderer->Draw2dLabel(5.0f,  40.0f, 1.5f, color, false, "actionPitch: %f", m_pitchAction);
		pRenderer->Draw2dLabel(5.0f,  55.0f, 1.5f, color, false, "actionYaw: %f", m_yawAction);
		pRenderer->Draw2dLabel(5.0f,  70.0f, 1.5f, color, false, "deltas: %f, %f", RAD2DEG(deltaX), RAD2DEG(deltaY));
		pRenderer->Draw2dLabel(5.0f,  85.0f, 1.5f, color, false, "impulse: %f, %f, %f", control.impulse.x, control.impulse.y, control.impulse.z);
		pRenderer->Draw2dLabel(5.0f, 100.0f, 1.5f, color, false, "angImpulse: %f, %f, %f", control.angImpulse.x, control.angImpulse.y, control.angImpulse.z);
		pRenderer->Draw2dLabel(5.0f, 115.0f, 1.5f, color, false, "velocity: %f, %f, %f (%f)", dyn.v.x, dyn.v.y, dyn.v.z, dyn.v.GetLength());
		pRenderer->Draw2dLabel(5.0f, 130.0f, 1.5f, color, false, "angular velocity: %f, %f, %f", dyn.w.x, dyn.w.y, dyn.w.z);
		pRenderer->Draw2dLabel(5.0f, 145.0f, 1.5f, color, false, "engine goal throttle: %f (%f)", m_engineGoalThrottle, m_engineGoalThrottle / (dyn.mass * m_enginePowerMax) );
		pRenderer->Draw2dLabel(5.0f, 160.0f, 1.5f, color, false, "angles: %f, %f, %f (%f, %f, %f)", localAngles.x, localAngles.y, localAngles.z, RAD2DEG(localAngles.x), RAD2DEG(localAngles.y), RAD2DEG(localAngles.z));
		pRenderer->Draw2dLabel(5.0f, 175.0f, 1.5f, color, false, "is stable: %i", IsStable());
		pRenderer->Draw2dLabel(5.0f, 220.0f, 1.5f, color, false, "actionRoll: %f", m_rollAction);

		Vec3 direction = m_pEntity->GetWorldTM().GetColumn(1);

		pRenderer->Draw2dLabel(5.0f, 235.0f, 1.5f, color, false, "direction: %f, %f, %f", direction.x, direction.y, direction.z);
		pRenderer->Draw2dLabel(5.0f, 250.0f, 1.5f, color, false, "up: %f, %f, %f", up.x, up.y, up.z);
	}

	if (m_netActionSync.PublishActions( CNetworkMovementHelicopter(this) ))
		m_pVehicle->GetGameObject()->ChangedNetworkState(eEA_GameClientDynamic);

	m_pitchAction = 0.0f;
	m_yawAction = 0.0f;
	m_rollAction = 0.0f;
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::UpdateEngine(float deltaTime)
{
	bool speedChanged = false;

	// will update the engine power up to the maximum according to the ignition time

	if (m_isEnginePowered && !m_isEngineGoingOff)
	{
		if (m_enginePower < m_enginePowerMax)
		{
			m_enginePower += deltaTime * (m_enginePowerMax / m_engineIgnitionTime);
			m_enginePower = min(m_enginePower, m_enginePowerMax);
			speedChanged = true;
		}
	}
	else
	{
		if (m_enginePower >= 0.0f)
		{
			float powerReduction = m_enginePowerMax / m_engineIgnitionTime;
			if (m_damage)
				powerReduction *= 2.0f;

			m_enginePower -= deltaTime * powerReduction;
			m_enginePower = max(m_enginePower, 0.0f);
			speedChanged = true;
		}
	}
}

// Lookup value from 3 point linear spline.
// (tn, vn) pairs define the spline values at certain points in time the inbetween values are linearly interpolated.
//------------------------------------------------------------------------
inline float LookupSpline( float t, float t0, float v0, float t1, float v1, float t2, float v2 )
{
	if( t < t0 )
		return v0;
	if( t > t2 )
		return t2;

	float	ts, dt, dv;

	if( t < t1 )
	{
		ts = t0;
		dt = t1 - t0;
		dv = v1 - v0;
	}
	else
	{
		ts = t1;
		dt = t2 - t1;
		dv = v2 - v1;
	}

	return v0 + (t - t0) / dt * dv;
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::Update(const float deltaTime)
{
  FUNCTION_PROFILER( GetISystem(), PROFILE_GAME );

	CVehicleMovementBase::Update(deltaTime);

	// update animation

	if (m_pLandingGearsAnim)
	{
		Vec3 worldPos = m_pVehicle->GetEntity()->GetWorldPos();
		float terrainZ = GetISystem()->GetI3DEngine()->GetTerrainZ((int)worldPos.x, (int)worldPos.y);
		float altitude = worldPos.z - terrainZ;

		if (altitude > 12)
		{
			if (m_pLandingGearsAnim->GetState() == string("opened"))
 				m_pLandingGearsAnim->ChangeState("closed");
		}
		else
		{
			if (m_pLandingGearsAnim->GetState() == string("closed"))
				m_pLandingGearsAnim->ChangeState("opened");
		}
	}

	if (m_pRotorAnim)
		m_pRotorAnim->SetSpeed(m_enginePower / m_enginePowerMax);

	if (m_enginePower && !m_isEngineGoingOff)
	{	
		if (ISound* pSound = GetSound(eSID_Run))
		{
			float rpmScale = (m_enginePower / m_enginePowerMax) * 0.7f;
			pSound->SetParam("rpm_scale", rpmScale, false);
		}		
	}

	IConsole* pConsole = GetISystem()->GetIConsole();
	if (ICVar* pVar = pConsole->GetCVar("v_invertPitchControl"))
	{
		if (pVar->GetIVal() == 1)
			m_pitchActionMul = -1.0f;
		else
			m_pitchActionMul = 1.0f;
	}
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::OnEngineCompletelyStopped()
{
	if (m_pRotorAnim)
		m_pRotorAnim->StopAnimation();
}

//------------------------------------------------------------------------
bool CVehicleMovementHelicopter::IsStable()
{
	Vec3 angles = m_pEntity->GetWorldAngles();

	if (angles.x > m_engineAngleForward || angles.x < -m_engineAngleForward)
		return false;

	if (angles.y > DEG2RAD(50) || angles.y < DEG2RAD(-50))
		return false;

	return true;
}

//------------------------------------------------------------------------
bool CVehicleMovementHelicopter::IsImmobile()
{
	pe_status_dynamics status;
	m_pEntity->GetPhysics()->GetStatus(&status);

	if (status.v.GetLengthSquared() > 1.0f)
		return false;

	if (status.w.GetLengthSquared() > 1.0f)
		return false;

	return true;
}

//------------------------------------------------------------------------
bool CVehicleMovementHelicopter::RequestMovement(CMovementRequest& movementRequest)
{
	// a minimum code for the helicopter 17/01/2006 Tetsuji

	if (!m_isEnginePowered)
		return false;

	if (movementRequest.HasLookTarget())
		m_aiRequest.SetLookTarget(movementRequest.GetLookTarget());

	if (movementRequest.HasMoveTarget())
		m_aiRequest.SetMoveTarget(movementRequest.GetMoveTarget());

	if (movementRequest.HasDesiredSpeed())
		m_aiRequest.SetDesiredSpeed(movementRequest.GetDesiredSpeed());

	return true;
}

//------------------------------------------------------------------------
void CVehicleMovementHelicopter::Serialize(TSerialize ser, unsigned aspects)
{
	if (ser.GetSerializationTarget() == eST_Network)
		m_netActionSync.Serialize(ser, aspects);
}

CNetworkMovementHelicopter::CNetworkMovementHelicopter()
{
}

CNetworkMovementHelicopter::CNetworkMovementHelicopter(CVehicleMovementHelicopter *pMovement)
{
}

void CNetworkMovementHelicopter::UpdateObject(CVehicleMovementHelicopter *pMovement)
{
}

void CNetworkMovementHelicopter::Serialize(TSerialize ser, unsigned aspects)
{
}
