/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2005.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: Implements a standard boat movement

-------------------------------------------------------------------------
History:
- 30:05:2005: Created by Michael Rauh

*************************************************************************/
#include "StdAfx.h"
#include "Game.h"

#include "IVehicleSystem.h"
#include "VehicleMovementStdBoat.h"
#include <IAgent.h>



//------------------------------------------------------------------------
CVehicleMovementStdBoat::CVehicleMovementStdBoat()
: m_velMax( 15 )
, m_accel( 5 )
, m_turnRateMax( 1 )
, m_turnAccel( 1 )
, m_cornerForceCoeff( 1 )
, m_turnAccelCoeff( 2 )
, m_accelCoeff( 2 )
, m_pushTilt( 0 )
, m_pushOffset(ZERO)
, m_cornerTilt( 0 )
, m_cornerOffset(ZERO)
, m_turnDamping( 0 )
, m_massOffset(ZERO)
{ 
  m_prevAngle = 0.0f;
}

//------------------------------------------------------------------------
CVehicleMovementStdBoat::~CVehicleMovementStdBoat()
{
}

//------------------------------------------------------------------------
bool CVehicleMovementStdBoat::Init(IVehicle* pVehicle, const SmartScriptTable &table)
{
  if (!CVehicleMovementBase::Init(pVehicle, table))
    return false;

  m_pVehicle = pVehicle;
  m_pEntity = pVehicle->GetEntity();
  m_pPhysics = m_pEntity->GetPhysics();

  table->GetValue("velMax", m_velMax);      
  table->GetValue("acceleration", m_accel);      
  table->GetValue("accelerationMultiplier", m_accelCoeff);       
  table->GetValue("pushOffset", m_pushOffset);      
  table->GetValue("pushTilt", m_pushTilt);     
  table->GetValue("turnRateMax", m_turnRateMax);
  table->GetValue("turnAccel", m_turnAccel);      
  table->GetValue("cornerForce", m_cornerForceCoeff);    
  table->GetValue("cornerOffset", m_cornerOffset);    
  table->GetValue("cornerTilt", m_cornerTilt);    
  table->GetValue("turnDamping", m_turnDamping); 
  table->GetValue("turnAccelMultiplier", m_turnAccelCoeff);   

  m_maxSpeed = m_velMax;

  // compute inertia [assumes box]
  AABB bbox;
	
	if (IVehiclePart* massPart = pVehicle->GetPart("mass"))
	{
		bbox = massPart->GetLocalBounds();
	}
	else
	{
		GameWarning("[CVehicleMovementStdBoat]: initialization: No \"mass\" geometry found!");
		m_pEntity->GetLocalBounds(bbox);
	}

	float mass = pVehicle->GetMass();

  float width = bbox.max.x - bbox.min.x;
  float length = bbox.max.y - bbox.min.y;
  float height = bbox.max.z - bbox.min.z;
  m_Inertia.x = mass * (sqr(length)+ sqr(height)) / 12;
  m_Inertia.y = mass * (sqr(width) + sqr(height)) / 12;
  m_Inertia.z = mass * (sqr(width) + sqr(length)) / 12;
  
  m_massOffset = bbox.GetCenter();
  CryLog("[StdBoat movement]: got mass offset (%f, %f, %f)", m_massOffset.x, m_massOffset.y, m_massOffset.z);


  // AI related
  // Initialise the direction PID.
  m_direction = 0.0f;
  m_dirPID.Reset();
  m_dirPID.m_kP = 0.6f;
  m_dirPID.m_kD = 0.1f;
  m_dirPID.m_kI = 0.01f;

  // Initialise the steering.
  m_steering = 0.0f;
  m_prevAngle = 0.0f;

  return true;
}

//------------------------------------------------------------------------
void CVehicleMovementStdBoat::Reset()
{
  CVehicleMovementBase::Reset();

  m_direction = 0;
  m_steering = 0;
  m_dirPID.Reset();
  m_prevAngle = 0.0f;
}

//------------------------------------------------------------------------
void CVehicleMovementStdBoat::Release()
{
  CVehicleMovementBase::Release();
  delete this;
}

//------------------------------------------------------------------------
void CVehicleMovementStdBoat::Physicalize()
{
	CVehicleMovementBase::Physicalize();

	pe_simulation_params paramsSim;
	m_pPhysics->GetParams( &paramsSim );
	m_gravity = abs(paramsSim.gravity.z);
}

//------------------------------------------------------------------------
bool CVehicleMovementStdBoat::SetParams(const SmartScriptTable &table)
{
  return true;
}

//------------------------------------------------------------------------
void CVehicleMovementStdBoat::ProcessMovement(const float deltaTime)
{  
  FUNCTION_PROFILER( GetISystem(), PROFILE_GAME );

  static const Vec3 pedalDir(FORWARD_DIRECTION);    
  static const Vec3 vecZero(ZERO);  
  static IPhysicalWorld* pWorld = GetISystem()->GetIPhysicalWorld();  
  static const float fWaterLevelMaxDiff = 0.15f; // max allowed height difference between propeller center and water level
  static const float fMinSpeedForTurn = 0.5f; // min speed so that turning becomes possible
  
  pe_status_dynamics statusDyn;

	if (!m_isEnginePowered)
		return;

  if (!m_pPhysics->GetStatus( &statusDyn )){
    CryError( "[CVehicleMovementStdBoat]: '%s' is missing physics status", m_pEntity->GetName() );
    return;
  }

  SmartScriptTable props;
  int bBraking;
  if (m_pEntity->GetScriptTable()->GetValue("Properties", props))
    if (props->GetValue("bBraking", bBraking) && bBraking > 0)
      return;

  pe_action_impulse linearImp, angularImp, dampImp; 
  const Matrix34& wTM = m_pEntity->GetWorldTM();  
  Matrix34 wTMInv = wTM.GetInvertedFast();

  float frameTime = min(deltaTime, 0.1f); 

  Vec3 localVel = wTMInv.TransformVector( statusDyn.v );
  Vec3 localW = wTMInv.TransformVector( statusDyn.w );
  
  // check if propeller is in water
  Vec3 worldPropPos = wTM.TransformPoint( m_pushOffset );  
  float fWaterLevelDiff = worldPropPos.z - GetISystem()->GetI3DEngine()->GetWaterLevel( &worldPropPos );
  
  // apply driving force     
  float a = 0;
  if (m_movementAction.power != 0 && fWaterLevelDiff < fWaterLevelMaxDiff)  
  {    
    a = m_movementAction.power * m_accel;
    if (sgn(a) * sgn(localVel.y) < 0){ // "braking"
      a *= m_accelCoeff;
    }
    if (abs(localVel.y) > m_velMax && sgn(a)*sgn(localVel.y)>0){ // check max vel
      a = 0;
    }
    Vec3 pushDir = pedalDir;
    if (a > 0){
      // apply force downwards a bit for more realistic response  
      pushDir = Quat_tpl<float>::CreateRotationAA( DEG2RAD(m_pushTilt), Vec3(-1,0,0) ) * pushDir;
    }
    if (a != 0)
    {
      pushDir = wTM.TransformVector( pushDir );  
      linearImp.impulse = pushDir * statusDyn.mass * a * frameTime;
      linearImp.point = wTM.TransformPoint( m_pushOffset );
      m_pPhysics->Action(&linearImp, 1);
    }    
  }  


  // apply steering       
  float turnAccel = 0;
  if (m_movementAction.rotateYaw != 0 && fWaterLevelDiff < fWaterLevelMaxDiff)
  {    
    Vec3 momentum(0,0,-1); // use momentum along -z to keep negative steering to the left
    
    if (abs(localVel.y) < fMinSpeedForTurn){ // if forward speed too small, no turning possible
      turnAccel = 0; 
    }
    else {
      turnAccel = m_movementAction.rotateYaw * m_turnAccel * sgnnz(m_movementAction.power);    
    }    

    // steering and current w in same direction?
    int sgnSteerW = sgn(m_movementAction.rotateYaw) * sgnnz(m_movementAction.power) * sgn(-localW.z);
    
    if (sgnSteerW < 0){ 
      turnAccel *= m_turnAccelCoeff; // "braking"      
    }
    else if (abs(localW.z) > m_turnRateMax){ 
      turnAccel = 0; // check max turn vel
    }
  }    
  else 
  { 
    // if no steering, damp rotation                
    turnAccel = localW.z * m_turnDamping;
  }
  if (turnAccel != 0)
  {
    Vec3 momentum(0,0,-1); // use momentum along -z to keep negative steering to the left
    momentum *= turnAccel * frameTime * m_Inertia.z;
    momentum = wTM.TransformVector( momentum );
    angularImp.angImpulse = momentum;
    angularImp.point = wTM.TransformPoint( m_massOffset );
    m_pPhysics->Action(&angularImp, 1);
  }
   

  // lateral force     
  if (localVel.x != 0 && fWaterLevelDiff < fWaterLevelMaxDiff)  
  {
    Vec3 cornerForce(0,0,0);
    cornerForce.x = -localVel.x * m_cornerForceCoeff * statusDyn.mass * frameTime;
    if (cornerForce.x != 0)
    {
      if (m_cornerTilt != 0)
        cornerForce = Quat_tpl<float>::CreateRotationAA( sgn(localVel.x)*DEG2RAD(m_cornerTilt), Vec3(0,1,0) ) * cornerForce;

      dampImp.impulse = wTM.TransformVector( cornerForce );

      dampImp.point = m_cornerOffset;      
      dampImp.point.x += m_massOffset.x;
      dampImp.point.y += m_massOffset.y;
      dampImp.point.x *= sgn(localVel.x);
      dampImp.point = wTM.TransformPoint( dampImp.point );
      m_pPhysics->Action(&dampImp, 1);   
    }    
  }  

  if (IsProfilingMovement())
  {
    IRenderer* pRenderer = GetISystem()->GetIRenderer();
    static float color[4] = {1,1,1,1};
    float y=50.f, step1=15.f, step2=20.f, size1=1.f, size2=1.5f;

    float speed = statusDyn.v.len();

    pRenderer->Draw2dLabel(5.0f,   y, size2, color, false, "Boat movement");
    pRenderer->Draw2dLabel(5.0f,  y+=step2, 1.5f, color, false, "Speed: %.1f (%.1f km/h)", speed, speed*3.6f);    
    pRenderer->Draw2dLabel(5.0f,  y+=step1, size1, color, false, "WaterLevel: %.2f (max: %.2f)", fWaterLevelDiff, fWaterLevelMaxDiff);

    pRenderer->Draw2dLabel(5.0f,  y+=step2, size2, color, false, "Driver input");
    pRenderer->Draw2dLabel(5.0f,  y+=step2, 1.5f, color, false, "power: %.2f", m_movementAction.power);
    pRenderer->Draw2dLabel(5.0f,  y+=step1, 1.5f, color, false, "steer: %.2f", m_movementAction.rotateYaw); 

    pRenderer->Draw2dLabel(5.0f,  y+=step2, size2, color, false, "Propelling");
    pRenderer->Draw2dLabel(5.0f,  y+=step2, 1.5f, color, false, "Impulse acc: %.2f", linearImp.impulse.len()); 
    pRenderer->Draw2dLabel(5.0f,  y+=step1, 1.5f, color, false, "Impulse steer: %.2f", angularImp.angImpulse.len()); 
    pRenderer->Draw2dLabel(5.0f,  y+=step1, 1.5f, color, false, "Impulse damp: %.2f", dampImp.impulse.len());     
  }
}

//------------------------------------------------------------------------
bool CVehicleMovementStdBoat::RequestMovement(CMovementRequest& movementRequest)
{
	SmartScriptTable stateTable;
	if (!m_pEntity->GetScriptTable()->GetValue("State", stateTable))
	{
		CryError( "Vehicle '%s' is missing 'State' LUA table. Required by AI", m_pEntity->GetName() );
		return false;
	}

	Vec3 worldPos = m_pEntity->GetWorldPos();

	float inputSpeed;
	if (movementRequest.HasDesiredSpeed())
		inputSpeed = movementRequest.GetDesiredSpeed();
	else
		inputSpeed = 0.0f;

	Vec3 moveDir;
	if (movementRequest.HasMoveTarget())
		moveDir = (movementRequest.GetMoveTarget() - worldPos).GetNormalizedSafe();
	else
		moveDir.zero();

	float	angle = 0;
	float	sideTiltAngle = 0;
	float	forwTiltAngle = 0;

	// If the movement vector is nonzero there is a target to drive at.
	if (moveDir.GetLengthSquared() > 0.01f)
	{
		SmartScriptTable movementAbilityTable;
		if (!m_pEntity->GetScriptTable()->GetValue("AIMovementAbility", movementAbilityTable))
		{
			CryError("Vehicle '%s' is missing 'AIMovementAbility' LUA table. Required by AI", m_pEntity->GetName());
			return false;
		}

		float	maxSpeed = 0.0f;
		if (!movementAbilityTable->GetValue("maxSpeed", maxSpeed))
		{
			CryError("Vehicle '%s' is missing 'MovementAbility.maxSpeed' LUA variable. Required by AI", m_pEntity->GetName());
			return false;
		}

		Matrix33 entRotMat(m_pEntity->GetRotation());
		Vec3 forwardDir = entRotMat.TransformVector(Vec3(0.0f, 1.0f, 0.0f));
		forwardDir.z = 0.0f;
		forwardDir.NormalizeSafe(Vec3Constants<float>::fVec3_OneY);
		Vec3 rightDir = forwardDir.Cross(Vec3(0.0f, 0.0f, 1.0f));

		// Normalize the movement dir so the dot product with it can be used to lookup the angle.
		Vec3 targetDir = moveDir;
		targetDir.z = 0.0f;
		targetDir.NormalizeSafe(Vec3Constants<float>::fVec3_OneX);

		/// component parallel to target dir
		float cosAngle = forwardDir.Dot(targetDir);
		Limit(cosAngle, -1.0f, 1.0f);
		angle = RAD2DEG((float) acos(cosAngle));
		if (targetDir.Dot(rightDir) < 0.0f)
			angle = -angle;

		if (!(angle < 181.0f && angle > -181.0f))
			angle = 0.0f;

		// Danny no need for PID - just need to get the proportional constant right
		// to turn angle into range 0-1. Predict the angle(=error) between now and 
		// next step - in fact over-predict to account for the lag in steering
		static float nTimesteps = 10.0f;
		static float steerConst = 0.2f;
		float predAngle = angle + nTimesteps * (angle - m_prevAngle);
		m_prevAngle = angle;
		m_steering = steerConst * predAngle;
		Limit(m_steering, -1.0f, 1.0f);

		// DEBUG: temporary hack to get the speed of the vehicle.
		pe_status_dynamics status;
		
		IPhysicalEntity* pPhysics = m_pEntity->GetPhysics();
		pPhysics->GetStatus(&status);
		Vec3 vel = status.v;
		float	speed = forwardDir.Dot(vel);

		if (!(speed < 100.0f && speed > -100.0f))
		{
			GameWarning("[CVehicleMovementStdBoat]: Bad speed from physics: %5.2f", speed);
			speed = 0.0f;
		}

		// If the desired speed is negative, it means that the maximum speed of the vehicle
		// should be used.
		float	desiredSpeed = inputSpeed;
		if (desiredSpeed > maxSpeed)
			desiredSpeed = maxSpeed;

		// Allow breaking if the error is too high.
		float	clampMin = 0.0f;
		if (abs( desiredSpeed - speed ) > desiredSpeed * 0.5f)
			clampMin = -1.0f;

		m_direction = m_dirPID.Update( speed, desiredSpeed, clampMin, 1.0f );

		stateTable->SetValue( "direction", m_direction );
		stateTable->SetValue( "steering", m_steering );
		m_movementAction.power = m_direction;
		m_movementAction.rotateYaw = m_steering;
		m_movementAction.isAI = true;
	}
	else
	{
		// let's break - we don't want to move anywhere
    m_direction = 0;
    m_steering = 0;
		stateTable->SetValue( "braking", 1 );
		stateTable->SetValue( "direction", m_direction );
		stateTable->SetValue( "steering", m_steering );
		m_movementAction.power = m_direction;
    m_movementAction.rotateYaw = m_steering;
		m_movementAction.brake = true;
	}

	return true;
}
