/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2005.
-------------------------------------------------------------------------
$Id$
$DateTime$
Description: Implements movement type for tracked vehicles

-------------------------------------------------------------------------
History:
- 13:06:2005: Created by MichaelR

*************************************************************************/
#include "StdAfx.h"
#include "Game.h"

#include "IVehicleSystem.h"
#include "VehicleMovementTank.h"

#include "VehicleSystem/Vehicle.h"
#include "VehicleSystem/VehiclePartTread.h"

CVehicleMovementTank::CVehicleMovementTank()
{
  m_pedalSpeed = 1.5f;
  m_pedalThreshold = 0.2f;
  m_steerSpeed = 0.f;
  m_steerLimit = 0.f;
  m_innerFrictionMod = 6.f;
  m_outerFrictionMod = 1.9f;
  m_minFriction = 1.f;
  m_maxFriction = 1.f;
  m_prevInnerFric = 0;
  m_prevOuterFric = 0;
  m_numWheels = 4;

  m_currPedal = 0;
  m_currSteer = 0;

  m_trackUVMaxRate = 20.f;
  m_trackUVSlices = 8;

  // AI specific
  m_bWideSteerThreshold = true;
}

CVehicleMovementTank::~CVehicleMovementTank()
{
}

bool CVehicleMovementTank::Init(IVehicle* pVehicle, const SmartScriptTable &table)
{
  if (!CVehicleMovementBase::Init(pVehicle, table))
    return false;
  
	if (!table->GetValue("Wheeled", m_wheeledTable))
		return false;

  table->GetValue("pedalSpeed", m_pedalSpeed);
  table->GetValue("pedalThreshold", m_pedalThreshold);
  table->GetValue("steerSpeed", m_steerSpeed);
  table->GetValue("steerLimit", m_steerLimit);
  table->GetValue("innerFrictionMod", m_innerFrictionMod);
  table->GetValue("outerFrictionMod", m_outerFrictionMod);

  table->GetValue("trackUVMaxRate", m_trackUVMaxRate);
  table->GetValue("trackUVSlices", m_trackUVSlices);

	/* todo: fix this
  pe_params_wheel wheelParams;
  wheelParams.iWheel = 0; // use first wheel to get friction
  m_pPhysics->GetParams(&wheelParams);
  m_minFriction = wheelParams.minFriction;
  m_maxFriction = wheelParams.maxFriction;
	*/
	m_minFriction = 1.2f;
	m_maxFriction = 1.2f;

  m_numWheels = pVehicle->GetWheelCount();
  //CryLog("[VehicleMovementTank] received number of wheels: %i", m_numWheels);

  // Initialise the direction PID.
  m_direction = 0.0f;
  m_dirPID.Reset();
  m_dirPID.m_kP = 0.6f;
  m_dirPID.m_kD = 0.1f;
  m_dirPID.m_kI = 0.01f;

  // Initialise the steering PID.
  m_steering = 0.0f;

  return true;
}

void CVehicleMovementTank::ProcessMovement(const float deltaTime)
{ 
  FUNCTION_PROFILER( GetISystem(), PROFILE_PHYSICS );

  LastMinuteInits(); // fix this (proper physicalization needed in movement init)

  if (!m_isEnginePowered)
  {
    m_action.bHandBrake = 1;
    m_action.pedal = 0;
    m_action.steer = 0;
    m_pPhysics->Action(&m_action, 1);
    return;
  }

  // tank specific:
  // full steering must be applied, otherwise there won't happen much (subject for improvement)    
  float actionSteer = m_movementAction.rotateYaw;
  
  if (m_steerSpeed == 0.f || m_movementAction.isAI)
  {
    m_currSteer =	sgn(actionSteer);
  }
  else
  { 
    m_currSteer += deltaTime * m_steerSpeed * sgn(actionSteer - m_currSteer);  
    m_currSteer = clamp_tpl(m_currSteer, -abs(actionSteer), abs(actionSteer));
  }

  if (m_steerLimit > 0.f)
    m_currSteer = clamp_tpl(m_currSteer, -m_steerLimit, m_steerLimit);
  
  // if steering, apply full throttle to have enough turn power    
  float actionPedal = (m_currSteer != 0) ? sgnnz(m_movementAction.power) : m_movementAction.power;

  pe_status_vehicle vehicleStatus;
  m_pPhysics->GetStatus(&vehicleStatus);
  int currGear = vehicleStatus.iCurGear - 1; // indexing for convenience: -1,0,1,2,..

  // pedal ramping 
  float absPedal = fabs_tpl(actionPedal);
  m_currPedal += deltaTime * m_pedalSpeed * sgn(actionPedal - m_currPedal);  
  m_currPedal = clamp_tpl(m_currPedal, -absPedal, absPedal);

  // only apply pedal after threshold is exceeded
  if (currGear == 0 && fabs_tpl(m_currPedal) < m_pedalThreshold) 
    m_action.pedal = 0;
  else
    m_action.pedal = m_currPedal;

  // reverse steering value for backward driving
  float effSteer = m_currSteer * sgnnz(actionPedal);   

  pe_status_dynamics statusDyn;
  m_pPhysics->GetStatus(&statusDyn);
  Vec3 localW( m_pEntity->GetRotation() * statusDyn.w );

  // increase friction with angular vel  	
  const static float angVelThreshold = 0.35f;    
  float innerFric = m_minFriction;
  float outerFric = innerFric;
  float fricMod = max( 0.f, fabs_tpl(localW.z) - angVelThreshold);

  // high outerFric means slower turning speed
  outerFric += m_outerFrictionMod * fricMod;

  // high innerFric increases the turning radius (especially at higher speeds)
  innerFric += m_innerFrictionMod * fricMod; 

  if (innerFric != m_prevInnerFric || outerFric != m_prevOuterFric)
  {
    pe_params_wheel wheelParams;  
    for (int i=0; i<m_numWheels; ++i)
    {    
      wheelParams.iWheel = i;
      bool bLeft = (i < 0.5f*m_numWheels); // left side?
      if ((bLeft && effSteer < 0) || (!bLeft && effSteer > 0 ))
      {
        wheelParams.minFriction = innerFric;
      }
      else 
      {
        wheelParams.minFriction = outerFric;
      }
      wheelParams.maxFriction = wheelParams.minFriction;
      m_pPhysics->SetParams(&wheelParams);
    }
    m_prevInnerFric = innerFric;
    m_prevOuterFric = outerFric;
  } 

  const static float maxSteer = gf_PI/4.f; // fix maxsteer, shouldn't change  
  m_action.steer = m_currSteer * maxSteer;

  m_action.bHandBrake = (m_movementAction.brake) ? 1 : 0;

  m_pPhysics->Action(&m_action, 1);

  // debug info
  if (IsProfilingMovement())
  {
    static IRenderer* pRenderer = GetISystem()->GetIRenderer();
    static float color[4] = {1,1,1,1};
    static const float step = 15.f, step2 = 20.f;

    float speed = vehicleStatus.vel.len();
    float y = 40.f;    

    pRenderer->Draw2dLabel(5.0f,  y,       2.0f, color, false, "Tank status");
    pRenderer->Draw2dLabel(5.0f,  y+=step2,1.5f, color, false, "Speed: %.1f (%.1f km/h)", speed, speed*3.6f);
    pRenderer->Draw2dLabel(5.0f,  y+=step, 1.5f, color, false, "RPM:   %i", (int)vehicleStatus.engineRPM); 
    pRenderer->Draw2dLabel(5.0f,  y+=step, 1.5f, color, false, "Gear:  %i", vehicleStatus.iCurGear-1);
    pRenderer->Draw2dLabel(5.0f,  y+=step, 1.5f, color, false, "Clutch:  %.2f", vehicleStatus.clutch);

    pRenderer->Draw2dLabel(5.0f,  y+=step2, 2.0f, color, false, "Driver input");
    pRenderer->Draw2dLabel(5.0f,  y+=step2, 1.5f, color, false, "power: %.2f", m_movementAction.power);
    pRenderer->Draw2dLabel(5.0f,  y+=step, 1.5f, color, false, "steer: %.2f", m_movementAction.rotateYaw); 
    pRenderer->Draw2dLabel(5.0f,  y+=step, 1.5f, color, false, "brake: %i", m_movementAction.brake);

    pRenderer->Draw2dLabel(5.0f,  y+=step2, 2.0f, color, false, "Tank action");
    pRenderer->Draw2dLabel(5.0f,  y+=step2, 1.5f, color, false, "pedal: %.2f", m_action.pedal);
    pRenderer->Draw2dLabel(5.0f,  y+=step, 1.5f, color, false, "steer: %.2f", m_action.steer); 
    pRenderer->Draw2dLabel(5.0f,  y+=step, 1.5f, color, false, "brake: %i", m_action.bHandBrake);    
  }

}

//------------------------------------------------------------------------
bool CVehicleMovementTank::RequestMovement(CMovementRequest& movementRequest)
{
	if (!CVehicleMovementStdWheeled::RequestMovement(movementRequest))
	{
		return false;
	}

  // tank seems to have reversed control when requesting reversing
  if (m_movementAction.isAI && m_movementAction.power < 0.0f)
    m_movementAction.rotateYaw *= -1.0f;

	// [mikko]: use hysterisis to control the steering.
	// Decrease this threshold too much and the tank is always correcting. Increase
	// it too much and hit deviates from its path and hits trees etc.
	const static float wideThreshold = 0.12f;
	const static float smallThreshold = 0.08f;

	float threshold = (m_bWideSteerThreshold) ? wideThreshold : smallThreshold;

	if (fabs_tpl(m_movementAction.rotateYaw) < threshold)
	{
		// inside the threshold
		m_movementAction.rotateYaw = 0.0f;
		m_bWideSteerThreshold = true;	    
	}
	else
	{
		// outside the threshold. if the wide steering threshold is exceed switch to smaller one.	    
		m_bWideSteerThreshold = false;	    
	}

	return true;
}

//------------------------------------------------------------------------
void CVehicleMovementTank::Update(const float deltaTime)
{
  CVehicleMovementStdWheeled::Update(deltaTime); 
}

void CVehicleMovementTank::UpdateSounds(const float deltaTime)
{

}


//------------------------------------------------------------------------
void CVehicleMovementTank::StopEngine()
{
  CVehicleMovementStdWheeled::StopEngine();

  // stop treads (this is temporary fix, the vehicle needs updating as long as its moving)
  int nParts = m_pVehicle->GetPartCount();
  IVehiclePart** parts = new IVehiclePart*[nParts];
  m_pVehicle->GetParts(parts, nParts);
  
  for (int i=0; i<nParts; ++i)
  {      
    if (parts[i]->GetType() == IVehiclePart::eVPT_TREAD)
    {
      CVehiclePartTread* pTread = (CVehiclePartTread*)(parts[i]);
      pTread->SetUVSpeed(0.f);
    } 
  }   
  delete[] parts;
}
 
