////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2005.
// -------------------------------------------------------------------------
//  File name:   TimeOfDayDialog.cpp
//  Version:     v1.00
//  Created:     12/3/2002 by Timur.
//  Compilers:   Visual C++ 7.0
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "TimeOfDayDialog.h"
#include "Controls/SplineCtrl.h"
#include "Controls/ColorGradientCtrl.h"
#include "NumberDlg.h"
#include "Mission.h"
#include "CryEditDoc.h"
#include "IViewPane.h"

#include <ITimer.h>
#include <I3DEngine.h>

#define IDC_TASKPANEL 1
#define IDC_SLIDERSPANEL 2
#define IDC_EXPRESSION_WEIGHT_SLIDER 3
#define IDC_TIME_OF_DAY 5
#define IDC_TIME_OF_DAY_START 6
#define IDC_TIME_OF_DAY_END 7
#define IDC_TIME_OF_DAY_SPEED 8
#define IDC_SKYUPDATE_CHECKBOX 11

#define SLIDER_SCALE 100

#define FIRST_SLIDER_ID 1000

#define ID_EXPAND_ALL_SPLINES 1000
#define ID_COLLAPSE_ALL_SPLINES 1001
#define ID_PLAY_TOD_ANIM 1002
#define ID_STOP_TOD_ANIM 1003
#define ID_IMPORT_SEUQENCE 1004


IMPLEMENT_DYNCREATE(CTimeOfDayDialog,CToolbarDialog)

//////////////////////////////////////////////////////////////////////////
class CTimeOfDayViewClass : public TRefCountBase<IViewPaneClass>
{
	//////////////////////////////////////////////////////////////////////////
	// IClassDesc
	//////////////////////////////////////////////////////////////////////////
	virtual ESystemClassID SystemClassID() { return ESYSTEM_CLASS_VIEWPANE; };
	virtual REFGUID ClassID()
	{
		// {85FB1272-D858-4ca5-ABB4-04D484ABF51E}
		static const GUID guid = { 0x85fb1272, 0xd858, 0x4ca5, { 0xab, 0xb4, 0x4, 0xd4, 0x84, 0xab, 0xf5, 0x1e } };
		return guid;
	}
	virtual const char* ClassName() { return "Time Of Day"; };
	virtual const char* Category() { return "Time Of Day"; };
	//////////////////////////////////////////////////////////////////////////
	virtual CRuntimeClass* GetRuntimeClass() { return RUNTIME_CLASS(CTimeOfDayDialog); };
	virtual const char* GetPaneTitle() { return _T("Time Of Day"); };
	virtual EDockingDirection GetDockingDirection() { return DOCK_FLOAT; };
	virtual CRect GetPaneRect() { return CRect(100,100,1000,800); };
	virtual bool SinglePane() { return false; };
	virtual bool WantIdleUpdate() { return true; };
};

CTimeOfDayDialog* CTimeOfDayDialog::m_pInstance = 0;

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::RegisterViewClass()
{
	GetIEditor()->GetClassFactory()->RegisterClass( new CTimeOfDayViewClass );
}

//////////////////////////////////////////////////////////////////////////
class CSplineCtrlContainerTOD : public CSplineCtrl
{
public:
	virtual void PostNcDestroy() { delete this; };
};

//////////////////////////////////////////////////////////////////////////
BEGIN_MESSAGE_MAP(CTimeOfDayDialog, CToolbarDialog)
	ON_WM_SIZE()
	ON_WM_CLOSE()
	ON_WM_HSCROLL()
	ON_NOTIFY_RANGE( NM_RCLICK,FIRST_SLIDER_ID,FIRST_SLIDER_ID+1000,OnSplineRClick )
	ON_NOTIFY_RANGE( SPLN_BEFORE_CHANGE,FIRST_SLIDER_ID,FIRST_SLIDER_ID+1000,OnBeforeSplineChange )
	ON_NOTIFY_RANGE( SPLN_CHANGE,FIRST_SLIDER_ID,FIRST_SLIDER_ID+1000,OnSplineChange )
	ON_NOTIFY_RANGE( CLRGRDN_BEFORE_CHANGE,FIRST_SLIDER_ID,FIRST_SLIDER_ID+1000,OnBeforeSplineChange )
	ON_NOTIFY_RANGE( CLRGRDN_CHANGE,FIRST_SLIDER_ID,FIRST_SLIDER_ID+1000,OnSplineChange )
	ON_COMMAND(ID_PLAY,OnPlayAnim)
	ON_UPDATE_COMMAND_UI(ID_PLAY,OnUpdatePlayAnim)
	ON_COMMAND(ID_RECORD,OnRecord)
	ON_UPDATE_COMMAND_UI(ID_RECORD,OnUpdateRecord)
	ON_COMMAND(ID_FACEIT_PLAY_FROM_0,OnPlayAnimFrom0)
	ON_UPDATE_COMMAND_UI(ID_FACEIT_PLAY_FROM_0,OnUpdatePlayAnimFrom0)
	ON_COMMAND(ID_FACEIT_GOTO_MINUS1,OnGotoMinus1)
	ON_COMMAND(ID_FACEIT_GOTO_0,OnGoto0)
	ON_COMMAND(ID_FACEIT_GOTO_1,OnGoto1)
	ON_MESSAGE(XTPWM_TASKPANEL_NOTIFY, OnTaskPanelNotify)
	ON_CONTROL_RANGE(TIMEOFDAYN_CHANGE, IDC_TIME_OF_DAY,IDC_TIME_OF_DAY_END, OnChangeCurrentTime)
	ON_EN_CHANGE(IDC_TIME_OF_DAY_SPEED,OnChangeTimeAnimSpeed)
END_MESSAGE_MAP()

/** Undo object stored when track is modified.
*/
class CUndoTimeOfDayObject : public IUndoObject
{
public:
	CUndoTimeOfDayObject()
	{
		m_undo = CreateXmlNode("Undo");
		m_redo = CreateXmlNode("Redo");
		GetIEditor()->Get3DEngine()->GetTimeOfDay()->Serialize( m_undo,false );
	}
protected:
	virtual int GetSize() { return sizeof(*this); }
	virtual const char* GetDescription() { return "Time Of Day"; };

	virtual void Undo( bool bUndo )
	{
		if (bUndo)
		{
			GetIEditor()->Get3DEngine()->GetTimeOfDay()->Serialize( m_redo,false );
		}
		GetIEditor()->Get3DEngine()->GetTimeOfDay()->Serialize( m_undo,true );
		CTimeOfDayDialog::UpdateValues();
	}
	virtual void Redo()
	{
		GetIEditor()->Get3DEngine()->GetTimeOfDay()->Serialize( m_redo,true );
		CTimeOfDayDialog::UpdateValues();
	}

private:
	XmlNodeRef m_undo;
	XmlNodeRef m_redo;
};

//////////////////////////////////////////////////////////////////////////
CTimeOfDayDialog::CTimeOfDayDialog()
: CToolbarDialog(IDD,NULL)
{
	m_pInstance = this;
	m_bAnimation = false;
	m_bPlayFrom0 = false;
	m_bIgnoreScroll = false;
	m_bIgnoreSetTimeToEngine = false;
	m_splineCtrlHeight = 80;
	m_bRecording = false;
	GetIEditor()->RegisterNotifyListener( this );
	Create( CTimeOfDayDialog::IDD,AfxGetMainWnd() );
}

//////////////////////////////////////////////////////////////////////////
CTimeOfDayDialog::~CTimeOfDayDialog()
{
	m_pInstance = 0;
	GetIEditor()->UnregisterNotifyListener( this );
	ClearCtrls();
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::PostNcDestroy()
{
	delete this;
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnClose()
{
	SavePlacement( "Dialogs\\TimeOfDay" );
	DestroyWindow();
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnSize(UINT nType, int cx, int cy)
{
	__super::OnSize(nType,cx,cy);

	RecalcLayout();
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::UpdateValues()
{
	if (!m_pInstance)
		return;

	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
	ITimeOfDay::SAdvancedInfo advInfo;
	pTimeOfDay->GetAdvancedInfo( advInfo );

	m_pInstance->SetTimeRange( advInfo.fStartTime,advInfo.fEndTime,advInfo.fAnimSpeed );
	m_pInstance->SetTime( m_pInstance->GetTime(),true,true );
	m_pInstance->RedrawWindow( NULL,NULL,RDW_INVALIDATE|RDW_ERASE|RDW_ALLCHILDREN);
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::RecalcLayout()
{
	if (m_wndSplinesTaskPanel.m_hWnd)
	{
		CRect rc;
		GetClientRect(rc);

		DWORD dwMode = LM_HORZ|LM_HORZDOCK|LM_STRETCH|LM_COMMIT;
		CSize sz = m_wndToolBar.CalcDockingLayout(32000, dwMode);

		CRect rcToolbar(rc);
		rcToolbar.bottom = sz.cy;
		m_wndToolBar.MoveWindow(rcToolbar);
		rc.top = rcToolbar.bottom+1;

		rc.right -= 300;

		CRect rcTask(rc);
		rcTask.right = 180;
		rc.left = rcTask.right+1;
		m_wndTaskPanel.MoveWindow(rcTask);

		CRect rcsldr(rc);
		int sh = 20;
		m_weightSlider.MoveWindow( CRect(rcsldr.left,rcsldr.top+10,rcsldr.right,rcsldr.top+10+sh) );
		m_textMinus100.MoveWindow( CRect(rcsldr.left+10,rcsldr.top+32,rcsldr.left+100,rcsldr.top+45) );
		m_textPlus100.MoveWindow( CRect(rcsldr.right-100,rcsldr.top+32,rcsldr.right-10,rcsldr.top+45) );
		m_text0.MoveWindow( CRect((rcsldr.right+rcsldr.left)/2-4,rcsldr.top+32,(rcsldr.right+rcsldr.left)/2+40,rcsldr.top+45) );

		rc.top += 50;
		m_wndSplinesTaskPanel.MoveWindow(rc);

		GetClientRect(rc);
		rc.top = rcToolbar.bottom+1;
		rc.left = rc.right - 299;
		m_propsCtrl.MoveWindow(rc);
	}
}

//////////////////////////////////////////////////////////////////////////
BOOL CTimeOfDayDialog::OnInitDialog()
{
	BOOL res = __super::OnInitDialog();

	CRect rc;
	GetClientRect(rc);

	//////////////////////////////////////////////////////////////////////////
	VERIFY(m_wndToolBar.CreateToolBar(WS_VISIBLE|WS_CHILD|CBRS_TOOLTIPS|CBRS_ALIGN_TOP, this, AFX_IDW_TOOLBAR));
	VERIFY(m_wndToolBar.LoadToolBar(IDR_TIME_OF_DAY_BAR));
	m_wndToolBar.SetFlags(xtpFlagAlignTop|xtpFlagStretched);
	m_wndToolBar.EnableCustomization(FALSE);

	//VERIFY(m_wndToolBar.Create(this,WS_VISIBLE|WS_CHILD|CBRS_TOOLTIPS,AFX_IDW_TOOLBAR));
	//VERIFY(m_wndToolBar.LoadToolBar(IDR_AVI_RECORDER_BAR));
	//m_wndToolBar.SetFlags(xtpFlagStretched);

	m_weightSlider.Create( WS_CHILD|WS_VISIBLE|TBS_BOTH|TBS_AUTOTICKS|TBS_NOTICKS,CRect(0,0,1,1),this,IDC_EXPRESSION_WEIGHT_SLIDER );
	m_weightSlider.SetRange(0,24*SLIDER_SCALE);
	m_weightSlider.SetTicFreq( 1*SLIDER_SCALE );

	m_textMinus100.Create( "0",WS_CHILD|WS_VISIBLE,rc,this,IDC_STATIC );
	m_textMinus100.SetFont( CFont::FromHandle( (HFONT)gSettings.gui.hSystemFont) );
	m_textPlus100.Create( "24",WS_CHILD|WS_VISIBLE|SS_RIGHT,rc,this,IDC_STATIC );
	m_textPlus100.SetFont( CFont::FromHandle( (HFONT)gSettings.gui.hSystemFont) );
	m_text0.Create( "12",WS_CHILD|WS_VISIBLE,rc,this,IDC_STATIC );
	m_text0.SetFont( CFont::FromHandle( (HFONT)gSettings.gui.hSystemFont) );

	//////////////////////////////////////////////////////////////////////////
	m_wndSplinesTaskPanel.Create( WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|WS_CLIPCHILDREN,rc,this,IDC_SLIDERSPANEL );
	m_wndSplinesTaskPanel.ModifyStyleEx( 0,WS_EX_CLIENTEDGE );

	m_wndSplinesTaskPanel.SetBehaviour(xtpTaskPanelBehaviourExplorer);
//	m_wndSplinesTaskPanel.SetTheme(xtpTaskPanelThemeListViewOffice2003);
	m_wndSplinesTaskPanel.SetTheme(xtpTaskPanelThemeToolboxWhidbey);

	//m_wndSplinesTaskPanel.SetHotTrackStyle(xtpTaskPanelHighlightItem);
	//m_wndSplinesTaskPanel.SetSelectItemOnFocus(TRUE);
	m_wndSplinesTaskPanel.AllowDrag(FALSE);

	m_wndSplinesTaskPanel.SetAnimation(xtpTaskPanelAnimationNo);

	m_wndSplinesTaskPanel.GetPaintManager()->m_rcGroupOuterMargins.SetRect(0,0,0,0);
	m_wndSplinesTaskPanel.GetPaintManager()->m_rcGroupInnerMargins.SetRect(0,0,0,0);
	m_wndSplinesTaskPanel.GetPaintManager()->m_rcItemOuterMargins.SetRect(0,0,0,0);
	m_wndSplinesTaskPanel.GetPaintManager()->m_rcItemInnerMargins.SetRect(0,0,0,0);
	m_wndSplinesTaskPanel.GetPaintManager()->m_rcControlMargins.SetRect(0,0,0,0);
	m_wndSplinesTaskPanel.GetPaintManager()->m_nGroupSpacing = 0;

	CMFCUtils::LoadTrueColorImageList( m_imageList,IDB_FACED_LIBTREE,16,RGB(255,0,255) );
	m_wndSplinesTaskPanel.SetGroupImageList( &m_imageList,CSize(16,16) );
	//////////////////////////////////////////////////////////////////////////

	{
		//////////////////////////////////////////////////////////////////////////
		//////////////////////////////////////////////////////////////////////////
		// Task panel.
		//////////////////////////////////////////////////////////////////////////
		m_wndTaskPanel.Create( WS_CHILD|WS_BORDER|WS_VISIBLE|WS_CLIPSIBLINGS|WS_CLIPCHILDREN,rc,this,IDC_TASKPANEL );

		//m_taskImageList.Create( IDR_DB_GAMETOKENS_BAR,16,1,RGB(192,192,192) );
		//VERIFY( CMFCUtils::LoadTrueColorImageList( m_taskImageList,IDR_DB_GAMETOKENS_BAR,15,RGB(192,192,192) ) );
		//m_wndTaskPanel.SetImageList( &m_taskImageList );
		m_wndTaskPanel.SetBehaviour(xtpTaskPanelBehaviourExplorer);
		m_wndTaskPanel.SetTheme(xtpTaskPanelThemeNativeWinXP);
		m_wndTaskPanel.SetSelectItemOnFocus(TRUE);
		m_wndTaskPanel.AllowDrag(TRUE);

		/*
		m_wndTaskPanel.GetPaintManager()->m_rcGroupOuterMargins.SetRect(1,3,1,3);
		m_wndTaskPanel.GetPaintManager()->m_rcGroupInnerMargins.SetRect(4,4,4,4);
		m_wndTaskPanel.GetPaintManager()->m_rcItemOuterMargins.SetRect(1,1,1,1);
		m_wndTaskPanel.GetPaintManager()->m_rcItemInnerMargins.SetRect(0,0,0,0);
		m_wndTaskPanel.GetPaintManager()->m_rcControlMargins.SetRect(2,0,2,0);
		m_wndTaskPanel.GetPaintManager()->m_nGroupSpacing = 3;
		*/

		// Add default tasks.
		CXTPTaskPanelGroupItem *pItem =  NULL;
		CXTPTaskPanelGroup *pGroup = NULL;
		{
			pGroup = m_wndTaskPanel.AddGroup(1);
			pGroup->SetCaption( "Time of Day Tasks" );

			pItem =  pGroup->AddLinkItem(ID_FILE_IMPORT); pItem->SetType(xtpTaskItemTypeLink);
			pItem->SetCaption( "Import From File" );
			pItem =  pGroup->AddLinkItem(ID_FILE_EXPORT); pItem->SetType(xtpTaskItemTypeLink);
			pItem->SetCaption( "Export To File" );

			pItem =  pGroup->AddLinkItem(ID_IMPORT_SEUQENCE); pItem->SetType(xtpTaskItemTypeLink);
			pItem->SetCaption( "Import Old Sequence" );

			pItem =  pGroup->AddLinkItem(ID_EXPAND_ALL_SPLINES); pItem->SetType(xtpTaskItemTypeLink);
			pItem->SetCaption( "Expand All" );
			pItem =  pGroup->AddLinkItem(ID_COLLAPSE_ALL_SPLINES); pItem->SetType(xtpTaskItemTypeLink);
			pItem->SetCaption( "Collapse All" );
		}

		m_timeEdit.Create( WS_CHILD|WS_VISIBLE|ES_CENTER,CRect(0,0,200,16),this,IDC_TIME_OF_DAY );
		m_timeEdit.SetFont( CFont::FromHandle((HFONT)gSettings.gui.hSystemFont) );
		m_timeEdit.SetParent(&m_wndTaskPanel);
		m_timeEdit.SetOwner(this);

		m_timeEditStart.Create( WS_CHILD|WS_VISIBLE|ES_CENTER,CRect(0,0,200,16),this,IDC_TIME_OF_DAY_START );
		m_timeEditStart.SetFont( CFont::FromHandle((HFONT)gSettings.gui.hSystemFont) );
		m_timeEditStart.SetParent(&m_wndTaskPanel);
		m_timeEditStart.SetOwner(this);

		m_timeEditEnd.Create( WS_CHILD|WS_VISIBLE|ES_CENTER,CRect(0,0,200,16),this,IDC_TIME_OF_DAY_END );
		m_timeEditEnd.SetFont( CFont::FromHandle((HFONT)gSettings.gui.hSystemFont) );
		m_timeEditEnd.SetParent(&m_wndTaskPanel);
		m_timeEditEnd.SetOwner(this);

		m_timeAnimSpeedCtrl.Create( this,CRect(0,0,200,18),IDC_TIME_OF_DAY_SPEED,CNumberCtrl::NOBORDER|CNumberCtrl::CENTER_ALIGN );
		m_timeAnimSpeedCtrl.SetParent(&m_wndTaskPanel);
		m_timeAnimSpeedCtrl.SetOwner(this);
		{
			pGroup = m_wndTaskPanel.AddGroup(2);
			pGroup->SetCaption( "Current Time" );

			pItem =  pGroup->AddControlItem(m_timeEdit);
			pItem->SetCaption( "Time: " );

			pItem =  pGroup->AddTextItem(_T("Start Time:"));
			pItem =  pGroup->AddControlItem(m_timeEditStart);
			pItem =  pGroup->AddTextItem(_T("End Time:"));
			pItem =  pGroup->AddControlItem(m_timeEditEnd);
			pItem =  pGroup->AddTextItem(_T("Play Speed:"));
			pItem =  pGroup->AddControlItem(m_timeAnimSpeedCtrl);
		}

		m_forceUpdateCheck.Create( "Force sky update",WS_CHILD|WS_VISIBLE|BS_AUTOCHECKBOX,CRect(0,0,60,16),&m_wndTaskPanel,IDC_SKYUPDATE_CHECKBOX );
		m_forceUpdateCheck.SetFont( CFont::FromHandle( (HFONT)gSettings.gui.hSystemFont) );
		{
			pGroup = m_wndTaskPanel.AddGroup(3);
			pGroup->SetCaption( "Update Tasks" );

			pItem =  pGroup->AddLinkItem(ID_PLAY_TOD_ANIM); pItem->SetType(xtpTaskItemTypeLink);
			pItem->SetCaption( "Play" );
			pItem =  pGroup->AddLinkItem(ID_STOP_TOD_ANIM); pItem->SetType(xtpTaskItemTypeLink);
			pItem->SetCaption( "Stop" );
			pItem =  pGroup->AddControlItem(m_forceUpdateCheck);
		}

		//////////////////////////////////////////////////////////////////////////
	}

	m_propsCtrl.Create( WS_CHILD|WS_VISIBLE,CRect(0,0,100,100),this );
	m_propsCtrl.ModifyStyleEx( 0,WS_EX_CLIENTEDGE );
	m_propsCtrl.SetUpdateCallback( functor(*this,&CTimeOfDayDialog::OnUpdateProperties) );

	RecalcLayout();
	ReloadCtrls();

	UpdateValues();
	SetTime( GetTime(),true );

	AutoLoadPlacement( "Dialogs\\TimeOfDay" );

	return res;
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::ClearCtrls()
{
	if (m_wndSplinesTaskPanel.m_hWnd)
		m_wndSplinesTaskPanel.GetGroups()->Clear();
	m_controllers.clear();
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::ReloadCtrls()
{
	m_wndSplinesTaskPanel.SetLockRedraw(TRUE);
	//m_wndSplinesTaskPanel.SetRedraw(FALSE);
	ClearCtrls();

	m_propsCtrl.DeleteAllItems();

	m_pVars = new CVarBlock;

	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
	int nCommand = 0;
	for (int i = 0; i < pTimeOfDay->GetVariableCount(); i++)
	{
		int nImage = 0;

		ITimeOfDay::SVariableInfo varInfo;
		if (!pTimeOfDay->GetVariableInfo( i,varInfo ))
			continue;

		{
			if (!varInfo.pInterpolator)
				continue;

			nCommand++;
			CXTPTaskPanelGroup* pFolder = m_wndSplinesTaskPanel.AddGroup(nCommand,nImage);
			pFolder->SetCaption( varInfo.name );
			
			pFolder->SetItemLayout(xtpTaskItemLayoutDefault);
			pFolder->SetExpanded(FALSE);

			if (varInfo.type == ITimeOfDay::TYPE_COLOR)
			{
				CRect sliderRc(0,0,1,40);
				CColorGradientCtrl *pColorCtrl = new CColorGradientCtrl;
				pColorCtrl->Create( WS_CHILD|WS_BORDER,sliderRc,this,FIRST_SLIDER_ID+nCommand );
				pColorCtrl->ModifyStyleEx( 0,WS_EX_CLIENTEDGE );
				pColorCtrl->SetParent( &m_wndSplinesTaskPanel );
				pColorCtrl->SetOwner( this );

				pColorCtrl->SetTimeRange( 0,1.0f );
				pColorCtrl->LockFirstAndLastKeys(true);
				pColorCtrl->SetTooltipValueScale(24,1);

				pColorCtrl->SetSpline( varInfo.pInterpolator );

				CXTPTaskPanelGroupItem *pItem =  pFolder->AddControlItem( *pColorCtrl );
				pItem->GetMargins().SetRect(0,0,0,0);

				pColorCtrl->MoveWindow( sliderRc );
				m_wndSplinesTaskPanel.ExpandGroup( pFolder,TRUE );
				pColorCtrl->ShowWindow( SW_SHOW );

				//////////////////////////////////////////////////////////////////////////
				// Add Var.
				IVariable *pVar = new CVariable<Vec3>;
				pVar->SetDataType( IVariable::DT_COLOR );
				pVar->SetName( varInfo.name );
				pVar->Set( Vec3(varInfo.fValue[0],varInfo.fValue[1],varInfo.fValue[2]) );
				m_pVars->AddVariable( pVar );

				ControllerInfo ci;
				ci.pVariable = pVar;
				ci.pColorCtrl = pColorCtrl;
				m_controllers.push_back(ci);
			}
			else if (varInfo.type == ITimeOfDay::TYPE_FLOAT)
			{
				CRect sliderRc(0,0,1,m_splineCtrlHeight);
				CSplineCtrlContainerTOD *pSplineCtrl = new CSplineCtrlContainerTOD;
				pSplineCtrl->Create( WS_CHILD|WS_BORDER,sliderRc,this,FIRST_SLIDER_ID+nCommand );
				pSplineCtrl->ModifyStyleEx( 0,WS_EX_CLIENTEDGE );
				pSplineCtrl->SetParent( &m_wndSplinesTaskPanel );
				pSplineCtrl->SetOwner( this );

				pSplineCtrl->SetGrid( 24,10 );
				pSplineCtrl->SetTimeRange( 0,1.0f );
				pSplineCtrl->LockFirstAndLastKeys(true);

				pSplineCtrl->SetValueRange( varInfo.fValue[1],varInfo.fValue[2] );
				pSplineCtrl->SetTooltipValueScale(24,1);

				pSplineCtrl->SetSpline( varInfo.pInterpolator );

				CXTPTaskPanelGroupItem *pItem =  pFolder->AddControlItem( *pSplineCtrl );
				pItem->GetMargins().SetRect(0,0,0,0);

				pSplineCtrl->MoveWindow( sliderRc );
				m_wndSplinesTaskPanel.ExpandGroup( pFolder,TRUE );
				pSplineCtrl->ShowWindow( SW_SHOW );

				// Add Var.
				IVariable *pVar = new CVariable<float>;
				pVar->SetName( varInfo.name );
				pVar->Set( varInfo.fValue[0] );
				pVar->SetLimits( varInfo.fValue[1],varInfo.fValue[2] );
				m_pVars->AddVariable( pVar );

				ControllerInfo ci;
				ci.pVariable = pVar;
				ci.pSplineCtrl = pSplineCtrl;
				m_controllers.push_back(ci);
			}
		}
	}
	m_wndSplinesTaskPanel.SetLockRedraw(FALSE);
	//m_wndSplinesTaskPanel.SetRedraw(TRUE);

	m_propsCtrl.AddVarBlock( m_pVars );
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnHScroll( UINT nSBCode,UINT nPos,CScrollBar* pScrollBar )
{
	if (m_bIgnoreScroll)
		return;
	if (pScrollBar == (CScrollBar*)&m_weightSlider)
	{
		float fWeight = (float)m_weightSlider.GetPos()/SLIDER_SCALE;
		m_bIgnoreScroll = true;
		SetTime( fWeight );
		m_bIgnoreScroll = false;
	}
}
#define ID_CHANGE_CONTROL_HEIGHT 10

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnSplineRClick( UINT nID, NMHDR* pNMHDR, LRESULT* lpResult )
{
	if (nID >= FIRST_SLIDER_ID && nID <= FIRST_SLIDER_ID+1000)
	{
		CWnd *pWnd = CWnd::FromHandle(pNMHDR->hwndFrom);
		if (pWnd->IsKindOf(RUNTIME_CLASS(CSplineCtrl)))
		{
			CSplineCtrl *pSlineCtrl = (CSplineCtrl*)pWnd;

			//for (int i = 0; i < (int)m_controllers.size(); i++)
			{
				//if (m_controllers[i].pSplineCtrl == pSlineCtrl)
				{
					CMenu menu;
					VERIFY( menu.CreatePopupMenu() );

					// create main menu items
					menu.AppendMenu( MF_STRING,ID_EXPAND_ALL_SPLINES, _T("Expand All") );
					menu.AppendMenu( MF_STRING,ID_COLLAPSE_ALL_SPLINES, _T("Collapse All") );
					menu.AppendMenu( MF_STRING,ID_CHANGE_CONTROL_HEIGHT, _T("Change Control Height") );

					CPoint pos;
					GetCursorPos(&pos);
					int cmd = menu.TrackPopupMenu( TPM_RETURNCMD|TPM_LEFTALIGN|TPM_LEFTBUTTON|TPM_NONOTIFY,pos.x,pos.y,this );
					switch (cmd)
					{
					case ID_EXPAND_ALL_SPLINES:
						OnExpandAll();
						break;
					case ID_COLLAPSE_ALL_SPLINES:
						OnCollapseAll();
						break;
					case ID_CHANGE_CONTROL_HEIGHT:
						{
							CNumberDlg dlg( this,m_splineCtrlHeight,"Enter New Height" );
							dlg.SetInteger(true);
							if (dlg.DoModal() == IDOK)
							{
								if (dlg.GetValue() > 10 && dlg.GetValue() < 500)
									m_splineCtrlHeight = dlg.GetValue();
								ReloadCtrls();
								return;
							}
						}
						break;
					}
					//break;
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnBeforeSplineChange( UINT nID, NMHDR* pNMHDR, LRESULT* lpResult )
{
	if (nID >= FIRST_SLIDER_ID && nID <= FIRST_SLIDER_ID+1000)
	{
		if (CUndo::IsRecording())
			CUndo::Record( new CUndoTimeOfDayObject );

		CWnd *pWnd = CWnd::FromHandle(pNMHDR->hwndFrom);
		if (pWnd->IsKindOf(RUNTIME_CLASS(CSplineCtrl)))
		{
			CSplineCtrl *pSlineCtrl = (CSplineCtrl*)pWnd;

			for (int i = 0; i < (int)m_controllers.size(); i++)
			{
				if (m_controllers[i].pSplineCtrl == pSlineCtrl)
				{
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnSplineChange( UINT nID, NMHDR* pNMHDR, LRESULT* lpResult )
{
	if (nID >= FIRST_SLIDER_ID && nID <= FIRST_SLIDER_ID+1000)
	{
		SetTime( GetTime() );

		CWnd *pWnd = CWnd::FromHandle(pNMHDR->hwndFrom);
		if (pWnd->IsKindOf(RUNTIME_CLASS(CSplineCtrl)))
		{
			CSplineCtrl *pSlineCtrl = (CSplineCtrl*)pWnd;

			for (int i = 0; i < (int)m_controllers.size(); i++)
			{
				if (m_controllers[i].pSplineCtrl == pSlineCtrl)
				{
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////
float CTimeOfDayDialog::GetTime() const
{
	float fTime = 0;
	if (GetIEditor()->GetDocument()->GetCurrentMission())
		fTime = GetIEditor()->GetDocument()->GetCurrentMission()->GetTime();
	return fTime;
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::SetTimeRange( float fTimeStart,float fTimeEnd,float fSpeed )
{
	{
		int nHour = floor(fTimeStart);
		int nMins = (fTimeStart - floor(fTimeStart)) * 60.0f;
		m_timeEditStart.SetTime( nHour,nMins );
	}
	{
		int nHour = floor(fTimeEnd);
		int nMins = (fTimeEnd - floor(fTimeEnd)) * 60.0f;
		m_timeEditEnd.SetTime( nHour,nMins );
	}
	m_timeAnimSpeedCtrl.SetValue( fSpeed );

	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
	ITimeOfDay::SAdvancedInfo advInfo;
	pTimeOfDay->GetAdvancedInfo( advInfo );
	advInfo.fStartTime = fTimeStart;
	advInfo.fEndTime = fTimeEnd;
	advInfo.fAnimSpeed = fSpeed;
	pTimeOfDay->SetAdvancedInfo( advInfo );
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::SetTime( float fHour,bool bOnlyUpdateUI,bool bNoPropertiesUpdate )
{
	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();

	bool bForceUpdate = m_forceUpdateCheck.GetCheck() == BST_CHECKED;

	if (!m_bIgnoreSetTimeToEngine && !bOnlyUpdateUI)
		pTimeOfDay->SetTime( fHour,bForceUpdate );

	if (GetIEditor()->GetDocument()->GetCurrentMission())
		GetIEditor()->GetDocument()->GetCurrentMission()->SetTime( fHour );

	int p = fHour*SLIDER_SCALE;
	if (!m_bIgnoreScroll)
	{
		m_bIgnoreScroll = true;
		m_weightSlider.SetPos( p );
		m_bIgnoreScroll = false;
	}
	if (p < 0)
		m_weightSlider.SetSelection(p,0);
	else if (p > 0)
		m_weightSlider.SetSelection(0,p);

	int nHour = floor(fHour);
	int nMins = (fHour - floor(fHour)) * 60.0f;
	m_timeEdit.SetTime( nHour,nMins );

	if (!bNoPropertiesUpdate)
	{
		m_propsCtrl.EnableUpdateCallback(false);
		for (int i = 0; i < (int)m_controllers.size(); i++)
		{
			if (m_controllers[i].pSplineCtrl)
				m_controllers[i].pSplineCtrl->SetTimeMarker(fHour/24.0f);
			if (m_controllers[i].pColorCtrl)
				m_controllers[i].pColorCtrl->SetTimeMarker(fHour/24.0f);

			IVariable *pVar = m_controllers[i].pVariable;
			if (!pVar)
				continue;

			ITimeOfDay::SVariableInfo varInfo;
			if (!pTimeOfDay->GetVariableInfo( i,varInfo ))
				continue;
			switch (varInfo.type)
			{
			case ITimeOfDay::TYPE_FLOAT:
				pVar->Set( varInfo.fValue[0] );
				break;
			case ITimeOfDay::TYPE_COLOR:
				pVar->Set( Vec3(varInfo.fValue[0],varInfo.fValue[1],varInfo.fValue[2]) );
				break;
			}
		}
		m_propsCtrl.EnableUpdateCallback(true);
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnPlayAnim()
{
	m_bAnimation = !m_bAnimation;
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnUpdatePlayAnim( CCmdUI* pCmdUI )
{
	pCmdUI->SetCheck( (m_bAnimation)?BST_CHECKED:BST_UNCHECKED );
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnRecord()
{
	m_bRecording = !m_bRecording;
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnUpdateRecord( CCmdUI* pCmdUI )
{
	pCmdUI->SetCheck( (m_bRecording)?BST_CHECKED:BST_UNCHECKED );
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnPlayAnimFrom0()
{
	m_bPlayFrom0 = !m_bPlayFrom0;
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnUpdatePlayAnimFrom0( CCmdUI* pCmdUI )
{
	pCmdUI->SetCheck(m_bPlayFrom0);
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnGotoMinus1()
{
	SetTime(0);
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnGoto0()
{
	SetTime(12);
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnGoto1()
{
	SetTime(24);
}


//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnEditorNotifyEvent( EEditorNotifyEvent event )
{
	switch (event)
	{
	case eNotify_OnEndSceneOpen:
		ReloadCtrls();
		UpdateValues();
		break;
	case eNotify_OnIdleUpdate:
		if (m_bAnimation)
		{
			ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
			float fHour = pTimeOfDay->GetTime();

			ITimeOfDay::SAdvancedInfo advInfo;
			pTimeOfDay->GetAdvancedInfo( advInfo );

			float dt = gEnv->pTimer->GetFrameTime();
			float fTime = fHour + dt*advInfo.fAnimSpeed;
			if (fTime > advInfo.fEndTime)
				fTime = advInfo.fStartTime;
			if (fTime < advInfo.fStartTime)
				fTime = advInfo.fEndTime;
			if (m_bPlayFrom0 && fTime < 0)
				fTime = 0;
			SetTime( fTime );
		}
		break;
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnChangeCurrentTime( UINT nID )
{
	switch (nID)
	{
	case IDC_TIME_OF_DAY:
		{
			CString str;
			m_timeEdit.GetWindowText(str);
			int h,m;
			sscanf( str,_T("%02d:%02d"),&h,&m );
			float fTime = (float)h + (float)m/60.0f;
			SetTime( fTime );
		}
		break;
	case IDC_TIME_OF_DAY_START:
		{
			CString str;
			m_timeEditStart.GetWindowText(str);
			int h,m;
			sscanf( str,_T("%02d:%02d"),&h,&m );
			float fTime = (float)h + (float)m/60.0f;
			ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
			ITimeOfDay::SAdvancedInfo advInfo;
			pTimeOfDay->GetAdvancedInfo( advInfo );
			advInfo.fStartTime = fTime;
			pTimeOfDay->SetAdvancedInfo( advInfo );
		}
		break;
	case IDC_TIME_OF_DAY_END:
		{
			CString str;
			m_timeEditEnd.GetWindowText(str);
			int h,m;
			sscanf( str,_T("%02d:%02d"),&h,&m );
			float fTime = (float)h + (float)m/60.0f;
			ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
			ITimeOfDay::SAdvancedInfo advInfo;
			pTimeOfDay->GetAdvancedInfo( advInfo );
			advInfo.fEndTime = fTime;
			pTimeOfDay->SetAdvancedInfo( advInfo );
		}
		break;
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnChangeTimeAnimSpeed()
{
	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
	ITimeOfDay::SAdvancedInfo advInfo;
	pTimeOfDay->GetAdvancedInfo( advInfo );
	advInfo.fAnimSpeed = m_timeAnimSpeedCtrl.GetValue();
	pTimeOfDay->SetAdvancedInfo( advInfo );
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnImport()
{
	char szFilters[] = "Time Of Day Settings (*.tod)|*.tod||";
	CString fileName;

	if (CFileUtil::SelectFile( szFilters,GetIEditor()->GetLevelFolder(),fileName ))
	{
		XmlNodeRef root = GetISystem()->LoadXmlFile( fileName );
		if (root)
		{
			ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
			float fTime = GetTime();
			pTimeOfDay->Serialize( root,true );
			pTimeOfDay->SetTime( fTime,true );

			UpdateValues();
		}
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnExport()
{
	char szFilters[] = "Time Of Day Settings (*.tod)|*.tod||";
	CString fileName;
	if (CFileUtil::SelectSaveFile( szFilters,"tod",GetIEditor()->GetLevelFolder(),fileName ))
	{
		// Write the light settings into the archive
		XmlNodeRef node = CreateXmlNode( "TimeOfDay" );
		ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
		pTimeOfDay->Serialize( node,false );
		SaveXmlNode( node,fileName );
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnExpandAll()
{
	for (int i = 0; i < m_wndSplinesTaskPanel.GetGroups()->GetCount(); i++)
	{
		m_wndSplinesTaskPanel.ExpandGroup( (CXTPTaskPanelGroup*)m_wndSplinesTaskPanel.GetGroups()->GetAt(i),TRUE );
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnCollapseAll()
{
	for (int i = 0; i < m_wndSplinesTaskPanel.GetGroups()->GetCount(); i++)
	{
		m_wndSplinesTaskPanel.ExpandGroup( (CXTPTaskPanelGroup*)m_wndSplinesTaskPanel.GetGroups()->GetAt(i),FALSE );
	}
}

//////////////////////////////////////////////////////////////////////////
LRESULT CTimeOfDayDialog::OnTaskPanelNotify( WPARAM wParam, LPARAM lParam )
{
	switch(wParam) {
	case XTP_TPN_CLICK:
		{
			CXTPTaskPanelGroupItem* pItem = (CXTPTaskPanelGroupItem*)lParam;
			UINT nCmdID = pItem->GetID();
			switch (nCmdID)
			{
				case ID_PLAY_TOD_ANIM:
					m_bAnimation = true;
					break;
				case ID_STOP_TOD_ANIM:
					m_bAnimation = false;
					break;
				case ID_FILE_IMPORT:
					OnImport();
					break;
				case ID_IMPORT_SEUQENCE:
					OnImportSequence();
					break;
				case ID_FILE_EXPORT:
					OnExport();
					break;
				case ID_EXPAND_ALL_SPLINES:
					OnExpandAll();
					break;
				case ID_COLLAPSE_ALL_SPLINES:
					OnCollapseAll();
					break;
			}
		}
		break;

	case XTP_TPN_RCLICK:
		break;
	}
	return 0;
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnUpdateProperties( IVariable *pVar )
{
	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();

	//////////////////////////////////////////////////////////////////////////
	for (int i = 0; i < (int)m_controllers.size(); i++)
	{
		if (m_controllers[i].pVariable != pVar)
			continue;

		float fTime = GetTime();
		float fSplineTime = fTime / 24;
		
		ITimeOfDay::SVariableInfo varInfo;
		if (!pTimeOfDay->GetVariableInfo( i,varInfo ))
			continue;

		int nKey = varInfo.pInterpolator->FindKey(fSplineTime);
		int nLastKey = varInfo.pInterpolator->GetKeyCount()-1;
		
		if (CUndo::IsRecording())
			CUndo::Record( new CUndoTimeOfDayObject );

		switch (varInfo.type)
		{
		case ITimeOfDay::TYPE_FLOAT:
			{
				float fVal = 0;
				pVar->Get(fVal);
				if (m_bRecording)
				{
					if (nKey < 0)
						varInfo.pInterpolator->InsertKeyFloat(fSplineTime,fVal);
					else
					{
						varInfo.pInterpolator->SetKeyValueFloat(nKey,fVal);
						if (nKey == 0)
							varInfo.pInterpolator->SetKeyValueFloat(nLastKey,fVal);
						else if (nKey == nLastKey)
							varInfo.pInterpolator->SetKeyValueFloat(0,fVal);
					}
				}

				float v3[3] = {fVal,varInfo.fValue[1],varInfo.fValue[2]};
				pTimeOfDay->SetVariableValue( i,v3 );
			}
			break;
		case ITimeOfDay::TYPE_COLOR:
			{
				Vec3 vVal;
				pVar->Get(vVal);
				float v3[3] = {vVal.x,vVal.y,vVal.z};
				if (m_bRecording)
				{
					if (nKey < 0)
						varInfo.pInterpolator->InsertKeyFloat3(fSplineTime,v3);
					else
					{
						varInfo.pInterpolator->SetKeyValueFloat3(nKey,v3);
						if (nKey == 0)
							varInfo.pInterpolator->SetKeyValueFloat3(nLastKey,v3);
						else if (nKey == nLastKey)
							varInfo.pInterpolator->SetKeyValueFloat3(0,v3);
					}
				}
				pTimeOfDay->SetVariableValue( i,v3 );
			}
			break;
		}

		if (m_controllers[i].pSplineCtrl)
			m_controllers[i].pSplineCtrl->Invalidate();
		if (m_controllers[i].pColorCtrl)
			m_controllers[i].pColorCtrl->Invalidate();

		bool bForceUpdate = m_forceUpdateCheck.GetCheck() == BST_CHECKED;
		pTimeOfDay->Update( false,bForceUpdate );
		break;
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::SetInterpolatorFloat( ISplineInterpolator *pInterpolator,float fTime,float fValue,int nIndex )
{
	int nKey = pInterpolator->FindKey(fTime);
	if (nKey < 0)
	{
		ISplineInterpolator::ValueType v = {0,0,0,0};
		v[nIndex] = fValue;
		pInterpolator->InsertKey(fTime,v);
	}
	else
	{
		ISplineInterpolator::ValueType v = {0,0,0,0};
		pInterpolator->GetKeyValue(nKey,v);
		v[nIndex] = fValue;
		pInterpolator->SetKeyValue(nKey,v);
	}
}

//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::LoadFloatTrack( const CString &name,const CString &varName,XmlNodeRef allNodes,int nIndex,float fValueScale )
{
	XmlNodeRef node = 0;
	{
		for (int i = 0; i < allNodes->getChildCount(); i++)
		{
			XmlNodeRef n = allNodes->getChild(i);
			CString nodeName = n->getAttr("Name");
			if (stricmp(nodeName,name) == 0)
			{
				node = n;
				break;
			}
		}
	}
	if (!node)
	{
		Warning( "Missing node" );
		return;
	}

	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
	ISplineInterpolator *pInterpolator = 0;
	{
		for (int i = 0; i < pTimeOfDay->GetVariableCount(); i++)
		{
			int nImage = 0;

			ITimeOfDay::SVariableInfo varInfo;
			if (!pTimeOfDay->GetVariableInfo( i,varInfo ))
				continue;
			if (stricmp(varInfo.name,varName) == 0)
				pInterpolator = varInfo.pInterpolator;
		}
	}
	if (!pInterpolator)
	{
		Warning( "Missing var name" );
		return;
	}

	XmlNodeRef track = node->findChild("Track");
	if (!track)
		return;

	if (track->getChildCount() > 0)
	{
		// Have keys in track.

		if (nIndex == 0)
		{
			// delete all keys.
			while (pInterpolator->GetKeyCount() > 0)
				pInterpolator->RemoveKey(0);
		}

		for (int i = 0; i < track->getChildCount(); i++)
		{
			XmlNodeRef keyNode = track->getChild(i);
			float time = (int)(atof(keyNode->getAttr("time")));
			time = time / (24*100.0f);
			if (time < 0)
				time = 0;
			if (time > 1)
				time = 1;
			float val = atof(keyNode->getAttr("value")) * fValueScale;

			SetInterpolatorFloat(pInterpolator,time,val,nIndex);
		}
		
		// Add 0/1 keys.
		float val0 = 0;
		pInterpolator->GetKeyValueFloat(0,val0);

		SetInterpolatorFloat(pInterpolator,0,val0,nIndex);
		SetInterpolatorFloat(pInterpolator,1,val0,nIndex);
	}
}

//////////////////////////////////////////////////////////////////////////
// For backward compatibility.
//////////////////////////////////////////////////////////////////////////
void CTimeOfDayDialog::OnImportSequence()
{
	char szFilters[] = "Time Of Day TrackView Sequence (*.*)|*.*||";
	CString fileName;

	if (!CFileUtil::SelectFile( szFilters,GetIEditor()->GetLevelFolder(),fileName ))
		return;

	XmlNodeRef root = GetISystem()->LoadXmlFile( fileName );
  if (!root)
		return;

	ITimeOfDay *pTimeOfDay = gEnv->p3DEngine->GetTimeOfDay();
	XmlNodeRef allNodes = root->findChild( "Nodes" );
	if (!allNodes)
		return;

	LoadFloatTrack( "EnvLighting.fSunMultiplier","Sun color multiplier",allNodes );
	LoadFloatTrack( "EnvLighting.fSkyMultiplier","Sky color multiplier",allNodes );
	
	LoadFloatTrack( "EnvLighting.fVolFogGlobalDensity",       "Volumetric fog: Global density",allNodes );
	LoadFloatTrack( "EnvLighting.fVolFogAtmosphereHeight",    "Volumetric fog: Atmosphere height",allNodes );
	LoadFloatTrack( "EnvLighting.fVolFogSunColorGlobalAtten", "Volumetric fog: Sun color global attenuation",allNodes );
	LoadFloatTrack( "EnvLighting.fVolFogSunColorAngularAtten","Volumetric fog: Sun color angular attenuation",allNodes );
	
	LoadFloatTrack( "EnvLighting.fWaveLenghtR","Sky light: Wavelength (R)",allNodes );
	LoadFloatTrack( "EnvLighting.fWaveLenghtG","Sky light: Wavelength (G)",allNodes );
	LoadFloatTrack( "EnvLighting.fWaveLenghtB","Sky light: Wavelength (B)",allNodes );
	
	LoadFloatTrack( "EnvLighting.fKm","Sky light: Mie scattering",allNodes );
	LoadFloatTrack( "EnvLighting.fKr","Sky light: Rayleigh scattering",allNodes );
	LoadFloatTrack( "EnvLighting.fG","Sky light: Sun anisotropy factor",allNodes );
	
	LoadFloatTrack( "EnvLighting.fOceanWaterTransp","Ocean water transparency",allNodes );
	
	LoadFloatTrack( "EnvLighting.fCloudShadingSunLightMultiplier","Cloud shading: Sun light multiplier",allNodes );
	LoadFloatTrack( "EnvLighting.fCloudShadingSkyLightMultiplier","Cloud shading: Sky light multiplier",allNodes );

	LoadFloatTrack( "r_CoronaSizeScale","Size of corona",allNodes );
	LoadFloatTrack( "fHDRDynamicPower","HDR dynamic power factor",allNodes );

	// Colors.
	float fColorScale = 1.0f/255.0f;
	LoadFloatTrack( "EnvLighting.fSunR","Sun color",allNodes,0,fColorScale );
	LoadFloatTrack( "EnvLighting.fSunG","Sun color",allNodes,1,fColorScale );
	LoadFloatTrack( "EnvLighting.fSunB","Sun color",allNodes,2,fColorScale );

	LoadFloatTrack( "EnvLighting.fSkyR","Sky color",allNodes,0,fColorScale );
	LoadFloatTrack( "EnvLighting.fSkyG","Sky color",allNodes,1,fColorScale );
	LoadFloatTrack( "EnvLighting.fSkyB","Sky color",allNodes,2,fColorScale );

	LoadFloatTrack( "EnvLighting.fFogR","Fog color",allNodes,0,fColorScale );
	LoadFloatTrack( "EnvLighting.fFogG","Fog color",allNodes,1,fColorScale );
	LoadFloatTrack( "EnvLighting.fFogB","Fog color",allNodes,2,fColorScale );

	LoadFloatTrack( "EnvLighting.fSunIntensityR","Sky light: Sun intensity",allNodes,0,fColorScale );
	LoadFloatTrack( "EnvLighting.fSunIntensityG","Sky light: Sun intensity",allNodes,1,fColorScale );
	LoadFloatTrack( "EnvLighting.fSunIntensityB","Sky light: Sun intensity",allNodes,2,fColorScale );

	LoadFloatTrack( "EnvLighting.fUnderWaterFogR","Underwater fog color",allNodes,0,fColorScale );
	LoadFloatTrack( "EnvLighting.fUnderWaterFogG","Underwater fog color",allNodes,1,fColorScale );
	LoadFloatTrack( "EnvLighting.fUnderWaterFogB","Underwater fog color",allNodes,2,fColorScale );
		
	UpdateValues();
}
