#include "stdafx.h"
#include "AutoRunCDApp.h"
#include "BackBuffer.h"
#include "Jukebox.h"
#include "MenuData.h"
#include "Utilities.h"
#include "ProductRegistry.h"
#include "LanguageID.h"
#include "resource.h"

#include <process.h>



#define FRAMELESS_WND



const unsigned int c_uiFadeScreenTimerID( 0 );
const unsigned int c_uiFadeTimerScreenInterval( 50 );

const unsigned char c_ucDefTransparency( 220 );



using namespace std;



static LRESULT CALLBACK
GlobalWndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	// delegate to wnd class
	return( CAutoRunCDApp::GetInstance().WndProc( hWnd, message, wParam, lParam ) );
}



CAutoRunCDApp::CAutoRunCDApp()
: m_hInstance( 0 )
, m_hWnd( 0 )
, m_hDefCursor( 0 )
, m_pkBackBuffer( 0 )
, m_pkJukebox( 0 )
, m_pkMenuData( 0 )
, m_ucCurButtonID( BID_NOBUTTON )
, m_iLastDragMouseScreenX( 0 )
, m_iLastDragMouseScreenY( 0 )
, m_bLeftMouseButtonPressed( false )
, m_bDragWindow( false )
, m_bScreenIsFading( false )
, m_ucFadeScreenCounter( 0 )
//, m_mediaChangeNotificationID(0)
{
}



CAutoRunCDApp::~CAutoRunCDApp()
{
	delete m_pkBackBuffer;
	delete m_pkJukebox;
	delete m_pkMenuData;

	// dump leaks here... be aware of stl container members and the existance of other static object 
	// instances (their destruction order can't be determined) which will be reported as "pseudo" leaks
	//_CrtDumpMemoryLeaks();
}



CAutoRunCDApp& 
CAutoRunCDApp::GetInstance()
{
	// returns the only instance of this class
	static CAutoRunCDApp s_kInstance;
	return( s_kInstance );
}



ATOM 
CAutoRunCDApp::RegisterClass()  const
{
	assert( 0 != m_hInstance );

	// get class name for resource
	tstring strClassName( GetStringFromResource( m_hInstance, IDC_AUTORUNCD ) );

	// initialize class structure
	WNDCLASSEX sWndClassEx;
	sWndClassEx.cbSize			= sizeof( WNDCLASSEX ); 
	sWndClassEx.style			= CS_HREDRAW | CS_VREDRAW;
	sWndClassEx.lpfnWndProc		= (WNDPROC) GlobalWndProc;
	sWndClassEx.cbClsExtra		= 0;
	sWndClassEx.cbWndExtra		= 0;
	sWndClassEx.hInstance		= m_hInstance;
	sWndClassEx.hIcon			= LoadIcon( m_hInstance, (LPCTSTR) IDI_AUTORUNCD );
	sWndClassEx.hCursor			= LoadCursor( 0, IDC_ARROW );
	sWndClassEx.hbrBackground	= (HBRUSH) ( COLOR_WINDOW + 1 );
	sWndClassEx.lpszMenuName	= (LPCTSTR) IDC_AUTORUNCD;
	sWndClassEx.lpszClassName	= strClassName.c_str();
	sWndClassEx.hIconSm			= LoadIcon( sWndClassEx.hInstance, (LPCTSTR) IDI_AUTORUNCD );

	// register class
	return( RegisterClassEx( &sWndClassEx ) );
}



void
CAutoRunCDApp::PrepareTransparency()
{
	// get access to "SetLayeredWindowAttributes" function
	HMODULE hUser32( LoadLibrary( _T( "user32.dll" ) ) );
	if( 0 != hUser32 )
	{
		typedef DWORD (WINAPI* PFSetLayeredWindowAttributes) ( HWND, DWORD, BYTE, DWORD );
		PFSetLayeredWindowAttributes pFunc( (PFSetLayeredWindowAttributes) GetProcAddress( hUser32, "SetLayeredWindowAttributes" ) );

		// set default transparency if functionality is available
		if( 0 != pFunc ) 
		{
			SetWindowLong( m_hWnd, GWL_EXSTYLE , GetWindowLong( m_hWnd , GWL_EXSTYLE ) | WS_EX_LAYERED );
			pFunc( m_hWnd, RGB( 255, 255, 255 ), c_ucDefTransparency, LWA_COLORKEY | LWA_ALPHA );
		}

		FreeLibrary( hUser32 );
	}
}



void
CAutoRunCDApp::InitInstance( const HINSTANCE& hInstance, int nCmdShow )
{
	// set instance handle
	m_hInstance = hInstance;

#ifdef FRAMELESS_WND
    DWORD dwWndStyle( WS_POPUP | /*WS_SYSMENU |*/ WS_MINIMIZEBOX );
#else
    DWORD dwWndStyle( WS_OVERLAPPEDWINDOW & ~WS_MAXIMIZEBOX );
#endif // #ifdef FRAMELESS_WND

		// try to create window
		if( 0 == ( m_hWnd = CreateWindow( (LPCTSTR) RegisterClass(), 
			GetStringFromResource( m_hInstance, IDS_APP_TITLE ).c_str(), 
			dwWndStyle,
			( GetSystemMetrics( SM_CXSCREEN ) - c_iWndWidth ) / 2, 
			( GetSystemMetrics( SM_CYSCREEN ) - c_iWndHeight ) / 2, 
			c_iWndWidth, c_iWndHeight, 0, 0, m_hInstance, 0 ) ) )
		{
			throw( texception( GetStringFromResource( m_hInstance, IDS_FAILEDTOCREATEWINDOW ) ) );
		}

#ifndef FRAMELESS_WND
	// adjust window size and position
	RECT rectWnd;
	GetWindowRect( m_hWnd, &rectWnd );

	RECT rectClient;
	GetClientRect( m_hWnd, &rectClient );

	int iNewWidth( c_iWndWidth + ( rectWnd.right - rectWnd.left ) - ( rectClient.right - rectClient.left ) );
	int iNewHeight( c_iWndHeight + ( rectWnd.bottom - rectWnd.top ) - ( rectClient.bottom - rectClient.top ) );

	MoveWindow( m_hWnd,
				( GetSystemMetrics( SM_CXSCREEN ) - iNewWidth  ) >> 1, 
				( GetSystemMetrics( SM_CYSCREEN ) - iNewHeight ) >> 1, 
				iNewWidth, iNewHeight, TRUE );
#endif // #ifndef FRAMELESS_WND

	m_hDefCursor = LoadCursor((HINSTANCE) m_hInstance, MAKEINTRESOURCE(IDC_CURSOR_DEFAULT));
	if (!m_hDefCursor)
		m_hDefCursor = LoadCursor(0, IDC_ARROW);

	// initialize back buffer
	m_pkBackBuffer = new CBackBuffer( m_hInstance, m_hWnd );

	// initialize sound system
	m_pkJukebox = new CJukebox( m_hInstance );

	// initialize menu data
	m_pkMenuData = new CMenuData( m_hInstance, m_pkBackBuffer );

	// draw inital state of window
	OnDraw();

	// present window
	//PrepareTransparency();
	ShowWindow( m_hWnd, nCmdShow );
	UpdateWindow( m_hWnd );

	// start playing background music
	m_pkJukebox->PlaySound( CJukebox::SND_BACKGROUND_MUSIC );

	//InitMediaChangeMonitor();	
}



void 
CAutoRunCDApp::SetCursorSafe( const TCHAR* pcCursorName )
{
	SetCursor(m_hDefCursor);

	//HCURSOR hCursor( LoadCursor( 0, pcCursorName ) );	
	//if( 0 == hCursor )
	//{
	//	// safety net... IDC_ARROW should always be available
	//	hCursor = LoadCursor( 0, IDC_ARROW );
	//	assert( 0 != hCursor );
	//}
	//SetCursor( hCursor );
}



int 
CAutoRunCDApp::Run()
{	
	int iRet;

	// standard message loop
	MSG sMessage;
	while( 0 != ( iRet = GetMessage( &sMessage, 0, 0, 0 ) ) )
	{ 
		if( -1 == iRet )
		{
			//ShutdownMediaChangeMonitor();
			return( -1 );
		}
		else
		{
			if( FALSE != PeekMessage( &sMessage, 0, 0, 0, PM_REMOVE ) )
			{
				if( WM_QUIT == sMessage.message ) 
				{
					break;
				}
			}

			TranslateMessage( &sMessage ); 
			DispatchMessage( &sMessage ); 
		}
	}
	//ShutdownMediaChangeMonitor();
	return( (int) sMessage.wParam );
}



void 
CAutoRunCDApp::HandleSysMenu()
{
	// handle system menu button clicks
	switch( m_ucCurButtonID )
	{
	case BID_SYSMENU_MINIMIZE:
		{
			ShowWindow( m_hWnd, SW_MINIMIZE );
			break;
		}
	case BID_SYSMENU_QUIT:
		{
			m_ucCurButtonID = BID_NOBUTTON;
			SendMessage( m_hWnd, WM_CLOSE, 0, 0 );
			break;
		}
	default:
		{
			assert( 0 );
			break;
		}
	}
}



void 
CAutoRunCDApp::ClientToScreen(  int iClientX, int iClientY, int& iScreenX, int& iScreenY  ) const
{
    POINT p;
    p.x = iClientX;
    p.y = iClientY;
    ::ClientToScreen( m_hWnd, &p );
    iScreenX = p.x;
    iScreenY = p.y;
}



void 
CAutoRunCDApp::FadeScreen()
{
	assert( 0 != m_pkBackBuffer );
	
	// cheap fade down effect using bit shifts and masks
	unsigned int* puiBackBufferBits( m_pkBackBuffer->GetPtr() );
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		puiBackBufferBits[ uiIndex ] = ( puiBackBufferBits[ uiIndex ] >> 1 ) & 0x7F7F7F7F;
	}

	InvalidateRect( m_hWnd, 0, FALSE );
}



void 
CAutoRunCDApp::OnPaint()
{
	// prepare paint struct
	PAINTSTRUCT sPS;	
	HDC hDC( BeginPaint( m_hWnd, &sPS ) );
	assert( 0 != hDC );

	// blit from back buffer to window
	BOOL bRet( BitBlt( hDC, 0, 0, c_iWndWidth, c_iWndHeight, m_pkBackBuffer->GetDC(), 0, 0, SRCCOPY ) );
	assert( FALSE != bRet );

	// mark end of window paining
	EndPaint( m_hWnd, &sPS );
}



void 
CAutoRunCDApp::OnDraw()
{
	assert( 0 != m_pkBackBuffer );

	// get access to back buffer
	HDC hBackBufferDC( m_pkBackBuffer->GetDC() );
	unsigned int* puiBackBufferBits( m_pkBackBuffer->GetPtr() );

	// draw window content
	const unsigned int* const pImageData[ 2 ] =
	{
		&m_pkMenuData->GetMenuBase()[ 0 ],
		( false == m_bLeftMouseButtonPressed ) ? ( &m_pkMenuData->GetMenuHighlighted()[ 0 ] ) : ( &m_pkMenuData->GetMenuSelected()[ 0 ] )
	};

	const Raw8BitImage& imgMenuMask( m_pkMenuData->GetMenuMask() );
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		// draw pixel from either base image or highligh/selected image (in case button ID at current mouse point == pixel's mask)
		puiBackBufferBits[ uiIndex ] = pImageData[ imgMenuMask[ uiIndex ] == m_ucCurButtonID ][ uiIndex ];
	}

	// draw frame using GDI functions
	MoveToEx( hBackBufferDC, 0, 0, 0 );
	LineTo( hBackBufferDC, c_iWndWidth - 1, 0 );
	LineTo( hBackBufferDC, c_iWndWidth - 1, c_iWndHeight - 1 );
	LineTo( hBackBufferDC, 0, c_iWndHeight - 1 );
	LineTo( hBackBufferDC, 0, 0 );

	// enforce WM_PAINT
	InvalidateRect( m_hWnd, 0, FALSE );
}




void 
CAutoRunCDApp::OnKeyDown( unsigned int uiChar, unsigned int uiRepCnt, unsigned int uiFlags )
{
	switch( uiChar )
	{
	case VK_ESCAPE:
		{
			m_ucCurButtonID = BID_NOBUTTON;
			SendMessage( m_hWnd, WM_CLOSE, 0, 0 );
		}
	default:
		{
		}
	}
}



void 
CAutoRunCDApp::OnMouseMove( int iNewX, int iNewY, unsigned int uiFlags )
{
	if( false == m_bScreenIsFading )
	{	
		if( false != m_bLeftMouseButtonPressed && 0 == ( uiFlags & MK_LBUTTON ) )
		{
			// mouse button was pressed but got release w/o notice of OnLButtonUp( ... )
			m_bLeftMouseButtonPressed = false;
			OnDraw();
		}

		if( false != m_bLeftMouseButtonPressed && false != m_bDragWindow )
		{
			// handle dragging of window
			RECT rectWnd;
			GetWindowRect( m_hWnd, &rectWnd );

			int iCurDragMouseScreenX( 0 );
			int iCurDragMouseScreenY( 0 );
			ClientToScreen( iNewX, iNewY, iCurDragMouseScreenX, iCurDragMouseScreenY );

			MoveWindow( m_hWnd, 
				rectWnd.left + ( iCurDragMouseScreenX - m_iLastDragMouseScreenX ), 
				rectWnd.top + ( iCurDragMouseScreenY - m_iLastDragMouseScreenY ), 
				rectWnd.right - rectWnd.left,
				rectWnd.bottom - rectWnd.top, TRUE );

			m_iLastDragMouseScreenX = iCurDragMouseScreenX;
			m_iLastDragMouseScreenY = iCurDragMouseScreenY;

			SetCursorSafe( IDC_SIZEALL );
		}
		else
		{
			// update current button ID as a result of the mouse move
			unsigned char ucNewButtonID( m_pkMenuData->GetMenuMask()[ GetOffset( iNewX, iNewY ) ] );
			if( m_ucCurButtonID != ucNewButtonID )
			{
				m_ucCurButtonID = ucNewButtonID;

				if( BID_NOBUTTON == m_ucCurButtonID  )
				{
					//m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_LEFTUNTOUCHED );
					m_bLeftMouseButtonPressed = false;
				}
				else
				{
					if( m_ucCurButtonID < __SYS_MENU_START )
					{
						m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_HOVERING );
					}
				}

				OnDraw();
			}

			// set mouse cursor accordingly
			if( BID_NOBUTTON != m_ucCurButtonID )
			{
				SetCursorSafe( IDC_HAND );
			}
			else
			{
				SetCursorSafe( IDC_ARROW );
			}
		}
	}
}



void
CAutoRunCDApp::OnLButtonDown( int iNewX, int iNewY )
{
	if( false == m_bScreenIsFading )
	{
		if( BID_NOBUTTON == m_ucCurButtonID )
		{
			// initiate window dragging
			SetCursorSafe( IDC_SIZEALL );
			ClientToScreen( iNewX, iNewY, m_iLastDragMouseScreenX, m_iLastDragMouseScreenY );
			SetCapture( m_hWnd );
			m_bDragWindow = true;
		}
		else
		{
			if( m_ucCurButtonID < __SYS_MENU_START )
			{
				m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_PRESSED );
			}
		}

		// mark LMB as pressed and redraw window content
		m_bLeftMouseButtonPressed = true;
		OnDraw();
	}
}



void
CAutoRunCDApp::OnLButtonUp()
{
	if( false == m_bScreenIsFading )
	{
		const TCHAR* strCursorID( IDC_ARROW );

		if( BID_NOBUTTON != m_ucCurButtonID )
		{
			// handle button events
			if( m_ucCurButtonID < __SYS_MENU_START )
			{
				switch( m_ucCurButtonID )
				{
				case BID_INSTALL:
				// case BID_UNINSTALL:
				case BID_PLAY:
				case BID_QUIT:
					{
						SendMessage( m_hWnd, WM_CLOSE, 0, 0 );
						break;
					}
				default:
					{
						RunButtonEvent();
						strCursorID = IDC_HAND;
						break;
					}
				}
			}
			else
			{
				HandleSysMenu();
			}
			//m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_RELEASED );
		}

		if( false != m_bDragWindow )
		{
			// leave window dragging mode
			ReleaseCapture();
			m_bDragWindow = false;
		}        

		// mark LMB as not-pressed and redraw window content
		m_bLeftMouseButtonPressed = false;
		SetCursorSafe( strCursorID );
		OnDraw();
	}
}



void
CAutoRunCDApp::OnTimer( unsigned int uiTimerID )
{
	switch( uiTimerID )
	{
	case c_uiFadeScreenTimerID:
		{
			if( m_ucFadeScreenCounter < 8 )
			{
				// still need to update screen fade effect
				FadeScreen();
				++m_ucFadeScreenCounter;

				// fade out sound as well
				//m_pkJukebox->SetVolume( ( 8 - (int) m_ucFadeScreenCounter ) * 255 / 8 );
			}
			else
			{
				// screen was faded down. kill timer execute button event and quit
				KillTimer( m_hWnd, c_uiFadeScreenTimerID );
				ShowWindow( m_hWnd, SW_HIDE );
				RunButtonEvent();
				PostQuitMessage( 0 );
			}
			break;
		}
	default:
		{
			break;
		}
	}
}



void
CAutoRunCDApp::OnClose()
{
	m_bScreenIsFading = true;
	SetTimer( m_hWnd, c_uiFadeScreenTimerID, c_uiFadeTimerScreenInterval, 0 );
}


void CAutoRunCDApp::ReportError() const
{
	unsigned int errCode(GetLastError());
	if (errCode)
	{
		TCHAR* pMsgBuf(0);
		FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM, 0, errCode, 0, (TCHAR*) &pMsgBuf, 0, 0);
		MessageBox(m_hWnd, pMsgBuf, GetStringFromResource(m_hInstance, IDS_APP_TITLE).c_str(), MB_OK | MB_ICONEXCLAMATION); 
		LocalFree(pMsgBuf);
	}
}


class CSEMGuard
{
public:
	CSEMGuard(unsigned int sem)
	: m_oldSEM(SetErrorMode(sem))
	{
	}
	~CSEMGuard()
	{
		SetErrorMode(m_oldSEM);
	}

private:
	unsigned int m_oldSEM;
};


static bool IsVistaKB940105Required()
{
	OSVERSIONINFO osv;
	memset(&osv, 0, sizeof(osv));
	osv.dwOSVersionInfoSize = sizeof(osv);
	GetVersionEx(&osv);

	if (osv.dwMajorVersion != 6 || osv.dwMinorVersion != 0 || (osv.dwBuildNumber > 6000))
	{
		// This QFE only ever applies to Windows Vista RTM. Windows Vista SP1 already has this fix,
		// and earlier versions of Windows do not implement WDDM
		return false;
	}

	//MEMORYSTATUSEX mex;
	//memset(&mex, 0, sizeof(mex));
	//mex.dwLength = sizeof(mex);
	//GlobalMemoryStatusEx(&mex);

	//if (mex.ullTotalVirtual >= 4294836224)
	//{
	//	// If there is 4 GB of VA space total for this process, then we are a
	//	// 32-bit Large Address Aware application running on a Windows 64-bit OS.

	//	// We could be a 32-bit Large Address Aware application running on a
	//	// Windows 32-bit OS and get up to 3 GB, but that has stability implications.
	//	// Therefore, we recommend the QFE for all 32-bit versions of the OS.

	//	// No need for the fix unless the game is pushing 4 GB of VA
	//	return false;
	//}

	const TCHAR* sysFile = _T("dxgkrnl.sys");

	// Ensure we are checking the system copy of the file
	TCHAR sysPath[MAX_PATH];
	GetSystemDirectory(sysPath, sizeof(sysPath) / sizeof(sysPath[0]));

	_t_strncat(sysPath, _T("\\drivers\\"), sizeof(sysPath) / sizeof(sysPath[0]));
	_t_strncat(sysPath, sysFile, sizeof(sysPath) / sizeof(sysPath[0]));

	TCHAR buf[2048];
	if (!GetFileVersionInfo(sysPath, 0, sizeof(buf), buf))
	{
		// This should never happen, but we'll assume it's a newer .sys file since we've
		// narrowed the test to a Windows Vista RTM OS.
		return false;
	}

	VS_FIXEDFILEINFO* ver;
	UINT size;
	if (!VerQueryValue(buf, _T("\\"), (void**) &ver, &size) || size != sizeof(VS_FIXEDFILEINFO) || ver->dwSignature != 0xFEEF04BD)
	{
		// This should never happen, but we'll assume it's a newer .sys file since we've
		// narrowed the test to a Windows Vista RTM OS.
		return false;
	}

	// File major.minor.build.qfe version comparison
	//   WORD major = HIWORD( ver->dwFileVersionMS ); WORD minor = LOWORD( ver->dwFileVersionMS );
	//   WORD build = HIWORD( ver->dwFileVersionLS ); WORD qfe = LOWORD( ver->dwFileVersionLS );

	if (ver->dwFileVersionMS > MAKELONG(0, 6) || (ver->dwFileVersionMS == MAKELONG(0, 6) && ver->dwFileVersionLS >= MAKELONG(20648, 6000)))
	{
		// QFE fix version of dxgkrnl.sys is 6.0.6000.20648
		return false;
	}

	return true;
}


#ifdef UNICODE
static tstring GetVistaKB940105WarningText()
{
	switch (GetPrimaryLanguageID())
	{	
	case 0x04:
		{ 
			// Chinese
			const TCHAR c_msgWarning[] = {0x4f60, 0x7684, 0x7cfb, 0x7d71, 0x7f3a, 0x5c11, 0x4e00, 0x500b, 0x91cd, 0x8981, 0x7684, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x66f4, 0x65b0, 0x3002, 0x000d, 0x000a, 0x4f60, 0x60f3, 0x8981, 0x66f4, 0x65b0, 0x55ce, 0xff1f, 0x0000};
			return c_msgWarning;
		}
	case 0x05:
		{
			// Czech
			const TCHAR c_msgWarning[] = {0x004e, 0x0061, 0x0020, 0x0076, 0x0061, 0x0161, 0x0065, 0x006d, 0x0020, 0x0073, 0x0079, 0x0073, 0x0074, 0x00e9, 0x006d, 0x0075, 0x0020, 0x0063, 0x0068, 0x0079, 0x0062, 0x00ed, 0x0020, 0x0064, 0x016f, 0x006c, 0x0065, 0x017e, 0x0069, 0x0074, 0x00e1, 0x0020, 0x0061, 0x006b, 0x0074, 0x0075, 0x0061, 0x006c, 0x0069, 0x007a, 0x0061, 0x0063, 0x0065, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x002e, 0x000d, 0x000a, 0x0050, 0x0159, 0x0065, 0x006a, 0x0065, 0x0074, 0x0065, 0x0020, 0x0073, 0x0069, 0x0020, 0x0070, 0x0072, 0x006f, 0x0076, 0x00e9, 0x0073, 0x0074, 0x0020, 0x0061, 0x006b, 0x0074, 0x0075, 0x0061, 0x006c, 0x0069, 0x007a, 0x0061, 0x0063, 0x0069, 0x003f, 0x0000};
			return c_msgWarning;
		}
	case 0x07:
		{ 
			// German
			const TCHAR c_msgWarning[] = {0x0049, 0x0068, 0x0072, 0x0065, 0x006d, 0x0020, 0x0053, 0x0079, 0x0073, 0x0074, 0x0065, 0x006d, 0x0020, 0x0066, 0x0065, 0x0068, 0x006c, 0x0074, 0x0020, 0x0065, 0x0069, 0x006e, 0x0020, 0x0077, 0x0069, 0x0063, 0x0068, 0x0074, 0x0069, 0x0067, 0x0065, 0x0073, 0x0020, 0x004d, 0x0069, 0x0063, 0x0072, 0x006f, 0x0073, 0x006f, 0x0066, 0x0074, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x002d, 0x0055, 0x0070, 0x0064, 0x0061, 0x0074, 0x0065, 0x002e, 0x000d, 0x000a, 0x004d, 0x00f6, 0x0063, 0x0068, 0x0074, 0x0065, 0x006e, 0x0020, 0x0053, 0x0069, 0x0065, 0x0020, 0x0049, 0x0068, 0x0072, 0x0020, 0x0053, 0x0079, 0x0073, 0x0074, 0x0065, 0x006d, 0x0020, 0x0061, 0x006b, 0x0074, 0x0075, 0x0061, 0x006c, 0x0069, 0x0073, 0x0069, 0x0065, 0x0072, 0x0065, 0x006e, 0x003f, 0x0000};
			return c_msgWarning;
		}		
	case 0x09:
	default:
		{
			// English
			return _T("Your system is missing an important Windows update.\nDo you want to update?");
		}
	case 0x10:
		{
			// Italian
			const TCHAR c_msgWarning[] = {0x0041, 0x006c, 0x0020, 0x0074, 0x0075, 0x006f, 0x0020, 0x0073, 0x0069, 0x0073, 0x0074, 0x0065, 0x006d, 0x0061, 0x0020, 0x006d, 0x0061, 0x006e, 0x0063, 0x0061, 0x0020, 0x0075, 0x006e, 0x0020, 0x0069, 0x006d, 0x0070, 0x006f, 0x0072, 0x0074, 0x0061, 0x006e, 0x0074, 0x0065, 0x0020, 0x0061, 0x0067, 0x0067, 0x0069, 0x006f, 0x0072, 0x006e, 0x0061, 0x006d, 0x0065, 0x006e, 0x0074, 0x006f, 0x0020, 0x0064, 0x0069, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x002e, 0x000d, 0x000a, 0x0056, 0x0075, 0x006f, 0x0069, 0x0020, 0x0061, 0x0067, 0x0067, 0x0069, 0x006f, 0x0072, 0x006e, 0x0061, 0x0072, 0x0065, 0x0020, 0x006f, 0x0072, 0x0061, 0x003f, 0x0000};
			return c_msgWarning;
		}		
	case 0x11:
		{
			// Japanese
			const TCHAR c_msgWarning[] = {0x304a, 0x4f7f, 0x3044, 0x306e, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x306b, 0x306f, 0x91cd, 0x8981, 0x306a, 0x66f4, 0x65b0, 0x30d7, 0x30ed, 0x30b0, 0x30e9, 0x30e0, 0x304c, 0x9069, 0x7528, 0x3055, 0x308c, 0x3066, 0x3044, 0x307e, 0x305b, 0x3093, 0x3002, 0x000d, 0x000a, 0x66f4, 0x65b0, 0x30d7, 0x30ed, 0x30b0, 0x30e9, 0x30e0, 0x306e, 0x60c5, 0x5831, 0x3092, 0x78ba, 0x8a8d, 0x3057, 0x307e, 0x3059, 0x304b, 0xff1f, 0x0000};
			return c_msgWarning;
		}		
	case 0x15:
		{
			// Polish
			const TCHAR c_msgWarning[] = {0x0057, 0x0020, 0x0054, 0x0077, 0x006f, 0x0069, 0x006d, 0x0020, 0x0073, 0x0079, 0x0073, 0x0074, 0x0065, 0x006d, 0x0069, 0x0065, 0x0020, 0x0062, 0x0072, 0x0061, 0x006b, 0x0075, 0x006a, 0x0065, 0x0020, 0x0077, 0x0061, 0x017c, 0x006e, 0x0065, 0x006a, 0x0020, 0x0061, 0x006b, 0x0074, 0x0075, 0x0061, 0x006c, 0x0069, 0x007a, 0x0061, 0x0063, 0x006a, 0x0069, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x002e, 0x000d, 0x000a, 0x0043, 0x007a, 0x0079, 0x0020, 0x0063, 0x0068, 0x0063, 0x0065, 0x0073, 0x007a, 0x002c, 0x0020, 0x0061, 0x0062, 0x0079, 0x0020, 0x006a, 0x0105, 0x0020, 0x0074, 0x0065, 0x0072, 0x0061, 0x007a, 0x0020, 0x007a, 0x0061, 0x0069, 0x006e, 0x0073, 0x0074, 0x0061, 0x006c, 0x006f, 0x0077, 0x0061, 0x0107, 0x003f, 0x0000};
			return c_msgWarning;
		}
	case 0x0a:
		{
			// Spanish
			const TCHAR c_msgWarning[] = {	0x0054, 0x0075, 0x0020, 0x0073, 0x0069, 0x0073, 0x0074, 0x0065, 0x006d, 0x0061, 0x0020, 0x006e, 0x006f, 0x0020, 0x0074, 0x0069, 0x0065, 0x006e, 0x0065, 0x0020, 0x0075, 0x006e, 0x0061, 0x0020, 0x0061, 0x0063, 0x0074, 0x0075, 0x0061, 0x006c, 0x0069, 0x007a, 0x0061, 0x0063, 0x0069, 0x00f3, 0x006e, 0x0020, 0x0064, 0x0065, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x0020, 0x0069, 0x006d, 0x0070, 0x006f, 0x0072, 0x0074, 0x0061, 0x006e, 0x0074, 0x0065, 0x002e, 0x000d, 0x000a, 0x00bf, 0x0051, 0x0075, 0x0069, 0x0065, 0x0072, 0x0065, 0x0073, 0x0020, 0x0061, 0x0063, 0x0074, 0x0075, 0x0061, 0x006c, 0x0069, 0x007a, 0x0061, 0x0072, 0x003f, 0x0000};
			return c_msgWarning;
		}
	case 0x0c:
		{
			// French
			const TCHAR c_msgWarning[] = {0x0056, 0x006f, 0x0074, 0x0072, 0x0065, 0x0020, 0x0073, 0x0079, 0x0073, 0x0074, 0x00e8, 0x006d, 0x0065, 0x0020, 0x0061, 0x0020, 0x0062, 0x0065, 0x0073, 0x006f, 0x0069, 0x006e, 0x0020, 0x0064, 0x0027, 0x0075, 0x006e, 0x0065, 0x0020, 0x006d, 0x0069, 0x0073, 0x0065, 0x0020, 0x00e0, 0x0020, 0x006a, 0x006f, 0x0075, 0x0072, 0x0020, 0x004d, 0x0069, 0x0063, 0x0072, 0x006f, 0x0073, 0x006f, 0x0066, 0x0074, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x0020, 0x0069, 0x006d, 0x0070, 0x006f, 0x0072, 0x0074, 0x0061, 0x006e, 0x0074, 0x0065, 0x002e, 0x000d, 0x000a, 0x0056, 0x006f, 0x0075, 0x006c, 0x0065, 0x007a, 0x002d, 0x0076, 0x006f, 0x0075, 0x0073, 0x0020, 0x0066, 0x0061, 0x0069, 0x0072, 0x0065, 0x0020, 0x006c, 0x0061, 0x0020, 0x006d, 0x0069, 0x0073, 0x0065, 0x0020, 0x00e0, 0x0020, 0x006a, 0x006f, 0x0075, 0x0072, 0x00a0, 0x003f, 0x0000};
			return c_msgWarning;
		}
	case 0x0e:
		{
			// Hungarian
			const TCHAR c_msgWarning[] = {0x0041, 0x0020, 0x0072, 0x0065, 0x006e, 0x0064, 0x0073, 0x007a, 0x0065, 0x0072, 0x0065, 0x0064, 0x0062, 0x0151, 0x006c, 0x0020, 0x0068, 0x0069, 0x00e1, 0x006e, 0x0079, 0x007a, 0x0069, 0x006b, 0x0020, 0x0065, 0x0067, 0x0079, 0x0020, 0x0066, 0x006f, 0x006e, 0x0074, 0x006f, 0x0073, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x0020, 0x0066, 0x0072, 0x0069, 0x0073, 0x0073, 0x00ed, 0x0074, 0x00e9, 0x0073, 0x002e, 0x000d, 0x000a, 0x0053, 0x007a, 0x0065, 0x0072, 0x0065, 0x0074, 0x006e, 0x00e9, 0x006c, 0x0020, 0x0066, 0x0072, 0x0069, 0x0073, 0x0073, 0x00ed, 0x0074, 0x0065, 0x006e, 0x0069, 0x003f, 0x0000};
			return c_msgWarning;
		}
	/*case 0x12:
		{
			// Korean
			const TCHAR c_msgWarning[] = {0xc0ac, 0xc6a9, 0xc790, 0xc758, 0x0020, 0xc2dc, 0xc2a4, 0xd15c, 0xc5d0, 0x0020, 0xc911, 0xc694, 0xd55c, 0x0020, 0xc5c5, 0xb370, 0xc774, 0xd2b8, 0xac00, 0x0020, 0xc124, 0xce58, 0xb418, 0xc9c0, 0x0020, 0xc54a, 0xc558, 0xc2b5, 0xb2c8, 0xb2e4, 0x002e, 0x000d, 0x000a, 0xc5c5, 0xb370, 0xc774, 0xd2b8, 0xd558, 0xaca0, 0xc2b5, 0xb2c8, 0xae4c, 0x003f, 0x0000};
			return c_msgWarning;
		}*/
	case 0x19:
		{
			// Russian
			const TCHAR c_msgWarning[] = {0x0412, 0x0020, 0x0432, 0x0430, 0x0448, 0x0435, 0x0439, 0x0020, 0x0441, 0x0438, 0x0441, 0x0442, 0x0435, 0x043c, 0x0435, 0x0020, 0x043d, 0x0435, 0x0020, 0x0443, 0x0441, 0x0442, 0x0430, 0x043d, 0x043e, 0x0432, 0x043b, 0x0435, 0x043d, 0x043e, 0x0020, 0x0432, 0x0430, 0x0436, 0x043d, 0x043e, 0x0435, 0x0020, 0x043e, 0x0431, 0x043d, 0x043e, 0x0432, 0x043b, 0x0435, 0x043d, 0x0438, 0x0435, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x002e, 0x000d, 0x000a, 0x0423, 0x0441, 0x0442, 0x0430, 0x043d, 0x043e, 0x0432, 0x0438, 0x0442, 0x044c, 0x0020, 0x0435, 0x0433, 0x043e, 0x0020, 0x0441, 0x0435, 0x0439, 0x0447, 0x0430, 0x0441, 0x003f, 0x0000};
			return c_msgWarning;
		}
	/*case 0x1e:
		{
			// Thai
			const TCHAR c_msgWarning[] = {0x0e23, 0x0e30, 0x0e1a, 0x0e1a, 0x0e02, 0x0e2d, 0x0e07, 0x0e04, 0x0e38, 0x0e13, 0x0e02, 0x0e32, 0x0e14, 0x0e01, 0x0e32, 0x0e23, 0x0e2d, 0x0e31, 0x0e1e, 0x0e40, 0x0e14, 0x0e17, 0x0e27, 0x0e34, 0x0e19, 0x0e42, 0x0e14, 0x0e27, 0x0e2a, 0x0e4c, 0x0e0a, 0x0e38, 0x0e14, 0x0e2a, 0x0e33, 0x0e04, 0x0e31, 0x0e0d, 0x000d, 0x000a, 0x0e04, 0x0e38, 0x0e13, 0x0e15, 0x0e49, 0x0e2d, 0x0e07, 0x0e01, 0x0e32, 0x0e23, 0x0e08, 0x0e30, 0x0e2d, 0x0e31, 0x0e1e, 0x0e40, 0x0e14, 0x0e17, 0x0e2b, 0x0e23, 0x0e37, 0x0e2d, 0x0e44, 0x0e21, 0x0e48, 0x003f, 0x0000};
			return c_msgWarning;
		}*/
	case 0x1f:
		{
			// Turkish
			const TCHAR c_msgWarning[] = {0x0053, 0x0069, 0x0073, 0x0074, 0x0065, 0x006d, 0x0069, 0x006e, 0x0069, 0x007a, 0x0064, 0x0065, 0x0020, 0x00f6, 0x006e, 0x0065, 0x006d, 0x006c, 0x0069, 0x0020, 0x0062, 0x0069, 0x0072, 0x0020, 0x0057, 0x0069, 0x006e, 0x0064, 0x006f, 0x0077, 0x0073, 0x0020, 0x0067, 0x00fc, 0x006e, 0x0063, 0x0065, 0x006c, 0x006c, 0x0065, 0x006d, 0x0065, 0x0073, 0x0069, 0x0020, 0x0065, 0x006b, 0x0073, 0x0069, 0x006b, 0x002e, 0x000d, 0x000a, 0x0047, 0x00fc, 0x006e, 0x0063, 0x0065, 0x006c, 0x006c, 0x0065, 0x006d, 0x0065, 0x006b, 0x0020, 0x0069, 0x0073, 0x0074, 0x0069, 0x0079, 0x006f, 0x0072, 0x0020, 0x006d, 0x0075, 0x0073, 0x0075, 0x006e, 0x0075, 0x007a, 0x003f, 0x0000};
			return c_msgWarning;
		}
	}				
}
#endif


void
CAutoRunCDApp::RunButtonEvent()
{
	// perform action assigned to clicked button
	CSEMGuard guard(SEM_FAILCRITICALERRORS);
	switch( m_ucCurButtonID )
	{
	case BID_INSTALL:
		{
			tstring strInstallCmd;
			if( false != ProductRegistry::GetInstallCmd( strInstallCmd ) )
			{
				try
				{
					if (FindFile(strInstallCmd.c_str()))
					{
#if 0
						bool visitKB940105UpdatePage(false);
						if (IsVistaKB940105Required())
						{
#	ifdef UNICODE
							if (MessageBox(m_hWnd, GetVistaKB940105WarningText().c_str(), GetStringFromResource(m_hInstance, IDS_APP_TITLE).c_str(), MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON1) == IDYES)
#	endif
								visitKB940105UpdatePage = true;
						}
						ShellExecute(0, 0, strInstallCmd.c_str(), 0, 0, SW_SHOWNORMAL);
						if (visitKB940105UpdatePage)
							ShellExecute(0, 0, _T("http://support.microsoft.com/kb/940105"), 0, 0, SW_SHOWNORMAL);
#else
						SHELLEXECUTEINFO info;
						memset(&info, 0, sizeof(info));
						info.cbSize = sizeof(info);
						info.lpFile = strInstallCmd.c_str();
						info.nShow = SW_SHOWNORMAL;
						info.fMask = SEE_MASK_NOCLOSEPROCESS;
						ShellExecuteEx(&info);

						if (info.hProcess)
						{
							DWORD exitCode(0);
							while (true)
							{
								if (!GetExitCodeProcess(info.hProcess, &exitCode) || exitCode != STILL_ACTIVE)
									break;
								Sleep(500);
							}
						}

						if (IsVistaKB940105Required())
						{
#	ifdef UNICODE
							if (MessageBox(m_hWnd, GetVistaKB940105WarningText().c_str(), GetStringFromResource(m_hInstance, IDS_APP_TITLE).c_str(), 
								MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON1) == IDYES)
#	endif
								ShellExecute(0, 0, _T("http://support.microsoft.com/kb/940105"), 0, 0, SW_SHOWNORMAL);
						}
#endif
					}
					else
						ReportError();
				}
				catch( ... )
				{
				}
			}
			break;
		}
	case BID_PLAY:
		{
			tstring strLaunchCmd;
			tstring strLaunchWorkingDir;
			if( false != ProductRegistry::GetLaunchCmd( strLaunchCmd, strLaunchWorkingDir ) )
			{
				try
				{
					SetCurrentDirectory(strLaunchWorkingDir.c_str());
					ShellExecute(0 , 0, strLaunchCmd.c_str(), 0, 0,SW_SHOWNORMAL);
					//ReportError();
				}
				catch( ... )
				{
				}
			}
			break;
		}
	case BID_README:
		{
			tstring strReadmePath;
			if( false != ProductRegistry::GetReadmePath( strReadmePath ) )
			{
				try
				{
					if (FindFile(strReadmePath.c_str()))
						ShellExecute(0, 0, strReadmePath.c_str(), 0, 0, SW_SHOWNORMAL);
					else
						ReportError();
				}
				catch( ... )
				{
				}
			}
			break;
		}
	case BID_REGISTER:
		{
			tstring strRegisterLink; 
			if( false != ProductRegistry::GetRegWebLink( strRegisterLink ) )
			{
				try
				{
					ShellExecute(0, 0, strRegisterLink.c_str(), 0, 0,SW_SHOWNORMAL);
					//ReportError();
				}
				catch( ... )
				{
				}
			}
			break;
		}
	case BID_WEB:
		{
			tstring strWeblinkPath;
			if( false != ProductRegistry::GetWebLinkPath( strWeblinkPath ) )
			{
				try
				{
					ShellExecute(0, 0, strWeblinkPath.c_str(), 0, 0, SW_SHOWNORMAL);
					//ReportError();
				}
				catch( ... )
				{
				}
			}
			break;
		}
	case BID_QUIT:
	default:
		{
			break;
		}
	}
}



void 
CAutoRunCDApp::OnSize( unsigned int uiType, unsigned short usWidth, unsigned short usHeight )
{
	if( SIZE_RESTORED == uiType && 0 != m_pkBackBuffer )
	{
		// reset internal states and redraw window content when window is restored from minimized state
		m_bLeftMouseButtonPressed = false;
		m_ucCurButtonID = BID_NOBUTTON;
		OnDraw();
	}
}



LRESULT 
CAutoRunCDApp::OnMouseActivate()
{
	// update LBM state when window is reactivated and redraw window content
	m_bLeftMouseButtonPressed = true;
	OnDraw();

	return( MA_ACTIVATE );
}



LRESULT 
CAutoRunCDApp::WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	// our window proc
	switch( message ) 
	{
	case WM_KEYDOWN:
		{
			OnKeyDown( (unsigned int) wParam, (unsigned int) LOWORD( lParam ), (unsigned int) HIWORD( lParam ) );
			break;
		}
	case WM_MOUSEMOVE:
		{
			OnMouseMove( (short) LOWORD( lParam ), (short) HIWORD( lParam ), (unsigned int) wParam );
			break;
		}
	case WM_LBUTTONDOWN:
		{
			OnLButtonDown( (short) LOWORD( lParam ), (short) HIWORD( lParam ) );
			break;
		}
	case WM_LBUTTONUP:
		{
			OnLButtonUp();
			break;
		}
	case WM_ERASEBKGND:
		{			
			break;
		}
	case WM_PAINT:
		{
			OnPaint();
			break;
		}
	case WM_TIMER:
		{
			OnTimer( (unsigned int) wParam );
			break;
		}
	case WM_CLOSE:
		{
			OnClose();
			break;
		}
	case WM_SETCURSOR:
		{		
			return( 1 );
		}
 	case WM_SIZE:
		{
			OnSize( (unsigned int) wParam, (unsigned short) LOWORD( lParam ), (unsigned short) HIWORD( lParam ) );
			break;
		}
	case WM_MOUSEACTIVATE:
		{
			return( OnMouseActivate() );
		}
	//case WM_USER_MEDIACHANGED:
	//	{
	//		return OnMediaChanged(wParam, lParam);
	//	}
	default:
		{
			return( DefWindowProc( hWnd, message, wParam, lParam ) );
		}
	}
	return( 0 );
}


//void CAutoRunCDApp::InitMediaChangeMonitor()
//{
//	LPITEMIDLIST ppidl;
//	if (SHGetSpecialFolderLocation(m_hWnd, CSIDL_DESKTOP, &ppidl) == NOERROR)
//	{
//		SHChangeNotifyEntry shCNE;
//		shCNE.pidl = ppidl;
//		shCNE.fRecursive = TRUE;
//
//		m_mediaChangeNotificationID = SHChangeNotifyRegister(m_hWnd, SHCNE_DISKEVENTS, 
//			SHCNE_DRIVEADD | SHCNE_DRIVEREMOVED | SHCNE_MEDIAINSERTED | SHCNE_MEDIAREMOVED, WM_USER_MEDIACHANGED, 1, &shCNE);
//	}
//}
//
//
//void CAutoRunCDApp::ShutdownMediaChangeMonitor()
//{
//	if (m_mediaChangeNotificationID)
//		SHChangeNotifyDeregister(m_mediaChangeNotificationID);
//}
//
//
//struct SHNOTIFYSTRUCT
//{
//	DWORD_PTR dwItem1;
//	DWORD_PTR dwItem2;
//};
//
//
//static tstring GetPathFromPIDL(DWORD_PTR pidl)
//{
//	TCHAR path[MAX_PATH];
//	path[0] = L'\0';
//	if (!SHGetPathFromIDList((ITEMIDLIST*) pidl, path))
//		path[0] = L'\0';
//	return path;
//}
//
//
//int CAutoRunCDApp::OnMediaChanged(WPARAM wParam, LPARAM lParam)
//{
//	SHNOTIFYSTRUCT* shns = (SHNOTIFYSTRUCT*) wParam;
//	tstring strPath;
//	switch (lParam)
//	{
//	case SHCNE_MEDIAINSERTED:
//	case SHCNE_DRIVEADD:
//		{
//			strPath = GetPathFromPIDL(shns->dwItem1);
//			break;
//		}
//	case SHCNE_MEDIAREMOVED:
//	case SHCNE_DRIVEREMOVED:
//		{
//			strPath = GetPathFromPIDL(shns->dwItem1);
//			break;
//		}		
//	}
//
//	char out[256];
//	sprintf(out, "0x%p, %ls", lParam, strPath.c_str());
//	MessageBoxA(0, out, 0, MB_OK);
//
//	return 0;
//}