#include "notifyicon.h"
#include "resource.h"
#include "clientthread.h"
#include <assert.h>

CNotifyIcon * CNotifyIcon::m_singleton;

struct CNotifyIcon::SNotifyIconData : public NOTIFYICONDATA
{
	SNotifyIconData( HWND hwnd )
	{
		memset( this, 0, sizeof(NOTIFYICONDATA) );
		cbSize = sizeof(NOTIFYICONDATA);
		hWnd = hwnd;
		uID = uidIcon;
	}

	void AddIcon( const CIconHandle& hdl )
	{
		uFlags |= NIF_ICON;
		hIcon = GetHIcon(hdl);
	}

	void AddTip( const std::string& text )
	{
		uFlags |= NIF_TIP;
		String( szTip, sizeof(szTip), text );
	}

	void AddInfo( const std::string& title, const std::string& info, DWORD flags, UINT timeout )
	{
		uFlags |= NIF_INFO;
		String( szInfoTitle, sizeof(szInfoTitle), title );
		String( szInfo, sizeof(szInfo), info );
		dwInfoFlags = flags;
		uTimeout = timeout;
	}

	void AddMessage( UINT msg )
	{
		uFlags |= NIF_MESSAGE;
		uCallbackMessage = msg;
	}

	void Notify( DWORD msg )
	{
		if (!Shell_NotifyIcon( msg, this ))
			throw std::runtime_error("Shell_NotifyIcon failed");
	}

private:
	void String( char * ptr, size_t sz, std::string text )
	{
		if (text.size()+1 > sz)
		{
			text.resize(sz-3-1);
			text += "...";
		}
		strcpy( ptr, text.c_str() );
	}
};

CNotifyIcon::CNotifyIcon(HWND hwnd) :
	m_notReady(IDI_NOTREADY),
	m_contextMenu(IDR_CONTEXTMENU),
	m_hwnd(hwnd)
{
	assert( !m_singleton );
	m_singleton = this;

	SNotifyIconData nd(m_hwnd);
	nd.AddIcon( m_notReady );
	nd.AddTip( "Blow: File distributer" );
	nd.AddMessage( msgIcon );
	nd.Notify( NIM_ADD );

	m_pThread.reset(new CClientThread(this));
}

void CNotifyIcon::Info( const std::string& title, const std::string& msg, DWORD flags, UINT timeout /* = 10000  */)
{
	SNotifyIconData nd(m_hwnd);
	nd.AddInfo( title, msg, flags, timeout );
	nd.Notify( NIM_MODIFY );
}

CNotifyIcon::~CNotifyIcon()
{
	m_pThread.reset();

	SNotifyIconData nd(m_hwnd);
	nd.Notify( NIM_DELETE );

	assert( m_singleton );
	m_singleton = NULL;
}

void CNotifyIcon::OnRButtonDown()
{
	SetForegroundWindow( m_hwnd );

	POINT point;
	GetCursorPos(&point);

	TrackPopupMenu(
		GetSubMenu(GetHMenu(m_contextMenu), 0),
		TPM_RIGHTBUTTON,
		point.x, point.y,
		0,
		m_hwnd,
		NULL );

	PostMessage( m_hwnd, WM_NULL, 0, 0 );
}

void CNotifyIcon::OnCommand( int command )
{
	switch (command)
	{
	case ID_QUIT:
		DestroyWindow(m_hwnd);
		break;
	}
}

void CNotifyIcon::OnUpdateNotification()
{
	CMutex::Lock lock(m_mutex);
	SNotifyIconData nd(m_hwnd);
	nd.AddInfo( m_notifyTitle, m_notifyText, NIF_INFO, 10000 );
	nd.Notify( NIM_MODIFY );
}

void CNotifyIcon::OnUpdateTip()
{
	CMutex::Lock lock(m_mutex);
	SNotifyIconData nd(m_hwnd);
	nd.AddTip( m_tip );
	nd.Notify( NIM_MODIFY );
}

void CNotifyIcon::SetNotification( const std::string& title, const std::string& text )
{
	CMutex::Lock lock( m_mutex );
	m_notifyTitle = title;
	m_notifyText = text;
	PostWindowMessage( msgUpdateNotification );
}

void CNotifyIcon::SetTip( const std::string& tip )
{
	CMutex::Lock lock( m_mutex );
	if (m_tip == tip)
		return;
	m_tip = tip;
	PostWindowMessage( msgUpdateTip );
}
