#ifndef SOCK_H
#define SOCK_H

#include <winsock2.h>
#include <ws2tcpip.h>
#include <windows.h>
#include <stdexcept>
#include <string>

class CSockException : public std::runtime_error
{
public:
	CSockException( int code = WSAGetLastError() ) : std::runtime_error(Format(code))
	{
	}

private:
	static std::string Format( int code )
	{
		char buf[1024];
		sprintf( buf, "WinSock error %d", code );
		return buf;
	}
};

class CConnReset : public CSockException
{
public:
	CConnReset() : CSockException(WSAECONNRESET)
	{
	}
};

class CWinSock
{
public:
	CWinSock()
	{
		if (int err = WSAStartup(MAKEWORD(2,0), &m_data))
			throw CSockException(err);
	}

	~CWinSock()
	{
		WSACleanup();
	}

private:
	WSADATA m_data;
};

class CSock
{
public:
	CSock()
	{
		m_hdl = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );
		if (m_hdl == INVALID_SOCKET)
			throw CSockException();
	}
	void Bind( unsigned short port )
	{
		sockaddr_in addr;
		memset( &addr, 0, sizeof(addr) );
		addr.sin_family = AF_INET;
		addr.sin_port = htons(port);
		if (0 != bind( m_hdl, (sockaddr*)&addr, sizeof(addr) ))
			throw CSockException();
	}
	template <class T>
	void SetSockOpt( int level, int option, const T& value )
	{
		if (0 != setsockopt( m_hdl, level, option, (const char*) &value, sizeof(T) ))
			throw CSockException();
	}
	~CSock()
	{
		closesocket( m_hdl );
	}
	void Send( unsigned addr, unsigned short port, const void * data, size_t length )
	{
		sockaddr_in saddr;
		memset( &saddr, 0, sizeof(saddr) );
		saddr.sin_family = AF_INET;
		saddr.sin_port = htons(port);
		saddr.sin_addr.S_un.S_addr = htonl(addr);
		if (length != sendto( m_hdl, (const char *)data, int(length), 0, (sockaddr*)&saddr, sizeof(saddr) ))
			throw CSockException();
	}
	size_t Receive( unsigned& addr, unsigned short& port, void * data, size_t length )
	{
		sockaddr_in saddr;
		int len = int(sizeof(saddr));
		int rec = recvfrom( m_hdl, (char*)data, int(length), 0, (sockaddr*)&saddr, &len );
		if (rec <= 0 || len != sizeof(saddr))
			throw CSockException();
		addr = htonl(saddr.sin_addr.S_un.S_addr);
		port = htons(saddr.sin_port);
		return rec;
	}
	bool Select( const timeval * timeout, bool input, bool output )
	{
		fd_set in;
		fd_set out;
		FD_ZERO(&in);
		FD_ZERO(&out);
		FD_SET(m_hdl, &in);
		FD_SET(m_hdl, &out);

		int err = select( int(m_hdl)+1, input? &in : NULL, output? &out : NULL, NULL, timeout );
		if (err == SOCKET_ERROR)
			throw CSockException();
		return err != 0;
	}

private:
	SOCKET m_hdl;
};

#endif
