#include "StdAfx.h"
#include "chartdata.h"
#include "ASCIIFile.h"					// CASCIIFile


DWORD String2DWORD( const char *inszName )
{
	assert(strlen(inszName)==4);

	return( (((DWORD)inszName[3])<<24) | (((DWORD)inszName[2])<<16) | (((DWORD)inszName[1])<<8) | ((DWORD)inszName[0]) );
}




template <class T>
T ParseData( char * &inoutpPtr )
{
	assert(inoutpPtr);
	DWORD dwSize=sizeof(T);

	T *pPtr=(T *)inoutpPtr;

	inoutpPtr+=dwSize;
	return(*pPtr);
}


string ParseCString( char * &inoutpPtr )
{
	assert(inoutpPtr);

	DWORD dwLen=(DWORD)strlen(inoutpPtr);

	string sRet(inoutpPtr);

	inoutpPtr+=(dwLen+1);

	return(sRet);
}



// constructor
CChartData::CChartData( void )
{
	m_iMaxPosition=0;
	m_iMaxValue=0;
}

// destructor
CChartData::~CChartData( void )
{
	vector<CChartLayer *>::iterator it;

	for(it=m_Layers.begin();it!=m_Layers.end();++it)
			delete (*it);

}


void CChartData::Draw( Graphics *inGraph, const CChartDisplayInfo &inInfo )
{
	vector<CChartLayer *>::iterator it;
	int iNo=0;

	switch(inInfo.m_iRenderMode)
	{
		case 0:		// lines
		{
			for(it=m_Layers.begin();it!=m_Layers.end();++it,iNo++)
				(*it)->Draw(inGraph,iNo,inInfo,m_iMaxValue);			
			break;
		}
/*
		case 1:		// 1=summed together
		{
			vector<>

			inGraph->Get

			for(it=m_Layers.begin();it!=m_Layers.end();++it,iNo++)
				(*it).Draw(inGraph,iNo,inInfo,m_iMaxValue);			
			break;
		}
*/
	}
}


// optimizable but not neccessary
void CChartData::CChartLayer::Draw( Graphics *inGraph, int iniNo, const CChartDisplayInfo &inInfo, const int iniMaxValue )
{
	if(m_bHidden)return;

	vector<CChartElement>::iterator it;

	bool bFirstOne=true;

	Color	col(m_dwColor|0xff000000);

	Pen pen(col, 1);
	SolidBrush brush(col);
//	Graphics myGraphics(inHdc);

//	myGraphics.SetSmoothingMode(SmoothingModeHighQuality);			// Antialiasing


	REAL oldx,oldy;

	for(it=m_Elements.begin();it!=m_Elements.end();++it)
	{
		REAL x,y;

		x=(REAL)(((*it).m_dwPosition*inInfo.GetXScale16()))/16;
		y=((iniMaxValue - (REAL)(*it).m_dwValue)*inInfo.GetYScale16k())/(16*1024);

		// lines with non 90 degree angle are drawn
//		if(!bFirstOne)
//			inGraph->DrawLine(&pen,oldx,oldy,x,y);

		// only horizontal and vertical lines
		if(!bFirstOne)
		{
			inGraph->DrawLine(&pen,oldx,oldy,x,oldy);	// horizontal
			inGraph->DrawLine(&pen,x,oldy,x,y);				// vertical
		}

		oldx=x;oldy=y;

		bFirstOne=false;
	}

//	SelectObject(inHdc,oldpen);
}

void CChartData::CChartLayer::GetMaximum( int &outiMaxPosition, int &outiMaxValue )
{
	vector<CChartElement>::iterator it;

	outiMaxPosition=0;
	outiMaxValue=0;

	for(it=m_Elements.begin();it!=m_Elements.end();++it)
	{
		DWORD x,y;

		x=(*it).m_dwPosition;
		y=(*it).m_dwValue;

		if((int)x>outiMaxPosition)outiMaxPosition=x;
		if((int)y>outiMaxValue)outiMaxValue=y;
	}
}



void CChartData::GetDimensions( int &outiWidth, int &outiHeight, const CChartDisplayInfo &inInfo ) const
{
	outiWidth=(m_iMaxPosition*inInfo.GetXScale16())/16;
	outiHeight=(m_iMaxValue*inInfo.GetYScale16k())/(16*1024);
}


void CChartData::GetDimensions( int &outiWidth, int &outiHeight ) const
{
	outiWidth=m_iMaxPosition;
	outiHeight=m_iMaxValue;
}


void CChartData::RecalculateDimensions( void )
{
	vector<CChartLayer *>::iterator it;

	m_iMaxPosition=0;
	m_iMaxValue=0;

	for(it=m_Layers.begin();it!=m_Layers.end();++it)
	{
		int x,y;

		(*it)->GetMaximum(x,y);

		if(x>m_iMaxPosition)m_iMaxPosition=x;
		if(y>m_iMaxValue)m_iMaxValue=y;
	}

	m_iMaxValue+=m_iMaxValue/10;		// 10% extra for easier browsing
}
/*
//
void CChartData::GenerateDebugData( void )
{
	m_Layers.push_back(new CChartLayer("Layer1 Red",0xff0000));

	CChartLayer *p1=m_Layers.back();

	p1->GenerateDebugData();

	m_Layers.push_back(new CChartLayer("Layer2 Green",0x00ff00));

	CChartLayer *p2=m_Layers.back();

	p2->GenerateDebugData();

	RecalculateDimensions();
}
*/


// uded after loading to put the frametime in all layers
void CChartData::CChartLayer::GeneratePositionFromSumOfGivenValues( CChartData::CChartLayer &refLayer, const DWORD indwDivideBy )
{
	assert(refLayer.m_Elements.size()==m_Elements.size());

	LONGLONG llPosition=0;

	for(DWORD i=0;i<m_Elements.size();i++)
	{
		m_Elements[i].m_dwPosition=(DWORD)(llPosition/(LONGLONG)indwDivideBy);
		llPosition+=refLayer.m_Elements[i].m_dwValue;
	}
}

/*
void CChartData::CChartLayer::GenerateDebugData( void )
{
	DWORD dwPos=0;

	for(int i=0;i<2000;i++)
	{
		dwPos+=(rand()%10)+1;
		DWORD dwValue=((i/100)*1000001*10)/10/19;		// 0..1000000 = 0..1 sec

		assert(dwValue>=0 && dwValue<=1000010);

		m_Elements.push_back(CChartElement(dwPos,dwValue));
	}
}
*/

// BGR, not RGB
static const DWORD g_Colors[18]=
{
	0x0000ff,0x00ff00,0xff0000,0xff00ff,			// blue,green,red,pink
	0xff8800,0x88ffff,0x88ff00,0x88ff88,
	0x00ffff,0xff88ff,0x00ff88,0xff0088,
	0x0088ff,0xffff00,0x8800ff,0xff8888,
	0x8888ff,0xffff88
};

void CChartData::CChartLayer::CycleColor()
{
	for(int i=0;i<18;i++)
		if(g_Colors[i]==m_dwColor)break;

	m_dwColor=g_Colors[(i+1)%18];
}

bool CChartData::Load( const char *inszFilePath )
{
	m_Layers.clear();

	CASCIIFile file;

	if(file.IO_LoadASCIIFile(inszFilePath))
	if(file.GetDataSize()>=12)									// minimum the Header
	{
		char *ptr=file.GetDataPtr();

/*
		header:

		8 bytes: "FPROFDAT"
			4 bytes: M: size (in ints) of each trace. all ints in little endian format
			4 bytes: N: number of traces

			followed by the traces themselves, M in total, each:

		strlen+1: name of trace, "__frametime" for the frame time
			N*4:      frame timings, in microseconds (1000000=1sec)
*/



		DWORD dwMagic1 = ParseData<DWORD>(ptr);		if(dwMagic1!=String2DWORD("FPRO"))return(false);
		DWORD dwMagic2 = ParseData<DWORD>(ptr);		if(dwMagic2!=String2DWORD("FDAT"))return(false);
		DWORD dwVersion= ParseData<DWORD>(ptr);		if(dwVersion!=2)return(false);

		DWORD dwLayerSize = ParseData<DWORD>(ptr);
		DWORD dwLayers = ParseData<DWORD>(ptr);

		CChartLayer *pFrameTimeLayer=0;

		// for all layers read data
		for(DWORD dwL=0;dwL<dwLayers;dwL++)
		{
			// create layer
			string sLayerName=ParseCString(ptr);
//			DWORD sLayerColor=ParseData<DWORD>(ptr);
			DWORD sLayerColor=g_Colors[dwL%18];

			CChartLayer *p1=new CChartLayer(sLayerName,sLayerColor);
			m_Layers.push_back(p1);

			if("__frametime"==sLayerName)
				pFrameTimeLayer=m_Layers.back();

			// load layer data
			if(!p1->Parse(ptr,dwLayerSize,dwVersion))
				return(false);											// load layer failed
		}

		// for all layers adjust frametime
		if(pFrameTimeLayer)
		for(DWORD dwL=0;dwL<dwLayers;dwL++)
		{
			CChartLayer *p1=GetLayerNo(dwL);

			p1->GeneratePositionFromSumOfGivenValues(*pFrameTimeLayer,1000);
		}

		RecalculateDimensions();
		return(true);
	}

	return(false);														// open failed
}

// in ms, rounded up
string CChartData::CChartLayer::GetNameAndStats( void ) const
{
	char str[1024];

//	 in microseconds (1000000=1sec)

	sprintf(str,"%.2f\t%.1f\t%.1f\t%s",
		(m_fAvg)/1000.0f,
		((m_dwMin)/100) * 0.1f,				// rounded down
		((m_dwMax+99)/100) * 0.1f,		// rounded up
		m_sName.c_str());

	return str;
}


DWORD CChartData::CChartLayer::GetIndexFromPosition( const int iniPos )
{
	for(DWORD i=1;i<m_Elements.size();i++)
	{
		if(m_Elements[i].m_dwPosition>(DWORD)iniPos)
			return i-1;
	}

	return 0;
}

DWORD CChartData::CChartLayer::GetValueInMicroSecAt( const int iniPos ) const
{
	return(m_Elements[iniPos].m_dwValue);
}

//!
DWORD CChartData::GetSumInMicroSecAt( const int iniPos ) const
{
	vector<CChartLayer *>::const_iterator it;

	DWORD dwRet=0;

	for(it=m_Layers.begin();it!=m_Layers.end();++it)
	{
		if((*it)->m_sName=="__frametime")
			return (*it)->GetValueInMicroSecAt(iniPos);
	}

	assert(0);		// no __frametime in the layers
	return 0;
}


string CChartData::CChartLayer::GetNameAndStatsAtIndex( const DWORD indwIndex, const float infSumInMS )
{
	char str[1024];

//	 in microseconds (1000000=1sec)

	float fRelative=0;

	if(m_Elements[indwIndex].m_wCount!=0)
		fRelative=(float)( (m_Elements[indwIndex].m_dwValue*100) / m_Elements[indwIndex].m_wCount / 1000 )*0.01f;

	float fOnePercent=infSumInMS*0.01f;

/*
	sprintf(str,"\t%.2f\t%.1f\t%s (%d)",
		fRelative,
		((m_Elements[indwIndex].m_dwValue)/100) * 0.1f,
		m_sName.c_str(),
		m_Elements[indwIndex].m_wCount);
*/
	sprintf(str,"%.2f\t%.1f\t%.1f%%\t%s (%d)",
		fRelative,
		((m_Elements[indwIndex].m_dwValue)/100) * 0.1f,
		(m_Elements[indwIndex].m_dwValue*0.001f)/fOnePercent,
		m_sName.c_str(),
		m_Elements[indwIndex].m_wCount);

	return str;
}




bool CChartData::CChartLayer::Parse( char * &inoutpPtr, const DWORD indwLayerSize, const DWORD indwVersion )
{
	m_Elements.reserve(indwLayerSize);

	m_dwMin=0xffffffff;
	m_dwMax=0;

	LONGLONG llSum=0;

	for(DWORD dwI=0;dwI<indwLayerSize;dwI++)
	{
		DWORD dwPos=dwI;
		DWORD dwValue=ParseData<DWORD>(inoutpPtr);								// value in milliseconds
		
		unsigned short wCount = ParseData<unsigned short>(inoutpPtr);		// to distinct between slow function or often called functions

		if(dwValue<m_dwMin)m_dwMin=dwValue;
		if(dwValue>m_dwMax)m_dwMax=dwValue;

		llSum+=dwValue;

		m_Elements.push_back(CChartElement(dwPos,dwValue,wCount));
	}

	LONGLONG llCount=indwLayerSize;

	m_fAvg=((llSum*100)/indwLayerSize)*0.01f;		// 2 digits after comma

	return(true);
}



CChartData::CChartLayer *CChartData::GetLayerNo( const int iniNo )
{
	if(iniNo>=(int)m_Layers.size())return(0);

	return(m_Layers[iniNo]);
}