// (c) 2000 by Zachary Booth Simpson
// This code may be freely used, modified, and distributed without any license
// as long as the original author is noted in the source file and all
// changes are made clear disclaimer using a "MODIFIED VERSION" disclaimer.
// There is NO warranty and the original author may not be held liable for any damages.
// http://www.totempole.net


/*
********************************** MODIFIED VERSION *************************************
	Modification History:
			01.03.2006 - Tamas Schlagl: Made it confortable with VC2005
********************************** MODIFIED VERSION *************************************
*/

#include "stdafx.h"
#include "dtime.h"

//#pragma warning(disable: 4312 4267 4244)

// Globals
//------------------------------------------------------------------------

#ifdef WIN32
	// These two hacky variables are used to coordinate the
	// GetSystemClock and the multimedia timer under Windows.
	// What a pain in the ass.
	unsigned int dTimeWin32MilsAtStart = 0;
	unsigned int dTimeWin32SecsAtStart = 0;
#endif

int dTimeFrameCount = 0;
int dTimeMasterDiffMils = 0;
DTime dTimeMasterFrameTime = 0.0;
DTime dTimeLocalFrameTime = 0.0;
DTime dTimeMasterDT = 0.0;
float dTimeMasterDTF = 0.0f;
DTime dTimeLocalDT = 0.0;
float dTimeLocalDTF = 0.0f;
DTime dTimeStart = 0.0;
int dTimePaused = 0;
DTime dTimeWhenPaused = 0.0;
DTime dTimePausedAccumulator = 0.0;
double dTimeFPS = 0.0;
double dTimeAvgFPS = 0.0;


// Private Interface Functions
//------------------------------------------------------------------------

void dTimeGetSecsAndMils( unsigned int &secs, unsigned int &mils, int realTime ) {
	// This function hides the OS dependency problems
	assert( dTimeStart > 0.0 );
		// You must initialize with dTimeInit() before making any other dTime calls

	int _masterDiffMils = realTime ? 0 : dTimeMasterDiffMils;

	#ifdef WIN32
		__int64 _mils = timeGetTime() - dTimeWin32MilsAtStart;
		_mils -= _masterDiffMils;
		_mils += (__int64)dTimeWin32SecsAtStart * (__int64)1000;
		secs = (unsigned int)_mils / 1000;
		mils = (unsigned int)_mils % 1000;
	#else
		struct timeval tv;
		struct timezone tz;
		gettimeofday( &tv, &tz );

		int _mils = tv.tv_usec/1000;
		secs = tv.tv_sec;

		_mils -= _masterDiffMils;
		if( _mils < 0 ) {
			secs += (_mils/1000)-1;
			mils = 1000-(-_mils % 1000);
		}
		else {
			secs += _mils / 1000;
			mils = _mils % 1000;
		}
	#endif
}	

// Public Interface Functions
//------------------------------------------------------------------------

DTime dTimeGetMaster() {
	assert( dTimeStart > 0.0 );
		// You must initialize with dTimeInit() before making any other dTime calls
	assert( dTimeMasterDiffMils == 0 || dTimePausedAccumulator == 0.0 );
		// If the time is slaved to a master clock, you are not allowed to
		// call dTimePauseTime.  Be sure to call dTimeResetPaused() first.
	if( dTimePaused ) {
		return dTimeWhenPaused;
	}
	unsigned int secs, mils;
	dTimeGetSecsAndMils( secs, mils, 0 );
	return DTime( (double)secs + (double)mils/1000.0 );
}

DTime dTimeGetLocal() {
	assert( dTimeStart > 0.0 );
		// You must initialize with dTimeInit() before making any other dTime calls
	unsigned int secs, mils;
	dTimeGetSecsAndMils( secs, mils, 1 );
	return DTime( (double)secs + (double)mils/1000.0 );
}

int dTimeIsPaused() {
	return dTimePaused > 0;
}

void dTimeLatchFrame() {
	assert( dTimeStart > 0.0 );
		// You must initialize with dTimeInit() before making any other dTime calls
	DTime lastMasterFrameTime = dTimeMasterFrameTime;
	DTime lastLocalFrameTime = dTimeLocalFrameTime;

 	dTimeMasterFrameTime = dTimeGetMaster();
 	dTimeLocalFrameTime  = dTimeGetLocal();

	dTimeMasterDT = dTimeMasterFrameTime - lastMasterFrameTime;
	dTimeMasterDTF = (float)dTimeMasterDT;

	dTimeLocalDT = dTimeLocalFrameTime - lastLocalFrameTime;
	dTimeLocalDTF = (float)dTimeLocalDT;

	// Stick local time into the rolling queue for averaging purposes
	static DTime lastFrameTimes[DTIME_FRAMES_TO_AVG];
	lastFrameTimes[ dTimeFrameCount%DTIME_FRAMES_TO_AVG ] = dTimeLocalFrameTime;

	// Set the current fps as long as we have one sample
	if( dTimeFrameCount > 0 ) {
		// It is possible that the system can run so fast that
		// no time elapses and we would get a divide by zero, so
		// we must check for this case.

		// NOTE: -1.0 is a magic value used only when we would divide by zero
		dTimeFPS = -1.0;
		if( dTimeLocalDT > 0.0 ) {
			dTimeFPS = 1.0 / dTimeLocalDT;
		}
	}

	// Set the average as long as we have enough samples
	if( dTimeFrameCount > DTIME_FRAMES_TO_AVG ) {
		double elapsed = dTimeLocalFrameTime -
			lastFrameTimes[ (dTimeFrameCount+1) % DTIME_FRAMES_TO_AVG ]
		;

		// NOTE: -1.0 is a magic value used only when we would divide by zero
		dTimeAvgFPS = -1.0;
		if( elapsed > 0.0 ) {
			dTimeAvgFPS = (double)DTIME_FRAMES_TO_AVG / elapsed;
		}
	}

	dTimeFrameCount++;
}

void dTimeInit() {
	// Unfortunatly, there is no one function which gives both
	// UTC and millisecond accuracy under windows.
	// Thus, we have to combine two seperate clocks by
	// spinning in a loop looking for the second to turn over
	// Yes, it is ugly!

 	#ifdef WIN32
		HANDLE process = GetCurrentProcess();
		DWORD oldPriority = GetPriorityClass( process );
		SetPriorityClass( process, HIGH_PRIORITY_CLASS );

		SYSTEMTIME t;
		GetSystemTime( &t );
		int lastSec = t.wSecond;
		while( 1 ) {
			GetSystemTime( &t );
			if( t.wSecond != lastSec ) {
				dTimeWin32MilsAtStart = timeGetTime();
				dTimeWin32SecsAtStart = (unsigned int)time(NULL);
				break;
			}
		}
		
		SetPriorityClass( process, oldPriority );
	#endif

	dTimeStart = 1.0;	// Set to defeat assert in dTimeGetLocal()
	dTimeStart = dTimeGetLocal();
	dTimeLatchFrame();
} 

void dTimePause() {
	assert( dTimeMasterDiffMils == 0 );
		// If the time is slaved to a master clock, you are not allowed to
		// call dTimePauseTime.  Be sure to call dTimeResetPaused() first.
	if( dTimePaused == 0 ) {
		dTimeWhenPaused = dTimeGetMaster();
	}
	dTimePaused++;
}

void dTimeResume() {
	if( dTimePaused ) {
		dTimePaused--;
		if( !dTimePaused ) {
			dTimePausedAccumulator.dTime += dTimeGetMaster() - dTimeWhenPaused;
		}
	}
}

void dTimeResetPaused() {
	dTimePausedAccumulator = 0.0;
}

void dTimeProcessSleepSecs( int secs ) {
	#ifdef WIN32
		Sleep( secs * 1000 );
	#else
		sleep( secs );
	#endif
}

void dTimeProcessSleepMils( int mils ) {
	#ifdef WIN32
		Sleep( mils );
	#else
		assert( 0 );	// TODO: Use select()
	#endif
}

void dTimeSetMasterDiff( int mils ) {
	dTimeResetPaused();
		// Pause is unallowed when you are slaved to a master clock
	dTimeMasterDiffMils = mils;
}


#ifdef SELFTEST_DTIME

#include "stdio.h"

void main() {
	dTimeInit();

	LocalTimer timer0( 2.0 );
	LocalTimer timer1( 0.5 );
	LocalTimer timer2;
	LocalTimer timer3( 0.1 );

	while( !timer0.isDone() ) {
		dTimeLatchFrame();

		if( timer3.isDone() ) {
			printf( "local=%lf master=%lf\n", dTimeLocalFrameTime-dTimeStart, dTimeMasterFrameTime-dTimeStart );
			timer3.start( 0.1 );
		}

		dTimeProcessSleepMils( 10 );

		if( timer1.isDone() ) {
			printf( "Pausing time\n" );
			dTimePause();
			timer1.stop();
			timer2.start( 0.5 );
		}

		if( timer2.isDone() ) {
			printf( "Resuming time\n" );
			dTimeResume();
			timer2.stop();

			printf( "Setting server difference to 1000ms\n" );
			dTimeSetMasterDiff( 1000 );
				// Pretend that the master clock is 1000 ms different.
		}
	}
}
#endif
