#!/usr/bin/env python
#
# File list generator.
#
# Sascha Demetrio, saschad@crytek.com

VERSION = '0.1'

import sys, os, os.path, string, getopt, re
import xml.parsers.expat

type_list = None
verbose = 0
create_dirlist = False
ignore_parent = False
p4_do_update = False
p4_do_submit = False
p4_comment = None
p4_change = None
p4_file_list = [ ]
fix_case = True
os_name = None

def version(out):
	out.write('GenFilelist version ' + str(VERSION) + '\n')
	out.flush()

def help(out):
	out.write("""Synopsis:
	python GenFilelist.py [(options)] (project-file-list)
Options:
-h|--help
	Display a help screen to the standard output an exit.
-V|--version
	Display the program version and exit.
-v|--verbose
	More verbose output.
-s|--silent
	Don't display info and warning messages.
-t|--types TYPELIST
	A comma separated list of file types. The default is "c,cpp".
-d|--dirlist
	Generate a directory list.
-u|--updatep4
	Update the perforce repository. Note that the tool will call the P4
	command line client, so the client must be set up correctly (i.e. the
	P4CLIENT environment variable must be set to the correct workspace
	name).
--submit
	Automatically submit changes to P4. If this option is omitted, then
	the P4 changelist is created but not submitted.
-l|--list LISTFILE
	Read a list of project files from the specified list file. Multiple
	file lists may be specified.
-c|--comment COMMENT
	The P4 comment to be used for the P4 change. GenFilelist will use a
	sensible default comment if this option is omitted.
""")
	out.flush()

def info(level, message):
  global verbose
  if (level <= verbose):
    sys.stdout.write('GenFilelist: ' + str(message) + '\n')
    sys.stdout.flush()

def error(message):
  sys.stderr.write('GenFilelist: ERROR: ' + str(message) + '\n')
  sys.stderr.flush()

def warning(level, message):
  global verbose
  if (level <= verbose):
    sys.stderr.write('GenFilelist: WARNING: ' + str(message) + '\n')
    sys.stderr.flush()

def normalize(filename, basedir = None):
  global fix_case

  filename = filename.replace('\\', '/')
  filename = re.sub('//+', '/', filename)
  if filename[:2] == './': filename = filename[2:]
  if fix_case:
    if basedir is None: basedir = '.'
    path = filename.split('/')
    try:
      for i in range(0, len(path)):
	if i > 0:
	  dname = basedir + '/' + string.join(path[0 : i - 1], '/')
	else:
	  dname = basedir
	fname = path[i]
	fname_lower = fname.lower()
	for f in os.listdir(dname):
	  f_lower = f.lower()
	  if f_lower == fname_lower:
	    path[i] = f
	    break
    except IOError:
      warning(0, 'Can not normalize path ' + repr(filename) + '!')
      return filename
    filename = string.join(path, '/')
  return filename

def dirname(path):
  index = path.rfind('/')
  if index == -1: return '.'
  return path[:index]

def basename(path):
  index = path.rfind('/')
  if index == -1: return path
  return path[index + 1:]

def filetype(filename):
  index = filename.rfind('.')
  if index == -1: return None
  return filename[index + 1:].lower()

def update(filename, item_list):
  global p4_do_update

  create_file = False
  f = None
  try:
    f = file(filename, 'r')
  except IOError:
    create_file = True
  if create_file:
    old_file = ''
  else:
    old_file = f.read()
    f.close()
  if len(item_list) > 0:
    new_file = string.join(item_list, '\n') + '\n'
  else:
    new_file = ''
  if not create_file and old_file == new_file:
    info(0, 'File ' + repr(filename) + ' unchanged')
    return
  elif create_file:
    info(0, 'Creating ' + repr(filename)
	+ ' (' + str(len(item_list)) + ' items) ...')
  else:
    info(0, 'Updating ' + repr(filename)
	+ ' (' + str(len(item_list)) + ' items) ...')

  if p4_do_update and not create_file: p4_edit(filename)

  f = file(filename, 'w')
  f.write(new_file)
  f.close()

  if p4_do_update and create_file: p4_add(filename)

def process(proj_filename):
  global type_list, create_dirlist, ignore_parent

  name = basename(proj_filename)
  if filetype(name) != 'vcproj':
    warning(0, 'Skipping ' + repr(proj_filename) + ' (not a VCPROJ file)!')
    return
  name = name[:-7]
  dname = dirname(proj_filename)

  info(1, 'Processing ' + repr(proj_filename) + ' [' + name + '] ...')

  # Extract the complete file list.
  parser = xml.parsers.expat.ParserCreate()
  file_list = { }
  dir_list = ([ ], [ ])
  def handler(name, attributes):
    if name == u'File':
      try:
	rel_path = attributes['RelativePath']
      except:
	return
      rel_path = normalize(rel_path, dname)
      if ignore_parent:
	if rel_path.find('..') != -1:
	  warning(1, 'Parent path ' + repr(rel_path) + ' ignored!')
	  return
      type = filetype(basename(rel_path))
      if type not in type_list: return
      if type not in file_list: file_list[type] = ([ ], [ ])
      if rel_path.lower() not in file_list[type][1]:
	file_list[type][0].append(rel_path)
	file_list[type][1].append(rel_path.lower())
	dir = dirname(rel_path)
	if dir.lower() not in dir_list[1]:
	  dir_list[0].append(dir)
	  dir_list[1].append(dir.lower())
  parser.StartElementHandler = handler
  parser.ParseFile(file(proj_filename, 'r'))

  # Update the listing files.
  if create_dirlist:
    list = dir_list[0]
    list.sort()
    update(proj_filename[:-7] + '.dirs', list)
  for type in type_list:
    if type not in file_list: file_list[type] = ([ ], None)
    list = file_list[type][0]
    list.sort()
    stdafx_index = None
    for index in range(0, len(list)):
      name = list[index]
      if (name[-7 - len(type):].lower() == 'stdafx.' + type
	  or name[-10 - len(type):].lower() == 'renderpch.' + type):
	if stdafx_index is not None:
	  warning(2, 'Multiple StdAfx/RenderPCH files in project!')
	else:
	  stdafx_index = index
	break
    if stdafx_index is not None:
      stdafx = list[stdafx_index]
      del list[stdafx_index]
      list.insert(0, stdafx)
    update(proj_filename[:-7] + '.files_' + type, list)

def p4_create_change():
  global p4_comment, p4_change

  assert p4_change is None
  info(1, 'Creating P4 changelist ...')
  desc = os.popen('p4 change -o', 'r').read()
  index = desc.index('\nDescription:')
  desc = (
      desc[:index]
      + '\nDescription:\r\n\t'
      + p4_comment
      + '\r\n\r\nFiles:\r\n\r\n')
  o, i = os.popen2('p4 change -i')
  o.write(desc)
  o.close()
  status = i.read().strip()
  i.close()
  if not status.startswith('Change ') or not status.endswith(' created.'):
    error('Unexpected status response from P4: ' + repr(status))
    sys.exit(1)
  p4_change = int(status[7:-9])
  info(0, 'Using P4 changelist #' + str(p4_change))

def p4_add(filename):
  global p4_change, os_name, p4_file_list

  if p4_change is None: p4_create_change()
  absolute = os.path.abspath(filename)
  if os_name == 'Cygwin':
    if not absolute.startswith('/cygdrive/'):
      error('Internal error: unexpected abspath() '+ repr(absolute))
      sys.exit(1)
    absolute = absolute[10:]
    absolute = absolute[0].upper() + ':' + absolute[1:]
  cmd = 'p4 add -t text -c ' + str(p4_change) + ' "' + str(absolute) + '"'
  info(1, 'P4: ' + cmd)
  f = os.popen(cmd)
  status = f.read().strip()
  exit_code = f.close()
  info(1, 'P4: ' + status)
  if status.find(' opened ') == -1:
    error('Unexpected status response from P4!')
    p4_revert()
    sys.exit(1)
  if exit_code:
    error('P4 add failed with exit code ' + str(exit_code))
    p4_revert()
    sys.exit(1)
  p4_file_list.append(absolute)

def p4_edit(filename):
  global p4_change, os_name, p4_file_list

  if p4_change is None: p4_create_change()
  absolute = os.path.abspath(filename)
  if os_name == 'Cygwin':
    if not absolute.startswith('/cygdrive/'):
      error('Internal error: unexpected abspath() '+ repr(absolute))
      sys.exit(1)
    absolute = absolute[10:]
    absolute = absolute[0].upper() + ':' + absolute[1:]
  failed = False
  cmd = 'p4 edit -c ' + str(p4_change) + ' "' + str(absolute) + '"'
  info(1, 'P4: ' + cmd)
  f = os.popen(cmd)
  status = f.read().strip()
  exit_code = f.close()
  info(1, 'P4: ' + status)
  if status.find(' opened ') == -1:
    error('Unexpected status response from P4!')
    failed = True
  if exit_code:
    error('P4 add failed with exit code ' + str(exit_code))
    failed = True
  if failed:
    p4_reopen(filename)
  else:
    p4_file_list.append(absolute)

def p4_reopen(filename):
  global p4_change, os_name, p4_file_list

  if p4_change is None: p4_create_change()
  absolute = os.path.abspath(filename)
  if os_name == 'Cygwin':
    if not absolute.startswith('/cygdrive/'):
      error('Internal error: unexpected abspath() '+ repr(absolute))
      sys.exit(1)
    absolute = absolute[10:]
    absolute = absolute[0].upper() + ':' + absolute[1:]
  cmd = 'p4 reopen -t text -c ' + str(p4_change) + ' "' + str(absolute) + '"'
  info(1, 'P4: ' + cmd)
  f = os.popen(cmd)
  status = f.read().strip()
  exit_code = f.close()
  info(1, 'P4: ' + status)
  if exit_code:
    error('P4 add failed with exit code ' + str(exit_code))
    p4_revert()
    sys.exit(1)
  p4_file_list.append(absolute)

def p4_submit():
  global p4_change

  if p4_change is None:
    info(0, 'P4 submit skipped, all files are unchanged')
    return
  cmd = 'p4 submit -c ' + str(p4_change)
  info(1, 'P4: ' + cmd)
  f = os.popen(cmd)
  status = f.read().strip()
  exit_code = f.close()
  info(1, 'P4: ' + status)
  if exit_code:
    error('Submit failed with exit code' + str(exit_code))
    p4_revert()
    sys.exit(1)

def p4_revert():
  global p4_change, p4_file_list

  if p4_change is None: return
  if len(p4_file_list) > 0:
    cmd = 'p4 revert -c ' + str(p4_change)
    for filename in p4_file_list: cmd += ' "' + filename + '"'
    info(1, 'P4: ' + cmd)
    f = os.popen(cmd)
    status = f.read().strip()
    exit_code = f.close()
    if exit_code:
      error('Revert failed with exit code ' + str(exit_code))

  cmd = 'p4 change -d ' + str(p4_change)
  info(1, 'P4: ' + cmd)
  f = os.popen(cmd)
  status = f.read().strip()
  exit_code = f.close()
  if exit_code:
    error('Delete changelist failed with exit code ' + str(exit_code))

def main():
  global type_list, verbose, create_dirlist
  global p4_do_update, p4_do_submit, p4_comment
  global os_name

  # Get the OS name.
  os_uname = os.uname()[0]
  if os_uname.lower() == 'linux':
    os_name = 'Linux'
  elif os_uname.lower().startswith('cygwin'):
    os_name = 'Cygwin'
  else:
    os_name = 'Unknown'

  # Parse the command line.
  opts, args = getopt.getopt(
      sys.argv[1:],
      'hVvst:duc:l:',
      [ 'help', 'version', 'verbose', 'silent', 'types=', 'dirlist',
	'updatep4', 'submit', 'comment=', 'list=' ])
  project_list = [ ]
  for a, o in opts:
    if a in ('-h', '--help'):
      version(sys.stdout)
      help(sys.stdout)
      sys.exit(0)
    elif a in ('-V', '--version'):
      version(sys.stdout)
      sys.exit(0)
    elif a in ('-v', '--verbose'):
      verbose += 1
    elif a in ('-s', '--silent'):
      verbose = -1
    elif a in ('-t', '--types'):
      if type_list is None: type_list = [ ]
      for t in o.split(','): type_list.append(t.lower())
    elif a in ('-d', '--dirlist'):
      create_dirlist = True
    elif a in ('-u', '--updatep4'):
      p4_do_update = True
    elif a == '--submit':
      p4_do_update = True
      p4_do_submit = True
    elif a in ('-c', '--comment'):
      p4_comment = o
    elif a in ('-l', '--list'):
      for project in file(o, 'r').read().split('\n'):
	project = project.strip()
	if project == '': continue
	if project not in project_list:
	  project_list.append(project)
    else:
      error('Unrecognized option ' + repr(a))
      sys.exit(1)
  project_list.extend(args)
  if len(project_list) == 0:
    error('No MVS project file specified!')
    sys.exit(1)

  # Set defaults.
  if type_list is None: type_list = [ 'c', 'cpp' ]
  if p4_comment is None: p4_comment = '!R project file lists updated'

  # Process files.
  for project in project_list: process(project)

  # Submit the P4 changeset.
  if p4_do_update and p4_do_submit: p4_submit()

if __name__ == '__main__': main()

# vim: sw=2 ts=8 tw=78

