#############################################################################
## Crytek Source File
## Copyright (C) 2006, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 31, 2006
## Description: GNU-make based build system
#############################################################################

# Common include of the GNU make-based build system.
#
# This file should be included by the top level makefile of the project.
# Common.mk expects the following make variables to be set:
#
# CODE_ROOT
#	The root directory of the source code.  This is the directory
#	containing the CryEngine sub-directory.
#
# BUILD_ROOT
#	The output root directory of the build process.  All files are
#	generated within the build root directory.
#
# REPO_ROOT
#	The root directory of the repository.
#
# CYGWIN_BASEDIR (optional, only when building on Windows)
#	The Windows path to the '/base' directory mounted in the Cygwin
#	environment.  Typically this is the common base directory of
#	$CODE_ROOT, $BUILD_ROOT, and $REPO_ROOT (in Windows DRIVE:PATH
#	notation).
#
# STAGES
#	A whitespace separated list of build stages to be performed by the
#	build system.  The make system will process all listed stages in order
#	- the current stage is exposed through the exported variable STAGE.
#
#	If STAGES is undefined but STAGE is defined, then STAGES is set to
#	contain only the named STAGE.
#
# ARCH
#	The target architecure.  This is a combination of an OS- or system
#	name and a processor type.  System names are Win32, Win64, Linux, PS3.
#	Processor type names are x86 (Intel 586 compatbible), amd64, cell.
#	The list of supported architecures can be obtained by listing the *.mk
#	files in the Arch.mk subdirectory.
#
# SOLUTION
#	The base name of the solution.  The build system expects a 'solution
#	makefile' in the Code/Solutions directory.  The solution makefile
#	defines the valid targets of the solution as well as build-order and
#	link dependencies between them.  (Note: These are not code level
#	dependencies.)
#
# TARGETS
#	The (symbolic) names of the build targets of the entire build process
#	(i.e. the combination of all stages).  Valid targets are defined
#	through the solution makefile.  The build system executes the named
#	targets in the specified order.  The current target is exposed through
#	the exported variable TARGET.
#
#	If TARGETS is undefined but TARGET is defined, then TARGETS is set to
#	contain only the named TARGET.
#
# This file provides the following make targets (we'll call them
# build-commands):
#
# default
#	The default build target for target and stage.  Typically this implies
#	the generation of all result files of the target and stage.
#
# clean
#	Remove all output files of the target and stage.  This includes all
#	files pulled out of the repository.
#
# post
#	Perform only the post action of the selected stages (use only if you
#	know what you are doing).  The post action does _not_ depend on the
#	default action of the stage!
#
# store (depends on 'default')
#	Store the results of the build process into the repository.
#
# The build system output may be controlled through some MKOPTION_* variables:
#
# MKOPTION_VERBOSE
#	Verbose operation of the make system.  This causes all build process
#	commands executed to be echoed.  If this is 0 or undefined, then a
#	less verbose output format is used.
#
# MKOPTION_MSVC
#	Convert output messages to MSVC format (required for intergration into
#	Visual Studio).
#
# MKOPTION_COMPACT
#	Compact output messages (similar to MSVC style).
#
# MKOPTION_NOWRAPPER
#	If this is 1, then the build wrapper is skipped when executing build
#	process commands.

# Archtecture specific makefiles ############################################
#
# All architecure specific makefiles live in '$(MAKE_ROOT)/Arch'.  The
# base-name if these makefile matches the '$(ARCH)' variable.  Architecture
# specific makefiles are included after the configuration makefiles
# (Config.mk) have been included and before any other makefile is included.

# Solution makefiles ########################################################
#
# All solution makefiles live in '$(SOLUTION_DIR)' (which may or may not
# reside withing '$(MAKE_ROOT)').  A solution makefile defines the following:
# - A list of projects.
# - (Optional) A default project.
# - Project dependencies.  These are link order dependencies, not code
#   dependencies.
#
# PROJECTS
#	A whitespace separated list of project names.  If some projects are
#	not relevant for all architecures, then the solution makefile must set
#	the PROJECTS variable depending on the selected architecure.  For
#	that, it may use the following variables: ARCH, ARCH_OS, ARCH_CPU.
#
# (ProjectName)_DEPS
#	A whitespace separated list of dependencies for the project
#	(ProjectName).
#
# (ProjectName)_CODE
#	The code directory for the project.  If this is undefined, then the
#	project is a 'phony' project, i.e. project that just causes the listed
#	dependencies to be built.
#
#	The specifed path _must_ be relative to the $(CODE_ROOT) directory.
#	The build system expects a project specific makefile with the name
#	Project.mk there.
#
# (ProjectName)_XFLAGS
#	Project specific extended flags (typically optimization flags).  These
#	settings override the architecure specific $(XFLAGS).
#
# (ProjectName)_XFLAGS_(SourceName)
#	Per-file override for the extended flags.  (SourceName) is the full
#	name of the source file, relative to the (ProjectName)_CODE, with all
#	dashes, slashes, and dots converted to underscores.  Example: The
#	XFLAGS for the source file Foo/Bar.cpp are specified in the variable
#	(ProjectName)_XFLAGS_Foo_Bar_cpp.  It is possible to specify extended
#	flags for a precompiled header file.
#
# DEFAULT_PROJECT
#	Optional.  The name of the default project.
#
# The projects specified in a solution makefile correspond to the valid
# targets for the TARGETS variable.  If both TARGET and TARGETS are undefined
# and a default project is defined for the selected solution, then the default
# project is chosen as the target.
#
# Note: It is valid to specify (ProjectName)_DEPS and/or (ProjectName)_CODE
# for a project and then skip that project in the PROJECTS variable.

# Stage Makefiles ###########################################################
#
# Stage specific makefiles live in the Stage subdirectory.  A stage specific
# makefile defines the following targets:
#
# _prepare
#	Phony target for setting up the compilation stage.
#
# _command_$(COMMAND)
#	Phony target executing the requested build command.

#############################################################################

# Check the directory layout variables.
ifeq ($(strip $(CODE_ROOT)),)
 $(error No CODE_ROOT specified)
endif
ifndef _CODE_CONFIG_INCLUDED
 -include $(CODE_ROOT)/Config.mk
_CODE_CONFIG_INCLUDED := 1
endif
ifeq ($(strip $(BUILD_ROOT)),)
 $(error No BUILD_ROOT specified)
endif
ifndef _BUILD_CONFIG_INCLUDED
 -include $(BUILD_ROOT)/Config.mk
 _BUILD_CONFIG_INCLUDED := 1
endif
ifeq ($(strip $(REPO_ROOT)),)
 $(error No REPO_ROOT specified)
endif
export CODE_ROOT
export BUILD_ROOT
export REPO_ROOT
ifdef CYGWIN_BASEDIR
 export CYGWIN_BASEDIR
endif

# Define the MAKE_ROOT variable.
#
# This is the root directory for all shared makefiles.  This variable is
# exported.  All shared makefiles should be accessed relative to this
# variable.
ifeq ($(origin MAKE_ROOT),undefined)
MAKE_ROOT := $(CODE_ROOT)/Tools/Make
endif
export MAKE_ROOT

ifeq ($(HOST_SYSTEM),Linux)
 HOST_BINDIR := host-linux
endif
ifeq ($(HOST_SYSTEM),Windows)
 HOST_BINDIR := host-win32
endif
ifndef HOST_BINDIR
 $(error set HOST_SYSTEM to 'Linux' or 'Windows')
endif
export HOST_BINDIR
MAKE_TOOLS_BINDIR = $(MAKE_ROOT)/Tools/$(HOST_BINDIR)
export MAKE_TOOLS_BINDIR

ifndef BUILD_TEMP
 BUILD_TEMP = $(BUILD_ROOT)/Temp
endif

# Directory containing the solution makefiles.
ifeq ($(origin SOLUTION_DIR),undefined)
SOLUTION_DIR := $(CODE_ROOT)/Solutions
endif
export SOLUTION_DIR

# Check if the architecure is supported.  An architecure is supported if and
# only if the associated architecure include file is present and can be
# included without indicating an error.  The include statement for the
# architecure file is below the solution file include clause.
ifeq ($(strip $(ARCH)),)
 $(error No target architecure specified)
endif # !ARCH
_arch := $(subst -, ,$(strip $(ARCH)))
ifneq ($(words $(_arch)),2)
 $(error Bad format in architecure string $(ARCH))
endif
ARCH_OS := $(word 1, $(_arch))
ARCH_CPU := $(word 2, $(_arch))
export ARCH_OS
export ARCH_CPU

# Check if the requested solution is supported (i.e. if a matching solution
# makefile exists).
ifeq ($(strip $(SOLUTION)),)
 $(error No solution specified)
endif # !SOLUTION
include $(SOLUTION_DIR)/$(SOLUTION).mk
ifndef TARGET
 ifndef TARGETS
  ifdef DEFAULT_PROJECT
   TARGET := $(DEFAULT_PROJECT)
  endif # DEFAULT_PROJECT
 endif # !TARGETS
endif # !TARGET
export SOLUTION

include $(MAKE_ROOT)/Arch/$(ARCH).mk
ifdef ERROR
 $(error $(ERROR))
endif # ERROR
export ARCH

DEFAULT_STAGES ?= compile link
STAGES ?= $(DEFAULT_STAGES)

# Common variable setup #####################################################

# Command targets recognized by the build system.
#
COMMANDS := default clean post store

# Location of this Common.mk makefile.
#
COMMON_MK := $(MAKE_ROOT)/Common.mk

# Common arguments passed to all sub-makes.
#
COMMON_ARGS = ARCH=$(ARCH) SOLUTION=$(SOLUTION)

# Default options ###########################################################

# For DISTCC builds, we'll use -j10 as a default for the MAKEOPTIONS of the
# 'compile' stage, for local build we'll use -j2.
ifeq ($(strip $(MKOPTION_MAKEFLAGS_compile)),)
ifeq ($(MKOPTION_DISTCC),1)
MKOPTION_MAKEFLAGS_compile := -j10
else
MKOPTION_MAKEFLAGS_compile := -j2
endif
endif
# For the stages 'link' and 'scan', default to -j2.
ifeq ($(strip $(MKOPTION_MAKEFLAGS_link)),)
MKOPTION_MAKEFLAGS_link := -j2
endif
ifeq ($(strip $(MKOPTION_MAKEFLAGS_scan)),)
MKOPTION_MAKEFLAGS_scan := -j2
endif

# Build wrapper setup #######################################################
#
# Most commands executed in the build process are filtered through a build
# wrapper script 'build.pl' (part of the MakeTools project).  The build
# wrapper is invoked by prepending the value of one of the $(BUILD*) variables
# to the build process command.
#
# $(BUILD)
#	Compile a C or C++ file.
# $(BUILD_PCH)
#	Compile a C++ header file to a precompiled header.
# $(BUILD_LINK)
#	Link to an executable, shared object, DLL, relocatable object, or
#	static library.
# $(BUILD_ECHO)
#	When building non-verbose, then the specified argument(s) are echoed.
#	May perform a simple path-normalization on the specified argument(s).
# $(BUILD_SILENT)
#	Silence command execution if $(MKOPTION_VERBOSE) is not set.
#
# The build commands come in a variant with an underscore prefix (e.g.
# $(_BUILD).  These variants contain the 'raw' command without any make
# specific prefixes.  Within a shell command (e.g. a shell-level if
# statement), the raw variant should be used instead of the default variant.

ifeq ($(MKOPTION_NOWRAPPER),1)
# No build wrapper.
 ifneq ($(MKOPTION_VERBOSE),1)
  _BUILD =
  _BUILD_PCH =
  _BUILD_LINK =
  _BUILD_ECHO = echo
  BUILD = $(SILENT) $(_BUILD)
  BUILD_PCH = $(SILENT) $(_BUILD_PCH)
  BUILD_LINK = $(SILENT) $(_BUILD_LINK)
  BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
 else # MKOPTION_VERBOSE
  _BUILD =
  _BUILD_PCH =
  _BUILD_LINK =
  _BUILD_ECHO = :
  BUILD = $(_BUILD)
  BUILD_PCH = $(_BUILD_PCH)
  BUILD_LINK = $(_BUILD_LINK)
  BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
 endif # MKOPTION_VERBOSE else
else # MKOPTION_NOWRAPPER
 BUILDTOOL := $(PERL) "$(MAKE_ROOT)/Tools/build.pl"
 BUILDECHOTOOL := $(PERL) "$(MAKE_ROOT)/Tools/echo.pl"
 ifeq ($(MKOPTION_MSVC),1)
  BUILDTOOL += -x
 endif
 ifeq ($(MKOPTION_STDERR),1)
  BUILDTOOL += --stderr
 endif
 ifeq ($(HOST_SYSTEM),Windows)
  # Cygwin mode.
  BUILDTOOL += -c
  BUILDECHOTOOL += -c
  ifdef CYGWIN_BASEDIR
   BUILDTOOL += -b '$(CYGWIN_BASEDIR)'
   BUILDECHOTOOL += -b '$(CYGWIN_BASEDIR)'
  endif
 endif
 ifeq ($(ARCH_OS),PS3)
  # PS3 mode (move stub libs to the end).
  BUILDTOOL += --ps3
 endif
 ifneq ($(MKOPTION_VERBOSE),1)
  BUILDTOOL += -s
  ifeq ($(MKOPTION_MSVC),1)
   _BUILD_ECHO = echo
   BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
  else
   _BUILD_ECHO = $(BUILDECHOTOOL) -t "$(TARGET)" --
   BUILD_ECHO = $(SILENT) $(BUILDECHOTOOL) -t "$(TARGET)" --
  endif
 else
  _BUILD_ECHO = :
  BUILD_ECHO = $(SILENT) $(_BUILD_ECHO)
 endif
 _BUILD = $(BUILDTOOL) -m compile --
 _BUILD_PCH = $(BUILDTOOL) -m compile_pch --
 _BUILD_LINK = $(BUILDTOOL) -m link --
 BUILD = $(SILENT) $(_BUILD)
 BUILD_PCH = $(SILENT) $(_BUILD_PCH)
 BUILD_LINK = $(SILENT) $(_BUILD_LINK)
endif # MKOPTION_NOWRAPPER else
ifeq ($(MKOPTION_VERBOSE),1)
 _BUILD_SILENT =
 BUILD_SILENT = $(_BUILD_SILENT)
else
 _BUILD_SILENT =
 BUILD_SILENT = $(SILENT) $(_BUILD_SILENT)
endif

# The log-rotate command.  Log files in the per-stage subdirectories of
# $(LOGDIR) are rotated at the beginning of the stage.
LOGDIR = $(BUILD_ROOT)/Log
LOGROTATE = $(PERL) "$(MAKE_ROOT)/Tools/logrotate.pl"
ifeq ($(MKOPTION_VERBOSE),1)
 LOGROTATE += --verbose
endif

# The special functions $(BUILD_MAKE_LDFILES) and $(BUILD_LDFILES) handle the
# generation and referencing of files passed to a linker.
#
# $(call BUILD_MAKE_LDFILES,$(OBJECTS),$(TARGET))
#   If $(MKOPTION_LDFILES) is set, generate the list file from the specified
#   objects.  On Windows systems, the file names are transformed into Windows
#   notation.  If $(MKOPTION_LDFILES) is not set, then the functions expands
#   to nothing.
#   Note: The expanded function already contains the appropriate silence
#   prefixes and it should be called without any prefixes.
# $(call BUILD_LDFILES,$(OBJECTS),$(TARGET))
#   If $(MKOPTION_LDFILES) is set, then the function expands to the list file
#   with a '@' prefix.  If $(MKOPTION_LDFILES) is not set, the function
#   expands to the specified objects list.
# $(call BUILD_RM_LDFILES,$(TARGET))
#   Remove the list file.
LDFILES_SUFFIX := files
ifeq ($(MKOPTION_LDFILES),1)
 build_make_ldfiles_echo := $(PERL) $(MAKE_ROOT)/Tools/echo.pl
 ifeq ($(HOST_SYSTEM),Windows)
  build_make_ldfiles_echo += -c
  ifdef CYGWIN_BASEDIR
   build_make_ldfiles_echo += -b '$(CYGWIN_BASEDIR)'
  endif
 endif
 BUILD_MAKE_LDFILES = \
	$(SILENT) $(RM) '$(2).$(LDFILES_SUFFIX)'; \
	$(build_make_ldfiles_echo) -- $(1) >'$(2).$(LDFILES_SUFFIX)'
 BUILD_LDFILES = '@$(2).$(LDFILES_SUFFIX)'
 BUILD_RM_LDFILES = $(SILENT) $(RM) '$(1).$(LDFILES_SUFFIX)'
else # MKOPTION_LDFILES == 1
 BUILD_MAKE_LDFILES = $(SILENT) :
 BUILD_LDFILES = $(1)
 BUILD_RM_LDFILES = $(SILENT) :
endif # else MKOPTION_LDFILES == 1

# Make target definitions. ##################################################

# Pseudo stages.  If $(STAGE) or $(STAGES) contain one of the pseudo stages,
# then the pseudo stage is expanded to the list of stages defined as
# STAGES_(pseudoStage).
STAGES_all := compile link

# Flags to be passed to all sub-makes.
# Note that the name is MAKE_FLAGS with an underscore in the middle!  GNU make
# uses a magic variable named $(MAKEFLAGS) for communication with sub-make
# processes.
MAKE_FLAGS := --no-print-directory
ifneq ($(MKOPTION_VERBOSE),1)
 MAKE_FLAGS += --quiet
endif

# Build progress messages.
#
# MSG_STAGE_BEGIN (stage, command)
#	Echo a message at the beginning of a stage.
#
# MSG_STAGE_END (stage, command)
#	Echo a message at the (successful) end of a stage.
#
# MSG_STAGE_FAILED (stage, command)
#	Echo a message indicating the failure of a stage.
#
# MSG_TARGET_BEGIN (target, stage, command)
#	Echo a message at the beginning of a target.
#
# MSG_TARGET_END (target, stage, command)
#	Echo a message at the (successful) end of a target.
#
# MSG_TARGET_FAILED (target, stage, command)
#	Echo a message indicating the failure of a target.
#
# MSG_READING_VCPROJ
#	Echo a message indicating that a VCPROJ file is being read.
include $(MAKE_ROOT)/Messages.mk

# Template for a sub-make invocation executing a stage.
# $(1) is the make command passed through the $(COMMAND) variable
#   (i.e.  'default', 'clean', ...);
# $(2) is the stage name.
define execstage_template
	if [ '$(1)' = default ]; then \
	  $$(LOGROTATE) -l "$$(LOGDIR)/$(2)"; \
	fi && \
	$$(MAKE) \
	  -f "$(COMMON_MK)" \
	  $$(MAKE_FLAGS) \
	  $$(MKOPTION_MAKEFLAGS) \
	  $$(MKOPTION_MAKEFLAGS_$(2)) \
	  COMMAND="$(1)" \
	  STAGE="$(2)" \
	  STAGES= \
	  $(COMMON_ARGS) \
	  _exec_stage &&
endef

# Define the make command targets for iterating over the specifed stages.  If
# STAGE is defined instead of STAGES, then we'll call the command target to
# execute only that stage.
ifneq ($(strip $(STAGES)),)
define command_template
.PHONY: $(1)
$(1):
	$(SILENT) $(foreach stage,\
	  $(STAGES),\
	  $(call execstage_template,$(1),$(stage))) :
endef
else # STAGES
ifndef STAGE
 $(error No STAGES specified)
endif # !STAGE
# Check if this is a pseudo stage.  We don't want pseudo stages to be
# introduces through environment variables or overrides, so we insist on the
# origin 'file' here.
ifeq ($(origin STAGES_$(STAGE)),file)
define command_template
.PHONY: $(1)
$(1):
	$(SILENT) $(foreach stage,\
	  $(STAGES_$(STAGE)),\
	  $(call execstage_template,$(1),$(stage))) :
endef
else # if pseudo stage
define command_template
.PHONY: $(1)
$(1):
	$(SILENT) $(call MSG_STAGE_BEGIN,$(STAGE),$(1))
	$(SILENT) $(call execstage_template,$(1),$(STAGE)) : || { \
	  $(call MSG_STAGE_FAILED,$(STAGE),$(1)); \
	  false; \
	}
	$(SILENT) $(call MSG_STAGE_END,$(STAGE),$(1))
endef
endif # if pseudo stage; else
endif # STAGES else
$(foreach command,$(COMMANDS),$(eval $(call command_template,$(command))))

.PHONY: _exec_stage
ifdef TARGETS
_exec_stage:
	$(SILENT) for TARGET in $(TARGETS); do \
	  $(MAKE) -f "$(COMMON_MK)" \
	    COMMAND=$(COMMAND) \
	    STAGE=$(STAGE) \
	    TARGET=$$TARGET \
	    $(COMMON_ARGS) \
	    _exec_target \
	  || exit $$?; \
	done
else # TARGETS
ifndef TARGET
 $(error No TARGETS specified)
endif # !TARGET
_exec_stage:
	$(SILENT) $(MAKE) -f "$(COMMON_MK)" \
	  COMMAND=$(COMMAND) \
	  STAGE=$(STAGE) \
	  TARGET=$(TARGET) \
	  $(COMMON_ARGS) \
	  _exec_target_$(TARGET)
endif # TARGETS else

ifdef TARGET

# The MakeTools project is always present.  All other (non-phony) projects
# depend on it.  Note that we're specifying an absolute path, because the make
# system might be called from a project outside of our code tree.
MakeTools_CODE := $(MAKE_ROOT)/Tools

ifdef $(TARGET)_CODE
define target_template
.PHONY: _exec_target_$(1)
_exec_target_$(1): $$(addprefix _exec_target_,$$($(1)_DEPS)) $(2)
	$(SILENT) mkdir -p "$(BUILD_ROOT)"
	$(SILENT) mkdir -p "$(BUILD_TEMP)"
	$(SILENT) $(call MSG_TARGET_BEGIN,$(1),$(STAGE),$(COMMAND))
	$(SILENT) $$(MAKE) -f "$(COMMON_MK)" \
	  COMMAND=$(COMMAND) \
	  STAGE=$(STAGE) \
	  TARGET=$(1) \
	  BUILD_STRIP_PREFIX=$(CURDIR) \
	  -C "$(BUILD_ROOT)" \
	  $(COMMON_ARGS) \
	  EXEC_TARGET=1 \
	  _exec_target || { \
	    $(call MSG_TARGET_FAILED,$(1),$(STAGE),$(COMMAND)); \
	    false; \
	  }
	$(SILENT) $(call MSG_TARGET_END,$(1),$(STAGE),$(COMMAND))
endef
else # $(TARGET)_CODE
# If  $(TARGET)_CODE is not defined, then the project is a phony project and
# no execute steps are preformed.
define target_template
.PHONY: _exec_target_$(1)
_exec_target_$(1): $$(addprefix _exec_target_,$$($(1)_DEPS))
	$(SILENT) :
endef
endif # $(TARGET)_CODE else
endif # TARGET
$(foreach project,\
	$(PROJECTS),\
	$(eval $(call target_template,$(project),_exec_target_MakeTools)))
$(eval $(call target_template,MakeTools,))

# The $(EXEC_TARGET) variable is defined only when we're called from an
# _exec_target_* command.  We'll include the stage specific code in this case.
#
ifdef EXEC_TARGET

# Define $(PROJECT_CODE) to point to the code directory of the target project.
# This is an absolute path.  Note that $($(TARGET)_CODE) may be an absolute
# directory.
ifeq ($(patsubst /%,,$($(TARGET)_CODE)),)
PROJECT_CODE := $($(TARGET)_CODE)
else
PROJECT_CODE := $(CODE_ROOT)/$($(TARGET)_CODE)
endif
export PROJECT_CODE

# $(PROJECT_BUILD) points to the build directory for the project.  We'll
# remove the strip prefix from $($(TARGET)_CODE).
target_code := $(patsubst $(BUILD_STRIP_PREFIX)%,%,$($(TARGET)_CODE))
ifdef DEPOT_MAIN
 target_code := $(patsubst $(DEPOT_MAIN)%,%,$($(TARGET)_CODE))
endif
target_code := $(patsubst /%,%,$(target_code))
PROJECT_BUILD := $(BUILD_ROOT)/$(target_code)
target_code :=
export PROJECT_BUILD

# The cleanup file lists for the supported stages.  All potentially generated
# files for a stage should be added to the corresponding clean lists for that
# stage.  The only exceptions are the 'Files.mk' files generated from the
# VCPROJ files - these are created on demand (in any stage) and are collected
# in the final cleanup list $(FINAL_CLEAN_$(TARGET)).
ifneq ($(TARGET),)
 SCAN_CLEAN_$(TARGET) :=
 COMPILE_CLEAN_$(TARGET) :=
 LINK_CLEAN_$(TARGET) :=
 FINAL_CLEAN_$(TARGET) :=
endif

include $(PROJECT_CODE)/Project.mk

# Load the architecure specific setup file, if present.
-include $(MAKE_ROOT)/Arch/$(ARCH).setup.mk

# Set default values for various PROJECT_* variables.
PROJECT_TYPE ?= module

include $(MAKE_ROOT)/Stage/$(STAGE).mk

.PHONY: _exec_target
_exec_target: _prepare _command_$(COMMAND)
ifeq ($(COMMAND),clean)
	$(BUILD_SILENT) rm -f $(FINAL_CLEAN_$(TARGET))
else
	$(SILENT) :
endif

endif # EXEC_TARGET

# Common.mk
# vim:ts=8:sw=2

