#############################################################################
## Crytek Source File
## Copyright (C) 2006, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 31, 2006
## Description: GNU-make based build system
#############################################################################

# Make include file for setting default values based on a selected
# configuration.
#
# This file is typically included by a top level makefile (before Common.mk is
# included).  All values are set only if undefined.  This file sets the
# following default values:
#
# STAGES
#	Set to 'compile'.  This default may change in the future (i.e. include
#	more stages.  It will always be set to a value that makes a rebuild
#	feasable.
#
# TARGETS
#	Kept undefined - will be set by Common.mk to the value of the the
#	$(DEFAULT_PROJECT) variable defined in the CryEngine.mk solution
#	makefile.
#
# CODE_ROOT
#	Set to the subdirectory 'Code' of the current working directory of the
#	caller.
#
# REPO_ROOT
#	Set to the subdirectory 'Repo/$(BUILD_NAME)' of the current working
#	directory of the caller.
#
# BUILD_ROOT
#	Set to the subdirectory 'Build/$(BUILD_NAME)' of the current working
#	directory of the caller.
#
# SOLUTION
#	Set to 'CryEngine'.
#
# OPTION_*
# MKOPTION_*
#	Set based on the chosen configuration.
#
# CONFIG_NAME
#	A short name for the chosen configuration.  May be equal to the
#	specified value of $(CONFIG).
#
# BUILD_NAME
#	The logical name of the build.  This is a combination of architecture
#	and configuration settings.  May be used as part of the $(BUILD_ROOT)
#	and/or $(REPO_ROOT).
#
# Some default values are set based on a chosen configuration.  To select a
# configuration, the caller must set the $(CONFIG) variable.  Configurations
# are kept in the $(MAKE_ROOT)/Config directory.
#
# Note that this file will chain-include the Common.mk makefile, so this file
# included by the top-level makefile _instead_of_ Common.mk, not in addition
# to it.

#############################################################################

# Try to include a Config.mk directory in the current working directory.  The
# $(CONFIG) variable may be set there.
-include $(CURDIR)/Config.mk

SOLUTION ?= CryEngine

SILENT := @
export SILENT

ifeq ($(CONFIG),)
 $(error No configuration selected)
endif

ifeq ($(origin CODE_ROOT),undefined)
 CODE_ROOT := $(CURDIR)/Code
endif

ifneq ($(origin CONFIG_NAME),undefined)
 $(error CONFIG_NAME must be undefined - will be defined by Default.mk)
endif

ifeq ($(origin MAKE_ROOT),undefined)
 MAKE_ROOT := $(CODE_ROOT)/Tools/Make
endif
export MAKE_ROOT

include $(MAKE_ROOT)/Config/$(CONFIG).mk

ifeq ($(origin CONFIG_NAME),undefined)
 CONFIG_NAME := $(CONFIG)
endif
ifeq ($(origin BUILD_NAME),undefined)
 ifneq ($(CONFIG_NAME),)
  BUILD_NAME := $(ARCH)-$(CONFIG_NAME)
 else
  BUILD_NAME := $(ARCH)
 endif
endif # origin(BUILD_NAME) != undefined
ifeq ($(origin BUILD_ROOT),undefined)
 BUILD_ROOT := $(CURDIR)/Build/$(BUILD_NAME)
endif
ifeq ($(origin REPO_ROOT),undefined)
 REPO_ROOT := $(CURDIR)/Repo/$(BUILD_NAME)
endif

ifeq ($(HOST_SYSTEM),Windows)
 MKOPTION_MSVC ?= 1
 MKOPTION_COMPACT ?= 1
 MKOPTION_LDFILES ?= 1
endif
ifeq ($(HOST_SYSTEM),Linux)
 MKOPTION_MSVC ?= 0
 MKOPTION_COMPACT ?= 0
endif

MKOPTION_VERBOSE ?= 0
MKOPTION_NOWRAPPER ?= 0

OPTION_MODULE_LINKMODE ?= shared
OPTION_COMPRESSED_XML ?= 1

# FIXME Should be removed some day.
# Compatibility link for obsolete MKOPTION_NOWRAPPER to OPTION_NODEBUGSYM.
ifdef MKOPTION_NODEBUGSYM
 $(warning obsolete MKOPTION_NODEBUGSYM used, use OPTION_NODEBUGSYM instead!)
 ifndef OPTION_NODEBUGSYM
  OPTION_NODEBUGSYM := $(MKOPTION_NODEBUGSYM)
 endif
endif

export HOST_SYSTEM

# Exported Configuration Variables ###########################################

# Build process options
# ---------------------
#
# These options start with a 'MKOPTION_' prefix and
# affect the build process.  These options should have no effect on the
# generated binaries.

# Enable message format conversion from the specified compiler to MSC
# compatible message formatting.  (VisualStudio integration.)
export MKOPTION_MSVC

# Enable verbose output.  If this option is enabled, then all compilation and
# linking commands are echoed (complete command line) and tool scripts are run
# with the -v (verbose) option (if supported).
export MKOPTION_VERBOSE

# Enable compact output.  If this option is set (and if MKOPTION_VERBOSE is
# not set), then build progress output is very compact (similar to MSC output
# in VS).
export MKOPTION_COMPACT

# Disable the build wrapper scripts.  This option is intended for debugging
# the make system and probably works only on Linux (if at all).
export MKOPTION_NOWRAPPER

# Enable local compiler output caching using ccache.  This option is generally
# not useful for local workstation builds, but may be useful for speeding up
# full-builds on build servers.
export MKOPTION_CCACHE

# Enable distributed compilation using distcc.  This option automatically
# enables 'MKOPTION_NOPCH', because precompiled header files don't work well
# together with distributed compilation.
export MKOPTION_DISTCC

# Disable the use and generation of precompiled headers.
export MKOPTION_NOPCH

# Use linker file lists.  If this option is enabled, then object file lists
# are passed to the linker using a command file (using the '@file' syntax).
# This helps to work around command line length limitations on some systems
# and is activated automatically if $(HOST_SYSTEM) is 'Windows'.
export MKOPTION_LDFILES

# The following options can be used to specify per-stage flags to $(MAKE).
export MKOPTION_MAKEFLAGS
export MKOPTION_MAKEFLAGS_scan
export MKOPTION_MAKEFLAGS_compile
export MKOPTION_MAKEFLAGS_link

# The following make options can be used to specify the version number and
# library directory locations of the PPU GCC for the PS3 build.  If these
# values are _not_ specified, then they are determined automatically through
# the 'gcc_version.pl' tool.
export MKOPTION_PS3_GCC_VERSION
export MKOPTION_PS3_GCC_LIBDIR
export MKOPTION_PS3_GCC_LIBEXECDIR

# This option enables the copying of the output ELF and PRX files to the
# specified output directory. The ELF/PRX files are required for debugging
# with GDB, since GDB can not read debug information from the SELF/SPRX files.
# The ELF/PRX files are _not_ needed for the ProDG debugger.
export MKOPTION_PS3_COPYELF

# This option enabled fast PRX linking.  If this option is set to 1, then stub
# libraries are created only on a full build.
export MKOPTION_PS3_PRXFASTLINK

# Define this if you want to scan all code by default (i.e. unless
# $(PROJECT_SCAN_C/CPP) is defined).  If this is not set (or set to 0), then
# only source files listed in $(PROJECT_SCAN_C/CPP) are scanned.  Note that
# this default behavior may change at some point.
export MKOPTION_PS3_SCANALL

# Build target options
# --------------------
#
# These options start with a 'OPTION_' prefix.  These options _may_ have an
# effect on the generated binaries.
#
# These options are typically set in the build configuration file
# 'Make/Config/$(CONFIG).mk', but may also be specified/overriden in the
# local configuration file 'Config.mk' or on the make command line.

# Flag indicating a debug build.  This option turns on various debugging
# facilities (by defining '_DEBUG' and by added debugging options to compilers
# and linkers).
export OPTION_DEBUG

# Flag indicating a release build.  Optizations are turned on.  Profiling
# facilities are turned off.
export OPTION_RELEASE

# Flag indicating a profile build.  Optizations and profiling facilities are
# turned on.  This option may be combined with 'OPTION_DEBUG'.
export OPTION_PROFILE

# Flag indicating that we're building a dedicated server.
export OPTION_DEDICATED

# Option specifying the link mode for modules.  This is either 'static' or
# 'shared'.  (Note that for PS3 shared linking is enabled through
# 'OPTION_PS3_PRX'.)
ifdef OPTION_MODULE_LINKMODE
 export OPTION_MODULE_LINKMODE
endif

# Disable all debugging symbols in compiled code.  This option is intended for
# compilation test builds to speed up linking time.  The generated binaries
# are not useful for debugging.
ifdef OPTION_NODEBUGSYM
 export OPTION_NODEBUGSYM
endif

# Enable user (developer) specific code.  Should be set to the login name of
# the user (developer).  It is safe to leave this option unset.
ifdef OPTION_USER
 export OPTION_USER
endif

# Enable whole project compilation.  If this option is set, then an entire
# project is compiled in a single translation unit.  This option may be set or
# cleared from the project makefiles ('Project.mk').
ifdef OPTION_WHOLE_PROJECT
 export OPTION_WHOLE_PROJECT
endif

# Enable skipping the link step when linking modules.  This may be useful in
# combination with $(OPTION_WHOLE_PROJECT) if the target is linked statically.
# Some configurations knows not to work with this option will turn it off
# automatically.
# NOTE: If a non-trivial module linking step is required, then this option
# will break the build process - use with care and only if you know what you
# are doing.
ifdef OPTION_SKIPLINK
 export OPTION_SKIPLINK
endif

# Enable whole project compilation for C files.  This option is effective only
# if $(OPTION_WHOLE_PROJECT) is also set.  By default (i.e. if
# $(OPTION_WHOLE_PROJECT_C) is not set) whole project compilation only affects
# C++ source files.
ifdef OPTION_WHOLE_PROJECT_C
 export OPTION_WHOLE_PROJECT_C
endif

# If specified, then this option sets an override for the configuration's
# renderer module name.  Configurations may override this option if the
# renderer is not selectable for the configuration.
ifdef OPTION_RENDERER
 export OPTION_RENDERER
endif

# Enable GPROF profiling.  GPROF profiling is supported only for GCC based
# builds (currently Linux and PS3).
ifdef OPTION_GPROF
 export OPTION_GPROF
endif

# PS3 specific build target options
# ---------------------------------

# Option enabling garbage collection using function sections.  When building
# PRX modules, this option affects only the non-PRX code (since PRXs do not
# support garbage collection).  The collection is currently implemented using
# the --gc-sections option in the final link, which results in a sub-optimal
# ordering of the function sections.  The final build environment will
# probably provide a more efficient collection.
ifdef OPTION_PS3_GCSECTIONS
 export OPTION_PS3_GCSECTIONS
endif

# Option enabling support for the SN-Systems profiler SN Tuner.
ifdef OPTION_PS3_SNTUNER
 export OPTION_PS3_SNTUNER
endif

# Option enabling optimized code guarded by '#ifdef PS3OPT'.  These are
# optimizations which should be kept out of non-PS3 build to avoid breakage.
ifdef OPTION_PS3_OPTIMIZED_CODE
 export OPTION_PS3_OPTIMIZED_CODE
endif

# Disable the use of the CryMemoryManager.  Note that CryMemoryManager is used
# for strings even if this option is set.
ifdef OPTION_PS3_NOMEMMAN
 export OPTION_PS3_NOMEMMAN
endif

# Enable the GCM HUD graphical debugger.
ifdef OPTION_PS3_GCMHUD
 export OPTION_PS3_GCMHUD
endif

# Links the GCM Debug library
ifdef OPTION_PS3_GCMDEBUG
 export OPTION_PS3_GCMDEBUG
endif

# Enable command buffer checking code.  If this option is set, then command
# buffer checking can be enabled through the launcher configuration option
# 'enable_cmdbuffer_check'.
#
# Possible values are:
# 1 - Basic checking.  This option only makes sure that the active part of the
#     command buffer is not overwritten before it is flushed.  This check can
#     be used to verify that a command buffer corruption is responsible for a
#     rendering problem.  The cost of this is relatively low.
# 2 - Full checking.  This option enables a full command buffer verification
#     on every operation.  This is very expensive but should be capable of
#     catching a command buffer corruption closer to the cause than basic
#     checking.
ifdef OPTION_PS3_GCMDEBUG_CMDBUFFERCHECK
 export OPTION_PS3_GCMDEBUG_CMDBUFFERCHECK
endif

# Links/builds against profile lib of SPU job manager
ifdef OPTION_PS3_SPU_PROFILE
 export OPTION_PS3_SPU_PROFILE
endif

# Skip the make_fself step after the final link.  The make_fself is sipped for
# the program ELF and for the PRX modules (if configured).
ifdef OPTION_PS3_NOMAKEFSELF
 export OPTION_PS3_NOMAKEFSELF
endif

# Option enabling the link-time malloc wrapper code.  Note that the malloc
# wrapper does not work within PRX modules!
ifdef OPTION_PS3_MALLOCWRAPPER
 export OPTION_PS3_MALLOCWRAPPER
endif

# Enable shared module linking using the PRX system.  Note that PS3 shared
# linking via PRX can currently not be enabled using the
# OPTION_MODULE_LINKMODE option, because PRX linking is much more restrictive
# than 'classical' shared linking and requires explicit support in the code.
ifdef OPTION_PS3_PRX
 export OPTION_PS3_PRX
endif

# Disable the export-pickup step when compiling PRX modules.  The default for
# this option is currently OFF but will be changed to ON if/when automatic
# function/method export will be needed.  Better leave this option undefined.
ifdef OPTION_PS3_NOPRXEXPORTPICKUP
 export OPTION_PS3_NOPRXEXPORTPICKUP
endif

# Enable the stripping of PRX modules.  This is an experimental feature
# possibly causing problems.  Try disabling this option in case of PRX
# trouble.
ifdef OPTION_PS3_PRXSTRIP
 export OPTION_PS3_PRXSTRIP
endif

# Disable optimization for SPU jobs.  Note that this may cause the SPU code
# size exeed the limits and may require special job page setups to handle the
# code size.
ifdef OPTION_PS3_SPU_NOOPTIMIZE
 export OPTION_PS3_SPU_NOOPTIMIZE
endif

# Enables SN-toolchain debugging for SPU jobs.
ifdef OPTION_PS3_SPU_SNDEBUG
 export OPTION_PS3_SPU_SNDEBUG
endif

# Enables SN-toolchain debugging for SPU jobs.
ifdef OPTION_PS3_SPU_NO_BRANCH_HINT
 export OPTION_PS3_SPU_NO_BRANCH_HINT
endif

# Enable SPU asserts for SPU jobs.
ifdef OPTION_PS3_SPU_ASSERT
 export OPTION_PS3_SPU_ASSERT
endif

# Enable SPU cache asserts for SPU jobs.
ifdef OPTION_PS3_SPUCACHE_ASSERT
 export OPTION_PS3_SPUCACHE_ASSERT
endif

# Enable runtime cache hazard checking for SPU jobs.
ifdef OPTION_PS3_SPUCACHE_HAZARD_CHECK
 export OPTION_PS3_SPUCACHE_HAZARD_CHECK
endif

# If $(MKOPTION_PS3_COPYELF) is enabled, then this is the path to the output
# directory for copying the ELF/SELF and PRX/SPRX files.
ifdef OPTION_PS3_OUTPUT_DIR
 export OPTION_PS3_OUTPUT_DIR
endif

# The copied files will use a suffix indicating the build type (typically a
# combination of '_debug' and '_profile').  This option may be used to specify
# a custom suffix.
ifdef OPTION_PS3_OUTPUT_SUFFIX
 export OPTION_PS3_OUTPUT_SUFFIX
endif

# Extra options to be passed to the CryCG tool.
ifdef OPTION_PS3_CRYCGFLAGS
 export OPTION_PS3_CRYCGFLAGS
endif

include $(MAKE_ROOT)/Common.mk

# Default.mk
# vim:ts=8:sw=8

