#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2006, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Fri Jun 29, 2007
## Description: GNU-make based build system
#############################################################################

# Get the GCC version number of the specified GCC compiler.

use warnings;
use strict;

use Getopt::Std;
use String::ShellQuote;

getopts('da:p:smq:h');
our($opt_d, $opt_a, $opt_p, $opt_s, $opt_m, $opt_q, $opt_h);

sub help
{
	print <<EOF
gcc_version.pl: Get the GCC version number for a specified GCC executable.
Synopsis:
  \$PERL gcc_version.pl (Options) -- (GCC-executable)
Options:
-d
	Return the output as an assignment to the variables.  If this option is
	omitted, then the queried values are returned in a whitespace separated
	list.
-p (PREFIX)
  Prefix for the generated variable names.
-a (ARCH)
  The GCC architecture string.
-s|-m
  Generate a variable assignment in shell syntax (-s) or make syntax (-m).
-q (TOKEN)[,(TOKEN) ...]
  Specify which values should be queried.  The following tokens are
	recognized:
	* version: The GCC version string (e.g. '4.1.2').
	* libdir: The GCC library directory.
	* libexecdir: The GCC 'libexec' directory.
EOF
  ;
}

if ($opt_h)
{
	help;
	exit 0;
}
if ($#ARGV != 0)
{
	print STDERR "invocation syntax error ('gcc_version.pl -h' for help)\n";
	exit 1
}
my $GCC = $ARGV[0];

# Get the GCC version number.
my $GCC_VERSION;
{
	local *GCCOUT;
	open GCCOUT, '-|', $GCC.' --version 2>&1 || exit @FAILED:$?@'
	  or die "can not execute GCC '$GCC': $!";
	my $line = <GCCOUT>;
	if ($line =~ /\@FAILED:([0-9]+)\@/)
	{
		die "GCC command terminated with exit code $1";
	}
	if (not $line =~ /([1-9][0-9]*\.[0-9]+(\.[0-9]+)?)/)
	{
		die "can not extract version number from result line '$line'";
	}
	$GCC_VERSION = $1;
}

# Locate the GCC base directory.
my $GCC_DIR;
my $GCC_BASENAME;
if ($GCC =~ /^(\/.*)\/bin\/([^\/]+)$/)
{
	$GCC_DIR = $1;
	$GCC_BASENAME = $2;
}

sub output($$)
{
	my $varName = shift;
	if ($opt_p) { $varName = $opt_p.$varName; }
	my $value = shift;
	if (not $opt_a)
	{
		print "$value\n";
	}
	elsif ($opt_s)
	{
		my $valueQuoted = shell_quote $value;
		print "$varName=$valueQuoted ";
	}
	elsif ($opt_m)
	{
		print "$varName := $value\n";
	}
	else
	{
		print "$varName = $value\n";
	}
}

my $queryString = $opt_q;
if (not $queryString) { $queryString = 'version'; }
my @queryList = split /,/, $queryString;
my $query;
foreach $query (@queryList)
{
	if ($query eq 'version')
	{
		output 'VERSION', $GCC_VERSION;
	}
	elsif ($query eq 'libdir' or $query eq 'libexecdir')
	{
		if (not $GCC_DIR)
		{
			die "GCC executable path '$GCC' not recognized";
		}
		if (not $opt_a)
		{
			die "architecture option -a not specified";
		}
		if ($query eq 'libdir')
		{
			my $libdir = "$GCC_DIR/lib/gcc/$opt_a/$GCC_VERSION";
			if (not -d $libdir) { die "libdir '$libdir' not found"; }
			output 'LIBDIR', $libdir;
		}
		else
		{
			my $libexecdir = "$GCC_DIR/libexec/gcc/$opt_a/$GCC_VERSION";
			if (not -d $libexecdir) { die "libexecdir '$libexecdir' not found"; }
			output 'LIBEXECDIR', $libexecdir;
		}
	}
}

# Tools/gcc_version.pl
# vim:ts=2:sw=2

